#ifdef WIN32

#define _CRT_SECURE_NO_WARNINGS

#include <Windows.h>
#include <io.h>
#include <direct.h>

#endif

#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <sys/resource.h>
#include <sys/wait.h>
#endif
#include <ctype.h>
#include <signal.h>
#include <setjmp.h>
#include <fcntl.h>
#include <time.h>
#include <stdio.h>
#include <string.h>
#ifdef WIN32
#define snprintf _snprintf
#else
#include <stdlib.h>
#include <unistd.h>
#endif

#include "xmalloc.h"
#include "util.h"
#include "object.h"
#include "code.h"
#include "source.h"
#include "detect.h"
#include "ta.h"
#include "clics_output.h"
#include "data.h"
#include "read_assembler.h"

// #define SANITY

int			g_source_cnt         = 0;
int			g_clone_cnt          = 0;
char		*g_html_dirP         = 0;
char		*g_ta_fileP          = 0;
char		*g_clics_fileP       = 0;
char		*g_clics_file1P      = 0;
int			g_match_lines        = 1;
int			g_match_macros       = 0;
int			g_min_match_sequence = 15;
int			g_min_function_sequence = 10;
int			g_ignore_macros      = 0;
int			g_mismatch_cost      = -1;
int			g_same_cost          = 1;
long long	g_comparisons        = 0;
int			g_report_c           = 1;
int			g_report_assembler   = 1;
int			g_report             = 1;
int			g_debug_dwarf        = 0;
int			g_debug_code         = 0;
int			g_use_frame_address  = 0;
int			g_use_temp_address   = 0;
int			g_top_clones         = 1024;
int			g_top_cols           = 8;
int			g_hillclimb			 = 0;
time_t		g_hillclimb_elapse   = 0;
int			g_hillclimb_timeout  = 60;
int			g_iterations         = 0x7FFFFFFF;
#ifdef CORRELATE
int			g_correlate          = 0;
#endif
int			g_best_weight        = 0;

int			g_read               = 0;
int			g_functions          = 0;
int			g_max_function_lth   = 0;
int			g_compiled_ok        = 0;
int			g_compiled_bad       = 0;

long long	g_original_matches   = 0;
long long	g_extra_matches      = 0;
long long	g_cancelled_matches  = 0;
long long	g_assembler_instructions = 0;
int			g_max_blocks_seen    = 0;
int			g_max_iterations_seen= -1;

char		**g_envPP;

FILE		*stdinF              = stdin;
FILE		*indexF              = 0;
FILE		*sourceF             = 0;
FILE		*cloneF              = 0;
FILE		*taF                 = 0;
FILE		*clicsF				 = 0;
FILE		*clics1F             = 0;
FILE		*tempF				 = 0;

int			g_verbose        = 0;
int			g_unlink         = 1;
int			g_skip_compile   = 0;
int			g_stdin_lineno   = 0;

jmp_buf		g_break_address;

typedef struct {
	int		signal;
	char	*nameP;
} sigtableT;

static sigtableT	sigtable[] = {
#ifndef WIN32
	{SIGHUP, "HUP"},
	{SIGBUS, "BUS"},
#endif
	{SIGINT, "INT"},
	{SIGILL, "ILL"},
	{SIGABRT, "ABRT"},
	{SIGFPE, "FPE"},
	{SIGSEGV, "SEGV"},
	{SIGTERM, "TERM"},
	{0, 0}
};

char *g_signalP = 0;

static void
handler(int sig)
{
	sigtableT	*sigtableP;

	for (sigtableP = sigtable; ; ++ sigtableP) {
		if (!sigtableP->nameP) {
			fprintf(stderr, "**UNEXPECTED SIGNAL %d CAUGHT**\n", sig);
			break;
		}
		if (sig == sigtableP->signal) {
			g_signalP = sigtableP->nameP;
			fprintf(stderr, "**SIG%s**\n", g_signalP);
			longjmp(g_break_address, sig);
			break;
}	}	}

static void
get_environment(int argc, char **argv)
{
	int		i, val, ret;
	char	*P;

	for (i = 1; i < argc;) {
		P = argv[i++];
		if (*P != '-') {
			goto dash_i;
		}	
		switch (P[1]) {
		case 'a':
			g_match_lines = 0;
			if (P[2]) {
				P += 2;
			} else {
				if (argc <= i) {
					break;
				}
				P = argv[i++];
			}
			switch (P[0]) {
			case 'l':
				g_match_lines = 1;	// Match start of source lines
				break;
			}
			break;
		case 'b':
			g_best_weight = 1;
			break;
		case 'C':
			if (P[2]) {
				P += 2;
			} else {
				if (argc <= i) {
					break;
				}
				P = argv[i++];
			}
			val = atoi(P);
			if (val <= 0) {
				fprintf(stderr, "-C %d not positive\n", val);
				error_exit(1);
			}
			g_top_cols = val;
			break;
		case 'd':
			g_debug_dwarf = 1;
			break;
		case 'D':
			g_debug_code  = 1;
			break;
		case 'e':
			g_match_macros = 1;
			break;
		case 'E':
			if (P[2]) {
				P += 2;
			} else {
				if (argc <= i) {
					break;
				}
				P = argv[i++];
			}
			val = atoi(P);
			if (val < 0) {
				fprintf(stderr, "-E %d is negative\n", val);
				error_exit(1);
			}
			g_hillclimb_timeout = val;
			break;
		case 'f':
			g_use_frame_address = 1;
			break;
		case 'F':
			g_use_temp_address = 1;
			break;
		case 'g':
			if (P[2]) {
				g_clics_file1P = P + 2;
			} else {
				if (argc <= i) {
					break;
				}
				g_clics_file1P = argv[i++];
			}
			break;
		case 'G':
			if (P[2]) {
				g_clics_fileP = P + 2;
			} else {
				if (argc <= i) {
					break;
				}
				g_clics_fileP = argv[i++];
			}
			break;
		case 'h':
			if (P[2]) {
				g_html_dirP = P + 2;
			} else {
				if (argc <= i) {
					break;
				}
				g_html_dirP = argv[i++];
			}
#ifdef WIN32
			CreateDirectoryA(g_html_dirP, 0);
#else
			mkdir(g_html_dirP, 0755);
#endif
			ret = snprintf(g_filename, FILENAME_MAX, "%s/index.html", g_html_dirP);
			if (ret < 0 || FILENAME_MAX <= ret) {
				fprintf(stderr, "index.html filename too long\n");
				error_exit(1);
			}

			indexF  = open_file(g_filename);
			break;
		case 'i':
			if (P[2]) {
				P += 2;
			} else {
				if (argc <= i) {
					break;
				}
				P = argv[i++];
			}
dash_i:		if (stdinF != stdin) {
				fprintf(stderr, "Multiple input files specified\n");
				error_exit(1);
			}
			stdinF = fopen(P, "r");
			if (!stdinF) {
				fprintf(stderr, "Couldn't read %s\n", P);
				error_exit(1);
			}
			break;
		case 'I':
			if (P[2]) {
				P += 2;
			} else {
				if (argc <= i) {
					break;
				}
				P = argv[i++];
			}
			val = atoi(P);
			if (val < 0) {
				fprintf(stderr, "-I/terate option must be non-negative\n");
				error_exit(1);
			}
			g_iterations = val;
			break;
		case 'k':
			g_unlink = 0;
			break;
		case 'l':
			if (P[2]) {
				P += 2;
			} else {
				if (argc <= i) {
					break;
				}
				P = argv[i++];
			}
			val = atoi(P);
			if (val < 1) {
				fprintf(stderr, "-l/ength must be > 0\n");
				error_exit(1);
			}
			g_min_match_sequence = val;
			break;
		case 'L':
			if (P[2]) {
				P += 2;
			} else {
				if (argc <= i) {
					break;
				}
				P = argv[i++];
			}
			val = atoi(P);
			if (val < 1) {
				fprintf(stderr, "-L/ength must be > 0\n");
				error_exit(1);
			}
			g_min_function_sequence = val;
			break;
		case 'm':
			if (P[2]) {
				P += 2;
			} else {
				if (argc <= i) {
					break;
				}
				P = argv[i++];
			}
			val = atoi(P);
			if (val < 0) {
				fprintf(stderr, "-m/ismatch cost must be => 0\n");
				error_exit(1);
			}
			g_mismatch_cost = -val;
			break;			
		case 'M':
			g_ignore_macros = 1;
			break;
		case 'r':
			g_report_c = 0;
			break;
		case 'R':
			g_report_assembler = 0;
			break;
		case 's':
			if (P[2]) {
				P += 2;
			} else {
				if (argc <= i) {
					break;
				}
				P = argv[i++];
			}
			val = atoi(P);
			if (val < 0) {
				fprintf(stderr, "-s/ame cost must be => 0\n");
				error_exit(1);
			}
			g_same_cost = val;
			break;
		case 'S':
			g_skip_compile = 1;
			break;
		case 't':
			if (P[2]) {
				g_ta_fileP = P + 2;
			} else {
				if (argc <= i) {
					break;
				}
				g_ta_fileP = argv[i++];
			}
			break;
		case 'T':
			if (P[2]) {
				P += 2;
			} else {
				if (argc <= i) {
					break;
				}
				P = argv[i++];
			}
			val = atoi(P);
			g_top_clones = val;
			break;
		case 'u':
			if (P[2]) {
				P += 2;
			} else {
				if (argc <= i) {
					break;
				}
				P = argv[i++];
			}
			val = atoi(P);
			if (val < 0) {
				fprintf(stderr, "-Use option must be non-negative\n");
				error_exit(1);
			}
			g_hillclimb = val;
			break;
		case 'v':
			g_verbose = 1;
			break;
#ifdef CORRELATE
		case 'z':
			g_correlate = 1;
			break;
#endif
		default:
			fprintf(stderr, "Unknown option \"%s\"\n\n", P);
			fputs("Options:\n\n"
				"  -a/ll [i/nstructions|l/ines]\n"
				"  -b/estweight\n"
				"  -C/ols <number>\n"
				"  -d/ebug\n"
				"  -D/ebug\n"
				"  -e/nable macro clones\n"
				"  -E/lapse timeout\n"
				"  -f/rame address\n"
				"  -F/rame temp address\n"
				"  -h/tml <directory>\n"
				"  -k/eep assembler\n"
				"  -l/ength_assember <min>\n"
				"  -M/acro detection\n"
				"  -m/ismatch <cost>\n"
				"  -r/emove source reporting\n"
				"  -R/emove assembler reporting\n"
				"  -s/ame <cost>\n"
				"  -S/kip compile\n"
				"  -t/a <file>\n"
				"  -g/ui <file> [-G (a/ssembler|s/ource|b/oth)]\n"
				"  -T/op <number>\n"
				"  -u/se <number> [-I/terate <number>]\n"
				"  -v/erbose\n"
#ifdef CORRELATE
				"  -z\n"
#endif
				, stderr);
			error_exit(1);
	}	}
	g_report = g_report_assembler || g_report_c;
	if (!g_html_dirP) {
		g_report = 0;
	}
	return;
}

static void
print_title(int argc, char *argv[])
{
	int	i;

	fputs( HTML_HEADER "<title>", indexF);
	fprintf(indexF, "ACD Version " VERSION " Compiled " __DATE__ " "  __TIME__ );
	fputs("</title>\n<body>\n", indexF);
	fprintf(indexF, "<h3>Arguments to %s</h3>\n<p>\n", argv[0]);
	for (i = 1; i < argc; ++i) {
		fprintf(indexF,"\"%s\" ", argv[i]);
	}
	if (g_top_clones > 0) {
		fprintf(indexF, "<p>\n<a href=top.html>Top %d clones </a>\n", g_top_clones);
	}
#ifdef CORRELATE
	if (g_correlate > 0) {
		fprintf(indexF, "<p>\n<a href=correlated.html>Correlations</a>\n");
	}
#endif
}

#ifdef WIN32
#define clock_t time_t
#endif

int
main(int argc, char *argv[], char *env[])
{
	char		*stdin_bufferP;
	int			stdin_buffer_lth;
	clock_t		start_clock, read_clock, end_clock;
	long long	total_matches;
	double		percentage;
	sigtableT	*sigtableP;
#ifndef WIN32
	struct		sigaction now;
#endif

/*
#ifndef WIN32
	struct rlimit rl;
	
	// Force a core file to be produced on an exception
	if (getrlimit(RLIMIT_CORE, &rl)) {
		fprintf(stderr, "Can't getrlimit\n");
	} else {
		fprintf(stderr, "getrlimit %ld %ld\n", (long) rl.rlim_cur, (long) rl.rlim_max);
	}
	rl.rlim_cur = RLIM_INFINITY;
	rl.rlim_max = RLIM_INFINITY;
	if (setrlimit(RLIMIT_CORE, &rl)) {
		fprintf(stderr, "Can't setrlimit\n");
	}
#endif
*/

	// Test that abort works
	// abort();
	
	start_clock = time(0);

	g_envPP = env;
	get_environment(argc, argv);
	initialise_detect();
	init_lookup_chars();

	print_title(argc, argv);
	clics_header(argc, argv);
	ta_header(argc, argv);
	fprintf(indexF, "<h3>Processing</h3>\n");

	stdin_buffer_lth = 256;
	stdin_bufferP    = 0;
	while (getline1(&stdin_bufferP, &stdin_buffer_lth, stdinF) >= 0) {
		++g_stdin_lineno;
		// fprintf(stderr, "%d: %s\n", g_stdin_lineno, stdin_bufferP);
		read_assembler(stdin_bufferP);
	}	
	Xfree(stdin_bufferP);
	if (stdinF != stdin) {
		fclose(stdinF);
		stdinF = stdin;
	}

	fprintf(stderr, "\n\nRead %d Compiled %d Failed %d\n", g_read, g_compiled_ok, g_compiled_bad);
	// Now perform clone detection

	read_clock = time(0);

	start_report();
#ifdef CORRELATE
	if (g_correlate) {
		init_data();
	}
#endif

#ifndef WIN32
	memset(&now, 0, sizeof(now));
	now.sa_handler = handler;
#endif

	if (!(setjmp(g_break_address))) {
		for (sigtableP = sigtable; sigtableP->nameP; ++sigtableP) {
#ifdef WIN32
			if (signal(sigtableP->signal, handler) == SIG_ERR) {
#else
			if (sigaction(sigtableP->signal, &now, 0)) {
#endif
				fprintf(stderr, "Error: Can't set SIG%s signal handler\n", sigtableP->nameP);
		}	}
		detect();
	}
	finalize_report();

	end_clock = time(0);
	total_matches = g_original_matches + g_extra_matches - g_cancelled_matches;
	percentage    = ((double) g_extra_matches)/((double) total_matches);
	percentage   *= 100.0;

	fprintf(indexF, "</table>\n");
	if (g_signalP) {
		fprintf(indexF, "<p><font color=red>**EXECUTION INTERRUPTED BY **SIG%s**</font>\n", g_signalP);
	}
	fprintf(indexF, "<p>\n<table>\n<tr><td>Start</td><td>%s</td></tr>\n", 	ctime(&start_clock));
	fprintf(indexF, "<tr><td>Read</td><td align=right>%s</td></tr>\n", 	ctime(&read_clock));
	fprintf(indexF, "<tr><td>End</td><td align=right>%s</td></tr>\n", 	ctime(&end_clock));
#ifndef WIN32
	fprintf(indexF, "<tr><td>Memory</td><td align=right>%ld</td></tr>\n", (long) sbrk(0));
#endif
	fprintf(indexF, "<tr><td>Instructions</td><td align=right>%lld</td></tr>\n", (long long) g_assembler_instructions);
	if (!g_hillclimb) {
		fprintf(indexF, "<tr><td>Detect Elapse</td><td align=right>%d</td></tr>\n", (int) (end_clock-read_clock));
	} else {
		fprintf(indexF, "<tr><td>Greedy Elapse</td><td align=right>%d</td></tr>\n", (int) (end_clock-read_clock-g_hillclimb_elapse));
		fprintf(indexF, "<tr><td>HillClimb Elapse</td><td align=right>%d</td></tr>\n", (int) g_hillclimb_elapse);
	}
	fprintf(indexF, "<tr><td>Sources</td><td align=right>%d</td></tr>\n", g_read);
	fprintf(indexF, "<tr><td>Compiled</td><td align=right>%d</td></tr>\n", g_compiled_ok);
	if (g_compiled_bad) {
		fprintf(indexF, "<tr><td>Uncompilable</td><td align=right>%d</td></tr>\n", g_compiled_bad);
	}
	fprintf(indexF, "<tr><td>Functions</td><td align=right>%d</td></tr>\n", g_functions);
	fprintf(indexF, "<tr><td>Comparisons</td><td align=right>%lld</td></tr>\n", g_comparisons);
	fprintf(indexF, "<tr><td>Original Matches Detected</td><td align=right>%lld</td></tr>\n", g_original_matches);
	if (g_hillclimb > 0) {
		fprintf(indexF, "<tr><td>Hillclimbing Added</td><td align=right>%lld</td></tr>\n", g_extra_matches);
	}
	fprintf(indexF, "<tr><td>Jump Matches Cancelled</td><td align=right>%lld</td></tr>\n", g_cancelled_matches);
	fprintf(indexF, "<tr><td>Matches detected</td><td align=right>%lld</td></tr>\n", total_matches);
	fprintf(indexF, "<tr><td>Cloned sources detected</td><td align=right>%d</td></tr>\n", g_source_cnt);
	fprintf(indexF, "<tr><td>Clones detected</td><td align=right>%d</td></tr>\n", g_clone_cnt);
	if (g_hillclimb > 0) {
		fprintf(indexF, "<tr><td>Max Hillclimbing Blocks</td><td align=right>%d</td></tr>\n", g_max_blocks_seen);
		fprintf(indexF, "<tr><td>Max Hillclimbing Iterations</td><td align=right>%d</td></tr>\n", g_max_iterations_seen+1);
		fprintf(indexF, "<tr><td>Hillclimbing Percentage</td><td align=right>%f%%</td></tr>\n", percentage );
	}

	fprintf(indexF, "</table>\n<p>\n<i>End of Report</i>\n</body>\n</html>\n");

	clics_trailer();
	ta_trailer();
	fprintf(stderr, "Done secs=%d clones=%d matches=%lld",
				(int) (end_clock - read_clock),
				g_clone_cnt,
				total_matches);
	if (g_hillclimb >= 0) {
		fprintf(stderr, " -u %d extra=%lld iterations=%d\n", 
				g_hillclimb,
				g_extra_matches,
				g_max_iterations_seen+1);
	}
	fprintf(stderr, "\n");
	return(0);
}
