#ifdef WIN32
#define _CRT_SECURE_NO_WARNINGS
#endif

#ifndef WIN32
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "xmalloc.h"
#include "util.h"
#include "object.h"
#include "label.h"
#include "location.h"
#include "dwarf.h"

extern int g_debug_dwarf;

/* Process debug information in Assembler as per Dwalf 2 & 3 specification */

#if 0
static void
trap(void)
{
}
#endif

static dw_formE	op_args_addr[]  = { DW_FORM_addr,DW_FORM_null};
static dw_formE	op_args_int8[]  = { DW_FORM_data1,DW_FORM_null};
static dw_formE	op_args_int16[] = { DW_FORM_data2,DW_FORM_null};
static dw_formE	op_args_int32[] = { DW_FORM_data4,DW_FORM_null};
static dw_formE	op_args_int64[] = { DW_FORM_data8,DW_FORM_null};
static dw_formE	op_args_udata[] = { DW_FORM_udata,DW_FORM_null};
static dw_formE	op_args_sdata[] = { DW_FORM_sdata,DW_FORM_null};
static dw_formE	op_args_udata_sdata[] = {
	DW_FORM_udata, DW_FORM_sdata, DW_FORM_null};
static dw_formE	op_args_udata_udata[] = {
	DW_FORM_udata,DW_FORM_udata, DW_FORM_null};
static dw_formE	op_args_indirect[] = { DW_FORM_indirect, DW_FORM_null};

typedef struct {
	const char	*nameP;
	dw_formE 	*argsP;
} op_tableT;

static op_tableT op_table[] = {
/* 0x00 */ 	 {0			,0}
/* 0x01 */	,{0			,0}
/* 0x02 */ 	,{0			,0}
/* 0x03 */	,{"addr"	,op_args_addr}	// constant address (size target specific)
/* 0x04 */ 	,{0			,0}
/* 0x05 */ 	,{0			,0}
/* 0x06 */	,{"deref"	,0}
/* 0x07 */ 	,{0			,0}
/* 0x08 */	,{"const1u"	,op_args_int8}	// 1-byte constant
/* 0x09 */	,{"const1s"	,op_args_int8} 	// 1-byte constant
/* 0x0a */	,{"const2u"	,op_args_int16} // 2-byte constant
/* 0x0b */	,{"const2s"	,op_args_int16}	// 2-byte constant
/* 0x0c */	,{"const4u"	,op_args_int32}	// 4-byte constant
/* 0x0d */	,{"const4s"	,op_args_int32}	// 4-byte constant
/* 0x0e */	,{"const8u"	,op_args_int64}	// 8-byte constant
/* 0x0f */	,{"const8s"	,op_args_int64}	// 8-byte constant
/* 0x10 */	,{"constu"	,op_args_udata}	// ULEB128 constant
/* 0x11 */	,{"consts"	,op_args_sdata}	// SLEB128 constant
/* 0x12 */	,{"dup"		,0} 
/* 0x13 */	,{"drop"	,0}
/* 0x14 */	,{"over"	,0}
/* 0x15 */	,{"pick"	,op_args_int8}	// 1-byte stack index
/* 0x16 */	,{"swap"	,0}
/* 0x17 */	,{"rot"		,0}
/* 0x18 */	,{"xderef"	,0}
/* 0x19 */	,{"abs"		,0}
/* 0x1a */	,{"and"		,0}
/* 0x1b */	,{"div"		,0}
/* 0x1c */	,{"minus"	,0}
/* 0x1d */	,{"mod"		,0}
/* 0x1e */	,{"mul"		,0}
/* 0x1f */	,{"neg"		,0}
/* 0x20 */	,{"not"		,0}
/* 0x21 */ ,{"or"		,0}
/* 0x22 */ ,{"plus"		,0}
/* 0x23 */ ,{"plus_uconst"	,op_args_udata}	// ULEB128 addend
/* 0x24 */ ,{"shl"		,0}
/* 0x25 */ ,{"shr"		,0}
/* 0x26 */ ,{"shra"		,0}
/* 0x27 */ ,{"xor"		,0}
/* 0x28 */ ,{"bra"		,op_args_int16}	// signed 2-byte constant
/* 0x29 */ ,{"eq"		,0}
/* 0x2a */ ,{"ge"		,0}
/* 0x2b */ ,{"gt"		,0}
/* 0x2c */ ,{"le"		,0}
/* 0x2d */ ,{"lt"		,0}
/* 0x2e */ ,{"ne"		,0}
/* 0x2f */ ,{"skip"		,op_args_int16}	// signed 2-byte constant
/* 0x30 */ ,{"lit0"		,0}
/* 0x31 */ ,{"lit1"		,0}
/* 0x32 */ ,{"lit2"		,0}
/* 0x33 */ ,{"lit3"		,0}
/* 0x34 */ ,{"lit4"		,0}
/* 0x35 */ ,{"lit5"		,0}
/* 0x36 */ ,{"lit6"		,0}
/* 0x37 */ ,{"lit7"		,0}
/* 0x38 */ ,{"lit8"		,0}
/* 0x39 */ ,{"lit9"		,0}
/* 0x3a */ ,{"lit10"	,0}
/* 0x3b */ ,{"lit11"	,0}
/* 0x3c */ ,{"lit12"	,0}
/* 0x3d */ ,{"lit13"	,0}
/* 0x3e */ ,{"lit14"	,0}
/* 0x3f */ ,{"lit15"	,0}
/* 0x40 */ ,{"lit16"	,0}
/* 0x41 */ ,{"lit17"	,0}
/* 0x42 */ ,{"lit18"	,0}
/* 0x43 */ ,{"lit19"	,0}
/* 0x44 */ ,{"lit20"	,0}
/* 0x45 */ ,{"lit21"	,0}
/* 0x46 */ ,{"lit22"	,0}
/* 0x47 */ ,{"lit23"	,0}
/* 0x48 */ ,{"lit24"	,0}
/* 0x49 */ ,{"lit25"	,0}
/* 0x4a */ ,{"lit26"	,0}
/* 0x4b */ ,{"lit27"	,0}
/* 0x4c */ ,{"lit28"	,0}
/* 0x4d */ ,{"lit29"	,0}
/* 0x4e */ ,{"lit30"	,0}
/* 0x4f */ ,{"lit31"	,0} // literals 0..31 = (DW_OP_lit0 + literal)
/* 0x50 */ ,{"reg0"		,0}
/* 0x51 */ ,{"reg1"		,0}
/* 0x52 */ ,{"reg2"		,0}
/* 0x53 */ ,{"reg3"		,0}
/* 0x54 */ ,{"reg4"		,0}
/* 0x55 */ ,{"reg5"		,0}
/* 0x56 */ ,{"reg6"		,0}
/* 0x57 */ ,{"reg7"		,0}
/* 0x58 */ ,{"reg8"		,0}
/* 0x59 */ ,{"reg9"		,0}
/* 0x5a */ ,{"reg10"	,0}
/* 0x5b */ ,{"reg11"	,0}
/* 0x5c */ ,{"reg12"	,0}
/* 0x5d */ ,{"reg13"	,0}
/* 0x5e */ ,{"reg14"	,0}
/* 0x5f */ ,{"reg15"	,0}
/* 0x60 */ ,{"reg16"	,0}
/* 0x61 */ ,{"reg17"	,0}
/* 0x62 */ ,{"reg18"	,0}
/* 0x63 */ ,{"reg19"	,0}
/* 0x64 */ ,{"reg20"	,0}
/* 0x65 */ ,{"reg21"	,0}
/* 0x66 */ ,{"reg22"	,0}
/* 0x67 */ ,{"reg23"	,0}
/* 0x68 */ ,{"reg24"	,0}
/* 0x69 */ ,{"reg25"	,0}
/* 0x6a */ ,{"reg26"	,0}
/* 0x6b */ ,{"reg27"	,0}
/* 0x6c */ ,{"reg28"	,0}
/* 0x6d */ ,{"reg29"	,0}
/* 0x6e */ ,{"reg30"	,0}
/* 0x6f */ ,{"reg31"	,0} // reg 0..31 = (DW_OP_reg0 + regnum)
/* 0x70 */ ,{"breg0"	,op_args_sdata}
/* 0x71 */ ,{"breg1"	,op_args_sdata}
/* 0x72 */ ,{"breg2"	,op_args_sdata}
/* 0x73 */ ,{"breg3"	,op_args_sdata}
/* 0x74 */ ,{"breg4"	,op_args_sdata}
/* 0x75 */ ,{"breg5"	,op_args_sdata}
/* 0x76 */ ,{"breg6"	,op_args_sdata}
/* 0x77 */ ,{"breg7"	,op_args_sdata}
/* 0x78 */ ,{"breg8"	,op_args_sdata}
/* 0x79 */ ,{"breg9"	,op_args_sdata}
/* 0x7a */ ,{"breg10" 	,op_args_sdata}
/* 0x7b */ ,{"breg11" 	,op_args_sdata}
/* 0x7c */ ,{"breg12" 	,op_args_sdata}
/* 0x7d */ ,{"breg13" 	,op_args_sdata}
/* 0x7e */ ,{"breg14" 	,op_args_sdata}
/* 0x7f */ ,{"breg15" 	,op_args_sdata}
/* 0x80 */ ,{"breg16" 	,op_args_sdata}
/* 0x81 */ ,{"breg17" 	,op_args_sdata}
/* 0x82 */ ,{"breg18" 	,op_args_sdata}
/* 0x83 */ ,{"breg19" 	,op_args_sdata}
/* 0x84 */ ,{"breg20" 	,op_args_sdata}
/* 0x85 */ ,{"breg21" 	,op_args_sdata}
/* 0x86 */ ,{"breg22" 	,op_args_sdata}
/* 0x87 */ ,{"breg23" 	,op_args_sdata}
/* 0x88 */ ,{"breg24" 	,op_args_sdata}
/* 0x89 */ ,{"breg25" 	,op_args_sdata}
/* 0x8a */ ,{"breg26" 	,op_args_sdata}
/* 0x8b */ ,{"breg27" 	,op_args_sdata}
/* 0x8c */ ,{"breg28" 	,op_args_sdata}
/* 0x8d */ ,{"breg29" 	,op_args_sdata}
/* 0x8e */ ,{"breg30" 	,op_args_sdata}
/* 0x8f */ ,{"breg31" 	,op_args_sdata} // SLEB128 offset base register 0..31 = (DW_OP_breg0 + regnum)
/* 0x90 */ ,{"regx"		,op_args_udata} // ULEB128 register
/* 0x91 */ ,{"fbreg"	,op_args_sdata} // SLEB128 offset
/* 0x92 */ ,{"bregx"	,op_args_udata_sdata} // ULEB128 register followed by SLEB128 offset
/* 0x93 */ ,{"piece"	,op_args_udata}	// ULEB128 size of piece addressed
/* 0x94 */ ,{"deref_size"	,op_args_int8}	// 1-byte size of data retrieved
/* 0x95 */ ,{"xderef_size"	,op_args_int8}	// 1-byte size of data retrieved
/* 0x96 */ ,{"nop"		,0}
/* 0x97 */ ,{"push_object_address"	,0}
/* 0x98 */ ,{"call2"	,op_args_int16}	// 2-byte offset of DIE
/* 0x99 */ ,{"call4"	,op_args_int32}	// 4-byte offset of DIE
/* 0x9a */ ,{"call_ref"	,op_args_indirect}	// 4- or 8-byte offset of DIE
/* 0x9b */ ,{"form_tls_address"	,0}
/* 0x9c */ ,{"call_frame_cfa"	,0}
/* 0x9d */ ,{"bit_piece",op_args_udata_udata}
};

static const char *
get_form_name(dw_formE form)
{
	static const char *name[] = {
/* 0x00 */	"null",
/* 0x01 */	"addr",
/* 0x02 */	0,
/* 0x03 */	"block2",
/* 0x04 */	"block4",
/* 0x05 */	"data2",
/* 0x06 */	"data4",
/* 0x07 */	"data8",
/* 0x08 */	"string",
/* 0x09 */	"block",
/* 0x0a */	"block1",
/* 0x0b */	"data1",
/* 0x0c */	"flag",
/* 0x0d */	"sdata",
/* 0x0e */	"strp",
/* 0x0f */	"udata",
/* 0x10 */	"ref_addr",
/* 0x11 */	"ref1",
/* 0x12 */	"ref2",
/* 0x13 */	"ref4",
/* 0x14 */	"ref8",
/* 0x15 */	"ref_udata",
/* 0x16 */	"indirect"
	};

	assert((sizeof(name)/sizeof(*name)) == 0x17);
	if (form < 0 || form > 0x16) {
		return 0;
	}
	return name[form];
}

#if 0

// Code to test that signed shift works correctly

#include <stdio.h>
int
main(int argc, char **argv)
{
	long long i;

	union {
		long long			k;
		unsigned long long  j;
	} val;

	i       = -2;
	val.k   = i;
	i     >>= 1;
	val.j >>= 1;

	printf("0x%16llx %lld 0x%16llx %lld\n", i, i, val.j, val.j);
	return(0);
}
#endif

static int g_offset;

static void
encode_uleb128(uint64T value, char **bufferPP, int load)
{
	char	*bufferP;
	uint8T	byte;

	bufferP = *bufferPP;
	do {
		byte    = value & 0x7F;	// Low order 7 bits of value
		value >>= 7;
		if (value != 0) { 		// more bytes to come
			byte |= 0x80;		// Set high bit of byte
		}
		if (load) {
			*bufferP = byte;
		}
		++bufferP;
	} while (value);
	*bufferPP = bufferP;
}

static uint64T
decode_uleb128(uint8T **bufferPP)
{
	uint8T			*bufferP;
	uint64T			ret, temp;
	int				shift;
	unsigned int	c;


	ret     = 0;
	shift   = 0;
	bufferP = *bufferPP;

	for (;;) {
		c = *bufferP++;
		assert(shift < 64);
		temp   = (c & 0x7F);
		temp <<= shift;
		ret   |= temp;
		if (!(c & 0x80)) {
			break;
		}
		shift += 7;
	}
	*bufferPP = bufferP;
	return(ret);
}

static void
encode_sleb128(sint64T value, char **bufferPP, int load)
{
	char	*bufferP;
	char	byte;
	int		more;

	bufferP = *bufferPP;
	more    = 1;

	do {
		byte    = value & 0x7F;	// low order 7 bits of value
		value >>= 7;
		if ((value ==  0 && !(byte & 0x40)) ||
		    (value == -1 &&  (byte & 0x40))) {
			more = 0;
		} else {
			byte |= 0x80;
		}
		if (load) {
			*bufferP = byte;
		}
		++bufferP;
	} while (more);
	*bufferPP = bufferP;
}

static sint64T
decode_sleb128(uint8T **bufferPP)
{
	uint8T	*bufferP;
	sint64T	ret;
	uint64T	temp;
	int		shift;
	uint32T	c;


	ret     = 0;
	shift   = 0;
	bufferP = *bufferPP;

	do {
		c = *bufferP++;
		assert(shift < 64);
		temp   = (c & 0x7F);
		temp <<= shift;
		ret   |= temp;
		shift += 7;
	} while (c & 0x80);

	if ((shift < 64) && (c & 0x40)) {
		temp   = 1;
		temp <<= shift;
		ret   |= -((sint64T) temp);
	}
	*bufferPP = bufferP;
	return(ret);
}

#if 0
// Code to test encode and decode

int
main(int argc, char **argv)
{
	unsigned long long uint64_1, uint64_2;
	signed long long   sint64_1, sint64_2;
	char		 buffer[24];
	char		*bufferP, *buffer1P;

#if 0
	uint64_1 = 0;
	do {
		bufferP = buffer;
		encode_uleb128(uint64_1, &bufferP ,1);
		buffer1P = buffer;
		uint64_2 = decode_uleb128(&buffer1P);
		assert(uint64_1 == uint64_2);
		assert(bufferP == buffer1P);
		++uint64_1;
	} while (uint64_1 <= 0xffffffff);
#endif


	sint64_1 = 0xffffffff80000000ll;
	do {
		bufferP = buffer;
		if (!(sint64_1 & 0xFFFFFFll)) {
			printf("sleb128: 0x%16llx\n", sint64_1);
		}
		encode_sleb128(sint64_1, &bufferP ,1);
		buffer1P = buffer;
		sint64_2 = decode_sleb128(&buffer1P);
		if (sint64_1 != sint64_2) {
			printf("sleb128: 0x%16llx 0x%16llx\n", sint64_1, sint64_2);
			assert(sint64_1 == sint64_2);
		}
		assert(bufferP == buffer1P);
		++sint64_1;
	} while (sint64_1 <= 0xffffffffll);


	return(0);
}
#endif

static void
print_ops(int lth, uint8T *blockP)
{
	uint8T		*P, *endP;
	const char 	*nameP;
	uint32T		code;
	dw_formE	*formP, form;
	uint8T		uint8;
	uint16T		uint16;
	uint32T		uint32;
	urefT		uref;
	uint64T		uint64;
	sint64T		sint64;

	assert((sizeof(op_table)/sizeof(*op_table)) == 0x9e);
	P = blockP + sizeof(uint32T);
	for (endP = P + lth; P < endP; ) {
		code = *P++;
		assert(code <= 0x9d);
		nameP = op_table[code].nameP;
		assert(nameP);
		printf(" [%s", nameP);
		formP = op_table[code].argsP;
		if (formP) {
			for (; (form = *formP) != DW_FORM_null; ++formP) {
				printf(" %s", get_form_name(form));
				switch (form) {
				case DW_FORM_data1:
					memcpy(&uint8, P, sizeof(uint8T));
					P += sizeof(uint8T);
					printf(" %d", (int) uint8);
					break;
				case DW_FORM_data2:
					memcpy(&uint16, P, sizeof(uint16T));
					P += sizeof(uint16T);
					printf(" %d", (int) uint16);
					break;
				case DW_FORM_data4:
				case DW_FORM_indirect:
					memcpy(&uint32, P, sizeof(uint32T));
					P += sizeof(uint32T);
					printf(" %d", uint32);
					break;
				case DW_FORM_addr:
					memcpy(&uref, P, sizeof(urefT));
					P += sizeof(urefT);
#if DWARF64
					printf(" %lld", uref);
#else
					printf(" %u", uref);
#endif
					break;

				case DW_FORM_data8:
					memcpy(&uint64, P, sizeof(uint64T));
					P += sizeof(uint64T);
					printf(" %lld", uint64);
					break;
				case DW_FORM_udata:
					uint64 = decode_uleb128(&P);
					printf(" %llu", uint64);
					break;
				case DW_FORM_sdata:
					sint64 = decode_sleb128(&P);
					printf(" %lld", sint64);
					break;
				default:
					assert(0);
		}	}	}
		printf("]");
	}
	assert(P == endP);
}

static const char *
get_node_name(dw_tagE tag)
{
	static const char *name[] = {
/* 0x00 */	"null",
/* 0x01 */	"array_type",
/* 0x02 */	"class_type",
/* 0x03 */	"entry_point",
/* 0x04 */	"enumeration_type",
/* 0x05 */	"formal_parameter",
/* 0x06 */	0,
/* 0x07 */	0,
/* 0x08 */	"imported_declaration",
/* 0x09 */	0,
/* 0x0a */	"label",
/* 0x0b */	"lexical_block",
/* 0x0c */	0,
/* 0x0d */	"member",
/* 0x0e */	0,
/* 0x0f */	"pointer_type",
/* 0x10 */	"reference_type",
/* 0x11 */	"compile_unit",
/* 0x12 */	"string_type",
/* 0x13 */	"structure_type",
/* 0x14 */	0,
/* 0x15 */	"subroutine_type",
/* 0x16 */	"typedef",
/* 0x17 */	"union_type",
/* 0x18 */	"unspecified_parameters",
/* 0x19 */	"variant",
/* 0x1a */	"common_block",
/* 0x1b */	"common_inclusion",
/* 0x1c */	"inheritance",
/* 0x1d */	"inlined_subroutine",
/* 0x1e */	"module",
/* 0x1f */	"ptr_to_member_type",
/* 0x20 */	"set_type",
/* 0x21 */	"subrange_type",
/* 0x22 */	"with_stmt",
/* 0x23 */	"access_declaration",
/* 0x24 */	"base_type",
/* 0x25 */	"catch_block",
/* 0x26 */	"const_type",
/* 0x27 */	"constant",
/* 0x28 */	"enumerator",
/* 0x29 */	"file_type",
/* 0x2a */	"friend",
/* 0x2b */	"namelist",
/* 0x2c */	"namelist_item",
/* 0x2d */	"packed_type",
/* 0x2e */	"subprogram",
/* 0x2f */	"template_type_parameter",
/* 0x30 */	"template_value_parameter",
/* 0x31 */	"thrown_type",
/* 0x32 */	"try_block",
/* 0x33 */	"variant_part",
/* 0x34 */	"variable",
/* 0x35 */	"volatile_type",
/* 0x36 */	"dwarf_procedure",
/* 0x37 */	"restrict_type",
/* 0x38 */	"interface_type",
/* 0x39 */	"namespace",
/* 0x3a */	"imported_module",
/* 0x3b */	"unspecified_type",
/* 0x3c */	"partial_unit",
/* 0x3d */	"imported_unit",
/* 0x3e */	0,
/* 0x3f */	"condition",
/* 0x40 */	"shared_type"
	};

	assert((sizeof(name)/sizeof(*name)) == 0x41);
	if (tag < 0 || tag > 0x40) {
		return 0;
	}
	return name[tag];
}

static const char *
get_attribute_name(dw_atE attribute)
{
	static const char *name[] = {
/* 0x00 */	0,
/* 0x01 */	"sibling",
/* 0x02 */	"location",
/* 0x03 */	"name",
/* 0x04 */	0,
/* 0x05 */	0,
/* 0x06 */	0,
/* 0x07 */	0,
/* 0x08 */	0,
/* 0x09 */	"ordering",
/* 0x0a */	0,
/* 0x0b */	"byte_size",
/* 0x0c */	"bit_offset",
/* 0x0d */	"bit_size",
/* 0x0e */	0,
/* 0x0f */	0,
/* 0x10 */	"stmt_list",
/* 0x11 */	"low_pc",
/* 0x12 */	"high_pc",
/* 0x13 */	"language",
/* 0x14 */	0,
/* 0x15 */	"discr",
/* 0x16 */	"discr_value",
/* 0x17 */	"visibility",
/* 0x18 */	"import",
/* 0x19 */	"string_length",
/* 0x1a */	"common_reference",
/* 0x1b */	"comp_dir",
/* 0x1c */	"const_value",
/* 0x1d */	"containing_type",
/* 0x1e */	"default_value",
/* 0x1f */	0,
/* 0x20 */	"inline",
/* 0x21 */	"is_optional",
/* 0x22 */	"lower_bound",
/* 0x23 */	0,
/* 0x24 */	0,
/* 0x25 */	"producer	",
/* 0x26 */	0,
/* 0x27 */	"prototyped",
/* 0x28 */	0,
/* 0x29 */	0,
/* 0x2a */	"return_addr",
/* 0x2b */	0,
/* 0x2c */	"start_scope",
/* 0x2d */	0,
/* 0x2e */	"bit_stride",
/* 0x2f */	"upper_bound",
/* 0x30 */	0,
/* 0x31 */	"abstract_origin",
/* 0x32 */	"accessibility",
/* 0x33 */	"address_class",
/* 0x34 */	"artificial",
/* 0x35 */	"base_types",
/* 0x36 */	"calling_convention",
/* 0x37 */	"count",
/* 0x38 */	"data_member_location",
/* 0x39 */	"decl_column",
/* 0x3a */	"decl_file",
/* 0x3b */	"decl_line",
/* 0x3c */	"declaration",
/* 0x3d */	"discr_list",
/* 0x3e */	"encoding",
/* 0x3f */	"external",
/* 0x40 */	"frame_base",
/* 0x41 */	"friend",
/* 0x42 */	"identifier_case",
/* 0x43 */	"macro_info",
/* 0x44 */	"namelist_item",
/* 0x45 */	"priority",
/* 0x46 */	"segment",
/* 0x47 */	"specification",
/* 0x48 */	"static_link",
/* 0x49 */	"type",
/* 0x4a */	"use_location",
/* 0x4b */	"variable_parameter",
/* 0x4c */	"virtuality",
/* 0x4d */	"vtable_elem_location",
/* 0x4e */	"allocated	",
/* 0x4f */	"associated",
/* 0x50 */	"data_location",
/* 0x51 */	"byte_stride",
/* 0x52 */	"entry_pc",
/* 0x53 */	"use_UTF8",
/* 0x54 */	"extension",
/* 0x55 */	"ranges",
/* 0x56 */	"trampoline",
/* 0x57 */	"call_column",
/* 0x58 */	"call_file",
/* 0x59 */	"call_line",
/* 0x5a */	"description",
/* 0x5b */	"binary_scale",
/* 0x5c */	"decimal_scale",
/* 0x5d */	"small",
/* 0x5e */	"decimal_sign",
/* 0x5f */	"digit_count",
/* 0x60 */	"picture_string",
/* 0x61 */	"mutable",
/* 0x62 */	"threads_scaled",
/* 0x63 */	"explicit",
/* 0x64 */	"object_pointer",
/* 0x65 */	"endianity",
/* 0x66 */	"elemental",
/* 0x67 */	"pure",
/* 0x68 */	"recursive"
};

	if (attribute == DW_AT_signature) {
		return("signature");
	}
	assert((sizeof(name)/sizeof(*name)) == 0x69);
	if (attribute < 0 || attribute > 0x68) {
		return(0);
	}
	return name[attribute];
}

static int
load_unsigned(char **atPP, uint64T *valueP)
{
	char	*atP;
	uint64T	ret;
	char	c;
	
	atP = *atPP;

	ret = 0;
	if ((c = *atP) == '0') {
		if ((c = atP[1]) == 'x' || c == 'X') {
			for (++atP ;;) {
				c = *(++atP);
				if (c >= '0' && c <= '9') {
					c -= '0';
				} else if ( c >= 'a' && c <= 'f') {
					c += 10 - 'a';
				} else if (c >= 'A' && c <= 'F') {
					c += 10 - 'A';
				} else {
					goto done;
				}
				ret = ret * 16 + c;
		}	}
	} else {
		if (c < '0' || c > '9') {
			goto fail;
	}	}

	do {
		ret = ret * 10 + c - '0';
		c   = *(++atP);
	} while (c >= '0' && c <= '9');

done:
	*valueP = ret;
	*atPP   = atP;
	return(1);
fail:
	assert(0);
	return(0);
}

static int
load_signed(char **atPP, sint64T *valueP)
{
	char	*atP;
	sint64T	ret;
	int		sign;
	char	c, c1;
	
	sign = 0;
	atP  = *atPP;

	ret = 0;
	if ((c = *atP) == '0') {
		if ((c1 = atP[1]) == 'x' || c1 == 'X') {
			for (++atP ;;) {
				c = *(++atP);
				if (c >= '0' && c <= '9') {
					c -= '0';
				} else if ( c >= 'a' && c <= 'f') {
					c += 10 - 'a';
				} else if (c >= 'A' && c <= 'F') {
					c += 10 - 'A';
				} else {
					goto done;
				}
				ret = ret * 16 + c;
		}	}
	} else {
		if (c == '-') {
			sign = 1;
			c = *(++atP);
		}
		if (c < '0' || c > '9') {
			goto fail;
	}	}

	do {
		ret = ret * 10 + c - '0';
		c   = *(++atP);
	} while (c >= '0' && c <= '9');

	if (sign) {
		ret = -ret;
	}

done:
	*valueP = ret;
	*atPP   = atP;
	return(1);
fail:
	assert(0);
	return(0);
}

static int
load_uint8(char **atPP, uint8T *valueP)
{
	char 	*atP;
	uint64T	uint64;

	atP = *atPP;
	if (strncmp(atP, " .byte ", 7)) {
		goto fail;
	}
	atP += 7;
	if (!load_unsigned(&atP, &uint64)) {
		goto fail;
	}
	if (uint64 > 0xff) {
		goto fail;
	}
	if (*atP++ != '\n') {
		goto fail;
	}
	*valueP = (uint8T) uint64;
	*atPP   = atP;
	++g_offset;
	return(1);
fail:
	assert(0);
	return(0);
}

static int
load_uint16(char **atPP, uint16T *valueP)
{
	char 	*atP;
	uint64T	uint64;

	atP = *atPP;
	if (strncmp(atP, " .value ", 8)) {
		goto fail;
	}
	atP += 8;
	if (!load_unsigned(&atP, &uint64)) {
		goto fail;
	}
	if (uint64 > 0xffff) {
		goto fail;
	}
	if (*atP++ != '\n') {
		goto fail;
	}
	*valueP   = (uint16T) uint64;
	*atPP     = atP;
	g_offset += 2;
	return(1);
fail:
	assert(0);
	return(0);
}

int
Cdwarf::load_label(char **atPP, urefT *valueP)
{
	char	*atP, *P;
	char	*P1;
	char	c;
	labelT	*labelP;

	atP = P = *atPP;
	for (; ; ++P) {
		switch (c = *P) {
		case 0:
		case ' ':
		case ',':
		case '\n':
			break;
		default:
			continue;
		}
		break;
	}
	*P = 0;

	for (labelP = m_labelsP; (P1 = labelP->labelP); ++labelP) {
		if (!strcmp(atP, P1)) {
			*valueP = labelP->lineno;
			*P    = c;
			*atPP = P;
			return(1);
	}	}
	*P    = c;
	*atPP = P;
	// Unknown label (hopefully something we don't care about)
	*valueP = 0xFFFF1111;
	return(1);
}

int
Cdwarf::load_uint32(char **atPP, uint32T *valueP)
{
	char 	*atP;
	uint32T	uint32;
	uint64T	uint64;
	urefT	uref;
	char	c;

	atP = *atPP;
	if (strncmp(atP, " .long ", 7)) {
		goto fail;
	}
	atP += 7;
	if ((c = *atP) >= '0' && c <= '9') {
		if (!load_unsigned(&atP, &uint64)) {
			goto fail;
		}
		if (uint64 > 0xffffffff) {
			goto fail;
		}
		uint32 = (uint32T) uint64;
	} else {
		if (!load_label(&atP, &uref)) {
			goto fail;
		}
		if (uref > 0xffffffff) {
			goto fail;
		}
		uint32 = uref;
	}
	if (*atP++ != '\n') {
		goto fail;
	}
	*valueP   = uint32;
	*atPP     = atP;
	g_offset += sizeof(uint32T);
	return(1);
fail:
	assert(0);
	return(0);
}

int
Cdwarf::load_uint64(char **atPP, uint64T *valueP)
{
	char 	*atP;
	uint64T	uint64;
	urefT	uref;
	char	c;

	atP = *atPP;
	if (strncmp(atP, " .quad ", 7)) {
		goto fail;
	}
	atP += 7;
	if ((c = *atP) >= '0' && c <= '9') {
		if (!load_unsigned(&atP, &uint64)) {
			goto fail;
		}
	} else {
		if (!load_label(&atP, &uref)) {
			goto fail;
		}
		uint64 = uref;
	}
	if (*atP++ != '\n') {
		goto fail;
	}
	*valueP   = uint64;
	*atPP     = atP;
	g_offset += sizeof(uint64T);
	return(1);
fail:
	assert(0);
	return(0);
}

static const uint64T max_uleb128[] = {
	0x000000000000007fll,
	0x0000000000003fffll,
	0x00000000001fffffll,
	0x000000000fffffffll,
	0x00000007ffffffffll,
	0x000003ffffffffffll,
	0x0001ffffffffffffll,
	0x00ffffffffffffffll,
	0x7fffffffffffffffll,
	0xffffffffFFFFFFFFll
};

static int
load_uleb128(char **atPP, uint32T *valueP)
{
	char 			*atP;
	uint64T			uint64;
	const uint64T	*maxP;

	atP = *atPP;
	if (strncmp(atP, " .uleb128 ", 10)) {
		if (!strncmp(atP, " .byte ", 7)) {
			atP += 7;
			if (!load_unsigned(&atP, &uint64)) {
				goto fail;
			}
			assert(!uint64);
			goto done;
		}
		goto fail;
	}
	atP += 10;
	if (!load_unsigned(&atP, &uint64)) {
		goto fail;
	}
done:
	if (uint64 > 0xffffffff) {
		goto fail;
	}
	*valueP = (uint32T) uint64;
	if (*atP++ != '\n') {
		goto fail;
	}
	for (maxP = max_uleb128; ; ++maxP) {
		++g_offset;
		if (uint64 <= *maxP) {
			break;
	}	}
	*atPP = atP;
	return(1);
fail:
	assert(0);
	return(0);
}

static const sint64T max_sleb128[] = {
	0x000000000000003fll,
	0x0000000000001fffll,
	0x00000000000fffffll,
	0x0000000007ffffffll,
	0x00000003ffffffffll,
	0x000001ffffffffffll,
	0x0000ffffffffffffll,
	0x007fffffffffffffll,
	0x3fffffffffffffffll,
	0x7fffffffFFFFFFFFll
};

static const sint64T min_sleb128[] = {
	0xffffffffffffffc0ll,
	0xffffffffffffe000ll,
	0xfffffffffff00000ll,
	0xfffffffff8000000ll,
	0xfffffffc00000000ll,
	0xfffffe0000000000ll,
	0xffff000000000000ll,
	0xff80000000000000ll,
	0xc000000000000000ll,
	0x8000000000000000ll
};

static int
load_sleb128(char **atPP, sint32T *valueP)
{
	char 			*atP;
	sint64T			sint64;
	const sint64T	*maxP;

	atP = *atPP;
	if (strncmp(atP, " .sleb128 ", 10)) {
/*
		if (!strncmp(atP, " .byte ", 7)) {
			atP += 7;
			if (!load_unsigned(&atP, valueP)) {
				goto fail;
			}
			assert(!*valueP);
			return(1);
		}
 */
		goto fail;
	}
	atP += 10;
	if (!load_signed(&atP, &sint64)) {
		goto fail;
	}
	*valueP = (sint32T) sint64;
	if (*atP++ != '\n') {
		goto fail;
	}
	if (0 <= sint64) { 
		for (maxP = max_sleb128; ; ++maxP) {
			++g_offset;
			if (sint64 <= *maxP) {
				break;
		}	}
	} else {
		for (maxP = min_sleb128; ; ++maxP) {
			++g_offset;
			if (*maxP <= sint64) {
				break;
	}	}	}
	*atPP = atP;
	return(1);
fail:
	assert(0);
	return(0);
}

int
load_string(char **atPP, char **bufferPP, int load, int escape)
{
	char 	*atP, *bufferP, *P, *P1;
	char	c;
	int		val;

	atP = *atPP;
	if (strncmp(atP, " .string \"", 10)) {
		goto fail;
	}
	atP    += 10;
	bufferP = P1 = *bufferPP;
	for (P = atP; (c = *P) != '"'; ++P) {
		if (c == '\\') {
			c   = *(++P);
			if (escape) {
				if (load) {
					*P1 = '\\';
				}
				++P1;
				if (c >= '0' && c <= '3') {
					if (load) {
						*P1 = c;		// Put back first octal digit
					}
					++P1;
					c = *(++P);
					assert(c >= '0' && c <= '7');
					if (load) {
						*P1 = c;		// Put back second octal digit
					}
					++P1;
					c = *(++P);
					assert(c >= '0' && c <= '7');
				}
			} else {
				switch (c) {
				case 'n':
					c = '\n';
					break;
				case 't':
					c = '\t';
					break;
				case 'v':
					c = '\v';
					break;
				case 'b':
					c = '\b';
					break;
				case 'r':
					c = '\r';
					break;
				case 'f':
					c = '\f';
					break;
				case 'a':
					c = '\a';
					break;
				case '0':
				case '1':
				case '2':
				case '3':
					// Three digit octal number
					val = c - '0';
					c   = *(++P);
					assert(c >= '0' && c <= '7');
					val = val * 8 + c - '0';
					c   = *(++P);
					c   = val * 8 + c - '0';
					break;
		}	}	}
		if (load) {
			*P1 = c;
		}
		++g_offset;
		++P1;
	}
	if (load) {
		*P1 = 0;
	}
	++g_offset;
	++P1;
	if (*(++P) != '\n') {
		goto fail;
	}
	*atPP     = P+1;
	*bufferPP = P1;
	return(1);
fail:
	assert(0);
	return(0);
}	

int
Cdwarf::load_addr(char **atPP, urefT *valueP)
{
	uint32T	uint32;
	uint64T	uint64;
	urefT	uref;

	if (!strncmp(*atPP, " .l", 3)) {
		if (!load_uint32(atPP, &uint32)) {
			goto fail;
		}
		uref = (urefT) uint32;
	} else {
		if (!load_uint64(atPP, &uint64)) {
			goto fail;
		}
		uref = (urefT) uint64;
	}
	*valueP = uref;
	return(1);
fail:
	assert(0);
	return(0);
}

int
Cdwarf::load_uint12(char **atPP, urefT *valueP)
{
	uint32T 	uint32;
	uint64T		uint64;
	urefT		uref;

	if (!load_uint32(atPP, &uint32)) {
		goto fail;
	}
	if (uint32 != 0xffffffff) {
		uref = uint32;
	} else {
#ifndef DWALF64
		assert(0);
#endif
		if (!load_uint64(atPP, &uint64)) {
			goto fail;
		}
		uref = (urefT) uint64;
	}
	*valueP   = uref;
	return(1);
fail:
	assert(0);
	return(0);
}

int
Cdwarf::load_block(char **atPP, char **bufferPP, int load, int lth)
{
	char 	*atP, *bufferP, *endP;
	uint64T	uint64;
	uint32T	uint32;
	sint32T	sint32;
	uint16T	uint16;
	uint8T	uint8;

	atP     = *atPP;
	bufferP = *bufferPP;
	endP    = bufferP + lth;

	while (bufferP < endP) {
		if (atP[0] != ' ' || atP[1] != '.') {
			goto fail;
		}
		switch (atP[2]) {
		case 'q':
			if (!load_uint64(&atP, &uint64)) {
				goto fail;
			}
			if (load) {
				memcpy(bufferP, &uint64, sizeof(uint64T));
			}
			bufferP += sizeof(uint64T);
			break;
		case 'l':
			if (!load_uint32(&atP, &uint32)) {
				goto fail;
			}
			if (load) {
				memcpy(bufferP, &uint32, sizeof(uint32T));
			}
			bufferP += sizeof(uint32T);
			break;
		case 'v':
			if (!load_uint16(&atP, &uint16)) {
				goto fail;
			}
			if (load) {
				memcpy(bufferP, &uint16, sizeof(uint16T));
			}
			bufferP += sizeof(uint16T);
			break;
		case 'b':
			if (!load_uint8(&atP, &uint8)) {
				goto fail;
			}
			if (load) {
				*bufferP = uint8;
			}
			++bufferP;
			break;
		case 'u':
			if (!load_uleb128(&atP, &uint32)) {
				goto fail;
			}
			encode_uleb128((uint64T) uint32, &bufferP, load);
			break;
		case 's':
			switch (atP[3]) {
			case 't':
				if (!load_string(&atP, &bufferP, load, 0 /* Dont escape */)) {
					goto fail;
				}
				break;
			case 'l':
				if (!load_sleb128(&atP, &sint32)) {
					goto fail;
				}
				encode_sleb128((sint64T) sint32, &bufferP, load);
				break;
			default:
				goto fail;
			}
			break;
		default:
			assert(0);
	}	}
	assert(bufferP == endP);
	*bufferPP = bufferP;
	*atPP     = atP;
	return(1);
fail:
	assert(0);
	return(0);
}

Cdwarf::Cdwarf(void)
{
	m_abbrevP     = 0;
	m_nodeP       = 0;
	m_strings_lth = 0;
	m_strings_max = 0;
	m_stringsP    = 0;
}

Cdwarf::~Cdwarf(void)
{
	Xfree(m_abbrevP);
	Xfree(m_nodeP);
	Xfree(m_stringsP);
}

int
Cdwarf::load_header(char **atPP)
{
	if (!load_uint12(atPP, &m_header.unit_length)) {
		goto fail;
	}
	if (!load_uint16(atPP, &m_header.version)) {
		goto fail;
	}
	assert(m_header.version >= 2 && m_header.version <= 3);
	if (!load_uint12(atPP, &m_header.debug_abbrev_offset)) {
		goto fail;
	}
	if (!load_uint8(atPP, &m_header.address_size)) {
		goto fail;
	}
#if DWARF64
	assert(m_header.address_size == 8);
#else
	assert(m_header.address_size == 4);
#endif
	return(1);
fail:
	assert(0);
	return(0);
}

int
Cdwarf::load_debug_abbrev(char *debug_abbrevP)
{
	char				*atP;
	uint32T				abbrev_code, abbrev_max, tag;
	uint8T				children;
	uint32T				name, form;
	int					abbrev_attr;
	dwarf_abbrevT		*abbrevP, *end_abbrevP, *abbrev1P;
	int					abbrev_lth;
	dwarf_abbrev_attrT	*start_attributesP, *attributeP, *startP;

	m_abbrevP    = 0;
	m_abbrev_max = 0;

	atP = debug_abbrevP;
	abbrev_max  = 0;
	abbrev_attr = 0;
	for (;;) {
		if (!load_uleb128(&atP, &abbrev_code)) {
			goto fail;
		}
		if (!abbrev_code) {
			break;
		}
		if (abbrev_code > abbrev_max) {
			abbrev_max = abbrev_code;
		}
		if (!load_uleb128(&atP, &tag)) {
			goto fail;
		}
		if (!load_uint8(&atP, &children)) {
			goto fail;
		}
		for (;;) {
			if (!load_uleb128(&atP, &name)) {
				goto fail;
			}
			if (!load_uleb128(&atP, &form)) {
				goto fail;
			}
			if (!name && !form) {
				break;
			}
			++abbrev_attr;
	}	}
	m_abbrev_max = abbrev_max;
	abbrev_lth   = (m_abbrev_max + 1) * sizeof(dwarf_abbrevT) + abbrev_attr * sizeof(dwarf_abbrev_attrT);
	m_abbrevP    = abbrevP = (dwarf_abbrevT *) Xmalloc(abbrev_lth);
	if (!abbrevP) {
		outofmemory();
	}
	for (end_abbrevP  = abbrevP + abbrev_max; abbrevP <= end_abbrevP; ++abbrevP) {
		Xcheck(m_abbrevP, abbrev_lth, abbrevP);
		abbrevP->tag         = DW_TAG_null;
		abbrevP->attributesP = 0;
	}
	attributeP = start_attributesP = (dwarf_abbrev_attrT *) abbrevP;
	abbrevP    = m_abbrevP;
	atP        = debug_abbrevP;
	for (;;) {
		if (!load_uleb128(&atP, &abbrev_code)) {
			goto fail;
		}
		if (!abbrev_code) {
			break;
		}
		assert(abbrev_code <= abbrev_max);
		abbrev1P = abbrevP + abbrev_code;
		Xcheck(abbrevP, abbrev_lth, abbrev1P);
		if (!load_uleb128(&atP, (unsigned int *) &abbrev1P->tag)) {
			goto fail;
		}
		if (!load_uint8(&atP, &children)) {
			goto fail;
		}
		assert(children < 2);
		abbrev1P->children    = (dw_childrenE) children;
		abbrev1P->attributesP = startP = attributeP;
		for (;;) {
			if (!load_uleb128(&atP, &name)) {
				goto fail;
			}
			if (!load_uleb128(&atP, &form)) {
				goto fail;
			}
			if (!name && !form) {
				abbrev1P->attributes = attributeP - startP;
				break;
			}
			Xcheck(abbrevP, abbrev_lth, attributeP);
			attributeP->name = (dw_atE)   name;
			attributeP->form = (dw_formE) form;
			++attributeP;
	}	}
	assert(((char *) attributeP) == ((char *) m_abbrevP) + abbrev_lth);
	return(1);
fail:
	assert(0);
	return(0);
}

/* Returns
   0:	No node found
   1:	Node found and loaded
  -1:	Error
 */

int
Cdwarf::load_node(char **atPP, char **bufferPP, dwarf_nodeT **nodePP)
{
	char				*startP, *atP;
	char				*bufferP, *extraP;
	dwarf_nodeT			*nodeP, **tailPP, *childP;
	dwarf_attrT			*attributeP;
	dwarf_abbrevT		*abbrevP;
	dwarf_abbrev_attrT	*abbrev_attributeP, *end_abbrev_attributeP;
	unsigned int		code;
	dw_formE			form;
	dw_valueT			value;
	uint32T				lth;
	uint32T				uint32;
	uint16T		      	uint16;
	uint8T				uint8;
	int					i;
	int					seen_children;
	int					offset;

	offset        = g_offset;
	atP = startP  = *atPP;
	for (bufferP = *bufferPP; ((long) bufferP) & 7; ++bufferP);

	if (!load_uleb128(&atP, &code)) {
		goto fail;
	}
	if (!code) {
		*atPP = atP;
		return(0);
	}

	assert(code <= m_abbrev_max);
	abbrevP           = m_abbrevP + code;
	nodeP             = (dwarf_nodeT *) bufferP;
	attributeP        = (dwarf_attrT *) (nodeP+1);
	abbrev_attributeP = abbrevP->attributesP;
	extraP            = (char *) (attributeP + abbrevP->attributes);

	if (nodePP) {
		*nodePP            = nodeP;
		nodeP->tag         = abbrevP->tag;
		nodeP->offset      = offset;
		nodeP->attributes  = abbrevP->attributes;
		nodeP->attributesP = attributeP;

		if (g_debug_dwarf) {
			for (i = m_depth; --i >= 0; ) {
				putchar(' ');
				putchar(' ');
			}
			printf("%u: %s", offset, get_node_name(abbrevP->tag));
		}
	}
	for (end_abbrev_attributeP = abbrev_attributeP + abbrevP->attributes; abbrev_attributeP < end_abbrev_attributeP; ++abbrev_attributeP) {
		form = abbrev_attributeP->form;
retry:
		switch (form) {
		case DW_FORM_null:
			value.uint32 = 0;
			break;
		case DW_FORM_flag:		// flag (one byte 0 or 1)
			if (!load_uint8(&atP, &uint8)) {
				goto fail;
			}
			assert(uint8 == 0 || uint8 == 1);
			value.uint32 = uint8;
			break;
		case DW_FORM_ref1:		// 1 byte offset reference
			if (!load_uint8(&atP, &uint8)) {
				goto fail;
			}
			value.uref = uint8;
			break;

		case DW_FORM_data1:		// 1 byte constant
			if (!load_uint8(&atP, &uint8)) {
				goto fail;
			}
			value.uint32 = uint8;
			break;
		case DW_FORM_ref2:		// 2 byte reference
			if (!load_uint16(&atP, &uint16)) {
				goto fail;
			}
			value.uref = uint16;
			break;

		case DW_FORM_data2:		// 2 byte constant
			if (!load_uint16(&atP, &uint16)) {
				goto fail;
			}
			value.uint32 = uint16;
			break;
		case DW_FORM_ref4:		// 4 byte reference
			if (!load_uint32(&atP, &uint32)) {
				goto fail;
			}
			value.uref = uint32;
			break;

		case DW_FORM_data4:		// int (constant, lineptr, loclistptr, macptr, rangelistptr)
			if (!load_uint32(&atP, &value.uint32)) {
				goto fail;
			}
			break;
		case DW_FORM_ref_udata:	// unsigned LEB128 reference
			if (!load_uleb128(&atP, &uint32)) {
				goto fail;
			}
			value.uref = uint32;
			break;
		case DW_FORM_addr:		// address
			if (!load_addr(&atP, &value.uref)) {
				goto fail;
			}
			break;
		case DW_FORM_strp:		// offset into string table in .debug_str
			if (!load_uint12(&atP, &value.uref)) {
				goto fail;
			}
			break;
		case DW_FORM_ref_addr:	// offset reference from start of debug info
			if (!load_uint12(&atP, &value.uref)) {
				goto fail;
			}
			break;
		case DW_FORM_ref8:		// 8 byte reference
		case DW_FORM_data8:		// long (constant, lineptr, loclistptr, macptr, rangelistptr)
			if (!load_uint64(&atP, &value.uint64)) {
				goto fail;
			}
			break;
		case DW_FORM_udata:		// unsigned LEB128 constant
			if (!load_uleb128(&atP, &value.uint32)) {
				goto fail;
			}
			break;
		case DW_FORM_sdata:		// signed LEB128 constant
			if (!load_sleb128(&atP, &value.sint32)) {
				goto fail;
			}
			break;
		case DW_FORM_block1:	// An unsigned char length followed by bytes
			if (!load_uint8(&atP, &uint8)) {
				goto fail;
			}
			lth = uint8;
			goto handle_block;
		case DW_FORM_block2:	// An unsigned short length followed by bytes
			if (!load_uint16(&atP, &uint16)) {
				goto fail;
			}
			lth = uint16;
			goto handle_block;
		case DW_FORM_block4:	// An unsigned int length followed by byte
			if (!load_uint32(&atP, &uint32)) {
				goto fail;
			}
			lth = uint32;
			goto handle_block;
		case DW_FORM_block:		// An unsigned LEB128 length followed by bytes
			if (!load_uleb128(&atP, &uint32)) {
				goto fail;
			}
			lth = uint32;
handle_block:
			for (; ((long) extraP) & 3; ++extraP);
			value.P = extraP;
			if (nodePP) {
				*((uint32T *) extraP) = lth;
			}
			extraP += sizeof(uint32T);

			if (!load_block(&atP, &extraP, nodePP != 0, lth)) {
				goto fail;
			}
			break;
		case DW_FORM_string:	// inlined string in the data
			value.P = extraP;
			if (!load_string(&atP, &extraP, nodePP != 0, 1)) {
				goto fail;
			}
			break;
		case DW_FORM_indirect:	// (see Section 7.5.3)
			if (!load_uleb128(&atP, &value.uint32)) {
				goto fail;
			}
			form = (dw_formE) value.uint32;
			goto retry;
		default:
			assert(0);
		}
		if (nodePP) {
			attributeP->name  = abbrev_attributeP->name;
			attributeP->form  = form;
			attributeP->value = value;

			if (g_debug_dwarf) {
				printf(" (%s %s ", get_attribute_name(abbrev_attributeP->name), get_form_name(form));
				switch(form) {
				case DW_FORM_flag:
					if (uint8) {
						printf("yes");
					} else {
						printf("no");
					}
					break;
				case DW_FORM_data1:
				case DW_FORM_data2:
				case DW_FORM_data4:
				case DW_FORM_udata:
				case DW_FORM_sdata:
					printf("%d", value.uint32);
					break;
				case DW_FORM_addr:
				case DW_FORM_ref_addr:
				case DW_FORM_strp:
				case DW_FORM_ref1:
				case DW_FORM_ref2:
				case DW_FORM_ref4:
				case DW_FORM_ref_udata:
				case DW_FORM_ref8:
#ifdef DWARF64
					printf("%lld", value.uref);
#else
					printf("%u", value.uref);
#endif
					break;
				case DW_FORM_data8:
					printf("%lld", value.uint64);
					break;
				case DW_FORM_block1:
				case DW_FORM_block2:
				case DW_FORM_block4:
				case DW_FORM_block:
					switch (attributeP->name) {
					case DW_AT_location:
						print_ops(lth, (uint8T *) value.P);
						break;
					default:
						printf("%d...", lth);
					}
					break;
				case DW_FORM_string:	// inlined string in the data
					printf("\"%s\"", value.P);
					break;
				case DW_FORM_null:
				case DW_FORM_indirect:
					break;
				}
				printf(")");
		}	}
		++attributeP;
	}

	if (nodePP) {
		if (g_debug_dwarf) {
			printf("\n");
		}
		tailPP = &nodeP->first_childP;
	} else {
		tailPP = 0;
	}

	if (abbrevP->children) {
		++m_depth;
		seen_children = 0;
		for (;;) {
			switch (load_node(&atP, &extraP, tailPP)) {
			case 1:
				if (tailPP) {
					childP = *tailPP;
					tailPP = &(childP->next_siblingP);
				}
				seen_children = 1;
				continue;
			case 0:
				if (!seen_children) {
					continue;
				}
				break;
			case -1:
				return(-1);
			}
			break;
		}
		--m_depth;
	}
	if (tailPP) {
		*tailPP = 0;
	}
	*atPP     = atP;
	*bufferPP = extraP;

	return(1);
fail:
	assert(0);
	return(-1);
} 

int
Cdwarf::load_debug_info(char *debug_infoP)
{
	char		*atP, *P;
	char		*startP, *bufferP;
	int			lth, lth1, offset, end_offset;

	g_offset = 0;
	atP = debug_infoP;
	if (!load_header(&atP)) {
		goto fail;
	}

	offset   = g_offset;
	P        = atP;
	bufferP  = 0;
	m_depth  = 0;
	if (load_node(&P, &bufferP, 0) != 1) {
		goto fail;
	}
	end_offset = g_offset;
	g_offset   = offset;
	lth      = bufferP - (char *) 0;
	startP   = bufferP = (char *) Xmalloc(lth);
	if (!bufferP) {
		outofmemory();
	}
	P        = atP;
	m_nodeP  = (dwarf_nodeT *) bufferP;
	m_depth  = 0;
	if (load_node(&P, &bufferP, &m_nodeP) != 1) {
		goto fail;
	}

	assert(end_offset == g_offset);
	lth1 = bufferP - startP;
	assert(lth == lth1);
	assert(m_nodeP->tag == DW_TAG_compile_unit || m_nodeP->tag == DW_TAG_partial_unit);
	return(1);
fail:
	assert(0);
	return(0);
}

int
Cdwarf::load_string_table(labelT *labelsP, char *debug_strP)
{
	char 	*stringsP, *P;
	labelT	*labelP;
	uint32T	lth;

	stringsP = 0;
	labelP   = labelsP;
	for (P = debug_strP; ; ) {
		if (!strncmp(P, " .ident", 7)) {
			for (; *P++ != '\n'; );
			continue;
		}

		if (strncmp(P, " .s", 3)) {
		 	for (;;++labelP) {
				if (!labelP->labelP) {
					goto fail;
				}
				if (labelP->labelP == P) {
					labelP->lineno = stringsP - ((char *) 0);
					break;
				}
				if (labelP->labelP > P) {
					goto fail;
			}	}
			for (; *P++ != '\n'; );
			continue;
		}
		if (!strncmp(P, " .section", 9)) {
			break;
		}
		if (!load_string(&P, &stringsP, 0, 0)) {
			goto fail;
	}	}
	m_strings_lth = lth = stringsP - ((char *) 0);
	if (m_strings_max < lth) {
		if (!m_strings_max) {
			m_strings_max = 1024;
		}
		do {
			m_strings_max <<= 1;
		} while (m_strings_max < lth);
		Xfree(m_stringsP);
		m_stringsP =  stringsP = (char *) Xmalloc(m_strings_max);
		if (!stringsP) {
			outofmemory();
	}	}
	stringsP = m_stringsP;
	for (P = debug_strP; ; ) {
		if (strncmp(P, " .s", 3)) {
			for (; *P++ != '\n'; );
			continue;
		}
		if (!strncmp(P, " .section", 9)) {
			break;
		}
		if (!load_string(&P, &stringsP, 1, 0)) {
			goto fail;
	}	}
	assert(lth == (uint32T) (stringsP - m_stringsP));
	assert(lth <= m_strings_max);
	return(1);
fail:
	assert(0);
	return(0);
}

void
Cdwarf::load(labelT *labelsP, char *debug_abbrevP, char *debug_infoP, char *debug_strP)
{
	m_labelsP     = labelsP;
	m_strings_lth = 0;

	if (debug_strP) {
		if (!load_string_table(labelsP, debug_strP)) {
			goto fail;
	}	}
	
	if (!load_debug_abbrev(debug_abbrevP)) {
		goto fail;
	}
	if (!load_debug_info(debug_infoP)) {
		goto fail;
	}
fail:
	return;
}

void
Cdwarf::load_locations1(dwarf_nodeT *nodeP, locationT **locationPP, int load)
{
	locationT		*locationP;
	location_typeE	type;
	dwarf_nodeT		*childP;
	dwarf_attrT		*attributeP, *end_attributesP;
	uint8T			*startP, *operationP;
	uint32T			lth;
	uint8T			operation;

	locationP = *locationPP;
	for (childP = nodeP->first_childP; childP; childP = childP->next_siblingP) {
		switch (childP->tag) {
		case DW_TAG_lexical_block:
		{
			locationT	*location1P;
			int			low_pc, high_pc;

			location1P = locationP;
			load_locations1(childP, &locationP, load);
			if (load) {
				attributeP = childP->attributesP;
				low_pc     = -1;
				high_pc    = -1;
				for (end_attributesP = attributeP + childP->attributes; attributeP < end_attributesP; ++attributeP) {
					switch (attributeP->name) {
					case DW_AT_low_pc:
						low_pc = (int) attributeP->value.uint64;
						break;
					case DW_AT_high_pc:
						high_pc = (int) attributeP->value.uint64;
					default:
						continue;
				}	}
				for (; location1P < locationP; ++location1P) {
					if (location1P->from_address == -1) {
						location1P->from_address = low_pc;
						location1P->to_address   = high_pc;
			}	}	}
			continue;
		}
		case DW_TAG_formal_parameter:
			type = LOCATION_parameter;
			goto common;
		case DW_TAG_variable:
			type = LOCATION_variable;
common:		if (load) {

				locationP->type			= type;
				locationP->nameP		= 0;
				locationP->frame_offset = -1;
				locationP->from_address = -1;
				locationP->to_address   = -1;
				locationP->size         = 0;	// Unknown
			
				attributeP = childP->attributesP;
				for (end_attributesP = attributeP + childP->attributes; attributeP < end_attributesP; ++attributeP) {
					switch (attributeP->name) {
					case DW_AT_name:
						assert(!locationP->nameP);
						switch (attributeP->form) {
						case DW_FORM_string:
							locationP->nameP = attributeP->value.P;
							break;
						case DW_FORM_strp:
							assert(attributeP->value.uref < m_strings_lth);
							locationP->nameP = m_stringsP + attributeP->value.uref;
							break;
						default:
							assert(0);
						}
						continue;
					case DW_AT_location:
						assert(locationP->frame_offset == -1);
						operationP  = startP = (uint8T *) attributeP->value.P;
						lth         = *((uint32T *) operationP);
						operationP += sizeof(uint32T);
						switch (operation = *operationP++) {
						case DW_OP_fbreg:
							locationP->frame_offset = (int) decode_sleb128((uint8T **) &operationP);
							if ((operationP - startP) == (int) (lth + sizeof(uint32T))) {
								break;
							}
						// case DW_OP_addr: // This is a static variable
						default:
							goto skip;
						}
						continue;
					case DW_AT_type:
						locationP->size = -((srefT) attributeP->value.uref);
						continue;
					default:
						continue;
			}	}	}
			++locationP;
		default:
skip:
			continue;
	}	}
	*locationPP = locationP;
}

void
Cdwarf::load_locations(char *signatureP, char *function_nameP, locationT **locationsPP, int load)
{
	dwarf_nodeT	*childP;
	dwarf_attrT	*attributeP, *end_attributesP;
	locationT	*start_locationP, *locationP, *end_locationsP;
	int			ref1, ref2, size, seen, array, ret;
	char		*nameP, *P;

	start_locationP = *locationsPP;
	for (childP = m_nodeP->first_childP; childP; childP = childP->next_siblingP) {
		if (childP->tag == DW_TAG_subprogram) {
			attributeP = childP->attributesP;
			for (end_attributesP = attributeP + childP->attributes; attributeP < end_attributesP; ++attributeP) {
				if (attributeP->name == DW_AT_signature ||
					attributeP->name == DW_AT_name) {

					switch (attributeP->form) {
					case DW_FORM_string:
						nameP = attributeP->value.P;
						break;
					case DW_FORM_strp:
						nameP = m_stringsP + attributeP->value.uref;
						break;
					default:
						assert(0);
					}
					if (attributeP->name == DW_AT_signature) {
						ret = strcmp(signatureP, nameP);
					} else if ((P = strchr(function_nameP, '('))) {
						*P  = 0;
						ret = strcmp(function_nameP, nameP);
						*P  = '(';
					} else {
						ret = strcmp(function_nameP, nameP);
					}
					if (!ret) {
						// Load parameters/variables of this function
						load_locations1(childP, locationsPP, load);
						goto discover_sizes;
	}	}	}	}	}
	return;
discover_sizes:
	if (!load) {
		return;
	}
	end_locationsP = *locationsPP;
	do {
		seen = 0;
		size = 0;		// Implies unknown
		for (locationP = start_locationP; locationP < end_locationsP; ++locationP) {
			if (locationP->size >= 0) {
				// Unknown size
				continue;
			}
			array = 0;
			seen  = 1;
			ref1  = locationP->size;
			ref2  = -locationP->size;
retry:
			for (childP = m_nodeP->first_childP; childP; childP = childP->next_siblingP) {
				if (childP->offset != ref2) {
					continue;
				}
				switch (childP->tag) {
				case DW_TAG_base_type:
				case DW_TAG_pointer_type:
				case DW_TAG_structure_type:
				case DW_TAG_union_type:
				case DW_TAG_enumeration_type:
				case DW_TAG_const_type:
					break;
				case DW_TAG_array_type:
					array = 1;
				case DW_TAG_volatile_type:
				case DW_TAG_typedef:
					attributeP = childP->attributesP;
					for (end_attributesP = attributeP + childP->attributes; attributeP < end_attributesP; ++attributeP) {
						if (attributeP->name == DW_AT_type) {
							ref2 = attributeP->value.uref;
							goto retry;
					}	}
				default:
					assert(0);
					continue;
				}
				attributeP = childP->attributesP;
				for (end_attributesP = attributeP + childP->attributes; attributeP < end_attributesP; ++attributeP) {
					if (attributeP->name != DW_AT_byte_size) {
						continue;
					}
					size = attributeP->value.uint32;
					goto know_size;
			}	}
know_size:
			for (; locationP < end_locationsP; ++locationP) {
				if (locationP->size == ref1) {
					locationP->size = size;
					if (array) {
						switch(locationP->type) {
						case LOCATION_parameter:
							locationP->type = LOCATION_array_parameter;
							break;
						case LOCATION_variable:
							locationP->type = LOCATION_array_variable;
						default:
							break;
					}	}
		}	}	}
	} while (seen);
	return;
}
#endif
