#ifdef WIN32
#define _CRT_SECURE_NO_WARNINGS
#endif

#ifndef NDEBUG
//#define DEBUG_DETECT
//#define DEBUG_HILLCLIMB
//#define TEST_HILLCLIMB
//#define PROOF
#endif

#include <assert.h>
#include <time.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#ifdef WIN32
#define snprintf _snprintf
#else
#include <unistd.h>
#endif

#include "object.h"
#include "xmalloc.h"
#include "util.h"
#include "file.h"
#include "location.h"
#include "code.h"
#include "function.h"
#include "source.h"
#include "data.h"
#include "clics_output.h"
#include "match.h"
#include "detect.h"

#ifdef WIN32
#define clock_t time_t
#endif

extern int			g_source_cnt;
extern int			g_clone_cnt;
extern int			g_top_clones;
extern int			g_top_cols;
extern int			g_report_c;
extern int			g_report_assembler;
extern int			g_report;
extern int			g_verbose;
extern char			*g_html_dirP;
extern int			g_ignore_macros;
extern int			g_min_match_sequence;
extern int			g_min_function_sequence;
extern int			g_mismatch_cost;
extern int			g_same_cost;
extern int			g_match_macros;
extern int			g_match_lines;
extern long long	g_comparisons;
#ifdef CORRELATE
extern int			g_correlate;
#endif
extern int			g_hillclimb;
extern int			g_hillclimb_timeout;
extern int			g_iterations;
extern int			g_best_weight;
extern time_t		g_hillclimb_elapse;

extern long long	g_original_matches;
extern long long	g_extra_matches;
extern long long	g_cancelled_matches;
extern int			g_max_blocks_seen;
extern int			g_max_iterations_seen;

static	Csource			*source1P;		// Source 1st side looking at
static	Csource			*source2P;		// Source 2nd side looking at
static	Cfunction		*function1P;	// Function 1st side looking at
static	Cfunction		*function2P;	// Function 2nd side looking at
static	codeT			*function_code1P;// Start of function1 code
static	codeT			*function_code2P;// Start of function2 code
static	codeT			*function_end1P;// Beyond end of function1 code
static	codeT			*function_end2P;// Beyond end of function2 code
static	codeT			*start_code1P;	// Comparing from 1st side
static	codeT			*start_code2P;	// Comparing from 2nd side
static	codeT			*end_code1P;	// May not compare to here on 1st side
static	codeT			*end_code2P;	// May not compare to here on 2nd side
static	codeT			*best_end1P;	// Best end of side 1
static	codeT			*best_end2P;	// Best end of side 2

extern FILE		*indexF;
extern FILE		*sourceF;
extern FILE		*cloneF;

static Csource	*last_sourceP = 0;

typedef struct {
	unsigned int	weight;
	unsigned int	html;
} topT;

static topT		*top_clonesP       = 0;
static int		top_clones_lth     = 0;
static topT		*top_clones_endP   = 0;
static int		top_clones_cnt     = 0;

void
initialise_detect(void)
{
	if (!g_html_dirP || !g_report) {
		g_top_clones = 0;
	}
	if (g_top_clones > 0) {
		top_clones_lth = g_top_clones * sizeof(topT);
		top_clonesP    = (topT *) Xmalloc(top_clones_lth);
		if (!top_clonesP) {
			outofmemory();
		}
		memset(top_clonesP, 0, top_clones_lth);
		top_clones_endP = (topT *) (((char *) top_clonesP) + top_clones_lth);
	}
}

void
start_report(void)
{
	clics_sources();
	if (g_html_dirP) {
		fputs("<p>\n<table>\n", indexF);
}	}

static int		g_lth1, g_lth2;
static clock_t	start_clock, end_clock;

#ifdef PROOF
static void
proof(void)
{
	static	codeT	*code1aP, *code2aP, *code1bP, *code2bP;
	static	codeT	*code1a_jumpP, *code2a_jumpP;
	static	codeT	*code1b_jumpP, *code2b_jumpP;

	assert(start_code1P->m_matchP);
	for (code1aP = start_code1P; code1aP < best_end1P; ++code1aP) {
		if (!(code2aP = code1aP->m_matchP)) {
			continue;
		}
		assert(code2aP->m_matchP == code1aP);
		code1a_jumpP = code1aP->m_jumpP;
		code2a_jumpP = code2aP->m_jumpP;
		if (!code1a_jumpP) {
			assert(!code2a_jumpP);
		} else if (code1a_jumpP == code1aP) {
			assert(code2a_jumpP == code2aP);
		} else if (code1a_jumpP > code1aP) {
			assert(code2a_jumpP > code2aP);
		} else {
			assert(code2a_jumpP && code2a_jumpP < code2aP);
			assert(code1a_jumpP >= start_code1P);
			assert(code2a_jumpP >= start_code2P);
		}
		for (code1bP = start_code1P; code1bP < best_end1P; ++code1bP) {
			if (code1aP == code1bP) {
				continue;
			}
			if (!(code2bP = code1bP->m_matchP)) {
				continue;
			}
			assert((code1aP > code1bP && code2aP > code2bP) ||
                   (code1aP < code1bP && code2aP < code2bP));
			if (!code1a_jumpP) {
				continue;
			}
			assert((code1a_jumpP <= code1bP && code2a_jumpP <= code2bP) ||
				   (code1a_jumpP > code1bP  && code2a_jumpP > code2bP));

			code1b_jumpP = code1bP->m_jumpP;
			code2b_jumpP = code2bP->m_jumpP;
			if (!code1b_jumpP || !code2b_jumpP) {
				continue;
			}
			if (code1a_jumpP >= best_end1P && code2a_jumpP >= best_end2P) {
				continue;
			}
			if (code1b_jumpP >= best_end1P && code2b_jumpP >= best_end2P) {
				continue;
			}
			assert((code1a_jumpP >= code1b_jumpP && code2a_jumpP >= code2b_jumpP) ||
				   (code1a_jumpP <= code1b_jumpP && code2a_jumpP <= code2b_jumpP));
	}	}	
}

#endif

extern FILE	*clicsF;
extern FILE	*clics1F;

static void
report_clone(void)
{
	extern	FILE			*taF;
	extern	FILE			*tempF;

	static	int				lineno1, lineno2, next_lineno1, next_lineno2;
	static	Cfile			*file1P, *file2P, *next_file1P, *next_file2P;
	static	const Cfile		*sourcefile1P, *sourcefile2P;
	static	int				different1, different2;

	static	char	*source_name1P, *source_name2P;
	static	codeT	*at_code1P, *at_code2P;
	static	int		offset1, offset2;
	static  unsigned int	source_cnt;

	static	unsigned int heap_index;
	static	topT	*heapP, *leftP, *rightP;
	static	unsigned int	weight;
	static	char			*P;
	static time_t			clock_now;
	static int				ret;
	static FILE				*F;

	sourcefile1P  = source1P->m_fileP;
	sourcefile2P  = source2P->m_fileP;
	source_name1P = sourcefile1P->m_nameP;
	source_name2P = sourcefile2P->m_nameP;

	if (!source1P->m_source_cnt) {
		source1P->m_source_cnt = ++g_source_cnt;
	}
	source_cnt = source1P->m_source_cnt;
	if (!source2P->m_source_cnt) {
		source2P->m_source_cnt = ++g_source_cnt;
	}
	
	// Increment the detail page number
	++g_clone_cnt;

	if (taF) {
		Cclone *clone1P, *clone2P;

		clone1P = function1P->locate_clone(start_code1P, best_end1P);
		clone2P = function2P->locate_clone(start_code2P, best_end2P);

		fprintf(taF, "R C%p C%p\n", clone1P, clone2P);
		if (g_report) {
			fprintf(tempF, "(R C%p C%p) { html=%u }\n", clone1P, clone2P, g_clone_cnt);
	}	}

	if (clicsF || clics1F) {
		clics_entry(start_code1P, best_end1P, start_code2P, best_end2P);
	}

	if (!g_html_dirP) {
		return;
	}

	if (top_clonesP) {
		// Use a heap with the smallest weight at the top

		weight = g_lth1 + g_lth2;
		if (top_clonesP->weight < weight) {
			++top_clones_cnt;
			heap_index = 1;
			heapP      = top_clonesP;
			// Add entry to the heap
			for (;;) {
				leftP = heapP + heap_index;
				if (top_clones_endP <= leftP) {
					goto found_heap_entry;
				}
				rightP = leftP + 1;
				if ((top_clones_endP <= rightP) || (leftP->weight <= rightP->weight)) {
					if (weight <= leftP->weight) {
						goto found_heap_entry;
					}
					Xcheck(top_clonesP, top_clones_lth, heapP);
					*heapP       = *leftP;
					heapP        = leftP;
					heap_index <<= 1;
					continue;
				}
				if (weight <= rightP->weight) {
found_heap_entry:	Xcheck(top_clonesP, top_clones_lth, heapP);
					heapP->weight = weight;
					heapP->html   = g_clone_cnt;
					break;
				}
				Xcheck(top_clonesP, top_clones_lth, heapP);
				*heapP       = *rightP;
				heapP        = rightP;
				heap_index   = (heap_index << 1) + 1;
			}
	}	}

	if (last_sourceP != source1P) {
		if (sourceF) {
			fprintf(sourceF,"</table>\n<p>\nEnd of report</body>\n</html>\n");
			/* Protect against interrupt */
			F       = sourceF;
			sourceF = 0;
			fclose(F);
		}
		ret = snprintf(g_filename, FILENAME_MAX, "%s/s%u.html", g_html_dirP, source_cnt);
		if (ret < 0 || ret >= FILENAME_MAX) {
			fprintf(stderr, "html file name too long\n");
			error_exit(1);
		}
		sourceF = open_file(g_filename);
		last_sourceP = source1P;

		fprintf(sourceF, HTML_HEADER "<title>Clones in %s", source_name1P);
		fprintf(sourceF, "</title>\n<body>\n<a href=index.html>Top</a>\n<h3>Clones in %s", source_name1P);
		fprintf(sourceF, "</h3>\n<table>\n");
			
		clock_now = time(0);
		P         = ctime(&clock_now);
		P[19]     = 0;
		fprintf(indexF, "<tr><td>%s</td><td><a href=\"s%u.html\">%s", P + 11, source_cnt, source_name1P);
		fprintf(indexF, "</a></td></tr>\n");
	}


	fputs("<tr><td><i>", sourceF);
	if (g_report) {
		fprintf(sourceF, "<a href=\"%u.html\">%s</a>", g_clone_cnt, function1P->m_nameP);
	} else {
		fprintf(sourceF, "%s", function1P->m_nameP);
	}
	fprintf(sourceF, "</i> <font color=\"red\">[%d]</font></td>\n    <td><i>%s.%s", (int) (best_end1P - start_code1P), source_name2P, function2P->m_nameP);
	fprintf(sourceF, "</i> <font color=\"blue\">[%d]</font></td>\n", (int) (best_end2P - start_code2P));
	fputs("</i></td>\n</tr>\n", sourceF);
	 
	if (!g_report) {
		return;
	}
	cloneF = 0;
	ret = snprintf(g_filename, FILENAME_MAX, "%s/%u.html", g_html_dirP, g_clone_cnt);
	if (ret < 0 || ret >= FILENAME_MAX) {
		fprintf(stderr, "html filename too long\n");
		error_exit(1);
	}
	cloneF = open_file(g_filename);
	fputs(
HTML_HEADER
"<script language=\"javascript\">\n"
"<!--\n"
"var a, s;\n"
"var a_shown='none'\n"
"var s_shown='none'\n"
"function toggleA() {\n"
"  for (i = 0; i < a.length; i++) { a[i].style.display = a_shown}\n"
"  a_shown = (a_shown == 'none' ? 'inline' : 'none')\n"
"}\n"
"function toggleS() {\n"
"  for (i = 0; i < s.length; i++) { s[i].style.display = s_shown}\n"
"  s_shown = (s_shown == 'none' ? 'inline' : 'none')\n"
"}\n"
"function init(){ \n"
"  var ai = 0;\n"
"  var si = 0;\n"
"  var tr = document.getElementsByTagName('*');\n"
"  var e, className, i;\n"
"  for (i = 0; i < tr.length; i++) {\n"
"    e = tr[i];\n"
"    if (e.tagName == 'TR') {\n"
"     className = \"\" + e.className;\n"
"     switch (className) {\n"
"      case 'a': ++ai; break;\n"
"      case 's': ++si; break;\n"
"  } } }\n"
"  a  = new Array(ai);\n"
"  s  = new Array(si);\n"
"  ai = 0;\n"
"  si = 0;\n"
"  for (i = 0; i < tr.length; i++) {\n"
"   e = tr[i];\n"
"   if (e.tagName == 'TR') {\n"
"    className = \"\" + e.className;\n"
"    switch(className) {\n"
"    case 'a': a[ai] = e; ++ai; break;\n"
"    case 's': s[si] = e; ++si; break;\n"
"} } } }\n"
"//-->\n"
"</script>\n"
"</head>\n"
"<body onLoad=\"init()\">\n"
"<p>\n"
"[<a href=index.html>Top</a>] "
	, cloneF);

	fprintf(cloneF, "[<a href=%u.html>Next</a>] ", g_clone_cnt+1);
	fputs(
"[<a href=\"javascript:toggleA()\">Toggle</a>] Assembler\n"
"[<a href=\"javascript:toggleS()\">Toggle</a>] Source\n"
"<p>\n"
	  , cloneF);
	fputs("<p>\n<table>\n", cloneF);
	fprintf(cloneF, "<tr><td><b><a href=s%u.html>%s</a>", source_cnt, source_name1P);
	fprintf(cloneF, "</b></td>\n    <td><b><a href=s%u.html>%s</a>", source2P->m_source_cnt, source_name2P);
	fputs("</b></td>\n</tr>\n", cloneF);
	fprintf(cloneF, "<tr><td><i>%s", function1P->m_nameP);
	fprintf(cloneF, "</i> <font color=\"red\">[%d]</font></td>\n    <td><i>%s", (int) (best_end1P - start_code1P), function2P->m_nameP);
	fprintf(cloneF, "</i> <font color=\"blue\">[%d]</font></td>\n</tr>\n", (int) (best_end2P - start_code2P));

	fputs("<tr><td><hr></hr></td><td><hr></hr></td></tr>", cloneF); 

	at_code1P = start_code1P;
	at_code2P = start_code2P;

	if (g_report_c) {

		if (at_code1P < best_end1P) {
			next_lineno1 = at_code1P->m_source_lineno;
			file1P       = at_code1P->m_fileP;
		} else {
			next_lineno1 = -1;
			file1P       = 0;
		}
		if (at_code2P < best_end2P) {
			next_lineno2 = at_code2P->m_source_lineno;
			file2P       = at_code2P->m_fileP;
		} else {
			next_lineno2 = -1;
			file2P       = 0;
		}
		different1   = 0;
		different2   = 0;
	}

	offset1 = at_code1P - function_code1P;
	offset2 = at_code2P - function_code2P;
	for (; at_code1P < best_end1P || at_code2P < best_end2P; ) {
			
		if (g_report_assembler) {
			if (best_end1P <= at_code1P || (at_code1P->m_matchP && at_code1P->m_matchP != at_code2P) ) {
				fprintf(cloneF, "<tr class=a><td></td>\n    <td><font color=\"blue\">%d: ", offset2);
				P = at_code2P->m_codeP;
				if (!at_code2P->m_jumpP) {
					put_html(P, cloneF);
				} else {
					put_opcode(P, cloneF);
					fprintf(cloneF, " -> %d", (int) (at_code2P->m_jumpP - function_code2P));
				}
				++at_code2P;
				++offset2;
				fputs("</font></td>\n</tr>\n", cloneF);
				different2 = 1;
			} else if (!at_code1P->m_matchP) {
				fprintf(cloneF, "<tr class=a><td><font color=\"red\">%d: ", offset1);
				P = at_code1P->m_codeP;
				if (!at_code1P->m_jumpP) {
					put_html(P, cloneF);
				} else {
					put_opcode(P, cloneF);
					fprintf(cloneF, " -> %d", (int) (at_code1P->m_jumpP - function_code1P));
				}	
				fputs("</font></td>\n    <td>", cloneF);
				++at_code1P;
				++offset1;
				different1 = 1;
			
				if (at_code2P < best_end2P && !at_code2P->m_matchP) {
					fprintf(cloneF, "<font color=\"blue\">%d: ", offset2);
					P = at_code2P->m_codeP;
					if (!at_code2P->m_jumpP) {
						put_html(P, cloneF);
					} else {
						put_opcode(P, cloneF);
						fprintf(cloneF, " -> %d", (int) (at_code2P->m_jumpP - function_code2P));
					}
					fputs("</font>", cloneF);
					++at_code2P;
					++offset2;
					different2 = 1;
				}
				fputs("</td>\n</tr>\n", cloneF);
			} else {
				fprintf(cloneF, "<tr class=a");
				if (at_code1P->m_fileP == at_code2P->m_fileP && at_code1P->m_source_lineno == at_code2P->m_source_lineno) {
					fprintf(cloneF, " bgColor=#D0D0D0");
				}
				fprintf(cloneF, "><td>%d: ", offset1);
				P = at_code1P->m_codeP;
				if (!at_code1P->m_jumpP) {
					put_html(P, cloneF);
				} else {
					put_opcode(P, cloneF);
					fprintf(cloneF, " -> %d", (int) (at_code1P->m_jumpP - function_code1P));
				}
				fprintf(cloneF, "</td>\n    <td>%d: ", offset2);
				++at_code1P;
				++offset1;
				P = at_code2P->m_codeP;
				if (!at_code2P->m_jumpP) {
					put_html(P, cloneF);
				} else {
					put_opcode(P, cloneF);
					fprintf(cloneF, " -> %d", (int) (at_code2P->m_jumpP - function_code2P));
				}
				fputs("</td>\n</tr>\n", cloneF);
				++at_code2P;
				++offset2;
			} 
		} else {
			// Not reporting assembler

			if (best_end1P <= at_code1P || (at_code1P->m_matchP && at_code1P->m_matchP != at_code2P) ) {
				++at_code2P;
				different2 = 1;
			} else if (!at_code1P->m_matchP) {
				++at_code1P;
				different1 = 1;
			
				if (at_code2P < best_end2P && !at_code2P->m_matchP) {
					++at_code2P;
				}
			} else {
				++at_code1P;
				++at_code2P;
		}	} 

		if (g_report_c) {

			lineno1      = next_lineno1;
			lineno2      = next_lineno2;
			if (at_code1P < best_end1P ) {
				next_lineno1 = at_code1P->m_source_lineno;
				next_file1P  = at_code1P->m_fileP;
			} else {
				next_lineno1 = -1;
				next_file1P  = 0;
			}
			if (at_code2P < best_end2P ) {
				next_lineno2 = at_code2P->m_source_lineno;
				next_file2P  = at_code2P->m_fileP;
			} else {
				next_lineno2 = -1;
				next_file2P  = 0;
			}

show_next_sourceline:
			if (!file1P || lineno1 == next_lineno1) {
				if (!file2P || lineno2 == next_lineno2) {
					continue;
				}
				fputs("<tr class=s><td></td>\n    <td><font color=\"blue\">", cloneF);
				file2P->showline(cloneF, &sourcefile2P, lineno2);
				fputs("</font></td>\n</tr>\n", cloneF);
				++lineno2;
				if (lineno2 < next_lineno2 && file2P == next_file2P) {
					goto show_next_sourceline;
				}
			} else if (!file2P || lineno2 == next_lineno2) {
				fputs("<tr class=s><td><font color=\"red\">", cloneF);
				file1P->showline(cloneF, &sourcefile1P, lineno1);
				fputs("</font></td>\n    <td></td></tr>\n", cloneF);
				++lineno1;
				if (lineno1 < next_lineno1 && file1P == next_file1P) {
					goto show_next_sourceline;
				}
			} else {
				fputs("<tr class=s", cloneF);
				if (!different1 && !different2 && sourcefile1P == sourcefile2P && lineno1 == lineno2) {
					fputs(" bgColor=#D0D0D0", cloneF);
				}
				fputs("><td>", cloneF);
				if (file1P) {
					if (!different1) {
						file1P->showline(cloneF, &sourcefile1P, lineno1);
						++lineno1;
					} else {
						fputs("<font color=\"red\">", cloneF);
						file1P->showline(cloneF, &sourcefile1P, lineno1);
						++lineno1;
						fputs("</font>", cloneF);
				}	}
				fputs("</td><td>", cloneF);
				if (file2P) {
					if (!different2) {
						file2P->showline(cloneF, &sourcefile2P, lineno2);
						++lineno2;
					} else {
						fputs("<font color=\"blue\">", cloneF);
						file2P->showline(cloneF, &sourcefile2P, lineno2);
						++lineno2;
						fputs("</font>", cloneF);
				}	}
				fputs("</td></tr>\n", cloneF);
				if ((lineno1 < next_lineno1 && file1P == next_file1P) || 
                    (lineno2 < next_lineno2 && file2P == next_file2P)) {
					goto show_next_sourceline;
			}	}
			file1P = next_file1P;
			file2P = next_file2P;
			different1 = 0;
			different2 = 0;
		}
	}
	fputs("</table>\n", cloneF);

#ifdef CORRELATE
	if (g_correlate) {
		correlate_data(function1P, start_code1P, best_end1P, function2P, start_code2P, best_end2P);
	}
#endif

	fprintf(cloneF, "<p><table>\n<tr><td>Start</td><td>%s</td></tr>\n", 	ctime(&start_clock));
	fprintf(cloneF, "<tr><td>End</td><td>%s</td></tr>\n</table>\n", 	ctime(&end_clock));
	fputs("<p>End of report\n</body>\n</html>\n", cloneF);
	fclose(cloneF);

}

void
finalize_report(void)
{
	extern char *g_signalP;

	static	topT	*topP, *atP, *endP, *heapP, *leftP, *rightP;
	static	int		heap_index, cnt, top_lth, i, ret;
	static	FILE	*topF;

	if (sourceF) {
		fprintf(sourceF,"</table>\n");
		if (g_signalP) {
			fprintf(sourceF, "<p><font color=red>**EXECUTION INTERRUPTED BY SIG%s**</font>\n", g_signalP);
		}
		fprintf(sourceF,"<p>\n<p>End of report</body>\n</html>\n");
		fclose(sourceF);
	}
	
#ifdef CORRELATE
	if (g_correlate) {
		report_correlated();
	}
#endif
	if (top_clonesP) {
		cnt = top_clones_cnt;
		if (cnt > g_top_clones) {
			cnt = g_top_clones;
		}
		top_lth = cnt * sizeof(topT);
		if (cnt) {
			topP = atP = (topT *) Xmalloc(top_lth);
			if (!topP) {
				outofmemory();
			}
			for (endP = atP + cnt; ; ) {
				if (top_clonesP->html) {
					Xcheck(topP, top_lth, atP);
					*atP++ = *top_clonesP;
					if (endP <= atP) {
						break;
				}	}
				heapP      = top_clonesP;
				heap_index = 1;
				for (;;) {
					leftP = heapP + heap_index;
					if (top_clones_endP <= leftP) {
						heapP->weight = 0xFFFFFFFF;
						break;
					}
					rightP = leftP + 1;
					if (top_clones_endP <= rightP || leftP->weight <= rightP->weight) {
						*heapP       = *leftP;
						heapP        = leftP;
						heap_index <<= 1;
						continue;
					}
					*heapP     = *rightP;
					heapP      = rightP;
					heap_index = (heap_index << 1) + 1;
			}	}
			ret = snprintf(g_filename, FILENAME_MAX, "%s/top.html", g_html_dirP);
			if (ret < 0 || ret >= FILENAME_MAX) {
				fprintf(stderr, "top filename too long\n");
				exit(1);
			}
			topF = open_file(g_filename);
			fprintf(topF, HTML_HEADER "<a href=index.html>Top</a>\n<title>Top %d largest clones</title>\n<body>\n<h3>Top %d largest clones</h3>\n<p>\n<table><tr>\n", cnt, cnt);
			for (i = 0; i < g_top_cols; ++i) {
				fprintf(topF, "<th><font color=red>Weight</font></th><th>Clone</th>");
			}
			for (i = g_top_cols; --atP >= topP; ++i) {
				if (i == g_top_cols) {
					fprintf(topF, "</tr>\n<tr>");
					i = 0;
				}
				fprintf(topF, "<td align=right>%u</td><td><a href=%u.html>%u.html</td>", atP->weight, atP->html, atP->html);
			}
			fprintf(topF, "</tr>\n</table>\n<P>\nEnd of report\n</body>\n</html>\n");
			fclose(topF);
	}	}
}

static int
permitted_clone(void)
{
	if ( (g_lth1 < g_min_match_sequence) || (g_lth2 < g_min_match_sequence)) {
		// Generally the clone is too short on one side or the other to be deemed worth reporting
		// Consider a clone long enough if it includes all of a function
		if ((start_code1P != function_code1P || best_end1P != function_end1P || g_lth1 < g_min_function_sequence) &&
			(start_code2P != function_code2P || best_end2P != function_end2P || g_lth2 < g_min_function_sequence)
		   ) {
			return(0);
	}	}
	return(1);
}

static int
same_branch(codeT *code1P, codeT *code2P)
{
	codeT	*to1P, *to2P, *matchP;

	to1P = code1P->m_jumpP;
	to2P = code2P->m_jumpP;
	if (to1P < code1P) {
		if (to2P >= code2P) {
			goto mismatch;
		}
		if (to1P < start_code1P) {
			goto mismatch;
		}
		if (to2P < start_code2P) {
			goto mismatch;
		}
		if ((matchP = to1P->m_matchP)) {
			if (to2P > matchP) {
				goto mismatch;
			}
			if (to2P->m_matchP) {
				if (to2P == matchP) {
					goto same;
				}
				goto mismatch;
			}
		} else if ((matchP = to2P->m_matchP)) {
			if (to1P > matchP) {
				goto mismatch;
			}
			if (to2P == start_code2P) {
				goto mismatch;
		}	}

		while (!((--to1P)->m_matchP));
		assert(start_code1P <= to1P);
		while (!((--to2P)->m_matchP));
		assert(start_code2P <= to2P);

		if (to1P->m_matchP != to2P) {
			goto mismatch;
		}
	} else if (to1P > code1P) {
		if (to2P <= code2P) {
			goto mismatch;
		}
		if (end_code1P <= to1P) {
			goto mismatch;
		}
		if (end_code2P <= to2P) {
			goto mismatch;
		}
	} else if (to2P != code2P) {
		goto mismatch;
	}
same:
	return(1);
mismatch:
	return(0);
}

static int
same_code(codeT *code1P, codeT *code2P, int *weightP) 
{
	register char	*assembler1P, *assembler2P;
	char			c;

	assembler1P = code1P->m_codeP;
	assembler2P = code2P->m_codeP;

	if (*assembler1P != *assembler2P) {
		goto mismatch;
	}

	if (code1P->m_jumpP) {
		if (!code2P->m_jumpP) {
			goto mismatch;
		}
		while (++assembler1P, ++assembler2P) {
			if ((c = *assembler1P) != *assembler2P) {
				goto mismatch;
			}
			switch (c) {
			case ' ':
				if (!same_branch(code1P, code2P)) {
					goto mismatch;
				}
			case 0:
				goto same;
	}	}	}

	if (code2P->m_jumpP) {
		goto mismatch;
	}
	if (strcmp(assembler1P, assembler2P)) {
		goto mismatch;
	}
same:
	*weightP = g_same_cost;
	return(1);
mismatch:
	*weightP = g_mismatch_cost;
	return(0);
}

// Hill Climbing Logic

typedef struct dllS {
	struct dllS		*prevP;
	struct dllS 	*nextP;
} dllT;

typedef unsigned short crossT;

typedef	struct edgeS {
	dllT			dll;			// Must be first element in struct
	codeT			*code1P;		// Left  side of match
	codeT			*code2P;		// Right side of match
	int				matched;		// This is a matching edge
	int				blocks;			// Number of edges that block this edge
	struct edgeS	*next_changeP;
	int				blocks1;

	int				crosses;
	crossT			*crossesP;
} edgeT;

typedef struct {
	edgeT			*edgeP;
	int				matched;
} reverseT;

typedef struct {
	edgeT			*edgeP;
	int				blocks;
} unblockedT;

static int
hill_climb(void)
{
	// Array of edges
	static	edgeT			*edgesP         = 0;
	static  int				edges_lth		= 0;
	static	edgeT			*edges_maxP     = 0;
	static	unsigned int	max_edges       = 0;
	static	edgeT			*end_edgesP     = 0;

	// Array of crossings of edges with edges
	static	crossT			*crossesP       = 0;
	static	unsigned int	max_crosses     = 0;

#define Backup(X) \
	if (X->next_changeP == (edgeT *) -1) { \
		X->blocks1      = X->blocks; \
		X->next_changeP = changesP; \
		changesP        = X; \
	}

#define Increment(X) \
	Backup(X); \
	++(X->blocks1);

#define Decrement(X) \
	Backup(X); \
	--(X->blocks1);

	// Array of edges whose matched/unmatched status is to be reversed
	static	reverseT 		*reversesP     = 0;
	static	int		 		reverses_lth   = 0;
	static	reverseT 		*reverses_maxP = 0;
	static	unsigned int	max_reverses   = 0;
	static	reverseT 		*end_reversesP = 0;

#define Reverse(X) \
	if (reverses_maxP <= end_reversesP) { \
		at = max_reverses; \
		if (!max_reverses) { \
			max_reverses   = 4096; \
		} else { \
			max_reverses <<= 1; \
		} \
		reverses_lth  = max_reverses * sizeof(reverseT); \
		reversesP     = (reverseT *) Xrealloc(reversesP, reverses_lth); \
		if (!reversesP) { \
			fprintf(stderr, "Hillclimb: Reversals outofmemory\n"); \
			max_reverses  = 0; \
			end_reversesP = 0; \
			reverses_maxP = 0; \
			goto abort_reversals; \
		} \
		reverses_maxP = reversesP + max_reverses; \
		end_reversesP = reversesP + at; \
	} \
	Xcheck(reversesP, reverses_lth, end_reversesP); \
	end_reversesP->edgeP   = X; \
	end_reversesP->matched = X->matched; \
	++end_reversesP; 

	static	unblockedT	*unblockedP = 0;
	static  int			unblocked_lth = 0;
	static	unblockedT	*unblocked_maxP = 0;
	static	unsigned int	 max_unblocked  = 0;
	static	unblockedT	*end_unblockedP = 0;

	// Array of dll's for each block which we wish to handle
	static	dllT 	*dllsP     = 0;
	static	int		dlls_lth   = 0;
	static	dllT	*end_dllsP = 0;

	// Linked list of edges whose blocks value changes
	static	edgeT	*changesP  = 0;

	static	int		iterations;

	static codeT	*code1aP, *code2aP, *code1bP, *code2bP;
	static crossT	*cross1P, *cross2P, *cross3P;
	static crossT	*end_cross1P, *end_cross2P, *end_cross3P;
	static edgeT	*edge1P, *next_edge1P;
	static edgeT	*edge2P, *edge3P, *edge4P;
	static unblockedT *unblocked1P, *unblocked_bestP;

	static int		at, phase, max_crossings, change, old_ret, ret;
	static int		weight;
	static int		matched;
	static reverseT *reverseP;
	static int		blocks;
	static dllT		*headP, *cycleP;
	static unsigned int crosses;
	static time_t	hillclimb_started;
	static time_t	hillclimb_terminate;
#ifdef TEST_HILLCLIMB
	static int		original_matches, final_matches;
#endif

	hillclimb_started   = time(0);
	g_hillclimb_elapse -= hillclimb_started;
	if (g_hillclimb_timeout) {
		hillclimb_terminate = hillclimb_started + g_hillclimb_timeout;
	} else {
		hillclimb_terminate = 0;
	}

#ifdef DEBUG_HILLCLIMB
	fprintf(stderr, "Hill Climb %p %d %p %d\n", 
		start_code1P, 
		best_end1P - start_code1P,
		start_code2P, 
		best_end2P - start_code2P);
#endif
#ifdef TEST_HILLCLIMB
	original_matches = 0;
	for (code1aP = start_code1P; code1aP <= best_end1P; ++code1aP) {
		if (code1aP->m_matchP) {
			++original_matches;
	}	}
#endif

	if (!dllsP) {
		dlls_lth  = g_hillclimb * sizeof(dllT);
		dllsP     = (dllT *) Xmalloc(dlls_lth);
		if (!dllsP) {
			fprintf(stderr, "hillclimbing: dlls outofmemory\n");
			return(0);
		}
		end_dllsP = dllsP + g_hillclimb;
	}

	// Find all possible edges

	end_edgesP       = edgesP;

	// Compute all the edges
	// These may change for jumps as matchings change

	// Can't change first line
	change    = 0;
	for (code1aP = start_code1P + 1; code1aP <= best_end1P; ++code1aP) {
		for (code2aP = start_code2P + 1; code2aP <= best_end2P; ++code2aP) {
			// Add the edge if appropriate
			if (code1aP->m_matchP == code2aP) {
				matched = 1;
			} else {
				if (!same_code(code1aP, code2aP, &weight)) {
					// Not appropriate
					continue;
				}
				matched = 0;
				change  = 1;
			}
			if (edges_maxP <= end_edgesP) {
				at = max_edges;
				if (!max_edges) {
					max_edges   = (1<<11);
				} else {
					max_edges <<= 1;
				}
				edges_lth  = max_edges * sizeof(edgeT);
				edgesP     = (edgeT *) Xrealloc(edgesP, edges_lth);
				if (!edgesP) {
					max_edges = 0;
					goto too_many_edges;
				}
				edges_maxP = edgesP + max_edges;
				end_edgesP = edgesP + at;
			}
			Xcheck(edgesP, edges_lth, end_edgesP);
			memset(end_edgesP, 0, sizeof(edgeT));
			end_edgesP->code1P       = code1aP;
			end_edgesP->code2P       = code2aP;
			end_edgesP->matched      = matched;
			end_edgesP->next_changeP = (edgeT *) -1;
			++end_edgesP;
	}	}

	if (!change) {
		// No unmatched edges
		ret = 0;
		goto done;
	}

	if ((end_edgesP - edgesP) > 0xFFFF) {
too_many_edges:
		// Can't handle the edge offsets in a short
		fprintf(stderr,
			"* Hill-climbing aborted: too many edges\n"
			"* %s:%s .v.\n* %s:%s\n",
			source1P->m_fileP->m_nameP,
			function1P->m_nameP,
			source2P->m_fileP->m_nameP,
			function2P->m_nameP);
		fprintf(stderr, "* Clone lths={%d,%d}, edges=%d\n",
			(int) (best_end1P - start_code1P),
			(int) (best_end2P - start_code2P),
			(int) (end_edgesP - edgesP));
		ret = 0;
		goto done;
	}

	// Add edge crossings

	crosses = 0;
	for (phase = 0; ; ++phase) {
		for (edge1P = edgesP; edge1P < end_edgesP; ++edge1P) {
			code1aP = edge1P->code1P;
			code2aP = edge1P->code2P;

			// Treat an edge point match as a cross
			for (edge2P = edge1P; ++edge2P < end_edgesP; ) {
				code1bP = edge2P->code1P;
				code2bP = edge2P->code2P;
				if ((code1aP >= code1bP && code2aP <= code2bP) ||
					(code1aP <= code1bP && code2aP >= code2bP)) {
					if (!phase) {
						++(edge1P->crosses);
						++(edge2P->crosses);
						crosses += 2;
					} else {
						--(edge1P->crossesP);
						*edge1P->crossesP = (edge2P - edgesP);
						--(edge2P->crossesP);
						*edge2P->crossesP = (edge1P - edgesP);
						if (edge1P->matched) {
							assert(!edge2P->matched);
							++(edge2P->blocks);
						} else if (edge2P->matched) {
							++(edge1P->blocks);
		}	}	}	}	}
		if (phase) {
			break;
		}
		if (!crosses) {
			ret = 0;
			goto done;
		}
		if (crosses > 0x40000000) {	// 1,073,741,824
too_many_crosses:
			fprintf(stderr,
				"* Hill-climbing aborted: too many crosses\n"
				"* %s:%s .v.\n* %s:%s\n",
				source1P->m_fileP->m_nameP,
				function1P->m_nameP,
				source2P->m_fileP->m_nameP,
				function2P->m_nameP);
			fprintf(stderr, "* Clone lths={%d,%d}, edges=%d crosses=%u\n",
				(int) (best_end1P - start_code1P),
				(int) (best_end2P - start_code2P),
				(int) (end_edgesP - edgesP),
				crosses);
			ret = 0;
			goto done;
		}

		if (max_crosses < crosses) {
			if (!max_crosses) {
				max_crosses = (1<<18);	// Must be multiple of 2
			} else {
				Xfree(crossesP);
			}
			while (max_crosses < crosses) {
				max_crosses <<= 1;
			}
			crossesP = (crossT *) Xmalloc(max_crosses * sizeof(unsigned short));
			if (!crossesP) {
				max_crosses = 0;
				goto too_many_crosses;
		}	}
		end_cross1P = crossesP + crosses;
		for (edge1P = end_edgesP; edgesP <= --edge1P; ) {
			edge1P->crossesP = end_cross1P;
			end_cross1P     -= edge1P->crosses;
		}
		assert(end_cross1P == crossesP);
	}

	for (headP = dllsP; headP < end_dllsP; ++headP) {
		Xcheck(dllsP, dlls_lth, headP);
		headP->nextP = headP;
		headP->prevP = headP;
	}

	for (edge2P = edgesP; edge2P < end_edgesP; ++edge2P) {
#ifdef TEST_HILLCLIMB
		assert(edge2P->blocks || edge2P->matched);
#endif
		blocks = edge2P->blocks;
		if (blocks > g_max_blocks_seen) {
			g_max_blocks_seen = blocks;
		}
		Xcheck(edgesP, edges_lth, edge2P);
		if (blocks && blocks <= g_hillclimb) {
			// Add to the start of the cycle
			--blocks;
			headP                    = dllsP + blocks;
			edge2P->dll.prevP        = headP;
			edge2P->dll.nextP        = headP->nextP;
			Xcheck(dllsP, dlls_lth, headP);
			headP->nextP             = (dllT *) edge2P;
			edge2P->dll.nextP->prevP = (dllT *) edge2P;
		} else {
			edge2P->dll.nextP        = 0;
			edge2P->dll.prevP        = 0;
	}	}

#ifdef DEBUG_HILLCLIMB
	fprintf(stderr, "%d edges %d edge crossings %d code1=%p %d code2=%p\n", 
				end_edgesP - edgesP,
				crosses,
				best_end1P - start_code1P,
				start_code1P,
				best_end2P - start_code2P,
				start_code2P);

	for (edge1P = edgesP; edge1P < end_edgesP; ++edge1P) {
		fprintf(stderr, "%d] %d %d<->%d %s<=>%s ", 
					edge1P - edgesP,
					edge1P->blocks,
					edge1P->code1P - start_code1P,
					edge1P->code2P - start_code2P,
					edge1P->code1P->m_codeP,
					edge1P->code2P->m_codeP);
		if (edge1P->matched) {
			fprintf(stderr, " match");
		}
		cross1P = edge1P->crossesP;
		for (end_cross1P = cross1P + edge1P->crosses; cross1P < end_cross1P; ++cross1P) {
			fprintf(stderr, " %hu", *cross1P);
		}
		fprintf(stderr, "\n");
	}
#endif

	old_ret = ret  = 0;
	iterations     = 0;
	for (max_crossings = 0; ; ++max_crossings) {
		if (max_crossings >= g_hillclimb) {
			if (ret == old_ret || iterations >= g_iterations) {
				if (iterations > g_max_iterations_seen) {
					g_max_iterations_seen = iterations;
				}
				break;
			}
			++iterations;
			old_ret        = ret;
			max_crossings  = 0;
			// All but first iteration consider all edges
		}

		cycleP = dllsP + max_crossings;
		for (edge1P = (edgeT *) cycleP->nextP; cycleP != (dllT *) edge1P; edge1P = next_edge1P) {
			next_edge1P = (edgeT *) edge1P->dll.nextP;
			assert(!edge1P->matched);
			assert(edge1P->blocks == max_crossings + 1);

#ifdef TEST_HILLCLIMB
			{
				int 	cnt;
				edgeT	*edge5P;
		
				for (edge4P = edgesP; edge4P < end_edgesP; ++edge4P) {
					code1aP = edge4P->code1P;
					code2aP = edge4P->code2P;
					if (edge4P->matched) {
						assert(code1aP->m_matchP == code2aP);
						assert(code2aP->m_matchP == code1aP);
						assert(!edge4P->blocks);
						continue;
					}
					assert(code1aP->m_matchP != code2aP);
					assert(code2aP->m_matchP != code1aP);
				
					cnt = 0;
					cross1P = edge4P->crossesP;
					for (end_cross1P = cross1P + edge4P->crosses; cross1P < end_cross1P; ++cross1P) {
						edge5P = edgesP + *cross1P;
						if (edge5P->matched) {
							++cnt;
					}	}
					assert(cnt == edge4P->blocks);
			}	}

			for (headP = dllsP; headP < end_dllsP; ++headP) {
				for (edge4P = (edgeT *) headP->nextP; headP != (dllT *) edge4P; edge4P = (edgeT *) edge4P->dll.nextP) {
					assert(edge4P->blocks == (headP - dllsP) + 1);
			}	}
#endif

			// Match1 is currently not an active edge
			// Compute the change in edges if all of the active edges crossing edge1P are made inactive
			// and edge1P is made active

#ifdef DEBUG_HILLCLIMB
			fprintf(stderr, "Considering activating %d<->%d\n",
								edge1P->code1P - start_code1P,
								edge1P->code2P - start_code2P);

			fprintf(stderr, "Initial add to reverse %p\n", edge1P);
#endif

			/* Edge 1 will become active if we commit this change which is
			 * an improvement of 1
			 */
			end_reversesP  = reversesP;
			change         = 1;
			/* Linked list of changed block numbers initially empty */
			changesP       = 0;
			end_unblockedP = unblockedP;

			cross1P = edge1P->crossesP;
			for (end_cross1P = cross1P + edge1P->crosses; cross1P < end_cross1P; ++cross1P) {
				/* Edge1 crosses edge2 */
				edge2P = edgesP + *cross1P;

				// Edge 1 will block this edge
				Increment(edge2P);

				if (!edge2P->matched) {
					continue;
				}
				// We have incremented all counts
				assert(edge2P->blocks1 == 1);
				// We must unactivate edge2P to permit activation of edge1
				Reverse(edge2P);
				// This is a match we loose
				--change;	
				cross2P = edge2P->crossesP;
				for (end_cross2P = cross2P + edge2P->crosses; cross2P < end_cross2P; ++cross2P) {
					// Because cross2 is inactive all edge3 crossing it are
					// no longer blocked by it
					edge3P = edgesP + *cross2P;
					assert(!edge3P->matched);
					Decrement(edge3P);
					if (edge3P->blocks1) {
						continue;
					}
					if (unblocked_maxP <= end_unblockedP) {
						at = max_unblocked;
						if (!max_unblocked) {
							max_unblocked   = 256;
						} else {
							max_unblocked <<= 1;
						}
						unblocked_lth = max_unblocked * sizeof(unblockedT);
						unblockedP    = (unblockedT *) Xrealloc(unblockedP, unblocked_lth);
						if (!unblockedP) {
							fprintf(stderr, "Hillclimb: unblocked outofmemory\n");
							max_unblocked = 0;
							goto abort_reversals;
						}
						unblocked_maxP = unblockedP + max_unblocked;
						end_unblockedP = unblockedP + at;
					}
					Xcheck(unblockedP, unblocked_lth, end_unblockedP);
					end_unblockedP->edgeP  = edge3P;
					++end_unblockedP;
			}	}

			/* We have now inactivated all edges crossing edge1 */
			assert(!(edge1P->blocks1));
			Reverse(edge1P);
			// Avoid allowing edge1P to match twice
			Increment(edge1P);

			for (;;) {
				for (unblocked1P = unblockedP; unblocked1P < end_unblockedP; ) {
					edge2P = unblocked1P->edgeP;
					if (edge2P->blocks1) {
						// No longer unblocked
						--end_unblockedP;
						*unblocked1P = *end_unblockedP;
						continue;
					} 
					edge2P->matched |= 2;
					++unblocked1P;
				}
				if (unblockedP == end_unblockedP) {
					break;
				}
				for (unblocked1P = unblockedP; unblocked1P < end_unblockedP; ++unblocked1P) {
					blocks  = 0;
					edge2P  = unblocked1P->edgeP;
					cross2P = edge2P->crossesP;
					for (end_cross2P = cross2P + edge2P->crosses; cross2P < end_cross2P; ++cross2P) {
						edge3P = edgesP + *cross2P;
						if (edge3P->matched & 2) {
							++blocks;
					}	}
					unblocked1P->blocks = blocks;
				}
				unblocked_bestP = unblocked1P = unblockedP;
				for (; unblocked1P < end_unblockedP; ++unblocked1P) {
					if (unblocked1P->blocks < unblocked_bestP->blocks) {
						unblocked_bestP = unblocked1P;
					}
					edge2P           = unblocked1P->edgeP;
					edge2P->matched &= ~2;
				}

				edge3P = unblocked_bestP->edgeP;
				--end_unblockedP;
				*unblocked_bestP = *end_unblockedP;

				assert(!edge3P->matched);
				assert(!edge3P->blocks1);

				// We will activate edge3P
				++change;	
				Reverse(edge3P);
				Increment(edge3P);	// Avoid choosing again

				cross3P = edge3P->crossesP;
				for (end_cross3P = cross3P + edge3P->crosses; cross3P < end_cross3P; ++cross3P) {
					// Edge3 activation blocks all things it crosses
					edge4P = edgesP + *cross3P;
					Increment(edge4P);
				}
			}
			if (change <= 0) {
				// This change does no good
				// Put all next_changeP pointers back to -1
	
abort_reversals:
				for (edge2P = changesP; edge2P; edge2P = edge3P) {
					edge3P               = edge2P->next_changeP;
					edge2P->next_changeP = (edgeT *) -1;
				}
			} else {

#ifdef DEBUG_HILLCLIMB
				fprintf(stderr, "Found a change of %d in edge %d-%d\n",
							change,
							edge1P->code1P - start_code1P,
							edge1P->code2P - start_code2P);
#endif

				ret += change;

				// Apply all the reversals

				for (reverseP = reversesP; reverseP < end_reversesP; ++reverseP) {
					edge2P  = reverseP->edgeP;
					code1bP = edge2P->code1P;
					code2bP = edge2P->code2P;
#ifdef TEST_HILLCLIMB
					if (edge2P->matched != reverseP->matched) {
						reverseT	*reverse1P;

						for (reverse1P = reversesP; reverse1P <= reverseP; ++reverse1P) {
							edge3P = reverse1P->edgeP;
							fprintf(stderr, "[%d] %d<->%d ",
								edge3P - edgesP,
								edge3P->code1P - start_code1P,
								edge3P->code2P - start_code2P);
							if (reverse1P->matched) {
								fprintf(stderr, " unmatch\n");
							} else {
								fprintf(stderr, " match\n");
						}	}
						assert(edge2P->matched == reverseP->matched);
					}
#endif
					if (edge2P->matched) {
#ifdef DEBUG_HILLCLIMB
						fprintf(stderr, "Unmatch %d<->%d\n",
							code1bP - start_code1P,
							code2bP - start_code2P);
#endif
						// Unmatch this edge
						assert(code1bP->m_matchP == code2bP);
						assert(code2bP->m_matchP == code1bP);
						code1bP->m_matchP = 0;
						code2bP->m_matchP = 0;
						edge2P->matched   = 0;
					} else {
#ifdef DEBUG_HILLCLIMB
						fprintf(stderr, "Match %d<->%d\n",
							code1bP - start_code1P,
							code2bP - start_code2P);
#endif
						// Match this edge
						assert(code1bP->m_matchP == 0);
						assert(code2bP->m_matchP == 0);
						code1bP->m_matchP = code2bP;
						code2bP->m_matchP = code1bP;
						edge2P->matched   = 1;
						--(edge2P->blocks1); // Undo the blocking ++
					}
				}
				// Having applied all the reversals correct
				// The blocks crossed in all affected edges
				for (edge2P = changesP; edge2P; edge2P = edge3P) {
					edge3P         = edge2P->next_changeP;
					edge2P->blocks = blocks = edge2P->blocks1;
					if (blocks > g_max_blocks_seen) {
						g_max_blocks_seen = blocks;
					}
					edge2P->next_changeP = (edgeT *) -1;
					if ((headP = edge2P->dll.nextP)) {
						// Remove edge2 from dll's
						if (edge2P == next_edge1P) {
							next_edge1P = (edgeT *) headP;
						}
						headP->prevP             = edge2P->dll.prevP;
						edge2P->dll.prevP->nextP = headP;
					}
					if (blocks && blocks <= g_hillclimb) {
						// Add to the start of the right dll
						// We won't see it this iteration if adding it to
						// our cycle because before us in the cycle
						--blocks;
						headP                    = dllsP + blocks;
						edge2P->dll.prevP        = headP;
						edge2P->dll.nextP        = headP->nextP;
						headP->nextP             = (dllT *) edge2P;
						edge2P->dll.nextP->prevP = (dllT *) edge2P;
					} else {
						edge2P->dll.nextP        = 0;
						edge2P->dll.prevP        = 0;
					}
				}
			}


#ifdef TEST_HILLCLIMB
			for (headP = dllsP; headP < end_dllsP; ++headP) {
				for (edge4P = (edgeT *) headP->nextP; headP != (dllT *) edge4P; edge4P = (edgeT *) edge4P->dll.nextP) {
					assert(edge4P->blocks == (headP - dllsP) + 1);
			}	}
#endif
			if (hillclimb_terminate && hillclimb_terminate < time(0)) {
				fprintf(stderr,
					"* Hill-climbing aborted: timeout %d secs\n"
					"* %s:%s .v.\n* %s:%s\n",
					g_hillclimb_timeout,
					source1P->m_fileP->m_nameP,
					function1P->m_nameP,
					source2P->m_fileP->m_nameP,
					function2P->m_nameP);
				fprintf(stderr, "* Clone lths={%d,%d}, edges=%d crosses=%u\n",
					(int) (best_end1P - start_code1P),
					(int) (best_end2P - start_code2P),
					(int) (end_edgesP - edgesP),
					crosses);
			
				goto terminate;
			}

	}	}

terminate:
	for (; !(best_end2P = best_end1P->m_matchP); --best_end1P);
#ifdef TEST_HILLCLIMB
	final_matches = 0;
	for (code1aP = start_code1P; code1aP <= best_end1P; ++code1aP) {
		if (code1aP->m_matchP) {
			++final_matches;
	}	}
	assert(final_matches - original_matches == ret);
#endif
done:
	g_hillclimb_elapse += time(0);
	return(ret);
}

extern int				g_max_function_lth;

static codeT *
detect_clone(void)
{
	static	codeT		*code1P, *code2P, *at_code1P, *at_code2P;
	static	codeT		*max_end1P, *max_end2P, *last_match1P;
	static	int			max_best, ret, weight, improved;
	static	int			original_matches, extra_matches, cancelled_matches;
	static  int			distance;

#ifdef DEBUG_DETECT
	static int debug_cnt = 0;

	fprintf(stderr, "%d Comparing %s(%d-%d) and %s(%d-%d)\n  %s\n  %s\n",  
		++debug_cnt,
		function1P->m_nameP,
		start_code1P - function_code1P,
		end_code1P - function_code1P,
		function2P->m_nameP,
		start_code2P - function_code2P, 
		end_code2P - function_code2P, 
		start_code1P->m_codeP, 
		start_code2P->m_codeP);
#endif
	start_code1P->m_matchP = start_code2P;
	start_code2P->m_matchP = start_code1P;
	last_match1P           = start_code1P;
	if (!same_code(start_code1P, start_code2P, &weight)) {
		goto fail;
	}
	max_best            = weight;
	max_end1P           = start_code1P;
	max_end2P           = start_code2P;

	start_clock = time(0);

	// Initially have to have a match
	best_end1P = code1P = start_code1P;
	best_end2P = code2P = start_code2P;
	original_matches    = 1;
	extra_matches       = 0;

extend_clone:
	improved = 0;
	for (; ++code1P < end_code1P && ++code2P < end_code2P;) {
		for (distance = 0; weight >= 0; weight += g_mismatch_cost) {
			at_code1P = code1P;
			at_code2P = code2P + distance;
			if (at_code2P >= end_code2P) {
				at_code1P += (at_code2P - end_code2P + 1);
				if (at_code1P >= end_code1P) {
					break;
				}
				at_code2P  = end_code2P - 1;
			} 
			while (at_code2P >= code2P && at_code1P < end_code1P) {
				if (same_code(at_code1P, at_code2P, &ret)) {
					goto found;
				}
				++at_code1P;
				--at_code2P;
			}
			++distance;
		}
		break;
found:	
		weight += ret;
		assert(weight >= 0);
		at_code1P->m_matchP = at_code2P;
		at_code2P->m_matchP = at_code1P;
		if (last_match1P < at_code1P) {
			last_match1P        = at_code1P;
		}
		improved            = 1;
		++original_matches;

		best_end1P  = at_code1P;
		best_end2P  = at_code2P;

		if (g_best_weight && (max_best <= weight)) {
			max_best  = weight;
			max_end1P = best_end1P;
			max_end2P = best_end2P;
		}
		code1P      = at_code1P;
		code2P      = at_code2P;
	}

	if (improved) {
		if (g_hillclimb > 0) {
			ret = hill_climb();
			if (ret > 0) {
				extra_matches += ret;
				weight += (ret * g_same_cost);
				code1P = best_end1P; 
				code2P = best_end2P;
				goto extend_clone;
	}	}	}
	
	if (g_best_weight) {
		weight     = max_best;
		best_end1P = max_end1P;
		best_end2P = max_end2P;
	}

	++best_end1P;
	++best_end2P;

	g_lth1     = best_end1P - start_code1P;
	g_lth2     = best_end2P - start_code2P;

	if (g_ignore_macros) {
		Cfile	*fileP;

		fileP  = 0;
		for (code1P = start_code1P; code1P < end_code1P; ++code1P) {
			if ((code2P = code1P->m_matchP)) {
				if (code1P->m_fileP         == code2P->m_fileP &&
					code1P->m_source_lineno == code2P->m_source_lineno) {
					if (fileP == code1P->m_fileP) {
						--g_lth1;
						--g_lth2;
						continue;
					}
					fileP = code1P->m_fileP;
					continue;
			}	}
			fileP = 0;
	}	}

	if (!permitted_clone()) {
		goto fail;
	}
	cancelled_matches = 0;
	for (code1P = start_code1P; code1P <= last_match1P; ++code1P) {
		at_code1P = code1P->m_jumpP;
		if (at_code1P <= code1P) {
			continue;
		}
		if (!(code2P = code1P->m_matchP)) {
			continue;
		}
		at_code2P = code2P->m_jumpP;
		if (at_code1P >= best_end1P) {
			if (at_code2P >= best_end2P) {
				// Both branch beyond end of clone
				// Might be turned into a return
				continue;
			}
		} else if (at_code2P < best_end2P) {
			if (at_code1P->m_matchP) {
				if (at_code2P == at_code1P->m_matchP) {
					// Both branch to the same matched instruction
					continue;
			}	}
			while (!((--at_code1P)->m_matchP));
			while (!((--at_code2P)->m_matchP));
			if (at_code1P->m_matchP == at_code2P) {
				// Both branch after the same matched instruction
				continue;
		}	}
		++cancelled_matches;
		if (code1P == start_code1P) {
			goto fail;
		}
		code1P->m_matchP = 0;
		code2P->m_matchP = 0;
	}

	g_original_matches  += original_matches;
	g_extra_matches     += extra_matches;
	g_cancelled_matches += cancelled_matches;
	end_clock = time(0);

#ifdef PROOF
	proof();
#endif

	report_clone();
	for (at_code1P = start_code1P; at_code1P <= last_match1P; ++at_code1P) {
		if ((at_code2P = at_code1P->m_matchP)) {
			Cmatch::add(at_code1P - function_code1P, at_code2P);
	}	}
done:
	for (; last_match1P >= start_code1P; --last_match1P) {
		if ((at_code2P = last_match1P->m_matchP)) {
			at_code2P->m_matchP    = 0;
			last_match1P->m_matchP = 0;
	}	}
	return(best_end2P);
fail:
	best_end2P = 0;
	goto done;
}

/*
void
trap_detect(void)
{
}
*/

void
detect(void)
{
	static	int				function_start_code, function_end_code;
	static	Cfunction		*function3P;
	static	Cfile			*fileP;	
	static	codeT			*codes1P, *codes2P, *skipP, *lastP;
	static	time_t 			clock_now;
	static	char			*P;
	static	int				source_cnt = 0;
#ifdef EXPLORE_SPEED
	static	long long		last_match_comparisons, last_comparisons;
	static	long long		last_match_comparisons1, last_comparisons1;
	static	int				clones_matched, codes_matched, rejected;
	static	time_t			clock_now1;
#endif

	Cmatch::init();

	for (source1P = Csource::g_headP; source1P; source1P = source1P->m_nextP) {
		if (g_verbose) {
			
			clock_now = time(0);
			P         = ctime(&clock_now);
			P[19]     = 0;
			fprintf(stderr, "%d) %s %s\n", ++source_cnt, P+11, source1P->m_fileP->m_nameP);
#ifdef EXPLORE_SPEED
			last_match_comparisons = Cmatch::m_comparisons;
			last_comparisons       = g_comparisons;
#endif
		}
		codes1P = source1P->m_codesP;
		for (function1P = source1P->m_functionsP; function1P; function1P = function1P->m_nextP) {
#ifdef EXPLORE_SPEED
			if (source_cnt == 28) {
				fprintf(stderr, "**function %s\n", function1P->m_nameP);
			}
#endif
			function_start_code = function1P->m_start_code;
			function_end_code   = function1P->m_end_code;
			start_code1P        = function_code1P = codes1P + function_start_code;
			function_end1P      = codes1P + function_end_code;

			Cmatch::setup(function_end_code - function_start_code);
			
			for ( ; start_code1P < function_end1P; ++start_code1P) {
				if (g_match_lines) {
					if (start_code1P != function_code1P && (fileP = start_code1P->m_fileP)) {
						lastP = start_code1P - 1;
						if (lastP->m_fileP == fileP && lastP->m_source_lineno == start_code1P->m_source_lineno) {
							continue;
				}	}	}
#ifdef EXPLORE_SPEED
				if (source_cnt == 28) {
					last_match_comparisons1 = Cmatch::m_comparisons;
					last_comparisons1       = g_comparisons;
					clock_now1              = time(0);
					clones_matched          = 0;
					codes_matched           = 0;
					rejected                = 0;
				}
#endif

				for (start_code2P = start_code1P->m_hashP; start_code2P; ) {
					if (!g_match_macros &&
						start_code1P->m_fileP &&
						start_code2P->m_fileP == start_code1P->m_fileP &&
						start_code2P->m_source_lineno == start_code1P->m_source_lineno) {
						// Obviously same lines will match trivially
						goto advance;
					}
					function2P      = start_code2P->m_functionP;
					source2P        = function2P->m_sourceP;
					codes2P         = source2P->m_codesP;

					// Detect subsequence within larger sequences already matched
					if (Cmatch::test(start_code1P - function_code1P, start_code2P)) {
#ifdef EXPLORE_SPEED
						++rejected;
#endif
						goto advance;
					}

					function_code2P = codes2P + function2P->m_start_code;
					function_end2P  = codes2P + function2P->m_end_code;
					if (function2P == function1P) {
						end_code2P = function_end1P;
						end_code1P = start_code2P;
					} else {
						end_code1P = function_end1P;
						end_code2P = function_end2P;
					}
					g_lth1 = end_code1P - start_code1P;
					g_lth2 = end_code2P - start_code2P;
					if (!permitted_clone()) {
						goto advance;
					}
					++g_comparisons;
					skipP         = detect_clone();
#ifdef EXPLORE_SPEED
					if (skipP) {
						++clones_matched;
						codes_matched += skipP - start_code2P + 1;
					}
#endif
					start_code2P  = start_code2P->m_hashP;

					if (skipP) {
						for (; start_code2P; start_code2P = start_code2P->m_hashP) {
							function3P = start_code2P->m_functionP;
							if (function3P != function2P || start_code2P >= skipP) {
								break;
					}	}	}
					continue;
advance:			start_code2P = start_code2P->m_hashP;
				}
#ifdef EXPLORE_SPEED
				if (source_cnt == 28) {
					fprintf(stderr, "%d %s %d %lld/%d %lld %d/%d\n",
						start_code1P->m_source_lineno,
						start_code1P->m_codeP,
						time(0) - clock_now1,
						Cmatch::m_comparisons - last_match_comparisons1,
						rejected, 
						g_comparisons - last_comparisons1,
						clones_matched,
						codes_matched
					);
				}
#endif
			}

			Cmatch::clear();
		}
#ifdef EXPLORE_SPEED
		if (g_verbose) {
			clock_now = time(0) - clock_now;
			fprintf(stderr, "Elapse=%d Codes=%d Match=%lld/%lld Comparisons=%lld/%lld/%lld\n",
					clock_now,
					source1P->m_codes,
					Cmatch::m_comparisons - last_match_comparisons,
					Cmatch::m_comparisons,
					g_comparisons - last_comparisons,
					g_comparisons,
					(clock_now ? (g_comparisons - last_comparisons) / ((long long) clock_now) : 0));
		}
#endif
}	}

