/* This code exists purely for research purposes.  When the -z option is
 * enabled this code is invoked with the intent of comparing the behaviour
 * of ACD to some other clone detection tool.  It is of no general use
 * since the inputs are hard coded into it.  Only by changing this code
 * on a needs basis might it serve any useful purpose to the general
 * user.
 */

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include "object.h"
#include "xmalloc.h"
#include "util.h"
#include "file.h"
#include "location.h"
#include "code.h"
#include "data.h"
#include "function.h"
#include "source.h"

#ifdef CORRELATE

#define SOURCE_TYPE 	0
#define ACD_TYPE    	1
#define CLIC_TYPE		2
#define OUTSIDE_BUILD	4

typedef struct {
	int		id;
	int		lines;
	char	*nameP;
	int		type;
	Csource	*sourceP;
} source_dataT;

typedef struct {
	int	lineno;
	int unknown1;
	int unknown2;
} line_dataT;

typedef struct {
	int			clone1_file;
	line_dataT	start_clone1;
	line_dataT	end_clone1;
	int			clone2_file;
	line_dataT	start_clone2;
	line_dataT	end_clone2;
	int			weight;
	int			overlap;
	Cfunction	*function1P;
	Cfunction	*function2P;
} clone_dataT;

#include "git-data30.h"
//#include "git-data60.h"
//#include "git-data69.h"

static FILE *dataF;
static FILE *missedF;

extern	char	*g_html_dirP;
extern int		g_min_match_sequence;
extern int		g_min_function_sequence;
extern int		g_functions;

static int		 g_outside_build = 0;
static int		 max_code_lth  = 0;
static long long sum_code_lth  = 0;
static long long sum_code2_lth = 0;

#define SOURCES 		((sizeof(source_data)/sizeof(source_data[0]))-1)
#define FUNCTION_PAIRS 	g_functions*(g_functions+1)/2

typedef struct {
	int					start;	// Lineno
	int					end;	// Lineno
} rangeT;

typedef struct boxS {
	struct boxS *nextP;
	rangeT		clone1;
	rangeT		clone2;
	int			type;
} boxT;

static boxT	**function_pairsPP = 0;

void
init_data(void)
{
	source_dataT	*source_dataP;
	Csource			*sourceP;
	Cfunction		*functionP;
	char			*nameP;
	clone_dataT		clone_data1, *clone_dataP, *end_clonesP;
	long long		lth1;
	int				lineno;

	sprintf(g_filename,"%s/correlated.html", g_html_dirP);
	dataF = open_file(g_filename);
	sprintf(g_filename,"%s/missed.html", g_html_dirP);
	missedF = open_file(g_filename);
	fprintf(dataF, HTML_HEADER "<title>Correlations</title>\n<body>\n<h3>Correlations</h3>\n<p>\nMin lengths=%d/%d\n<p>\n<a href=missed.html>missed</a><p>\n", g_min_match_sequence, g_min_function_sequence);
	fprintf(missedF, HTML_HEADER "<title>ACD missed these clones</title>\n<body>\n<h3>ACD missed these clones</h3>\n<p>\n");

	for (source_dataP = source_data; (nameP = source_dataP->nameP); ++source_dataP) {

		if (!Cfile::exists(nameP)) {
			if (!g_outside_build) {
				fprintf(missedF, "<h3>Outside build</h3>\n<p>\n");
			}
			++g_outside_build;
			fprintf(missedF, "%s\n<BR>\n", nameP);
			source_dataP->type = OUTSIDE_BUILD;
			continue;
		}
		lth1           = source_dataP->lines;
		if (lth1 > max_code_lth) {
			max_code_lth = lth1;
		}
		sum_code_lth  += lth1;
		sum_code2_lth += (lth1 * lth1);

		for (sourceP = Csource::g_headP; ; sourceP = sourceP->m_nextP) {
			if (!sourceP) {
				assert(0);
				break;
			}
			if (!strcmp(sourceP->m_fileP->m_nameP, nameP)) {
				source_dataP->sourceP = sourceP;
				break;
	}	}	}

	lth1           = sizeof(clone_data)/sizeof(clone_data[0]);
	clone_dataP    = clone_data;
	end_clonesP    = clone_data + lth1;
	for (; clone_dataP < end_clonesP; ++clone_dataP) {
		// Change from offset 0 to offset 1 to agree with ACD
		clone_dataP->start_clone1.lineno++;
		clone_dataP->end_clone1.lineno++;
		clone_dataP->start_clone2.lineno++;
		clone_dataP->end_clone2.lineno++;

		clone_dataP->function1P = 0;
		clone_dataP->function2P = 0;
		sourceP = source_data[clone_dataP->clone1_file].sourceP;
		if (sourceP) {
			lineno  = clone_dataP->start_clone1.lineno;
			for (functionP = sourceP->m_functionsP; ; functionP = functionP->m_nextP) {
				if (!functionP) {
					assert(0);
					break;
				}
				if (functionP->m_source_lineno <= lineno) {
					clone_dataP->function1P = functionP;
					break;
		}	}	}

		sourceP = source_data[clone_dataP->clone2_file].sourceP;
		if (sourceP) {
			lineno  = clone_dataP->start_clone2.lineno;
			for (functionP = sourceP->m_functionsP; ; functionP = functionP->m_nextP) {
				if (!functionP) {
					assert(0);
					break;
				}
				if (functionP->m_source_lineno <= lineno) {
					clone_dataP->function2P = functionP;
					break;
		}	}	}
		if (clone_dataP->function1P->m_number < clone_dataP->function2P->m_number) {
			continue;
		}
		if (clone_dataP->function1P == clone_dataP->function2P) {
			if (clone_dataP->start_clone1.lineno <= clone_dataP->start_clone2.lineno) {
				continue;
		}	}
		clone_data1               = *clone_dataP;
		clone_dataP->clone1_file  = clone_data1.clone2_file;
		clone_dataP->start_clone1 = clone_data1.start_clone2;
		clone_dataP->end_clone1   = clone_data1.end_clone2;
		clone_dataP->clone2_file  = clone_data1.clone1_file;
		clone_dataP->start_clone2 = clone_data1.start_clone1;
		clone_dataP->end_clone2   = clone_data1.end_clone1;
		clone_dataP->function1P   = clone_data1.function2P;
		clone_dataP->function2P   = clone_data1.function1P;
	}
	lth1             = FUNCTION_PAIRS * sizeof(boxT *);
	function_pairsPP = (boxT **) Xmalloc(lth1);
	memset(function_pairsPP, 0, lth1);
}


static boxT	*g_box_freeP = 0;
static int	g_boxes_max  = 0;
static int	g_boxes      = 0;
static boxT	*g_boxesP    = 0;

static boxT *
get_box(int start1, int end1, int start2, int end2, boxT *afterP, int type)
{
	boxT	*boxP;

	if (g_box_freeP) {
		boxP        = g_box_freeP;
		g_box_freeP = boxP->nextP;
	} else {
		if (g_boxes_max <= g_boxes) {
			if (!g_boxes_max) {
				g_boxes_max   = 2<<18;
			} else {
				g_boxes_max <<= 1;
			}
			g_boxesP = (boxT *) Xrealloc(g_boxesP, g_boxes_max*sizeof(boxT));
		}
		boxP               = g_boxesP + g_boxes++;
	}
	assert(start1 > 0);
	assert(start2 > 0);
	assert(start1 <= end1);
	assert(start2 <= end2);
	boxP->clone1.start = start1;
	boxP->clone1.end   = end1;
	boxP->clone2.start = start2;
	boxP->clone2.end   = end2;
	boxP->type         = type;
	if (afterP) {
		boxP->nextP    = afterP->nextP;
		afterP->nextP  = boxP;
	} else {
		boxP->nextP    = 0;
	}
	return (boxP);
}

static void
free_box(boxT *boxP)
{
	boxP->nextP = g_box_freeP;
	g_box_freeP = boxP;
}

static void
compute_lineno(Cfunction *functionP, codeT *startP, codeT *endP, int *P, int *P1)
{
	Csource	*sourceP;
	Cfile	*fileP;
	codeT	*codesP, *codeP;
	codeT	*function_startP;
	int		start, end;

	sourceP = functionP->m_sourceP;
	fileP = sourceP->m_fileP;
	start = 0x7FFFFFFF;
	end   = -1;

	for (codeP = startP; codeP < endP; ++codeP) {
		if (codeP->m_fileP == fileP) {
			if (codeP->m_source_lineno < start) {
				start = codeP->m_source_lineno;
			}
			if (codeP->m_source_lineno > end) {
				end = codeP->m_source_lineno;
	}	}	}

	if (end < start) {
		codesP = sourceP->m_codesP;
		codeP  = startP;
		for (function_startP = sourceP->m_codesP + functionP->m_start_code; --codeP >= function_startP; ) {
			if (codeP->m_fileP == fileP) {
				start = end = codeP->m_source_lineno;
		}	}
	
		if (end < start) {
			start = end = functionP->m_source_lineno;
	}	}
	*P  = start;
	*P1 = end;	
}

static void
apply_box(boxT	*boxP, boxT	*headP)
{
	boxT	*targetP, *nextP;
	int		seen, left, right, top, bottom;

	for (; boxP; boxP = nextP) {
		seen = 0;
		for (targetP = headP; targetP; targetP = targetP->nextP) {
			if (boxP->clone1.end   < targetP->clone1.start ||
				boxP->clone1.start > targetP->clone1.end   ||
				boxP->clone2.end   < targetP->clone2.start ||
				boxP->clone2.start > targetP->clone2.end) {
				// No overlap
				continue;
			}
			seen   = 1;
			left   = boxP->clone1.start  - targetP->clone1.start;
			if (left < 0) {
				// Slice off the left part of box
				get_box(boxP->clone1.start,
						targetP->clone1.start - 1,
						boxP->clone2.start,
						boxP->clone2.end,
						boxP, boxP->type);
				boxP->clone1.start = targetP->clone1.start;
			} else if (left > 0) {	
				// Slice off the left part of target
				get_box(targetP->clone1.start,
						boxP->clone1.start - 1,
						targetP->clone2.start,
						targetP->clone2.end,
						targetP, targetP->type);
				targetP->clone1.start = boxP->clone1.start;
			}

			right  = targetP->clone1.end - boxP->clone1.end;
			if (right < 0) {
				// Slice off the right part of box
				get_box(targetP->clone1.end+1,
						boxP->clone1.end,
						boxP->clone2.start,
						boxP->clone2.end,
						boxP, boxP->type);
				boxP->clone1.end = targetP->clone1.end;
			} else if (right > 0) {	
				// Slice off the right part of target
				get_box(boxP->clone1.end+1,
						targetP->clone1.end,
						targetP->clone2.start,
						targetP->clone2.end,
						targetP, targetP->type);
				targetP->clone1.end = boxP->clone1.end;
			}

			top    = boxP->clone2.start  - targetP->clone2.start;
			if (top < 0) {
				// Slice off the top of the box
				get_box(boxP->clone1.start,
						boxP->clone1.end,
						boxP->clone2.start,
						targetP->clone2.start - 1,
						boxP, boxP->type);
				boxP->clone2.start = targetP->clone2.start;
			} else if (top > 0) {
				// Slice off the top of the target
				get_box(targetP->clone1.start,
						targetP->clone1.end,
						targetP->clone2.start,
						boxP->clone2.start - 1,
						targetP, targetP->type);
				targetP->clone2.start = boxP->clone2.start;
			}

			bottom = targetP->clone2.end - boxP->clone2.end;
			if (bottom < 0) {
				// Slice off bottom of box
				get_box(boxP->clone1.start,
						boxP->clone1.end,
						targetP->clone2.end + 1,
						boxP->clone2.end,
						boxP, boxP->type);
				boxP->clone2.end = targetP->clone2.end;
			} else if (bottom > 0) {
				// Slice off the bottom of the target
				get_box(targetP->clone1.start,
						targetP->clone1.end,
						boxP->clone2.end + 1,
						targetP->clone2.end,
						targetP, targetP->type);
				targetP->clone2.end = boxP->clone2.end;
			}
			targetP->type |= boxP->type;
			nextP          = boxP->nextP;
			free_box(boxP);
		}
		if (!seen) {
			fprintf(stderr, "Couldn't find target overlap for {%d-%d,%d-%d}\n",
				boxP->clone1.start, boxP->clone1.end, boxP->clone2.start, boxP->clone2.end);

			for (targetP = headP; targetP; targetP = targetP->nextP) {
				fprintf(stderr, "{%d\t%d\t%d\t%d}\n",
				targetP->clone1.start, targetP->clone1.end, targetP->clone2.start, targetP->clone2.end);
			}

			assert(0);
			error_exit(1);
		}
	}
}
			
static boxT *
get_function_pair(int i, int j)
{
	boxT	**function_pairPP, *targetP;

	assert(i <= j);
	function_pairPP = function_pairsPP + ((j * (j + 1))/2) + i;
	if (!(targetP = *function_pairPP)) {
		*function_pairPP = targetP = 
					get_box(1,
							source_data[i].lines,
							1,	
							source_data[j].lines,
							0, SOURCE_TYPE);
	}
	return(targetP);
}

static long long g_acd_pairs   = 0;
static long long g_acd_overlap = 0;
static long long g_clic_pairs  = 0;
static long long g_clic_overlap= 0;
static long long g_clic_bad    = 0;			
static long long g_area[4]     = {0};
static long long g_source[4]   = {0};
static long long g_total_area  = 0;
static long long sum_acd_lth   = 0;
static long long sum_acd2_lth  = 0;
static int		 max_acd_lth   = 0;
static long long cnt_acd_lth   = 0;
static long long sum_clic_lth  = 0;
static long long sum_clic2_lth = 0;
static long long cnt_clic_lth  = 0;
static int       max_clic_lth  = 0;

extern FILE	*stdoutF;

void
correlate_data(Cfunction *function1P, codeT *start1P, codeT *end1P, Cfunction *function2P, codeT *start2P, codeT *end2P)
{
	Cfunction	*functionP;
	codeT		*codeP;
	boxT		*targetP, *boxP;
	int			i, j, temp;
	int			start1, start2, end1, end2;
	clone_dataT	*clone_dataP, *endP;
	int			lth, seen;
	Cfile		*file1P, *file2P;
	const Cfile *default1P, *default2P;
	int		 	lineno1, lineno2, end_lineno1, end_lineno2;

	compute_lineno(function1P, start1P, end1P, &start1, &end1);
	compute_lineno(function2P, start2P, end2P, &start2, &end2);

	if (start1 < 0 || start2 < 0 || end1 < 0 || end2 < 0) {
		return;
	}

	i = function1P->m_number;
	j = function2P->m_number;

	if (i > j || ((i == j) && (start1 > start2))) {
		functionP  = function1P;
		function1P = function2P;
		function2P = functionP;
		codeP      = start1P;
		start1P    = start2P;
		start2P    = codeP;
		codeP      = end1P;
		end1P      = end2P;
		end2P      = codeP;
		temp       = i;
		i          = j;
		j          = temp;
		temp       = start1;
		start1     = start2;
		start2     = temp;
		temp       = end1;
		end1       = end2;
		end2       = temp;
	}

	// Data now in same ordering as clone data output

	++g_acd_pairs;

	file1P = function1P->m_sourceP->m_fileP;
	file2P = function2P->m_sourceP->m_fileP;

	seen           = 0;
	clone_dataP    = clone_data;
	endP    = clone_data + (sizeof(clone_data)/sizeof(clone_data[0]));
	for (; clone_dataP < endP; ++clone_dataP) {
		if (clone_dataP->function1P != function1P) {
			continue;
		}
		if (clone_dataP->function2P != function2P) {
			continue;
		}
		lineno1     = clone_dataP->start_clone1.lineno;
		lineno2     = clone_dataP->start_clone2.lineno;
		end_lineno1 = clone_dataP->end_clone1.lineno;
		end_lineno2 = clone_dataP->end_clone2.lineno;

		if (end1   < lineno1 ||
			start1 > end_lineno1 ||
			end2   < lineno2 ||
			start2 > end_lineno2) {
			// No overlap
			continue;
		}
		++(clone_dataP->overlap);
		seen = 1;

		fprintf(stdoutF,
			"<h3>Correlates with CLIC clone pairs</h3>\n<p>\n"
			"<table><tr><th>%s</th><th>%s</th></tr>\n",
					file1P->m_nameP, file2P->m_nameP);
		default1P = file1P;
		default2P = file2P;
		while (lineno1 <= end_lineno1 || lineno2 <= end_lineno2) {
			fputs("<tr><td>", stdoutF);
			if (lineno1 <= end_lineno1) {
				file1P->showline(&default1P, lineno1);
				++lineno1;
			}
			fputs("</td><td>", stdoutF);
			if (lineno2 <= end_lineno2) {
				file2P->showline(&default2P, lineno2);
				++lineno2;
			}
			fputs("</td></tr>\n", stdoutF);
		}
		fputs("</table>\n", stdoutF);
	}
	if (seen) {
		++g_acd_overlap;
	}

	targetP = get_function_pair(i, j);

	lth = end1 - start1 + 1;
	assert(lth > 0);
	if (lth > max_acd_lth) {
		max_acd_lth = lth;
	}
	sum_acd_lth  += lth;
	sum_acd2_lth += lth*lth;

	lth = end2 - start2 + 1;
	assert(lth > 0);
	if (lth > max_acd_lth) {
		max_acd_lth = lth;
	}
	sum_acd_lth  += lth;
	sum_acd2_lth += lth*lth;
	
	cnt_acd_lth  += 2;

	boxP = get_box(start1, end1, start2, end2, 0, ACD_TYPE);
	apply_box(boxP, targetP);
}

static void
correlate_clone_data(void)
{
	FILE			*saveF;
	clone_dataT		*clone_dataP, *endP;
	Cfunction		*function1P, *function2P;
	boxT			*boxP, *targetP;
	int	 			start1, start2, end1, end2;
	int				i, j, lth;
	char			*name1P, *name2P;
	Cfile			*file1P, *file2P;
	const Cfile		*default1P, *default2P;
	int				lineno1, lineno2;
	char			*P;
	int				skip;

	clone_dataP    = clone_data;
	endP    = clone_data + (sizeof(clone_data)/sizeof(clone_data[0]));
	for (; clone_dataP < endP; ++clone_dataP) {
		function1P = clone_dataP->function1P;
		if (!function1P) {
			continue;
		}
		function2P = clone_dataP->function2P;
		if (!function2P) {
			continue;
		}

		i     = function1P->m_number;
		j     = function2P->m_number;

		assert (i <= j);

		start1 = clone_dataP->start_clone1.lineno;
		end1   = clone_dataP->end_clone1.lineno;
		start2 = clone_dataP->start_clone2.lineno;
		end2   = clone_dataP->end_clone2.lineno;

		assert(start1 <= end1);
		assert(start2 <= end2);
		assert(start1 > 0 && end1 > 0 && start2 > 0 && end2 > 0);

		++g_clic_pairs;


		name1P   = source_data[i].nameP;
		name2P   = source_data[j].nameP;
		file1P   = Cfile::locate(name1P);
		assert(file1P);
		skip = 0;
		P    = "ACD missed clone pair";
		if (i != j) {
			file2P   = Cfile::locate(name2P);
			assert(file2P);
			if (clone_dataP->overlap) {
				++g_clic_overlap;
				P = 0;
			} 
		} else {
			file2P = file1P;
			if (start2 <= end1 && start1 <= end2) {
				++g_clic_bad;
				P = "CLIC overlapped pair";
				skip = 1;
			} else if (clone_dataP->overlap){
				++g_clic_overlap;
				P = 0;
		}	}

		if (P) {
			saveF   = stdoutF;
			stdoutF = missedF;
			fprintf(stdoutF, "<h3>%s</h3>\n", P);
			fprintf(stdoutF,
				"<p>\n<table><tr><th>%s</th><th>%s</th></tr>\n",
					name1P, name2P);
			default1P = file1P;
			default2P = file2P;
			lineno1   = start1;
			lineno2   = start2;
			while (lineno1 <= end1 || lineno2 <= end2) {
				fputs("<tr><td>", stdoutF);
				if (lineno1 <= end1) {
					file1P->showline(&default1P, lineno1);
					++lineno1;
				}
				fputs("</td><td>", stdoutF);
				if (lineno2 <= end2) {
					file2P->showline(&default2P, lineno2);
					++lineno2;
				}
				fputs("</td></tr>\n", stdoutF);
			}
			fputs("</table>\n<p>\n", stdoutF);
			stdoutF = saveF;
			if (skip) {
				continue;
		}	}

		source_data[i].type |= CLIC_TYPE;
		source_data[j].type |= CLIC_TYPE;
		targetP = get_function_pair(i, j);

		lth = end1 - start1 + 1;
		if (lth > max_clic_lth) {
			max_clic_lth = lth;
		}
		sum_clic_lth  += lth;
		sum_clic2_lth += lth*lth;
	
		lth = end2 - start2 + 1;
		if (lth > max_clic_lth) {
			max_clic_lth = lth;
		}
		sum_clic_lth  += lth;
		sum_clic2_lth += lth*lth;
		cnt_clic_lth  += 2;

		boxP    = get_box(start1, end1, start2, end2, 0, CLIC_TYPE);

		apply_box(boxP, targetP);
}	}

static void
compute_statistics(void)
{
	Csource			*sourceP;
	Cfunction		*functionP;
	boxT			**function_pairPP, **endPP, *boxP;
	long long	lth1, lth2;
	int		i;

	for (sourceP = Csource::g_headP; sourceP; sourceP = sourceP->m_nextP) {
		for (functionP = sourceP->m_functionsP; functionP; functionP = functionP->m_nextP) {
			g_source[functionP->m_type] += 1;
	}	}

	function_pairPP = function_pairsPP;
	for (endPP = function_pairPP + FUNCTION_PAIRS; function_pairPP < endPP; ++function_pairPP) {
		for (boxP = *function_pairPP; boxP; boxP = boxP->nextP) {

			lth1 = (boxP->clone1.end - boxP->clone1.start)+1;
			assert(lth1 > 0);
			lth2 = (boxP->clone2.end - boxP->clone2.start)+1;
			assert(lth2 > 0);
			lth1 *= lth2;
			g_area[boxP->type] += lth1;
	}	}

	for (i = 0; i < 4; ++i) {
		g_total_area += g_area[i];
	}
}

void
report_correlated(void)
{
	int				i;
	double			d, d2, region;
	char			*P;
	long long		sum;

	correlate_clone_data();
	compute_statistics();

	fprintf(dataF, "<h3>Statistics</h3>\n<p>\n<table>\n");
	fprintf(dataF, "<tr><th align=right>Source files</th><td>%d</td></tr>\n", (int) SOURCES);
	fprintf(dataF, "<tr><th align=right>Outside build</th><td>%d</td></tr>\n", g_outside_build);
	fprintf(dataF, "<tr><th align=right>Max code lth</th><td>%d</td></tr>\n", max_code_lth);
	d  = ((double) sum_code_lth) / ((double) g_functions);
	fprintf(dataF, "<tr><th align=right>Avg function lth</th><td>%lg</td></tr>\n", d);
	d2 = ((double) sum_code2_lth) / ((double) g_functions);
	d  = sqrt(d2 - d*d);
	fprintf(dataF, "<tr><th align=right>Std function lth</th><td>%lg</td></tr>\n", d);
	fprintf(dataF, "<tr><th align=right>CLIC pairs</th><td>%lld</td></tr>\n", g_clic_pairs);
	fprintf(dataF, "<tr><th align=right>CLIC bad</th><td>%lld</td></tr>\n", g_clic_bad);
	fprintf(dataF, "<tr><th align=right>Total CLIC lth</th><td>%lld</td></tr>\n", sum_clic_lth);
	fprintf(dataF, "<tr><th align=right>Max CLIC lth</th><td>%d</td></tr>\n", max_clic_lth);
	d  = ((double) sum_clic_lth) / ((double) cnt_clic_lth);
	fprintf(dataF, "<tr><th align=right>Avg CLIC lth</th><td>%lg</td></tr>\n", d);
	d2 = ((double) sum_clic2_lth) / ((double) cnt_clic_lth);
	d  = sqrt(d2 - d*d);
	fprintf(dataF, "<tr><th align=right>Std CLIC lth</th><td>%lg</td></tr>\n", d);


	fprintf(dataF, "<tr><th></th><td></td></tr>\n");
	fprintf(dataF, "<tr><th align=right>ACD pairs</th><td>%lld</td></tr>\n", g_acd_pairs);
	fprintf(dataF, "<tr><th align=right>ACD overlap</th><td>%lld</td></tr>\n", g_acd_overlap);
	fprintf(dataF, "<tr><th align=right>CLIC overlap</th><td>%lld</td></tr>\n", g_clic_overlap);

	sum = 0;
	for (i = 0; i < 4; ++i) {
		switch(i) {
		case 0:
			P = "Neither";
			break;
		case ACD_TYPE:
			P = "ACD";
			break;
		case CLIC_TYPE:
			P = "CLIC";
			break;
		case ACD_TYPE | CLIC_TYPE:
			P = "Both";
			break;
		default:
			assert(0);
			P = 0;
		}
		fprintf(dataF, "<tr><th align=right>%s hit function</th><td>%lld</td></tr>\n", P, g_source[i]);
		if (i) {
			sum += g_source[i];
		}
	}
	fprintf(dataF, "<tr><th align=right>total hit function</th><td>%lld</td></tr>\n", sum);

	fprintf(dataF, "<tr><th align=right>Total ACD lth</th><td>%lld</td></tr>\n", sum_acd_lth);
	fprintf(dataF, "<tr><th align=right>Max ACD lth</th><td>%d</td></tr>\n", max_acd_lth);
	d  = ((double) sum_acd_lth) / ((double) cnt_acd_lth);
	fprintf(dataF, "<tr><th align=right>Avg ACD lth</th><td>%lg</td></tr>\n", d);
	d2 = ((double) sum_acd2_lth) / ((double) cnt_acd_lth);
	d  = sqrt(d2 - d*d);
	fprintf(dataF, "<tr><th align=right>Std ACD lth</th><td>%lg</td></tr>\n", d);

	d2 = 0;
	for (i = 0; i < 4; ++i) {
		switch(i) {
		case 0:
			P = "Neither";
			break;
		case ACD_TYPE:
			P = "ACD ";
			break;
		case CLIC_TYPE:
			P = "CLIC";
			break;
		case ACD_TYPE | CLIC_TYPE:
			P = "Both";
			break;
		default:
			assert(0);
			P = 0;
		}
		d = ( ((double) g_area[i]) / ((double) g_total_area) ) * 100.0;

		fprintf(dataF, "<tr><th align=right>%s area</th><td>%lg%%</td></tr>\n", P, d);
		if (i) {
			d2 += d;
		}
	}

	fprintf(dataF, "<tr><th align=right>Clone area</th><td>%lg%%</td></tr>\n", d2);
	region = sqrt(g_total_area);
	d2 = 0;
	for (i = 0; i < 4; ++i) {
		switch(i) {
		case 0:
			P = "Neither";
			break;
		case ACD_TYPE:
			P = "ACD ";
			break;
		case CLIC_TYPE:
			P = "CLIC";
			break;
		case ACD_TYPE | CLIC_TYPE:
			P = "Both";
			break;
		default:
			assert(0);
			P = 0;
		}
		d = (sqrt(g_area[i]) / region) * 100.0;

		fprintf(dataF, "<tr><th align=right>%s range</th><td>%lg%%</td></tr>\n", P, d);
		if (i) {
			d2 += d;
	}	}

	fprintf(dataF, "<tr><th align=right>Clone region</th><td>%lg%%</td></tr>\n", d2);
	fprintf(missedF, "<P>\nEnd of report\n</body>\n</html>\n");
	fclose(missedF);
	fprintf(dataF, "</table>\n<P>\nEnd of report\n</body>\n</html>\n");
	fclose(dataF);
}
#endif
