/* objdump.c -- dump information about an object file.
   Copyright 1990, 1991, 1992, 1993, 1994, 1995, 1996, 1997, 1998, 1999,
   2000, 2001, 2002, 2003, 2004, 2005
   Free Software Foundation, Inc.

   This file is part of GNU Binutils.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   Heavily hacked by Ian Davis
*/

#include <stdio.h>
#include <unistd.h>

#include "bfd.h"
#include "bfdver.h"
#include "bucomm.h"
#include "safe-ctype.h"
#include "dis-asm.h"
#include "libiberty.h"
#include "asxo.h"

#ifdef NEED_DECLARATION_FPRINTF
/* This is needed by init_disassemble_info().  */
extern int fprintf (FILE *, const char *, ...);
#endif

/* #define TRACE */

#if defined(TRACE)
static int trace_depth = 0;

#define IN(X) \
{ \
	++trace_depth; \
	fprintf(stderr, "%*.*s>%s:%d\n", trace_depth, trace_depth, "", X, __LINE__); \
}
#define OUT(X) \
{ \
	fprintf(stderr, "%*.*s<%s:%d\n", trace_depth, trace_depth, "", X, __LINE__); \
	--trace_depth; \
}
#else
#define IN(X)
#define OUT(X)
#endif

/* Exit status.  */
static int exit_status = 0;

static void (*g_callback)(int lth, char *lineP);

/* Extra info to pass to the section disassembler and address printing
   function.  */
struct objdump_disasm_info
{
	bfd *              abfd;
	asection *         sec;
	bfd_boolean        require_sec;
	arelent **         dynrelbuf;
	long               dynrelcount;
  	disassembler_ftype disassemble_fn;
#ifdef DISASSEMBLER_NEEDS_RELOCS
  	arelent *          reloc;
#endif
};

/* Architecture to disassemble for, or default if NULL.  */
static char *machine = NULL;

/* Endianness to disassemble for, or default if BFD_ENDIAN_UNKNOWN.  */
static enum bfd_endian endian = BFD_ENDIAN_UNKNOWN;

/* The symbol table.  */
static asymbol **syms = 0;

/* Number of symbols in `syms'.  */
static long symcount = 0;

/* The sorted symbol table.  */
static asymbol **sorted_syms = 0;

/* Number of symbols in `sorted_syms'.  */
static long sorted_symcount = 0;

/* The dynamic symbol table.  */
static asymbol **dynsyms = 0;

/* The synthetic symbol table.  */
static asymbol *synthsyms = 0;
static long synthcount = 0;

/* Number of symbols in `dynsyms'.  */
static long dynsymcount = 0;

static void
nonfatal(const char *msg)
{
	IN("nonfatal")
	bfd_nonfatal (msg);
	exit_status = 1;
	OUT("nonfatal")
}

/* Pseudo FILE object for strings.  */
typedef struct {
  char *buffer;
  size_t pos;
  size_t alloc;
} SFILE;

SFILE	output;

/* sprintf to a "stream".  */

static int
burp(SFILE *f, const char *format, ...)
{
	size_t space, n;
	va_list args;

	IN("burp")
    va_start (args, format);
  	for (;;) {
		space = f->alloc - f->pos;
		if (!space) {
			n = 0;
		} else {
      		n = vsnprintf (f->buffer + f->pos, space, format, args);

      		if (space > n) {
				break;
		}	}
      
      	f->alloc = (f->alloc + n) * 2;
      	f->buffer = xrealloc (f->buffer, f->alloc+1);
	}
  	f->pos += n;
  
	va_end (args);
  	OUT("burp")
  	return n;
}

static void
flush_burp(SFILE *f)
{
	char	*P, *P1, *endP;

	if (f->pos) {
		if (g_callback) {
			P     = f->buffer;
			endP  = P + f->pos;
			*endP = 0;
			for (endP = P + f->pos; P < endP; P = P1+1) {
				P1 = strchr(P,'\n');
				if (!P1) {
					break;
				}
				*P1 = 0;
				g_callback(P1-P, P);
		}	}
		f->pos = 0;
}	}

static asymbol **
slurp_symtab(bfd *abfd)
{
	asymbol **sy = NULL;
  	long storage;

  	IN("slurp_symtab")
  	if (!(bfd_get_file_flags (abfd) & HAS_SYMS)) {
		symcount = 0;
      	goto done;
    }

  	storage = bfd_get_symtab_upper_bound (abfd);
  	if (storage < 0) {
    	bfd_fatal (bfd_get_filename (abfd));
  	}
  	if (storage) {
    	sy = xmalloc (storage);
  	}

  	symcount = bfd_canonicalize_symtab (abfd, sy);
  	if (symcount < 0) {
    	bfd_fatal (bfd_get_filename (abfd));
  	}
done:
  	OUT("slurp_symtab")
  	return sy;
}

/* Read in the dynamic symbols.  */

static asymbol **
slurp_dynamic_symtab(bfd *abfd)
{
  	asymbol **sy = NULL;
  	long storage;

  	IN("slurp_dynamic_symtab")
  	storage = bfd_get_dynamic_symtab_upper_bound (abfd);
  	if (storage < 0) {
		if (!(bfd_get_file_flags (abfd) & DYNAMIC)) {
			non_fatal ("%s: not a dynamic object", bfd_get_filename (abfd));
	  		dynsymcount = 0;
	  		goto done;
		}
      	bfd_fatal (bfd_get_filename (abfd));
    }
  	if (storage) {
    	sy = xmalloc (storage);
	}

  	dynsymcount = bfd_canonicalize_dynamic_symtab (abfd, sy);
  	if (dynsymcount < 0) {
    	bfd_fatal (bfd_get_filename (abfd));
	}
done:
  	OUT("slurp_dynamic_symtab")
	return sy;
}

/* Filter out (in place) symbols that are useless for disassembly.
   COUNT is the number of elements in SYMBOLS.
   Return the number of useful symbols.  */

static long
remove_useless_symbols(asymbol **symbols, long count)
{
	asymbol **in_ptr = symbols;
	asymbol **out_ptr = symbols;

  	IN("remove_useless_symbols")
  	while (--count >= 0) {
		asymbol *sym = *in_ptr++;

      	if (sym->name == NULL || sym->name[0] == '\0') {
			continue;
		}
      	if (sym->flags & (BSF_DEBUGGING | BSF_SECTION_SYM)) {
			continue;
		}
      	if (bfd_is_und_section (sym->section)) {
			continue;
		}
		if (bfd_is_com_section (sym->section)) {
			continue;
		}
      	*out_ptr++ = sym;
	}
  	OUT("remove_useless_symbols")
  	return out_ptr - symbols;
}

/* Sort symbols into value order.  */

static int
compare_symbols(const void *ap, const void *bp)
{
  	const asymbol *a = * (const asymbol **) ap;
  	const asymbol *b = * (const asymbol **) bp;
  	const char *an;
  	const char *bn;
  	size_t anl;
  	size_t bnl;
  	bfd_boolean af;
  	bfd_boolean bf;
  	flagword aflags;
  	flagword bflags;
  	int  ret;

  	IN("compare_symbols")
  	if (bfd_asymbol_value (a) > bfd_asymbol_value (b)) {
    	ret = 1;
    	goto done;
  	}
  	if (bfd_asymbol_value (a) < bfd_asymbol_value (b)) {
    	ret = -1;
    	goto done;
  	}
  	if (a->section > b->section) {
    	ret = 1;
    	goto done;
  	}
  	if (a->section < b->section) {
    	ret = -1;
    	goto done;
  	}
  	an = bfd_asymbol_name (a);
  	bn = bfd_asymbol_name (b);
  	anl = strlen (an);
  	bnl = strlen (bn);

  	/* The symbols gnu_compiled and gcc2_compiled convey no real
       information, so put them after other symbols with the same value.  */

  	af = (strstr (an, "gnu_compiled") != NULL || strstr (an, "gcc2_compiled") != NULL);
  	bf = (strstr (bn, "gnu_compiled") != NULL || strstr (bn, "gcc2_compiled") != NULL);
  	if (af && ! bf) {
    	ret = 1;
    	goto done;
  	}
  	if (! af && bf) {
    	ret = -1;
    	goto done;
  	}

  	/* We use a heuristic for the file name, to try to sort it after
       more useful symbols.  It may not work on non Unix systems, but it
       doesn't really matter; the only difference is precisely which
       symbol names get printed.  */

#define file_symbol(s, sn, snl)			\
  (((s)->flags & BSF_FILE) != 0			\
   || ((sn)[(snl) - 2] == '.'			\
       && ((sn)[(snl) - 1] == 'o'		\
	   || (sn)[(snl) - 1] == 'a')))

  	af = file_symbol(a, an, anl);
  	bf = file_symbol(b, bn, bnl);

  	if (af && ! bf) {
    	ret = 1;
    	goto done;
  	}
  	if (! af && bf) {
    	ret = -1;
    	goto done;
  	}

  	/* Try to sort global symbols before local symbols before function
       symbols before debugging symbols.  */

  	aflags = a->flags;
  	bflags = b->flags;

  	if ((aflags & BSF_DEBUGGING) != (bflags & BSF_DEBUGGING)) {
		if ((aflags & BSF_DEBUGGING) != 0) {
        	ret = 1;
        	goto done;
      	}
      	ret = -1;
      	goto done;
	}
  	if ((aflags & BSF_FUNCTION) != (bflags & BSF_FUNCTION)) {
		if ((aflags & BSF_FUNCTION) != 0) {
			ret = -1;
        	goto done;
      	}
      	ret = 1;
      	goto done;
	}
  	if ((aflags & BSF_LOCAL) != (bflags & BSF_LOCAL)) {
		if ((aflags & BSF_LOCAL) != 0) {
			ret = 1;
        	goto done;
		}
		ret = -1;
		goto done;
	}
  	if ((aflags & BSF_GLOBAL) != (bflags & BSF_GLOBAL)) {
		if ((aflags & BSF_GLOBAL) != 0) {
			ret = -1;
        	goto done;
      	}
      	ret = 1;
      	goto done;
	}

  /* Symbols that start with '.' might be section names, so sort them
     after symbols that don't start with '.'.  */
	if (an[0] == '.' && bn[0] != '.') {
    	ret = 1;
    	goto done;
  	}
  	if (an[0] != '.' && bn[0] == '.') {
    	ret = -1;
    	goto done;
  	}

  	/* Finally, if we can't distinguish them in any other way, try to
       get consistent results by sorting the symbols by name.  */
  	ret = strcmp (an, bn);
done:
  	OUT("compare_symbols")
  	return(ret);
}

/* Sort relocs into address order.  */

static int
compare_relocs(const void *ap, const void *bp)
{
	const arelent *a = * (const arelent **) ap;
  	const arelent *b = * (const arelent **) bp;
  	int ret;

  	IN("compare_relocs")
  	if (a->address > b->address) {
    	ret = 1;
    	goto done;
  	}
  	if (a->address < b->address) {
    	ret = -1;
    	goto done;
  	}

  	/* So that associated relocations tied to the same address show up
       in the correct order, we don't do any further sorting.  */
  	if (a > b) {
    	ret = 1;
    	goto done;
  	}
  	if (a < b) {
    	ret = -1;
    	goto done;
  	}
  	ret = 0;
done:
  	OUT("compare_relocs")
  	return ret;
}

/* Print an address (VMA) to the output stream in INFO.
   If SKIP_ZEROES is TRUE, omit leading zeroes.  */

static void
objdump_print_value(bfd_vma vma, struct disassemble_info *info, bfd_boolean skip_zeroes)
{
	char buf[30];
  	char *p;
  	struct objdump_disasm_info *aux;

  	IN("objdump_print_value")
  	aux = (struct objdump_disasm_info *) info->application_data;
  	bfd_sprintf_vma(aux->abfd, buf, vma);
	p = buf;
  	if (skip_zeroes) {
      	for (; *p == '0'; ++p) ;
      	if (*p == '\0') {
			--p;
	}	}
  	(*info->fprintf_func) (info->stream, "%s", p);
  	OUT("objdump_print_value")
}

/* Print the name of a symbol.  */

static void
objdump_print_symname(struct disassemble_info *info, asymbol *sym)
{
	const char *name;

  	IN("objdump_print_symname")
  	name = bfd_asymbol_name (sym);
  	(*info->fprintf_func) (info->stream, "%s", name);
  	OUT("objdump_print_symname")
}

/* Locate a symbol given a bfd and a section (from INFO->application_data),
   and a VMA.  If INFO->application_data->require_sec is TRUE, then always
   require the symbol to be in the section.  Returns NULL if there is no
   suitable symbol.  If PLACE is not NULL, then *PLACE is set to the index
   of the symbol in sorted_syms.  */

static asymbol *
find_symbol_for_address(bfd_vma vma, struct disassemble_info *info, long *place)
{
	/* @@ Would it speed things up to cache the last two symbols returned,
       and maybe their address ranges?  For many processors, only one memory
       operand can be present at a time, so the 2-entry cache wouldn't be
       constantly churned by code doing heavy memory accesses.  */

  	/* Indices in `sorted_syms'.  */

  	long min = 0;
  	long max = sorted_symcount;
  	long thisplace;
  	struct objdump_disasm_info *aux;
  	bfd *abfd;
  	asection *sec;
  	unsigned int opb;
  	asymbol *ret;

  	IN("find_symbol_for_address")
  	if (sorted_symcount < 1) {
    	ret = NULL;
    	goto done;
  	}

  	aux = (struct objdump_disasm_info *) info->application_data;
  	abfd = aux->abfd;
  	sec = aux->sec;
  	opb = bfd_octets_per_byte (abfd);

  	/* Perform a binary search looking for the closest symbol to the
       required value.  We are searching the range (min, max].  */
  	while (min + 1 < max) {
		asymbol *sym;

      	thisplace = (max + min) / 2;
      	sym = sorted_syms[thisplace];

      	if (bfd_asymbol_value (sym) > vma) {
			max = thisplace;
		} else if (bfd_asymbol_value (sym) < vma) {
			min = thisplace;
      	} else {
	  		min = thisplace;
	  		break;
		}
	}

  	/* The symbol we want is now in min, the low end of the range we
       were searching.  If there are several symbols with the same
       value, we want the first one.  */

  	thisplace = min;
  	while (thisplace > 0
	 && (bfd_asymbol_value (sorted_syms[thisplace])
	     == bfd_asymbol_value (sorted_syms[thisplace - 1]))) {
    	--thisplace;
	}

  	/* If the file is relocatable, and the symbol could be from this
       section, prefer a symbol from this section over symbols from
       others, even if the other symbol's value might be closer.

       Note that this may be wrong for some symbol references if the
       sections have overlapping memory ranges, but in that case there's
       no way to tell what's desired without looking at the relocation
       table.  */

	if (sorted_syms[thisplace]->section != sec
         && (aux->require_sec
	     || ((abfd->flags & HAS_RELOC) != 0
	      && vma >= bfd_get_section_vma (abfd, sec)
	      && vma < (bfd_get_section_vma (abfd, sec)
			+ bfd_section_size (abfd, sec) / opb)))) {
		long i;

      	for (i = thisplace + 1; i < sorted_symcount; i++) {
	  		if (bfd_asymbol_value (sorted_syms[i]) != bfd_asymbol_value (sorted_syms[thisplace])) {
	    		break;
			}
		}

      	--i;

      	for (; i >= 0; i--) {
	  		if (sorted_syms[i]->section == sec
	      		&& (i == 0
		  		|| sorted_syms[i - 1]->section != sec
		  		|| (bfd_asymbol_value (sorted_syms[i])
		      	!= bfd_asymbol_value (sorted_syms[i - 1])))) {
	      		thisplace = i;
	      		break;
	    	}
		}

      	if (sorted_syms[thisplace]->section != sec) {

	  		/* We didn't find a good symbol with a smaller value.
	     	   Look for one with a larger value.  */
	  		for (i = thisplace + 1; i < sorted_symcount; i++) {
	      		if (sorted_syms[i]->section == sec) {
		  			thisplace = i;
		  			break;
				}
	    	}
		}

      	if (sorted_syms[thisplace]->section != sec
	  		&& (aux->require_sec
	      	|| ((abfd->flags & HAS_RELOC) != 0
		  	&& vma >= bfd_get_section_vma (abfd, sec)
		  	&& vma < (bfd_get_section_vma (abfd, sec)
			    + bfd_section_size (abfd, sec))))) {
			/* There is no suitable symbol.  */
        	ret = NULL;
        	goto done;
      	}
	}

  	/* Give the target a chance to reject the symbol.  */
  	while (! info->symbol_is_valid (sorted_syms [thisplace], info)) {
		++thisplace;
      	if (thisplace >= sorted_symcount
	  		|| bfd_asymbol_value (sorted_syms [thisplace]) > vma) {
        	ret = NULL;
        	goto done;
      	}
	}

  	if (place != NULL) {
    	*place = thisplace;
	}
  	ret = sorted_syms[thisplace];
done:
  	OUT("find_symbol_for_address")
  	return ret;
}

/* Print an address and the offset to the nearest symbol.  */

static void
objdump_print_addr_with_sym(asection *sec, asymbol *sym,
			     bfd_vma vma, struct disassemble_info *info,
			     bfd_boolean skip_zeroes)
{
	IN("objdump_print_addr_with_sym")
  	objdump_print_value (vma, info, skip_zeroes);

  	if (sym == NULL) {
		bfd_vma secaddr;

      	(*info->fprintf_func) (info->stream, " {%s", bfd_get_section_name (abfd, sec));
      	secaddr = bfd_get_section_vma (abfd, sec);
      	if (vma < secaddr) {
	  		(*info->fprintf_func) (info->stream, "-0x");
	  		objdump_print_value (secaddr - vma, info, TRUE);
		} else if (vma > secaddr) {
	  		(*info->fprintf_func) (info->stream, "+0x");
	  		objdump_print_value (vma - secaddr, info, TRUE);
		}
      	(*info->fprintf_func) (info->stream, "}");
	} else {
		(*info->fprintf_func) (info->stream, " <");
      	objdump_print_symname (info, sym);
      	(*info->fprintf_func) (info->stream, ">");
		if (bfd_asymbol_value (sym) > vma) {
	  		(*info->fprintf_func) (info->stream, "-0x");
	  		objdump_print_value (bfd_asymbol_value (sym) - vma, info, TRUE);
		} else if (vma > bfd_asymbol_value (sym)) {
	  		(*info->fprintf_func) (info->stream, "+0x");
	  		objdump_print_value (vma - bfd_asymbol_value (sym), info, TRUE);
		}
	}
	OUT("objdump_print_addr_with_sym")
}

/* Print an address (VMA), symbolically if possible.
   If SKIP_ZEROES is TRUE, don't output leading zeroes.  */

static void
objdump_print_addr(bfd_vma vma, struct disassemble_info *info, bfd_boolean skip_zeroes)
{
	struct objdump_disasm_info *aux;
  	asymbol *sym;

#ifdef DISASSEMBLER_NEEDS_RELOCS
  	bfd_boolean skip_find = FALSE;
#endif

	IN("objdump_print_addr")
  	if (sorted_symcount < 1) {
		(*info->fprintf_func) (info->stream, "0x");
      	objdump_print_value (vma, info, skip_zeroes);
      	goto done;
	}

  	aux = (struct objdump_disasm_info *) info->application_data;

#ifdef DISASSEMBLER_NEEDS_RELOCS
	if (aux->reloc != NULL
      	&& aux->reloc->sym_ptr_ptr != NULL
      	&& * aux->reloc->sym_ptr_ptr != NULL) {
		sym = * aux->reloc->sym_ptr_ptr;

      	/* Adjust the vma to the reloc.  */
      	vma += bfd_asymbol_value (sym);

      	if (bfd_is_und_section (bfd_get_section (sym))) {
			skip_find = TRUE;
		}
	}

  	if (!skip_find)
#endif
    sym = find_symbol_for_address (vma, info, NULL);

  	objdump_print_addr_with_sym (aux->sec, sym, vma, info, skip_zeroes);

done:
  	OUT("objdump_print_addr")
  	return;
}

/* Print VMA to INFO.  This function is passed to the disassembler
   routine.  */

static void
objdump_print_address(bfd_vma vma, struct disassemble_info *info)
{
	IN("objdump_print_address")
  	objdump_print_addr (vma, info, 1);
  	OUT("objdump_print_address")
}

/* Determine of the given address has a symbol associated with it.  */

static int
objdump_symbol_at_address (bfd_vma vma, struct disassemble_info * info)
{
	asymbol * sym;
	int	ret;

	IN("objdump_symbol_at_address")
	sym = find_symbol_for_address (vma, info, NULL);

	ret = (sym != NULL && (bfd_asymbol_value (sym) == vma));
	OUT("objdump_symbol_at_address")
	return ret;
}

/* Hold the last function name and the last line number we displayed
   in a disassembly.  */

static char		prev_source[2048];
static unsigned int prev_line;

/* Show the line number, in a disassembly listing.  */

static void
show_line (bfd *abfd, asection *section, bfd_vma addr_offset)
{
	const char *filename;
	const char *functionname;
	unsigned int line;

  	IN("show_line")
  	if (! bfd_find_nearest_line (abfd, section, syms, addr_offset, &filename, &functionname, &line)) {
    	goto done;
  	}

  	if (filename != NULL && *filename != '\0') {
		if (strcmp(filename, prev_source)) {
			burp(&output, "Source %s\n", filename);
			flush_burp(&output);
			strcpy(prev_source, filename);
	    	prev_line = -1;
  	}	}
    
  	if (line > 0 && line != prev_line) {
    	burp(&output, "Line %u\n", line);
		flush_burp(&output);
    	prev_line = line;
  	}
done:
  	OUT("show_line")
  	return;
}

/* The number of zeroes we want to see before we start skipping them.
   The number is arbitrarily chosen.  */

#define DEFAULT_SKIP_ZEROES 8

/* The number of zeroes to skip at the end of a section.  If the
   number of zeroes at the end is between SKIP_ZEROES_AT_END and
   SKIP_ZEROES, they will be disassembled.  If there are fewer than
   SKIP_ZEROES_AT_END, they will be skipped.  This is a heuristic
   attempt to avoid disassembling zeroes inserted by section
   alignment.  */

#define DEFAULT_SKIP_ZEROES_AT_END 3


/* Disassemble some data in memory between given values.  */

static void
disassemble_bytes(struct disassemble_info * info,
		   disassembler_ftype        disassemble_fn,
		   bfd_boolean               insns,
		   bfd_byte *                data,
		   bfd_vma                   start_offset,
		   bfd_vma                   stop_offset,
		   bfd_vma		     		 rel_offset,
		   arelent ***               relppp,
		   arelent **                relppend)
{
	struct objdump_disasm_info *aux;
	asection *section;
	int octets_per_line;
	bfd_boolean done_dot;
	bfd_vma addr_offset;
	unsigned int opb = info->octets_per_byte;
	unsigned int skip_zeroes = info->skip_zeroes;
	unsigned int skip_zeroes_at_end = info->skip_zeroes_at_end;
	int octets = opb;
	SFILE sfile;
	char buf[50];
	char *s;
	bfd_vma j;

  	IN("disassemble_bytes")
  	aux = (struct objdump_disasm_info *) info->application_data;
  	section = aux->sec;

  	sfile.alloc = 120;
  	sfile.buffer = xmalloc (sfile.alloc+1);
  	sfile.pos = 0;
  
  	if (insns) {
    	octets_per_line = 4;
  	} else {
    	octets_per_line = 16;
	}

	bfd_sprintf_vma (aux->abfd, buf, (section->vma + bfd_section_size (section->owner, section) / opb));
	s = buf;

  	info->insn_info_valid = 0;
  	done_dot = FALSE;
  	addr_offset = start_offset;
  	while (addr_offset < stop_offset) {
		bfd_vma z;
#ifdef DISASSEMBLER_NEEDS_RELOCS
		int previous_octets;

		/* Remember the length of the previous instruction.  */
      	previous_octets = octets;
#endif
      	octets = 0;

      	/* If we see more than SKIP_ZEROES octets of zeroes, we just print `...'.  */
      	for (z = addr_offset * opb; z < stop_offset * opb; z++) {
			if (data[z] != 0) {
	  			break;
      	}	}
      	if ((info->insn_info_valid == 0 || info->branch_delay_insns == 0) && (z - addr_offset * opb >= skip_zeroes || (z == stop_offset * opb && z - addr_offset * opb < skip_zeroes_at_end))) {
	  		burp(&output, "\t...\n");

	  		/* If there are more nonzero octets to follow, we only skip
	     	   zeroes in multiples of 4, to try to avoid running over
	     	   the start of an instruction which happens to start with
	     	   zero.  */
	  		if (z != stop_offset * opb) {
	    		z = addr_offset * opb + ((z - addr_offset * opb) &~ 3);
			}
	  		octets = z - addr_offset * opb;
		} else {
	  		int bpc = 0;
	  		int pb = 0;

	  		done_dot = FALSE;

	  		/* The line number tables will refer to unadjusted
	     	   section VMAs, so we must undo any VMA modifications
	     	   when calling show_line.  */
	  		show_line (aux->abfd, section, addr_offset);

	  		bfd_sprintf_vma (aux->abfd, buf, section->vma + addr_offset);

	  		burp(&output, "%s:\t", buf);

	  		if (insns) {
	      		sfile.pos = 0;
	      		info->fprintf_func = (fprintf_ftype) burp;
	      		info->stream       = (FILE *) &sfile;
	      		info->bytes_per_line = 0;
	      		info->bytes_per_chunk = 0;
	      		info->flags = 0;

#ifdef DISASSEMBLER_NEEDS_RELOCS
	      		if (*relppp < relppend) {
		  			bfd_signed_vma distance_to_rel;

		  			distance_to_rel = (**relppp)->address - (rel_offset + addr_offset);

		  			/* Check to see if the current reloc is associated with
		     		   the instruction that we are about to disassemble.  */
		  			if (distance_to_rel == 0
		      			/* FIXME: This is wrong.  We are trying to catch
			 				relocs that are addressed part way through the
			 				current instruction, as might happen with a packed
			 				VLIW instruction.  Unfortunately we do not know the
			 				length of the current instruction since we have not
			 				disassembled it yet.  Instead we take a guess based
			 				upon the length of the previous instruction.  The
			 				proper solution is to have a new target-specific
			 				disassembler function which just returns the length
			 				of an instruction at a given address without trying
			 				to display its disassembly. */
		      			|| (distance_to_rel > 0 && distance_to_rel < (bfd_signed_vma) (previous_octets/ opb))) {
		      			info->flags = INSN_HAS_RELOC;
		      			aux->reloc = **relppp;
		    		} else {
		    			aux->reloc = NULL;
					}
				}
#endif
	      		octets = (*disassemble_fn) (section->vma + addr_offset, info);
	      		info->fprintf_func = (fprintf_ftype) burp;
	      		info->stream       = &output;
	      		if (info->bytes_per_line != 0) {
					octets_per_line = info->bytes_per_line;
				}
	      		if (octets < 0) {
		  			if (sfile.pos) {
		    			burp(&output, "%s\n", sfile.buffer);
						flush_burp(&output);
	      			}
		  			break;
				}
	    	} else {
	      		bfd_vma j;

	      		octets = octets_per_line;
	      		if (addr_offset + octets / opb > stop_offset) {
					octets = (stop_offset - addr_offset) * opb;
				}

	      		for (j = addr_offset * opb; j < addr_offset * opb + octets; ++j) {
		  			if (ISPRINT (data[j])) {
		    			buf[j - addr_offset * opb] = data[j];
					} else {
		    			buf[j - addr_offset * opb] = '.';
					}
				}
	      		buf[j - addr_offset * opb] = '\0';
	    	}

      		/* we print octets_per_line octets per line.  */
      		pb = octets;
      		if (pb > octets_per_line) {
				pb = octets_per_line;
              	}

      		if (info->bytes_per_chunk) {
				 bpc = info->bytes_per_chunk;
			} else {
				bpc = 1;
			}

      		for (j = addr_offset * opb; j < addr_offset * opb + pb; j += bpc) {
	  			int k;

	  			if (bpc > 1 && info->display_endian == BFD_ENDIAN_LITTLE) {
	      			for (k = bpc - 1; k >= 0; k--) {
						burp(&output, "%02x", (unsigned) data[j + k]);
					}
	    		} else {
	      			for (k = 0; k < bpc; k++) {
						burp(&output, "%02x", (unsigned) data[j + k]);
					}
	    		}
	      		burp(&output, " ");
			}

      		for (; pb < octets_per_line; pb += bpc) {
	  			int k;

	  			for (k = 0; k < bpc; k++) {
	    			burp(&output, "  ");
	  			}
	  			burp(&output, " ");
			}

      		/* Separate raw data from instruction by extra space.  */
      		if (insns) {
				burp(&output, "\t");
			} else {
				burp(&output, "    ");
			}

	  		if (! insns) {
	    		burp(&output, "%s", buf);
	  		} else if (sfile.pos) {
	    		burp(&output, "%s", sfile.buffer);
          	}

      		while (pb < octets) {
	  			burp(&output, "\n");
				flush_burp(&output);
	  			j = addr_offset * opb + pb;

	  			bfd_sprintf_vma (aux->abfd, buf, section->vma + j / opb);
	  			burp(&output, "%s:\t", buf);

	  			pb += octets_per_line;
	  			if (pb > octets) {
	    			pb = octets;
				}
	  			for (; j < addr_offset * opb + pb; j += bpc) {
	      			int k;

	      			if (bpc > 1 && info->display_endian == BFD_ENDIAN_LITTLE) {
		  				for (k = bpc - 1; k >= 0; k--) {
		    				burp(&output, "%02x", (unsigned) data[j + k]);
						}
					} else {
		  				for (k = 0; k < bpc; k++) {
		    				burp(&output, "%02x", (unsigned) data[j + k]);
						}
					}
		  			burp(&output, " ");
	    		}
			}
	    	burp(&output, "\n");
			flush_burp(&output);
		}

      	while ((*relppp) < relppend && (**relppp)->address < rel_offset + addr_offset + octets / opb) {
	  		++(*relppp);
		}
      	addr_offset += octets / opb;
	}

  	free(sfile.buffer);
  	OUT("disassemble_bytes")
}

static void
disassemble_section (bfd *abfd, asection *section, void *info)
{
  struct disassemble_info *    pinfo = (struct disassemble_info *) info;
  struct objdump_disasm_info * paux;
  unsigned int                 opb = pinfo->octets_per_byte;
  bfd_byte *                   data = 0;
  bfd_size_type                datasize = 0;
  arelent **                   rel_pp = NULL;
  arelent **                   rel_ppstart = 0;
  arelent **                   rel_ppend;
  unsigned long                stop_offset;
  asymbol *                    sym = NULL;
  long                         place = 0;
  long                         rel_count;
  bfd_vma                      rel_offset;
  unsigned long                addr_offset;

  /* Sections that do not contain machine
     code are not normally disassembled.  */
  IN("disassemble_section")
  if (((section->flags & (SEC_CODE | SEC_HAS_CONTENTS)) != (SEC_CODE | SEC_HAS_CONTENTS)))
    goto done;

  datasize = bfd_get_section_size (section);
  if (datasize == 0)
    goto done;

  /* Decide which set of relocs to use.  Load them if necessary.  */
  paux = (struct objdump_disasm_info *) pinfo->application_data;
  if (paux->dynrelbuf)
    {
      rel_pp = paux->dynrelbuf;
      rel_count = paux->dynrelcount;
      /* Dynamic reloc addresses are absolute, non-dynamic are section
	 relative.  REL_OFFSET specifies the reloc address corresponding
	 to the start of this section.  */
      rel_offset = section->vma;
    }
  else
    {
      rel_count = 0;
      rel_pp = NULL;
      rel_offset = 0;

      if ((section->flags & SEC_RELOC) != 0)
	{
	  long relsize;

	  relsize = bfd_get_reloc_upper_bound (abfd, section);
	  if (relsize < 0)
	    bfd_fatal (bfd_get_filename (abfd));

	  if (relsize > 0)
	    {
	      rel_ppstart = rel_pp = xmalloc (relsize);
	      rel_count = bfd_canonicalize_reloc (abfd, section, rel_pp, syms);
	      if (rel_count < 0) {
		bfd_fatal (bfd_get_filename (abfd));
	      }

	      /* Sort the relocs by address.  */
	      qsort (rel_pp, rel_count, sizeof (arelent *), compare_relocs);
	    }
	}

    }
  rel_ppend = rel_pp + rel_count;

  data = xmalloc (datasize);

  bfd_get_section_contents (abfd, section, data, 0, datasize);

  paux->sec = section;
  pinfo->buffer = data;
  pinfo->buffer_vma = section->vma;
  pinfo->buffer_length = datasize;
  pinfo->section = section;

  addr_offset = 0;
  stop_offset = datasize / opb;

  /* Skip over the relocs belonging to addresses below the start address.  */
  while (rel_pp < rel_ppend
	 && (*rel_pp)->address < rel_offset + addr_offset)
    ++rel_pp;

  burp(&output, "Disassembly of section %s\n", section->name);
  flush_burp(&output);

  /* Find the nearest symbol forwards from our current position.  */
  paux->require_sec = TRUE;
  sym = find_symbol_for_address (section->vma + addr_offset, info, &place);
  paux->require_sec = FALSE;

  /* Disassemble a block of instructions up to the address associated with
     the symbol we have just found.  Then print the symbol and find the
     next symbol on.  Repeat until we have disassembled the entire section
     or we have reached the end of the address range we are interested in.  */
  while (addr_offset < stop_offset)
    {
      bfd_vma addr;
      asymbol *nextsym;
      unsigned long nextstop_offset;
      bfd_boolean insns;

      addr = section->vma + addr_offset;

      if (sym != NULL && bfd_asymbol_value (sym) <= addr)
	{
	  int x;

	  for (x = place;
	       (x < sorted_symcount
		&& (bfd_asymbol_value (sorted_syms[x]) <= addr));
	       ++x)
	    continue;

	  pinfo->symbols = sorted_syms + place;
	  pinfo->num_symbols = x - place;
	}
      else
	{
	  pinfo->symbols = NULL;
	  pinfo->num_symbols = 0;
	}

	
	pinfo->fprintf_func (pinfo->stream, "\n");
	show_line(abfd, section, addr - section->vma);
	objdump_print_addr_with_sym (section, sym, addr, pinfo, FALSE);
	pinfo->fprintf_func (pinfo->stream, ":\n");

      if (sym != NULL && bfd_asymbol_value (sym) > addr)
	nextsym = sym;
      else if (sym == NULL)
	nextsym = NULL;
      else
	{
#define is_valid_next_sym(SYM) \
  ((SYM)->section == section \
   && (bfd_asymbol_value (SYM) > bfd_asymbol_value (sym)) \
   && pinfo->symbol_is_valid (SYM, pinfo))
	    
	  /* Search forward for the next appropriate symbol in
	     SECTION.  Note that all the symbols are sorted
	     together into one big array, and that some sections
	     may have overlapping addresses.  */
	  while (place < sorted_symcount
		 && ! is_valid_next_sym (sorted_syms [place]))
	    ++place;

	  if (place >= sorted_symcount)
	    nextsym = NULL;
	  else
	    nextsym = sorted_syms[place];
	}

      if (sym != NULL && bfd_asymbol_value (sym) > addr)
	nextstop_offset = bfd_asymbol_value (sym) - section->vma;
      else if (nextsym == NULL)
	nextstop_offset = stop_offset;
      else
	nextstop_offset = bfd_asymbol_value (nextsym) - section->vma;

      if (nextstop_offset > stop_offset)
	nextstop_offset = stop_offset;

      /* If a symbol is explicitly marked as being an object
	 rather than a function, just dump the bytes without
	 disassembling them.  */
      if (sym == NULL || bfd_asymbol_value (sym) > addr
	  || ((sym->flags & BSF_OBJECT) == 0
	      && (strstr (bfd_asymbol_name (sym), "gnu_compiled")
		  == NULL)
	      && (strstr (bfd_asymbol_name (sym), "gcc2_compiled")
		  == NULL))
	  || (sym->flags & BSF_FUNCTION) != 0)
	insns = TRUE;
      else
	insns = FALSE;

      disassemble_bytes (pinfo, paux->disassemble_fn, insns, data,
			 addr_offset, nextstop_offset,
			 rel_offset, &rel_pp, rel_ppend);

      addr_offset = nextstop_offset;
      sym = nextsym;
    }

  free(data);
  pinfo->buffer        = 0;
  pinfo->buffer_length = 0;

  if (rel_ppstart != NULL) {
    free(rel_ppstart);
  }
done:
  OUT("disassemble_section")
  return;
}

/* Disassemble the contents of an object file.  */

static void
disassemble_data (bfd *abfd)
{
  struct disassemble_info disasm_info;
  struct objdump_disasm_info aux;
  long i, size;

  IN("disassemble_data")
  prev_line = -1;
  prev_source[0] = 0;

  /* We make a copy of syms to sort.  We don't want to sort syms
     because that will screw up the relocs.  */
  sorted_symcount = symcount ? symcount : dynsymcount;
  size = (sorted_symcount + synthcount) * sizeof (asymbol *);
  sorted_syms = xmalloc (size);
  memcpy (sorted_syms, symcount ? syms : dynsyms, sorted_symcount * sizeof (asymbol *));

  sorted_symcount = remove_useless_symbols (sorted_syms, sorted_symcount);

  for (i = 0; i < synthcount; ++i)
    {
      sorted_syms[sorted_symcount] = synthsyms + i;
      ++sorted_symcount;
    }

  /* Sort the symbols into section and symbol order.  */
  qsort (sorted_syms, sorted_symcount, sizeof (asymbol *), compare_symbols);

  init_disassemble_info (&disasm_info, &output, (fprintf_ftype) burp);

  disasm_info.application_data = (void *) &aux;
  aux.abfd = abfd;
  aux.require_sec = FALSE;
  aux.dynrelbuf = NULL;
  aux.dynrelcount = 0;
#ifdef DISASSEMBLER_NEEDS_RELOCS
  aux.reloc = NULL;
#endif

  disasm_info.print_address_func = objdump_print_address;
  disasm_info.symbol_at_address_func = objdump_symbol_at_address;

  if (machine != NULL)
    {
      const bfd_arch_info_type *info = bfd_scan_arch (machine);

      if (info == NULL)
	fatal ("Can't use supplied machine %s", machine);

      abfd->arch_info = info;
    }

  if (endian != BFD_ENDIAN_UNKNOWN)
    {
      struct bfd_target *xvec;

      size = sizeof(struct bfd_target);
      xvec = xmalloc (size);
      memcpy (xvec, abfd->xvec, sizeof (struct bfd_target));
      xvec->byteorder = endian;
      abfd->xvec = xvec;
    }

  /* Use libopcodes to locate a suitable disassembler.  */
  aux.disassemble_fn = disassembler (abfd);
  if (!aux.disassemble_fn)
    {
      non_fatal ("Can't disassemble for architecture %s\n",
		 bfd_printable_arch_mach (bfd_get_arch (abfd), 0));
      exit_status = 1;
      goto done;
    }

  disasm_info.flavour = bfd_get_flavour (abfd);
  disasm_info.arch = bfd_get_arch (abfd);
  disasm_info.mach = bfd_get_mach (abfd);
  disasm_info.disassembler_options = 0;
  disasm_info.octets_per_byte = bfd_octets_per_byte (abfd);
  disasm_info.skip_zeroes = DEFAULT_SKIP_ZEROES;
  disasm_info.skip_zeroes_at_end = DEFAULT_SKIP_ZEROES_AT_END;

  if (bfd_big_endian (abfd))
    disasm_info.display_endian = disasm_info.endian = BFD_ENDIAN_BIG;
  else if (bfd_little_endian (abfd))
    disasm_info.display_endian = disasm_info.endian = BFD_ENDIAN_LITTLE;
  else
    /* ??? Aborting here seems too drastic.  We could default to big or little
       instead.  */
    disasm_info.endian = BFD_ENDIAN_UNKNOWN;

  /* Allow the target to customize the info structure.  */
  disassemble_init_for_target (& disasm_info);

  bfd_map_over_sections (abfd, disassemble_section, & disasm_info);

  if (aux.dynrelbuf != NULL) {
    free(aux.dynrelbuf);
    aux.dynrelbuf = 0;
  }
  free(sorted_syms);
  sorted_syms = 0;
done:
  OUT("disassemble_data")
  return;
}

/* Dump selected contents of ABFD.  */

static void
dump_bfd (bfd *abfd)
{
	IN("dump_bfd")
	burp(&output, "\n");
	flush_burp(&output);
	burp(&output, "File %s\n", bfd_get_filename (abfd));
	flush_burp(&output);
	burp(&output, "Format %s\n", abfd->xvec->name);
	flush_burp(&output);

	syms = slurp_symtab (abfd);
  	if (bfd_get_dynamic_symtab_upper_bound (abfd) > 0) {
    	dynsyms = slurp_dynamic_symtab (abfd);
	}
	synthcount = bfd_get_synthetic_symtab (abfd, symcount, syms, dynsymcount, dynsyms, &synthsyms);
	if (synthcount < 0) {
		synthcount = 0;
	}
    disassemble_data (abfd);

  	if (syms) {
		free(syms);
		syms = 0;
    }

  	if (dynsyms) {
		free(dynsyms);
      	dynsyms = 0;
    }

  	if (synthsyms) {
		free(synthsyms);
      	synthsyms = 0;
    }

  	symcount    = 0;
  	dynsymcount = 0;
  	synthcount  = 0;
	OUT("dump_bfd")
}

static void
display_bfd (bfd *abfd)
{
	char **matching;

	IN("display_bfd")
  	if (bfd_check_format_matches (abfd, bfd_object, &matching)) {
		dump_bfd (abfd);
		goto done;
    }

  	if (bfd_get_error () == bfd_error_file_ambiguously_recognized) {
		nonfatal (bfd_get_filename (abfd));
      	list_matching_formats (matching);
      	free(matching);
		matching = 0;
      	goto done;
    }

  	if (bfd_get_error () != bfd_error_file_not_recognized) {
		nonfatal (bfd_get_filename (abfd));
      	goto done;
    }

  	if (bfd_check_format_matches (abfd, bfd_core, &matching)) {
		dump_bfd (abfd);
      	goto done;
    }

  	nonfatal (bfd_get_filename (abfd));

  	if (bfd_get_error () == bfd_error_file_ambiguously_recognized) {
		list_matching_formats (matching);
      	free(matching);
		matching = 0;
    }
done:
	OUT("display_bfd")
	return;
}

void
objdumpfile(const char *filenameP, void (*callback)(int lth, char *lineP))
{
	bfd *file;
	bfd *arfile = NULL;

	g_callback = callback;

  	output.alloc = 120;
  	output.buffer = xmalloc (output.alloc+1);
  	output.pos = 0;
  
	IN("objdumpfile")
  	if (get_file_size(filenameP) < 1) {
    	goto done;
	}
  	file = bfd_openr (filenameP, 0 /* default target */);
  	if (!file) {
		nonfatal (filenameP);
		goto done;
    }

  	/* If the file is an archive, process all of its elements.  */
  	if (bfd_check_format (file, bfd_archive)) {
		bfd *last_arfile = NULL;

      	burp(&output, "Archive %s\n", bfd_get_filename (file));
		flush_burp(&output);
      	for (;;) {
	  		bfd_set_error (bfd_error_no_error);

	  		arfile = bfd_openr_next_archived_file (file, arfile);
	  		if (arfile == NULL) {
	      		if (bfd_get_error () != bfd_error_no_more_archived_files) {
					nonfatal (bfd_get_filename (file));
				}
	      		break;
	    	}
	  		display_bfd(arfile);

	  		if (last_arfile != NULL) {
	    		bfd_close (last_arfile);
			}
	  		last_arfile = arfile;
		}
      	if (last_arfile != NULL) {
			bfd_close (last_arfile);
    	}
	} else {
  		if (bfd_check_format(file, bfd_core)) {
      		burp(&output, "Core %s\n", bfd_get_filename (file));
			flush_burp(&output);
		}
    	display_bfd (file);
	}
  	bfd_close (file);

	burp(&output, "End of report\n");
	flush_burp(&output);

	free(output.buffer);
done:
	OUT("objdumpfile")
	return;
}

void
objdump_init(char *programP)
{
	IN("objdump_init")
#if defined (HAVE_SETLOCALE)
#if defined (HAVE_LC_MESSAGES)
  	setlocale (LC_MESSAGES, "");
#endif
  	setlocale (LC_CTYPE, "");
#endif
	bindtextdomain ("binutils", "/usr/local/share/locale");
	textdomain ("binutils");


  	program_name = programP;
  	xmalloc_set_program_name (program_name);

  	bfd_init ();
  	set_default_bfd_target ();
	OUT("objdump_init")
}

int
objdump_finalize(void)
{
	IN("objdump_finalize")
	OUT("objdump_finalize")
  	return exit_status;
}

