#include <assert.h>
#include <stdio.h>
#include <string.h>

#include "xmalloc.h"
#include "util.h"
#include "signature.h"

void
Clth_string::dump(FILE *F, const char *prefixP)
{
	char *P, *endP;

	if ((P = m_stringP)) {
		if (prefixP) {
			fputs(prefixP, F);
		}
		fputc('"', F);
		for (endP = P + m_lth; P < endP; ++P) {
			fputc(*P, F);
		}
		fputc('"', F);
}	}
	
int
Clth_string::same(const Clth_string *otherP) const
{
	const char 	*string1P, *string2P;
	int			lth;

	string1P = m_stringP;
	if (!string1P) {
		return(0);
	}
	string2P = otherP->m_stringP;
	if (!string2P) {
		return(0);
	}
	lth = m_lth;
	if (lth != otherP->m_lth) {
		return(0);
	}
	if (strncmp(string1P, string2P, lth)) {
		return(0);
	}
	return (1);
}

void
Cfragment::dump(FILE *F, const char *prefixP)
{
	Clth_string	*valueP, *typeP;

	typeP  = &m_type;
	valueP = &m_value;

	if (valueP->m_stringP) {
		if (prefixP) {
			fputs(prefixP, F);
		}
		if (typeP->m_stringP) {
			fputc('[', F);
			typeP->dump(F, 0);
			fputc(']', F);
		}
		valueP->dump(F, 0);
}	}

void
Cfragments::dump(FILE *F, const char *prefixP)
{
	Cfragment	*nsP, *classnameP, *nameP, *discriminatorP, *typeP;

	nsP            = &m_ns;
	classnameP     = &m_classname;
	nameP          = &m_name;
	typeP          = &m_type;
	discriminatorP = &m_discriminator;
	
	if (prefixP) {
		if (nsP->m_value.m_stringP ||
			classnameP->m_value.m_stringP ||
			nameP->m_value.m_stringP ||
			discriminatorP->m_value.m_stringP ||
			typeP->m_value.m_stringP) {
			fputs(prefixP, F);
	}	}
	nsP->dump(F, "ns=");
	classnameP->dump(F, "class=");
	nameP->dump(F, "name=");
	discriminatorP->dump(F, "_");
	typeP->dump(F, "type=");
};

#define End_unresolved_qualifier_level end_simple_id

class Csignature_error {
};

Csignature::Csignature(void)
{
	m_templateP = 0;
}

Csignature::~Csignature(void)
{
	Xfree(m_templateP);
}

void
Csignature::fail(void)
{
	assert(0);
	throw new Csignature_error();
}

void
Csignature::reset(void)
{
	m_name.reset();
	m_localname.reset();
	Xfree(m_templateP);
}

void
Csignature::dump(FILE *F)
{
	m_name.dump(F, 0);
	m_localname.dump(F, "+");
	if (m_templateP) {
		fprintf(stderr, " template=\"%s\"", m_templateP);
};	}

void
Csignature::saw_ns(char *typeP, int type_lth, char *valueP, int lth)
{
	Cfragments	*nameP;
	Cfragment	*nsP;

	if ((nameP = m_nameP)) {
		nsP = &nameP->m_ns;
		if (nsP->m_value.m_stringP) {
			fail();
		}
		nsP->m_type.m_stringP  = typeP;
		nsP->m_type.m_lth      = type_lth;
		nsP->m_value.m_stringP = valueP;
		nsP->m_value.m_lth     = lth;
	}
}

void
Csignature::saw_name(char *typeP, int type_lth, char *valueP, int lth)
{
	Cfragments	*fragmentsP;
	Cfragment	*nameP, *classnameP, *nsP;
	
	if ((fragmentsP = m_nameP)) {
		nameP = &(fragmentsP->m_name);
		if (nameP->m_value.m_stringP) {
			classnameP = &(fragmentsP->m_classname);
			if (classnameP->m_value.m_stringP) {
				nsP = &(fragmentsP->m_ns);
				if (nsP->m_value.m_stringP) {
					fail();
				}
				*nsP = *classnameP;
			}
			*classnameP = *nameP;
		}
		nameP->m_type.m_stringP  = typeP;
		nameP->m_type.m_lth      = type_lth;
		nameP->m_value.m_stringP = valueP;
		nameP->m_value.m_lth     = lth;
}	}

void
Csignature::saw_discriminator(char *valueP, int lth)
{
	Cfragments	*fragmentsP;
	Cfragment	*discriminatorP;
	
	if ((fragmentsP = m_nameP)) {
		discriminatorP = &(fragmentsP->m_discriminator);
		if (discriminatorP->m_value.m_stringP) {
			fail();
		}
		discriminatorP->m_value.m_stringP = valueP;
		discriminatorP->m_value.m_lth     = lth;
}	}

void
Csignature::saw_type(char *valueP, int lth)
{
	Cfragments	*fragmentsP;
	Cfragment	*typeP;
	
	if ((fragmentsP = m_nameP)) {
		typeP = &(fragmentsP->m_type);
		if (typeP->m_value.m_stringP) {
			fail();
		}
		typeP->m_value.m_stringP = valueP;
		typeP->m_value.m_lth     = lth;
}	}

/* static */ char *
Csignature::end_call_offset(char *startP)
{
	char	*P;
	int 	c, i, max, lth;

	P = startP;
	switch(*P++) {
	case 'h':
		max = 1;
		break;
	case 'v':
		max = 2;
		break;
	default:
		fail();
	}
	for (i = 0; i < max; ++i) {
		if (*P == 'n') {	// Negative
			++P;
		}
		// Recover the call offset
		for (lth = 0; (c = *P) >= '0' && c <= '9'; ++P) {
			lth = lth * 10 + c - '0';
		}
		if (*P != '_') {
			fail();
		}
		++P;
	}
	return(P);
}

// <source-name> ::= <positive length number> <identifier>

char *
Csignature::end_source_name(char *startP)
{
	char	*P;
	int		lth, c;

	lth = 0;
	for (P = startP; (c = *P) >= '0' && c <= '9'; ++P) {
		lth = lth * 10 + c - '0';
	}
	if (!lth) {
		fail();
	}
	saw_name(startP, P - startP, P, lth);
	return(P + lth);
}


char *
Csignature::end_operator_name(char *startP)
{
	char 		*P;
	int			c;
	Cfragments	*saveP;

	P = startP;
	switch(*P++) {
	case 'v':
		// ::= v <digit> <source-name>	# vendor extended operator
		if ((c = *P) < '0' || c > '9') {
			fail();
		}
		saveP   = m_nameP;
		m_nameP = 0;
		P = end_source_name(++P);
		m_nameP = saveP;
		break;
	case 'c':
		// ::= cv <type>	# (cast)        
		if (*P == 'v') {
			saveP   = m_nameP;
			m_nameP = 0;
			P = end_type(++P);
			m_nameP = saveP;
			break;
		}
	default:
		++P;
	}
	saw_name("o", 1, startP, P - startP);
	return(P);
}

// <unqualified-name> ::= <operator-name>
//                    ::= <ctor-dtor-name>  
//                    ::= <source-name>   
//					  ::= <unnamed-type-name>

char *
Csignature::end_unqualified_name(char *startP)
{
	char 	*P;
	int		c, lth;

	P = startP;
	if (*P == 'L') {	// GNU extention to denote local function
		++P;
	}
	if ((c = *P) >= '0' && c <= '9') {
		return end_source_name(P);
	}

	if (c >= 'a' && c <= 'z') {
		// <operator-name>
		return end_operator_name(P);
	}

	// <ctor-dtor-name> ::= C1	# complete object constructor
	//             	    ::= C2	# base object constructor
	//					::= C3	# complete object allocating constructor
	//					::= D0	# deleting destructor
	//					::= D1	# complete object destructor
	//					::= D2	# base object destructor

	// <unnamed-type-name> ::= Ut [ <nonnegative number> ] _ 

	P = startP;
	switch (c) {
	case 'U':
		++P;
		if (*P != 't') {
			fail();
		}
		for (lth = 0; (c = *++P) >= '0' && c <= '9'; ) {
			lth = lth * 10 + c - '0';
		}
		if (c != '_') {
			fail();
		}
		++P;
		saw_name(startP, 2, startP, P - startP);
		break;
	case 'C':
		if ((c = *++P) < '1' || c > '3') {
			fail();
		}
		++P;
		saw_name(startP, 1, startP, P - startP);
		break;
	case 'D':
		if ((c = *++P) < '0' && c > '2') {
			fail();
		}
		++P;
		saw_name(startP, 1, startP, P - startP);
		break;
	default:
		fail();
	}
	return P;
}

// <template-param> ::= T_	# first template parameter
//    		        ::= T <parameter-2 non-negative number> _

/* static */ char *
Csignature::end_template_param(char *startP)
{
	char 	*P;
	int		c;
	
	P = startP;
	if (*P != 'T') {
		fail();
	}
	while ((c = *++P) >= '0' && c <= '9');
	if (*P != '_') {
		fail();
	}
	return ++P;
}

//  <expr-primary> ::= L <type> <value number> E   # integer literal
//                 ::= L <type <value float> E     # floating literal
//                 ::= L <mangled-name> E          # external name

char *
Csignature::end_expr_primary(char *startP)
{
	char	*P;
	int		c;

	P = startP;
	if (*P++ != 'L') {
		fail();
	}
	if (*P == '_' && P[1] == 'Z') {
		P = end_mangled_name(P);
		if (P) {
			if (*P != 'E') {
				fail();
			}
			++P;
		}
		return(P);
	}
	P = end_type(P);
	if (P) {
		if (*P == 'n') {
			++P;
		}
		for (; ((c = *P) >= '0' && c <= '9') || (c >= 'a' && c <= 'f'); ++P);
		if (*P == 'E') {
			++P;
		} else {
			fail();
	}	}
	return (P);
}

//  <template-arg> ::= <type>			# type or template
//			  	   ::= X <expression> E	# expression
//                 ::= <expr-primary>   # simple expressions
//				   ::= J <template-arg>* E

char *
Csignature::end_template_arg(char *startP)
{
	char 		*P;

	P     = startP;

	switch (*P) {
	case 'X':
		P = end_expression(++P);
		if (*P != 'E') {
			fail();
		}
		++P;
		break;
	case 'L':
		P = end_expr_primary(P);
		break;
	case 'J':
		++P;
		for (++P; *P != 'E';) {
			P = end_template_arg(P);
		}
		++P;
		break;
	default:
		P = end_type(P);
		break;
	}
	return(P);
}

// <template-args> ::= I <template-arg>+ E

char *
Csignature::end_template_args(char *startP)
{
	char 		*P, *P1, *P2;
	int			args;
	Cfragments	*saveP;

	saveP = m_nameP;
	P     = startP;

	if (*P++ != 'I') {
		fail();
	}
	m_nameP = 0;
	for (args = 0; *P != 'E';++args) {
		P = end_template_arg(P);
	}
	++P;

	if (!m_templateP) {
		m_templateP = dupstring(m_startP);
	}
	P2 = m_templateP + (startP - m_startP);
	for (P1 = startP; P1 < P; ++P1) {
		*P2++ = 1;
	}

	if (args < 1) {
		fail();
	}
	m_nameP = saveP;
	return(P);
}

char *
Csignature::end_simple_id(char *startP)
{
	char *P;

	P = end_source_name(startP);
	if (*P == 'I') {
		P = end_template_args(P);
	}
	return P;
}

// <base-unresolved-name> ::= <simple-id> 
//                            # unresolved name
//                        ::= on <operator-name> 
//                            # unresolved operator-function-id
//                        ::= on <operator-name> <template-args>
//                            # unresolved operator template-id
//                        ::= dn <destructor-name>
//                            # destructor or pseudo-destructor;
//                            # e.g. ~X or ~X<N-1>


char *
Csignature::end_base_unresolved_name(char *startP)
{
	char 	*P;
	int		c;
	
	P = startP;
	switch (*P) {
	case 'o':	// on <operator-name>  # unresolved operator-function-id
				// on <operator-name> <template-args>  # unresolved operator template-id
		if (*++P != 'n') {
			fail();
		}
		P = end_operator_name(++P);
		if (*P == 'I') {
			P = end_template_args(P);
		}
		break;
	case 'd':
		if (*++P != 'n') {
			fail();
		}
		switch (c = *++P) {
		case 'T':
		case 'D':
		case 'S':
			P = end_unresolved_type(P);
			break;
		default:
			P = end_simple_id(P);
		}
		break;
	default:
		P = end_simple_id(P);
	}
	return P;
}

// <decltype>  ::= Dt <expression> E  
//                 # decltype of an id-expression or class member access (C++0x)
//             ::= DT <expression> E  
//                 # decltype of an expression (C++0x)

char *
Csignature::end_decltype(char *startP)
{
	char 		*P;
	Cfragments	*saveP;

	P       = startP;
	if (*P != 'D') {
		fail();
	}
	switch (*++P) {
	case 't':
	case 'T':
		saveP   = m_nameP;
		m_nameP = 0;
		P = end_expression(++P);
		if (*P != 'E') {
			fail();
		}
		m_nameP = saveP;
		++P;
		break;
	default:
		fail();
	}
	return ++P;
}

// <unresolved-type> ::= <template-param>
//                    ::= <decltype>
//                    ::= <substitution>

char *
Csignature::end_unresolved_type(char *startP)
{
	char *P;

	P = startP;
	switch (*P) {
	case 'T':
		P = end_template_param(P);
		break;
	case 'D':
		P = end_decltype(P);
		break;
	case 'S':
		P = end_substitution(P);
		break;
	default:
		fail();
	}
	return P;
}

// <unresolved-name> ::= [gs] <base-unresolved-name> 
//                       # x or (with "gs") ::x
//                       sr <unresolved-type> <base-unresolved-name>
//                       # T::x / decltype(p)::x
//                       srN <unresolved-type> <unresolved-qualifier-level>+ E <base-unresolved-name>
//						 # T::N::x /decltype(p)::N::x
//                       [gs] sr <unresolved-qualifier-level>+ E <base-unresolved-name>  
//						 # A::x, N::y, A<T>::z; "gs" means leading "::"


char *
Csignature::end_unresolved_name(char *startP)
{
	char 	*P;
	int		saw_gs;

	saw_gs = 0;
	P = startP;
	switch (*P) {
	case 'g':
		if (*++P != 's') {
			fail();
		}
		saw_gs = 1;
		++P;
	}
	switch (*P) {
	case 'o':	// <base-unresolved-name>
		break;
	case 's':
		++P;
		switch (*P) {
		case 'N':
			if (saw_gs) {
				fail();
			}
			P = end_unresolved_type(++P);
			for (; *P != 'E'; ) {
				P = End_unresolved_qualifier_level(P);
			}
			break;
		case 'T':	// <template-param>
		case 'D':	// <decltype>
		case 'S':	// <substitution>
			if (saw_gs) {
				fail();
			}
			P = end_unresolved_type(P);
			break;
		default:
			for (; *P != 'E'; ) {
				P = End_unresolved_qualifier_level(P);
			}
			++P;
			break;
		}
		break;
	default:
		fail();
	}
	return end_base_unresolved_name(P);
}

// <CV-qualifiers> ::= [r] [V] [K] 	# restrict (C99), volatile, const

char *
Csignature::end_cv_qualifiers(char *startP)
{
	char *P;

	for (P = startP;; ++P) {
		switch(*P) {
		case 'r':
		case 'V':
		case 'K':
			continue;
		}
		return(P);
}	}

// <function-param> ::= fp <top-level CV-qualifiers> _
//                      # L == 0, first parameter
//        		    ::= fp <top-level CV-qualifiers> <parameter-2 non-negative number> _
//                      # L == 0, second and later parameters
//         		    ::= fL <L-1 non-negative number> p <top-level CV-qualifiers> _
//						# L > 0, first parameter
//				    ::= fL <L-1 non-negative number> p <top-level CV-qualifiers>
//                         <parameter-2 non-negative number> _   
//                      # L > 0, second and later parameters

char *
Csignature::end_function_param(char *startP)
{
	char *P;
	int	 lth, c;

	P   = startP;
	lth = 0;
	if (*P != 'f') {
		fail();
	}
	switch (*++P) {
	case 'L':
		for (++P; (c = *P) >= '0' && c <= '9'; ++P) {
			lth = lth * 10 + c - '0';
		}
		if (*P != 'p') {
			fail();
		}
	case 'p':
		P = end_cv_qualifiers(++P);
		for (; (c = *P) >= '0' && c <= '9'; ++P) {
			lth = lth * 10 + c - '0';
		}
		if (*P == '_') {
			break;
		}
	default:
		fail();
	}
	return ++P;
}

// <initializer> ::= pi <expression>* E       

char *
Csignature::end_initializer(char *startP)
{
	char *P;

	P = startP;
	if (*P != 'p' || P[1] != 'i') {
		fail();
	}
	P += 2;
	while (*P != 'E') {
		P = end_expression(P);
	}
	return ++P;
}

//<expression> ::= <unary operator-name> <expression>
//	           ::= <binary operator-name> <expression> <expression>
//	           ::= <ternary operator-name> <expression> <expression> <expression>
//			   ::= c1 <expression>+ E

//             ::= st <type>
//             ::= <template-param>
//             ::= sr <type> <unqualified-name>   # dependent name
//             ::= sr <type> <unqualified-name> <template-args>   # dependent template-id
//	           ::= <expr-primary>

char *
Csignature::end_expression(char *startP)
{
	char 	*P;
	int		c, expressions;

	for (P = startP;;) {
		switch (*P) {
		case 'a':
			switch (P[1]) {
			case 't':	// at <type> # alignof (a type)
				return end_type(P+2);
			}
			break;
		case 'c':
			switch (P[1]) {
			case 'v':	// cv <type> <expression> # conversion with one argument
						//::= cv <type> _ <expression>* E
				P = end_type(P+2);
				if (*P != '_') {
					continue;
				}
			case 'l':	// cl <expression>+ E  # call
				for (P +=2; *P != 'E'; )  {
					P = end_expression(P);
				}
				return ++P;
			case 'c': // cc <type> <expression> # const_cast<type> (expression)
				P = end_type(P+2);
				continue;
			}
			break;
		case 'd':
			switch (P[1]) {
			case 'l': // dl <expression> # delete expression
			case 'a': // da <expression> # delete[] expression
				P += 2;
				continue;
			case 'c': // dc <type> <expression> # dynamic_cast<type> (expression)
				P = end_type(P+2);
				continue;
			case 's': // ds <expression> <expression> # expr.*expr
				P = end_expression(P+2);
				continue;
			case 't': // dt <expression> <unresolved-name> # expr.name
				P = end_expression(P+2);
				return end_unresolved_name(P);
			}	
			break;
		case 'f':	// <function-param>
			switch (P[1]) {
			case 'L':
			case 'p':
				return end_function_param(P);
			}
			break;
		case 'g':
			if (P[1] != 's') {
				break;
			}
			switch (P[2]) {
			case 'n':
			case 'd':
				P += 2;
				continue;
			default:
				return end_unresolved_name(P);
			}
			break;
		case 'm':	// mm_ <expression> # prefix --
			switch (P[1]) {
			case 'm':
				if (P[2] == '_') {
					P += 3;
					continue;
				}
				break;
			}
			break;
		case 'n':
			switch (P[1]) {
			case 'w': // nw <expression>* _ <type> E 
					  // # new (expr-list) type
           	          // nw <expression>* _ <type> <initializer>
					  // # new (expr-list) type (init)

			case 'a': // na <expression>* _ <type> E
					  // # new[] (expr-list) type
               	  	  // na <expression>* _ <type> <initializer>
					  // # new[] (expr-list) type (init)
				for (P += 2; *P != '_'; ) {
					P = end_expression(P);
				}
				++P;
				P = end_type(P);
				if (*P != 'E') {
					P = end_initializer(P);
				} else {
					++P;
				}
				return(P);
			}
			break;
		case 'p':	// pp_ <expression> # prefix ++
			switch (P[1]) {
			case 'p':
				if (P[2] == '_') {
					P += 3;
					continue;
				}
				break;
			case 't': // pt <expression> <unresolved-name> # expr->name
				P = end_expression(P+2);
				return end_unresolved_name(P);
			}
			break;
		case 'r':	// rc <type> <expression> 
					// # reinterpret_cast<type> (expression)
			switch (P[1]) {
			case 'c':
				P = end_type(P+2);
				continue;
			}
			break;
		case 's':
			switch (P[1]) {
			case 'c': // sc <type> <expression>
					  // # static_cast<type> (expression)
				P = end_type(P+2);
				continue;
			case 'p': // sp <expression> # pack expansion
				P += 2;
				continue;
			case 'r': // <unresolved-name>
				return end_unresolved_name(P);
			case 't': // st <type> # sizeof (a type)
				return end_type(P+2);
			case 'Z': // sZ <template-param> # size of a parameter pack
               	  	  // sZ <function-param> # size of a function parameter pack
				P += 2;
				if (*P == 'T') {
					P = end_template_param(P);
				} else {
					P = end_function_param(P);
				}
				return(P);
			}
			break;
		case 't':
			switch (P[1]) {
			case 'i': // ti <type> # typeid (type)
				return end_type(P+1);
			case 'e': // te <expression> # typeid (expression)
			case 'w': // tw <expression> # throw expression
				P += 2;
				continue;
			case 'r': // tr # throw with no operand (rethrow)
				return P+2;
			}
			break;
		case 'T': // <template-param>
			return end_template_param(P);
		case 'L': // <expr-primary> ::= L <type> <value number> E 
				  // # integer literal
				  //                 ::= L <type <value float> E
				  // # floating literal
				  //                 ::= L <mangled-name> E
				  // # external name
			return end_expr_primary(P);
		}

		//<expression> ::= <unary operator-name> <expression>
		//	           ::= <binary operator-name> <expression> <expression>
		//	           ::= <trinary operator-name> <expression> <expression> <expression>
		P = end_operator_name(P);
		for (expressions = 0;; ++expressions) {
			if (((c = *P) < 'a' || c > 'z') &&	/* Not Operator */
				c != 'L'	&&					/* Not primary */
				c != 'T') {						/* Not template parameter */
				// Not an expression
				break;
			}
			P = end_expression(P);
		}
		if (expressions < 1 || expressions > 3) {
			fail();
		}		
		return (P);
}	}

// <substitution> ::= S <seq-id> _
//                ::= S_

char *
Csignature::end_substitution(char *startP)
{
	char	*P;
	int		c;

	P = startP;
	if (*P != 'S') {
		fail();
	}
	while (((c = *++P) >= '0' && c <= '9') || (c >= 'A' && c <= 'Z'));
	if (*P != '_') {
		fail();
	}
	++P;
	saw_name(startP, 1, startP+1, P - startP - 1);
	return P;
}

// <name> ::= <nested-name>
//        ::= <unscoped-name>
//        ::= <unscoped-template-name> <template-args>
//        ::= <local-name>	# See Scope Encoding below

// <unscoped-name> ::= <unqualified-name>
//      		   ::= St <unqualified-name>   # ::std::

// <unscoped-template-name> ::= <unscoped-name>
//          			    ::= <substitution>


char *
Csignature::end_name(char *startP)
{
	char 		*P, *atP, *classP;
	int			seen_prefix, last_source_name, lth, c;
	Cfragments	*saveP;

	P = classP = startP;
	switch (*P) {
	case 'S':
		if (P[1] == 't') {
			// <unscoped-name> ::= St <unqualified-name>   # ::std::
			saw_ns(P, 2, "::std::", 7);
			P = end_unqualified_name(P+2);
			if (*P != 'I') {
				// <name> ::= <unscoped-name>
				return(P);
			}
		} else {
			// <unscoped-template-name> ::= <substituion>
			P = end_substitution(P);
		}
		// <name> ::= <unscoped-template-name> <template-args>
		return end_template_args(P);
	case 'N':
		// <nested-name> ::= N [<CV-qualifiers>] <prefix> <unqualified-name> E
		//               ::= N [<CV-qualifiers>] <template-prefix> <template-args> E
		P = end_cv_qualifiers(++P);

		//          <prefix> ::= <prefix> <unqualified-name>
		//   	             ::= <template-prefix> <template-args>
		//                   ::= <template-param>
		//					 ::= <decltype>
		// 	                 ::= # empty
		// 	                 ::= <substitution>
		//					 ::= <prefix> <data-member-prefix>
		// 
		// <template-prefix> ::= <prefix> <template unqualified-name>
		//                   ::= <template-param>
		//                   ::= <substitution>

		/* A template-param or a substitution must occur at the start
		 * of a prefix if present
		 */

		seen_prefix = 1;
		switch (*P) {
		case 'T':
			P = end_template_param(P);
			break;
		case 'S':
			P = end_substitution(P);
			break;
		case 'D': // <decltype>  ::= Dt <expression> E
             	  //             ::= DT <expression> E
			return end_decltype(P);
		default:
			seen_prefix = 0;
			break;
		}
		
		last_source_name = 0;
		do {
			if (last_source_name) {
				// <data-member-prefix> := <member source-name> M
				// Horrible clash with other uses of M
				if (*P == 'M') {		
					++P;
				}
				last_source_name = 0;
			}
			switch (*P) {
			case 'I':
				if (!seen_prefix) {
					fail();
				}
				P = end_template_args(P);
				break;
			default:
				seen_prefix = 1;
				if (*P >= '0' && *P <= '9') {
					last_source_name = 1;
				}
				P = end_unqualified_name(P);
				break;
			}
		} while (*P != 'E');
		return ++P;
	case 'Z':
		// <local-name> := Z <function encoding> E <entity name> [<discriminator>]
        //              := Z <function encoding> E s [<discriminator>]
		// <local-name> := Z <function encoding> Ed [ <parameter number> ] _ <entity name>

		
		P = end_encoding(++P);
		if (*P++ != 'E') {
			fail();
		}
		saveP   = m_nameP;
		m_nameP = &m_localname;
		switch (*P) {
		case 's':
			++P;
			break;
		case 'd':
			while ((c = *++P) >= '0' && c <= '9');
			if (*P != '_') {
				fail();
			}
			return end_name(P);
		default:
			P = end_name(P);
		}
		if (*P == '_') {
  			// <discriminator> := _ <non-negative number> 
			atP = P;
			for (lth = 0; (c = *++P) >= '0' && c <= '9'; ) {
				lth = lth * 10 + c - '0';
			}
			saw_discriminator(atP, P - atP);
		}
		m_nameP = saveP;
		return(P);
	default:
		// <name> ::= <unqualified-name> [<template_args>]

		// <unqualified-name> ::= <operator-name>
        //                    ::= <ctor-dtor-name>  
        //                    ::= <source-name>   

		P = end_unqualified_name(P);
		if (*P == 'I') {
			P = end_template_args(P);
		}
		return(P);
}	}

char *
Csignature::end_type(char *startP)
{
	char 		*P;
	int			c, lth;
	Cfragments	*saveP;

	P = end_cv_qualifiers(startP);
	for (;;) {
		// <type> ::= <CV-qualifiers> <type>
		//        ::= P <type>	# pointer-to
		// 	      ::= R <type>	# reference-to
		// 	      ::= C <type>	# complex pair (C 2000)
		// 	      ::= G <type>	# imaginary (C 2000)
		//        ::= U <source-name> <type> # vendor extended type qualifier
		// 

		switch (*P) {
		case 'U': // ::= U <source-name> <type> # vendor extended type qualifier
			P = end_source_name(++P);
			continue;
		case 'P': // P <type>	# pointer-to
		case 'R': // R <type>	# reference-to
		case 'O': // rvalue reference-to (C++0x)
		case 'C': // complex pair (C 2000)
		case 'G': // imaginary (C 2000)
			++P;
			continue;
		case 'v':	// void
		case 'w':	// wchar_t
		case 'b':	// bool
		case 'c':	// char
		case 'a':	// signed char
		case 'h':	// unsigned char
		case 's':	// short
		case 't':	// unsigned short
		case 'i':	// int
		case 'j':	// unsigned int
		case 'l':	// long
		case 'm':	// unsigned long
		case 'x':	// long long, __int64
		case 'y':	// unsigned long long, __int64
		case 'n': 	// __int128
		case 'o':	// unsigned __int128
		case 'f':	// float
		case 'd':	// double
		case 'e':	// long double, __float80
		case 'g':	// __float128
		case 'z':	// ellipsis ...
			// builtin
			++P;
			break;
		case 'u': 	// u <source-name>	# vendor extended builtin type
			P = end_source_name(++P);
			break;
		case 'D':
			switch(*++P) {
			case 'p':	// Dp <type> # pack expansion (C++0x)
				++P;
				continue;
			case 't':	// <decltype>
			case 'T':	// <decltype>
				P = end_decltype(--P);
				break;
						// IEEE 754r floating point
			case 'd':	// 64 bit
			case 'e':	// 128 bit
			case 'f':	// 32 bit
			case 'h':	// 16 bit
			case 'i':	// char32_t
			case 's':	// char16_t
			case 'a':	// auto
			case 'n':	// std::nullptr_t (ie. decltype(nullptr))
				++P;
				break;
			default:
				fail();
			}
			break;
		case 'F':
			// <function-type> ::= F [Y] <bare-function-type> E
  			// <bare-function-type> ::= <signature type>+
			// # types are possible return type, then parameter types

			++P;
			if (*P == 'Y') {	// extern "C". 
				++P;
			}
			P = end_bare_function_type(P);
			if (*P != 'E') {
				fail();
			}
			++P;
			break;
		case 'A':
			// <array-type> ::= A <positive dimension number> _ <element type>
	       	//              ::= A [<dimension expression>] _ <element type>
			++P;
			if ((c = *P) >= '0' && c <= '9') {
				for (lth = c - '0'; (c = *++P) >= '0' && c <= '9'; lth = lth * 10 + c - '0');
			} else if (*P != '_') {
				saveP   = m_nameP;
				m_nameP = 0;
				P       = end_expression(P);
				m_nameP = saveP;
			}
			if (*P != '_') {
				fail();
			}
			continue;
		case 'M':
			// <pointer-to-member-type> ::= M <class type> <member type>
			P = end_type(++P);
			if (P) {
				P = end_type(P);
			}
			break;
		case 'T':
			// <template-param> ::= T_	# first template parameter
		   	//					::= T <parameter-2 non-negative number> _
			P = end_template_param(P);
			if (*P == 'I') {
				// ::= <template-template-param> <template-args>
				// <template-template-param> ::= <template-param>
			    //                           ::= <substitution>


				P = end_template_args(P);
			}
			break;
		case 'S':
			P = end_substitution(P);
			// <substitution> ::= S <seq-id> _
		 	//                ::= S_
			while (((c = *++P) >= '0' && c <= '9') || (c >= 'A' && c <= 'Z'));
			if (*P != '_') {
				fail();
			}
			++P;
			if (*P == 'I') {
				// ::= <template-template-param> <template-args>
				P = end_template_args(P);
			}
			break;
		default:
			// <class-enum-type> ::= <name>
			P = end_name(P);
		}
		return(P);
}	}

// <bare-function-type> ::= <signature type>+
//	# types are possible return type, then parameter types

char *
Csignature::end_bare_function_type(char *startP)
{
	char 		*P, *start_typeP;
	char		c;
	Cfragments	*saveP;
	
	P       = startP;
	saveP   = m_nameP;
	m_nameP = 0;

	start_typeP = 0;
	while ((c = *P) && c != 'E') {
		if (!start_typeP) {
			start_typeP = P;
		}
		P = end_type(P);
	}
	m_nameP = saveP;
	if (start_typeP) {
		saw_type(start_typeP, P - start_typeP);
	}
	return(P);
}

// <encoding> ::= <function name> <bare-function-type>
//            ::= <data name>
//			  ::= <special-name>

char *
Csignature::end_encoding(char *startP)
{
	char 	*P;

	P = startP;
	switch (*P) {
	case 'G': 	// <special-name> ::= GV <object name>
				// # Guard variable for one-time initialization # No <type>
		++P;
		if (*P != 'V') {
			fail();
		}
		return end_name(++P);
	case 'T':
		switch (*++P) {
		case 'V':	// ::= TV <type>	# virtual table
		case 'T':	// ::= TT <type>	# VTT structure (construction vtable index)
		case 'I':	// ::= TI <type>	# typeinfo structure
		case 'S':	// ::= TS <type>	# typeinfo name (null-terminated byte str)
			P = end_type(++P);
			// The type is not a name but the classname
			saw_name(0, 0, 0, 0);
			return P;
		case 'h':		// Virtual overide thunk with fixed this adjustment
		case 'v':		// Virtual override thunk with virtual adjustment
			// <special-name> ::= Th <offset number> _ <base encoding>
			// <special-name> ::= Tv <offset number> _ <virtual offset number> _ <base encoding>
			//      # base is the nominal target function of thunk
			P = end_call_offset(P);
			break;
		case 'c':
			// ::= Tc <call-offset> <call-offset> <base encoding>
		    //  # base is the nominal target function of thunk
		    //  # first call-offset is 'this' adjustment
		    //  # second call-offset is result adjustment
			P =	end_call_offset(P);
			P =	end_call_offset(P);
			break;
		default:
			fail();
		}
		return end_encoding(P);	// <base encoding>
	default:
		// <encoding> ::= <function name> <bare-function-type>
		//            ::= <data name>
		P = end_name(startP);
		P = end_bare_function_type(P);
		return(P);
}	}

// <mangled-name> ::= _Z <encoding>

char *
Csignature::end_mangled_name(char *startP)
{
	char 	*P;

	if (startP[0] == '_' && startP[1] == 'Z') {
		P = end_encoding(startP+2);
	} else {
		// Not mangled
		P = startP + strlen(startP);
		saw_name(0, 0, startP, P - startP);
	}
	return(P);
}

/* This logic must fully scan a mangled_name and return the logical end
   of that mangled name or 0 on error.  In addition it will populate
   *classPP with a pointer to the start of a class name if one found
   *namePP  with the start of the actual function name
   *name_lthP with the length of this function name.
   The class name length is simply *namePP - *classPP since function names
   always immediately follow their class name
 */

char *
Csignature::signature(char *startP)
{
	char 	*P, *P1, *P2;
	int		c;

	try {
		reset();
		
		m_startP = startP;
		m_nameP  = &m_name;
		P = end_mangled_name(startP);
		if (*P) {
			fail();
		}
		if ((P1 = m_templateP)) {
			P2 = P1;
			do {
				c = *P1++;
				if (c != 1) {
					*P2++ = c;
				}
			} while (c);
		}

/*
		fprintf(stderr, "signature %s:", startP);
		dump(stderr);
		fputc('\n', stderr);
*/
	} catch (Csignature_error e) {
		P = 0;
	}
	return(P);
}

int
Csignature::same_class(const Csignature *otherP) const
{
	const Clth_string	*string1P, *string2P;

	string1P = &(m_name.m_classname.m_value);
	string2P = &(otherP->m_name.m_classname.m_value);
	if (!string1P->same(string2P)) {
		return(0);
	}
	string1P = &(m_name.m_ns.m_value);
	string2P = &(otherP->m_name.m_ns.m_value);
	
	if (string1P->m_stringP || string2P->m_stringP) {
		if (!string1P->same(string2P)) {
			return(0);
	}	}
	return(1);
}

int
Csignature::same_function_name(const Csignature *otherP) const
{
	const Clth_string	*string1P, *string2P;

	string1P = &(m_name.m_name.m_value);
	string2P = &(otherP->m_name.m_name.m_value);

	return string1P->same(string2P);
}

