#include "typedef.h"

#include <assert.h>
#include <time.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "object.h"
#include "xmalloc.h"
#include "util.h"
#include "opcode.h"
#include "code.h"
#include "source.h"
#include "function.h"
#include "detect.h"
#include "clics_output.h"

#ifndef NDEBUG
//#define DEBUG_DETECT
//#define DEBUG_HILLCLIMB
//#define TEST_HILLCLIMB
#define PROOF
#endif

extern int		g_hypertext_cnt;
extern int		g_top_clones;
extern int		g_top_cols;
extern int		g_report_java;
extern int		g_report_assembler;
extern int		g_verbose;
extern char		*g_html_dirP;
extern int		g_comments;
extern int		g_min_match_sequence;
extern int		g_min_function_sequence;
extern int		g_mismatch_cost;
extern int		g_same_cost;
extern int		g_match_lines;
extern int  	g_comparisons;
extern int		g_hillclimb;
extern int		g_hillclimb_timeout;
extern int		g_iterations;
extern int		g_best_weight;
extern time_t	g_hillclimb_elapse;

extern long long	g_original_matches;
extern long long	g_extra_matches;
extern long long	g_cancelled_matches;
extern int			g_max_blocks_seen;
extern int			g_max_iterations_seen;

static	Csource			*source1P;		// Source 1st side looking at
static	Csource			*source2P;		// Source 2nd side looking at
static	Cfunction		*function1P;	// Function 1st side looking at
static	Cfunction		*function2P;	// Function 2nd side looking at
static	codeT			*function_code1P;// Start of function1 code
static	codeT			*function_code2P;// Start of function2 code
static	codeT			*function_end1P;// Beyond end of function1 code
static	codeT			*function_end2P;// Beyond end of function2 code
static	codeT			*start_code1P;	// Comparing from 1st side
static	codeT			*start_code2P;	// Comparing from 2nd side
static	codeT			*end_code1P;	// May not compare to here on 1st side
static	codeT			*end_code2P;	// May not compare to here on 2nd side
static	codeT			*best_end1P;	// Best end of side 1
static	codeT			*best_end2P;	// Best end of side 2
static  int				forward_jumps;	// Matched forward jumps in clone

extern FILE		*stdoutF;
extern FILE		*redirectF;

static Csource	*last_sourceP = 0;
static FILE		*sourceF      = 0;
static FILE		*cloneF       = 0;

typedef struct {
	unsigned int	weight;
	unsigned int	html;
} topT;

static topT		*top_clonesP       = 0;
static topT		*top_clones_endP   = 0;
static int		top_clones_cnt     = 0;

/*
void
trap(void)
{
}
*/

void
initialise_detect(void)
{
	int	need;

	if (!g_html_dirP || (!g_report_assembler && !g_report_java)) {
		g_top_clones = 0;
	}
	if (g_top_clones > 0) {
		need = g_top_clones * sizeof(topT);
		top_clonesP = (topT *) Xmalloc(need);
		if (!top_clonesP) {
			outofmemory();
		}
		memset(top_clonesP, 0, need);
		top_clones_endP = (topT *) (((char *) top_clonesP) + need);
	}
}

void
start_report(void)
{
	clics_sources();
	if (g_html_dirP) {
		fputs("<p>\n<table>\n", redirectF);
}	}

static int		g_lth1, g_lth2;
static time_t	start_clock, end_clock;

#ifdef PROOF
static void
proof(void)
{
	static	codeT	*code1aP, *code2aP, *code1bP, *code2bP;
	static	codeT	*code1a_jumpP, *code2a_jumpP;
	static	codeT	*code1b_jumpP, *code2b_jumpP;

	assert(start_code1P->m_matchP == start_code2P);
	assert(start_code2P->m_matchP == start_code1P);
	
	for (code1aP = start_code1P; code1aP < best_end1P; ++code1aP) {
		if (!(code2aP = code1aP->m_matchP)) {
			continue;
		}
		assert(code2aP->m_matchP == code1aP);
		code1a_jumpP = code1aP->m_jump.P;
		code2a_jumpP = code2aP->m_jump.P;
		if (!code1a_jumpP) {
			assert(!code2a_jumpP);
		} else if (code1a_jumpP == code1aP) {
			assert(code2a_jumpP == code2aP);
		} else if (code1a_jumpP > code1aP) {
			assert(code2a_jumpP > code2aP);
		} else {
			assert(code2a_jumpP && code2a_jumpP < code2aP);
			assert(code1a_jumpP >= start_code1P);
			assert(code2a_jumpP >= start_code2P);
		}
		for (code1bP = start_code1P; code1bP < best_end1P; ++code1bP) {
			if (code1aP == code1bP) {
				continue;
			}
			if (!(code2bP = code1bP->m_matchP)) {
				continue;
			}
			assert((code1aP > code1bP && code2aP > code2bP) ||
                   (code1aP < code1bP && code2aP < code2bP));
			if (!code1a_jumpP) {
				continue;
			}
			assert((code1a_jumpP <= code1bP && code2a_jumpP <= code2bP) ||
				   (code1a_jumpP > code1bP  && code2a_jumpP > code2bP));

			code1b_jumpP = code1bP->m_jump.P;
			code2b_jumpP = code2bP->m_jump.P;
			if (!code1b_jumpP || !code2b_jumpP) {
				continue;
			}
			if (code1a_jumpP >= best_end1P && code2a_jumpP >= best_end2P) {
				continue;
			}
			if (code1b_jumpP >= best_end1P && code2b_jumpP >= best_end2P) {
				continue;
			}
			assert((code1a_jumpP >= code1b_jumpP && code2a_jumpP >= code2b_jumpP) ||
				   (code1a_jumpP <= code1b_jumpP && code2a_jumpP <= code2b_jumpP));
	}	}	
}
#endif

extern FILE	*clicsF;

static void
dump_clics_clone(codeT *start_codeP, codeT *best_endP)
{
	Csource	*sourceP;
	char	**sourcePP;
	int		clone_start_lineno, clone_start_offset, clone_end_lineno, clone_end_column, clone_end_offset;
	char	*start_sourceP, *last_charP;


	sourceP  = start_codeP->m_functionP->m_sourceP;
	sourcePP = sourceP->m_sourcePP;
	if (sourcePP == (char **) -1) {
		clone_start_lineno = 0;
		clone_start_offset = 0;
		clone_end_lineno   = 0;
		clone_end_column   = 0;
		clone_end_offset   = 0;
	} else {
		start_sourceP      = sourcePP[0];
		clone_start_lineno = start_codeP->m_source_lineno - 1;
		assert(0 <= clone_start_lineno && clone_start_lineno < sourceP->m_source_lines);
		clone_start_offset = sourcePP[clone_start_lineno] - start_sourceP;
		clone_end_lineno   = best_endP[-1].m_source_lineno;
		assert(0 < clone_end_lineno && clone_end_lineno <= sourceP->m_source_lines);
		last_charP         = sourcePP[clone_end_lineno] - 1;
		clone_end_offset   = last_charP - start_sourceP;
		--clone_end_lineno;
		clone_end_column   = last_charP - sourcePP[clone_end_lineno]; 
	}
	fprintf(clicsF, "%d\t%d,0,%d\t%d,%d,%d\t", 
		sourceP->m_number, 
		clone_start_lineno, 
		clone_start_offset,
		clone_end_lineno,
		clone_end_column,
		clone_end_offset);
}

static void
report_clone(void)
{
	extern	FILE			*taF;
	extern	FILE			*tempF;

	static	int				lineno1, lineno2, next_lineno1, next_lineno2;
	static	int				different1, different2;

	static	u2T				*source_name1P, *source_name2P;
	static	codeT			*at_code1P, *at_code2P;
	static	int				offset1, offset2;
	static	int				report_java;
	static	const ocodeT	*opcode1P, *opcode2P;
	static	unsigned int	hypertext_cnt;

	static	unsigned int heap_index;
	static	topT	*heapP, *leftP, *rightP;
	static	unsigned int	weight;
	static	char			*P;
	static time_t			clock_now;

	source_name1P = source1P->get_classname(source1P->m_this_class);
	source_name2P = source2P->get_classname(source2P->m_this_class);

	// Increment the detail page number
	g_hypertext_cnt += 1;

	if (taF) {
		Cclone *clone1P, *clone2P;

		clone1P = function1P->locate_clone(start_code1P, best_end1P);
		clone2P = function2P->locate_clone(start_code2P, best_end2P);

		fprintf(taF, "R C%p C%p\n", clone1P, clone2P);
		if (g_html_dirP && (g_report_java || g_report_assembler)) {
			fprintf(tempF, "(R C%p C%p) { html=%u }\n", clone1P, clone2P, g_hypertext_cnt);
	}	}

	if (clicsF) {
		dump_clics_clone(start_code1P, best_end1P);
		dump_clics_clone(start_code2P, best_end2P);
		fprintf(clicsF, "%ld\n", (long) (best_end1P - start_code1P + best_end2P - start_code2P));
	}

	if (top_clonesP) {
		// Use a heap with the smallest weight at the top

		weight = g_lth1 + g_lth2;
		if (top_clonesP->weight < weight) {
			++top_clones_cnt;
			heap_index = 1;
			heapP      = top_clonesP;
			// Add entry to the heap
			for (;;) {
				leftP = heapP + heap_index;
				if (top_clones_endP <= leftP) {
					goto found_heap_entry;
				}
				rightP = leftP + 1;
				if ((top_clones_endP <= rightP) || (leftP->weight <= rightP->weight)) {
					if (weight <= leftP->weight) {
						goto found_heap_entry;
					}
					*heapP       = *leftP;
					heapP        = leftP;
					heap_index <<= 1;
					continue;
				}
				if (weight <= rightP->weight) {
found_heap_entry:	heapP->weight = weight;
					heapP->html   = g_hypertext_cnt;
					break;
				}
				*heapP       = *rightP;
				heapP        = rightP;
				heap_index   = (heap_index << 1) + 1;
			}
	}	}

	if (g_html_dirP) {
		if (last_sourceP != source1P) {
			if (sourceF) {
				fprintf(sourceF,"</table>\n<p>\nEnd of report</body>\n</html>\n");
				fclose(sourceF);
			}
			hypertext_cnt = source1P->m_hypertext_cnt;
			sprintf(g_filename,"%s/%u.html", g_html_dirP, hypertext_cnt);
			sourceF = open_file(g_filename);
			last_sourceP = source1P;

			stdoutF       = sourceF;
			
			fputs( HTML_HEADER "<title>Clones in ", stdoutF);
			dump_classname(source_name1P);
			fputs("</title>\n<body>\n<a href=index.html>Top</a>\n<h3>Clones in ", stdoutF);
			dump_classname(source_name1P);
			fputs("</h3>\n<table>\n", stdoutF);

			// On top level page write
			stdoutF   = redirectF;
			clock_now = time(0);
			P         = ctime(&clock_now);
			P[19]     = 0;
			fprintf(stdoutF, "<tr><td>%s</td><td><a href=\"%u.html\">", P + 11, hypertext_cnt);
			dump_classname(source_name1P);
			fprintf(stdoutF, "</a></td></tr>\n");
		}

		cloneF = 0;
		if (g_report_java || g_report_assembler) {
			sprintf(g_filename,"%s/%u.html", g_html_dirP, g_hypertext_cnt);
			cloneF = open_file(g_filename);
			fputs(
HTML_HEADER
"<script language=\"javascript\">\n"
"<!--\n"
"var a, s;\n"
"var a_shown='none'\n"
"var s_shown='none'\n"
"function toggleA() {\n"
"  for (i = 0; i < a.length; i++) { a[i].style.display = a_shown}\n"
"  a_shown = (a_shown == 'none' ? 'inline' : 'none')\n"
"}\n"
"function toggleS() {\n"
"  for (i = 0; i < s.length; i++) { s[i].style.display = s_shown}\n"
"  s_shown = (s_shown == 'none' ? 'inline' : 'none')\n"
"}\n"
"function init(){ \n"
"  var ai = 0;\n"
"  var si = 0;\n"
"  var tr = document.all.tags(\"tr\");\n"
"  var e, className, i;\n"
"  for (i = 0; i < tr.length; i++) {\n"
"    e = tr[i];\n"
"    className = \"\" + e.className;\n"
"    switch (className) {\n"
"      case 'a': ++ai; break;\n"
"      case 's': ++si; break;\n"
"  } }\n"
"  a  = new Array(ai);\n"
"  s  = new Array(si);\n"
"  ai = 0;\n"
"  si = 0;\n"
"  for (i = 0; i < tr.length; i++) {\n"
"    e = tr[i];\n"
"    className = \"\" + e.className;\n"
"    switch(className) {\n"
"    case 'a': a[ai] = e; ++ai; break;\n"
"    case 's': s[si] = e; ++si; break;\n"
"} } }\n"
"//-->\n"
"</script>\n"
"</head>\n"
"<body onLoad=\"init()\">\n"
"<p>\n"
"[<a href=index.html>Top</a>] [<a href=\"javascript:toggleA()\">Toggle</a>] Pcode\n"
"[<a href=\"javascript:toggleS()\">Toggle</a>] Source\n"
"<p>\n"
				  , cloneF);
		}
			
		stdoutF = sourceF;
		fputs("<tr><td><i>", stdoutF);
		fprintf(stdoutF, "<a href=\"%u.html\">", g_hypertext_cnt);
		dump_access(function1P->m_access_flags);
		dumpUnicode(function1P->m_nameP);
		source1P->dump_descriptor(function1P->m_descriptorP, function1P, (function1P->m_access_flags & 8));
		fputs("</a>", stdoutF);
		fprintf(stdoutF, "</i> <font color=\"red\">[%d]</font></td>\n    <td><i>", (int) (best_end1P - start_code1P));
		dump_access(function2P->m_access_flags);
		dump_classname(source_name2P);
		fputc('.', stdoutF);
		dumpUnicode(function2P->m_nameP);
		source2P->dump_descriptor(function2P->m_descriptorP, function2P, (function2P->m_access_flags & 8));
		fprintf(stdoutF, "</i> <font color=\"blue\">[%d]</font></td>\n", (int) (best_end2P - start_code2P));

		fputs("</i></td>\n</tr>\n", stdoutF);
		stdoutF = cloneF;
	} 
	fputs("<p>\n<table>\n", stdoutF);
	fprintf(stdoutF, "<tr><td><b><a href=%u.html>", source1P->m_hypertext_cnt);
	dump_classname(source_name1P);
	fprintf(stdoutF, "</a></b></td>\n    <td><b><a href=%u.html>", source2P->m_hypertext_cnt);
	dump_classname(source_name2P);
	fputs("</a></b></td>\n</tr>\n<tr><td><i>", stdoutF);
	dump_access(function1P->m_access_flags);
	dumpUnicode(function1P->m_nameP);
	source1P->dump_descriptor(function1P->m_descriptorP, function1P, (function1P->m_access_flags & 8));
	fprintf(stdoutF, "</i> <font color=\"red\">[%d]</font></td>\n    <td><i>", (int) (best_end1P - start_code1P));
	dump_access(function2P->m_access_flags);
	dumpUnicode(function2P->m_nameP);
	source2P->dump_descriptor(function2P->m_descriptorP, function2P, (function2P->m_access_flags & 8));
	fprintf(stdoutF, "</i> <font color=\"blue\">[%d]</font></td>\n</tr>\n", (int) (best_end2P - start_code2P));

	if (g_report_java || g_report_assembler) {

		fputs("<tr><td><hr></hr></td><td><hr></hr></td></tr>", stdoutF); 

		at_code1P = start_code1P;
		at_code2P = start_code2P;

		if ((report_java = g_report_java)) {

			if (!source1P->m_sourcePP) {
				source1P->load_source();
			}
			if (!source2P->m_sourcePP) {
				source2P->load_source();
			}
			if (!source1P->m_source_lines && !source2P->m_source_lines) {
				// Can open neither source
				report_java = 0;
			}
			next_lineno1 = ((at_code1P < best_end1P) ? (int) at_code1P->m_source_lineno : -1);
			next_lineno2 = ((at_code2P < best_end2P) ? (int) at_code2P->m_source_lineno : -1);
			different1   = 0;
			different2   = 0;
		}

		offset1 = at_code1P - function_code1P;
		offset2 = at_code2P - function_code2P;
		for (; at_code1P < best_end1P || at_code2P < best_end2P; ) {

			if (g_comments) {
				opcode1P = opcodes + at_code1P->m_pcode;
				opcode2P = opcodes + at_code2P->m_pcode;

				fputs("<tr class=a><td>", stdoutF);
				if (best_end1P <= at_code1P || (at_code1P->m_matchP && at_code1P->m_matchP != at_code2P) ) {
					fputs("</td>\n    <td><font color=\"blue\">", stdoutF);
					dumpString(opcode2P->commentP);
					fputs("</font></td>\n</tr>\n", stdoutF);
				} else if (!at_code1P->m_matchP) {
					fputs("<font color=\"red\">", stdoutF);
					dumpString(opcode1P->commentP);
					fputs("</font></td>\n    <td>", stdoutF);
				
					if (at_code2P < best_end2P && !at_code2P->m_matchP) {
						fputs("<font color=\"blue\">", stdoutF);
						dumpString(opcode2P->commentP);
						fputs("</font>", stdoutF);
					}
					fputs("</td>\n</tr>\n", stdoutF);
				} else {
					dumpString(opcode1P->commentP);
					fputs("</td>\n    <td>", stdoutF);
					dumpString(opcode2P->commentP);
					fputs("</td>\n</tr>\n", stdoutF);
			}	} 

			if (g_report_assembler) {
				if (best_end1P <= at_code1P || (at_code1P->m_matchP && at_code1P->m_matchP != at_code2P) ) {
					fprintf(stdoutF, "<tr class=a><td></td>\n    <td><font color=\"blue\">%d: ", offset2);
					source2P->dump_pcode(function2P, at_code2P);
					fputs("</font></td>\n</tr>\n", stdoutF);
					++at_code2P;
					++offset2;
					different2 = 1;
				} else if (!at_code1P->m_matchP) {
					fprintf(stdoutF, "<tr class=a><td><font color=\"red\">%d: ", offset1);
					source1P->dump_pcode(function1P, at_code1P);
					fputs("</font></td>\n    <td>", stdoutF);
					++at_code1P;
					++offset1;
					different1 = 1;
				
					if (at_code2P < best_end2P && !at_code2P->m_matchP) {
						fprintf(stdoutF, "<font color=\"blue\">%d: ", offset2);
						source2P->dump_pcode(function2P, at_code2P);
						fputs("</font>", stdoutF);
						++at_code2P;
						++offset2;
						different2 = 1;
					}
					fputs("</td>\n</tr>\n", stdoutF);
				} else {
					fputs("<tr class=a><td>", stdoutF);
					source1P->dump_pcode(function1P, at_code1P);
					fputs("</td>\n    <td>", stdoutF);
					source2P->dump_pcode(function2P, at_code2P);
					fputs("</td>\n</tr>\n", stdoutF);

					++at_code1P;
					++offset1;
					++at_code2P;
					++offset2;
				} 
			} else {
				// Not reporting assembler

				if (best_end1P <= at_code1P || (at_code1P->m_matchP && at_code1P->m_matchP != at_code2P) ) {
					++at_code2P;
					different2 = 1;
				} else if (!at_code1P->m_matchP) {
					++at_code1P;
					different1 = 1;
				
					if (at_code2P < best_end2P && !at_code2P->m_matchP) {
						++at_code2P;
					}
				} else {
					++at_code1P;
					++at_code2P;
			}	} 

			if (report_java) {

				lineno1      = next_lineno1;
				lineno2      = next_lineno2;
				next_lineno1 = ((at_code1P < best_end1P) ? (int) at_code1P->m_source_lineno : -1);
				next_lineno2 = ((at_code2P < best_end2P) ? (int) at_code2P->m_source_lineno : -1);

				if (lineno1 == next_lineno1) {
					if (!source2P->m_source_lines || lineno2 == next_lineno2) {
						continue;
					}
					fputs("<tr class=s><td></td>\n    <td><font color=\"blue\">", stdoutF);
					source2P->showline(lineno2);
					fputs("</font></td>\n</tr>\n", stdoutF);
				} else if (lineno2 == next_lineno2) {
					if (!source1P->m_source_lines || lineno1 == next_lineno1) {
						continue;
					}
					fputs("<tr class=s><td><font color=\"red\">", stdoutF);
					source1P->showline(lineno1);
					fputs("</font></td>\n    <td></td></tr>\n", stdoutF);
				} else {
					fputs("<tr class=s><td>", stdoutF);
					if (source1P->m_source_lines) {
						if (!different1) {
							source1P->showline(lineno1);
						} else {
							fputs("<font color=\"red\">", stdoutF);
							source1P->showline(lineno1);
							fputs("</font>", stdoutF);
					}	}
					fputs("</td><td>", stdoutF);
					if (source2P->m_source_lines) {
						if (!different2) {
							source2P->showline(lineno2);
						} else {
							fputs("<font color=\"blue\">", stdoutF);
							source2P->showline(lineno2);
							fputs("</font>", stdoutF);
					}	}
					fputs("</td></tr>\n", stdoutF);
				}
				different1 = 0;
				different2 = 0;
			}
		}
	}
	fputs("</table>\n", stdoutF);
	if (g_html_dirP) {
		fprintf(stdoutF, "<p><table>\n<tr><td>Start</td><td>%s</td></tr>\n", 	ctime(&start_clock));
		fprintf(stdoutF, "<tr><td>End</td><td>%s</td></tr>\n</table>\n", 	ctime(&end_clock));
		fputs("<p>End of report\n</body>\n</html>\n", stdoutF);
		fclose(stdoutF);
		stdoutF = redirectF;
	}
}

void
finalize_report(void)
{
    extern char *g_signalP;

	static	topT	*topP, *atP, *endP, *heapP, *leftP, *rightP;
	static	int		heap_index, cnt, i;
	static	FILE	*topF;

	if (sourceF) {
		fprintf(sourceF,"</table>\n");
        if (g_signalP) {
            fprintf(sourceF, "<p><font color=red>**EXECUTION INTERRUPTED BY SIG%s**</font>\n", g_signalP);
        }
		fprintf(sourceF,"<p>\n<p>End of report</body>\n</html>\n");
		fclose(sourceF);
	}
	
	if (top_clonesP) {
		cnt = top_clones_cnt;
		if (cnt > g_top_clones) {
			cnt = g_top_clones;
		}
		if (cnt) {
			topP = atP = (topT *) Xmalloc(cnt *sizeof(topT));
			if (!topP) {
				outofmemory();
			}
			for (endP = atP + cnt; ; ) {
				if (top_clonesP->html) {
					*atP++     = *top_clonesP;
					if (endP <= atP) {
						break;
				}	}
				heapP      = top_clonesP;
				heap_index = 1;
				for (;;) {
					leftP = heapP + heap_index;
					if (top_clones_endP <= leftP) {
						heapP->weight = 0xFFFFFFFF;
						break;
					}
					rightP = leftP + 1;
					if (top_clones_endP <= rightP || leftP->weight <= rightP->weight) {
						*heapP       = *leftP;
						heapP        = leftP;
						heap_index <<= 1;
						continue;
					}
					*heapP     = *rightP;
					heapP      = rightP;
					heap_index = (heap_index << 1) + 1;
			}	}
			sprintf(g_filename,"%s/top.html", g_html_dirP);
			topF = open_file(g_filename);
			fprintf(topF, HTML_HEADER "<title>Top %d largest clones</title>\n<body>\n<h3>Top %d largest clones</h3>\n<p>\n<table><tr>\n", cnt, cnt);
			for (i = 0; i < g_top_cols; ++i) {
				fprintf(topF, "<th><font color=red>Weight</font></th><th>Clone</th>");
			}
			for (i = g_top_cols; --atP >= topP; ++i) {
				if (i == g_top_cols) {
					fprintf(topF, "</tr>\n<tr>");
					i = 0;
				}
				fprintf(topF, "<td align=right>%u</td><td><a href=%u.html>%u.html</td>", atP->weight, atP->html, atP->html);
			}
			fprintf(topF, "</tr>\n</table>\n<P>\nEnd of report\n</body>\n</html>\n");
			fclose(topF);
	}	}
}

static int
permitted_clone(void)
{
	if ( (g_lth1 < g_min_match_sequence) || (g_lth2 < g_min_match_sequence)) {
		// Generally the clone is too short on one side or the other to be deemed worth reporting
		// Consider a clone long enough if it includes all of a function
		if ((start_code1P != function_code1P || best_end1P != function_end1P || g_lth1 < g_min_function_sequence) &&
			(start_code2P != function_code2P || best_end2P != function_end2P || g_lth2 < g_min_function_sequence)
		   ) {
			return(0);
	}	}
	return(1);
}

static int
same_branch(codeT *code1P, codeT *code2P)
{
	codeT	*to1P, *to2P, *matchP;

	to1P = code1P->m_jump.P;
	to2P = code2P->m_jump.P;
	if (to1P < code1P) {
		if (to2P >= code2P) {
			goto mismatch;
		}
		if (to1P < start_code1P) {
			goto mismatch;
		}
		if (to2P < start_code2P) {
			goto mismatch;
		}
		if ((matchP = to1P->m_matchP)) {
			if (to2P > matchP) {
				goto mismatch;
			}
			if (to2P->m_matchP) {
				if (to2P == matchP) {
					goto same;
				}
				goto mismatch;
			}
		} else if ((matchP = to2P->m_matchP)) {
			if (to1P > matchP) {
				goto mismatch;
			}
			if (to2P == start_code2P) {
				goto mismatch;
		}	}

		while (!((--to1P)->m_matchP));
		assert(start_code1P <= to1P);
		while (!((--to2P)->m_matchP));
		assert(start_code2P <= to2P);

		if (to1P->m_matchP != to2P) {
			goto mismatch;
		}
	} else if (to1P > code1P) {
		if (to2P <= code2P) {
			goto mismatch;
		}
		if (end_code1P <= to1P) {
			goto mismatch;
		}
		if (end_code2P <= to2P) {
			goto mismatch;
		}
	} else if (to2P != code2P) {
		goto mismatch;
	}
same:
	return(1);
mismatch:
	return(0);
}

static int
same_code(codeT *code1P, codeT *code2P, int *weightP) 
{
	int	pcode1;

	pcode1 = code1P->m_pcode;
	if (pcode1 != code2P->m_pcode) {
		goto mismatch;
	}

	switch (pcode1) {
	case ILOAD:
	case ILOAD_0:
	case ILOAD_1:
	case ILOAD_2:
	case ILOAD_3:
	case LLOAD:
	case LLOAD_0:
	case LLOAD_1:
	case LLOAD_2:
	case LLOAD_3:
	case FLOAD:
	case FLOAD_0:
	case FLOAD_1:
	case FLOAD_2:
	case FLOAD_3:
	case DLOAD:
	case DLOAD_0:
	case DLOAD_1:
	case DLOAD_2:
	case DLOAD_3:
	case ALOAD:
	case ALOAD_0:
	case ALOAD_1:
	case ALOAD_2:
	case ALOAD_3:

	case ISTORE:
	case ISTORE_0:
	case ISTORE_1:
	case ISTORE_2:
	case ISTORE_3:
	case LSTORE:
	case LSTORE_0:
	case LSTORE_1:
	case LSTORE_2:
	case LSTORE_3:
	case FSTORE:
	case FSTORE_0:
	case FSTORE_1:
	case FSTORE_2:
	case FSTORE_3:
	case DSTORE:
	case DSTORE_0:
	case DSTORE_1:
	case DSTORE_2:
	case DSTORE_3:
	case ASTORE:
	case ASTORE_0:
	case ASTORE_1:
	case ASTORE_2:
	case ASTORE_3:
	{
		varT	*var1P, *var2P;
		
		var1P = (varT *) code1P->m_tableP;
		var2P = (varT *) code2P->m_tableP;

		if (var1P->m_nameP || var2P->m_nameP) {
			if (var1P->m_nameP == var2P->m_nameP) {
				break;
			}
			goto mismatch;
		}
		if (var1P->m_number != var2P->m_number) {
			goto mismatch;
		}
		break;
	}
	case IINC:
	{
		iincT	*var1P, *var2P;
		
		var1P = (iincT *) code1P->m_tableP;
		var2P = (iincT *) code2P->m_tableP;

		if (var1P->m_s4 != var2P->m_s4) {
			goto mismatch;
		}

		if (var1P->m_nameP && var2P->m_nameP) {
			if (var1P->m_nameP == var2P->m_nameP) {
				break;
			}
			goto mismatch;
		}
		if (var1P->m_number != var2P->m_number) {
			goto mismatch;
		}
		break;
	}
	case GETSTATIC:	
	case GETFIELD:
	case PUTSTATIC:
	case PUTFIELD:
	{
		fieldrefT	*fieldref1P, *fieldref2P;

		fieldref1P = (fieldrefT *) code1P->m_tableP;
		fieldref2P = (fieldrefT *) code2P->m_tableP;

		if (fieldref1P->m_classnameP != fieldref2P->m_classnameP ||
			fieldref1P->m_nameP      != fieldref2P->m_nameP) {
			goto mismatch;
		}
		break;
	}
	case INVOKEVIRTUAL:
	case INVOKESPECIAL:
	case INVOKESTATIC:
	{
		methodrefT	*methodref1P, *methodref2P;

		methodref1P = (methodrefT *) code1P->m_tableP;
		methodref2P = (methodrefT *) code2P->m_tableP;

		if (methodref1P->m_classnameP   != methodref2P->m_classnameP ||
			methodref1P->m_nameP        != methodref2P->m_nameP      ||
			methodref1P->m_descriptorP  != methodref2P->m_descriptorP) {
			goto mismatch;
		}
		break;
	}
	case INVOKEINTERFACE:
	{
		interfacerefT	*methodref1P, *methodref2P;

		methodref1P = (interfacerefT *) code1P->m_tableP;
		methodref2P = (interfacerefT *) code2P->m_tableP;

		if (methodref1P->m_classnameP   != methodref2P->m_classnameP ||
			methodref1P->m_nameP        != methodref2P->m_nameP      ||
			methodref1P->m_descriptorP  != methodref2P->m_descriptorP ||
			methodref1P->m_args         != methodref2P->m_args) {
			goto mismatch;
		}
		break;
	}
	case MULTIANEWARRAY:
	{
		arrayrefT	*arrayref1P, *arrayref2P;

		arrayref1P = (arrayrefT *) code1P->m_tableP;
		arrayref2P = (arrayrefT *) code2P->m_tableP;

		if (arrayref1P->m_classnameP != arrayref2P->m_classnameP ||
			arrayref1P->m_dimensions != arrayref2P->m_dimensions) {
			goto mismatch;
		}
		break;
	}
	case ANEWARRAY:				// anewarray
	case NEW:					// new
	case CHECKCAST:				// checkcast
	case INSTANCEOF:			// instanceof
	{
		classrefT	*classref1P, *classref2P;
		
		classref1P = (classrefT *) code1P->m_tableP;
		classref2P = (classrefT *) code2P->m_tableP;

		if (classref1P->m_classnameP != classref2P->m_classnameP) {
			goto mismatch;
		}
		break;
	}
	case LDC:
	case LDC_W:
	case LDC2_W:
	{
		cp_infoT	*cp1P, *cp2P;

		cp1P = (cp_infoT *) code1P->m_tableP;
		cp2P = (cp_infoT *) code2P->m_tableP;

		if (cp1P->tag != cp2P->tag) {
			goto mismatch;
		}
		switch (cp1P->tag) {
		case CONSTANT_Integer:
			if (cp1P->u.ival != cp2P->u.ival) {
				goto mismatch;
			}
			break;
		case CONSTANT_Float:
			if (cp1P->u.fval != cp2P->u.fval) {
				goto mismatch;
			}
			break;
		case CONSTANT_Long:
			if (cp1P->u.lval != cp2P->u.lval) {
				goto mismatch;
			}
			break;
		case CONSTANT_Double:
			if (cp1P->u.dval != cp2P->u.dval) {
				goto mismatch;
			}
			break;
		case CONSTANT_String:
		case CONSTANT_Class:
			// N.B. Even though the spec says that there
			//      is an indirection to the string via
			//		cpP->u.ref.index1 we've already removed
			//		it by the time we get here.
			if (cp1P->u.stringP != cp2P->u.stringP) {
				goto mismatch;
			}
			break;
		default:
			assert(0);
		}
		break;
	}
	case IFEQ:
	case IFNE:
	case IFLT:
	case IFGE:
	case IFGT:
	case IFLE:
	case IF_ICMPEQ:
	case IF_ICMPNE:
	case IF_ICMPLT:
	case IF_ICMPGE:
	case IF_ICMPGT:
	case IF_ICMPLE:
	case IF_ACMPEQ:
	case IF_ACMPNE:
	case IFNULL:
	case IFNOTNULL:
	case GOTO:
	case GOTO_W:
	case JSR:
	case JSR_W:
	case TABLESWITCH:
	case LOOKUPSWITCH:
	case DEFAULT:
		if (!same_branch(code1P, code2P)) {
			goto mismatch;
		}
		break;
	case TRY_START:
		if (!same_branch(code1P, code2P)) {
			goto mismatch;
		}
	case TRY_END:
	case TRY_LABEL:
	{
		exceptionT	*exception1P = (exceptionT *) code1P->m_tableP;
		exceptionT	*exception2P = (exceptionT *) code2P->m_tableP;

		if (exception1P->m_catch_typeP != exception2P->m_catch_typeP) {
			goto mismatch;
		}
		break;
	}
	default:
		if (code1P->m_u4 != code2P->m_u4) {
			goto mismatch;
	}	}

	*weightP = g_same_cost;
	return(1);
mismatch:
	*weightP = g_mismatch_cost;
	return(0);
}

// Hill Climbing Logic

typedef struct dllS {
	struct dllS		*prevP;
	struct dllS 	*nextP;
} dllT;

typedef unsigned short crossT;

typedef	struct edgeS {
	dllT			dll;			// Must be first element in struct
	codeT			*code1P;		// Left  side of match
	codeT			*code2P;		// Right side of match
	int				matched;		// This is a matching edge
	int				blocks;			// Number of edges that block this edge
	struct edgeS	*next_changeP;
	int				blocks1;

	int				crosses;
	crossT			*crossesP;
} edgeT;

typedef struct {
	edgeT			*edgeP;
	int				matched;
} reverseT;

typedef struct {
	edgeT			*edgeP;
	int				blocks;
} unblockedT;

static int
hill_climb(void)
{
	// Array of edges
	static	edgeT	*edgesP         = 0;
	static	edgeT	*edges_maxP     = 0;
	static	unsigned int		max_edges       = 0;
	static	edgeT	*end_edgesP     = 0;

	// Array of crossings of edges with edges
	static	crossT	*crossesP       = 0;
	static	unsigned int	max_crosses     = 0;

#define Backup(X) \
	if (X->next_changeP == (edgeT *) -1) { \
		X->blocks1      = X->blocks; \
		X->next_changeP = changesP; \
		changesP        = X; \
	}

#define Increment(X) \
	Backup(X); \
	++(X->blocks1);

#define Decrement(X) \
	Backup(X); \
	--(X->blocks1);

	// Array of edges whose matched/unmatched status is to be reversed
	static	reverseT		*reversesP     = 0;
	static	reverseT		*reverses_maxP = 0;
	static	unsigned int	max_reverses   = 0;
	static	reverseT		*end_reversesP = 0;

#define Reverse(X) \
	if (reverses_maxP <= end_reversesP) { \
		at = max_reverses; \
		if (!max_reverses) { \
			max_reverses   = 4096; \
		} else { \
			max_reverses <<= 1; \
		} \
		reversesP     = (reverseT *) Xrealloc(reversesP, max_reverses * sizeof(reverseT)); \
		if (!reversesP) { \
            fprintf(stderr, "Hillclimb: Reversals outofmemory\n"); \
			reverses_maxP = 0; \
            max_reverses  = 0; \
            goto abort_reversals; \
        } \
		reverses_maxP = reversesP + max_reverses; \
		end_reversesP = reversesP + at; \
	} \
	end_reversesP->edgeP   = X; \
	end_reversesP->matched = X->matched; \
	++end_reversesP; 

	static	unblockedT	*unblockedP = 0;
	static	unblockedT	*unblocked_maxP = 0;
	static	unsigned int	 max_unblocked  = 0;
	static	unblockedT	*end_unblockedP = 0;

	// Array of dll's for each block which we wish to handle
	static	dllT 	*dllsP     = 0;
	static	dllT	*end_dllsP = 0;

	// Linked list of edges whose blocks value changes
	static	edgeT	*changesP  = 0;

	static	int		iterations;

	static codeT	*code1aP, *code2aP, *code1bP, *code2bP;
	static crossT	*cross1P, *cross2P, *cross3P;
	static crossT	*end_cross1P, *end_cross2P, *end_cross3P;
	static edgeT	*edge1P, *next_edge1P;
	static edgeT	*edge2P, *edge3P, *edge4P;
	static unblockedT *unblocked1P, *unblocked_bestP;

	static int		at, phase, max_crossings, change, old_ret, ret;
	static int		weight;
	static int		matched;
	static reverseT *reverseP;
	static int		blocks;
	static dllT		*headP, *cycleP;
	static unsigned int crosses;

	static time_t	hillclimb_started;
	static time_t	hillclimb_terminate;

#ifdef TEST_HILLCLIMB
	static int		original_matches, final_matches;
#endif

	hillclimb_started   = time(0);
	if (g_hillclimb_timeout) {
		hillclimb_terminate = hillclimb_started + g_hillclimb_timeout;
	} else {
		hillclimb_terminate = 0;
    }
	g_hillclimb_elapse -= hillclimb_started;

#ifdef DEBUG_HILLCLIMB
	fprintf(stderr, "Hill Climb %p %d %p %d\n", 
		start_code1P, 
		best_end1P - start_code1P,
		start_code2P, 
		best_end2P - start_code2P);
#endif
#ifdef TEST_HILLCLIMB
	original_matches = 0;
	for (code1aP = start_code1P; code1aP <= best_end1P; ++code1aP) {
		if (code1aP->m_matchP) {
			++original_matches;
	}	}
#endif

	if (!dllsP) {
		dllsP     = (dllT *) Xmalloc(g_hillclimb * sizeof(dllT));
		if (!dllsP) {
			fprintf(stderr, "Hillclimb: dlls outofmemory\n");
			return(0);
		}
		end_dllsP = dllsP + g_hillclimb;
	}

	// Find all possible edges

	end_edgesP       = edgesP;

	// Compute all the edges
	// These may change for jumps as matchings change

	// Can't change first line
	change    = 0;
	for (code1aP = start_code1P + 1; code1aP <= best_end1P; ++code1aP) {
		for (code2aP = start_code2P + 1; code2aP <= best_end2P; ++code2aP) {
			// Add the edge if appropriate
			if (code1aP->m_matchP == code2aP) {
				matched = 1;
			} else {
				if (!same_code(code1aP, code2aP, &weight)) {
					// Not appropriate
					continue;
				}
				matched = 0;
				change  = 1;
			}
			if (edges_maxP <= end_edgesP) {
				at = max_edges;
				if (!max_edges) {
					max_edges   = (1<<11);
				} else {
					max_edges <<= 1;
				}
				edgesP     = (edgeT *) Xrealloc(edgesP, max_edges * sizeof(edgeT));
				if (!edgesP) {
					max_edges = 0;
					goto too_many_edges;
				}
				edges_maxP = edgesP + max_edges;
				end_edgesP = edgesP + at;
			}
			memset(end_edgesP, 0, sizeof(edgeT));
			end_edgesP->code1P       = code1aP;
			end_edgesP->code2P       = code2aP;
			end_edgesP->matched      = matched;
			end_edgesP->next_changeP = (edgeT *) -1;
			++end_edgesP;
	}	}

	if (!change) {
		// No unmatched edges
		ret = 0;
		goto done;
	}

	if ((end_edgesP - edgesP) > 0xFFFF) {
too_many_edges:
		// Can't handle the edge offsets in a short
		fprintf(stderr,
			"* Hill-climbing aborted: too many edges\n"
			"* %S:%S .v.\n* %S:%S\n",
			(wchar_t *) source1P->m_nameP,
			(wchar_t *) function1P->m_nameP,
			(wchar_t *) source2P->m_nameP,
			(wchar_t *) function2P->m_nameP);
		fprintf(stderr, "* Clone lths={%ld,%ld}, edges=%ld\n",
			(long) (best_end1P - start_code1P),
			(long) (best_end2P - start_code2P),
			(long) (end_edgesP - edgesP));
		ret = 0;
		goto done;
	}

	// Add edge crossings

	crosses = 0;
	for (phase = 0; ; ++phase) {
		for (edge1P = edgesP; edge1P < end_edgesP; ++edge1P) {
			code1aP = edge1P->code1P;
			code2aP = edge1P->code2P;
			// Treat an edge point match as a cross

			for (edge2P = edge1P; ++edge2P < end_edgesP; ) {
				code1bP = edge2P->code1P;
				code2bP = edge2P->code2P;
				if ((code1aP >= code1bP && code2aP <= code2bP) ||
					(code1aP <= code1bP && code2aP >= code2bP)) {
					if (!phase) {
						++(edge1P->crosses);
						++(edge2P->crosses);
						crosses += 2;
					} else {
						--(edge1P->crossesP);
						*edge1P->crossesP = (edge2P - edgesP);
						--(edge2P->crossesP);
						*edge2P->crossesP = (edge1P - edgesP);
						if (edge1P->matched) {
							assert(!edge2P->matched);
							++(edge2P->blocks);
						} else if (edge2P->matched) {
							++(edge1P->blocks);
		}	}	}	}	}
		if (phase) {
			break;
		}
		if (!crosses) {
			ret = 0;
			goto done;
		}
		if (crosses > 0x40000000) {
too_many_crosses:
			fprintf(stderr,
				"* Hill-climbing aborted: too many crosses\n"
				"* %S:%S .v.\n* %S:%S\n",
				(wchar_t *) source1P->m_nameP,
				(wchar_t *) function1P->m_nameP,
				(wchar_t *) source2P->m_nameP,
				(wchar_t *) function2P->m_nameP);
			fprintf(stderr, "* Clone lths={%ld,%ld}, edges=%ld crosses=%u\n",
				(long) (best_end1P - start_code1P),
				(long) (best_end2P - start_code2P),
				(long) (end_edgesP - edgesP),
				crosses);
			ret = 0;
			goto done;
		}

		if (max_crosses < crosses) {
			if (!max_crosses) {
				max_crosses = (1<<18);	// Must be multiple of 2
			} else {
				Xfree(crossesP);
			}
			while (max_crosses < crosses) {
				max_crosses <<= 1;
			}
			crossesP = (crossT *) Xmalloc(max_crosses * sizeof(unsigned short));
			if (!crossesP) {
				max_crosses = 0;
				goto too_many_crosses;
			}
		}
		end_cross1P = crossesP + crosses;
		for (edge1P = end_edgesP; edgesP <= --edge1P; ) {
			edge1P->crossesP = end_cross1P;
			end_cross1P     -= edge1P->crosses;
		}
		assert(end_cross1P == crossesP);
	}

	for (headP = dllsP; headP < end_dllsP; ++headP) {
		headP->nextP = headP;
		headP->prevP = headP;
	}

	for (edge2P = edgesP; edge2P < end_edgesP; ++edge2P) {
#ifdef TEST_HILLCLIMB
		assert(edge2P->blocks || edge2P->matched);
#endif
		blocks = edge2P->blocks;
		if (blocks > g_max_blocks_seen) {
			g_max_blocks_seen = blocks;
		}
		if (blocks && blocks <= g_hillclimb) {
			// Add to the start of the cycle
			--blocks;
			headP                    = dllsP + blocks;
			edge2P->dll.prevP        = headP;
			edge2P->dll.nextP        = headP->nextP;
			headP->nextP             = (dllT *) edge2P;
			edge2P->dll.nextP->prevP = (dllT *) edge2P;
		} else {
			edge2P->dll.nextP        = 0;
			edge2P->dll.prevP        = 0;
	}	}

#ifdef DEBUG_HILLCLIMB
	fprintf(stderr, "%d edges %d edge crossings %d code1=%p %d code2=%p\n", 
				end_edgesP - edgesP,
				crosses,
				best_end1P - start_code1P,
				start_code1P,
				best_end2P - start_code2P,
				start_code2P);

	for (edge1P = edgesP; edge1P < end_edgesP; ++edge1P) {
		fprintf(stderr, "%d] %d %d<->%d %s<=>%s ", 
					edge1P - edgesP,
					edge1P->blocks,
					edge1P->code1P - start_code1P,
					edge1P->code2P - start_code2P,
					edge1P->code1P->m_codeP,
					edge1P->code2P->m_codeP);
		if (edge1P->matched) {
			fprintf(stderr, " match");
		}
		cross1P = edge1P->crossesP;
		for (end_cross1P = cross1P + edge1P->crosses; cross1P < end_cross1P; ++cross1P) {
			fprintf(stderr, " %hu", *cross1P);
		}
		fprintf(stderr, "\n");
	}
#endif

	old_ret = ret  = 0;
	iterations     = 0;
	for (max_crossings = 0; ; ++max_crossings) {
		if (max_crossings >= g_hillclimb) {
			if (ret == old_ret || iterations >= g_iterations) {
				if (iterations > g_max_iterations_seen) {
					g_max_iterations_seen = iterations;
				}
				break;
			}
			++iterations;
			old_ret        = ret;
			max_crossings  = 0;
			// All but first iteration consider all edges
		}

		cycleP = dllsP + max_crossings;
		for (edge1P = (edgeT *) cycleP->nextP; cycleP != (dllT *) edge1P; edge1P = next_edge1P) {
			next_edge1P = (edgeT *) edge1P->dll.nextP;
			assert(!edge1P->matched);
			assert(edge1P->blocks == max_crossings + 1);

#ifdef TEST_HILLCLIMB
			{
				int 	cnt;
				edgeT	*edge5P;
		
				for (edge4P = edgesP; edge4P < end_edgesP; ++edge4P) {
					code1aP = edge4P->code1P;
					code2aP = edge4P->code2P;
					if (edge4P->matched) {
						assert(code1aP->m_matchP == code2aP);
						assert(code2aP->m_matchP == code1aP);
						assert(!edge4P->blocks);
						continue;
					}
					assert(code1aP->m_matchP != code2aP);
					assert(code2aP->m_matchP != code1aP);
				
					cnt = 0;
					cross1P = edge4P->crossesP;
					for (end_cross1P = cross1P + edge4P->crosses; cross1P < end_cross1P; ++cross1P) {
						edge5P = edgesP + *cross1P;
						if (edge5P->matched) {
							++cnt;
					}	}
					assert(cnt == edge4P->blocks);
			}	}

			for (headP = dllsP; headP < end_dllsP; ++headP) {
				for (edge4P = (edgeT *) headP->nextP; headP != (dllT *) edge4P; edge4P = (edgeT *) edge4P->dll.nextP) {
					assert(edge4P->blocks == (headP - dllsP) + 1);
			}	}
#endif

			// Match1 is currently not an active edge
			// Compute the change in edges if all of the active edges crossing edge1P are made inactive
			// and edge1P is made active

#ifdef DEBUG_HILLCLIMB
			fprintf(stderr, "Considering activating %d<->%d\n",
								edge1P->code1P - start_code1P,
								edge1P->code2P - start_code2P);

			fprintf(stderr, "Initial add to reverse %p\n", edge1P);
#endif

			/* Edge 1 will become active if we commit this change which is
			 * an improvement of 1
			 */
			end_reversesP  = reversesP;
			change         = 1;
			/* Linked list of changed block numbers initially empty */
			changesP       = 0;
			end_unblockedP = unblockedP;

			cross1P = edge1P->crossesP;
			for (end_cross1P = cross1P + edge1P->crosses; cross1P < end_cross1P; ++cross1P) {
				/* Edge1 crosses edge2 */
				edge2P = edgesP + *cross1P;

				// Edge 1 will block this edge
				Increment(edge2P);

				if (!edge2P->matched) {
					continue;
				}
				// We have incremented all counts
				assert(edge2P->blocks1 == 1);
				// We must unactivate edge2P to permit activation of edge1
				Reverse(edge2P);
				// This is a match we loose
				--change;	
				cross2P = edge2P->crossesP;
				for (end_cross2P = cross2P + edge2P->crosses; cross2P < end_cross2P; ++cross2P) {
					// Because cross2 is inactive all edge3 crossing it are
					// no longer blocked by it
					edge3P = edgesP + *cross2P;
					assert(!edge3P->matched);
					Decrement(edge3P);
					if (edge3P->blocks1) {
						continue;
					}
					if (unblocked_maxP <= end_unblockedP) {
						at = max_unblocked;
						if (!max_unblocked) {
							max_unblocked   = 256;
						} else {
							max_unblocked <<= 1;
						}
						unblockedP = (unblockedT *) Xrealloc(unblockedP, max_unblocked * sizeof(unblockedT));
                        if (!unblockedP) {
                            fprintf(stderr, "Hillclimb: unblocked outofmemory\n");
                            max_unblocked = 0;
                            goto abort_reversals;
                        }

						unblocked_maxP = unblockedP + max_unblocked;
						end_unblockedP = unblockedP + at;
					}
					end_unblockedP->edgeP  = edge3P;
					++end_unblockedP;
			}	}

			/* We have now inactivated all edges crossing edge1 */
			assert(!(edge1P->blocks1));
			Reverse(edge1P);
			// Avoid allowing edge1P to match twice
			Increment(edge1P);

			for (;;) {
				for (unblocked1P = unblockedP; unblocked1P < end_unblockedP; ) {
					edge2P = unblocked1P->edgeP;
					if (edge2P->blocks1) {
						// No longer unblocked
						--end_unblockedP;
						*unblocked1P = *end_unblockedP;
						continue;
					} 
					edge2P->matched |= 2;
					++unblocked1P;
				}
				if (unblockedP == end_unblockedP) {
					break;
				}
				for (unblocked1P = unblockedP; unblocked1P < end_unblockedP; ++unblocked1P) {
					blocks  = 0;
					edge2P  = unblocked1P->edgeP;
					cross2P = edge2P->crossesP;
					for (end_cross2P = cross2P + edge2P->crosses; cross2P < end_cross2P; ++cross2P) {
						edge3P = edgesP + *cross2P;
						if (edge3P->matched & 2) {
							++blocks;
					}	}
					unblocked1P->blocks = blocks;
				}
				unblocked_bestP = unblocked1P = unblockedP;
				for (; unblocked1P < end_unblockedP; ++unblocked1P) {
					if (unblocked1P->blocks < unblocked_bestP->blocks) {
						unblocked_bestP = unblocked1P;
					}
					edge2P           = unblocked1P->edgeP;
					edge2P->matched &= ~2;
				}

				edge3P = unblocked_bestP->edgeP;
				--end_unblockedP;
				*unblocked_bestP = *end_unblockedP;

				assert(!edge3P->matched);
				assert(!edge3P->blocks1);

				// We will activate edge3P
				++change;	
				Reverse(edge3P);
				Increment(edge3P);	// Avoid choosing again

				cross3P = edge3P->crossesP;
				for (end_cross3P = cross3P + edge3P->crosses; cross3P < end_cross3P; ++cross3P) {
					// Edge3 activation blocks all things it crosses
					edge4P = edgesP + *cross3P;
					Increment(edge4P);
				}
			}
			if (change <= 0) {
				// This change does no good
				// Put all next_changeP pointers back to -1
abort_reversals:	
				for (edge2P = changesP; edge2P; edge2P = edge3P) {
					edge3P               = edge2P->next_changeP;
					edge2P->next_changeP = (edgeT *) -1;
				}
			} else {

#ifdef DEBUG_HILLCLIMB
				fprintf(stderr, "Found a change of %d in edge %d-%d\n",
							change,
							edge1P->code1P - start_code1P,
							edge1P->code2P - start_code2P);
#endif

				ret += change;

				// Apply all the reversals

				for (reverseP = reversesP; reverseP < end_reversesP; ++reverseP) {
					edge2P  = reverseP->edgeP;
					code1bP = edge2P->code1P;
					code2bP = edge2P->code2P;
#ifdef TEST_HILLCLIMB
					if (edge2P->matched != reverseP->matched) {
						reverseT	*reverse1P;

						for (reverse1P = reversesP; reverse1P <= reverseP; ++reverse1P) {
							edge3P = reverse1P->edgeP;
							fprintf(stderr, "[%d] %d<->%d ",
								edge3P - edgesP,
								edge3P->code1P - start_code1P,
								edge3P->code2P - start_code2P);
							if (reverse1P->matched) {
								fprintf(stderr, " unmatch\n");
							} else {
								fprintf(stderr, " match\n");
						}	}
						assert(edge2P->matched == reverseP->matched);
					}
#endif
					if (edge2P->matched) {
#ifdef DEBUG_HILLCLIMB
						fprintf(stderr, "Unmatch %d<->%d\n",
							code1bP - start_code1P,
							code2bP - start_code2P);
#endif
						// Unmatch this edge
						assert(code1bP->m_matchP == code2bP);
						assert(code2bP->m_matchP == code1bP);
						code1bP->m_matchP = 0;
						code2bP->m_matchP = 0;
						edge2P->matched   = 0;
						if (code1bP->m_jump.P > code1bP) {
							--forward_jumps;
						}
					} else {
#ifdef DEBUG_HILLCLIMB
						fprintf(stderr, "Match %d<->%d\n",
							code1bP - start_code1P,
							code2bP - start_code2P);
#endif
						// Match this edge
						assert(code1bP->m_matchP == 0);
						assert(code2bP->m_matchP == 0);
						code1bP->m_matchP = code2bP;
						code2bP->m_matchP = code1bP;
						edge2P->matched   = 1;
						if (code1bP->m_jump.P > code1bP) {
							++forward_jumps;
						}
						--(edge2P->blocks1); // Undo the blocking ++
					}
				}
				// Having applied all the reversals correct
				// The blocks crossed in all affected edges
				for (edge2P = changesP; edge2P; edge2P = edge3P) {
					edge3P         = edge2P->next_changeP;
					edge2P->blocks = blocks = edge2P->blocks1;
					if (blocks > g_max_blocks_seen) {
						g_max_blocks_seen = blocks;
					}
					edge2P->next_changeP = (edgeT *) -1;
					if ((headP = edge2P->dll.nextP)) {
						// Remove edge2 from dll's
						if (edge2P == next_edge1P) {
							next_edge1P = (edgeT *) headP;
						}
						headP->prevP             = edge2P->dll.prevP;
						edge2P->dll.prevP->nextP = headP;
					}
					if (blocks && blocks <= g_hillclimb) {
						// Add to the start of the right dll
						// We won't see it this iteration if adding it to
						// our cycle because before us in the cycle
						--blocks;
						headP                    = dllsP + blocks;
						edge2P->dll.prevP        = headP;
						edge2P->dll.nextP        = headP->nextP;
						headP->nextP             = (dllT *) edge2P;
						edge2P->dll.nextP->prevP = (dllT *) edge2P;
					} else {
						edge2P->dll.nextP        = 0;
						edge2P->dll.prevP        = 0;
					}
				}
			}


#ifdef TEST_HILLCLIMB
			for (headP = dllsP; headP < end_dllsP; ++headP) {
				for (edge4P = (edgeT *) headP->nextP; headP != (dllT *) edge4P; edge4P = (edgeT *) edge4P->dll.nextP) {
					assert(edge4P->blocks == (headP - dllsP) + 1);
			}	}
#endif
            if (hillclimb_terminate && hillclimb_terminate < time(0)) {
                fprintf(stderr,
                    "* Hill-climbing aborted: timeout %d secs\n"
                    "* %S:%S .v.\n* %S:%S\n",
                    g_hillclimb_timeout,
                    (wchar_t *) source1P->m_nameP,
                    (wchar_t *) function1P->m_nameP,
                    (wchar_t *) source2P->m_nameP,
                    (wchar_t *) function2P->m_nameP);
                fprintf(stderr, "* Clone lths={%ld,%ld}, edges=%ld crosses=%u\n",
                    (long) (best_end1P - start_code1P),
                    (long) (best_end2P - start_code2P),
                    (long) (end_edgesP - edgesP),
                    crosses);

                goto terminate;
            }
	}	}

terminate:
	for (; !(best_end2P = best_end1P->m_matchP); --best_end1P);
#ifdef TEST_HILLCLIMB
	final_matches = 0;
	for (code1aP = start_code1P; code1aP <= best_end1P; ++code1aP) {
		if (code1aP->m_matchP) {
			++final_matches;
	}	}
	assert(final_matches - original_matches == ret);
#endif
done:
	g_hillclimb_elapse += time(0);
	return(ret);
}

typedef struct matchesS {
	struct matchesS *m_nextP;
	struct codeS	*m_codeP;
} matchesT;

typedef struct block_matchesS {
	struct block_matchesS	*m_nextP;
	matchesT				m_cache[8192];
} block_matchesT;

extern int				g_max_function_lth;

static int				g_matches_lth          = 0;
static matchesT			**g_matchesPP          = 0;
static matchesT			***g_tail_matchesPPP   = 0;
static block_matchesT	*g_head_block_matchesP = 0;		// Head of the linked list of blocks we can get matches from
static block_matchesT	*g_block_matchesP      = 0;		// The current block we are getting matches from
static matchesT	        *g_matches_cacheP      = 0;		// Next available match in the current block
static matchesT	        *g_matches_endP        = 0;		// End of available matches in the current block

static codeT *
detect_clone(void)
{
	static	codeT	*code1P, *code2P, *at_code1P, *at_code2P, *last_match1P;
	static	codeT	*max_end1P, *max_end2P;
	static	int		max_best, ret, weight, improved;
	static	int		original_matches, extra_matches, cancelled_matches;
	static  int		distance;
	static	block_matchesT	*next_blockP;
	static	matchesT	**tailPP, ***tailPPP;

#ifdef DEBUG_DETECT
	static int debug_cnt = 0;

	fprintf(stderr, "%d Comparing %s(%d-%d) and %s(%d-%d)\n  %s\n  %s\n",  
		++debug_cnt,
		function1P->m_nameP,
		start_code1P - function_code1P,
		end_code1P - function_code1P,
		function2P->m_nameP,
		start_code2P - function_code2P, 
		end_code2P - function_code2P, 
		start_code1P->m_codeP, 
		start_code2P->m_codeP);
#endif
	start_code1P->m_matchP = start_code2P;
	start_code2P->m_matchP = start_code1P;
	last_match1P           = start_code1P;

	if (!same_code(start_code1P, start_code2P, &weight)) {
		goto fail;
	}
	max_best            = weight;
	max_end1P           = start_code1P;
	max_end2P           = start_code2P;

	start_clock = time(0);

	// Initially have to have a match
	best_end1P = code1P = start_code1P;
	best_end2P = code2P = start_code2P;
	original_matches    = 1;
	extra_matches       = 0;

extend_clone:
	improved = 0;
	for (; ++code1P < end_code1P && ++code2P < end_code2P;) {
		for (distance = 0; weight >= 0; weight += g_mismatch_cost) {
			at_code1P = code1P;
			at_code2P = code2P + distance;
			if (at_code2P >= end_code2P) {
				at_code1P += (at_code2P - end_code2P + 1);
				if (at_code1P >= end_code1P) {
					break;
				}
				at_code2P  = end_code2P - 1;
			} 
			while (at_code2P >= code2P && at_code1P < end_code1P) {
				if (same_code(at_code1P, at_code2P, &ret)) {
					goto found;
				}
				++at_code1P;
				--at_code2P;
			}
			++distance;
		}
		break;
found:	
		weight += ret;
		assert(weight >= 0);
		at_code1P->m_matchP = at_code2P;
		at_code2P->m_matchP = at_code1P;
		if (last_match1P < at_code1P) {
			last_match1P    = at_code1P;
		}
		improved        = 1;
		++original_matches;

		best_end1P  = at_code1P;
		best_end2P  = at_code2P;

		if (g_best_weight && (max_best <= weight)) {
			max_best  = weight;
			max_end1P = best_end1P;
			max_end2P = best_end2P;
		}
		code1P      = at_code1P;
		code2P      = at_code2P;
	}

	if (improved) {
		if (g_hillclimb > 0) {
			ret = hill_climb();
			if (ret > 0) {
				extra_matches += ret;
				weight += (ret * g_same_cost);
				code1P = best_end1P; 
				code2P = best_end2P;
				goto extend_clone;
	}	}	}
	
	if (g_best_weight) {
		weight     = max_best;
		best_end1P = max_end1P;
		best_end2P = max_end2P;
	}

	++best_end1P;
	++best_end2P;

	g_lth1     = best_end1P - start_code1P;
	g_lth2     = best_end2P - start_code2P;

	if (!permitted_clone()) {
		goto fail;
	}
	
	cancelled_matches = 0;
	for (code1P = start_code1P; code1P <= last_match1P; ++code1P) {
		at_code1P = code1P->m_jump.P;
		if (at_code1P <= code1P) {
			continue;
		}
		if (!(code2P = code1P->m_matchP)) {
			continue;
		}
		at_code2P = code2P->m_jump.P;
		if (at_code1P >= best_end1P) {
			if (at_code2P >= best_end2P) {
				// Both branch beyond end of clone
				// Might be turned into a return
				continue;
			}
		} else if (at_code2P < best_end2P) {
			if (at_code1P->m_matchP) {
				if (at_code2P == at_code1P->m_matchP) {
					// Both branch to the same matched instruction
					continue;
			}	}
			while (!((--at_code1P)->m_matchP));
			while (!((--at_code2P)->m_matchP));
			if (at_code1P->m_matchP == at_code2P) {
				// Both branch after the same matched instruction
				continue;
		}	}
		++cancelled_matches;
		if (code1P == start_code1P) {
			goto fail;
		}
		code1P->m_matchP = 0;
		code2P->m_matchP = 0;
	}

	g_original_matches  += original_matches;
	g_extra_matches     += extra_matches;
	g_cancelled_matches += cancelled_matches;
	end_clock = time(0);
#ifdef PROOF
	proof();
#endif

	report_clone();

	for (at_code1P = start_code1P; at_code1P <= last_match1P; ++at_code1P) {
		if ((at_code2P = at_code1P->m_matchP)) {

			if (g_matches_cacheP == g_matches_endP) {
				if (g_block_matchesP) {
					next_blockP = g_block_matchesP->m_nextP;
				} else {
					next_blockP = 0;
				}
				if (!next_blockP) {
					next_blockP = (block_matchesT *) Xmalloc(sizeof(block_matchesT));
					if (!next_blockP) {
						outofmemory();
					}
					next_blockP->m_nextP = 0;
					if (g_block_matchesP) {
						g_block_matchesP->m_nextP = next_blockP;
					} else {
						g_head_block_matchesP = next_blockP;
				}	}
				g_block_matchesP = next_blockP;

				g_matches_cacheP = g_block_matchesP->m_cache;
				g_matches_endP   = g_matches_cacheP + 8192;
			}
			tailPPP  = g_tail_matchesPPP + (at_code1P - function_code1P);
			tailPP   = *tailPPP;
			g_matches_cacheP->m_codeP = at_code2P;
			g_matches_cacheP->m_nextP = 0;
			*tailPPP = &g_matches_cacheP->m_nextP;
			*tailPP  = g_matches_cacheP++;
	}	}
done:
	for (; last_match1P >= start_code1P; --last_match1P) {
		if ((at_code2P = last_match1P->m_matchP)) {
			at_code2P->m_matchP    = 0;
			last_match1P->m_matchP = 0;
	}	}
	return(best_end2P);
fail:
	best_end2P = 0;
	goto done;
}

void
detect(void)
{
	static	Cfunction		*end_functions1P, *function3P;
	static	codeT			*skipP;
	static	matchesT		*matchesP, **matchesPP;
	static	int				lineno;
	static	time_t			clock_now;
	static	char			*P;
	static	int				lth;

	g_matches_lth     = sizeof(matchesT *) * g_max_function_lth;
	g_matchesPP       = (matchesT **)  Xmalloc(g_matches_lth);
	if (!g_matchesPP) {
		outofmemory();
	}
	g_tail_matchesPPP = (matchesT ***) Xmalloc(g_matches_lth);
	if (!g_tail_matchesPPP) {
		outofmemory();
	}

	for (source1P = Csource::g_headP; source1P; source1P = source1P->m_nextP) {
		if (g_verbose) {
			clock_now = time(0);
			P         = ctime(&clock_now);
			P[19]     = 0;
			fprintf(stderr, "%s: ", P+11);
			dump_wchar(source1P->m_nameP, stderr);
			fputc('\n', stderr);
		}
		function1P     = source1P->m_functionsP;
		for (end_functions1P = function1P + source1P->m_functions_count; function1P < end_functions1P; ++function1P) {
			start_code1P   = function_code1P = function1P->m_codeP;
			lth            = function1P->m_code_count;
			function_end1P = function_code1P + lth;
			assert(lth <= g_max_function_lth);
			memset(g_matchesPP, 0, lth * sizeof(matchesT *));
			while (--lth >= 0) {
				g_tail_matchesPPP[lth] = g_matchesPP + lth;
			}
			
			for (; start_code1P < function_end1P; ++start_code1P) {
				switch (g_match_lines) {
				case 2:
					lineno = start_code1P->m_source_lineno;
					if (lineno) {
						if (start_code1P != function_code1P && lineno == (int) start_code1P[-1].m_source_lineno) {
							continue;
						}
						break;
					}
				case 1:
					if (start_code1P->m_pop) {
						continue;
				}	}
					 
				for (start_code2P = start_code1P->m_hashP; start_code2P; ) {
					function2P      = start_code2P->m_functionP;
					source2P        = function2P->m_sourceP;

					if (g_match_lines == 1) {
						if (start_code2P->m_pop) {
							goto advance;
					}	}

					// Detect subsequence within sequences already matched

					for (matchesPP = g_matchesPP + (start_code1P - function_code1P); (matchesP = *matchesPP); matchesPP = &(matchesP->m_nextP)) {
						if (matchesP->m_codeP == start_code2P) {
							*matchesPP = matchesP->m_nextP;
							goto advance;
					}	}

					function_code2P = function2P->m_codeP;
					function_end2P  = function_code2P + function2P->m_code_count;
					if (function2P == function1P) {
						end_code2P = function_end1P;
						end_code1P = start_code2P;
					} else {
						end_code1P = function_end1P;
						end_code2P = function_end2P;
					}
					g_lth1 = end_code1P - start_code1P;
					g_lth2 = end_code2P - start_code2P;
					if (!permitted_clone()) {
						goto advance;
					}
					++g_comparisons;
					skipP         = detect_clone();
					start_code2P  = start_code2P->m_hashP;

					if (skipP) {
						for (; start_code2P; start_code2P = start_code2P->m_hashP) {
							function3P = start_code2P->m_functionP;
							if (function3P != function2P || start_code2P >= skipP) {
								break;
					}	}	}
					continue;
advance:			start_code2P = start_code2P->m_hashP;
			}	}

			g_block_matchesP = g_head_block_matchesP;
			if (g_block_matchesP) {
				g_matches_cacheP = g_block_matchesP->m_cache;
				g_matches_endP   = g_matches_cacheP + 8192;
			}
}	}	}

