#include "typedef.h"

#ifdef WIN32
#include <Windows.h>
#else
#include <sys/stat.h>
#include <sys/types.h>
#endif


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <signal.h>
#include <setjmp.h>

#ifndef WIN32
#include <unistd.h>
#endif

#include "object.h"
#include "util.h"
#include "ta.h"
#include "clics_output.h"
#include "opcode.h"
#include "code.h"
#include "detect.h"
#include "source.h"

int		g_dynamic            = 0;
int		g_hypertext_cnt      = 0;
char	*g_html_dirP         = 0;
char	*g_ta_fileP          = 0;
char	*g_clics_fileP       = 0;
int		g_match_lines		 = 1;
int		g_min_match_sequence = 15;
int		g_min_function_sequence = 10;
int		g_mismatch_cost      = -1;
int		g_same_cost			 = 1;
int		g_comparisons        = 0;
int		g_report_java        = 1;
int		g_report_assembler   = 1;
int		g_comments           = 0;
int		g_verbose            = 0;

int		g_debug_code         = 0;
int		g_top_clones         = 1024;
int		g_top_cols           = 8;
int		g_hillclimb          = 0;
time_t	g_hillclimb_elapse   = 0;
int		g_hillclimb_timeout  = 60;
int		g_iterations         = 0x7FFFFFFF;
int		g_best_weight        = 0;
int		g_exceptions         = 1;

int			g_read               = 0;
int			g_functions          = 0;
int			g_max_function_lth   = 0;
int			g_compiled_ok        = 0;
int			g_compiled_bad       = 0;

long long	g_original_matches   = 0;
long long	g_extra_matches      = 0;
long long	g_cancelled_matches  = 0;
int			g_max_blocks_seen    = 0;
int			g_max_iterations_seen= -1;

char	*g_classpathP        = 0;
char	*g_java[10]	         = {0,"-g","-classpath", ".", 0, 0};

char	**g_envPP;

FILE	*stdinF    = stdin;
FILE	*stdoutF   = stdout;
FILE	*redirectF = stdoutF;
FILE	*taF       = 0;
FILE	*tempF	   = 0;
FILE	*clicsF    = 0;

int		g_skip_compile   = 0;

jmp_buf     g_break_address;

typedef struct {
    int     signal;
    char    *nameP;
} sigtableT;

static sigtableT    sigtable[] = {
#ifndef WIN32
	{SIGHUP,	"HUP"},
	{SIGBUS,	"BUS"},
#endif
    {SIGINT,	"INT"},
    {SIGILL,	"ILL"},
    {SIGABRT,	"ABRT"},
    {SIGFPE,	"FPE"},
    {SIGSEGV,	"SEGV"},
	{SIGTERM,   "TERM"},
    {0, 0}
};

char *g_signalP = 0;

static void
handler(int sig)
{
    sigtableT   *sigtableP;

    for (sigtableP = sigtable; ; ++ sigtableP) {
        if (!sigtableP->nameP) {
            fprintf(stderr, "**UNEXPECTED SIGNAL %d CAUGHT**\n", sig);
            break;
        }
        if (sig == sigtableP->signal) {
            g_signalP = sigtableP->nameP;
            fprintf(stderr, "**SIG%s**\n", g_signalP);
            longjmp(g_break_address, sig);
            break;
}   }   }

static void
print_title(int argc, char *argv[])
{
	static int printed_title = 0;

	int	i;

	if (!printed_title) {
		fputs(HTML_HEADER "<title>", stdoutF);
		fprintf(stdoutF, "JCD Version " VERSION " Compiled " __DATE__ " "  __TIME__ );
		fputs("</title>\n<body>\n", stdoutF);
		fprintf(stdoutF, "<h3>Arguments to %s</h3>\n<p>\n", argv[0]);
		for (i = 1; i < argc; ++i) {
			fprintf(stdoutF,"\"%s\" ", argv[i]);
		}
		if (g_top_clones > 0) {
			fprintf(stdoutF, "<p>\n<a href=top.html>Top %d clones </a>\n", g_top_clones);
		}
		initialise_detect();
		ta_header(argc, argv);
		clics_header(argc, argv);
		fputs("<h3>Processing</h3>\n<p>\n", stdoutF);
		printed_title = 1;
}	}



static void
read_input(char *filenameP)
{
	readclassfile(filenameP, 0);
}

static void
read_inputs(const char *filenamesP)
{
	char	filename[2048];
	char	*P;
	int		c;
	FILE	*F;

	if (!strcmp(filenamesP, "-")) {
		F = stdin;
	} else {
		F = fopen(filenamesP, "r");
		if (!F) {
			fprintf(stderr, "Couldn't open %s\n", filenamesP);
			error_exit(1);
	}	}
	do {
		P = filename;
		for (;;) {
			c = getc(F);
			if (c == '\r' || c == '\n' || c == EOF) {
				break;
			}
			*P++ = c;
		}
		if (P != filename) {
			*P = 0;
			read_input(filename);
		}
	} while (c != EOF);

	if (F != stdin) {
		fclose(F);
}	}

static void
get_environment(int argc, char *argv[])
{
	char	*filenamesP = 0;
	int		i, val;
	char	*P;

	for (i = 1; i < argc;) {
		P = argv[i++];
		if (*P == '-') {
			switch (P[1]) {
			case 'a':
				g_match_lines = 0;
				if (P[2]) {
					P += 2;
				} else {
					if (argc <= i) {
						break;
					}
					P = argv[i++];
				}
				switch (P[0]) {
				case 'l':
					g_match_lines = 1;	// Match start of java source lines
					break;
				case 's':
					g_match_lines = 2;	// Match start of pcode steps
					break;
				}
				break;
			case 'b':
				g_best_weight = 1;
				break;
			case 'c':
				if (g_classpathP) {
					fprintf(stderr, "-c option may only be specified once\n");
					error_exit(1);
				}
				if (i < argc) {
					g_classpathP = argv[i++];
					g_java[3]    = g_classpathP;
				}
				break;
			case 'd':
				g_dynamic = 1;
				break;
			case 'D':
				g_debug_code  = 1;
				break;
			case 'e':
				g_exceptions = 0;
				break;
			case 'E':
				if (P[2]) {
					P += 2;
				} else {
					if (argc <= i) {
						break;
					}
					P = argv[i++];
				}
				val = atoi(P);
				if (val < 0) {
					fprintf(stderr, "-E %d is negative\n", val);
					error_exit(1);
				}
				g_hillclimb_timeout = val;
				break;
			case 'G':
				if (P[2]) {
					g_clics_fileP = P + 2;
				} else {
					if (argc <= i) {
						break;
					}
					g_clics_fileP = argv[i++];
				}
				break;
			case 'i':
				if (filenamesP) {
					fprintf(stderr, "-i option may only be specified once\n");
					error_exit(1);
				}
				if (P[2]) {
					filenamesP = P + 2;
				} else if (i < argc) {
					filenamesP = argv[i++];
				} else {
					filenamesP = "-";
				} 
				break;
			case 'h':
				if (P[2]) {
					g_html_dirP = P + 2;
				} else {
					if (argc <= i) {
						break;
					}
					g_html_dirP = argv[i++];
				}
#ifdef WIN32
				CreateDirectoryA(g_html_dirP, 0);
#else
				mkdir(g_html_dirP, 0755);
#endif
				sprintf(g_filename, "%s/index.html", g_html_dirP);
				redirectF = open_file(g_filename);
				stdoutF   = redirectF;
				break;
			case 'j':
				if (g_java[0]) {
					fprintf(stderr, "-j/ava option may only be specified once\n");
					error_exit(1);
				}
				if (P[2]) {
					P += 2;
				} else {
					if (argc <= i) {
						break;
					}
					P = argv[i++];
				}
				g_java[0] = P;
				break;
			case 't':
				if (P[2]) {
					g_ta_fileP = P + 2;
				} else {
					if (argc <= i) {
						break;
					}
					g_ta_fileP = argv[i++];
				}
				break;
			case 'S':
				g_skip_compile = 1;
				break;
			case 'T':
				if (P[2]) {
					P += 2;
				} else {
					if (argc <= i) {
						break;
					}
					P = argv[i++];
				}
				val = atoi(P);
				g_top_clones = val;
				break;
			case 'C':
				if (P[2]) {
					P += 2;
				} else {
					if (argc <= i) {
						break;
					}
					P = argv[i++];
				}
				val = atoi(P);
				if (val <= 0) {
					fprintf(stderr, "-C %d not positive\n", val);
					error_exit(1);
				}
				g_top_cols = val;
				break;
			case 'l':
				if (P[2]) {
					P += 2;
				} else {
					if (argc <= i) {
						break;
					}
					P = argv[i++];
				}
				val = atoi(P);
				if (val < 1) {
					fprintf(stderr, "-l/ength must be > 0\n");
					error_exit(1);
				}
				g_min_match_sequence = val;
				break;
			case 'L':
				if (P[2]) {
					P += 2;
				} else {
					if (argc <= i) {
						break;
					}
					P = argv[i++];
				}
				val = atoi(P);
				if (val < 1) {
					fprintf(stderr, "-L/ength must be > 0\n");
					error_exit(1);
				}
				g_min_function_sequence = val;
				break;
			case 'm':
				if (P[2]) {
					P += 2;
				} else {
					if (argc <= i) {
						break;
					}
					P = argv[i++];
				}
				val = atoi(P);
				if (val < 0) {
					fprintf(stderr, "-m/ismatch cost must be => 0\n");
					error_exit(1);
				}
				g_mismatch_cost = -val;
				break;
			case 's':
				if (P[2]) {
					P += 2;
				} else {
					if (argc <= i) {
						break;
					}
					P = argv[i++];
				}
				val = atoi(P);
				if (val < 0) {
					fprintf(stderr, "-s/ame cost must be => 0\n");
					error_exit(1);
				}
				g_same_cost = val;
				break;
			case 'v':
				g_verbose = 1;
				break;
			case 'r':
				g_report_java = 0;
				break;
			case 'R':
				g_report_assembler = 0;
				break;
			case 'x':
				g_comments = 1;
				break;
			case 'u':
				if (P[2]) {
					P += 2;
				} else {
					if (argc <= i) {
						break;
					}
					P = argv[i++];
				}
				val = atoi(P);
				if (val < 0) {
					fprintf(stderr, "-Use option must be non-negative\n");
					error_exit(1);
				}
				g_hillclimb = val;
				break;
			case 'I':
				if (P[2]) {
					P += 2;
				} else {
					if (argc <= i) {
						break;
					}
					P = argv[i++];
				}
				val = atoi(P);
				if (val < 0) {
					fprintf(stderr, "-i/terate option must be non-negative\n");
					error_exit(1);
				}
				g_iterations = val;
				break;
			default:
				fprintf(stderr, "Unknown option '%s'\n\n", P);
				fputs("\n\nOptions:\n\n"
					"  -a/ll [i/nstructions|l/ines|s/teps]\n"
					"  -b/estweight\n"
					"  -c/lasspath <paths>\n"
					"  -C/ols <number>\n"
					"  -d/ynamic\n"
					"  -D/ebug\n"
					"  -e/xceptions\n"
					"  -Elapse timeout\n"
					"  -h/tml <directory>\n"
					"  -i/nput filenames\n"
					"  -j/ava <program>\n"
					"  -k/eep assembler\n"
					"  -l/ength_assembler <min>\n"
					"  -L/ength_function <min>\n"
					"  -m/ismatch <cost>\n"
					"  -r/emove java reporting\n"
					"  -R/emove pcode reporting\n"
					"  -x/tra comments\n"
					"  -s/ame <cost>\n"
					"  -S/kip compile\n"
					"  -t/a <file>\n"
					"  -G/ui <file>\n"
					"  -T/op <number>\n"
					"  -u/se <number> [-i/terate <number>]\n"
					"  -v/erbose\n"
					, stderr);
				error_exit(1);
			}
			continue;
		}
		print_title(argc, argv);
		read_input(P); 
	}
	print_title(argc, argv);
	if (filenamesP) {
		read_inputs(filenamesP);
	}
}

int
main(int argc, char *argv[], char *env[])
{
	time_t		start_clock, read_clock, end_clock;
	int			hypertext_cnt;
	long long	total_matches;
	double		percentage;

    sigtableT   *sigtableP;

#ifndef WIN32
    struct      sigaction now;
#endif
	
	start_clock = time(0);

	g_envPP = env;
	get_environment(argc, argv);

	// Now read recursively any classes which are cited but not yet processed
	if (g_dynamic) {
		fprintf(stdoutF, "<h3>Dynamic</h3>\n<p>\n");
		pseudo_source();
	}

	fprintf(stderr, "\n\nRead %d Compiled %d Failed %d\n", g_read, g_compiled_ok, g_compiled_bad);
	// Now perform clone detection

	read_clock = time(0);

	start_report();
	hypertext_cnt = g_hypertext_cnt;

#ifndef WIN32
    memset(&now, 0, sizeof(now));
    now.sa_handler = handler;
#endif

    if (!(setjmp(g_break_address))) {
        for (sigtableP = sigtable; sigtableP->nameP; ++sigtableP) {
#ifdef WIN32
			if (signal(sigtableP->signal, handler) == SIG_ERR) {
#else
            if (sigaction(sigtableP->signal, &now, 0)) {
#endif
                fprintf(stderr, "Error: Can't set SIG%s signal handler\n", sigtableP->nameP);
        }   }
        detect();
    }

	finalize_report();

	end_clock = time(0);

	total_matches = g_original_matches + g_extra_matches - g_cancelled_matches;
	percentage    = ((double) g_extra_matches)/((double) total_matches);
	percentage   *= 100.0;

	fprintf(redirectF, "</table>\n");
	if (g_signalP) {
        fprintf(redirectF, "<p><font color=red>**EXECUTION INTERRUPTED BY **SIG%s**</font>\n", g_signalP);
    }
	fprintf(redirectF, "<p>\n<table>\n<tr><td>Start</td><td>%s</td></tr>\n", 	ctime(&start_clock));

	fprintf(redirectF, "<tr><td>Read</td><td align=right>%s</td></tr>\n", 	ctime(&read_clock));
	fprintf(redirectF, "<tr><td>End</td><td align=right>%s</td></tr>\n", 	ctime(&end_clock));
#ifndef WIN32
	fprintf(redirectF, "<tr><td>Memory</td><td align=right>%ld</td></tr>\n", (long) sbrk(0));
#endif
	if (!g_hillclimb) {
		fprintf(redirectF, "<tr><td>Detect Elapse</td><td align=right>%ld</td></tr>\n", (long) (end_clock-read_clock));
	} else {
		fprintf(redirectF, "<tr><td>Greedy Elapse</td><td align=right>%ld</td></tr>\n", (long) (end_clock-read_clock-g_hillclimb_elapse));
		fprintf(redirectF, "<tr><td>HillClimb Elapse</td><td align=right>%ld</td></tr>\n", (long) g_hillclimb_elapse);
	}
	fprintf(redirectF, "<tr><td>Sources</td><td align=right>%d</td></tr>\n", g_read);
	fprintf(redirectF, "<tr><td>Compiled</td><td align=right>%d</td></tr>\n", g_compiled_ok);
	if (g_compiled_bad) {
		fprintf(redirectF, "<tr><td>Uncompilable</td><td align=right>%d</td></tr>\n", g_compiled_bad);
	}
	fprintf(redirectF, "<tr><td>Functions</td><td align=right>%d</td></tr>\n", g_functions);
	fprintf(redirectF, "<tr><td>Comparisons</td><td align=right>%d</td></tr>\n", g_comparisons);
	fprintf(redirectF, "<tr><td>Original Matches Detected</td><td align=right>%lld</td></tr>\n", g_original_matches);
	if (g_hillclimb > 0) {
		fprintf(redirectF, "<tr><td>Hillclimbing Added</td><td align=right>%lld</td></tr>\n", g_extra_matches);
	}
	fprintf(redirectF, "<tr><td>Jump Matches Cancelled</td><td align=right>%lld</td></tr>\n", g_cancelled_matches);
	fprintf(redirectF, "<tr><td>Matches detected</td><td align=right>%lld</td></tr>\n", total_matches);
	fprintf(redirectF, "<tr><td>Clones detected</td><td align=right>%d</td></tr>\n", g_hypertext_cnt - hypertext_cnt);
	if (g_hillclimb > 0) {
		fprintf(redirectF, "<tr><td>Max Hillclimbing Blocks</td><td align=right>%d</td></tr>\n", g_max_blocks_seen);
		fprintf(redirectF, "<tr><td>Max Hillclimbing Iterations</td><td align=right>%d</td></tr>\n", g_max_iterations_seen+1);
		fprintf(redirectF, "<tr><td>Hillclimbing Percentage</td><td align=right>%f%%</td></tr>\n", percentage );
	}

	fprintf(redirectF, "</table>\n<p>\n<i>End of Report</i>\n</body>\n</html>\n");

	clics_trailer();
	ta_trailer();
	fprintf(stderr, "Done secs=%ld clones=%d matches=%lld",
				(long) (end_clock - read_clock),
				g_hypertext_cnt - hypertext_cnt,
				total_matches);
	if (g_hillclimb >= 0) {
		fprintf(stderr, " -u %d extra=%lld iterations=%d\n", 
				g_hillclimb,
				g_extra_matches,
				g_max_iterations_seen+1);
	}
	fprintf(stderr, "\n");
	return(0);
}

