#include "typedef.h"

#include <assert.h>
//#include <sys/types.h>
//#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>

#ifdef WIN32
#include <io.h>
#include <direct.h>
#include <sys/types.h>
#include <sys/stat.h>

#define open _open
#define read _read
#define close _close
#define alloca _alloca
#define getcwd _getcwd
#define lseek _lseek
#define access _access
#define stat _stat

#define F_OK 0
#else
#include <wchar.h>
#include <errno.h>
#include <unistd.h>
#include <alloca.h>

#define _O_BINARY 0
#endif

#include "stringcache.h"
#include "util.h"
#include "xmalloc.h"
#include "object.h"
#include "opcode.h"
#include "code.h"
#include "source.h"
#include "function.h"
#include "sort.h"

extern int				g_dynamic;
extern int				g_verbose;
extern int				g_exceptions;
extern int				g_match_lines;
extern char				*g_classpathP;
extern char				*g_java[10];
extern char				**g_envPP;
extern int				g_skip_compile;

extern int				g_read;
extern int				g_compiled_ok;
extern int				g_compiled_bad;
extern int				g_functions;
extern int				g_max_function_lth;

extern FILE				*stdoutF;

static u1T				*g_imageP         = 0;
static const u1T		*g_image_atP      = 0;
static int				g_image_size      = 0;
static int				g_image_max       = -1;

static linenumberT		*g_linenumbersP    = 0;
static int				g_linenumbers_max = 0;

#define Align(X) ((((long) (X)) + 3) & ~3)

#define AssertIndex(X)		assert(((unsigned int) X) < m_constant_pool_count)
#define AssertCode(X)    	assert(((unsigned int) X) < code_length)
#define AssertEndCode(X)  	assert(((unsigned int) X) <= code_length)

static const u1T	*
getU2(const u1T *startP, u2T *valP)
{
	register	const u1T	*P = startP;
	register	u2T			val;

	assert(!(((long) valP) & 1));
	val   = (*P++ << 8);
	val  |= *P++;
	*valP = val;
	return(P);
}

static const u1T	*
getU4(const u1T *startP, u4T *valP)
{
	register	const u1T	*P = startP;
	register	u4T			val;

	assert(!(((long) valP) & 3));
	val   = (*P++ << 24);
	val  |= (*P++ << 16);
	val  |= (*P++ << 8);
	val  |= *P++;
	*valP = val;
	return (P);
}
	
static const u1T *
getU8(const u1T *startP, u8T *valP)
{
	register	const u1T	*P = startP;

	u8T	val;

//	assert(!(((int) valP) & 7));
	val   = (((u8T) *P) << 56); ++P;
	val  |= (((u8T) *P) << 48); ++P;
	val  |= (((u8T) *P) << 40); ++P;
	val  |= (((u8T) *P) << 32); ++P;
	val  |= (((u8T) *P) << 24); ++P;
	val  |= (((u8T) *P) << 16); ++P;
	val  |= (((u8T) *P) << 8);  ++P;
	val  |= *P++;
	*valP = val;
	return (P);
}

/*
  4.4.7 The CONSTANT_Utf8_info Structure

  The CONSTANT_Utf8_info structure is used to represent constant string values. UTF-8 strings are encoded so that character 
  sequences that contain only non-null ASCII characters can be represented using only 1 byte per character, but characters 
  of up to 16 bits can be represented. All characters in the range '\u0001' to '\u007F' are represented by a single byte: 

0  bits 6-0  

The 7 bits of data in the byte give the value of the character represented. 
The null character ('\u0000') and characters in the range '\u0080' to '\u07FF' are represented by a pair of bytes x and y:

x:  1  1  0  bits 10-6  
y:  1  0  bits 5-0  

The bytes represent the character with the value ((x & 0x1f) << 6) + (y & 0x3f). 

Characters in the range '\u0800' to '\uFFFF' are represented by 3 bytes x, y, and z:

x:  1  1  1  0  bits 15-12  
y:  1  0  bits 11-6  
z:  1  0  bits 5-0  

The character with the value ((x & 0xf) << 12) + ((y & 0x3f) << 6) + (z & 0x3f) is represented by the bytes. 

The bytes of multibyte characters are stored in the class file in big-endian (high byte first) order.

*/

static const u1T *
getUtf8(int bytes, const u1T *compressedP, u2T **stringPP, int load)
{
	const u1T	*P       = compressedP;
	const u1T	*endP    = P + bytes;
	u2T			*startP  = *stringPP;
	u2T			*stringP = startP;
	u1T			c;
	u2T			val;

	for (; P < endP; ++P) {
		c = *P;
		if (!(c & 0x80)) {
			// One byte encoding
			val = c;
		} else {
			if (!(c & 0x20)) {
				// Two byte encoding
				val  = (c & 0x1F) << 6;
			} else {
				// Three byte encoding
				val  = (c & 0xF) << 12;
				c    = *++P;
				val |= (c & 0x3F) << 6;
			}
			c    = *++P;
			val |= (c & 0x3F);
		}
		if (load) {
			*stringP = val;
		}
		++stringP;
	}
	if (load) {
		*stringP = 0;
	}
	++stringP;
	*stringPP  = stringP;
	assert(P == endP);
	return(endP);
}

/* Read the class image from disk */

static int
read_image(const char *filenameP)
{
	int		fileno;
	int		ret;

	fileno = open(filenameP, O_RDONLY | _O_BINARY, 0);
	if (fileno == -1) {
		return(1);
	}
	g_image_size = lseek(fileno, 0L, SEEK_END);
	if (g_image_size < 0) {
		fprintf(stderr, "Can't lseek end of %s\n", filenameP);
		goto fail;
	}
	if (lseek(fileno, 0L, SEEK_SET) < 0) {
		fprintf(stderr, "Can't lseek start of %s\n", filenameP);
		goto fail;
	}
	if (g_image_size > g_image_max) {
		if (g_imageP) {
			Xfree(g_imageP);
		}
		g_imageP     = (u1T *) Xmalloc(g_image_size + 1);
		if (!g_imageP) {
			outofmemory();
		}
		g_image_max  = g_image_size;
	}

	ret = read(fileno, g_imageP, g_image_size);
	if (ret != g_image_size) {
		fprintf(stderr, "Can't read %s\n", filenameP);
		goto fail;
	}

	g_imageP[g_image_size] = 0;
	ret = 0;
done:
	if (close(fileno)) {
		fprintf(stderr, "Can't close %s\n", filenameP);
		ret = -1;
	}
	return(ret);
fail:
	ret = -1;
	goto done;
}

/*
    ClassFile {
    	u4 magic;
    	u2 minor_version;
    	u2 major_version;
    	u2 constant_pool_count;
    	cp_info constant_pool[constant_pool_count-1];
    	u2 access_flags;
    	u2 this_class;
    	u2 super_class;
    	u2 interfaces_count;
    	u2 interfaces[interfaces_count];
    	u2 fields_count;
    	field_info fields[fields_count];
    	u2 methods_count;
    	method_info methods[methods_count];
    	u2 attributes_count;
    	attribute_info attributes[attributes_count];
    }
*/

static int			g_memory         = 0;
static u1T			*g_memory_startP = 0;
static int			g_extra          = 0;
static u1T			*g_extra_startP  = 0;
static int			g_string_need    = 0;
static const u1T	*g_image_endP    = 0;

int
Csource::load_constant_pool(const char *filenameP, int load)
{
	u1T				*memoryP;
	u1T				*stringP, *startP;
	cp_infoT		*cpP, *endP;
	constantE		tag;
	int				i, need;
	u4T				magic;
	u2T				length;

	g_image_atP = g_imageP;
	if (!load) {
		memoryP     = (u1T *) 0;
		stringP     = (u1T *) 0;
		g_image_atP = getU4(g_image_atP, &magic);

		if (magic != 0xCAFEBABE) {
			fprintf(stderr, "%s is not a valid class file - has magic value 0x%08x\n", filenameP, magic);
			goto fail;
		}
		g_image_atP   = getU2(g_image_atP, &m_minor_version);
		g_image_atP   = getU2(g_image_atP, &m_major_version);
		g_image_atP   = getU2(g_image_atP, &m_constant_pool_count);
	} else {
		memoryP      = (u1T *) m_constant_poolP;
		stringP      = (u1T *) (m_constant_poolP + m_constant_pool_count);
		g_image_atP += 10;
	}

	cpP = (cp_infoT *) memoryP;
	if (m_constant_pool_count) {

		if (load) {
			cpP->tag = CONSTANT_Null;
		}
		++cpP;
		for (i = 1; i < m_constant_pool_count; ++i) {
			tag = (constantE) *g_image_atP++;
			if (load) {
				cpP->tag = tag;
			}
			switch (tag) {
			case CONSTANT_Utf8:
				g_image_atP  = getU2(g_image_atP, &length);
				startP       = stringP;
				g_image_atP  = getUtf8(length, g_image_atP, (u2T **) &stringP, load);
				if (load) {
					cpP->u.stringP = stringCache((u2T *) startP);
				}
				break;
			case CONSTANT_Integer:
			case CONSTANT_Float:
				if (load) {
					g_image_atP     = getU4(g_image_atP, &cpP->u.u4val);
				} else {
					g_image_atP    += 4;
				}
				break;
			case CONSTANT_NameAndType:
				if (load) {
					g_image_atP     = getU2(g_image_atP, &cpP->u.ref.index1);
					AssertIndex(cpP->u.ref.index1);
					g_image_atP     = getU2(g_image_atP, &cpP->u.ref.index2);
					AssertIndex(cpP->u.ref.index2);
				} else {
					g_image_atP    += 4;
				}
				break;
			case CONSTANT_Long:
			case CONSTANT_Double:
				if (load) {
					g_image_atP     = getU8(g_image_atP, &cpP->u.u8val);
				} else {
					g_image_atP    += 8;
				}
				// 8 byte values take 2 slots in the constant pool table
				++cpP;
				if (load) {
					cpP->tag = CONSTANT_Null;
				}
				++i;
				break;
			case CONSTANT_Class:
			case CONSTANT_String:
				if (load) {
					g_image_atP     = getU2(g_image_atP, &cpP->u.ref.index1);
					AssertIndex(cpP->u.ref.index1);
				} else {
					g_image_atP    += 2;
				}
				break;
			case CONSTANT_Fieldref:
			case CONSTANT_Methodref:
			case CONSTANT_InterfaceMethodref:
				if (load) {
					g_image_atP     = getU2(g_image_atP, &cpP->u.ref.index1);
					AssertIndex(cpP->u.ref.index1);
					g_image_atP     = getU2(g_image_atP, &cpP->u.ref.index2);
					AssertIndex(cpP->u.ref.index2);
				} else {
					g_image_atP    += 4;
				}
				break;
			default:
				fprintf(stderr, "Unknown constant code %d\n", tag);
				goto fail;
			}
			++cpP;
		}
		assert(cpP == m_constant_poolP + m_constant_pool_count);
		memoryP = (u1T *) cpP;
	}
	if (!load) {
		g_image_endP     = g_image_atP;
		g_memory         = memoryP - (u1T *) 0;
		g_string_need    = stringP - (u1T *) 0;
		need             = g_memory + g_string_need;
		memoryP          = (u1T *) Xmalloc(need + strlen(filenameP) + 1);
		if (!memoryP) {
			outofmemory();
		}
		m_constant_poolP = (cp_infoT *) memoryP;
		m_filenameP      = (char *) (memoryP + need);
		strcpy(m_filenameP, filenameP);
	} else {
		assert((memoryP - ((u1T *) m_constant_poolP)) == g_memory);
		assert((stringP - ((u1T *) m_constant_poolP)) == g_memory + g_string_need);
		assert(g_image_atP == g_image_endP);

		cpP  = m_constant_poolP;
		endP = cpP + m_constant_pool_count;
		while (++cpP < endP) {
			switch (cpP->tag) {
			case CONSTANT_String:
			case CONSTANT_Class:
				// Simplify the indirection here up front to help in managing constants later
				// Do it after the entire table is loaded to anticipate forward references if
				// they should exist
				cpP->u.stringP = get_Utf8(cpP->u.ref.index1);
			default:
				break;
		}	}
	}
	return(0);
fail:
	return(-1);
}

#define HASH_SIZE 0x190760

static codeT	**hash_table[HASH_SIZE] = {0};

static void
hash_pcode(codeT *pcodeP)
{
	int			 pcode;
	unsigned int hash;
	unsigned int hash1;
	codeT		 **hashPP;

	pcode           = pcodeP->m_pcode;
	hash            = opcodes[pcode].hash;

	assert(hash != 0xFFFFFF);

	switch (pcode) {
	case ILOAD:		// 0x000000
	case LLOAD:		// 0x010000
	case FLOAD:		// 0x020000
	case DLOAD:		// 0x030000
	case ALOAD:		// 0x040000
	case ISTORE:	// 0x050000
	case LSTORE:	// 0x060000
	case FSTORE:	// 0x070000
	case DSTORE:	// 0x080000
	case ASTORE:	// 0x090000
	{
		varT	*varP = (varT *) pcodeP->m_tableP;
		
		if (varP->m_nameP) {
			hash1 = ((unsigned long) varP->m_nameP) >> 2;
		} else {
			hash1 = varP->m_number;
		}
		hash |= (hash1 & 0xFFFF);
		break;
	}
	case GETSTATIC:	// 0x0a0000	
	case GETFIELD:	// 0x0b0000
	case PUTSTATIC: // 0x0c0000
	case PUTFIELD:	// 0x0d0000
	{
		fieldrefT	*fieldrefP = (fieldrefT *) pcodeP->m_tableP;

		hash1 = (((unsigned long) fieldrefP->m_classnameP) ^ 
			     ((unsigned long) fieldrefP->m_nameP)
			    ) >> 2;
		hash |= (hash1 & 0xFFFF);
		break;
	}
	case INVOKEVIRTUAL:	// 0x0e0000
	case INVOKESPECIAL:	// 0x0f0000
	case INVOKESTATIC:	// 0x100000
	{
		methodrefT	*methodrefP = (methodrefT *) pcodeP->m_tableP;

		hash1     = (((unsigned long) methodrefP->m_classnameP) ^ 
			         ((unsigned long) methodrefP->m_nameP)      ^
					 ((unsigned long) methodrefP->m_descriptorP)
					) >> 2;
		hash |= (hash1 & 0xFFFF);
		break;
	}
	case INVOKEINTERFACE: // 0x110000
	{
		interfacerefT	*methodrefP = (interfacerefT *) pcodeP->m_tableP;

		hash1     = (((unsigned long) methodrefP->m_classnameP) ^ 
			         ((unsigned long) methodrefP->m_nameP)      ^
					 ((unsigned long) methodrefP->m_descriptorP)
					 ) >> 2;
		hash1 ^= methodrefP->m_args;
		hash  |= (hash1 & 0xFFFF);
		break;
	}
	case NEW:		// 0x120000
	case ANEWARRAY:	// 0x130000
	case CHECKCAST:	// 0x140000
	case INSTANCEOF:// 0x150000
	{
		classrefT	*classrefP;
		
		classrefP = (classrefT *) pcodeP->m_tableP;

		hash1  = ((unsigned long) classrefP->m_classnameP) >> 2;
		hash  |= (hash1 & 0xFFFF);
		break;
	}
	case MULTIANEWARRAY: // 0x160000
	{
		arrayrefT	*arrayrefP = (arrayrefT *) pcodeP->m_tableP;

		hash1  = ((unsigned long) arrayrefP->m_classnameP) >> 2;
		hash1 ^= arrayrefP->m_dimensions;
		hash  |= (hash1 & 0xFFFF);
		break;
	}
	case IINC:		// 0x17000
	{
		iincT	*varP = (iincT *) pcodeP->m_tableP;
		
		if (varP->m_nameP) {
			hash1     = ((unsigned long) varP->m_nameP) >> 2;
		} else {
			hash1     = varP->m_number;
		}
		hash |= ((hash1 + varP->m_s4) & 0xFFFF);
		break;
	}
	case LDC:
	case LDC_W:
	case LDC2_W:	// 0x180000
	{
		cp_infoT	*cpP = (cp_infoT *) pcodeP->m_tableP;

		switch (cpP->tag) {
		case CONSTANT_Integer:
			hash1 = cpP->u.ival & 0x3FFF;
			break;
		case CONSTANT_Float:
			hash1 = 0x4000 | (cpP->u.ival & 0x3FFF);
			break;
		case CONSTANT_Double:
		case CONSTANT_Long:
			hash1 = 0x8000 | ((cpP->u.ival ^ (unsigned int) cpP->u.lval) & 0x3FFF);
			break;
		case CONSTANT_String:
		case CONSTANT_Class:
		{
			const u2T *P;

			hash1 = 0;
			for (P = cpP->u.stringP; *P; ++P) {
				hash1 <<= 1;
				hash1  ^= *P;
			}
			hash1 = 0xc000 | (hash1 & 0x3FFF);
			break;
		}
		default:
			assert(0);
		}
		hash |= hash1;
		break;
	}
	case IFEQ:		// 0x190000 - 0x19003F
	case IFNE:		
	case IFLT:		
	case IFGE:		
	case IFGT:
	case IFLE:
	case IF_ICMPEQ:
	case IF_ICMPNE:
	case IF_ICMPLT:
	case IF_ICMPGE:
	case IF_ICMPGT:
	case IF_ICMPLE:
	case IF_ACMPEQ:
	case IF_ACMPNE:
	case IFNULL:
	case IFNOTNULL:
	case GOTO:
	case GOTO_W:
	case JSR:
	case JSR_W:
	case RET_W:
	case LOOKUPSWITCH:
	case TABLESWITCH:
	case TRY_START:
	case DEFAULT:
		if (pcodeP > pcodeP->m_jump.P) {
			hash |= 0x20;
		}
		break;
	case BIPUSH:	// 0x190400
	case SIPUSH:	// 0x190500
	case NEWARRAY:	// 0x190600
		hash |= pcodeP->m_u4 & 0xFF;
		break;
	}

	assert(hash >= 0 && hash < HASH_SIZE);

	if ((hashPP = hash_table[hash])) {
		*hashPP = pcodeP; 
	}
	hash_table[hash] = &(pcodeP->m_hashP);
}

const static u2T	sourceFile[] =
	{'S','o','u','r','c','e','F','i','l','e',0};
const static u2T	constantValue[] =
	{'C','o','n','s','t','a','n','t','V','a','l','u','e',0};
const static u2T	code[] =
	{'C','o','d','e',0};
const static u2T	deprecated[] =
	{'D','e','p','r','e','c','a','t','e','d',0};
const static u2T	enclosingMethod[] = 
	{'E','n','c','l','o','s','i','n','g','M','e','t','h','o','d',0};
const static u2T	exceptions[] = 
	{'E','x','c','e','p','t','i','o','n','s',0};
const static u2T	innerClasses[] =
	{'I','n','n','e','r','C','l','a','s','s','e','s',0};
const static u2T	lineNumberTable[] =
	{'L','i','n','e','N','u','m','b','e','r','T','a','b','l','e',0};
const static u2T	localVariableTable[] =
	{'L','o','c','a','l','V','a','r','i','a','b','l','e','T','a','b','l','e',0};
const static u2T	synthetic[] =
	{'S','y','n','t','h','e','t','i','c',0};

int
Csource::load_tables(int load)
{
	const u1T		*image_atP   = g_image_atP;
	const u1T		*image_at1P;
	const u1T		*image_at2P;
	u1T				*memoryP, *extraP;
	field_infoT		*fieldP, *field_endP;
	Cfunction		*functionP, *function_endP;
	u2T				attribute_name_index, throws_count;
	u2T				*nameP, *interfaceP, *interface_endP, *descriptorP;
	u4T				method_number, i, j;
	u4T				attribute_length;
	u2T				fields_count, functions_count;
	u2T				attributes_count;
	labelT			*sort_atP, *sort_endP;
	int				last_lineno;
	int				lth;
	
	/*
		u2 access_flags;
    	u2 this_class;
    	u2 super_class;
    	u2 interfaces_count;
	*/
	if (load) {
		sort_atP      = Csort::g_labelsP;
		sort_endP     = sort_atP + Csort::g_labels_count;
		memoryP       = g_memory_startP;
		extraP        = g_extra_startP;
		m_interfacesP = (u2T *) memoryP;
		image_atP     = getU2(image_atP, &m_access_flags);
		image_atP     = getU2(image_atP, &m_this_class);
		AssertIndex(m_this_class);
		image_atP     = getU2(image_atP, &m_super_class);
		AssertIndex(m_super_class);
		image_atP    += 2;

		nameP = get_classname(m_this_class);
		if (!m_nameP) {
			m_nameP = nameP;
			chain(this);
		} else {
			if (unicodecmp(nameP, m_nameP)) {
				fputs("Class \"", stderr);
				dump_wchar(m_nameP, stderr);
				fputs( "\" has internal classname \"", stderr);
				dump_wchar(nameP, stderr);
				fputs("\"\n", stderr);
		}	}
		*g_tailPP = this;
		g_tailPP  = &m_nextP;
	} else {
		memoryP    = 0;
		extraP     = 0;
		image_atP += 6;
		image_atP  = getU2(image_atP, &m_interfaces_count);
		Csort::clear_sort();
	}

	// interfaces

	interfaceP     = (u2T *) memoryP;
	interface_endP = interfaceP + m_interfaces_count;	
	if (load) {
		for (; interfaceP < interface_endP; ++interfaceP) {
			image_atP = getU2(image_atP, interfaceP);
			AssertIndex(*interfaceP);
		} 
	} else {
		image_atP += 2 * m_interfaces_count;
	}
	memoryP   = (u1T *) interface_endP;

	// fields 

	image_atP = getU2(image_atP, &fields_count);
	m_fields_count = fields_count;
	m_fieldsP  = fieldP = (field_infoT *) Align(memoryP);
	field_endP = fieldP + fields_count;
	memoryP    = (u1T *) field_endP;

	for (; fieldP < field_endP; ++fieldP) {
		if (load) {
			image_atP = getU2(image_atP, &fieldP->m_access_flags);
			image_atP = getU2(image_atP, &fieldP->m_name_index);			// get_Utf8(fieldsP->name_index);
			AssertIndex(fieldP->m_name_index);
			image_atP = getU2(image_atP, &fieldP->m_descriptor_index);	// get_Utf8(fieldsP->descriptor_index);
			AssertIndex(fieldP->m_descriptor_index);
			image_atP = getU2(image_atP, &attributes_count);
			fieldP->m_constant_index = 0;
			fieldP->m_flags          = 0;
		} else {
			image_atP += 6;
			image_atP = getU2(image_atP, &attributes_count);
		}
		for (i = 0; i < attributes_count; ++i) {
			image_atP  = getU2(image_atP, &attribute_name_index);
			AssertIndex(attribute_name_index);
			image_atP  = getU4(image_atP, &attribute_length);
			image_at1P = image_atP;
			image_atP += attribute_length;

			if (load) {
				nameP     = get_Utf8(attribute_name_index);
				if (!unicodecmp(nameP, constantValue)) {
					assert(attribute_length == 2);
					getU2(image_at1P, &fieldP->m_constant_index);
					AssertIndex(fieldP->m_constant_index);
					fieldP->m_flags |= ConstantX;
					continue;
				}
				if (!unicodecmp(nameP, deprecated)) {
					fieldP->m_flags |= DeprecatedX;
					continue;
			}	}
		}
	}

	// methodsP

	image_atP = getU2(image_atP, &functions_count);
	m_functions_count = functions_count;
	m_functionsP      = functionP = (Cfunction *) Align(memoryP);
	function_endP     = functionP + functions_count;
	g_functions      += functions_count;
	memoryP         = (u1T *) function_endP;
	for (method_number = 0; functionP < function_endP; ++functionP, ++method_number) {

		if (load) {
			u2T index;

			memset(functionP, 0, sizeof(Cfunction));
			functionP->m_sourceP = this;
			image_atP = getU2(image_atP, &functionP->m_access_flags);
			image_atP = getU2(image_atP, &index);
			AssertIndex(index);
			functionP->m_nameP = get_Utf8(index);
			image_atP = getU2(image_atP, &index);
			AssertIndex(index);
			functionP->m_descriptorP = get_Utf8(index);
			image_atP = getU2(image_atP, &attributes_count);
		} else {
			image_atP += 6;
			image_atP = getU2(image_atP, &attributes_count);
		}
		for (i = 0; i < attributes_count; ++i) {
			image_atP  = getU2(image_atP, &attribute_name_index);
			AssertIndex(attribute_name_index);
			image_atP  = getU4(image_atP, &attribute_length);
			nameP      = get_Utf8(attribute_name_index);

			image_at1P = image_atP;
			image_atP += attribute_length;
			if (load) {
				if (!unicodecmp(nameP, synthetic)) {
					functionP->m_flags |= SyntheticX;
					continue;
				}
				if (!unicodecmp(nameP, deprecated)) {
					functionP->m_flags |= DeprecatedX;
					continue;
			}	}

			if (!unicodecmp(nameP, exceptions)) {
				image_at1P = getU2(image_at1P, &throws_count);
				memoryP    = (u1T *) Align(memoryP);
				if (load) {
					functionP->m_throws_count  = throws_count;
					functionP->m_throws_indexP = (u2T *) memoryP;
				}
				if (load) {
					for (j = 0; j < throws_count; ++j) {
						image_at1P = getU2(image_at1P, (u2T *) memoryP);
						AssertIndex(*((u2T *) memoryP));
						memoryP   += 2;
					}
				} else {
					memoryP += (2 * throws_count);
				}
				continue;
			}
			if (!unicodecmp(nameP, code)) {
				localvariableT	*localvariablesP     = 0;
				localvariableT	*localvariable_endP  = 0;
				u2T				localvariables_count;
				localvariableT	*localvariableP;

				u2T				u2, exception_table_count;
				s2T				s2;
				u4T				u4, code_length;
				s4T				s4;
				const u1T		*startCodeP, *endCodeP;
				u1T				u1;
				u4T				attributes_count, lineno_changes_at_pc, current_lineno, pc, flags;
				pcodeE			pcode;
				const ocodeT	*opcodeP, *opcode1P;
				u4T				wide_index = 0;
				u4T				attribute_length;
				const u2T		*nameP;
				u2T				attribute_name_index;
				linenumberT		*linenumberP, *linenumber_endP;
				codeT			*pcodeP, *code1P, *code2P;

				if (load) {
					image_at1P  = getU2(image_at1P, &functionP->m_max_stack);
					image_at1P  = getU2(image_at1P, &functionP->m_max_locals);
				} else {
					image_at1P += 4;
				}

				current_lineno       = 0xFFFFFFFF;
				lineno_changes_at_pc = 0x7FFFFFFF;
				linenumberP          = 0;
				
				startCodeP     = getU4(image_at1P, &code_length);
				endCodeP       = image_at1P = startCodeP + code_length;

				image_at1P     = getU2(image_at1P, &exception_table_count);

				if (load || !g_exceptions) {
					image_at1P += 8 * exception_table_count;
				} else {
					exceptionT	*exceptionP;
					exceptionT	*exception_endP;
					u2T	catch_type;

					exceptionP = (exceptionT *) Xmalloc(sizeof(exceptionT) * exception_table_count);
					if (!exceptionP) {
						outofmemory();
					}
					for (exception_endP = exceptionP + exception_table_count; exceptionP < exception_endP; ++exceptionP) {
						image_at1P = getU2(image_at1P, &exceptionP->m_start_pc);
						AssertCode(exceptionP->m_start_pc);
						image_at1P = getU2(image_at1P, &exceptionP->m_end_pc);
						AssertEndCode(exceptionP->m_end_pc);
						image_at1P = getU2(image_at1P, &exceptionP->m_handler_pc);
						AssertCode(exceptionP->m_handler_pc);
						image_at1P = getU2(image_at1P, &catch_type);

						if (!catch_type) {
							exceptionP->m_catch_typeP = 0;
						} else {
							exceptionP->m_catch_typeP = get_classname(catch_type);
							if (g_dynamic) {
								locate1(exceptionP->m_catch_typeP);
						}	}
						Csort::append_exception(method_number, exceptionP);
					}
				} 

				image_at1P = getU2(image_at1P, &u2);
				attributes_count = u2;
				linenumberP      = 0;
				current_lineno   = 0;
				for (j = 0; j < attributes_count; ++j) {
					image_at1P = getU2(image_at1P, &attribute_name_index);
					AssertIndex(attribute_name_index);
					nameP  = get_Utf8(attribute_name_index);
					image_at1P  = getU4(image_at1P, &attribute_length);
					image_at2P  = image_at1P;
					image_at1P += attribute_length;
						
					if (load && !unicodecmp(nameP, lineNumberTable)) {
						u2T			linenumbers_count;

						image_at2P      = getU2(image_at2P, &linenumbers_count);
						if (linenumbers_count >= g_linenumbers_max) {
							g_linenumbers_max = (linenumbers_count + 1024) & ~1023;
							if (g_linenumbersP) {
								Xfree((void **) &g_linenumbersP);
							}
							g_linenumbersP = (linenumberT *) Xmalloc(g_linenumbers_max * sizeof(linenumberT));
							if (!g_linenumbersP) {
								outofmemory();
							}	}

						linenumberP     = g_linenumbersP;
						linenumber_endP = g_linenumbersP + linenumbers_count;

						for (; linenumberP < linenumber_endP; ++linenumberP) {
							image_at2P = getU2(image_at2P, &linenumberP->m_start_pc);
							image_at2P = getU2(image_at2P, &linenumberP->m_line_number);
						}
						assert(image_at2P == image_at1P);
														
						linenumber_endP->m_line_number = 0;
						linenumber_endP->m_start_pc    = 0xFFFF;
						current_lineno                 = g_linenumbersP->m_line_number;
						linenumberP                    = g_linenumbersP+1;
						lineno_changes_at_pc           = linenumberP->m_start_pc;
						continue;
					}

					if (!unicodecmp(nameP, localVariableTable)) {

						assert(!localvariablesP);
						image_at2P         = getU2(image_at2P, &localvariables_count);
						assert(localvariables_count * 10 + 2 == (int) attribute_length);
						localvariablesP    = (localvariableT *) Align(memoryP);
						localvariable_endP = localvariablesP + localvariables_count;

						if (load) {
							for (localvariableP = localvariablesP; localvariableP < localvariable_endP; ++localvariableP) {
								image_at2P = getU2(image_at2P, &localvariableP->m_start_pc);
								image_at2P = getU2(image_at2P, &u2);
								localvariableP->m_end_pc = localvariableP->m_start_pc + u2;
								image_at2P = getU2(image_at2P, &u2);
								localvariableP->m_nameP = get_Utf8(u2);
								image_at2P = getU2(image_at2P, &u2);
								localvariableP->m_descriptorP = get_Utf8(u2);
								if (g_dynamic) {
									locate(localvariableP->m_descriptorP);
								}
								image_at2P = getU2(image_at2P, &localvariableP->m_index);
								AssertCode(localvariableP->m_start_pc);
								AssertEndCode((unsigned int) (localvariableP->m_end_pc));
							}
							functionP->m_localvariablesP      = localvariablesP;
							functionP->m_localvariables_count = localvariables_count;
							assert (image_at2P == image_at1P);
						} 
						memoryP = (u1T *) localvariable_endP;
						continue;
					}
				}

				memoryP = (u1T *) Align(memoryP);
				pcodeP  = (codeT *) memoryP;
				if (load) {
					functionP->m_codeP = pcodeP;
				}
				
				for (image_at2P = startCodeP; image_at2P < endCodeP; ++pcodeP) {
					pc = (u4T) (image_at2P - startCodeP);
					if (linenumberP) {
						while (pc >= lineno_changes_at_pc) {
							current_lineno = linenumberP->m_line_number;
							++linenumberP;
							lineno_changes_at_pc = linenumberP->m_start_pc;
					}	}

					pcode   = (pcodeE) *image_at2P++;
					opcodeP = opcodes + pcode;
					flags   = opcodeP->flags;

					if (flags & INVALIDX) {
						fprintf(stderr, ": Unknown pcode operation of %d 0x%02x\n", pcode, pcode);
						goto fail;
					}

					s4 = opcodeP->variable;
					u4 = (u4T) s4;

					if (flags & (U1X|U2X|U4X)) {
						if ((flags & U1X)) {
							if (wide_index) {
								image_at2P = getU2(image_at2P, &u2);
								u4         = u2;
								s4         = ((signed short) u2);
								if (pcode != IINC) {
									wide_index = 0;
								}
							} else {
								u1 = *image_at2P++;
								u4 = u1;
								s4 = ((signed char) u1);
							}
						} else if (flags & U2X) {
							image_at2P = getU2(image_at2P, &u2);
							u4         = u2;
							s4         = ((signed short) u2);
						} else {
							image_at2P = getU4(image_at2P, &u4);
							s4         = (signed int) u4;
					}	}

					if (load) {

						for (;sort_atP < sort_endP && sort_atP->method == (int) method_number && ((int) sort_atP->to) <= pc; ++sort_atP) {
							memset(pcodeP, 0, sizeof(codeT));
							pcodeP->m_functionP = functionP;
							pcodeP->m_source_lineno = current_lineno;
							pcodeP->m_pc        = sort_atP->to;
							opcode1P            = opcodes + sort_atP->pcode;
							pcodeP->m_pcode     = opcode1P->pcode;
							pcodeP->m_pop       = opcode1P->pop;
							pcodeP->m_push      = opcode1P->push;
							pcodeP->m_tableP    = (u4T *) sort_atP->exceptionP;
							if (pcodeP->m_pcode == TRY_START) {
								pcodeP->m_jump.offset = sort_atP->exceptionP->m_handler_pc - pcodeP->m_pc;
							}
							++pcodeP;
						}

						assert(((u1T *) pcodeP) >= g_memory_startP);
						assert(((u1T *) (pcodeP + 1)) <=  g_memory_startP + g_memory);
						memset(pcodeP, 0, sizeof(codeT));
						pcodeP->m_functionP     = functionP;
						pcodeP->m_source_lineno = current_lineno;
						pcodeP->m_pc            = pc;
						pcodeP->m_pcode         = (u1T) opcodeP->pcode;
						pcodeP->m_pop           = opcodeP->pop;
						pcodeP->m_push          = opcodeP->push;
					
						if (flags & IX) {
							pcodeP->m_u4 = u4;
						} else {
							pcodeP->m_s4 = s4;
					}	}
					
					// Pushing Constants
					// http://sunsite.ee/java/vmspec/vmspec-14.html#HEADING14-0

					switch (pcode) {
					case ILOAD:
					case ILOAD_0:
					case ILOAD_1:
					case ILOAD_2:
					case ILOAD_3:
					case LLOAD:
					case LLOAD_0:
					case LLOAD_1:
					case LLOAD_2:
					case LLOAD_3:
					case FLOAD:
					case FLOAD_0:
					case FLOAD_1:
					case FLOAD_2:
					case FLOAD_3:
					case DLOAD:
					case DLOAD_0:
					case DLOAD_1:
					case DLOAD_2:
					case DLOAD_3:
					case ALOAD:
					case ALOAD_0:
					case ALOAD_1:
					case ALOAD_2:
					case ALOAD_3:

					case ISTORE:
					case ISTORE_0:
					case ISTORE_1:
					case ISTORE_2:
					case ISTORE_3:
					case LSTORE:
					case LSTORE_0:
					case LSTORE_1:
					case LSTORE_2:
					case LSTORE_3:
					case FSTORE:
					case FSTORE_0:
					case FSTORE_1:
					case FSTORE_2:
					case FSTORE_3:
					case DSTORE:
					case DSTORE_0:
					case DSTORE_1:
					case DSTORE_2:
					case DSTORE_3:
					case ASTORE:
					case ASTORE_0:
					case ASTORE_1:
					case ASTORE_2:
					case ASTORE_3:
					{
						varT	*varP;
						
						varP   = (varT *) Align(extraP);
						extraP = (u1T *) (varP + 1);
						if (load) {
							int	pc1;
							
							varP->m_number       = u4;
							varP->m_nameP        = 0;
							varP->m_descriptorP  = 0;

							pc1 = (u4T) (image_at2P - startCodeP);
							for (localvariableP = localvariablesP; localvariableP < localvariable_endP; ++localvariableP) {
								if (localvariableP->m_index == u4 && localvariableP->m_start_pc <= pc1 && localvariableP->m_end_pc >= pc1) {
									varP->m_nameP       = localvariableP->m_nameP;
									varP->m_descriptorP = localvariableP->m_descriptorP;
									break;
							}	}
							pcodeP->m_tableP     = (u4T *) varP;
						}
						break;
					}
					case IINC:		// iinc
					{
						iincT	*varP;
						
						varP   = (iincT *) Align(extraP);
						extraP = (u1T *) (varP + 1);
						
						if (wide_index) {
							image_at2P = getU2(image_at2P, &u2);
							s2 = ((signed short) u2);
							wide_index = 0;
						} else {
							s2 = *((signed char *) image_at2P);
							++image_at2P;
						}

						if (load) {
							int	pc1;

							varP->m_number       = u4;
							varP->m_nameP        = 0;
							varP->m_descriptorP  = 0;
							varP->m_s4           = s2;

							pc1 = (u4T) (image_at2P - startCodeP);
							for (localvariableP = localvariablesP; localvariableP < localvariable_endP; ++localvariableP) {
								if (localvariableP->m_index == u4 && localvariableP->m_start_pc <= pc1 && localvariableP->m_end_pc >= pc1) {
									varP->m_nameP       = localvariableP->m_nameP;
									varP->m_descriptorP = localvariableP->m_descriptorP;
									break;
							}	}
							pcodeP->m_tableP     = (u4T *) varP;
						}
						break;
					}
					case GETSTATIC:	
					case GETFIELD:
					case PUTSTATIC:
					case PUTFIELD:
					{
						fieldrefT	*fieldrefP;
						
						fieldrefP = (fieldrefT *) Align(extraP);
						extraP    = (u1T *) (fieldrefP + 1);

						if (load) {
							cp_infoT	*cpP;

							assert(u4 > 0 && u4 < m_constant_pool_count);
							cpP = m_constant_poolP + u4;
							assert(cpP->tag == CONSTANT_Fieldref);
							fieldrefP->m_classnameP   = get_classname(cpP->u.ref.index1);
							if (g_dynamic) {
								locate1(fieldrefP->m_classnameP);
							}
							u4 = cpP->u.ref.index2;
							assert(u4 > 0 && u4 < m_constant_pool_count);
							cpP = m_constant_poolP + u4;
							assert(cpP->tag == CONSTANT_NameAndType);
							fieldrefP->m_nameP       = get_Utf8(cpP->u.ref.index1);
							fieldrefP->m_descriptorP = get_Utf8(cpP->u.ref.index2);
							pcodeP->m_tableP = (u4T *) fieldrefP;
							if (!fieldrefP->m_descriptorP[1]) {
								switch (fieldrefP->m_descriptorP[0]) {
								case 'J':
								case 'D':
									switch (pcode) {
									case GETSTATIC:
									case GETFIELD:
										pcodeP->m_push = 2;
										break;
									case PUTSTATIC:
										pcodeP->m_pop = 2;
										break;
									case PUTFIELD:
										pcodeP->m_pop = 3;
									default:
										break;
									}
							}	}
						}
						break;
					}
					case INVOKEVIRTUAL:
					case INVOKESPECIAL:
					case INVOKESTATIC:
					{
						methodrefT	*methodrefP;
						
						methodrefP = (methodrefT *) Align(extraP);
						extraP    = (u1T *) (methodrefP + 1);

						if (load) {
							cp_infoT	*cpP;

							assert(u4 > 0 && u4 < m_constant_pool_count);
							cpP = m_constant_poolP + u4;
							assert(cpP->tag == CONSTANT_Methodref);
							methodrefP->m_classnameP   = get_classname(cpP->u.ref.index1);
							if (g_dynamic) {
								locate1(methodrefP->m_classnameP);
							}
							u4 = cpP->u.ref.index2;
							assert(u4 > 0 && u4 < m_constant_pool_count);
							cpP = m_constant_poolP + u4;
							assert(cpP->tag == CONSTANT_NameAndType);
							methodrefP->m_nameP       = get_Utf8(cpP->u.ref.index1);
							methodrefP->m_descriptorP = descriptorP = get_Utf8(cpP->u.ref.index2);
							pcodeP->m_tableP          = (u4T *) methodrefP;

							count_parameters(descriptorP, &(pcodeP->m_pop), &(pcodeP->m_push));
						}
						break;
					}
					case INVOKEINTERFACE:
					{
						interfacerefT	*methodrefP;
						
						methodrefP = (interfacerefT *) Align(extraP);
						extraP    = (u1T *) (methodrefP + 1);

						if (load) {
							cp_infoT	*cpP;

							assert(u4 > 0 && u4 < m_constant_pool_count);
							cpP = m_constant_poolP + u4;
							assert(cpP->tag == CONSTANT_InterfaceMethodref);
							methodrefP->m_classnameP   = get_classname(cpP->u.ref.index1);
							if (g_dynamic) {
								locate1(methodrefP->m_classnameP);
							}
							u4 = cpP->u.ref.index2;
							assert(u4 > 0 && u4 < m_constant_pool_count);
							cpP = m_constant_poolP + u4;
							assert(cpP->tag == CONSTANT_NameAndType);
							methodrefP->m_nameP       = get_Utf8(cpP->u.ref.index1);
							methodrefP->m_descriptorP = descriptorP = get_Utf8(cpP->u.ref.index2);
							methodrefP->m_args        = *image_at2P;
							pcodeP->m_tableP          = (u4T *) methodrefP;

							count_parameters(descriptorP, &(pcodeP->m_pop), &(pcodeP->m_push));
							pcodeP->m_pop++;
							assert(pcodeP->m_pop == methodrefP->m_args + 1);

						}
						++image_at2P;
						assert(!*image_at2P); // 2nd byte reserved (should always be 0)
						++image_at2P;
						break;
					}

					// Managing arrays
					// http://sunsite.ee/java/vmspec/vmspec-18.html#HEADING18-0

					case MULTIANEWARRAY:
					{
						arrayrefT	*arrayrefP = (arrayrefT *) Align(extraP);

						extraP = (u1T *) (arrayrefP + 1);
						if (load) {
							int			dimensions;

							arrayrefP->m_classnameP = get_classname(u4);
							if (g_dynamic) {
								locate1(arrayrefP->m_classnameP);
							}
							arrayrefP->m_dimensions = dimensions = *image_at2P;
							pcodeP->m_tableP        = (u4T *) arrayrefP;
							pcodeP->m_pop           = dimensions;
						}
						++image_at2P;
						break;
					}
					case NEW:
					case CHECKCAST:
					case INSTANCEOF:
					{
						classrefT	*classrefP = (classrefT *) Align(extraP);

						extraP = (u1T *) (classrefP + 1);
						if (load) {
							classrefP->m_classnameP = get_classname(u4);
							if (g_dynamic) {
								locate1(classrefP->m_classnameP);
							}
							pcodeP->m_tableP = (u4T *) classrefP;
						}
						break;
					}
					case ANEWARRAY:
					{
						classrefT	*classrefP = (classrefT *) Align(extraP);

						extraP = (u1T *) (classrefP + 1);
						if (load) {
							classrefP->m_classnameP = get_classname(u4);
							if (g_dynamic) {
								locate(classrefP->m_classnameP);
							}
							pcodeP->m_tableP = (u4T *) classrefP;
						}
						break;
					}
					case LDC:
					case LDC_W:
					case LDC2_W:
					{
						if (load) {
							assert(u4 > 0 && u4 < m_constant_pool_count);
							pcodeP->m_tableP = (u4T *) (m_constant_poolP + u4);
						}
						break;
					}
					case IFEQ:
					case IFNE:
					case IFLT:
					case IFGE:
					case IFGT:
					case IFLE:
					case IF_ICMPEQ:
					case IF_ICMPNE:
					case IF_ICMPLT:
					case IF_ICMPGE:
					case IF_ICMPGT:
					case IF_ICMPLE:
					case IF_ACMPEQ:
					case IF_ACMPNE:
					case IFNULL:
					case IFNOTNULL:
					case GOTO:
					case GOTO_W:
					case JSR:
					case JSR_W:
					{
						if (load) {
							pcodeP->m_jump.offset = s4;
							AssertCode((pc + s4));
						}
						break;
					} 
					case TABLESWITCH:	// tableswitch
					{
						u4T	def;
						s4T	low, high;
						s4T	i;

						while ((image_at2P-startCodeP) & 3) {
							assert(!*image_at2P);
							++image_at2P;
						}
						extraP     = (u1T *) Align(extraP);
						image_at2P = getU4(image_at2P,         &def);	// default
						image_at2P = getU4(image_at2P, (u4T *) &low);	// low
						image_at2P = getU4(image_at2P, (u4T *) &high);	// high
						assert(high >= low);
						if (load) {
							AssertCode((pc + def));
							for (i = low; i <= high; ++i) {
								pcodeP->m_s4 = i;
								image_at2P = getU4(image_at2P, (u4T *) &(pcodeP->m_jump.offset));	// jump table
								AssertCode((pc + pcodeP->m_jump.offset));
								++pcodeP;
								memset(pcodeP, 0, sizeof(codeT));
								pcodeP->m_functionP = functionP;
								pcodeP->m_source_lineno = current_lineno;
								pcodeP->m_pc        = pc;
								pcodeP->m_pcode     = (u1T) opcodeP->pcode;
								pcodeP->m_pop       = opcodeP->pop;
								pcodeP->m_push      = opcodeP->push;
							}
							pcodeP->m_pcode         = DEFAULT;
							pcodeP->m_s4            = 0;
							pcodeP->m_jump.offset   = def;
						} else {
							for (i = low; i <= high; ++i) {
								image_at2P = getU4(image_at2P, &u4); // jump table
							}
							pcodeP  += (high - low + 1);
						}
						break;
					}
					case LOOKUPSWITCH:
					{
						codeT *start_tableP;
						u4T	def, pairs, i;

						while ((image_at2P - startCodeP) & 3) {
							assert(!*image_at2P);
							++image_at2P;
						}
						extraP     = (u1T *) Align(extraP);
						image_at2P = getU4(image_at2P, &def);	// default
						image_at2P = getU4(image_at2P, &pairs);	// number of pairs
						assert(pairs > 0);
						if (load) {
							start_tableP              = pcodeP;
							pcodeP->m_pcode           = TABLESWITCH;
							for (i = 0; i < pairs; ++i) {
								image_at2P = getU4(image_at2P, &(pcodeP->m_u4));					// match
								image_at2P = getU4(image_at2P, (u4T *) &(pcodeP->m_jump.offset));	// jump
								AssertCode((pc + pcodeP->m_jump.offset));
								++pcodeP;
								memset(pcodeP, 0, sizeof(codeT));
								pcodeP->m_functionP = functionP;
								pcodeP->m_source_lineno = current_lineno;
								pcodeP->m_pc        = pc;
								pcodeP->m_pcode     = TABLESWITCH;
								pcodeP->m_pop       = opcodeP->pop;
								pcodeP->m_push      = opcodeP->push;
							}
							pcodeP->m_pcode         = DEFAULT;
							pcodeP->m_jump.offset   = def;

							// Now sort the values (because can't be sure that pre-sorted)
							// Use an exchange sort in hope that pre-sorted
							code1P = start_tableP;
							code2P = code1P + 1;
							for (; ;) {
								if (pcodeP <= code2P) {
									break;
								}
								s4 = code2P->m_s4;
								if (s4 < code1P->m_s4) {
									code2P->m_s4          = code1P->m_s4;
									code1P->m_s4          = s4;
									s4                    = code2P->m_jump.offset;
									code2P->m_jump.offset = code1P->m_jump.offset;
									code1P->m_jump.offset = s4;
									if (code1P != start_tableP) {
										--code1P;
										--code2P;
										continue;
								}	}
								++code1P;
								++code2P;
							}
						} else {
							for (i = 0; i < pairs; ++i) {
								image_at2P += sizeof(u4T);				// match
								image_at2P = getU4(image_at2P, &u4);	// jump
							}	
							pcodeP  += pairs;
						}
						break;
					}
					case WIDE:					// wide
						wide_index = 1;
						--pcodeP;				// Don't include in the assembler
						continue;
					default:
						break;
					}
				}
				assert(image_at2P == endCodeP);
				memoryP = (u1T *) pcodeP;
				if (load) {
					last_lineno = -1;
					functionP->m_code_count = lth = pcodeP - functionP->m_codeP;
					if (g_max_function_lth < lth) {
						g_max_function_lth = lth;
					}
					for (code1P = functionP->m_codeP; code1P < pcodeP; ++code1P) {
						switch (code1P->m_pcode) {
						case IFEQ:
						case IFNE:
						case IFLT:
						case IFGE:
						case IFGT:
						case IFLE:
						case IF_ICMPEQ:
						case IF_ICMPNE:
						case IF_ICMPLT:
						case IF_ICMPGE:
						case IF_ICMPGT:
						case IF_ICMPLE:
						case IF_ACMPEQ:
						case IF_ACMPNE:
						case IFNULL:
						case IFNOTNULL:
						case GOTO:
						case GOTO_W:
						case JSR:
						case JSR_W:
						case LOOKUPSWITCH:
						case TABLESWITCH:
						case TRY_START:
						case DEFAULT:
							pc = code1P->m_pc + code1P->m_jump.offset;
							if (code1P->m_jump.offset > 0) {
								for (code2P = code1P; ; ++code2P) {
									if (pcodeP <= code2P) {
										assert(0);
										code1P->m_jump.P = 0;
										break;
									}
									if (code2P->m_pc == pc) {
										code1P->m_jump.P = code2P;
										break;
								}	}
							} else {
								for (code2P = functionP->m_codeP; ; ++code2P) {
									if (code1P < code2P) {
										assert(0);
										code1P->m_jump.P = 0;
										break;
									}
									if (code2P->m_pc == pc) {
										if (code1P == code2P) {
											code1P->m_jump.P = 0;
										} else {
											code1P->m_jump.P = code2P;
										}
										break;
								}	}
							}
							break;
						default:
							code1P->m_jump.P = 0;
						}
						if (g_match_lines) {
							if (last_lineno == (int) code1P->m_source_lineno) {
								continue;
							}
							last_lineno = code1P->m_source_lineno;
						}
						hash_pcode(code1P);
				}	}
			}
		}
	}

	// Classfile attributes follow

	image_atP = getU2(image_atP, &attributes_count);
	for (j = 0; j < attributes_count; ++j) {
		image_atP  = getU2(image_atP, &attribute_name_index);
		AssertIndex(attribute_name_index);
		image_atP  = getU4(image_atP, &attribute_length);
		nameP      = get_Utf8(attribute_name_index);
		image_at1P = image_atP;
		image_atP += attribute_length;

		if (load) {
			if (!unicodecmp(nameP, synthetic)) {
				assert(attribute_length == 0);
				m_flags |= SyntheticX;
				continue;
			}
			if (!unicodecmp(nameP, deprecated)) {
				assert(attribute_length == 0);
				m_flags |= DeprecatedX;
				continue;
			}
			if (!unicodecmp(nameP, sourceFile)) {
				assert(attribute_length == 2);
				if (m_source_name) {
					fprintf(stderr, "Multiple source declarations in %s\n", m_filenameP);
					error_exit(1);
				}
				getU2(image_at1P, &m_source_name);
				AssertIndex(m_source_name);
				continue;
			}
			if (!unicodecmp(nameP, enclosingMethod)) {
				assert(attribute_length == 4);
				image_at1P = getU2(image_at1P, &m_enclosingMethod.m_class_index);
				AssertIndex(m_enclosingMethod.m_class_index);
				image_at1P = getU2(image_at1P, &m_enclosingMethod.m_method_index);
				AssertIndex(m_enclosingMethod.m_method_index);
				continue;
			}
		}
		if (!unicodecmp(nameP, innerClasses)) {
			innerclassT	*innerclassP;
			u2T			innerclasses_count;
			
			image_at1P           = getU2(image_at1P, &innerclasses_count);
			m_innerclasses_count = innerclasses_count;
			m_innerclassesP      = innerclassP = (innerclassT *) Align(memoryP);
			if (load) {
				for (i = 0; i < innerclasses_count; ++i) {
					image_at1P = getU2(image_at1P, &innerclassP->m_inner_class_info_index);
					AssertIndex(innerclassP->m_inner_class_info_index);
					image_at1P = getU2(image_at1P, &innerclassP->m_outer_class_info_index);
					AssertIndex(innerclassP->m_outer_class_info_index);
					image_at1P = getU2(image_at1P, &innerclassP->m_inner_class_name_index);
					AssertIndex(innerclassP->m_inner_class_name_index);
					image_at1P = getU2(image_at1P, &innerclassP->m_inner_class_access_flags);
					++innerclassP;
				}
				assert (image_at1P == image_atP);
			} else {
				innerclassP += innerclasses_count;
			}
			memoryP = (u1T *) innerclassP;
			continue;
		}
	}
	
	assert((image_atP - g_imageP) == g_image_size);
	if (!load) {
		g_memory = (memoryP - (u1T *) 0) + Csort::g_labels_count * sizeof(codeT);
		g_extra  = extraP  - (u1T *) 0;
		g_memory_startP = (u1T *) Xmalloc(g_memory);
		if (!g_memory_startP) {
			outofmemory();
		}
		g_extra_startP  = (u1T *) Xmalloc(g_extra);
		if (!g_extra_startP) {
			outofmemory();
		}
		Csort::sort();
	} else {
		assert((memoryP - g_memory_startP) == g_memory);
		assert((extraP  - g_extra_startP)  == g_extra);
	}
	return(0);
fail:
	return(-1);
}

#ifdef WIN32

#include <process.h>

static int
call_java(char *pathP)
{
	char		*sourceP, *P, *P1;
	int			ret;
	int			i;

	sourceP = pathP;
	P       = strrchr(sourceP, '/');
	P1      = strchr(sourceP, '\\');
	if (P1 > P) {
		P = P1;
	}
	if (!P) {
		P = sourceP;
	}
	P = strchr(P, '$');
	if (P) {
		int	lth = P-pathP;

		strncpy(g_filename, pathP, lth);
		strcpy(g_filename+lth, ".java");
		sourceP = g_filename;
	}	
	g_java[4] = sourceP;

	if (g_verbose) {
		for (i = 0; (P = g_java[i]); ++i) {
			fputs(P, stderr);
			fputc(' ', stderr);
		}
		fputc('\n', stderr);
	}

	ret = _spawnvpe(_P_WAIT, g_java[0], g_java+1, g_envPP);
	if (!ret) {
		++g_compiled_ok;
	} else {
		++g_compiled_bad;
		if (!g_verbose) {
			for (i = 0; P = g_java[i]; ++i) {
				fputs(P, stderr);
				fputc(' ', stderr);
		}	}
		fprintf(stderr, "returned %d\n", ret);
	} 
	return(ret);
}
#else
#include <sys/wait.h>

static int
call_java(char *pathP)
{
	char		*sourceP, *P;
	int			ret, ret1, pid, status;
	int			i;

	sourceP = pathP;
	P       = strrchr(sourceP, '/');
	if (!P) {
		P = sourceP;
	}
	P       = strchr(P, '$');
	if (P) {
		int	lth = P-pathP;

		sourceP = (char *) alloca(lth+6);
		strncpy(sourceP, pathP, lth);
		strcpy(sourceP+lth, ".java");
	}	
	g_java[4] = sourceP;

	if (g_verbose) {
		for (i = 0; (P = g_java[i]); ++i) {
			fputs(P, stderr);
			fputc(' ', stderr);
		}
		fputc('\n', stderr);
	}

	ret  = fork();
	ret1 = -2;

	switch (ret) {
	case 0:
		execvp(g_java[0], g_java);
		fprintf(stderr,"Return from execve %s\n", g_java[0]);
		error_exit(1);
	case -1:
		fprintf(stderr,"Fork failed\n");
		break;
	default:
		pid = wait(&status);
		if (pid != ret) {
			fprintf(stderr,"Wait for %s to finish failed\n", g_java[0]);
		} else if (!WIFEXITED(status)) {
			fprintf(stderr,"%s did not terminate normally\n", g_java[0]);
		} else {
			ret1 = WEXITSTATUS(status);
	}	}
	if (!ret1) {
		++g_compiled_ok;
	} else {
		++g_compiled_bad;
		if (!g_verbose) {
			for (i = 0; (P = g_java[i]); ++i) {
				fputs(P, stderr);
				fputc(' ', stderr);
		}	}
		fprintf(stderr, "returned %d\n", ret1);
	} 
	return(ret1);
}
#endif

int
readclassfile(char *pathP, Csource *classP)
{
	int				lth;
	Csource			*class1P;
	int				load, ret;

	if (classP) {
		if (access(pathP, F_OK)) {
			return(1);
	}	}

	lth = strlen(pathP);
	if (lth > 5 && !strcmp(pathP+lth-5,".java")) {
		if (!g_java[0]) {
			fprintf(stderr,"Use -j option if you want to compile %s\n", pathP);
			return(-1);
		}
		fputs("Compiled ", stdoutF);
		fputs(pathP, stdoutF);
		ret = call_java(pathP);
		if (ret) {
			fprintf(stdoutF, " failed ret=%d", ret);
		}
		fputs("<br>\n", stdoutF);
		
		strcpy(g_filename, pathP);
		strcpy(g_filename+lth-4,"class");
		pathP = g_filename;
	}

	ret = read_image(pathP);
	if (ret != 0) {
		if (ret == 1 && !classP) {
			fprintf(stderr, "Can't open %s\n", pathP);
		}
		goto done;
	}

	++g_read;
	fputs("Read ", stdoutF);
	fputs(pathP, stdoutF);
	fputs("\n<br>\n", stdoutF);

	class1P = classP;
	if (!class1P) {
		class1P = new Csource();
	}
	class1P->m_flags |= ProcessedX;

	for (load = 0; load < 2; ++load) {
		ret = class1P->load_constant_pool(pathP, load);
		if (ret != 0) {
			goto done;
	}	}

	for (load = 0; load < 2; ++load) {
		ret = class1P->load_tables(load);
		if (ret != 0) {
			goto done;
	}	}

	if (g_verbose) {
		if (classP) {
			fputs("Dynamically processed", stderr);
		} else {
			fputs("Processed", stderr);
		}
		fprintf(stderr, " \"%s\"\n", pathP);
	}
done:
	return(ret);
}

void
pseudo_source(void)
{
	int			i, ret, skip_compile;
	Csource		*classP;
	const char	*classpathP;
	int			repeat;
	char		path[2048];
	const char	*P;
	char		*pathP;
	u2T			*P2;
	char		c;

	struct stat	source_stat;
	struct stat	assembler_stat;

	classpathP = g_classpathP;
	if (!classpathP) {
		classpathP = ".";
	}
	do {
		repeat = 0;
		for (i = 0; i < HASHSIZE; ++i) {
			for (classP = Csource::g_hash[i]; classP; classP = classP->m_hashP) {
				if (!(classP->m_flags & ProcessedX)) {
					pathP = path;
					for (P = classpathP;; ++P) {
						switch (c = *P) {
						case 0:
						case ';':
						case ',':
							*pathP++ = '/';
							break;
						default:
							*pathP++ = c;
							continue;
						}
						for (P2 = classP->m_nameP; (*pathP++ = (u1T) *P2++); );

						--pathP;
						ret          = 1;
						skip_compile = g_skip_compile;
						if (g_java[0]) {
							if (skip_compile) {
								strcpy(pathP, ".class");
								if (stat(path, &assembler_stat)) {
									skip_compile = 0;
							}	}
							strcpy(pathP, ".java");
							if (access(path, F_OK)) {
								skip_compile = 1;
							} else if (skip_compile) {
								if (stat(path, &source_stat) ||
									source_stat.st_mtime >= assembler_stat.st_mtime) {
									skip_compile = 0;
							}	}
							if (!skip_compile) {
								ret = readclassfile(path, classP);
						}	}
						if (ret == 1) {
							strcpy(pathP, ".class");
							ret = readclassfile(path, classP);
						}
						switch (ret) {
						case 0:
							// Could read
							repeat = 1;
							break;
						case 1:
							if (c && P[1]) {
								// Wrong item on class path used; move to next item on class path
								pathP = path;
								continue;
							}
							fputs("<font color=red>Can't locate class ", stdoutF);
							dumpUnicode(classP->m_nameP);
							fputs("</font>\n<br>\n", stdoutF);
							break;
						}
						break;
				}	}
		}	}
	} while (repeat);
}


