#include "typedef.h"

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>

#if defined(WIN32)
#include <io.h>
#include <direct.h>

#define open _open
#define read _read
#define close _close
#define alloca _alloca
#define getcwd _getcwd
#define lseek _lseek
#else
#include <wchar.h>
#include <errno.h>
#include <unistd.h>
#include <alloca.h>

#define _O_BINARY 0
#endif

#include "xmalloc.h"
#include "util.h"
#include "object.h"
#include "directory.h"
#include "code.h"
#include "source.h"
#include "function.h"
#include "opcode.h"

Csource	*Csource::g_hash[HASHSIZE] = {0};
Csource	*Csource::g_headP          = 0;
Csource	**Csource::g_tailPP        = &g_headP;

extern int g_verbose;

extern FILE *stdoutF;

/* static */ u4T
Csource::hash(const u2T *nameP)
{
	int		ret  = 0;
	u2T		c;
	const	u2T *P;

	for (P = nameP; (c = *P) ; ++P) {
		ret  += c;
		ret <<= 1;
		if (ret & HASHSIZE) {
			ret |= 1;
	}	}
	return(ret & (HASHSIZE-1));
}

static int	g_number = 0;

Csource::Csource(void)
{
	memset(this, 0, sizeof(Csource));
	m_number = g_number++;
}

Csource::Csource(const u2T *nameP)
{
	memset(this, 0, sizeof(Csource));
	m_number = g_number++;
	m_nameP    = unicodedup(nameP);
}

void
Csource::showline(int lineno) const
{
	if (lineno > 0 && lineno <= m_source_lines) {
		char	*atP = m_sourcePP[lineno-1];
		char	c;

		fprintf(stdoutF, "<i>%d: ", lineno);
		while ((c = *atP++) != '\n' && c) {
			switch (c) {
			case '\r':
				continue;
			case '>':
				fputs("&gt;", stdoutF);
				continue;
			case '<':
				fputs("&lt;", stdoutF);
				continue;
			case '&':
				fputs("&amp;", stdoutF);
				continue;
			case '\t':
				c = ' ';
			}
			fputc(c, stdoutF);
		}
		fputs("</i>", stdoutF);
	}
}

void
Csource::chain(Csource *classP)
{
	u2T				*nameP;
	unsigned int	index;
	Csource			*class1P;
	Csource			**classPP;

	nameP = classP->m_nameP;
	index = hash(nameP);

	// Remove any earlier created reference to this class if we have now independently read it
	// as result of any explicit instruction to read it.
	for (classPP = g_hash + index; (class1P = *classPP); classPP = &(class1P->m_hashP)) {
		if (!unicodecmp(class1P->m_nameP, nameP)) {
			*classPP = class1P->m_hashP;
			delete class1P;
			break;
	}	}
	classP->m_hashP = g_hash[index];
	g_hash[index]   = classP;
}

// This is called directly when the class name is not mangled

Csource *
Csource::locate1(u2T *nameP)
{
	unsigned int	index;
	Csource			*classP;
	Csource			**classPP;

	index = hash(nameP);
	for (classPP = g_hash + index; ; classPP = &(classP->m_hashP)) {

		if (!(classP = *classPP)) {
			classP = new Csource(nameP);
			if (!classP) {
				outofmemory();
			}
			*classPP = classP;	// Add to hash list
			break;
		}

		if (!unicodecmp(classP->m_nameP, nameP)) {
			break;
	}	}
	return(classP);
}

// This is called when the class name is mangled

Csource *
Csource::locate(u2T *nameP)
{
	Csource			*classP;
	u2T				*P;
	u2T				c;

	for (; *nameP == '['; ++nameP); // ignore array subscripts on class
	if (*nameP == 'L') {
		++nameP;	// Advance nameP so avoids 'L' at start
	} else {
		// Not a class name but a primative data type
		if (!nameP[1]) {
			switch(*nameP) {
			case 'B':
			case 'C':
			case 'D':
			case 'F':
			case 'I':
			case 'J':
			case 'S':
			case 'Z':
				return(0);
	}	}	}

	for (P = nameP; (c = *P); ++P) {
		if (c == ';') {
			break;
	}	}
	*P = 0;		// Remove trailing ';'
	classP = locate1(nameP);
	*P = c; // Put the name back the way it was
	return(classP);
}

u2T *
Csource::get_Utf8(int index) const
{
	cp_infoT	*cpP = m_constant_poolP + index;
	u2T			*stringP;

	assert(index > 0 && index < m_constant_pool_count);
	assert(cpP->tag == CONSTANT_Utf8);
	stringP = cpP->u.stringP;
	return(stringP);
}

void
Csource::load_source(void)
{
	// Can't use L".java" because wchar_t is 4 bytes long on unix
	static u2T		java_string[] = {'.','j','a','v','a',0};
	Csource			*classP;
	char			*filenameP;
	u2T				*P, *P1;
	int				lth, size, lineno;
	int				fileno;
	char			**sourcePP;
	char			*source_codeP, *atP, *eolP;
	int				ret;

	classP = 0;
	if (m_source_name) {
		u2T	source[2048];

		P  = source;
		P1 = U2rchr(m_nameP, '/');
		if (P1) {
			lth = (P1 - m_nameP) + 1;
			assert(lth < 256);
			U2ncpy(P, m_nameP, lth);
			P += lth;
		}
		P1  = get_Utf8(m_source_name);
		lth = U2len(P1);
		if (lth > 5 && !U2cmp(P1 + lth - 5, java_string)) {
			lth -= 5;
		}
		U2ncpy(P, P1, lth);
		P += lth;
		*P = 0;
		if (U2cmp(m_nameP, source)) {
			classP = locate1((u2T *) source);
		}
	} else { 
		P = U2chr(m_nameP, (u2T) '$');
		if (P) {
			*P = 0;
			classP = locate1(m_nameP);
			*P = '$';
	}	}

	if (classP) {
		if (!classP->m_sourcePP) {
			classP->load_source();
		}
		m_sourcePP     = classP->m_sourcePP;
		m_source_lines = classP->m_source_lines;
		return;
	}

	if ((filenameP = m_filenameP)) {
		lth = strlen(filenameP);
		if (lth > 6 && !strcmp(filenameP+lth-6, ".class")) {
			strcpy(filenameP+lth-5, "java");
			fileno = open(filenameP, O_RDONLY | _O_BINARY, 0);
			strcpy(filenameP+lth-5, "class");
			if (fileno != -1) {
				size = lseek(fileno, 0L, SEEK_END);
				if (size > 0) {
					if (lseek(fileno, 0L, SEEK_SET) < 0) {
						ret = -1;
					} else {
						source_codeP = (char *) Xmalloc(size + 2);
						if (!source_codeP) {
							outofmemory();
						}
						ret = read(fileno, source_codeP, size);
					}
					close(fileno);
					if (ret == size) {
						atP         = source_codeP;
						if (size == 0 || atP[size-1] != '\n') {
							atP[size]   = '\n';
							atP[size+1] = 0;
						} else {
							atP[size]   = 0;
						}

						lineno      = 0;
						for (; (eolP = strchr(atP, '\n')); atP = eolP + 1) {
							++lineno;
						}
						m_sourcePP     = sourcePP = (char **) Xmalloc((lineno+1) * sizeof(char *));
						if (!sourcePP) {
							outofmemory();
						}
						m_source_lines = lineno;

						lineno         = 0;
						for (atP = source_codeP; (eolP = strchr(atP, '\n')); atP = eolP + 1) {
							sourcePP[lineno++] = atP;
							*eolP              = 0;
						}
						sourcePP[lineno]       = atP;
						return;
	}	}	}	}	}
	m_sourcePP = ((char **) -1);
}

void
Csource::dump_Utf8(int index) const
{
	dumpUnicode(get_Utf8(index));
}

u2T *
Csource::get_classname(int index) const
{
	cp_infoT	*cpP = m_constant_poolP + index;

	assert(index > 0 && index < m_constant_pool_count);
	assert(cpP->tag == CONSTANT_Class);
	// N.B. Even though the spec says that there
	//      is an indirection to the string via
	//		cpP->u.ref.index1 we've already removed
	//		it by the time we get here.
	return(cpP->u.stringP);
}

void
Csource::dump_descriptor(const u2T *descriptorP, Cfunction *functionP, int isStatic) const
{
	const	char *P;
	u2T		c, c1;
	int		dimension = 0;
	int		parameter = 0;
	int		offset;
	
//	dumpUnicode(descriptorP);
//	fputs("->", stdoutF);

	for (;;) {
		switch (c = *descriptorP++) {
		case 0:
			return;
		case '[':
			++dimension;
			continue;
			break;
		case 'L':
			if (parameter > 1) {
				fputs(", ", stdoutF);
			}
			for (; (c1 = *descriptorP) && c1 != ';'; ++descriptorP) {
				if (c1 == '/') {
					c1 = '.';
				}
				fputc(c1, stdoutF);
			}
			goto show_array;
		case '(':
			fputc('(', stdoutF);
			parameter = 1;
			continue;
		case ')':
			fputs(") returns ", stdoutF);
			parameter = 0;
			continue;
		case 'B':
			P = "byte";
			break;
		case 'C':
			P = "char";
			break;
		case 'D':
			P = "double";
			break;
		case 'F':
			P = "float";
			break;
		case 'I':
			P = "int";
			break;
		case 'J':
			P = "long";
			break;
		case 'S':
			P = "short";
			break;
		case 'Z':
			P = "boolean";
			break;
		case 'V':
			P = "void";
			break;
		case ';':
			continue;
		default:
			assert(0);
		}
		if (parameter > 1) {
			fputs(", ", stdoutF);
		}
		fputs(P, stdoutF);
show_array:
		for (; dimension > 0; --dimension) {
			fputs("[]", stdoutF);
		}
		if (parameter) {
			if (functionP) {
				localvariableT	*localVariablesP = functionP->m_localvariablesP;
				localvariableT	*localVariableP;

				if ((localVariableP = localVariablesP)) {
					localvariableT	*endP;

					offset = parameter;
					if (isStatic) {
						--offset;
					}

					for (endP = localVariableP + functionP->m_localvariables_count; localVariableP < endP; ++localVariableP) {
						if (localVariableP->m_index == offset) {
							fputc(' ', stdoutF);
							dumpUnicode(localVariableP->m_nameP);
							break;
				}	}	}
			}
			++parameter;
		}
	}	
	dumpUnicode(descriptorP);
	return;
}

static void
dump_constant(cp_infoT *cpP)
{
	switch (cpP->tag) {
	case CONSTANT_Integer:
		fprintf(stdoutF, "/* Integer */ %d", cpP->u.ival);
		break;
	case CONSTANT_Float:
		fprintf(stdoutF, "/* Float */ %f", cpP->u.fval);
		break;
	case CONSTANT_Long:
		fprintf(stdoutF, "/* Long */ %lld", cpP->u.lval);
		break;
	case CONSTANT_Double:
		fprintf(stdoutF, "/* Double */ %lf", cpP->u.dval);
		break;
	case CONSTANT_String:
		fputc('"', stdoutF);
		// N.B. Even though the spec says that there
		//      is an indirection to the string via
		//		cpP->u.ref.index1 we've already removed
		//		it by the time we get here.
		dumpUnicode(cpP->u.stringP);
		fputc('"', stdoutF);
		break;
	case CONSTANT_Class:
		// N.B. Even though the spec says that there
		//      is an indirection to the string via
		//		cpP->u.ref.index1 we've already removed
		//		it by the time we get here.

		fputs("/* Class */ ", stdoutF);
		dumpUnicode(cpP->u.stringP);
		break;
	default:
		assert(0);
}	}

void
Csource::dump_pcode(Cfunction *functionP, codeT *pcodeP) const
{
	static const char *array_type[] = 
	{0,0,0,0,
	"boolean", "char", "float", "double", "byte", "short", "int", "long"};

	// Can't use L"finally" because wchar_t is 4 bytes long on unix
	static u2T		finally_string[] = {'f','i','n','a','l','l','y',0};

	const ocodeT	*opcodeP;
	int				code;
	u4T				u4;

	fprintf(stdoutF, "%u[%u] ", (int) pcodeP->m_source_lineno, (int) pcodeP->m_pc);
	code    = pcodeP->m_pcode;
	opcodeP = opcodes + code;
	fprintf(stdoutF, "%s ", opcodeP->nameP);
	u4    = pcodeP->m_u4;

	// Pushing Constants
	// http://sunsite.ee/java/vmspec/vmspec-14.html#HEADING14-0

	switch (code) {
	case BIPUSH:
	case SIPUSH:
		fprintf(stdoutF, "%d", pcodeP->m_s4);
		break;
	case ILOAD:
	case ILOAD_0:
	case ILOAD_1:
	case ILOAD_2:
	case ILOAD_3:
	case LLOAD:
	case LLOAD_0:
	case LLOAD_1:
	case LLOAD_2:
	case LLOAD_3:
	case FLOAD:
	case FLOAD_0:
	case FLOAD_1:
	case FLOAD_2:
	case FLOAD_3:
	case DLOAD:
	case DLOAD_0:
	case DLOAD_1:
	case DLOAD_2:
	case DLOAD_3:
	case ALOAD:
	case ALOAD_0:
	case ALOAD_1:
	case ALOAD_2:
	case ALOAD_3:

	case ISTORE:
	case ISTORE_0:
	case ISTORE_1:
	case ISTORE_2:
	case ISTORE_3:
	case LSTORE:
	case LSTORE_0:
	case LSTORE_1:
	case LSTORE_2:
	case LSTORE_3:
	case FSTORE:
	case FSTORE_0:
	case FSTORE_1:
	case FSTORE_2:
	case FSTORE_3:
	case DSTORE:
	case DSTORE_0:
	case DSTORE_1:
	case DSTORE_2:
	case DSTORE_3:
	case ASTORE:
	case ASTORE_0:
	case ASTORE_1:
	case ASTORE_2:
	case ASTORE_3:
	{
		varT *varP;
		
		varP = (varT *) pcodeP->m_tableP;
		if (varP->m_nameP) {
			dump_descriptor(varP->m_descriptorP, 0, 0);
			fputc(' ' , stdoutF);
			dumpUnicode(varP->m_nameP);
		} else {
			fprintf(stdoutF, "%d", varP->m_number);
		}
		break;
	}
	case IINC:
	{
		iincT	*varP;

		varP = (iincT *) pcodeP->m_tableP;
		if (varP->m_nameP) {
			dump_descriptor(varP->m_descriptorP, 0, 0);
			fputc(' ' , stdoutF);
			dumpUnicode(varP->m_nameP);
		} else {
			fprintf(stdoutF, "%d", varP->m_number);
		}
		fprintf(stdoutF, " %d", varP->m_s4);
		break;
	}
	case LDC:
	case LDC_W:
	case LDC2_W:
		dump_constant((cp_infoT *) pcodeP->m_tableP);
		break;
	case GETSTATIC:
	case PUTSTATIC:
	case GETFIELD:
	case PUTFIELD:
	{
		const fieldrefT	*fieldrefP = (const fieldrefT *) pcodeP->m_tableP;

		dump_descriptor(fieldrefP->m_descriptorP, 0, 0);
		fputc(' ', stdoutF);
		dump_classname(fieldrefP->m_classnameP);
		fputc('.', stdoutF);
		dumpUnicode(fieldrefP->m_nameP);
		break;
	}
	case INVOKEVIRTUAL:
	case INVOKESPECIAL:
	case INVOKESTATIC:
	{
		const methodrefT	*methodrefP = (const methodrefT *) pcodeP->m_tableP;
	
		dump_classname(methodrefP->m_classnameP);
		fputc('.', stdoutF);
		dumpUnicode(methodrefP->m_nameP);
		dump_descriptor(methodrefP->m_descriptorP, 0, 0);
		break;
	}
	case INVOKEINTERFACE:
	{
		const interfacerefT	*methodrefP = (const interfacerefT *) pcodeP->m_tableP;

		dump_classname(methodrefP->m_classnameP);
		fputc('.', stdoutF);
		dumpUnicode(methodrefP->m_nameP);
		dump_descriptor(methodrefP->m_descriptorP, 0, 0);
		fputc(' ', stdoutF);
		fprintf(stdoutF, "%d\t", (int) methodrefP->m_args);
		break;
	}
	// Managing arrays
	// http://sunsite.ee/java/vmspec/vmspec-18.html#HEADING18-0

	case NEW:
	case ANEWARRAY:
	case CHECKCAST:
	case INSTANCEOF:
	{
		const classrefT	*classrefP = (const classrefT *) pcodeP->m_tableP;
		dump_classname(classrefP->m_classnameP);
		break;
	}
	case NEWARRAY:	// newarray
		if (u4 >= 4 && u4 <= 11) {
			fprintf(stdoutF, "%s", array_type[u4]);
		} else {
			fprintf(stdoutF, "%d", u4);
		}
		break;
	case MULTIANEWARRAY:
	{
		const arrayrefT	*arrayrefP = (const arrayrefT *) pcodeP->m_tableP;
		dump_classname(arrayrefP->m_classnameP);
		fprintf(stdoutF, " X %d", arrayrefP->m_dimensions);
		break;
	}
	case IFEQ:
	case IFNE:
	case IFLT:
	case IFGE:
	case IFGT:
	case IFLE:
	case IF_ICMPEQ:
	case IF_ICMPNE:
	case IF_ICMPLT:
	case IF_ICMPGE:
	case IF_ICMPGT:
	case IF_ICMPLE:
	case IF_ACMPEQ:
	case IF_ACMPNE:
	case IFNULL:
	case IFNOTNULL:
	case GOTO:
	case GOTO_W:
	case JSR:
	case JSR_W:
	case TABLESWITCH:
	case LOOKUPSWITCH:
	case DEFAULT:
		if (pcodeP->m_jump.P) {
			fprintf(stdoutF, "->%d", (int) pcodeP->m_jump.P->m_pc);
		} else {
			fprintf(stdoutF, "<->%d", (int) pcodeP->m_pc);
		}
		break;
	case TRY_START:
	{
		exceptionT	*exceptionP  = (exceptionT *) pcodeP->m_tableP;
		u2T			*catch_typeP = exceptionP->m_catch_typeP;

		if (!catch_typeP) {
			catch_typeP = finally_string;
		}
		fprintf(stdoutF, "[- %d] ", exceptionP->m_end_pc);
		dumpUnicode(catch_typeP);
		fprintf(stdoutF, " ->%d", pcodeP->m_jump.P->m_pc);
		break;
	}
	case TRY_END:
	{
		exceptionT	*exceptionP  = (exceptionT *) pcodeP->m_tableP;
		u2T			*catch_typeP = exceptionP->m_catch_typeP;

		if (!catch_typeP) {
			catch_typeP = finally_string;
		}
		fprintf(stdoutF, "[%d -] ", exceptionP->m_start_pc);
		dumpUnicode(catch_typeP);
		fprintf(stdoutF, " ->%d", exceptionP->m_handler_pc);
		break;
	}
	case TRY_LABEL: 
	{
		exceptionT	*exceptionP  = (exceptionT *) pcodeP->m_tableP;
		u2T			*catch_typeP = exceptionP->m_catch_typeP;

		if (!catch_typeP) {
			catch_typeP = finally_string;
		}
		fprintf(stdoutF, "[%d, %d] ", exceptionP->m_start_pc, exceptionP->m_end_pc);
		dumpUnicode(catch_typeP);
		break;
	} }
}

void
Csource::emit_ta(void) const
{
	extern FILE	*taF;
	extern FILE	*tempF;

	Cdirectory	*parentP;
	u2T			*nameP, *P, *P1;

	parentP = m_parentP;

	if (parentP) {
		// Making the count 2 causes automatic directory to treat as must
		// not be collapsed
		parentP->has_clones(2);
	}
	fprintf(taF, "$INSTANCE S%p S\n", this);
	if (parentP) {
		fprintf(taF, "contain D%p S%p\n", parentP, this);
	}
	nameP = m_nameP;
	P     = 0;
	for (P1 = nameP; ; ++P1) {
		switch (*P1) {
		case 0:
			break;
		case '/':
			P = P1;
		default:
			continue;
		}
		break;
	}

	if (P && P[1]) {
		nameP = P+1;
	}
	fprintf(tempF, "S%p {label=\"%S\" file=\"./%S\"}\n", this, (wchar_t *) nameP, (wchar_t *) nameP);
}
