package lsedit;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import java.applet.Applet;
import javax.swing.*;
import javax.swing.table.*;
import javax.swing.border.*;

public class EditAttribute extends JDialog { //Class definition

	// This is the class which maps to the data being presented/updated to lsedit variables

	protected class AttributeTableModel extends AbstractTableModel {
		
		LandscapeObject	m_object;
		JTable			m_table;

		public AttributeTableModel(LandscapeObject object)
		{
			m_object = object;
		}

		public void setJTable(JTable table)
		{
			m_table = table;
		}

		public int getRowCount()
		{
			return(m_object.getAttributeCount());
		}

		public int getColumnCount()
		{
			return(2);
		}

		public String getColumnName(int col)
		{
			if (col == 0) {
				return("Name");
			} else {
				return("Value");
			}
		}

		public Object getValueAt(int row, int col)
		{
			if (col == 0) {
				return(m_object.getAttributeNameAt(row));
			} 
			return(m_object.getAttributeValueAt(row));
		}

		public Class getColumnClass(int col) 
		{
			if (col == 0) {
				return(getValueAt(0,col).getClass());
			}
			Object	x = new Object();
			return(x.getClass());
		}

		public boolean isCellEditable(int row, int col)
		{
			if (col == 0) {
				return(m_object.canEditName(row));
			}
			return(m_object.canEditAttribute(row));
		}

		public void setValueAt(Object value, int row, int col)
		{
			if (col == 0) {
				if (m_object.setAttributeNameAt(row, value)) {
					// Order of rows may change
					m_table.revalidate();
					m_table.repaint();
				}
			} else {
				m_object.setAttributeValueAt(row, value);
			}
		}
	}
	
	// This is the class which decides how the table is drawn and edited

	protected class MyJTable extends JTable {

		protected class ColorRenderer extends JLabel implements TableCellRenderer {
			Border unselectedBorder = null;
			Border selectedBorder   = null;

			public ColorRenderer() 
			{
				super();
				setOpaque(true); //MUST do this for background to show up.
			}

			public Component getTableCellRendererComponent(JTable table, Object color, boolean isSelected, boolean hasFocus, int row, int column) 
			{
				setBackground((Color)color);
				if (isSelected) {
					if (selectedBorder == null) {
						selectedBorder = BorderFactory.createMatteBorder(2,5,2,5, table.getSelectionBackground());
					}
					setBorder(selectedBorder);
				} else {
					if (unselectedBorder == null) {
						unselectedBorder = BorderFactory.createMatteBorder(2,5,2,5, table.getBackground());
					}
					setBorder(unselectedBorder);
				}
				return this;
			}
		}

		/*
		 * This interface returns a button which is placed where the old value used to be and when fired brings up the
		 * actual editor to change the old value.
		 */

		class ColorEditor extends DefaultCellEditor {
			JFrame			m_frame;
			Color			m_currentColor;
			boolean			m_allowNull;
			JButton			m_button;

			public ColorEditor(JFrame frame, boolean allowNull) 
			{
				super(new JCheckBox()); //Unfortunately, the constructor expects a check box, combo box, or text field.

				m_frame     = frame;
				m_allowNull = allowNull;

				//First, set up the button that brings up the dialog.

				m_button =  new JButton() 
										{
											public void setText(String s) {
												//Button never shows text -- only color.
											}
										};
        
				m_button.setBackground(Color.white);
				m_button.setBorderPainted(false);
				m_button.setMargin(new Insets(0,0,0,0));

				editorComponent = m_button;
				setClickCountToStart(1); //This is usually 1 or 2.

				//Must do this so that editing stops when appropriate.
				m_button.addActionListener(new ActionListener() 
											 {
												public void actionPerformed(ActionEvent e) {
													fireEditingStopped();
												}
											 });

				//Here's the code that brings up the dialog.
				m_button.addActionListener(new ActionListener() 
											 {
												public void actionPerformed(ActionEvent e) 
												{
													m_button.setBackground(m_currentColor);
													m_currentColor = ColorChooser.create(m_frame, "Pick a color", m_currentColor, true, m_allowNull); 
												}
											 });

			}

			public Object getCellEditorValue() 
			{
				return m_currentColor;
			}

			public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) 
			{
				m_currentColor = (Color) value;
				return editorComponent;
			}
		}

		// Do something sensible to handle new lines in input string

		protected class TextRenderer extends JTextField implements TableCellRenderer {

			public TextRenderer() 
			{
				super();
			}

			public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) 
			{
				setText((String) value);
				return this;
			}
		}

		// Bring up a large box for editting multiple lines of text

		class TextEditor extends DefaultCellEditor {

			JFrame			m_frame;
			String			m_currentText;
			JButton			m_button;
			JTextArea		m_textArea;
			JDialog			m_dialog;

			public TextEditor(JFrame frame) 
			{
				super(new JCheckBox()); //Unfortunately, the constructor expects a check box, combo box, or text field.

				//First, set up the button that brings up the dialog.

				m_frame  = frame;
				m_button = new JButton(); 
        
				m_button.setBackground(Color.white);
				m_button.setBorderPainted(false);
				m_button.setMargin(new Insets(0,0,0,0));

				editorComponent = m_button;
				setClickCountToStart(1); //This is usually 1 or 2.

				//Must do this so that editing stops when appropriate.
				m_button.addActionListener(new ActionListener() 
											 {
												public void actionPerformed(ActionEvent e) {
													fireEditingStopped();
												}
											 });

				//Here's the code that brings up the dialog.
				m_button.addActionListener(new ActionListener() 
											 {
												public void actionPerformed(ActionEvent e) 
												{
													ActionListener	okListener;
													Container		contentPane;
													JScrollPane		scrollPane;
													Button			OK_button = new Button("OK");
													Button			Cancel_button = new Button("Cancel");

													m_textArea = new JTextArea(m_currentText);

													okListener   = new ActionListener() 
												                    {
																		public void actionPerformed(ActionEvent e) {
																			m_currentText = m_textArea.getText();
																			m_dialog.setVisible(false);
																		}
																	};

													m_dialog = new JDialog(m_frame, "Change Text", true);
													m_dialog.resize(438,369);

													contentPane = m_dialog.getContentPane();
													contentPane.setLayout(null);

													m_dialog.setForeground(new Color(0,0,0));
													m_dialog.setBackground(new Color(192,192,192));
													m_dialog.setFont(new Font("Dialog",0,12));

													scrollPane = new JScrollPane(m_textArea);
													scrollPane.setVisible(true);
													scrollPane.reshape(5,5,423,300);
													contentPane.add(scrollPane);
													OK_button = new Button("OK");
													OK_button.reshape(5, 310, 40, 20);
													contentPane.add(OK_button);
													OK_button.addActionListener(okListener); 
											 		m_dialog.setVisible(true);
												}
											 });

			}
			public Object getCellEditorValue() 
			{
				return m_currentText;
			}

			public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) 
			{
				m_currentText = (String) value;
				return editorComponent;
			}
		}

		// Bring up a large box for editting multiple lines of attribute value text

		class AviTextEditor extends TextEditor {

			AttributeValueItem	m_avi;
			String				m_oldText;
			
			public AviTextEditor(JFrame frame) 
			{
				super(frame); 
			}

			public Object getCellEditorValue() 
			{
				if (!m_oldText.equals(m_currentText)) {
					int	i;

					for (i = m_currentText.length()-1; i >= 0; --i) {
						if (!java.lang.Character.isWhitespace(m_currentText.charAt(i))) {
							break;
					}	}
					if (i < 0) {
						m_avi = null;
					} else {
						StringBufferInputStream stream = new StringBufferInputStream(m_currentText);
						LandscapeTokenStream	ts     = new LandscapeTokenStream(stream, "Attribute Editor");
						try {
							AttributeValueItem	avi;

							avi = ts.parseAttributeValueItem();
							if (avi != null) {
								m_avi = avi;
							} else {
								System.out.println("Can't parse '" + m_currentText + "' as attribute value");
							}
						} catch (Exception e) {
							System.out.println("Can't parse '" + m_currentText + "' as attribute value - exception thrown");
						}
				}	}
				return m_avi;
			}

			public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) 
			{
				m_avi         = (AttributeValueItem) value;
				if (value == null) {
					m_currentText = "";
				} else {
					m_currentText = value.toString();
				}
				m_oldText     = m_currentText;
				return editorComponent;
			}
		}

		// Ignore changes which do not produce double values

		class DoubleEditor extends DefaultCellEditor {

			Double			m_current;
			JTextField		m_textField;

			public DoubleEditor() 
			{
				super(new JCheckBox()); //Unfortunately, the constructor expects a check box, combo box, or text field.

				//First, set up the button that brings up the dialog.

				m_textField =  new JTextField(); 
        		editorComponent = m_textField;
				setClickCountToStart(1); //This is usually 1 or 2.

				//Must do this so that editing stops when appropriate.
				m_textField.addActionListener(new ActionListener() 
											 {
												public void actionPerformed(ActionEvent e) {
													fireEditingStopped();
												}
											 });

				m_textField.addActionListener(new ActionListener() 
											 {
												public void actionPerformed(ActionEvent e) 
												{
													Double	newValue;

													try {
														newValue  = new Double(m_textField.getText());
														m_current = newValue;
													} catch (Exception e1) {
														System.out.println("\"" + m_textField.getText() + "\" is not a double");
													}
												}
											 });

			}

			public Object getCellEditorValue() 
			{
				return m_current;
			}

			public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) 
			{
				m_current = (Double) value;
				m_textField.setText(m_current.toString());
				return editorComponent;
			}
		}

		class IntegerEditor extends DefaultCellEditor {

			Integer			m_current;
			JTextField		m_textField;

			public IntegerEditor() 
			{
				super(new JCheckBox()); //Unfortunately, the constructor expects a check box, combo box, or text field.

				//First, set up the button that brings up the dialog.

				m_textField =  new JTextField(); 
        		editorComponent = m_textField;
				setClickCountToStart(1); //This is usually 1 or 2.

				//Must do this so that editing stops when appropriate.
				m_textField.addActionListener(new ActionListener() 
											 {
												public void actionPerformed(ActionEvent e) {
													fireEditingStopped();
												}
											 });

				m_textField.addActionListener(new ActionListener() 
											 {
												public void actionPerformed(ActionEvent e) 
												{
													Integer	newValue;

													try {
														newValue  = new Integer(m_textField.getText());
														m_current = newValue;
													} catch (Exception e1) {
														System.out.println("\"" + m_textField.getText() + "\" is not an integer");
													}
												}
											 });

			}

			public Object getCellEditorValue() 
			{
				return m_current;
			}

			public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) 
			{
				m_current = (Integer) value;
				m_textField.setText(m_current.toString());
				return editorComponent;
			}
		}

		// Do something sensible to handle limited choice of styles

		protected class StyleRenderer extends JTextField implements TableCellRenderer {

			public StyleRenderer() 
			{
				super();
			}

			public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) 
			{
				int		style;
				String	description;

				style = ((Integer) value).intValue();

				if (style < 0 || style >= EntityClass.styleName.length) {
					description = "";
				} else {
					description = EntityClass.styleName[style];
				}
				setText(description);
				return this;
			}
		}

		class StyleEditor extends DefaultCellEditor {

			Integer			m_current;
			JComboBox		m_comboBox;

			public StyleEditor() 
			{
				super(new JCheckBox()); //Unfortunately, the constructor expects a check box, combo box, or text field.

				//First, set up the combo box
				
				int	i;

				m_comboBox =  new JComboBox(); 

				for (i = 0; i < EntityClass.styleName.length; ++i) {
					m_comboBox.addItem(EntityClass.styleName[i]);
				}
        		editorComponent = m_comboBox;
				setClickCountToStart(1); //This is usually 1 or 2.

				//Must do this so that editing stops when appropriate.
				m_comboBox.addActionListener(new ActionListener() 
											 {
												public void actionPerformed(ActionEvent e) {
													fireEditingStopped();
												}
											 });

				m_comboBox.addActionListener(new ActionListener() 
											 {
												public void actionPerformed(ActionEvent e) 
												{
													int		selected;

													selected = m_comboBox.getSelectedIndex();
													if (selected >= 0) {
														m_current = new Integer(selected);
													}
												}
											 });

			}

			public Object getCellEditorValue() 
			{
				return m_current;
			}

			public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) 
			{
				int	selected;

				m_current = (Integer) value;
				selected  = m_current.intValue();
				if (selected >= 0 && selected < m_comboBox.getItemCount()) {
					m_comboBox.setSelectedIndex(m_current.intValue());
				}
				return editorComponent;
			}
		}

		// Do something sensible to handle limited choice of styles

		protected class RelStyleRenderer extends JTextField implements TableCellRenderer {

			public RelStyleRenderer() 
			{
				super();
			}

			public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) 
			{
				int		style;
				String	description;

				style = ((Integer) value).intValue();

				if (style < 0 || style >= Util.lineStyleName.length) {
					description = "";
				} else {
					description = Util.lineStyleName[style];
				}
				setText(description);
				return this;
			}
		}

		class RelStyleEditor extends DefaultCellEditor {

			Integer			m_current;
			JComboBox		m_comboBox;

			public RelStyleEditor() 
			{
				super(new JCheckBox()); //Unfortunately, the constructor expects a check box, combo box, or text field.

				//First, set up the combo box
				
				int	i;

				m_comboBox =  new JComboBox(); 

				for (i = 0; i < Util.lineStyleName.length; ++i) {
					m_comboBox.addItem(Util.lineStyleName[i]);
				}
        		editorComponent = m_comboBox;
				setClickCountToStart(1); //This is usually 1 or 2.

				//Must do this so that editing stops when appropriate.
				m_comboBox.addActionListener(new ActionListener() 
											 {
												public void actionPerformed(ActionEvent e) {
													fireEditingStopped();
												}
											 });

				m_comboBox.addActionListener(new ActionListener() 
											 {
												public void actionPerformed(ActionEvent e) 
												{
													int		selected;

													selected = m_comboBox.getSelectedIndex();
													if (selected >= 0) {
														m_current = new Integer(selected);
													}
												}
											 });

			}

			public Object getCellEditorValue() 
			{
				return m_current;
			}

			public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) 
			{
				int	selected;

				m_current = (Integer) value;
				selected  = m_current.intValue();
				if (selected >= 0 && selected < m_comboBox.getItemCount()) {
					m_comboBox.setSelectedIndex(m_current.intValue());
				}
				return editorComponent;
			}
		}

		JFrame			m_frame;
		LandscapeObject	m_object;

		public MyJTable(JFrame frame, AbstractTableModel tableModel, LandscapeObject object)
		{
			super(tableModel);
			m_frame  = frame;
			m_object = object;
		}

		// Overload how cells are rendered

		public TableCellRenderer getCellRenderer(int row, int column)
		{
			Object				value;
			TableCellRenderer	ret;
			String				tip;

			ret = null;
			tip = null;
			if (convertColumnIndexToModel(column) == 1) {

				int type;

				value = dataModel.getValueAt(row, column);
				type  = m_object.getAttributeTypeAt(row);

				switch (type) {
				case Attribute.STRING_TYPE:
					if (row != 0) {
						tip = "Click to change string";
					} else {
						tip = "This value is fixed";
					}
					break;
				case Attribute.DOUBLE_TYPE:
					tip = "Click to change double";
					break;
				case Attribute.INT_TYPE:
					tip = "Click to change integer";
					break;
				case Attribute.COLOR_TYPE:
				case Attribute.COLOR_OR_NULL_TYPE:
					if (value != null) {
						ret = new ColorRenderer();
					}
					tip = "Click to change color";
					break;
				case Attribute.TEXT_TYPE:
					if (value != null) {
						ret = new TextRenderer();
					}
					tip = "Click to change text";
					break;
				case Attribute.STYLE_TYPE:
					if (value != null) {
						ret = new StyleRenderer();
					}
					tip = "Click to change entity style";
					break;
				case Attribute.REL_STYLE_TYPE:
					if (value != null) {
						ret = new RelStyleRenderer();
					}
					tip = "Click to change relation style";
					break;
				case Attribute.AVI_TYPE:
					tip = "Click to change attribute value";
					break;
				case Attribute.NULL_TYPE:
					tip = "First enter name to create new attribute";
					break;
				}
				if (ret == null) {
					ret = super.getCellRenderer(row, column);
					((JLabel) ret).setForeground(Color.black);
				}
			} else {
				Color	foreground;

				ret = super.getCellRenderer(row, column);
				if (row < m_object.getPrimaryAttributeCount()) {
					foreground = Color.blue;
					tip        = "This attribute name is fixed";
				} else {
					foreground = Color.red;
					if (row == m_object.getAttributeCount()-1) {
						tip    = "Click to add new attribute name";
					} else {
						tip    = "Click to change/delete attribute name";
				}	}
				((JLabel) ret).setForeground(foreground);
			}
			if (ret != null) {
				((JComponent) ret).setToolTipText(tip);
			}
			return(ret);
		}

		// Overload how cells are editted

		public TableCellEditor getCellEditor(int row, int column)
		{
			int		type;

			if (convertColumnIndexToModel(column) == 1) {
				type = m_object.getAttributeTypeAt(row);
				switch (type) {
				case Attribute.COLOR_TYPE:
					return(new ColorEditor(m_frame, false));
				case Attribute.COLOR_OR_NULL_TYPE:
					return(new ColorEditor(m_frame, true));
				case Attribute.TEXT_TYPE:
					return(new TextEditor(m_frame));
				case Attribute.DOUBLE_TYPE:
					return(new DoubleEditor());
				case Attribute.INT_TYPE:
					return(new IntegerEditor());
				case Attribute.STYLE_TYPE:
					return(new StyleEditor());
				case Attribute.REL_STYLE_TYPE:
					return(new RelStyleEditor());
				case Attribute.AVI_TYPE:
					return(new AviTextEditor(m_frame));
			}	}
			return(super.getCellEditor(row, column));
		}
	}
		
	protected EditAttribute(JFrame frame, LandscapeObject e) //Constructor
	{
		super(frame, "Edit Attributes of " + e.getLabel(),true); //false if non-modal

		Container	contentPane;
		JScrollPane	scrollPane;

		resize(438,369);

		contentPane = getContentPane();
		contentPane.setLayout(new BorderLayout());

		setForeground(new Color(0,0,0));
		setBackground(new Color(192,192,192));
		setFont(new Font("Dialog",0,12));

		AttributeTableModel	tableModel;
		MyJTable			table;

		tableModel = new AttributeTableModel(e);
		table      = new MyJTable(frame, tableModel, e);
		tableModel.setJTable(table);
		table.setVisible(true);
		scrollPane = new JScrollPane(table);

		scrollPane.setVisible(true);
		contentPane.add(scrollPane, BorderLayout.CENTER);
		repaint();
		setVisible(true);
	}

	private static JFrame findFrame(LandscapeViewerCore ls)
	{
		Container c = ls; 

		do { 
			c = c.getParent(); 
		} while (!(c instanceof JFrame)); 
		return((JFrame) c);
	}

	private static boolean sameColors(Color color1, Color color2)
	{
		if (color1 == color2) {
			return(true);
		}
		if (color1 == null || color2 == null) {
			return(false);
		}
		return(color1.equals(color2));
	}


	public static void Create(LandscapeViewerCore ls, EntityInstance e) 
	{
		new EditAttribute(findFrame(ls), e);
		ls.repaint(); 
	}

	public static void Create(LandscapeViewerCore ls, EntityClass e) 
	{
		Color	oldColorWhenOpen, oldColor, oldLabelColor, newColorWhenOpen, newColor, newLabelColor;
		int		oldStyle, newStyle;
		boolean	sameColorWhenOpen, sameColor, sameLabelColor, sameStyle;

		oldColorWhenOpen = e.getObjectColorWhenOpen();
		oldColor         = e.getObjectColor();
		oldLabelColor    = e.getLabelColor();
		oldStyle         = e.style;


		new EditAttribute(findFrame(ls), e);

		newColorWhenOpen = e.getObjectColorWhenOpen();
		newColor         = e.getObjectColor();
		newLabelColor    = e.getLabelColor();
		newStyle         = e.style;

		sameColorWhenOpen = sameColors(oldColorWhenOpen, newColorWhenOpen);
		sameColor         = sameColors(oldColor,         newColor);
		sameLabelColor    = sameColors(oldLabelColor,    newLabelColor);
		sameStyle         = (oldStyle == newStyle);

		if (!sameColorWhenOpen || !sameColor || !sameLabelColor || !sameStyle) {
			if (!sameColorWhenOpen) {
				ls.setEntitiesColorWhenOpen(e, newColorWhenOpen);
			}
			if (!sameColor) {
				ls.setEntitiesColor(e, newColor);
			}
			if (!sameLabelColor) {
				ls.setEntitiesLabelColor(e, newLabelColor);
			}
			if (!sameStyle) {
				ls.setEntitiesStyle(e, newStyle);
			}
			ls.repaint(); 
		} else {
			ls.repaintTabs();
		}
	}

	public static void Create(LandscapeViewerCore ls, RelationClass e) 
	{

		Color	oldColor, newColor;
		int		oldStyle, newStyle;

		oldStyle = e.style;
		oldColor = e.getObjectColor();


		new EditAttribute(findFrame(ls), e);

		newColor = e.getObjectColor();
		newStyle = e.style;
		if (!oldColor.equals(newColor) || oldStyle != newStyle) {
			if (!oldColor.equals(newColor)) {
				ls.setRelationColor(e, newColor);
			}
			if (oldStyle != newStyle) {
				ls.setRelationStyle(e, newStyle);
			}
			ls.repaint(); 
		} else {
			ls.repaintTabs();
		}
	}

} //End of class EditAttribute



