package lsedit;

import java.util.*;

// Base class for layout classes accessed by Landscape Editor
//
// For a layouter to be available in the Landscape Editor, an instance entry
// must appear in the layouterList array in class LayouterManager.java

abstract public class LandscapeLayouter 
{
	protected Diagram dg;

	// ----------------
	// Utility routines
	// ----------------

	// IMPORTANT:
	//
	// These routines are wrappers to the Landscape Editor
	// Use these wrappers to insure compatibility with future releases.

	// Returns the current EntityInstance which is the visible root.

	protected EntityInstance getRoot() 
	{
		return dg.getRoot();
	}

	// Returns a vector containing EntityInstance objects consisting
	// of the currently selected group. Returns null if no entities
	// are selected.

	protected Vector getGroup() 
	{
		return dg.getGroup();
	}

	// Returns the particular EntityInstance which is the selected key entity
	// of the group.

	protected EntityInstance getKeyEntity() 
	{
		return dg.getKeyEntity();
	}

	// Returns the EntityInstance which is parent container of passed entity

	protected EntityInstance getParent(EntityInstance e) 
	{
		return e.getParent();
	}

	// Returns a vector containing EntityInstance objects consisting
	// of the top level children of the passed EntityInstance. 
	// Returns null if the passed entity has no children.

	protected Vector getChildren(EntityInstance e) 
	{
		Vector v       = new Vector();
		Enumeration en = e.getChildren();

		while (en.hasMoreElements()) {
			v.addElement(en.nextElement());
		}
		return v;
	}

	protected Enumeration getChildrenIteration(EntityInstance e) 
	{
		return e.getChildren();
	}

	// Returns the current global layout of the passed entity.

	protected Layout getLayout(EntityInstance e) 
	{
		return e.getLayout();
	}

	// Sets the layout of the passed entity.

	protected void setLayout(EntityInstance e, Layout lyt) 
	{
		e.setLayout(lyt);
	}

	// Returns a vector of the RelationInstance edges to which the
	// passed entity is the source.

	protected Vector getEdgesToSuppliers(EntityInstance e) 
	{
		Vector		v  = new Vector();
		Enumeration en = e.srcRelationElements();

		while (en.hasMoreElements()) {
			v.addElement(en.nextElement());
		}
		return v;
	}

	protected Enumeration getEdgesToSuppliersIteration(EntityInstance e) 
	{
		return e.srcRelationElements();
	}

	protected Vector getEdgesAbstractedToSuppliers(EntityInstance e) 
	{
		Vector		v  = new Vector();
		Enumeration en = e.srcRelationAbstractedElements();

		while (en.hasMoreElements()) {
			v.addElement(en.nextElement());
		}
		return v;
	}

	protected Enumeration getEdgesAbstractedToSuppliersIteration(EntityInstance e)
	{
		return e.srcRelationAbstractedElements();
	}

	// Returns a vector of the RelationInstance edges to which the
	// passed entity is the destination.

	protected Vector getEdgesFromClients(EntityInstance e) 
	{
		Vector		v  = new Vector();
		Enumeration en = e.dstRelationElements();

		while (en.hasMoreElements()) {
			v.addElement(en.nextElement());
		}
		return v;
	}

	protected Enumeration getEdgesFromClientsIteration(EntityInstance e) 
	{
		return e.dstRelationElements();
	}

	protected Vector getEdgesAbstractedFromClients(EntityInstance e) 
	{
		Vector		v  = new Vector();
		Enumeration en = e.dstRelationAbstractedElements();

		while (en.hasMoreElements()) {
			v.addElement(en.nextElement());
		}
		return v;
	}

	protected Enumeration getEdgesAbstractedFromClientsIteration(EntityInstance e)
	{
		return e.dstRelationAbstractedElements();
	}

	// Returns a vector consisting of the EntityClass active in the Diagram.

	protected Vector getEntityClasses() 
	{
		Vector		v  = new Vector();
		Enumeration en = dg.enumEntityClasses();

		while (en.hasMoreElements()) {
			v.addElement(en.nextElement());
		}
		return v;
	}

	// Returns an Enumeration of the EntityClass active in the Diagram.

	protected Enumeration getEntityClassIteration() 
	{
		return dg.enumEntityClasses();
	}

	// Returns a vector consisting of the RelationClass active in the Diagram.

	protected Vector getRelationClasses() 
	{
		Vector		v  = new Vector();
		Enumeration en = dg.enumRelations();

		while (en.hasMoreElements()) {
			v.addElement(en.nextElement());
		}
		return v;
	}

	// Returns an Enumeration of the RelationClass active in the Diagram.

	protected Enumeration getRelationClassIteration() 
	{
		return dg.enumRelations();
	}

	protected void sortVectorByString(Vector v) 
	{
		Util.sortVector(v);
	}

	protected void sortVectorByString(Vector v, boolean ascendingFlag) 
	{
		Util.sortVector(v, ascendingFlag);
	}

	protected void beep() 
	{
		System.out.print("\007");
		System.out.flush();
	}	 

	// --------------
	// Public methods
	// --------------

	public LandscapeLayouter() 
	{
	}

	// Called by the system to set the current Diagram

	public final void setDiagram(Diagram dg) 
	{
		this.dg = dg;
	}

	// -----------------------
	// Abstract public methods
	// -----------------------

	// Called by system to obtain menu entry label for this layout routine

	abstract String getMenuLabel();

	// Called by the system when a layout is requested
	// Needs to be supplied by layout classes which derive from this
	// base class.
	//
	// Returns a status/error message string.

	abstract public String doLayout();
}

