package lsedit;

import java.util.*;

import java.io.*;



public class LandscapeTokenStream {


	public final static int SCHEME_TUPLE		= 0;
	public final static int SCHEME_ATTRIBUTE	= 1;
	public final static int FACT_TUPLE			= 2;
	public final static int FACT_ATTRIBUTE		= 3;
	public final static int INCLUDE_FILE		= 4; 

	public final static int ERROR				= -1;
	public final static int NONE				= -2; 
	public final static int EOF					= 99;

	protected final static String SCHEME_ID = "SCHEME";
	protected final static String FACT_ID	= "FACT";
	protected final static String TUPLE_ID	= "TUPLE";
	protected final static String ATTRIBUTE_ID = "ATTRIBUTE";
	protected final static String INCLUDE_ID = "INCLUDE"; 

	protected static boolean debug = false;	 // true outputs debug info 

	protected boolean fatalError = false;

	protected String fileName; 

	protected TA_StreamTokenizer ts;

	// push backed tokens 

	protected String token1, token2, includeFile;
	protected int ttype3; 



	public static void setDebugOn() {
		debug = true;
	}


	protected void debugOut(String msg) {
		if (debug) {
			System.out.println(msg); 
	}	}


	protected int nextToken() throws IOException {

		int ttype = ts.nextToken();

		if (debug) {
			switch(ttype)
			{
			case TA_StreamTokenizer.TT_EOF: 
				System.out.println("Token(EOF)");
				break; 
			case TA_StreamTokenizer.TT_WORD:
				System.out.println("WordToken(" + '"' + ts.sval + '"' + ")");
				break;
			case TA_StreamTokenizer.TT_NUMBER:
				System.out.println("NumberToken(#" + ts.nval + ")"); 
				break; 
			default:
				System.out.println("CharToken('" + ((char) ttype) + "')");
				break; 
		}	}
		return ttype; 
	}

	protected void skipRecord() throws IOException {
		int ttype;

		do {
			ttype =	 nextToken();
		} while (ttype != TA_StreamTokenizer.TT_EOF && ttype != '{');


		if (ttype != '{')
			return;

		do {
			ttype = nextToken();
		} while (ttype != TA_StreamTokenizer.TT_EOF && ttype != '}');
	}



	protected void pushBack() {
		ts.pushBack();
	}

	protected AttributeValueItem parseList() throws IOException {
		debugOut("parseList:");
		// Called when value field started with '('
		AttributeValueItem avi = null;
		AttributeValueItem tail = null; 

		for (;;) {
			switch(nextToken())
			{
			case ')':
				return avi;
			case '(':
				{
				// Nested list
				AttributeValueItem list = parseList();
				if (avi == null) 
					avi = list;					// Make head 
				else 
					tail.nextList = list;		// Chain to tail
				tail = list;
				}
				break;
			case TA_StreamTokenizer.TT_WORD:
				{
				AttributeValueItem navi = new AttributeValueItem(ts.sval);
				if (avi == null)
					avi = navi;					// Make head 
				else
					tail.next = navi;			// Chain to tail 
				tail = navi; 
				}
				break; 
			default:
				error("Expecting list, value, or ')'");
				pushBack(); 
				return null;
			}
		}
	}

	protected AttributeValueItem parseNested(String id) throws IOException {
		debugOut("parseNested:");
		// Called when value field started with '{'
		// The attribute value item is an entire AttributeRecord 
		AttributeRecord ar = new AttributeRecord(id); 
		ar.attributes = parseAttributes();	// Parse for attribute list 
		return new AttributeValueItem(ar);
	}

	protected AttributeValueItem parseAVI(String attributeId) throws IOException 
	{
		AttributeValueItem	avi;
		int					ttype;

		ttype = nextToken();
		switch(ttype) {
		case '(':
			avi = parseList();
			break;
		case '{':
			avi = parseNested(attributeId);
			break;
		case TA_StreamTokenizer.TT_WORD:
			avi = new AttributeValueItem(ts.sval);
			break;
		default:
			error("Expecting attribute value");
			avi = null;
		}
		return(avi);
	}

	protected Attribute parseAttributes() throws IOException {

		debugOut("parseAttributes:");
		Attribute attrList = null; 
		Attribute prev = null; 

		// Return a linked list of attributes

		for (;;) {
			int ttype = nextToken();
			if (ttype == '}') {
				return attrList;
			}

			if (ttype != TA_StreamTokenizer.TT_WORD) {
				error("Expecting attribute id");
				return null;
			}


			String attributeId = ts.sval;
			AttributeValueItem avi;

			if (nextToken() != '=') {
				// Attribute declaration with no value 
				pushBack();
				avi = null; 
			} else { 
				avi = parseAVI(attributeId);
				if (avi == null) {
					return null;
				}
			}

			Attribute newAttr = new Attribute(attributeId, avi);
			if (attrList == null)
				attrList = newAttr;	 // Make head 
			else
				prev.next = newAttr; // Chain to tail 
			prev = newAttr; 
		}
	}



	// --------------

	// Public methods

	// --------------

	public LandscapeTokenStream(InputStream is, String fileName) {

		this.fileName = fileName; 
		ts = new TA_StreamTokenizer(is);
	}



	// Parse calls

	protected AttributeValueItem parseAttributeValueItem() throws IOException 
	{
		AttributeValueItem	avi;
		int					ttype;

		avi = parseAVI("");
		ttype = nextToken();
		if (ttype != TA_StreamTokenizer.TT_EOF) {
			avi = null;
		}
		return(avi);
	}

	public int nextSection() throws IOException {

		// Called when a section id is expected
		// Returns section id if new section found, or 
		// EOF when at stream end.

		debugOut("nextSection: "); 

		int ttype;

		String graph, type;

		if (fatalError) {
			return EOF;
		}


		if (token1 != null) {

			if (ttype3 == ':') {

				graph = token1; 

				type = token2;



				token1 = null;

				token2 = null;

			}

			else {

				error("Expecting section header ':'"); 

				return ERROR; 

			}

		}

		else { 

			ttype = nextToken();



			if (ttype == TA_StreamTokenizer.TT_EOF)

				return EOF;



			if (ttype != TA_StreamTokenizer.TT_WORD) {

				error("Expecting graph id");

				return ERROR;

			} 



			graph = ts.sval;



			if (nextToken() != TA_StreamTokenizer.TT_WORD) { 

				error("Expecting section type id");

				return ERROR;

			}



			type = ts.sval; 



			if (nextToken() != ':') {

				error("Expecting ':'");

				return ERROR;

			}

		}

		

		if (graph.equals(SCHEME_ID)) {

			if (type.equals(TUPLE_ID))

				return SCHEME_TUPLE;

			else if (type.equals(ATTRIBUTE_ID))

				return SCHEME_ATTRIBUTE;

			else {

				error("Bad section type");

				return ERROR; 

			}

		}

		else if (graph.equals(FACT_ID)) {

			if (type.equals(TUPLE_ID))

				return FACT_TUPLE;

			else if (type.equals(ATTRIBUTE_ID))

				return FACT_ATTRIBUTE;

			else {

				error("Bad section type");

				return ERROR;

			}

		}

		else if (graph.equals(INCLUDE_ID)) {

			includeFile = type; 

			return INCLUDE_FILE; 

		}

		else {

			error("Bad section id");

			return ERROR;

		}

	}



	public String getIncludeFile() { 

		return includeFile; 

	}



	public void setForTuples() {

	} 



	public Tuple nextTuple() throws IOException {

		debugOut("nextTuple:");



		if (fatalError)
			return null;

		int ttype = nextToken();

		if (ttype == TA_StreamTokenizer.TT_EOF) {
			// End of section 
			return null; 
		}

		if (ttype != TA_StreamTokenizer.TT_WORD) {
			error("Expecting word token");
			fatalError = true;
			return null;
		}


		String verb = ts.sval;

		if (nextToken() != TA_StreamTokenizer.TT_WORD) {
			error("Expecting word token"); 
			fatalError = true;
			return null;
		}

		String object = ts.sval;

		ttype = nextToken();

		if (ttype == ':') {
			token1 = verb;
			token2 = object; 
			ttype3 = ttype; 
			return null; 
		} 

		if (ttype != TA_StreamTokenizer.TT_WORD) {
			error("Error in tuple parse");
			fatalError = true;
			return null;
		}

		String subject = ts.sval;

		return new Tuple(verb, object, subject);
	}



	public void setForRecords() {

	}



	public AttributeRecord nextRecord() throws IOException {

		debugOut("nextRecord: ");

		if (fatalError) {
			return null;
		}

		// 
		// id "{" {attribute} "}"
		// 



		String	  id; 
		Attribute attrList; 



		do {

			int ttype = nextToken();

			if (ttype == TA_StreamTokenizer.TT_EOF) {
				// End of section
				return null; 
			}



			if (ttype == '(') {
				// Relation or relation class 
				id = "(";

				boolean firstToken = true;

				for (;;) {
					ttype = nextToken();
					if (ttype == ')') {
						break;
					}
					if (ttype == TA_StreamTokenizer.TT_EOF) {
						error("Expected ')'");
						return null; 
					} else if (ttype != TA_StreamTokenizer.TT_WORD) {

						errorNS("Unexpected char '" + ((char) ts.ttype) + "' in relation id - Partial id: " + id);
						skipRecord();
						return nextRecord();
					}

					if (firstToken) {
						id += ts.sval;
						firstToken = false;
					}  else {
						id += " " + ts.sval;
					}
				} 
				id += ")"; 
			} else if (ttype != TA_StreamTokenizer.TT_WORD) {
				error("Expecting object id");
				skipRecord();
				return nextRecord();
			} else {

				// Entity class or entity 
				id = ts.sval;
			}

			debugOut("Processing record for: " + id);

			if (nextToken() != '{') {

				token1 = id; 
				token2 = ts.sval;

				if (nextToken() != ':') {
					token1 = null;
					token2 = null; 
					error("Expecting section header"); 
					skipRecord();
					return nextRecord();
				}

				// End of section 

				ttype3 = ':';

				return null;
			}

			attrList = parseAttributes(); 

		} while(attrList == null); // loop until non-empty record 

		AttributeRecord ar = new AttributeRecord(id); 
		ar.attributes = attrList;

		return ar;

	}



	public void errorNS(String msg) {

/*
		private static int first_time = 1;

		if (first_time == 1) {
				first_time = 0;
				System.out.println("errorNS");
				java.lang.Thread.dumpStack();
				System.out.println("-----");
		}
*/

		MsgOut.println("*** Error (" + fileName + ":" + ts.lineno() + "): "

								  + msg);

	}



	public void error(String msg) {

		MsgOut.println("*** Error (" + fileName + ":" + ts.lineno() + "): "

								  + msg + ". Found " + ts.sval);

	}



	public void warning(String msg) {

		MsgOut.println(">>> Warning (" + fileName + ":" + 

						   ts.lineno() + "): " + msg);

	}

}

