package lsedit;

import java.applet.*;
import java.awt.*;
import java.net.*;
import java.io.*;
import java.util.*;
import javax.swing.JFrame;

abstract public class LandscapeViewerCore extends Applet implements ToolBarEventHandler

{
	protected final static String TITLE = "Landscape Viewer";

	protected final static Font menuFont = new Font("Helvetica", Font.PLAIN, 12);

	protected final static int TOC_WIDTH = 175;	   // Width of right box
	protected final static int LB_WIDTH = 200;	   // Width of right box
	protected final static int NB_WIDTH = 200;	   // Name box width
	protected final static int DESC_HEIGHT = 110;  // height of desc box 
	protected final static int NAV_DIM	   = 16;
	protected final static int SGAP = 2;		  // Gap between Toolbar objects
	protected final static int GAP = 4;			  // Gap between GUI objects
	protected final static int FB_GAP = 5;		  // Gap in FB bar

	protected final static double DESC_HEIGHT_FACTOR = 0.12;

	protected final static int DROPDOWN_WIDTH = 150;
	protected final static int DROPDOWN_HEIGHT = 25;
	protected final static int TOP_CTRL_DELTA = 50;

	protected static final String LANDSCAPE_ITEM = "tool"; 
	protected static final String BOOKSHELF_ITEM = "bshelf";

	public static final int SHOW_DESC	   = 1100;
	public static final int SHOW_FB		   = 1101;
	public static final int SHOW_RIGHT	   = 1102;
	public static final int SHOW_TOC	   = 1103;
	public static final int SHOW_CLIENTS   = 1104;
	public static final int SHOW_SUPPLIERS = 1105;
	public static final int SHOW_CARDINALS = 1106;

	public static final int TOP_CLIENTS	   = 1107;
	public static final int SHOW_ANIMATE   = 1108;

	public static final int STUMP_BASE = 1110;
	public static final int EDGE_BASE  = 1120;

	public static final int QUERY_PERSIST = 1140;



	public static final int FA_LOAD = 0;



	/* Parameters */



	protected String editURL;
	protected String lsPath;
	protected Vector lsPath_bg = new Vector();
	protected String aboutURL, helpURL;
	protected String prefURL, prefInvokeURL;
	protected String user;
	protected String startEntity = null;

	protected LandscapeModeHandler modeHandler;
	// protected HelpEngine helpEngine;

	protected static final int RELN_ALL = 10;

	protected AppletContext ac = null; 
	protected JFrame		af = null;

	protected ScrollableDiagram dg = null;		// Active diagram 

	protected static final String resultBoxHelp	  = "Displays the results of visual queries.";
	protected static final String leftTextBoxHelp = "Displays the 'description' for the current landscape.";

	protected static final String rightTextBoxHelp =

		"Displays the 'description' for the closed (not a container) entity currently under the mouse cursor.";



	protected static final String feedbackHelp = 

		"Displays feedback from the program. Examples include errors, warnings, and confirmations of action.";



	protected static final String nameBoxHelp = 

		"Displays the landscape entity, edge, or application button/box currently under the mouse cursor.";



	// GUI compontents

	protected MenuButton editMenuButton = null;

	protected Scrollbar dgScrollBarRight, dgScrollBarBottom;

	protected TextBox leftTextBox, rightTextBox, tocBox;

	protected LegendBox legendBox;

	protected MapBox mapBox;

	protected QueryBox queryBox;

	protected ResultBox resultBox;

	protected Feedback feedback, nameBox; 

	protected HelpBox helpBox;



	protected PopupHelp popupHelp;

	protected TitleBox	titleBox;



	protected EntityInstance currentDescEntity = null;

	protected EntityInstance currentNameEntity = null;

	protected RelationInstance currentEdge = null;



	// Init flags



	protected boolean appInitialized = false;



	protected Dimension viewportDim;



	protected Point lastMousePos = new Point(0, 0);



	// Reference to object which draws the right listing box

	protected TextBox boxDrawer;



	// 



	protected int mode = 0;



	protected boolean tocOn = false;



	protected boolean showClients	= true;

	protected boolean showSuppliers = true;

	protected boolean showCardinals = false;



	protected boolean topClients = true;



	protected Dimension diagDim = null;



	private TabbedBox optionBox;



	abstract protected Dimension getViewDimensions();



	// abstract protected HelpEngine getHelpEngine();



	//



	protected int topWidth;

	private	  boolean show_animation = false;


	// ----------------- 

	// Protected methods

	// ----------------- 


	// Draw the tab which we actually want to see last

	protected void repaintTabs(Graphics gc) 
	{
		if (optionBox != legendBox) {
			legendBox.draw(gc);
		}
		if (optionBox != mapBox) {
			mapBox.draw(gc);
		}
		if (optionBox != queryBox) {
			queryBox.draw(gc);
		}
		if (optionBox != resultBox) {
			resultBox.draw(gc);
		}
		if (optionBox == legendBox) {
			legendBox.draw(gc);
		} else if (optionBox == mapBox) {
			mapBox.draw(gc);
		} else if (optionBox == queryBox) {
			queryBox.draw(gc);
		} else if (optionBox == resultBox) {
			resultBox.draw(gc);
		}
	}

	protected void repaintTabs() 
	{
		repaintTabs(getGraphics());
	}


	protected void hideTabs()
	{
		resultBox.hide();
		legendBox.hide();
		queryBox.hide();
		mapBox.hide();
	}

	protected void showTabs()
	{
		legendBox.show();
		mapBox.show();
		queryBox.show();
		resultBox.show();
		if (optionBox != null) {
			optionBox.activate();
	}	}


	public void activateTabBox(TabbedBox tb) {

		if (optionBox == tb) {
			return;
		}

		if (optionBox != null) {
			optionBox.inactivate();
		}

		optionBox = tb;

		tb.activate();
		repaintTabs();
	}



	protected void setMenuCheck(int key, boolean state) {

	}



	protected String getTitle() {

		return TITLE;

	}



	protected void undo() {

		if (dg.isUndoAvailable()) {

			dg.undo(); 

			repaintDg(); 

		}

		else {

			doFeedback("Require action before undo"); 

		}

	}





	// Generate top of screen GUI 



	protected ToolBarButton[] toolButton = {

				new Find_Button(this),	
				new Query_f_Button(this),

				new Query_b_Button(this), 
				new Query_C_Button(this),

				new Query_Clear_Button(this),

				new Elision_c_Button(this),

				new Elision_I_Button(this), 

				new Elision_u_Button(this),

				new Elision_CU_Button(this), 
				new Elision_s_Button(this),
				new Elision_CS_Button(this),

				new FontSmallerButton(this),
				new FontBiggerButton(this)

		};



	protected int getNumToolButtons() {

		return 6;

	}



	protected MaxMinWindowButton maxMin;



	protected void genToolBar() {

		for (int i = 0; i<getNumToolButtons() ; i++) {

			 this.add(toolButton[i]);
		}
		maxMin = null;

/*

		String st = getParameter("MM_STATE");



		if (st != null) {

			maxMin = new MaxMinWindowButton(this);

			this.add(maxMin);



			if (st != null) 

				maxMin.setState(st.equals("FS"));

		}
*/

	}



	protected void genMainGUI() {

		helpBox = new HelpBox();

		this.add(helpBox);



		popupHelp = new PopupHelp(helpBox, getViewDimensions());

		// helpEngine = getHelpEngine();



		titleBox = new TitleBox(this);

		this.add(titleBox);



		this.add(dgScrollBarRight);

		this.add(dgScrollBarBottom);



		leftTextBox = new TextBox(this, true);

		rightTextBox = new TextBox(this, true);



		tocBox = new TextBox(this, false);

		if (!tocOn)

			tocBox.hide();

		// this.add(tocBox);



		legendBox = new LegendBox(this);

		mapBox = new MapBox(this);

		queryBox = new QueryBox(this);

		resultBox = new ResultBox(this);



		feedback = new Feedback(this);

		nameBox = new Feedback(this);



		// this.add(leftTextBox);

		// this.add(rightTextBox);

		// this.add(feedback);

		// this.add(nameBox);

		// this.add(leftNavButton);

		// this.add(rightNavButton);

		

		genToolBar();

	}



	abstract protected void genMenuButtons();



	protected void genDropDowns() {

		// None in viewer

	}



	protected void genModeHandlers() {

		modeHandler = new ViewModeHandler();

		modeHandler.init(this);

	}



	protected void genGUI() {

		genModeHandlers();

		genMenuButtons();

		genDropDowns(); 

		genMainGUI();

		this.validate();

		this.setLayout(null);

	}



	protected void paintToolButtons(Graphics gc) {



	}

	// Shared by LandscapeEditorCore and LandscapeViewerCore

	protected void layoutToolButtons(int xpos, int ypos)
	{

		for (int i = 0; i < getNumToolButtons(); i++) {
			toolButton[i].move(xpos, ypos);
			Dimension dim = toolButton[i].size();
			xpos += dim.width + SGAP;
		}

		if (maxMin != null) {
			Dimension dim = maxMin.size();
			maxMin.move(topWidth + GAP - dim.width, ypos);
		}
	}





	public void paintTopControls(Graphics gc) {
	}



	protected void layoutTopControls() {

		int width;

		int xpos = GAP;
		int ypos = GAP;

		if (editMenuButton != null) {
			Rectangle dim;

			editMenuButton.resizeButton();
			editMenuButton.move(xpos, ypos);
			this.add(editMenuButton);

			dim = editMenuButton.bounds();

			xpos += dim.x + dim.width + GAP;
		}

		layoutToolButtons(xpos, ypos);

	}



	protected void setLeftBox() {

		EntityInstance root = dg.getRoot();


		
		String label = root.getLabel();

		String desc = root.getDescription();

		String title = root.getTitle();



		if (desc.length() == 0) {

			if (root.getParent() == null) 

				desc = "The " + label + " landscape.";

			else

				desc = "The " + label + " " + root.getEntityClass().getLabel();

		}



		String topline;



		if (title != null) 

			topline = title + " (" + label + ")";

		else

			topline = label;


		leftTextBox.set(topline, desc);
	}


/*
	Rough sketch of layout

	********************* TOC ******************************

	*********************** *******************	 ***********
	*  Leftbox			  * * Right box		  *	 *		   *
	*********************** *******************	 *		   *
												 *		   *
	******Feedback********* * Edge under mouse*	 * TABBED  *
												 * TABLE   *
	*******************************************	 *		   *
	*										  *	 *		   *
	*					DIAGRAM				  *	 *		   *
	*										  *	 *		   *
	*******************************************	 ***********
*/


	protected void layoutGUI() {

		MsgOut.dprintln("Layout");

		viewportDim = getViewDimensions();

		int lbwa = (int) (((float) viewportDim.width)*0.20);



		// Tab box width is larger of 20% or LB_WIDTH

		int tabbw = legendBox.isVisible() ? Math.max(LB_WIDTH, lbwa) + GAP : 0;

		
		int toc = (tocBox.isVisible()) ? TOC_WIDTH + GAP : 0;

		int ph = (int) (((float) viewportDim.height)*DESC_HEIGHT_FACTOR);

		int tbh = !leftTextBox.isVisible() ? 0 : Math.min(DESC_HEIGHT, ph) + GAP;



		int fbh = feedback.isVisible() ? Util.fontHeight(feedback.getFont()) + FB_GAP*2 : 0;

		int w = viewportDim.width-tabbw-toc-GAP*2;
		int h = viewportDim.height-GAP*2; 

		int xm;
		int ypos;

		topWidth = w;

		move(0, 0);		

		resize(viewportDim.width, viewportDim.height);



		if (editMenuButton != null) {
			xm	 = GAP;
			ypos = 45;
		} else {

			Insets ins = af.getContentPane().getInsets();

			xm	 = ins.left + GAP;
			ypos = ins.top + DROPDOWN_HEIGHT + GAP*2;
		}



		if (toc > 0) {
			tocBox.reshape(xm, ypos, toc-GAP, h-ypos);
		}



		if (tabbw > 0) {

			int lbh = (h - ypos);

			legendBox.reshape(xm+toc+w+GAP, ypos, tabbw-GAP, lbh);

			queryBox.reshape(xm+toc+w+GAP, ypos, tabbw-GAP, lbh);

			mapBox.reshape(xm+toc+w+GAP, ypos, tabbw-GAP, lbh);

			resultBox.reshape(xm+toc+w+GAP, ypos, tabbw-GAP, lbh);

		}



		int wtop = (w-GAP)/2;



		if (tbh > 0) {

			int ht = Math.max(tbh - GAP, leftTextBox.bounds().height);



			leftTextBox.reshape(xm+toc, ypos, wtop, ht);



			ht = Math.max(tbh - GAP, rightTextBox.bounds().height);



			rightTextBox.reshape(xm+toc+wtop+GAP, ypos, w-wtop-GAP, ht);



			ypos += tbh;

		}



		if (fbh > 0) {

			feedback.reshape(xm+toc, ypos, wtop, fbh);

			nameBox.reshape(xm+toc+wtop+GAP, ypos, w-wtop-GAP, fbh);

			ypos += fbh + GAP;

		}



		int dgw = viewportDim.width-tabbw-toc-GAP*2;

		int dgh = h-ypos;



		if (dg != null) {
			dg.reshape(xm+toc, ypos, dgw, dgh);
		}

		layoutTopControls();

	}



	protected void editLandscape() {

		MsgOut.dprintln("Edit: " + editURL); 

		showURL(editURL, LsLink.TARGET_NEW); 

	}



	protected void about() {

		if (aboutURL != null) {

			showURL(aboutURL, LsLink.TARGET_FRAME);

		}

	}



	protected void help() {

		if (helpURL != null) {

			showURL(helpURL, LsLink.TARGET_NEW);

		}

	}



	protected void toggleVisibility(LandscapeBoxDrawer bd) {

		if (bd.isVisible()) {

			bd.hide();

		}

		else { 

			bd.show();

		}

	} 



	protected void toggleVisibility(MyComponent bd) {

		if (bd.isVisible()) {

			bd.hide();

		}

		else { 

			bd.show();

		}

		repaint();

	} 



	protected void readyMsg() {

		doFeedback(getTitle() + " " + Version.MAJOR + "." + Version.MINOR + 

				" (build " + Version.BUILD + ") Started.");

	}



	protected static final String indAdd = "  ";



	protected void fillTOC(EntityInstance e, StringBuffer sb, String ind) {

		Enumeration en = e.getChildren();



		while (en.hasMoreElements()) {

			EntityInstance child = (EntityInstance) en.nextElement();



			if (child.hasChildren()) {

				sb.append(ind + child.getLabel() + " {\n");

				fillTOC(child, sb, ind + indAdd);

				sb.append(ind + "}\n");

			}

			else {

				sb.append(ind + child.getLabel() + "\n");

			}

		}

	}



	protected void fillTOC() {

		StringBuffer sb = new StringBuffer();

		String name = Util.prefixOf(dg.getName());

		EntityInstance root = dg.getRoot(); 
		EntityInstance main = null;
		Enumeration en = root.getChildren();

		boolean cont = true;

		while (cont && en.hasMoreElements()) {

			main = (EntityInstance) en.nextElement();



			String idprefix = Util.prefixOf(main.getLabel());



			if (name.equals(idprefix))

				cont = false;

		}



		if (main != null) {

			// Found a main subsystem



			sb.append(main.getLabel() + " {\n");

			fillTOC(main, sb, indAdd);

			sb.append("}\n\n");



			// All others are clients or services



			Vector clients = new Vector();

			Vector services = new Vector();



			en = root.getChildren();



			while (en.hasMoreElements()) {

				EntityInstance e = (EntityInstance) en.nextElement();



				if (e != main) {

					if (e.isClientOf(main)) {

						clients.addElement(e);

					}

					else {

						services.addElement(e);

					}

				}

			}





			if (clients.size() > 0) {

				sb.append("\nClients:\n\n");



				en = clients.elements();



				while (en.hasMoreElements()) {

					EntityInstance e = (EntityInstance) en.nextElement();



					sb.append(e.getLabel() + " {\n");

					fillTOC(e, sb, indAdd);

					sb.append("}\n");

				}

			}

			else {

				sb.append("\nClients: NONE\n");

			}





			if (services.size() > 0) {

				sb.append("\nServices:\n\n");



				en = services.elements();



				while (en.hasMoreElements()) {

					EntityInstance e = (EntityInstance) en.nextElement();



					sb.append(e.getLabel() + " {\n");

					fillTOC(e, sb, indAdd);

					sb.append("}\n");

				}

			}

			else {

				sb.append("\nServices: NONE");

			}

		}

		else {

			// All are in root



			sb.append(name + " {\n");

			fillTOC(root, sb, indAdd);

			sb.append("}\n");

		}



		tocBox.set("Table of contents:", new String(sb), null, true);

	}





	// Called to initialize applet component



	protected void init_core(boolean isApplet) {



		this.setBackground(Color.lightGray);



		if (isApplet) {
			ac = this.getAppletContext();
		}


		dgScrollBarRight = new Scrollbar(Scrollbar.VERTICAL); 
		dgScrollBarBottom = new Scrollbar(Scrollbar.HORIZONTAL); 
		dgScrollBarRight.hide();
		dgScrollBarBottom.hide();

		// Generate fonts

		EntityInstance.generateFonts();
		genGUI();
		showInfo("");
	}

	// Called as super from LandscapeEditorCore.initialLoad()
	// Called from LandscapeViewerCore.start()

	protected void initialLoad() {

		if (lsPath == null) {
			layoutGUI();
			MsgOut.dprintln("Empty landscape");
			leftTextBox.set("Empty Landscape", "Select 'Open landscape' from 'File' menu to load a new landscape.");
			readyMsg();
		} else {

			MsgOut.dprintln("Load: " + lsPath);

			String rc = dg.loadDiagram(lsPath, lsPath_bg, null);

			layoutGUI();

			// System.exit(0);	// Load profiling exit

			fillTOC();


			if (rc != null) {
				error("Load failed (" + rc + ") for: " + lsPath);
				System.out.println("Load failed (" + rc + ") for: " + lsPath);
			} else {
				dg.setToViewport();
				setLeftBox();
				readyMsg();

				if (startEntity != null) {
					if (!dg.navigateTo(startEntity)) {
						error("Entity not found: '" + startEntity + "'");		// IJD
					}
		}	}	}

		layoutTopControls();

		if (legendBox.isVisible()) {
			activateTabBox(legendBox);
		}

		this.requestFocus();

		setEdgeModeMenu(dg.getEdgeMode());
	}



	protected void goTo(Vector res) {

		doFeedback("");



		EntityInstance e  = (EntityInstance) res.elementAt(0);

		EntityInstance pe = e.getParent();



		if (pe != dg.getRoot()) {

			dg.navigateTo(pe);

		}



		dg.clearFlags();



		Enumeration en = res.elements();



		while (en.hasMoreElements()) {

			e = (EntityInstance) en.nextElement();



			e.setHighlightFlag();

		}



		Util.sortVector(res);

		showList("FIND RESULTS:", res);



		setLeftBox();

		rightTextBox.set("", "");

		redrawDg();

	}



	protected Find findResults = null;



	protected void find() {

		SearchCookie sc = 

			FindBox.Create(this, "Find Landscape Entities");



		if (sc.subPatterns.size() > 0) {

			findResults = new Find(sc, dg.getTopInstance());



			if (findResults.foundCount() > 0) {

				goTo(findResults.firstResult());

			}

			else {

				error("No entities found which match search pattern.");

			}

		}

	}



	protected void findNext() {

		if (findResults == null) {

			error("No search has occured.");

			return;

		}



		if (findResults.foundCount() == 0) {

			error("No search results.");

			return;

		}



		if (!findResults.haveNextResult()) {

			error("No more results available.");

			return;

		}



		goTo(findResults.nextResult());

	}



	protected void findPrev() {

		if (findResults == null) {

			error("No search has occured.");

			return;

		}



		if (findResults.foundCount() == 0) {

			error("No search results.");

			return;

		}



		if (!findResults.havePrevResult()) {

			error("At first result.");

			return;

		}



		goTo(findResults.prevResult());

	}









	// --------------

	// Public methods 

	// --------------



	public void init() {

				MsgOut.dprintln("Applet initialization starts");
				init_core(true);

				MsgOut.dprintln("Applet initialization ends");
	}



	public void init_app(JFrame af, Dimension diagDim) {

		MsgOut.dprintln("App initialization starts");
		this.af = af;
		this.diagDim = diagDim;
		init_core(false);

		MsgOut.dprintln("App initialization ends");
	}



	public void start() {

		waitCursorOn();

		dg = new ScrollableDiagram(this, dgScrollBarRight, dgScrollBarBottom);

		if (modeHandler != null) {
			modeHandler.select(dg);
		}
		initialLoad();
		waitCursorOff();
		appInitialized = true;	// No paints or events allowed till this set
		repaint();
	}

/*
	public void repaint()
	{
		System.out.println("LandscapeViewerCore rePaint");		// IJD
		java.lang.Thread.dumpStack();
		System.out.println("-----");
		super.repaint();
	}
*/

	public void paint(Graphics gc) {

		Rectangle cr;
		Rectangle dr;
		Dimension dim;

		if (!appInitialized) {
			return;
		}

		//waitCursorOn();

		dim = getViewDimensions();

		if (dim.width != viewportDim.width || dim.height != viewportDim.height) {
			layoutGUI();
		}

		cr = gc.getClipRect();

		paintTopControls(gc);

		dr = dg.bounds();
		if (cr.intersects(dr)) {
			dg.draw(gc);
		}

		if (editMenuButton != null) {
			editMenuButton.repaint();
		}

		// Unnecessary once a full component (again)

		if (tocBox.isVisible()) {
			tocBox.draw(gc);
		}
		if (feedback.isVisible()) {
			feedback.draw(gc);
		}
		if (nameBox.isVisible()) {
			nameBox.draw(gc);
		}
		if (legendBox.isVisible()) {
			repaintTabs(gc);
		}
		if (leftTextBox.isVisible()) {
			leftTextBox.draw(gc);
		}
		if (rightTextBox.isVisible()) {
			rightTextBox.draw(gc);
		}

		//waitCursorOff();
	}



	public void repaintDg() {

		waitCursorOn();

		Graphics gc = getGraphics();
		dg.draw(gc);
		if (leftTextBox.isVisible()) {
			leftTextBox.draw(gc);
		}
		if (rightTextBox.isVisible()) {
			rightTextBox.draw(gc);
		}
		if (mapBox.isVisible() && optionBox == mapBox) {
			mapBox.draw(gc);
		}
		gc.dispose();

		waitCursorOff();
	}

	public void repaintEntity(EntityInstance e) {

		Graphics gc = getGraphics();

		dg.draw(gc, e);
		gc.dispose();
	}


	public void redrawDg() {

		waitCursorOn();

		dg.rescale();
		repaintDg();
	}


	// Fill the entire canvas with background color
	// Then repaint

	public void repaintFull() {

		MsgOut.dprintln("Landscape ViewerCore repaintFull");

		Dimension sz = size();
		Graphics gc = getGraphics();

		gc.setColor(this.getBackground());
		gc.fillRect(0, 0, sz.width, sz.height);
		gc.dispose();

		if (editMenuButton != null) {
			editMenuButton.repaint();
		}

		dg.rescale();
		repaint();
	}

	public void saveForUndo() {

		dg.saveForUndo();

	}



	public ResultBox getListBox() {

		return resultBox;

	}



	public Point getMousePos() {

		return lastMousePos;
		
	}



	protected boolean isReadWrite() {

		return false;	// No write with viewer

	}



	public void showURL(String urlName, int target) {

		MsgOut.vprintln("URL: " + urlName + " - target: " + target);

		try {

			URL newURL = new URL(getDocumentBase(), urlName);



			// URLConnection urlCon = newURL.openConnection();



			// urlCon.connect();



			switch(target)

			{

			case LsLink.TARGET_TOP:

				ac.showDocument(newURL, "_top");

				break;



			case LsLink.TARGET_NEW:

				ac.showDocument(newURL, "_blank");

				break;



			case LsLink.TARGET_FRAME:

				ac.showDocument(newURL, "Map");

				break;



			case LsLink.TARGET_LIST:

				ac.showDocument(newURL, "List");

				break;

			}

		}

		catch (MalformedURLException ex) {

			MsgOut.println("Malformed URL: " + urlName);

		}

		catch (IOException ex) {

			MsgOut.println("Couldn't access URL: " + urlName);

		}

	}



	public void maxMinLandscape() {

				String id = dg.getName();

				String link = "$LSPATH$/$IDPREFIX$.$LSSUFFIX_ALT$"; 
				String file = Util.expand(link, id, this);

				System.out.println(file);

				showURL(file, LsLink.TARGET_TOP);

	}



	public void showList(String title, Vector v, boolean activateBox) {

				StringBuffer sb = new StringBuffer();



				if (v.size() > 0) {
					Enumeration en = v.elements();



					while(en.hasMoreElements()) {
						String str = en.nextElement().toString();

						sb.append(str + "\n");
					}

				} else {

						sb.append("No entities");

				}
				resultBox.set(title, new String(sb), null, activateBox);
	}



	public void showList(String title, Vector v) {

				showList(title, v, true);

	}



	public void doFeedback(String str) {

		feedback.set(str);

	}



	public void showInfo(String str) {

		if (!rightTextBox.isVisible() || !rightTextBox.isOpen()) {
			nameBox.set(str);
		}

	}



	public void setHelpPopup(String str, int x, int y) {

		popupHelp.set(str, x, y);

	}



	public void clearHelpPopup() {

		popupHelp.clear();

	}



	public void error(String msg) {

		doFeedback(msg);

		System.out.print("\007");

		System.out.flush();

	}

   



	public void clearFeedback() {

		feedback.set("");

	}



	protected String filePrompt(String banner, String file, int mode) {

		String name;
		Container c;

		c = this;
		do { 
			c = c.getParent(); 
		} while (!(c instanceof Frame));

		Frame f = (Frame) c;
		int		  mode1;

/*
		FilePromptDialog.Create(this, f, banner, file, mode);
 */

		if (mode == FA_LOAD) {
				mode1 = FileDialog.LOAD;
		} else {
				mode1 = FileDialog.SAVE;
		}

		FileDialog fd = new FileDialog(f, banner, mode1);
		if (file != null) {
				fd.setFile(file);
		}
		fd.show();		// Blocks until done



		name = fd.getFile();
		if (name != null) {
			name = fd.getDirectory() + name;
		}
		return name;

	}


	protected void checkForPopupHelp(int x, int y) {

		String str = null;



		if (leftTextBox.isMouseOver(x, y)) {

			str = leftTextBoxHelp;

		}

		else if (rightTextBox.isMouseOver(x, y)) {

			str = rightTextBoxHelp;

		}

		else if (feedback.isMouseOver(x, y)) {
			String current;

			str = feedbackHelp;
			current = feedback.get();
			if (current != null) {
				str += "\n\n" + current;
			}
		}

		else if (nameBox.isMouseOver(x, y)) {

			str = nameBoxHelp;

		}



		if (str == null) {

			popupHelp.clear();

		}

		else {

			popupHelp.set(str, x, y);

		}

	}



	protected boolean modeHandlingActive = false;

	protected boolean mouseClickActive = false;



	public boolean mouseDown(Event ev, int x, int y) {

//		System.out.println("LandscapeViewerCore mousedown\n");


		boolean rightButton = ((ev.modifiers & Event.META_MASK) != 0);


		int adjx = dg.adjustX(x);

		int adjy = dg.adjustY(y);

		if (optionBox.mouseDown(ev, x, y)) {
			return true;
		}



		EntityInstance e = dg.mouseOverEx(adjx, adjy);

		if (e != null) {

			RelationInstance ri = dg.mouseOverEdge(e, adjx, adjy);



			showDescription(e, ri, true);



			modeHandlingActive = true;

			mouseClickActive = true;



			modeHandler.start();


			if (!modeHandler.mouseDown(ev, adjx, adjy)) {

				modeHandler.cleanup();

				modeHandlingActive = false;

			}

		}

		else {

			if (legendBox.isOverTab(x, y)) {

				activateTabBox(legendBox);

			}

			else if (mapBox.isOverTab(x, y)) {

				activateTabBox(mapBox);

			}

			else if (queryBox.isOverTab(x, y)) {

				activateTabBox(queryBox);

			}

			else if (resultBox.isOverTab(x, y)) {

				activateTabBox(resultBox);

			}

		}



		return true;

	}



	public boolean mouseDrag(Event ev, int x, int y) {



		if (modeHandlingActive) {

			int adjx = dg.adjustX(x);

			int adjy = dg.adjustY(y);



			if (modeHandler.mouseDrag(ev, adjx, adjy))

				return true;

		}



		if (optionBox.mouseDrag(ev, x, y)) {

			return true;

		}



		return mouseMove(ev, x, y); // Must remain as unadjusted values!

	}



	protected long lastUp = 0;

	

	public boolean mouseUp(Event ev, int x, int y) {

		int adjx = dg.adjustX(x);

		int adjy = dg.adjustY(y);



		if (optionBox.mouseUp(ev, x, y)) {

			return true;

		}



		if (mouseClickActive) {

			if (ev.when - lastUp < 300) {

				// System.out.println("Double click on " + dg.mouseOverEx(adjx, adjy));

				setCursor(Frame.DEFAULT_CURSOR);

				modeHandler.mouseDoubleClick(ev, adjx, adjy);

				lastUp = 0;

				/*[irbull]

				 * Since we are handling double click, don't need 

				 * to handle it as single click as well

				 */

				modeHandler.cleanup();

				modeHandlingActive = false;

			}

			else {

				lastUp = ev.when;

			}

			mouseClickActive = false;

		}

		

		if (modeHandlingActive) {

			modeHandler.mouseUp(ev, adjx, adjy);

			modeHandler.cleanup();

			modeHandlingActive = false;

		}

		else {

			optionBox.mouseUp(ev, x, y);

		}



		if (dg.adjustY(y) > 0) {

			toolButton[0].requestFocus();

			this.requestFocus();

		}		



		return true;

	}



	protected void showDescription(EntityInstance e, RelationInstance ri, boolean showOpens)

	{

		if (ri != null) {

			if (ri != currentEdge) {

				currentEdge = ri;

				currentNameEntity = null;



				EntityInstance root = dg.getRoot();



				EntityInstance src = ri.getSrc().getVisibleEntity();

				EntityInstance dst = ri.getDst().getVisibleEntity();



				showInfo(Util.quoted(src.getLabel()) + " " + 

						 ri.getRelationClass().getLabel() + " " +

						 Util.quoted(dst.getLabel()));

			}

		} else {

			if (e != currentNameEntity) {

				currentNameEntity = e;

				currentEdge = null;



				String str;

				

				if (e == null)

					str = "";

				else {

					EntityInstance pe = e.getEnterableParent();



					if (pe != null) {

						str = pe.getLabel() + " . " + e.getLabel();

					}

					else {

						str = e.getLabel();

					}

				}



				showInfo(str);

			}



			if (e == null)

				return;



			if (currentDescEntity != e && 

				(!e.isOpen() || showOpens) &&

				e.getEntityClass() != null)

			{

				String label = e.getLabel();

				String title = e.getTitle(); 



				String desc = e.getDescription();



				if (desc.length() == 0) {

					desc = "The " + e.getLabel() + " " +

						   e.getEntityClass().getLabel() + ".";

				}



				String topline = " (" + e.getEntityClass().getLabel() +

								 (e.hasChildren() ? 

										" - " + e.numChildren() + " items)" :

								 ")" );



				if (title != null) {

					topline = title + topline;

				}

				else {

					topline = label + topline;

				}



				rightTextBox.set(topline, desc);

				currentDescEntity = e;

			}

		}

	}





	public boolean mouseMove(Event ev, int x, int y) {



		int adjx = dg.adjustX(x);

		int adjy = dg.adjustY(y);



		lastMousePos.x = adjx;

		lastMousePos.y = adjy;



		if (optionBox.isMouseOver(x, y)) {

			optionBox.mouseMotion(ev, x, y);

			return true;

		}

		else {

			checkForPopupHelp(x, y);

		}



		if (leftTextBox.mouseMotion(ev, x, y))

			return true;



		if (rightTextBox.mouseMotion(ev, x, y))

			return true;



		EntityInstance e = dg.mouseOverEx(adjx, adjy);

		RelationInstance ri = dg.mouseOverEdge(e, adjx, adjy);



		showDescription(e, ri, false);



		if (e != null) {

			modeHandler.mouseMotion(e, adjx, adjy);

			if (!e.isOpen()) {

				if (e.isClippedLabel()) {

					titleBox.set(e, dg);

				}

			}

			else {

				titleBox.hide();

			}

		}

		else {

			titleBox.hide();

			setCursor(Frame.DEFAULT_CURSOR);

		}



		return true;

	}



	protected void toggleOption(int key) {

		// System.out.println("Toggle: " + key);



		switch(key)

		{

		case SHOW_DESC:

			toggleVisibility(leftTextBox);

			toggleVisibility(rightTextBox);

			setMenuCheck(SHOW_DESC, leftTextBox.isVisible());

			break;



		case SHOW_FB:

			toggleVisibility(feedback);

			toggleVisibility(nameBox); 

			setMenuCheck(SHOW_FB, feedback.isVisible());

			break;



		case SHOW_RIGHT:

			if (legendBox.isVisible()) {
				hideTabs();
				setMenuCheck(SHOW_RIGHT, false);

			} else {
				showTabs();
				setMenuCheck(SHOW_RIGHT, true);
			}

			break;

		case SHOW_ANIMATE:
			show_animation = !show_animation;
			setMenuCheck(SHOW_ANIMATE, show_animation);
			break;

		case SHOW_TOC:

			toggleVisibility(tocBox);

			setMenuCheck(SHOW_TOC, tocBox.isVisible());

			break;



		case SHOW_CLIENTS:

			showClients = !showClients;

			setMenuCheck(SHOW_CLIENTS, showClients);

			break;



		case SHOW_SUPPLIERS:

			showSuppliers = !showSuppliers;

			setMenuCheck(SHOW_SUPPLIERS, showSuppliers);

			break;



		case SHOW_CARDINALS:

			showCardinals = !showCardinals;

			setMenuCheck(SHOW_CARDINALS, showCardinals);

			break;



		case TOP_CLIENTS:

			setGlobalOption(TOP_CLIENTS, !topClients);

			break;

		}

	}



	protected void setEdgeModeMenu(int mode) {

		for (int i=Diagram.BEST_EDGE; i<=Diagram.DIRECT_EDGE; i++) {

			setMenuCheck(EDGE_BASE+i, false);

		}



		setMenuCheck(EDGE_BASE+mode, true);

	}

 

	protected boolean processAltKey(int key) {

		switch(key)

		{

		case 'l':

			activateTabBox(legendBox);

			return true;



		case 'm':

			activateTabBox(mapBox);

			return true;



		case 'q':

			activateTabBox(queryBox);

			return true;



		case 'r':

			activateTabBox(resultBox);

			return true;

		}



		return false;

	}



	// Keyboard event handling 



	public boolean processKey(int key, int modifiers) {

		

		String str;

		boolean state; 

		EntityInstance e;





		boolean ctrl = ((modifiers & Event.CTRL_MASK) != 0); 

		boolean shift = ((modifiers & Event.SHIFT_MASK) != 0); 



		switch(key)

		{

		case CTRL.PAGE_UP:

		case CTRL.PAGE_DOWN:

			rightTextBox.keyProcess(key);

			return true;



		case 'Q':

			queryBox.toggleGroupingFlag();

			return true;



		case CTRL.U:

		case CTRL.Z:

			undo();

			return true;



		case 'u':

			if (ctrl)  {

				return processKey(CTRL.U, modifiers); 

			}

			return false; 



		case CTRL.R:

			layoutGUI();

			repaint();

			return true; 



		case 'r':

			if (ctrl)

				return processKey(CTRL.R, modifiers); 

			return false;



		case '/':

			state = Util.adjustFontToggle();



			doFeedback("Font adjustment set to " +

				(state ? "on" : "off") + "."); 



			if (editMenuButton != null) { 
				editMenuButton.resizeButton();
			}

			setMenuCheck('/', state);

			repaintFull();

			return true;  



		case CTRL.ESC:

			dg.clearFlags();

			doFeedback("Query/selection cleared");

			redrawDg();

			resultBox.clear();

			modeHandler.reset();

			return true;



		case SHOW_RIGHT:

		case SHOW_DESC:

		case SHOW_FB:

		case SHOW_TOC:

		case SHOW_ANIMATE:

			toggleOption(key);

			layoutGUI();

			repaintFull();

			return true;



		case SHOW_CLIENTS:

		case SHOW_SUPPLIERS:

		case TOP_CLIENTS:

			toggleOption(key);

			navigateTo(dg.getRoot());

			repaintDg();

			return true;



		case SHOW_CARDINALS:

			toggleOption(key);

			repaintDg();

			return true;



		case Diagram.BEST_EDGE+EDGE_BASE:

		case Diagram.TB_EDGE+EDGE_BASE:

		case Diagram.DIRECT_EDGE+EDGE_BASE:

			{

			int mode = key-EDGE_BASE;

			int i;



			switch(mode)

			{ 

			case Diagram.BEST_EDGE:

				str = "best edge";

				break; 



			case Diagram.TB_EDGE:

				str = "top/bottom";

				break; 



			case Diagram.DIRECT_EDGE:

				str = "direct"; 

				break; 



			default: 

				return true;

			}



			dg.setEdgeMode(mode);

			setEdgeModeMenu(mode);

			navigateTo(dg.getRoot());



			doFeedback("Edge mode set to: " + str + "."); 

			repaintDg(); 

			}

			return true; 



		case '!':

			return processKey('1', Event.SHIFT_MASK);

		case '@':

			return processKey('2', Event.SHIFT_MASK);

		case '#':

			return processKey('3', Event.SHIFT_MASK);

		case '$':

			return processKey('4', Event.SHIFT_MASK);

		case '%':

			return processKey('5', Event.SHIFT_MASK);

		case '^':

			return processKey('6', Event.SHIFT_MASK);

		case '&':

			return processKey('7', Event.SHIFT_MASK);

		case '*':

			return processKey('8', Event.SHIFT_MASK);

		case '(':

			return processKey('9', Event.SHIFT_MASK);

		case ')':

			return processKey('0', Event.SHIFT_MASK);



		case CTRL.F1:

			// helpEngine.setHelp(HelpEngine.MAIN);

			return true;



		case '1':

		case '2':

		case '3':

		case '4':

		case '5':

		case '6':

		case '7':

		case '8':

		case '9':

		case '0':

			if (shift) {

				activateTabBox(queryBox);

				queryBox.toggleRelationActivity(dg, 

										(key == '0' ? 10 : key - '1'));

			}

			else {

				activateTabBox(legendBox);

				legendBox.toggleRelationVisibility(dg, 

										(key == '0' ? 10 : key - '1'));

			}

			repaintDg();

			requestFocus();

			return true;



		case 'S':

			dg.setToViewport();
			repaintDg();
			doFeedback("Landscape adjusted to viewport");
			return true;

		case CTRL.F:
			find();
			return true;

		case CTRL.F3:
			findNext();
			return true;

		case CTRL.F2:
			findPrev();
			return true;
/*
		case :
			dg.fitTo(dg.getRoot());
			repaintDg();
			doFeedback("Landscape fitted to viewport");	   
			return true;
*/

		case QUERY_PERSIST:
			state = dg.toggleQueryPersistence();
			doFeedback("Query persistence set to: " + (state ? "On" : "Off"));
			setMenuCheck(QUERY_PERSIST, state);
			return true;

		case -1:
			about();
			return true;

		case -2:

			help();

			return true;



		case CTRL.MAXMIN:

			maxMinLandscape();

			return true;

		}

		return false; 



	} // end processKey	  





	public boolean keyDown(Event event, int key) {



		if ((event.modifiers & Event.ALT_MASK) != 0) {

			return processAltKey(key);

		}



		if (modeHandler.processKey(key, event.modifiers))

			return true;



		return processKey(key, event.modifiers);

	}



	public boolean action(Event event, Object what) {

		return false;

	}



	public boolean doHandleEvent(Event event, Graphics gc) {

		if (leftTextBox.handleEvent(dg, gc, event)) {

			return true;

		}

		else if (rightTextBox.handleEvent(dg, gc, event)) {

			return true;

		}

		else if (tocBox.handleEvent(dg, gc, event)) {

			return true;

		}

		else if (optionBox.handleEvent(dg, gc, event)) {

			return true;

		}

		else if (event.target == dgScrollBarRight ||

				 event.target == dgScrollBarBottom)

		{

			dg.scroll(gc, event);			

			return true;

		} 

		else

			return super.handleEvent(event); 

	}



	public boolean handleEvent(Event event) {

		if (!appInitialized)

			return false;



		Graphics gc = getGraphics(); 



		boolean state = doHandleEvent(event, gc);



		gc.dispose();



		return state;

	}





	public void dialogAction(Object[] results, int type) {

		// Null

	}



	public void processMenuSelection(MenuButton b, int key) {

		processKey(key, 0); 

	}



	public ScrollableDiagram getDiagram() {

		return dg;

	}



	protected void sleep(int ms) {

		try {

			Thread.sleep(ms);

		}

		catch(Exception e) {

		}

	}

	static protected int EC_CNT = 40;


	public Graphics getDiagramContext() {

		return(dg.getContext(getGraphics()));
	}


	protected void expandAnimate(EntityInstance e) {

		if (!show_animation) {
			return;
		}

		Graphics gc = getDiagramContext();


		gc.setXORMode(e.getBackground());



		Layout lyt = e.getLayout();



		Rectangle r = dg.bounds();



		double dl = lyt.x/EC_CNT;

		double dr = (r.width-(lyt.x+lyt.width))/EC_CNT;

		double dt = lyt.y/EC_CNT;

		double db = (r.height-(lyt.y+lyt.height))/EC_CNT;



		dg.drawEntityOutline(gc, lyt, e);



		for (int i=1; i<EC_CNT; i++) {

			dg.drawEntityOutline(gc, lyt, e);

			lyt.x -= dl;

			lyt.y -= dt;

			lyt.width += (dl+dr);

			lyt.height += (dt+db);

			dg.drawEntityOutline(gc, lyt, e);

			sleep(5);

		}

	}



	protected void contractAnimate(EntityInstance e) {

		if (!show_animation) {
			return;
		}

		Graphics gc = getDiagramContext();



		gc.setXORMode(e.getBackground());



		Layout lyt = e.getLayout();



		Rectangle r = dg.bounds();



		double dl = lyt.x/EC_CNT;

		double dr = (r.width-(lyt.x+lyt.width))/EC_CNT;

		double dt = lyt.y/EC_CNT;

		double db = (r.height-(lyt.y+lyt.height))/EC_CNT;



		Layout clyt = new Layout(0, 0, (double) r.width, (double) r.height);



		dg.drawEntityOutline(gc, clyt, e);



		for (int i=1; i<EC_CNT; i++) {

			dg.drawEntityOutline(gc, clyt, e);

			clyt.x += dl;

			clyt.y += dt;

			clyt.width -= (dl+dr);

			clyt.height -= (dt+db);

			dg.drawEntityOutline(gc, clyt, e);

			sleep(5);

		}

	}



	protected void navigateTo(EntityInstance e) {

		EntityInstance oldRoot = dg.getRoot();

		if (e == dg.getRoot().getParent()) {

			dg.navigateTo(e);

			contractAnimate(oldRoot);

		}

		else {
			expandAnimate(e);
			dg.navigateTo(e);
		}



		setLeftBox();

		dg.clearFlags();

		rightTextBox.set("", "");

		resultBox.clear();

		activateTabBox(mapBox);

		doFeedback("Now showing: " + e.getLabel());

		repaint();		// This repaint is needed 
	}

		

	public void followLink(String url, int target) {

		showURL(url, target);

	}



	public void followLink(EntityInstance e, boolean newViewer) {

		Attribute attr = e.getAttribute(EntityInstance.LINK_ID);

		if (attr == null) {

			// This seems to be the case when going down.
			navigateTo(e);

			return;

		}



		AttributeValueItem avi = attr.avi;

		if (avi == null) {

		   // This seems to be the case when going up
		   /* [irbull] if avi is null, just use the navigate to? */

			navigateTo(e);

			return;

		}



		while (avi != null) {

			String file = LsLink.expand(avi.value, e, this);

			if (file != null) {

				if (newViewer) {

					followLink(file, LsLink.TARGET_NEW);

				}

				else if (attr.avi.next != null) {

					String tgtStr = Util.expand(avi.next.value, this);


					int target = LsLink.convertTarget(tgtStr);

					if (target == LsLink.TARGET_APP) {

						// Eventually the link name should be resolved

						// we may want to allow a navigate on one 

						// entity to go to another

						navigateTo(e);

					}

					else {

						followLink(file, target);

					}

				}

				else {

					error("No target: " + attr.avi.value);

				}

			}



			avi = avi.nextList;

		}

	}



	public void enter(EntityInstance e) {

		dg.enter(e);
		dg.setToViewport();
		repaintDg();
	}



	abstract void doSetCursor(int cursor);



	private int curCursor = Frame.DEFAULT_CURSOR;



	protected boolean waitActive = false;



	public void setCursor(int cursor) {

		if (curCursor != cursor) {

			curCursor = cursor;

			if (!waitActive)

				doSetCursor(cursor);

		}

	}



	public void waitCursorOn() {

		if (!waitActive) {

			waitActive = true;

			doSetCursor(Frame.WAIT_CURSOR);

		}

	}



	public void waitCursorOff() {

		if (waitActive) {

			waitActive = false;

			doSetCursor(curCursor);

		}

	}



	public void processKeyEvent(int key, int modifiers) {

		if (modeHandler.processKey(key, modifiers))

			return;

		

		processKey(key, modifiers);

	}



	public boolean isViewer() {

		return true;

	}



	public void toggleRightBoxes() {

/*
		boolean isOpen = !listTextBox.isVisible();

		if (isOpen) {
			listTextBox.show();
		} else {
			listTextBox.hide();
		}

		legendBox.setButtonState(isOpen);
		queryBox.setButtonState(isOpen);

		layoutGUI();

		dg.setToViewport();
		doFeedback("Landscape adjusted to viewport");
		repaint();
*/
	}



	public boolean getGroupQueryFlag() {

		return queryBox.getGroupingFlag();

	}



	public void setGlobalOption(int opt, boolean state) {

		switch(opt)
		{
		case TOP_CLIENTS:
			topClients = state;
			// Thread.dumpStack();
			break;

		case SHOW_CLIENTS:
			showClients = state;
			break;

		case SHOW_SUPPLIERS:
			showSuppliers = state;
			break;

		case SHOW_CARDINALS:
			showCardinals = state;
			break;
		}

		setMenuCheck(opt, state);
	}

	public boolean getOption(int opt) {

		switch(opt)
		{
		case TOP_CLIENTS:
			return topClients;

		case SHOW_CLIENTS:
			return showClients;

		case SHOW_SUPPLIERS:
			return showSuppliers;

		case SHOW_CARDINALS:
			return showCardinals;
		}
		return false;
	}

	abstract public InternalBufferStream getInternalBufferStream();


	// The changing of color behaviour for a diagram is routed through
	// the applet so that it can do more if it wishes than just change
	// one diagram (supposing it has more than one)

	public void setEntitiesColorWhenOpen(EntityClass ofClass, Color color)
	{
		dg.setEntitiesColorWhenOpen(ofClass, color);
	}


	public void setEntitiesColor(EntityClass ofClass, Color color) 
	{
		dg.setEntitiesColor(ofClass, color);
	}	

	public void setEntitiesLabelColor(EntityClass ofClass, Color color)
	{
		dg.setEntitiesLabelColor(ofClass, color);
	}

	public void setEntitiesStyle(EntityClass ofClass, int style)
	{
		dg.setEntitiesStyle(ofClass, style);
	}

	public void setRelationColor(RelationClass ofClass, Color color)
	{
		dg.setRelationColor(ofClass, color);
	}

	public void setRelationStyle(RelationClass ofClass, int style)
	{
		dg.setRelationStyle(ofClass, style);
	}

}


