package lsedit;

import java.awt.*;
import java.util.*;
import java.io.*; 

// EntityInstance class
// 
// Provides most of the data and methods for an entity.
// Can't be instantiated directly. Requires a subclass to provide
// the draw and other functions based on a shape (e.g. box).
//

public abstract class EntityInstance extends LandscapeObject {

	public final static int SMALL_FONT = 0;
	public final static int REG_FONT   = 1;

	public final static int NOT_USED   = -1;
	public final static int CLOSED	   = 0;
	public final static int OPEN	  = 1;

	public final static String LABEL_ID =			 "label";
	public final static String XPOSITION_ID =		 "x";
	public final static String YPOSITION_ID =		 "y";
	public final static String WIDTH_ID =			 "width";
	public final static String HEIGHT_ID =			 "height";

	public final static String XRELPOSITION_ID =	 "xrel";
	public final static String YRELPOSITION_ID =	 "yrel";
	public final static String WIDTHREL_ID =		 "widthrel";
	public final static String HEIGHTREL_ID =		 "heightrel";

	public final static String IN_ELISION_ID =		 "elision";
	public final static String OUT_ELISION_ID =		 "outelision";
	public final static String CLIENT_ELISION_ID =	 "clientelision";
	public final static String SUPPLIER_ELISION_ID = "supplierelision";
	public final static String INTERNAL_ELISION_ID = "internalelision";
	public final static String AGG_ELISION_ID =		 "aggelision";
	public final static String LINK_ID =			 "navlink";
	public final static String INPOINT_ID =			 "inpoints";
	public final static String OUTPOINT_ID =		 "outpoints";
	public final static String LEFTPOINT_ID =		 "leftpoints";
	public final static String RIGHTPOINT_ID =		 "rightpoints";
	public final static String DESC_ID =			 "description";
	public final static String TITLE_ID =			 "title";
	public final static String FONTDELTA_ID =		 "fontdelta";

	protected static final int RSZ_NONE = -1;
	protected static final int RSZ_NW	= 0;
	protected static final int RSZ_N	= 1;
	protected static final int RSZ_NE	= 2; 
	protected static final int RSZ_E	= 3;
	protected static final int RSZ_SE	= 4;
	protected static final int RSZ_S	= 5;
	protected static final int RSZ_SW	= 6;
	protected static final int RSZ_W	= 7;

	protected static final int DEF_CONT_CAP  = 35; // Default contains capacity
	protected static final int ROOT_CONT_CAP = 1000; // root contains capacity

	protected EntityClass parentClass;				// The class this entity belongs to

	protected int mark = 0;

	// Annotation tab constants

	protected static final double TAB_HEIGHT = 16.0; 

	protected static final int MOUSE_NEAR_EDGE_THRESHOLD = 4;
	protected static final int SEP_THRESHOLD = 8;

	// Static font info 

	protected static final int MARGIN = 5; 
	protected static final int MIN_HEIGHT = 10;
	protected static final int MIN_WIDTH = 30;

	protected static final int CONTENTS_FLAG_DIM = 8;

	protected static final String OPEN_FAMILY = "Helvetica";
	protected static final int	  OPEN_FONT_SIZE = 12;
	protected static final Font openFont = new Font(OPEN_FAMILY, Font.PLAIN, OPEN_FONT_SIZE);

	protected static final String CLOSED_FAMILY = "Helvetica";
	protected static final int	  CLOSED_FONT_MIN_SIZE = 6; 
	protected static final int	  CLOSED_FONT_SMALL_SIZE = 10; 
	protected static final int	  NUM_FONTS = 25;
	protected static final int	  START_FONT_NUM = 6;

	protected static Font closedFont[]    = new Font[NUM_FONTS];
	protected static final Font smallFont =	new Font(CLOSED_FAMILY, Font.PLAIN, CLOSED_FONT_SMALL_SIZE);

	protected static boolean mungeIdToLabel = true;
	protected static final Color CARD_FORE_COLOUR = new Color(1.0f, 1.0f, 0.85f);

	// First order attributes

	protected Vector	 inElision       = new Vector();
	protected Vector	 outElision      = new Vector();
	protected Vector	 clientElision   = new Vector();
	protected Vector	 supplierElision = new Vector();
	protected Vector	 internalElision = new Vector();

	protected EdgePoint[] topPoints, bottomPoints, leftPoints, rightPoints;

	// Changed width and height to have some [reasonable?] default IJD

	// These are the coordinates with respect to the parent node

	private double		m_xrelLocal	     = 1.0/32.0;
	private double		m_yrelLocal      = 1.0/32.0;
	private double		m_widthrelLocal  = 15.0/16.0;
	private	double		m_heightrelLocal = 15.0/16.0;

	// These are the coordinates of the object on the screen

	private int			 m_x;					// Top left x position of object relative to diagram top left (0,0)
	private int			 m_y;					// Top left y position of object relative to diagram top, left (0,0)
	private int			 m_width;				// Width of this object
	private int			 m_height;				// Height of this object

	protected int		 fontDelta        = 0;

	/* keep tight control over the group flags so can draw changes at a low level */

	private boolean		 m_groupFlag	  = false;				// This element is a member of the group
	private boolean		 m_groupKeyFlag	  = false;				// This is the active member of the group

	protected boolean	 highlightFlag    = false;
	protected int		 tmpOpen          = NOT_USED;
	protected boolean	 fullLabel        = true;
	protected boolean	 drewClientFlag   = true;
	protected boolean	 drewSupplierFlag = true;

	protected EntityInstance visibleEntityCache = null;

	// Relation info

	// Our container (null -> $ROOT)

	protected EntityInstance containedBy;

	// Entities contained by this one

	protected Hashtable containsList;

	// List of relations for which this entity is the source 

	protected Vector srcRelList = new Vector();

	// List of relations for which this entity is the destination 

	protected Vector dstRelList = new Vector(); 

	protected int[] dstCardinals;

	// List of bends contained in us

	protected Vector bends = new Vector();

	// Temp layout storage

	Layout tempLyt;

	// Stat gathering

	protected int dstCount;

	protected Object	m_ancilliary;

	// -----------------
	// Protected methods
	// -----------------

	protected void setFont(Graphics g, int type) 
	{
		switch(type) {
		case SMALL_FONT:
			g.setFont(smallFont);
			return;
		default:
			g.setFont(closedFont[START_FONT_NUM + fontDelta]);
			return;
		}
	}

	protected void setFontDelta(int fd) 
	{
		// Keep it in bounds
		fontDelta = Math.max(-START_FONT_NUM, Math.min(fd, NUM_FONTS-START_FONT_NUM-1));
	}

	// Called from BaseEntity.undo()

	protected void doSaveForUndo(EntityInstance undoEntity) 
	{
		// Save the things we know about, and then hand down to parent

		undoEntity.srcRelList   = (Vector) srcRelList.clone(); 
		undoEntity.dstRelList   = (Vector) dstRelList.clone(); 
		undoEntity.containsList = (Hashtable) containsList.clone(); 
		undoEntity.containedBy  = containedBy; 

		// Now make copy of attributes;			

		undoEntity.label = label;  
		undoEntity.description = description; 
		undoEntity.setObjectColor(getObjectColor()); 

		// undoEntity.views = (views != null) ? views.clone() : null; 

		undoEntity.inElision       = (Vector) inElision.clone();
		undoEntity.outElision      = (Vector) outElision.clone();
		undoEntity.clientElision   = (Vector) clientElision.clone();
		undoEntity.supplierElision = (Vector) supplierElision.clone();
		undoEntity.internalElision = (Vector) internalElision.clone();

		undoEntity.setxRelLocal(xRelLocal());
		undoEntity.setyRelLocal(yRelLocal()); 
		undoEntity.setwidthRelLocal(widthRelLocal());
		undoEntity.setheightRelLocal(heightRelLocal());

		undoEntity.setX(x());
		undoEntity.setY(y());
		undoEntity.setWidth(width()); 
		undoEntity.setHeight(height()); 

		// I/O points

		int nc = dg.numRelationClasses;

		if (topPoints != null) {
			undoEntity.topPoints = new EdgePoint[nc];
			for (int i = 0; i<nc; i++) {
				if (topPoints[i] != null) {
					undoEntity.topPoints[i] = (EdgePoint) topPoints[i].clone();
				}
			}
		}

		if (bottomPoints != null) {
			undoEntity.bottomPoints = new EdgePoint[nc];
			for (int i = 0; i<nc; i++) {
				if (bottomPoints[i] != null) {
					undoEntity.bottomPoints[i] = (EdgePoint) bottomPoints[i].clone();
				}
			}
		}

		if (leftPoints != null) {
			undoEntity.leftPoints = new EdgePoint[nc];
			for (int i = 0; i<nc; i++) {
				if (leftPoints[i] != null) {
					undoEntity.leftPoints[i] = (EdgePoint) leftPoints[i].clone();
				}
			}
		}

		if (rightPoints != null) {
			undoEntity.rightPoints = new EdgePoint[nc];
			for (int i = 0; i<nc; i++) {
				if (rightPoints[i] != null) {
					undoEntity.rightPoints[i] = (EdgePoint) rightPoints[i].clone();
				}
			}
		}

		super.saveForUndo(undoEntity); 
	}

	protected void saveForUndo(EntityInstance undoEntity) 
	{
		// Handle children

		Enumeration en = getChildren();

		while (en.hasMoreElements()) {
			EntityInstance e = (EntityInstance) en.nextElement();
			e.saveForUndo();
		}

		doSaveForUndo(undoEntity); 
	}

	protected void undo(EntityInstance undoEntity) 
	{
		// Handle children
		Enumeration en = getChildren();

		while (en.hasMoreElements()) {
			EntityInstance e = (EntityInstance) en.nextElement();
			e.undo();
		}

		// Undo the things we know about, and then hand down to parent

		srcRelList   = undoEntity.srcRelList; 
		dstRelList   = undoEntity.dstRelList; 
		containsList = undoEntity.containsList; 
		containedBy  = undoEntity.containedBy;

		// Copy back the attributes 

		label        = undoEntity.label; 
		description  = undoEntity.description; 

		setObjectColor(undoEntity.getObjectColor()); 

		// undoEntity.views = (views != null) ? views.clone() : null;

		inElision       = undoEntity.inElision; 
		outElision      = undoEntity.outElision;
		clientElision   = undoEntity.clientElision;
		supplierElision = undoEntity.supplierElision;
		internalElision = undoEntity.internalElision;

		setxRelLocal(undoEntity.xRelLocal()); 
		setyRelLocal(undoEntity.yRelLocal());
		setwidthRelLocal(undoEntity.widthRelLocal());
		setheightRelLocal(undoEntity.heightRelLocal());

		setX(undoEntity.x());
		setY(undoEntity.y());
		setWidth(undoEntity.width());
		setHeight(undoEntity.height());

		topPoints	    = undoEntity.topPoints;
		bottomPoints    = undoEntity.bottomPoints;
		leftPoints	    = undoEntity.leftPoints;
		rightPoints	    = undoEntity.rightPoints;

		super.undo(undoEntity);
	}



	protected double tabHeight() 
	{
		return height()/3-2;
	}

	protected void drawIOpoint(Graphics g, RelationClass rc, EdgePoint pt, boolean sc)
	{

/*

		// Cache here if needed
		// sc = set colour flag

		if (sc) {
			Color cc = g.getColor();

			RelationInstance.drawBend(g, pt.x, pt.y, true);

			g.setColor(cc);
		} else {
			 RelationInstance.drawBend(g, pt.x, pt.y, false);
		}

*/
	}


	protected void setGraphicsColor(Graphics g) {

		g.setColor(getObjectColor());

	}

	public LandscapeObject derivedFrom(int i)
	{
		return((i == 0) ? parentClass : null);
	}

	public Color getLabelColour() {

		if (highlightFlag && !isOpen()) {

			return Color.yellow;

		}

		return getLabelColor();

	}

	//
	// Parse an attribute record to set the appropriate edge point factors
	// supplied in an attribute record.

	protected void parsePoints(EdgePoint[] ept, Attribute attr) 
	{
		AttributeValueItem avi = attr.avi;

		while (avi != null) {
			RelationClass rc = dg.nameToRelationClass(avi.value);

			if (rc != null) {
				int nid = rc.getNid();

				double wf = Util.parseReal(avi.next.value);
				double hf = Util.parseReal(avi.next.next.value);

				if (ept[nid] == null) {
					ept[nid] = new EdgePoint(this, rc, wf, hf);
				} else {
					if (rc != ept[nid].getRc()) {
						MsgOut.println("Class nid mismatch");
					}
					ept[nid].wf = wf;
					ept[nid].hf = hf;
				}

				ept[nid].isDefault = false;
				avi = avi.nextList;
			}
		}
	}

	static protected Hashtable est = new Hashtable();

	protected void processElision(Attribute attr, Vector list) 
	{
		AttributeValueItem av = attr.avi;

		while(av != null) {
			String str = (String) est.get(av.value);
			if (str == null) {
				str = av.value;
				est.put(str, str);
			}
			if (!list.contains(str)) {
				list.addElement(str);
			}
			av = av.next;
		}
	}

	protected void processRawElision(String[] sl, Vector list) 
	{
		for (int i = 0; i<sl.length; i++) {
			list.addElement(sl[i]);
		}
	}

	//
	// Parse the values of any first order attributes into their storage.
	// 

	// First thing called by EntityInstance::AddAttributes
	// If the attribute is a known member of storage -- don't save attribute -- just store value

	protected boolean processFirstOrderAttributes(Attribute attr)
	{
		boolean hasVal = (attr.avi != null);

		if (attr.hasId(XPOSITION_ID)) {			
			if (hasVal) {
				setxLocal(attr.parseReal());		// Legacy code
			}
			return true; 
		}

		if (attr.hasId(YPOSITION_ID)) {
			if (hasVal) {
				setyLocal(attr.parseReal());		// Legacy code
			}
			return true; 
		}

		if (attr.hasId(WIDTH_ID)) {
			if (hasVal) {
				setwidthLocal(attr.parseReal());	// Legacy code
			}
			return true; 
		}

		if (attr.hasId(HEIGHT_ID)) {
			if (hasVal) {
				setheightLocal(attr.parseReal());	// Legacy code
			}
			return true; 
		}

		if (attr.hasId(XRELPOSITION_ID)) {
			if (hasVal) {
				setxRelLocal(attr.parseReal());
			}
			return true; 
		}

		if (attr.hasId(YRELPOSITION_ID)) {
			if (hasVal) {
				setyRelLocal(attr.parseReal());
			}
			return true; 
		}

		if (attr.hasId(WIDTHREL_ID)) {
			if (hasVal) {
				setwidthRelLocal(attr.parseReal());
			}
			return true; 
		}

		if (attr.hasId(HEIGHTREL_ID)) {
			if (hasVal) {
				setheightRelLocal(attr.parseReal());
			}
			return true; 
		}

		if (attr.hasId(LABEL_ID)) {
			if (hasVal) {
				label = attr.parseString();
			}
			return true; 
		}

		if (attr.hasId(DESC_ID)) {
			if (hasVal) { 
				description = attr.parseString();
			}
			return true; 
		} 

		if (attr.hasId(TITLE_ID)) {
			if (hasVal) {
				title = attr.parseString();
			}
			return true;
		}

		if (attr.hasId(FONTDELTA_ID)) {
			if (hasVal) {
				setFontDelta(attr.parseInt());
			}
			return true; 
		}

		if (attr.hasId(COLOUR_ID)) {
			if (hasVal) {
				setObjectColor(attr.parseColour());
			}
			return true;
		}

		if (attr.hasId(OPEN_COLOUR_ID)) {
			if (hasVal) {
				setObjectColorWhenOpen(attr.parseColour());
			}
			return true;
		}

		if (attr.hasId(LABEL_COLOUR_ID)) {
			if (hasVal) {
				setLabelColor(attr.parseColour());
			}
			return true;
		}

		if (attr.hasId(IN_ELISION_ID)) {
			if (hasVal) { 
				processElision(attr, inElision); 
			}
			return true; 
		}

		if (attr.hasId(OUT_ELISION_ID)) {
			if (hasVal) {
				processElision(attr, outElision);
			}
			return true;
		}

		if (attr.hasId(CLIENT_ELISION_ID)) {
			if (hasVal) {
				processElision(attr, clientElision);
			}
			return true;
		}

		if (attr.hasId(SUPPLIER_ELISION_ID)) {
			if (hasVal) {
				processElision(attr, supplierElision);
			}
			return true;
		}

		if (attr.hasId(INTERNAL_ELISION_ID)) {
			if (hasVal) {
				processElision(attr, internalElision);
			}
			return true;
		}

		if (attr.hasId(INPOINT_ID)) {
			if (hasVal) {
				if (topPoints == null) {
					topPoints = new EdgePoint[dg.numRelationClasses];
				}
				parsePoints(topPoints, attr);
			}
			return true; 
		}

		if (attr.hasId(OUTPOINT_ID)) {
			if (hasVal) {
				if (bottomPoints == null) {
					bottomPoints = new EdgePoint[dg.numRelationClasses];
				}
				parsePoints(bottomPoints, attr); 
			}
			return true; 
		}

		if (attr.hasId(LEFTPOINT_ID)) {
			if (hasVal) {
				if (leftPoints == null) {
					leftPoints = new EdgePoint[dg.numRelationClasses];
				}
				parsePoints(leftPoints, attr); 
			}
			return true; 
		}

		if (attr.hasId(RIGHTPOINT_ID)) {
			if (hasVal) {
				if (rightPoints == null) {
					rightPoints = new EdgePoint[dg.numRelationClasses];
				}
				parsePoints(rightPoints, attr); 
			}
			return true; 
		}

		return false; 
	}



	protected boolean hasRelation(RelationClass rc, boolean srcList) {

		Enumeration enRels = (srcList ? srcRelList.elements() : dstRelList.elements());



		while(enRels.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) enRels.nextElement();



			if (ri.parentClass.id.equals(rc.id))

				return true; 

		}



		// Now check for infered relations based on what we contain 



		Enumeration contains = getChildren();



		while (contains.hasMoreElements()) {

			EntityInstance e = (EntityInstance) contains.nextElement();



			if (e.hasRelation(rc, srcList))

				return true;

		}
		return false; 
	}


	protected void doEntityDelete() 
	{
		// Remove dstRelList relations from entities on our srcRelList.
		// Remove srcRelList relations from entities on our dstRelList. 
		// Remove entity from parent's contain list. 

		Enumeration en = srcRelList.elements(); 
		while(en.hasMoreElements()) {
			RelationInstance ri = (RelationInstance) en.nextElement();
			ri.getDst().removeDstRelation(ri); 
		} 

		en = dstRelList.elements();
		while(en.hasMoreElements()) {
			RelationInstance ri = (RelationInstance) en.nextElement();
			ri.getSrc().removeSrcRelation(ri);
		}
		containedBy.removeContainment(this);
	}

	// Determine the bounding box of all children

	public Layout getChildrenRelBoundingBox() 
	{
		Enumeration		en;
		EntityInstance	e;
		Bend			bend;
		double			x1, x2, y1, y2, temp;
		boolean			seen;

		x1   = Double.MAX_VALUE;
		x2   = Double.MIN_VALUE;
		y1   = Double.MAX_VALUE;
		y2   = Double.MIN_VALUE;
		seen = false;

		for (en = getChildren(); en.hasMoreElements(); ) {
			seen = true;
			e = (EntityInstance) en.nextElement();

			temp = e.xRelLocal();
			if (temp < x1) {
				x1 = temp;
			}

			temp = e.yRelLocal();
			if (temp < y1) {
				y1 = temp;
			}

			temp = e.xRelLocal() + e.widthRelLocal();
			if (temp > x2) {
				x2 = temp;
			}

			temp = e.yRelLocal() + e.heightRelLocal();
			if (temp > y2) {
				y2 = temp;
		}	}

		if (!seen) {
			MsgOut.dprintln("No children");
			return(null);
		} 
		return new Layout(x1, y1, x2-x1, y2-y1);
	}

	protected String endSet[] = { 
		".ss",
		".dup",
		"__funcdef",
		"__funcdcl",
		"__vardef",
		"__vardcl",
		"__macro", 
		"__struct", 
		"__union", 
		"__enum", 
		"__type"
	}; 

	public String mungeId(String id) 
	{
		if (mungeIdToLabel && id.length() > 0) {

			for (int i = 0; i < endSet.length; i++) {
				if (id.endsWith(endSet[i])) {
					id = id.substring(0, id.length()-endSet[i].length());
					break;
				}
			}

			if (id.charAt(0) != '<') {
				int lpi = id.lastIndexOf('/');

				if (lpi < 0) {
					lpi = id.lastIndexOf('\\');
				}
				if (lpi >= 0 && id.length() != lpi+1) {
					return id.substring(lpi+1);
				}
			}
		}
		return id;
	}



	// --------------

	// Static methods

	// -------------- 


	/* Allocates helvetica fonts of increasing point size */

	static public void generateFonts() {

		for (int i=0; i<NUM_FONTS; i++) {

			closedFont[i] =

				new Font(CLOSED_FAMILY, Font.PLAIN, CLOSED_FONT_MIN_SIZE+i /* Point size */);

		}

	}



	static public void setLabelMungeFlag(boolean state) {

		mungeIdToLabel = state;

	}





	// ----------------
	// Abstract methods
	// ----------------

	// draw()
	//
	// Must be provided by a subclass of EntityInstance to handle
	// drawing of actual entity shape (e.g. box).

	abstract public void saveForUndo(); 
	abstract public void undo(); 
	abstract public void draw( Graphics g);

	// abstract public void drawIOpoint(Graphics g, EdgePoint pt);
	// Calculate the intercept between the entity and the line between
	// points p1 and p2, where p1 is in the entity.



	abstract public RealPoint calculateIntercept(RealPoint p1, RealPoint p2);



	abstract public boolean isPointOver(int x, int y);
	abstract public boolean isPointOverIO(EdgePoint pt, int x, int y);
	abstract public boolean containsLayout(Layout lyt); 
	abstract public boolean intersectsLayout(Layout lyt); 
	abstract public int		overResizeTab(int x, int y);

	abstract public void drawHighlight(Graphics g);
	abstract public void undrawHighlight(Graphics g);
	abstract public void drawOutline(Graphics g, Layout lyt); 

	abstract public EdgePoint getPoint(RelationClass rc, int side);
	abstract public EdgePoint getOutPoint(RelationInstance ri, int edge_mode, Layout srcLyt, Layout dstLyt);

	abstract public EdgePoint getMouseOverIO(int x, int y);
	abstract public void writeIOpoints(PrintStream ps);

	// --------------
	// Public methods
	// --------------

	protected EntityInstance(Diagram dg) 
	{
		this.dg = dg;
	}


	protected String extendedLabel(boolean wParent) 
	{
		if (wParent) {
			EntityInstance pe = getEnterableParent();
			if (pe != null) {
				return pe.getLabel() + " .\n" + label;
			}
		}
		return label;
	}



	public void setup(EntityClass parentClass, String id, Diagram dg) 
	{
		this.dg = dg;
		this.id = id;
		this.parentClass = parentClass;

		this.label = mungeId(id); // Default

		// A default layout

		this.setX(10);
		this.setY(10);
		this.setWidth(20);
		this.setHeight(10);

		containsList = new Hashtable(id.equals(Diagram.ROOT_ID) ? ROOT_CONT_CAP : DEF_CONT_CAP);
	}

 

	public EntityInstance(EntityClass parentClass, String id, Diagram dg) 
	{
		setup(parentClass, id, dg);
	}



	public void clearCache() 
	{
		Enumeration en;

		// Reset any caching

		visibleEntityCache = null;

		topPoints = bottomPoints = rightPoints = leftPoints = null;

		for (en = getChildren(); en.hasMoreElements();) {
			EntityInstance e = (EntityInstance) en.nextElement();
			e.clearCache();
		}
	}

	public void drawAll(Graphics g) {

		drewClientFlag	 = false;
		drewSupplierFlag = false;
		dstCount		 = 0;

		draw(g);

		if (!isOpen()) {
//			System.out.println("EntityInstance::drawAll " + this + " closed");

			if (hasChildren()) {
				drawContentsFlag(g);
			}
		} else {
			drawChildren(g);
		}
	}



	public void drawChildren(Graphics g) 
	{
		Enumeration en;
		EntityInstance e;

		if (!isOpen()) {
//			System.out.println("EntityInstance::drawChildren " + this + " closed");
			return;
		}

		for (en = getChildren(); en.hasMoreElements(); ) {
			e = (EntityInstance) en.nextElement();
//			System.out.println("EntityInstance::drawChildren " + this + " being drawn");
			e.drawAll(g);
		}
	}

	// Destroy just the entity, leaving the children 

	public void delete() 
	{
		Enumeration en;

		// Promote our children

		for (en = getChildren(); en.hasMoreElements(); ) {
			EntityInstance e = (EntityInstance) en.nextElement();
			Layout lyt = e.getLayout();
			containedBy.addContainment(e);
			e.setLayout(lyt); 
		}
		doEntityDelete(); 
	}



	// Destroy entity and all it contains 

	public void deleteAll(boolean delThis) 
	{
		// deleteAll on our children

		Enumeration en;

		for (en = getChildren(); en.hasMoreElements(); ) {
			EntityInstance e = (EntityInstance) en.nextElement();
			e.deleteAll(true);
		} 

		if (delThis) {
			doEntityDelete();
	}	}



	public void addContainment(EntityInstance e) {

		// This entity contains the passed entity

		// Put on this' contains list



		if (containsList.put(e.id, e) != null) {

			MsgOut.println("Entity " + id + " already contains " + e.id);

		}

		else {

			e.containedBy = this;

		}

	}



	public boolean removeContainment(EntityInstance e) {

		if (containsList.containsKey(e.id)) {

			containsList.remove(e.id);

			return true;

		}

		else {

			MsgOut.println("Entity " + id + " doesn't contain " + e.id);

			return false;

		}

	}



	public void moveEntityContainment(EntityInstance newContainer) {

		// Transfer containment iff we need to



		if (newContainer != containedBy) {

			containedBy.removeContainment(this);

			newContainer.addContainment(this); 

		}

	}



	public void addRelation(RelationInstance ri, EntityInstance dst) {

		// Should check to see if relation is already present



		addSrcRelation(ri); 

		dst.addDestRelation(ri); 

	}



	public RelationInstance getRelation(RelationClass rc, EntityInstance dst) {

		// Return relation instance if present



		Enumeration en = srcRelList.elements();



		while (en.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) en.nextElement();



			if (ri.getRelationClass() == rc && ri.getDst() == dst)

				return ri;

		}



		return null;

	}



	public void addSrcRelation(RelationInstance ri) {

		srcRelList.addElement(ri); 

	}



	public void removeSrcRelation(RelationInstance ri) {

		srcRelList.removeElement(ri); 

	}



	public void addDestRelation(RelationInstance ri) {

		dstRelList.addElement(ri); 

	}



	public void removeDstRelation(RelationInstance ri) {

		dstRelList.removeElement(ri);

	}



	public boolean isRelationPresent(RelationClass rc, EntityInstance dst) {

		String dstId = dst.getId();



		Enumeration en = srcRelList.elements();



		while(en.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) en.nextElement();



			if (ri.isSameRelation(rc, this, dst)) {

				return true; 

			}

		}



		return false; 

	}



	public void addAttribute(Attribute attr) {

		// Process first order attributes, and add all others 
		// to an attribute database

		if (processFirstOrderAttributes(attr)) {
			return;
		}
		super.addAttribute(attr);
	}



	public boolean addRawAttribute(String id, int val) {

		if (id.equals(FONTDELTA_ID)) {
			setFontDelta(val);
			return true; 
		}
		return false;
	}



	public boolean addRawAttribute(String id, double val) {

		if (id.equals(XPOSITION_ID)) {			// Legacy code
			setxLocal(val);						// Legacy code
			return true; 
		}
		else if (id.equals(YPOSITION_ID)) {
			setyLocal(val);						// Legacy code
			return true; 
		}
		else if (id.equals(WIDTH_ID)) {
			setwidthLocal(val);					// Legacy code
			return true; 
		}
		else if (id.equals(HEIGHT_ID)) {
			setheightLocal(val);				// Legacy code
			return true; 
		}
		if (id.equals(XRELPOSITION_ID)) {
			setxRelLocal(val);
			return true; 
		}
		else if (id.equals(YRELPOSITION_ID)) {
			setyRelLocal(val);
			return true; 
		}
		else if (id.equals(WIDTHREL_ID)) {
			setwidthRelLocal(val);
			return true; 
		}
		else if (id.equals(HEIGHTREL_ID)) {
			setheightRelLocal(val);
			return true; 
		}
		return false;
	}



	public boolean addRawAttribute(String id, String val) {

		if (id.equals(LABEL_ID)) {
			label = val;
			return true; 
		}

		else if (id.equals(DESC_ID)) {
			description = val;
			return true; 
		} 
		else if (id.equals(TITLE_ID)) {
			title = val;
			return true;
		}
		return super.addRawAttribute(id, val);
	}

	public boolean addRawAttribute(String id, int[] il) {
		return false;
	}



	public boolean addRawAttribute(String id, double[] dl) {
		if (id.equals(COLOUR_ID)) {
			setObjectColor( new Color((float) dl[0], (float) dl[1], (float) dl[2]));
			return true;
		}
		if (id.equals(LABEL_COLOUR_ID)) {
			setLabelColor( new Color((float) dl[0], (float) dl[1], (float) dl[2]) );
			return true;
		}
		if (id.equals(OPEN_COLOUR_ID)) {
			setObjectColorWhenOpen( new Color((float) dl[0], (float) dl[1], (float) dl[2]));
			return true;
		}
		return false;
	}



	public boolean addRawAttribute(String id, String[] sl) {

		if (id.equals(IN_ELISION_ID)) {
			processRawElision(sl, inElision); 
			return true; 
		}
		else if (id.equals(OUT_ELISION_ID)) {
			processRawElision(sl, outElision);
			return true;
		}
		else if (id.equals(CLIENT_ELISION_ID)) {
			processRawElision(sl, clientElision);
			return true;
		}
		else if (id.equals(SUPPLIER_ELISION_ID)) {
			processRawElision(sl, supplierElision);
			return true;
		}
		else if (id.equals(INTERNAL_ELISION_ID)) {
			processRawElision(sl, internalElision);
			return true;
		}

		return super.addRawAttribute(id, sl);
	}



	public void assignAttributes(Attribute attr) {

		// Passed list of attribute assignments

		while (attr != null) {

			// For each attribute assignment, find the variable,
			// and assign the value.

			if (attr.avi == null) {
				MsgOut.println("Null attribute assignment");
			} else if (!processFirstOrderAttributes(attr)) {
				Attribute attrVar = getAttribute(attr.id);

				if (attrVar == null) {
					MsgOut.println("Missing attribute '" + attr.toString() + "'");
				} else {
					if (attrVar.cloneOnAssign) {
						attrVar = (Attribute) attrVar.clone();
						attrVar.cloneOnAssign = false;
						replaceAttribute(attrVar);
					}
					attrVar.avi = attr.avi;
				}
			}
			attr = attr.next;
		}
	} 

	public void assignRelnAttributes(RelationClass rc, EntityInstance dst, Attribute attr)
	{	
		// Find the associated edge
		System.out.println("Asgn: " + rc.getId() + " " + id + " " + dst.getId());
	}


/*
		v6.0.3
		Stupid to rescale -- might as well recalculate 
 */


	protected void rescale(EdgePoint[] pts) {

		if (pts != null) {
			for (int i = 0; i < pts.length; i++) {
				if (pts[i] != null) {
					pts[i].clearRc();
	}	}	}	}

	// Global coordinate transform
	//
	// Calculate global values for layout from local with scale.
	// The absolute x and y offsets of the parent are passed.
	//
	// This also handles calling the rescale on the containing entities.
	//

	private void rescale(int parent_x, int parent_y, int parent_width, int parent_height)
	{
		int new_x, new_y, new_width, new_height;

		new_x      = parent_x + (int) (parent_width  * m_xrelLocal);
		new_y      = parent_y + (int) (parent_height * m_yrelLocal);
		new_width  =            (int) (parent_width  * m_widthrelLocal);
		new_height =            (int) (parent_height * m_heightrelLocal);

/*		System.out.println("EntityInstance.rescale(" + parent_x + ", " + parent_y + ", " + parent_width + ", " + parent_height + ") using " + this + " [" + m_xrelLocal + ", " + m_yrelLocal + ", " + m_widthrelLocal + ", " + m_heightrelLocal + "] = {" + new_x + ", " + new_y + ", " + new_width + ", " + new_height + "}");
		java.lang.Thread.dumpStack();
		System.out.println("-----");
*/
		setGlobalLayout(new_x, new_y, new_width, new_height); 
	}

	public void rescaleChildren() 
	{
		Enumeration en;
		
		// rescale edge points

		rescale(topPoints);
		rescale(bottomPoints);
		rescale(leftPoints);
		rescale(rightPoints);


		// rescale bends

		for (en = bends.elements(); en.hasMoreElements(); ) {
			Bend bend = (Bend) en.nextElement();
			bend.x    = m_x + bend.xRelLocal() * m_width;
			bend.y    = m_y + bend.yRelLocal() * m_height;
			MsgOut.dprintln(id + " " + bend.x + " " + bend.y);
		}

		for (en = getChildren(); en.hasMoreElements(); ) {
			EntityInstance e = (EntityInstance) en.nextElement();
			e.rescale(m_x, m_y, m_width, m_height);
		}
	}

	protected void setX(int x) {
		m_x = x;
	}

	protected void setY(int y) {
		m_y = y;
	}

	protected void setWidth(int width) {
		m_width = width;
	}

	protected void setHeight(int height) {
		m_height = height;
	}

	protected void setGlobalLayout(int x, int y, int width, int height) 
	{
		m_x      = x;
		m_y      = y;
		m_width  = width;
		m_height = height;

/*
		System.out.println(this + ":setGlobalLayout(" + x + "," + y + "," + width + "," + height + ")");		// IJD
		java.lang.Thread.dumpStack();
		System.out.println("-----");
*/

		rescaleChildren();
	}

	public void setGlobalLayout(Layout lyt) 
	{
		setGlobalLayout((int) lyt.x, (int) lyt.y, (int) lyt.width, (int) lyt.height);
	} 

	protected void scale(double wf, double hf, boolean doContainer, boolean doPos)
	{
		// Scale ourselves 

		if (doContainer) {
			double x1, y1, w1, h1; 

			x1 = xRelLocal();
			y1 = yRelLocal();
			w1 = widthRelLocal()  * wf;
			h1 = heightRelLocal() * hf;

			if (doPos) {
				x1 *= wf;
				y1 *= hf;
			}

			if (x1 >= 0.0 && w1 > 0.0 && y1 >= 0.0 && h1 > 0.0 && (x1+w1) <= 1.0 && (y1+h1) <= 1.0) {
				setxRelLocal(x1);
				setyRelLocal(y1);
				setwidthRelLocal(w1);
				setheightRelLocal(h1);	
			}
			return;

		} 
		
		// Insure the scale of the bounding box of the contained elements
		// will be within the container.
		
		Layout bb = getChildrenRelBoundingBox();

		if (bb == null) {
			return;
		}

		bb.width  *= wf;
		bb.height *= hf;

		if (doPos) {
			bb.x = bb.x * wf;
			bb.y = bb.y * hf;
		}

		// Don't bother, if it won't fit right
		if (bb.x >= 0.0 && bb.width > 0.0 && bb.y >= 0.0 && bb.height > 0.0 && (bb.x + bb.width) < 1.0 && (bb.y + bb.height) < 1.0) {
		
			// ... any bends we contain ...

			Enumeration en;

			for (en = bends.elements(); en.hasMoreElements(); ) {
				Bend bend = (Bend) en.nextElement();

				bend.setxRelLocal(bend.xRelLocal() * wf);
				bend.setyRelLocal(bend.yRelLocal() * hf);
			}

			// ... and then our children

			for (en = getChildren(); en.hasMoreElements(); ) {
				EntityInstance e = (EntityInstance) en.nextElement();
				e.scale(wf, hf, true, true);
		}	}
	}

	public void scale(double sf, boolean doContainer, boolean doPos) 
	{
		scale(sf, sf, doContainer, doPos);
	}

	public void scale(double sf, boolean doContainer) 
	{
		scale(sf, sf, doContainer, false);
	}

	public void scale(double sf) 
	{
		scale(sf, sf, true, false);
	}

	public void scaleX(double sf, boolean doContainer, boolean doPos) 
	{
		scale(sf, 1.0, doContainer, doPos);
	}

	public void scaleX(double sf, boolean doContainer) 
	{
		scale(sf, 1.0, doContainer, false);
	}

	public void scaleX(double sf) 
	{
		scale(sf, 1.0, true, false);
	}

	public void scaleY(double sf, boolean doContainer, boolean doPos) 
	{
		scale(1.0, sf, doContainer, doPos);
	}

	public void scaleY(double sf, boolean doContainer) 
	{
		scale(1.0, sf, doContainer, false); 
	}

	public void scaleY(double sf) 
	{
		scale(1.0, sf, true, false);
	}

	// Attempt to fit children to us  

	private void doFitScale(double dx, double dy, double wf, double hf) 
	{
		setwidthRelLocal(dx + widthRelLocal() * wf);
		setheightRelLocal(dy + heightRelLocal() * hf);	
		setxRelLocal(xRelLocal() * wf + dx);
		setyRelLocal(yRelLocal() * hf - dy);
	}


	public void fitTo(boolean alwaysFit) 
	{
		Enumeration en;

		MsgOut.dprintln("Fit to: " + id );

		if ( (widthRelLocal() == 0) || (heightRelLocal() == 0)) {
			containsList = new Hashtable();
		}

		if (containsList.size() == 0) {
			MsgOut.dprintln("No children");
			return;
		}

		Layout bb = getChildrenRelBoundingBox();

		if (bb.width <= 0.0 || bb.height <= 0.000) {
			return;
		}

		if (!alwaysFit) {

			// Only continue if bb isn't completely contained in this

			if (bb.x >= 0.0 && bb.y >= 0.0 && (bb.x + bb.width) <= 1.0 && (bb.y + bb.height) <= 1.0) {
				return;
		}	}
		

		// Translate bb to top left of parent and scale to fit

		// want new_relx = 1/15 and (new_relx + new_relwidth) = 14/15 for bounding box
		// so:
		//	m*x         + c =  1/15
		//	m*(x+width) + c = 14/15
		//	m*(x+width-x)   = 13/15 => m = 13/(15*width)
		//							=> c = (1/15) - m * x;

		double scaleX = 13.0 / (15.0 * bb.width);
		double scaleY = 13.0 / (15.0 * bb.height);
		double dx     = (1.0 / 15.0) - scaleX * bb.x;
		double dy     = (1.0 / 15.0) - scaleY * bb.y;

		for (en = getChildren(); en.hasMoreElements(); ) {
			EntityInstance e = (EntityInstance) en.nextElement();

			e.doFitScale(dx, dy, scaleX, scaleY);
		}

		MsgOut.dprintln("Fit: " + scaleX + " " + scaleY);
		return;
	}

	// Change the relative dimensions if anything is out of wack -- ie relative < 1/15 or > 14/15

	public void verifyFit() 
	{
		fitTo( false );
	}


	public void deltaFont(int delta) 
	{
		setFontDelta(fontDelta + delta);
	}

	protected void drawTopLeftLabel(Graphics g) 
	{
		g.setFont(smallFont);
//		g.setColor(Color.black); 

		Util.drawStringClipped(g, label, x()+MARGIN, y()+MARGIN, width()-MARGIN*2, height()-MARGIN*2);
	}



	public boolean doDrawLabel(Graphics g, double x, double y,	double width, double height, boolean wParent)
	{
		return Util.drawStringWrapped(g, extendedLabel(wParent), x+MARGIN, y+MARGIN, width-MARGIN*2, height-MARGIN*2, true, false);
	}



	public void drawLabel(Graphics g, int type, boolean wParent) 
	{
		if (getParent() == null) {
			return; 
		}

		// Draw our own label

		if (isLabelDrawable()) {
			g.setColor(getLabelColour());
			if (isOpen()) {
				// top left corner 
				drawTopLeftLabel(g);
			} else {
				// centered, perhaps multi-line
				setFont(g, type);
				fullLabel = doDrawLabel(g, x(), y(), width(), height(), wParent);
			}
		} else {
			fullLabel = false;
		}
	}



	public void drawLabel(Graphics g) 
	{
		drawLabel(g, REG_FONT, false);
	}



	public void drawLabels(Graphics g) {



		if (width() < MIN_WIDTH || height() < MIN_HEIGHT) {

			fullLabel = false;

			return;

		}



		// Call method on each child



		if (isOpen()) {

			Enumeration en = getChildren();



			while (en.hasMoreElements()) {

				EntityInstance e = (EntityInstance) en.nextElement();



				e.drawLabels(g);

			}

		}



		drawLabel(g);

	}



	public Dimension getLabelDim(Graphics g, int type, boolean wParent) {

		setFont(g, type);



		FontMetrics fm = g.getFontMetrics();



		int h = Util.fontHeight(fm);

		int w;



		if (wParent) {

			EntityInstance pe = getEnterableParent();



			if (pe != null) {

				w = Math.max(fm.stringWidth(pe.getLabel() + " | "), 

							 fm.stringWidth(label));

				h += h/2;

			}

			else {

				w = fm.stringWidth(label);

			}

		}

		else {

			w = fm.stringWidth(label);

		}



		return new Dimension(w, h);

	}



	public Dimension getLabelDim(Graphics g, int type) {

		return getLabelDim(g, type, false);

	}



	public Dimension getFitDim(Graphics g, int type, boolean wParent) {

		Dimension dim = getLabelDim(g, type, wParent);



		if (hasChildren() && !isOpen()) {

			dim.width += CONTENTS_FLAG_DIM;

		}



		dim.width += MARGIN*3;

		dim.height += (MARGIN*3)/2;

		

		return dim;

	}



	public Dimension getFitDim(Graphics g, int type) {

		return getFitDim(g, type, false);

	}





	// Flags



	public void drawClientFlag(Graphics g) {

		// Draw flag that shows client exists



		if (drewClientFlag)

			return;



		RealPoint p1 = 

			new RealPoint(x() + width() - 5, y() + Util.ARROW_L);



		RealPoint p2 =

			new RealPoint(x() + width() - Util.ARROW_L*2 - 5, y() + Util.ARROW_L);



		Util.drawArrow(g, p1, p2);



		drewClientFlag = true;

	}



	public void drawSupplierFlag(Graphics g) {

		// Draw flag that shows supplier exists



		if (drewSupplierFlag)

			return;



		RealPoint p1 = new RealPoint(x() + width() - Util.ARROW_L*2 - 5, y() + Util.ARROW_L);

		RealPoint p2 = new RealPoint(x() + width() - 5, y() + Util.ARROW_L);



		Util.drawArrow(g, p1, p2);



		drewSupplierFlag = true;

	}

	/*  Draw a small mark as shown in top left of object

		x---
        |
		|   --------
		|  |        |
		   |    |   |
		   |    |   |
	  	   |  ----  |
		   |    |   |
		   |    |   |
		   |        |
		   |________|
	*/

	public void drawContentsFlag(Graphics g) 
	{
		int x1, y1;

		if (width() < CONTENTS_FLAG_DIM*2 || height() < CONTENTS_FLAG_DIM*2) {
			return; 
		}

		g.setColor(getObjectColor().darker());

		x1 = x();
		y1 = y();

		g.drawRect(x1+3, y1+3, CONTENTS_FLAG_DIM /* 8 */, CONTENTS_FLAG_DIM);

		// g.setColor(Color.black);
		g.drawLine(x1+5,y1+7, x1+9, y1+7);
		g.drawLine(x1+7, y1+5, x1+7, y1+9);
	}



	// Edges


	public void drawEdges(Graphics g) { 

		EntityInstance ve, pe;

		ve = getVisibleEntity();

		if (ve == null) {
			pe = null;
		} else {
			pe = ve.getParent();
		}

		// Drill down to draw edges from each child

		Enumeration en = getChildren();

		while (en.hasMoreElements()) {

			EntityInstance e = (EntityInstance) en.nextElement();

			e.drawEdges(g);

		}

		// All that follows seems to depend on pe not being null

		if (pe == null) {
			return;
		}

		// Now draw edges from this entity



		EntityInstance root = dg.getRoot();



		en = srcRelList.elements();



		while (en.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) en.nextElement();



			if (ri.getSrc() != this) {

				System.out.println("Error!!!!!");

			}



			if (ri.draw(g, true)) { // allow elision

				EntityInstance dst = ri.getDst();


				if (!root.descendent(dst) && 

					pe.isSupplierRelationElided(ri.getRelationClass()))

				{

					ve.drawSupplierFlag(g);

				}

			}

		}



		// There may be edges that end with us that have no source

		// in the current diagram root. 



		en = dstRelList.elements();



		while (en.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) en.nextElement();



			EntityInstance src = ri.getSrc();



			if (!root.descendent(src)) {

				if (ri.draw(g, true)) { // allow elision

					if (pe.isClientRelationElided(ri.getRelationClass())) {

						ve.drawClientFlag(g);

					}

				}

			}

		}

	}



	public void drawHighlightedEdges(Graphics g) { 

		// Draw our edges (edges starting with us)



		EntityInstance root = dg.getRoot();



		Enumeration en = srcRelList.elements();



		while (en.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) en.nextElement();



			if (ri.getHighlightFlag() && ri.getRelationClass().isActive()) {

				ri.draw(g, false);		// no elision

			}

		}



		en = dstRelList.elements();



		while (en.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) en.nextElement();



			EntityInstance src = ri.getSrc();



			if (!root.descendent(src) && ri.getHighlightFlag() &&

						ri.getRelationClass().isActive())

			{

				ri.draw(g, false);		// no elision

			}

		}





		// Call method on each child



		en = getChildren();



		while (en.hasMoreElements()) {

			EntityInstance e = (EntityInstance) en.nextElement();



			e.drawHighlightedEdges(g);

		}

	}



	protected void doDrawCardinals(Graphics g, FontMetrics fm, int ht) {

		for (int i=0; i<dstCardinals.length; i++) {

			RelationClass rc = dg.numToRelationClass(i);



			if (dstCardinals[i] > 0 && 

				(!dg.allowElision() || rc.isVisible()))

			{

				String cs = "" + dstCardinals[i];



				int w = fm.stringWidth(cs);



				double f = rc.getIOfactor();

		

				ScreenPoint pt = 

						new ScreenPoint(x()+width()*f-w/2, y()+height()+MARGIN+ht);



				g.setColor(rc.getObjectColor());



				if (g instanceof PSGr) {

					g.fillOval(pt.x-MARGIN, pt.y-ht+3, 

								Math.max(ht, w+MARGIN*2), ht);

				}

				else {

					g.fillOval(pt.x-MARGIN, pt.y-ht+3, 

								Math.max(ht, w+MARGIN*2)+1, ht+1);

				}

		

				g.setColor(Color.black);



				g.drawOval(pt.x-MARGIN, pt.y-ht+3, 

								Math.max(ht, w+MARGIN*2), ht);



				g.setColor(CARD_FORE_COLOUR);



				g.drawString(cs, pt.x, pt.y);

			}

		}



		if (!isOpen())

			return;



		Enumeration en = getChildren();



		while (en.hasMoreElements()) {

			EntityInstance e = (EntityInstance) en.nextElement();



			e.doDrawCardinals(g, fm, ht);

		}

	}



	public void drawCardinals(Graphics g) {

		g.setFont(smallFont);



		FontMetrics fm = g.getFontMetrics();



		doDrawCardinals(g, fm, Util.fontHeight(fm));

	}





	public void getHighlightedEdges(Vector v) { 



		Enumeration en = srcRelList.elements();



		while (en.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) en.nextElement();



			if (ri.getHighlightFlag() && ri.getRelationClass().isActive()) {

				v.addElement(ri);

			}

		}



		en = dstRelList.elements();



		EntityInstance root = dg.getRoot();



		while (en.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) en.nextElement();



			EntityInstance src = ri.getSrc();



			if (!root.descendent(src) && ri.getHighlightFlag() &&

						ri.getRelationClass().isActive())

			{

				v.addElement(ri);

			}

		}





		// Call method on each child



		en = getChildren();



		while (en.hasMoreElements()) {

			EntityInstance e = (EntityInstance) en.nextElement();



			e.getHighlightedEdges(v);

		}

	}







	// draw a line segement between entity and coordinates



	public void drawSegment(Graphics gc, int x1, int y1) {

		int x0 = (int) Math.round(x() + width()/2);

		int y0 = (int) Math.round(y() + height()/2);



		gc.setColor(Color.black);

		gc.drawLine(x0, y0, x1, y1); 

	}



	// TA raw info routines



	public void getInstancesRaw(Vector v, Hashtable st) {

		// Get instance info for ourself, and then all our children



		if (parentClass != null) {

			// Not $ROOT 



			Integer[] pair = new Integer[2];



			pair[0] = (Integer) st.get(id);

			pair[1] = (Integer) st.get(parentClass.getId());



			v.addElement(pair);

		}



		Enumeration en = getChildren();



		while(en.hasMoreElements()) {

			EntityInstance child = (EntityInstance) en.nextElement();



			child.getInstancesRaw(v, st);

		}

	}



	public void getRelationsRaw(Vector v, Hashtable st, Integer cid) {

		// Get the relations on us, and then all our children



		Enumeration en = srcRelList.elements(); 



		while(en.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) en.nextElement();



			Integer[] triple = new Integer[3];



			RelationClass rc  = ri.getRelationClass();

			EntityInstance e1 = ri.getSrc();

			EntityInstance e2 = ri.getDst();



			triple[0] = (Integer) st.get(rc.getId());

			triple[1] = (Integer) st.get(e1.getId());

			triple[2] = (Integer) st.get(e2.getId());



			v.addElement(triple);

		}				



		en = getChildren();



		while(en.hasMoreElements()) {

			EntityInstance child = (EntityInstance) en.nextElement();



			if (parentClass != null) {

				// If not root, output contain relation



				Integer[] triple = new Integer[3];



				triple[0] = cid;

				triple[1] = (Integer) st.get(id);

				triple[2] = (Integer) st.get(child.id);



				v.addElement(triple);

			}



			child.getRelationsRaw(v, st, cid);

		}

	}





	// TA file output routines 



	public void writeInstances(PrintStream ps) throws IOException {

		// Write an instance line for ourself, and then all our children



		if (parentClass != null) {

			// Not $ROOT 



			ps.print(Diagram.INSTANCE_ID + " " + 

						qt(id) + " " + qt(parentClass.getId()) + "\n");

		}



		Enumeration en = getChildren();



		while(en.hasMoreElements()) {

			EntityInstance child = (EntityInstance) en.nextElement();



			child.writeInstances(ps);

		}

	}



	public void writeRelations(PrintStream ps) throws IOException {

		// Write out the relations on us, and then all our children



		Enumeration en = srcRelList.elements(); 



		while(en.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) en.nextElement();



			ri.writeRelation(ps); 

		}				



		en = getChildren();



		while(en.hasMoreElements()) {

			EntityInstance child = (EntityInstance) en.nextElement();



			if (parentClass != null) {

				// If not root, output contain relation



				ps.print(Diagram.CONTAIN_ID + " " + qt(id) + " " + qt(child.id) + "\n"); 

			}



			child.writeRelations(ps);

		}

	}



	protected void writeLayout(PrintStream ps) throws IOException 
	{
		// Write out layout attributes

		ps.print(Attribute.indent + "xrel      = " + xRelLocal() + "\n");
		ps.print(Attribute.indent + "yrel      = " + yRelLocal() + "\n"); 
		ps.print(Attribute.indent + "widthrel  = " + widthRelLocal() + "\n"); 
		ps.print(Attribute.indent + "heightrel = " + heightRelLocal() + "\n");
	}



	public void writeLayoutAttributes(PrintStream ps) throws IOException 
	{

		// Write out attribute record for us, and then our children



		ps.print(qt(id) + " {\n");



		writeLayout(ps); 



		ps.print("}\n\n");



		// Recurse for contained children



		Enumeration en = getChildren();



		while(en.hasMoreElements()) {

			EntityInstance child = (EntityInstance) en.nextElement();



			child.writeLayoutAttributes(ps);

		}

	}



	protected void writeElision(PrintStream ps, Vector ev, String id) {

		if (ev.size() > 0) {

			ps.print(Attribute.indent + id + " = (");

			   

			Enumeration en = ev.elements();



			while(en.hasMoreElements()) {

				String rcId = (String) en.nextElement(); 



				if (en.hasMoreElements())

					ps.print(rcId + " ");

				else

					ps.print(rcId); 

			}

			ps.print(")\n"); 

		}

	}



	protected void regRawElision(Vector v, Hashtable st, Vector ev, String id) 

	{

		if (ev.size() > 0) {

			int len = ev.size();

			String[] sl = new String[len];



			for (int i=0; i<len; i++) {

				sl[i] = (String) ev.elementAt(i);

			}



			regRawAttribute(v, st, id, sl);

		}

	}







	public void

	writeAttributes(PrintStream ps, boolean outputLyt) throws IOException

	{  

		// Write out attribute record for us, and then our children



		ps.print(qt(id) + " {\n");



		if (outputLyt)	{

			writeLayout(ps);
		}


		if (label != null && !label.equals(mungeId(id))) {

			// Save only if it differs from munged id (ie. auto-label)



			ps.print(Attribute.indent + LABEL_ID + " = " +

						qt(label) + "\n");

		}



		if (description != null) {

			ps.print(Attribute.indent + DESC_ID + " = \"" + 

						description + "\"\n");

		}



		if (parentClass != null) {

			// First order attributes not associated with $ROOT 

			writeElision(ps, inElision, IN_ELISION_ID);

			writeElision(ps, outElision, OUT_ELISION_ID);

			writeElision(ps, clientElision, CLIENT_ELISION_ID);

			writeElision(ps, supplierElision, SUPPLIER_ELISION_ID);

			writeElision(ps, internalElision, INTERNAL_ELISION_ID);

			writeIOpoints(ps);



			if (fontDelta != 0) {

				ps.print(Attribute.indent + FONTDELTA_ID + " = ");

				ps.print(fontDelta + "\n");

			}

		}



		// Finally output the second-class attributes 


		super.writeAttributes(ps, false); 

		// End the record 

		ps.print("}\n\n"); 



		// Write any attributes of src relations



		Enumeration en = srcRelList.elements();



		while(en.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) en.nextElement();



			ri.writeAttributes(ps);

		}



		// Recurse for contained children



		en = getChildren();



		while(en.hasMoreElements()) {

			EntityInstance child = (EntityInstance) en.nextElement();



			child.writeAttributes(ps, outputLyt);

		}

	}



	public void getAttributesRaw(Vector mv, Hashtable st) {

		Vector v = new Vector();

 

		regRawAttribute(v, st, XPOSITION_ID, x());

		regRawAttribute(v, st, YPOSITION_ID, y());

		regRawAttribute(v, st, WIDTH_ID, width());

		regRawAttribute(v, st, HEIGHT_ID, height());



		if (label != null && !label.equals(mungeId(id))) {

			// Save only if it differs from munged id (ie. auto-label)



			regRawAttribute(v, st, LABEL_ID, label);

		}



		if (description != null) {

			regRawAttribute(v, st, DESC_ID, description);

		}



		if (parentClass != null) {

			// First order attributes not associated with $ROOT 



			Color pc	 = parentClass.getObjectColor();
			Color color1 = getObjectColor();



			if (!color1.equals(pc)) {

				regRawAttribute(v, st, COLOUR_ID, color1);

			}

			pc = parentClass.getLabelColor();


			Color labelcolor1;

			labelcolor1 = getLabelColor();

			if (!labelcolor1.equals(pc)) {

				regRawAttribute(v, st, LABEL_COLOUR_ID, labelcolor1);

			}


			color1 = getObjectColorWhenOpen();
			if (color1 != null) {
				regRawAttribute(v, st, OPEN_COLOUR_ID, labelcolor1);
			}


			regRawElision(v, st, inElision, IN_ELISION_ID);

			regRawElision(v, st, clientElision, OUT_ELISION_ID);

			regRawElision(v, st, supplierElision, SUPPLIER_ELISION_ID);

			regRawElision(v, st, internalElision, INTERNAL_ELISION_ID);



			// addRawIOpoints(v, st);



			if (fontDelta != 0) {

				regRawAttribute(v, st, FONTDELTA_ID, fontDelta);

			}

		}



		// Finally output the second-class attributes 



		if (attributes.size() > 0) 

			super.getAttributesRaw(v, st, false);



		Object[] obj = new Object[2];



		obj[0] = st.get(id);

		obj[1] = v;



		mv.addElement(obj);



		Enumeration en;



/*

		// Write any attributes of src relations



		Enumeration en = srcRelList.elements();



		while(en.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) en.nextElement();



			ri.writeAttributes(ps);

		}

*/

		// Recurse for contained children



		en = getChildren();



		while(en.hasMoreElements()) {

			EntityInstance child = (EntityInstance) en.nextElement();



			child.getAttributesRaw(mv, st);

		}

	}





	// Accessor functions and queries 



	public boolean canOpen() {

		return !containsList.isEmpty();

	}



	public boolean isOpen() {

		if (tmpOpen != NOT_USED) {
			return (tmpOpen == OPEN);
		}
		return ((dg.getRoot() == this) || (!isInRelationElided(Diagram.CONTAIN_ID) && canOpen()));
	}



	public boolean isOpenElided() {

		 return !isInRelationElided(Diagram.CONTAIN_ID);

	}



	public EntityInstance getParent() { 

		return containedBy;

	}	



	public EntityInstance getEnterableParent() {

		if (containedBy == null || containedBy.isEnterable())

			return containedBy;

		

		return containedBy.getEnterableParent();

	}



	public EntityClass getEntityClass() {

		return parentClass;

	}



	public boolean isRoot() {

		return (containedBy == null);

	}



	public boolean hasChildren() {

		return !containsList.isEmpty();

	}



	public int numChildren() {

		return containsList.size();

	}



	public Enumeration getChildren() {

		return containsList.elements();

	}



	public void resetCardinals(int numRelations) {

		dstCardinals = new int[numRelations];



		Enumeration en = getChildren();



		while (en.hasMoreElements()) {

			EntityInstance e = (EntityInstance) en.nextElement();



			e.resetCardinals(numRelations);

		}

	}



	public void incCardinal(int ind) {

		dstCardinals[ind]++;

	}



	public void calcEdgeCardinals(boolean hOnly) {

		Enumeration en = srcRelList.elements();



		EntityInstance ve = getVisibleEntity();



		while (en.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) en.nextElement();

		   

			int ind = ri.getRelationClass().getNid();



			EntityInstance vsrc = ri.getDst().getVisibleEntity();



			if (vsrc != null && vsrc != ve && (!hOnly || ri.getHighlightFlag()))

				vsrc.incCardinal(ind);

		}



		EntityInstance root = dg.getRoot();



		en = dstRelList.elements();



		while (en.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) en.nextElement();



			if (!root.descendent(ri.getSrc())) {

				int ind = ri.getRelationClass().getNid();



				if (!hOnly || ri.getHighlightFlag())

					ve.incCardinal(ind);

			}

		}



		en = getChildren();



		while (en.hasMoreElements()) {

			EntityInstance e = (EntityInstance) en.nextElement();



			e.calcEdgeCardinals(hOnly);

		}

	}

	



	public void getClients(Vector v, EntityInstance root) {

		Enumeration en = dstRelList.elements();



		while (en.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) en.nextElement();



			EntityInstance src = ri.getSrc();


			if ((this == getVisibleEntity(root) || 

				!getParent().isInRelationElided(AGG_ELISION_ID)) &&

				!root.descendent(src) && !v.contains(src))

			{

				v.addElement(src);

			}
/*
			else if (root.descendent(src)) {
				System.out.println("GetClients " + src + " root.descendant()");
			} else if (v.contains(src)) {
				System.out.println("GetClients " + src + " v.contains()");
			} else {
				System.out.println("GetClients " + src + " visible=" + getVisibleEntity(root) + " relationElided=" + getParent().isInRelationElided(AGG_ELISION_ID));
			}
*/

		}



		en = containsList.elements();



		while (en.hasMoreElements()) {

			EntityInstance e = (EntityInstance) en.nextElement();



			e.getClients(v, root);

		}

	}



	public void getSuppliers(Vector v, EntityInstance root) {

		Enumeration en = srcRelList.elements();



		while (en.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) en.nextElement();



			EntityInstance dst	= ri.getDst();

			EntityInstance vsrc = getVisibleEntity(root);



			if ((this == vsrc || !getParent().isInRelationElided(AGG_ELISION_ID)) && !root.descendent(dst) && !v.contains(dst))

			{

				v.addElement(dst);

			}

		}



		en = containsList.elements();



		while (en.hasMoreElements()) {

			EntityInstance e = (EntityInstance) en.nextElement();



			e.getSuppliers(v, root);

		}

	}



	public String toString() {

		return label;

	}



	public Diagram getDiagram() {

		return dg;

	}



	public String getDescription() {

		if (description != null)

			return description; 

		else

			return "";	

	}



	public void setDescription(String str) {

		description = str;

	}



	public void setLabel(String str) {

		label = str;

	}



	public Layout getLayout() {
		return new Layout(x(), y(), width(), height());
	}



	public Layout getRelLayout() 
	{
		return new Layout(xRelLocal(), yRelLocal(), widthRelLocal(), heightRelLocal());
	}

	// Layout is global, adjust it to coords local to parent

	public void setLayout(Layout lyt) 
	{
		double	width1, height1;

		width1  = parentsWidth();
		height1 = parentsHeight();

		if (width1 <= 0.0) {
			setxRelLocal(0);
			setwidthRelLocal(0);
		} else {
			setxRelLocal((lyt.x - parentsX())/width1);
			setwidthRelLocal(lyt.width/width1);
		}

		if (height1 <= 0.0) {
			setyRelLocal(0);
			setheightRelLocal(0);
		} else {
			setyRelLocal((lyt.y - parentsY())/height1);
			setheightRelLocal(lyt.height/height1);
		}
		setGlobalLayout(lyt);
	} 

	public void saveLayout() {

		tempLyt = new Layout(x(), y(), width(), height());
		Enumeration en = getChildren();

		while (en.hasMoreElements()) {
			EntityInstance e = (EntityInstance) en.nextElement();
			e.saveLayout();
		}
	}

	public void restoreLayout() {

		if (tempLyt == null) {
			System.out.println("Layout: restore without save " + label);
			return;
		}

		setGlobalLayout(tempLyt);
		tempLyt = null;

		Enumeration en = getChildren();

		while (en.hasMoreElements()) {
			EntityInstance e = (EntityInstance) en.nextElement();
			e.restoreLayout();
		}
	}

	public void addBend(Bend bend) {

		if (!bends.contains(bend)) {
			bends.addElement(bend);
		}
	}

	public void deleteBend(Bend bend) {
		bends.removeElement(bend);
	}

	public Layout getChildBoundingBox() 
	{
		Enumeration en;

		// Calc the global bounding box of children

		if (containsList.isEmpty()) {
			return null;
		}

		double x1 = Double.MAX_VALUE;
		double x2 = Double.MIN_VALUE;
		double y1 = Double.MAX_VALUE;
		double y2 = Double.MIN_VALUE;

		for (en = getChildren(); en.hasMoreElements(); ) {

			EntityInstance e = (EntityInstance) en.nextElement();

			if (e.x() < x1) {
				x1 = e.x();
			}

			if (e.y() < y1) {
				y1 = e.y();
			}

			if (e.x() + e.width() > x2) {
				x2 = e.x() + e.width();
			}


			if (e.y() + e.height() > y2) {
				y2 = e.y() + e.height();
		}	}

		return new Layout(x1, y1, x2-x1, y2-y1);
	}

	public Color getBackgroundWhenOpen() {

		Color c;

		// Container is handled specially
		// Find colour from depth

		c = getObjectColorWhenOpen();
		if (c == null) {

			float v = Diagram.BG - 0.05f;

			EntityInstance b = this;

			if (isTransparent()) {
				b = b.containedBy;
			}

			EntityInstance root = dg.getRoot();

			if (!root.descendent(this)) {
				root = root.common(this);
			}


			while(b != root) {
				v -= 0.05f;
				b = b.containedBy; 
			}

			c = new Color(v, v, v);

			if (Util.isBlack(c)) {
				c = Color.lightGray;
		}	}
		return c;
	}
			

	public Color getBackground() {

		Color c;

		if (isOpen()) {
			c = getBackgroundWhenOpen();
		} else {
			c = getObjectColor();
		}
		return c;
	}



	public boolean isRelationElided(Vector v, String id) {

		// String str = (String) est.get(id);
		//if (str == null)
			//return false;
		// [irbull] commented this out because if an attributre was not present when it was first created, it would never use it.

		return v.contains(id);

	}



	protected boolean toggleElision(Vector v, String id) {

		if (isRelationElided(v, id)) {
			if (!v.removeElement(id)) {
				MsgOut.println("remove() failed");
			}
			return false; 
		} 

		v.addElement(id); 
		return true; 
	}

 

	public boolean toggleInElision(String id) {

		MsgOut.dprintln("toggle in " + id);

		return toggleElision(inElision, id);
	}

 

	public boolean toggleInElision(RelationClass rc) {
		return toggleInElision(rc.getId());
	}



	public boolean toggleOutElision(String id) {

		MsgOut.dprintln("toggle out " + id);



		return toggleElision(outElision, id);

	}



	public boolean toggleOutElision(RelationClass rc) {

		return toggleOutElision(rc.getId());

	}



	public boolean toggleClientElision(String id) {

		MsgOut.dprintln("toggle client " + id);



		return toggleElision(clientElision, id);

	}



	public boolean toggleClientElision(RelationClass rc) {

		return toggleClientElision(rc.getId());

	}



	public boolean toggleSupplierElision(String id) {

		MsgOut.dprintln("toggle supplier " + id);



		return toggleElision(supplierElision, id);

	}



	public boolean toggleSupplierElision(RelationClass rc) {

		return toggleSupplierElision(rc.getId());

	}



	public boolean toggleInternalElision(String id) {

		MsgOut.dprintln("toggle internal " + id);



		return toggleElision(internalElision, id);

	}



	public boolean toggleInternalElision(RelationClass rc) {

		return toggleInternalElision(rc.getId());

	}



	public void setTempOpenFlag(int state) 
	{
		tmpOpen = state;
	}



	// Queries 



	public boolean hasId(String id) 
	{
		return this.id.equals(id);
	}



	public boolean isInRelationElided(RelationClass rc) 
	{
		return isRelationElided( inElision, rc.getId() );
	}



	public boolean isInRelationElided(String id) 
	{
		return isRelationElided(inElision, id);
	}

	public boolean isOutRelationElided(RelationClass rc) 
	{
		return isRelationElided(outElision, rc.getId());
	}

	public boolean isOutRelationElided(String id) 
	{
		return isRelationElided(outElision, id);
	}



	public boolean isClientRelationElided(RelationClass rc) 
	{
		return isRelationElided(clientElision, rc.getId());
	}



	public boolean isClientRelationElided(String id) 
	{
		return isRelationElided(clientElision, id);
	}



	public boolean isSupplierRelationElided(RelationClass rc) 
	{
		return isRelationElided(supplierElision, rc.getId());
	}



	public boolean isSupplierRelationElided(String id) 
	{
		return isRelationElided(supplierElision, id);
	}



	public boolean isInternalRelationElided(RelationClass rc) 
	{
		return isRelationElided(internalElision, rc.getId());
	}



	public boolean isInternalRelationElided(String id) 
	{
		return isRelationElided(internalElision, id);
	}



	public boolean isContainedBy(EntityInstance e2) 
	{
		EntityInstance c = containedBy;

		while (c != null) {
			if (c == e2) {
				return true;
			}
			c = c.containedBy;
		}
		return false;
	}



	public boolean descendent(EntityInstance e) 
	{
		EntityInstance c = e;

		while (c != null) {
			if (c == this) {
				return true;
			}
			c = c.containedBy;
		}
		return false;
	}

	// Return the common parent entity

	public EntityInstance common(EntityInstance e) 
	{
		EntityInstance e1 = this.containedBy;

		if (e1 == null) {
			return e1; // All things have root as containment ancestor
		}
		while(e1 != null) {
			EntityInstance e2 = e;
			while(e2 != null) {
				if (e1 == e2) {
					return e1;
				}
				e2 = e2.containedBy;
			}
			e1 = e1.containedBy;
		}
		MsgOut.println("EntityInstance.common: failed");
		return null;
	}

	public EntityInstance getVisibleEntity(EntityInstance root) 
	{
		if (this == root) {
			return root;
		}

		if (visibleEntityCache == null) {

			// Calculate it once per draw

			EntityInstance e = this;

			EntityInstance ce = e;
			EntityInstance top;
			if (root.descendent(e)) {
				top = root;
			} else {
				top = dg.getClientOrSupplier(e);
				if (top == null) {
					return null;
				}
				top = top.getParent();
			}
			do {
				ce = ce.containedBy;
				if (ce == null) {
					return e;
				}
				if (ce != top && !ce.isOpen()) {
					e = ce;
				}
			} while (ce != top);
			visibleEntityCache = e;
		}
		return visibleEntityCache;

	}



	public EntityInstance getVisibleEntity() 
	{
		return getVisibleEntity(dg.getRoot());
	}

	// Return true if we're a client of passed entity.
	// That is, a relation exist between us, or our children and 
	// the passed entity or its children.

	public boolean isClientOf(EntityInstance pc) 
	{

		Enumeration en = srcRelList.elements();

		while (en.hasMoreElements()) {
			RelationInstance ri = (RelationInstance) en.nextElement();
			EntityInstance e = ri.getDst();

			if (e == pc) {
				return true;
			}

			if (pc.isContainedBy(e)) {
				return true;
			}
		}

		en = getChildren();

		while(en.hasMoreElements()) {
			EntityInstance e = (EntityInstance) en.nextElement();
			if (e.isClientOf(pc)) {
				return true;
		}	}

		return false;
	}

	// Return reference to entity the cursor is over. 
	// Return null if none.

	public EntityInstance getMouseOver(int x, int y) 
	{
		if (isPointOver(x, y)) {
			if (isOpen()) {
				Enumeration en = getChildren();

				while(en.hasMoreElements()) {
					EntityInstance e = (EntityInstance) en.nextElement();

					EntityInstance over = e.getMouseOver(x, y);
					if (over != null)
						return over; 
				}

			}
			// It's not over any of our children, so it's over us.
			return this; 
		}
		return null;
	}



	public Bend getMouseOverBend(int x, int y) 
	{
		Enumeration en = bends.elements();

		while (en.hasMoreElements()) {
			Bend bend = (Bend) en.nextElement();
			if (bend.isOver(x, y)) {
				return bend;
		}	}

		en = getChildren();

		while(en.hasMoreElements()) {
			EntityInstance e = (EntityInstance) en.nextElement();

			if (e.isOpen()) {
				Bend bend = e.getMouseOverBend(x, y);
				if (bend != null) {
					return bend; 
		}	}	}
		return null;
	}


	public RelationInstance getMouseOverEdge(int x, int y, int timeStamp) {


		Enumeration en1 = getChildren();

		while(en1.hasMoreElements()) {

			// Look at each child
			EntityInstance ce = (EntityInstance) en1.nextElement();

			Enumeration en2 = ce.dstRelationElements();

			RelationInstance ri;

			while (en2.hasMoreElements()) {
				ri = (RelationInstance) en2.nextElement();

				if (ri.isNearEdge(x, y, MOUSE_NEAR_EDGE_THRESHOLD, timeStamp)) {
					return ri;
				}
			}

			en2 = ce.srcRelationElements();

			while (en2.hasMoreElements()) {
				ri = (RelationInstance) en2.nextElement();
				if (ri.isNearEdge(x, y, MOUSE_NEAR_EDGE_THRESHOLD, timeStamp)) {
					return ri;
				}
			}

			ri = ce.getMouseOverEdge(x, y, timeStamp);
			if (ri != null) {
				return ri;
			}
		}
		return null;
	}



	public void getMouseOverAllEdges(int x, int y, Vector v)
	{

		Enumeration en = dstRelationElements();

		while (en.hasMoreElements()) {
			RelationInstance ri = (RelationInstance) en.nextElement();
			if (ri.isNearEdge(x, y, MOUSE_NEAR_EDGE_THRESHOLD, -1)) {
				if (!v.contains(ri)) {
					v.addElement(ri);
			}	}
		}

		en = srcRelationElements();

		while (en.hasMoreElements()) {
			RelationInstance ri = (RelationInstance) en.nextElement();

			if (ri.isNearEdge(x, y, MOUSE_NEAR_EDGE_THRESHOLD, -1)) {
				if (!v.contains(ri)) {
					v.addElement(ri);
				}
			}
		}
	}

	public EntityInstance intersects(Layout lyt) {

		if (containsLayout(lyt)) {

			if (!containsList.isEmpty()) {

				// Check children



				Enumeration en = getChildren();



				while(en.hasMoreElements()) {

					EntityInstance e = (EntityInstance) en.nextElement();



					EntityInstance oe = e.intersects(lyt); 



					if (oe != null)

						return oe; 

				}



				return null; 

			}

			return this;  // over us 

		}

		else if (intersectsLayout(lyt)) 

			return this;



		return null; 

	}





	public EntityInstance containing(Layout lyt) {

		if (containsLayout(lyt)) {

			if (!containsList.isEmpty() && isOpen()) {

				// Check children



				Enumeration en = getChildren();



				while(en.hasMoreElements()) {

					EntityInstance e = (EntityInstance) en.nextElement();



					EntityInstance oe = e.containing(lyt);



					if (oe != null)

						return oe;

				}

			}

			return this;

		}

		return null;

	}



	public Vector getCoverList(Layout coverLyt) {

		// Return the set of children contained in layout



		Vector contents = new Vector();



		Enumeration en = getChildren();



		while(en.hasMoreElements()) {

			EntityInstance e = (EntityInstance) en.nextElement();



			Layout lyt = e.getLayout();



			if (coverLyt.x < lyt.x && lyt.x+lyt.width < coverLyt.x+width() &&

				coverLyt.y < lyt.y && lyt.y+lyt.height < coverLyt.y+height())

			{

				contents.addElement(e);

			}

		}



		return contents;

	}



	public Enumeration srcRelationElements() {

		return srcRelList.elements(); 

	}



	public Enumeration dstRelationElements() {

		return dstRelList.elements(); 

	}



	protected void addSrcRels(Vector v) {

		Enumeration en = srcRelList.elements();



		while(en.hasMoreElements()) {

			v.addElement(en.nextElement());

		}



		en = getChildren();



		while(en.hasMoreElements()) {

			EntityInstance e = (EntityInstance) en.nextElement();



			e.addSrcRels(v);

		}

	}



	public Enumeration srcRelationAbstractedElements() {

		// Return enumeration for all edges which

		// have this entity as the source (real and abstracted).



		Vector v = new Vector();



		addSrcRels(v);



		return v.elements();

	}



	protected void addDstRels(Vector v) {

		Enumeration en = dstRelList.elements();



		while(en.hasMoreElements()) {

			v.addElement(en.nextElement());

		}



		en = getChildren();



		while(en.hasMoreElements()) {

			EntityInstance e = (EntityInstance) en.nextElement();



			e.addDstRels(v);

		}

	}



	public Enumeration dstRelationAbstractedElements() {

		// Return enumeration for all edges which

		// have this entity as the destination (real and abstracted).



		Vector v = new Vector();



		addDstRels(v);



		return v.elements();

	}



	public Vector groupRegion(Layout lyt) {

		// This entity isn't in the group, but any children which

		// intersect the passed region are.



		if (containsList.isEmpty())

			return null;



		Vector grp = new Vector();



		Enumeration en = getChildren();



		while (en.hasMoreElements()) {

			EntityInstance e = (EntityInstance) en.nextElement(); 



			if (e.intersectsLayout(lyt)) {

				e.setGroupFlag(); 

				grp.addElement(e);

			}

		}



		if (grp.isEmpty())

			return null;



		return grp;

	}



	public void getGroup(Vector grp) 
	{

		if (getGroupFlag()) {
			grp.addElement(this);
		}

		if (containsList.isEmpty()) {
			return;
		}

		Enumeration en = getChildren();

		while (en.hasMoreElements()) {

			EntityInstance e = (EntityInstance) en.nextElement(); 
			e.getGroup(grp);
		}
	}


	protected void setGroupFlag() {

	   m_groupFlag = true; 
	}



	protected void clearGroupFlag() {

		m_groupFlag = false;
	}



	public boolean getGroupFlag() {

		return m_groupFlag;

	}



	protected void setGroupKeyFlag() {

		m_groupKeyFlag = true;

	}



	protected void clearGroupKeyFlag() {

		m_groupKeyFlag = false;

	}



	public boolean getGroupKeyFlag() {

		return m_groupKeyFlag;

	}



	public void getHighlightGroup(Vector grp) {

		if (highlightFlag)

			grp.addElement(this);



		if (containsList.isEmpty())

			return;



		Enumeration en = getChildren();



		while (en.hasMoreElements()) {

			EntityInstance e = (EntityInstance) en.nextElement(); 



			e.getHighlightGroup(grp);

		}

	}



	public void setHighlightFlag() {

		highlightFlag = true;

	}



	public boolean getHighlightFlag() {

		return highlightFlag;

	}



	public boolean clearHighlightFlag() {

		boolean ret = highlightFlag;
		if (ret) {
			highlightFlag = false;
		}
		return(ret);
	}

	public boolean clearAllFlags() {

		boolean ret;

		clearGroupFlag();

		clearGroupKeyFlag();

		ret		= clearHighlightFlag();

		tmpOpen = NOT_USED;



		Enumeration en = srcRelList.elements();



		while (en.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) en.nextElement();



			ret |= ri.clearHighlightFlag();

			ri.clearGroupFlag();

		}



		en = getChildren();



		while (en.hasMoreElements()) {

			EntityInstance e = (EntityInstance) en.nextElement(); 



			ret |= e.clearAllFlags();

		}
		return(ret);
	}



	public boolean clearHighlightFlags() {

		boolean ret;

		ret = clearHighlightFlag();

		Enumeration en = srcRelList.elements();

		while (en.hasMoreElements()) {

			RelationInstance ri = (RelationInstance) en.nextElement();



			ret |= ri.clearHighlightFlag();

		}

		en = getChildren();

		while (en.hasMoreElements()) {

			EntityInstance e = (EntityInstance) en.nextElement(); 

			ret |= e.clearHighlightFlags();
		}
		return(ret);
	}



	public void clearGroupFlags() {

		clearGroupFlag();

		clearGroupKeyFlag();



		Enumeration en = getChildren();



		while (en.hasMoreElements()) {

			EntityInstance e = (EntityInstance) en.nextElement(); 



			e.clearGroupFlags();

		}

	}



	public boolean isClippedLabel() 
	{
		return !fullLabel;
	}



	// Internal attributes of different entities



	public boolean isTransparent() {

		return false;

	}



	public boolean isCloseable() {

		return true;

	}



	public boolean isClickable() {

		return true;

	}



	public boolean isLabelDrawable() {

		return true;

	}



	public boolean isEnterable() {

		return true;

	}



	public void incDstCount() {

		dstCount++;

	}



	public int getDstCount() {

		return dstCount;

	}



	public void setMark(int val) {

		mark = val;

	}



	public int getMark()
	{
		return mark;
	}

	public void clearMarks() 
	{
		mark = 0;
		Enumeration en = getChildren();

		while (en.hasMoreElements()) {
			EntityInstance e = (EntityInstance) en.nextElement();
			e.clearMarks();
		}
	}

	static public Font getSmallFont() 
	{
		return smallFont;
	}

	public int x() {
		return(m_x);
	}

	public int y() {
		return(m_y);
	}

	public int width() {
		return(m_width);
	}

	public int height() {
		return(m_height);
	}

	public double parentsWidthLocal()
	{
		if (containedBy == null) {
			if (dg == null) {
				System.out.println("Null diagram\n");
			}
			return(dg.getDiagramWidth());
		} 
		return(containedBy.parentsWidthLocal() * containedBy.widthRelLocal());
	}

	public double parentsHeightLocal()
	{
		if (containedBy == null) {
			return(dg.getDiagramHeight());
		} 
		return(containedBy.parentsHeightLocal() * containedBy.heightRelLocal());
	}

	public double parentsX()
	{
		if (containedBy == null) {
			if (dg == null) {
				System.out.println("Null diagram\n");
			}
			return(dg.getDiagramX());
		} 
		return(containedBy.x());
	}

	public double parentsY()
	{
		if (containedBy == null) {
			if (dg == null) {
				System.out.println("Null diagram\n");
			}
			return(dg.getDiagramY());
		} 
		return(containedBy.y());
	}

	public double parentsWidth()
	{
		if (containedBy == null) {
			if (dg == null) {
				System.out.println("Null diagram\n");
			}
			return(dg.getDiagramWidth());
		} 
		return(containedBy.width());
	}

	public double parentsHeight()
	{
		if (containedBy == null) {
			return(dg.getDiagramHeight());
		} 
		return(containedBy.height());
	}

	/* Relative local coordinate system
     *
     * 0 same offset as top/left hand edge of parent
     * 1 same offset as bottom/right hand edge of parent
     */

	public double xRelLocal()  
	{
		return(m_xrelLocal);
	}

	public double yRelLocal() 
	{
		return(m_yrelLocal);
	}

	public double widthRelLocal() 
	{
		return(m_widthrelLocal);
	}

	public double heightRelLocal() 
	{
		return(m_heightrelLocal);
	}

	public void setxRelLocal(double xRelLocal)  
	{
		m_xrelLocal = xRelLocal;
	}

	public void setyRelLocal(double yRelLocal) 
	{
		m_yrelLocal = yRelLocal;
	}	

	public void setwidthRelLocal(double widthRelLocal) 
	{
		m_widthrelLocal = widthRelLocal;
	}

	public void setheightRelLocal(double heightRelLocal) 
	{
		m_heightrelLocal = heightRelLocal;
	}

	/* Local coordinate system (obsoleted because of numeric underflow)
	 *
	 * 0 - Same as top/left edge of parent
	 * n - As position n while parent at position m.
     */

	/* This is only called if we are using legacy coordinate system..
	 * It is called after loading coordinates
	 * and converts the old local coordinates temporarily stored into relative ones
	 */

	public void computeRelCoordinates(double xParent, double yParent, double widthParent, double heightParent)
	{
		double			xLocal, yLocal, widthLocal, heightLocal;
		Enumeration		en;
		EntityInstance	e;

		xLocal      = m_xrelLocal;
		yLocal      = m_yrelLocal;
		widthLocal  = m_widthrelLocal;
		heightLocal = m_heightrelLocal;

		if (widthParent == 0.0) {
			m_xrelLocal      = 0.0;
			m_widthrelLocal  = 0.0;
		} else {
			m_xrelLocal      = xLocal/widthParent;
			m_widthrelLocal  = widthLocal/widthParent;
		}
		if (heightParent == 0.0) {
			m_yrelLocal      = 0.0;
			m_heightrelLocal = 0.0;
		} else {
			m_yrelLocal      = yLocal/heightParent;
			m_heightrelLocal = heightLocal/heightParent;
		}

		for (en = getChildren(); en.hasMoreElements(); ) {
			e = (EntityInstance) en.nextElement();
			e.computeRelCoordinates(xLocal, yLocal, widthLocal, heightLocal);
		}
//		System.out.println("EntityInstance:computeRelCoordinates(" + xParent + ", " + yParent + ", " + widthParent + ", " + heightParent + ") = {" + m_xrelLocal + ", " + m_yrelLocal + ", " + m_widthrelLocal + ", " + m_heightrelLocal + "}");
	}

	public void setxLocal(double xLocal)  
	{
		dg.m_uses_local_coordinates = true;
		m_xrelLocal                 = xLocal;
	}


	public void setyLocal(double yLocal) 
	{
		dg.m_uses_local_coordinates = true;
		m_yrelLocal                 = yLocal;
	}

	public void setwidthLocal(double widthLocal) 
	{
		dg.m_uses_local_coordinates = true;
		m_widthrelLocal             = widthLocal;
	}

	public void setheightLocal(double heightLocal) 
	{
		dg.m_uses_local_coordinates = true;
		m_heightrelLocal            = heightLocal;
	}

	public Rectangle getBounds()
	{
		return (new Rectangle(x(), y(), (int) width(), (int) height()) );
	}

	public void	  ancilliary(Object value)
	{
		m_ancilliary = value;
	}

	public Object ancilliary()
	{
		return(m_ancilliary);
	}

	// The routines that follow hide the complexity of getting/setting attribute values
	// from EditAttributes
 
	public int getPrimaryAttributeCount()
	{
		return(13);
	}

	public String getAttributeNameAt(int index)
	{
		String	name;

		switch (index) {
		case 0:
			name  = "id";
			break;
		case 1:
			name  = "class";
			break;
		case 2:
			name  = LABEL_ID;
			break;
		case 3:
			name  = TITLE_ID;
			break;
		case 4:
			name  = DESC_ID;
			break;	
		case 5:
			name  = COLOUR_ID;
			break;
		case 6:
			name  = LABEL_COLOUR_ID;
			break;
		case 7:
			name  = OPEN_COLOUR_ID;
			break;
		case 8:
			name  = XRELPOSITION_ID;
			break;
		case 9:
			name  = YRELPOSITION_ID;
			break;
		case 10:
			name  = WIDTHREL_ID;
			break;
		case 11:
			name  = HEIGHTREL_ID;
			break;
		case 12:
			name  = FONTDELTA_ID;
			break;
		default:
			name  = super.getAttributeNameAt(index);
		}
		return(name);
	}

	public Object getAttributeValueAt(int index)
	{
		Object	value;

		switch (index) {
		case 0:
			value = getId();
			break;
		case 1:
			if (parentClass == null) {
				value = null;
			} else {
				value = parentClass.getId();
			}
			break;
		case 2:
			value = label;
			break;
		case 3:
			value = title;
			break;
		case 4:
			value = description;
			break;
		case 5:
			value = getObjectColor();
			break;
		case 6:
			value = getLabelColor();
			break;
		case 7:
			value = getObjectColorWhenOpen();
			break;
		case 8:
			value = new Double(xRelLocal());
			break;
		case 9:
			value = new Double(yRelLocal());
			break;
		case 10:
			value = new Double(widthRelLocal());
			break;
		case 11:
			value = new Double(heightRelLocal());
			break;
		case 12:
			value = new Integer(fontDelta);
			break;
		default:
			value = super.getAttributeValueAt(index);
		}
		return(value);
	}

	public void setAttributeValueAt(int index, Object value)
	{
		switch (index) {
		case 0:
			setId((String) value);
			break;
		case 1:
			if (value != null) {
				String newId = (String) value;
				if (parentClass == null || !parentClass.getId().equals(newId)) {
					Enumeration	en;
					EntityClass	ec;

					for (en = dg.enumEntityClasses(); en.hasMoreElements(); ) {
						ec = (EntityClass) en.nextElement();
						if (ec.getId().equals(newId)) {
							parentClass = ec;
							break;
			}	}	}	}
			break;
		case 2:
			label          = (String) value;
			break;
		case 3:
			title          = (String) value;
			break;
		case 4:
			description    = (String) value;
			break;
		case 5:
			setObjectColor((Color) value);
			break;
		case 6:
			setLabelColor((Color) value);
			break;
		case 7:
			setObjectColorWhenOpen((Color) value);
			break;
		case 8:
			setxRelLocal(((Double) value).doubleValue());
			break;
		case 9:
			setyRelLocal(((Double) value).doubleValue());
			break;
		case 10:
			setwidthRelLocal(((Double) value).doubleValue());
			break;
		case 11:
			setheightRelLocal(((Double) value).doubleValue());
			break;
		case 12:
			fontDelta      = ((Integer) value).intValue();
			break;
		default:
			super.setAttributeValueAt(index, value);
	}	}


	// Need to know the type in cases where value might be null
	// For example with some colors

	public int getAttributeTypeAt(int index)
	{
		int		ret;
		
		switch (index) {
		case 0:
		case 2:
		case 3:
			ret = Attribute.STRING_TYPE;
			break;
		case 1:
			ret = Attribute.ENTITY_CLASS_TYPE;
			break;
		case 4:
			ret = Attribute.TEXT_TYPE;
			break;
		case 5:
		case 6:
		case 7:
			ret = Attribute.COLOR_OR_NULL_TYPE;
			break;
		case 8:
		case 9:
		case 10:
		case 11:
			ret = Attribute.DOUBLE_TYPE;
			break;
		case 12:
			ret = Attribute.INT_TYPE;
			break;
		default:
			ret = super.getAttributeTypeAt(index);
		}
		return(ret);
	}
}
