package lsedit;
import java.applet.*;
import java.awt.*;
import java.net.*;
import java.io.*;
import java.util.*;

abstract public class LandscapeEditorCore extends LandscapeViewerCore {

	protected final static int A_HORIZ_TOP		 = 10000;
	protected final static int A_HORIZ_CENTER	 = 10001;
	protected final static int A_VERTICAL_LEFT	 = 10002; 
	protected final static int A_VERTICAL_RIGHT	 = 10003;
	protected final static int A_VERTICAL_CENTER = 10004;

	protected final static int A_FIT			 = 10010;
	protected final static int A_GROUP			 = 10013;
	protected final static int A_FIT_LABEL		 = 10014;

	protected final static int SZ_WIDTH			 = 10020;
	protected final static int SZ_HEIGHT		 = 10021;
	protected final static int SZ_WIDTH_HEIGHT	 = 10022;

	protected final static int SPC_HORIZ		 = 10030;
	protected final static int SPC_VERTICAL		 = 10031;

	protected final static int PRINT_LS			 = 19993;
	protected final static int PRINT_BOOK		 = 19994;
	protected final static int PRINT_ALL		 = 19995;

	protected final static int SAVE_RAW			 = 19996;

	protected final static int SAVE_AS			 = 19997;
	protected final static int SAVE_AS_RAW		 = 19998;

	protected final static int ABOUT_HELP		 = 19999;

	protected final static int GRID_MAX			 = 10;

	protected final static int NUM_TOOL_BUTTONS	 = 10;

	protected final static String TITLE = "Landscape Editor";

	protected final static int FA_SAVE		 = FA_LOAD+1;
	protected final static int FA_PRINT		 = FA_LOAD+2;
	protected final static int FA_PRINT_ALL	 = FA_LOAD+3;
	protected final static int FA_PRINT_BOOK = FA_LOAD+4;

	protected LayoutManager layoutManager = new LayoutManager();

	protected LandscapeModeHandler editModeHandler, viewModeHandler;

	protected Choice lsDropDown; 

	protected Vector diagrams = new Vector();
	protected Vector diagramNames = new Vector();

	/* Parameters */

	protected String lsSavePath, lsSaveSuffix, lsSaveCmd;

	protected static String aboutStr = TITLE + " " + Version.Details();

	protected String printPrompt(String banner, String prompt, String file, int mode)
	{
		Container c = this;

		do { 
			c = c.getParent(); 
		} while (!(c instanceof Frame));

		Frame f = (Frame) c;

		PrintPromptDialog.Create(this, f, banner, prompt, file, mode);

		return null;
	}

	// Driving component of quick sort

	private static final int partition( Vector v, int l, int r, int mode ) 
	{
		// Arbitrarily pick the left element as the pivot element:

		Layout lytp = ((EntityInstance) v.elementAt(l)).getLayout();

		l--; 
		r++;

		while(true) {
			// Figure out what's before and after the pivot:

			boolean f;
			do {
				r--;

				Layout lyt = ((EntityInstance) v.elementAt(r)).getLayout();

				if (mode == SPC_HORIZ) {
					f = lyt.x > lytp.x;
				} else {
					f = lyt.y > lytp.y;
				}
			} while (f);

			do {
				l++;
				Layout lyt = ((EntityInstance) v.elementAt(l)).getLayout();
				if (mode == SPC_HORIZ) {
					f = lyt.x < lytp.x;
				} else {	
					f = lyt.y < lytp.y;
				}
			} while (f);
		
			// Swap elements if we can:

			if (l < r) {
				Object o = v.elementAt(l);
				v.setElementAt(v.elementAt(r), l);
				v.setElementAt(o, r);
			} else {
				return r;
	}	}	}

	private static final void sort( Vector v, int l, int r, int mode ) 
	{
		// If we haven't reached a termination condition...

		if (l < r) {
			//	Partition the vector into left and right halves:
			int p = partition(v, l, r, mode);

			//	Recursively sort each half:
			sort(v, l,   p, mode);
			sort(v, p+1, r, mode);
		}
	}

	protected static void sortVectorByLayout(Vector v, int mode) 
	{
		sort(v, 0, v.size() - 1, mode);
	}

	protected String getTitle() 
	{
		return TITLE;
	}


	public void paintTopControls(Graphics gc) 
	{
		lsDropDown.paint(gc);
		paintToolButtons(gc);
	}

	protected void layoutTopControls() 
	{
		int xpos, ypos;

		Dimension dim;

		Insets ins = af.getContentPane().getInsets();

		xpos = ins.left + GAP;
		ypos = ins.top + GAP;

		dim = lsDropDown.size();

		lsDropDown.reshape(xpos, ypos, 100, dim.height);

		MsgOut.dprintln("lsDropDown width: " + dim.width);

		xpos += 100 + GAP;

		layoutToolButtons(xpos, ypos);
	}

	// Overrides LandscapeViewerCore.genDropDowns()
	protected void genDropDowns() {
		MyChoice c = new MyChoice();
		this.add(c);
		lsDropDown = c;
		lsDropDown.disable();
	}

	protected void genModeHandlers() {
		editModeHandler = new EditModeHandler();
		editModeHandler.init(this);

		viewModeHandler = new ViewModeHandler();
		viewModeHandler.init(this);

		modeHandler = editModeHandler;
	}

	protected int getNumToolButtons() {
		return toolButton.length;
	}

	protected void registerLsName(String name) {
		lsDropDown.addItem(name);

		if (lsDropDown.countItems() > 1) 
			lsDropDown.enable();
	}


	protected void setDiagram(ScrollableDiagram dg) {
		this.dg = dg;

		modeHandler.select(dg);
		dg.setDrawBends(!(modeHandler instanceof ViewModeHandler));

		lsDropDown.select(dg.getName());
		fillTOC();
		layoutGUI();
		repaint();

		doFeedback("Set to: " + dg.getName());

		setLeftBox();
	}

	protected void setDiagram(int ind) {
		ScrollableDiagram newDg = (ScrollableDiagram) diagrams.elementAt(ind);

		if (newDg == dg)
			return;

		setDiagram(newDg);
	}

	protected void addDiagram(ScrollableDiagram newDg, String path) 
	{
		diagrams.addElement(newDg);
		diagramNames.addElement(path);

		registerLsName(newDg.getName());

		setDiagram(newDg);
		newDg.setToViewport();
	}

	public void attach(String lsPath) 
	{
		int					ind;
		ScrollableDiagram	dgLink;

		ind = diagramNames.indexOf(lsPath);
		if (ind < 0) {
			dgLink = new ScrollableDiagram(this, dgScrollBarRight, dgScrollBarBottom);

			String rc = dgLink.loadDiagram(lsPath, null, null);
		  
			if (rc != null) {
				error("Attach failed: " + rc);
				return;
			}
			addDiagram(dgLink, lsPath);
		}
		else {
			setDiagram(ind);
		}
	}

	public void followLink(String url, int target) {
		if (target == LsLink.TARGET_APP)
			attach(url);
		else
			showURL(url, target);
	}

	protected void loadLs(String file) 
	{
		if (file != null && file.length() > 0) {
			lsPath = file;

			doFeedback("Reading: " + lsPath);

			ScrollableDiagram newDg = new ScrollableDiagram(this, dgScrollBarRight, dgScrollBarBottom);

			String rc = newDg.loadDiagram(lsPath, null, null);

			if (rc != null) {
				error("Failed to load (" + rc + ")");
			} else {
				addDiagram(newDg, lsPath);
			}
		}

		this.requestFocus();
		repaint();
	}

	protected void fitChildren() {
		dg.saveForUndo();
		dg.fitTo(dg.getRoot());
		doFeedback("Children fitted to: " + dg.getRoot().getLabel());
		dg.rescaleDiagram();
		repaintDg();
	}

	protected void fitToLabel() {
		Vector v = startGroupOp();

		if (v == null)
			return;
		
		int H_MARGIN = 8; 
		int V_MARGIN = 4;

		Enumeration en = v.elements();

		while (en.hasMoreElements()) {
			EntityInstance e = (EntityInstance) en.nextElement();

			Dimension ld = 
				e.getLabelDim(getGraphics(), EntityInstance.REG_FONT);

			Layout curLayout = e.getLayout();

			curLayout.width = (double) (ld.width+H_MARGIN*2);
			curLayout.height = (double) (ld.height+V_MARGIN*2);

			dg.saveForUndo();
			e.setLayout(curLayout);
		}

		dg.rescaleDiagram();
		redrawDg();
	}

	public void doLayout(int menuIndex) {
		dg.saveForUndo();

		String rmsg = layoutManager.executeLayout(menuIndex, dg);

		doFeedback(rmsg);

		dg.rescaleDiagram();
		repaintDg();
	}

	protected Vector startGroupOp() {
		dg.saveForUndo();

		Vector grp = dg.getGroup();

		if (grp == null) {
			error("Group not selected");
			return null;
		}

		return grp;
	}

	protected void treeGroup() {
		Vector grp = startGroupOp();

		if (grp == null) {
			return;
		}

		RelationClass rc = null;

		Enumeration en = dg.enumRelations();

		while (en.hasMoreElements()) {
			RelationClass arc = (RelationClass) en.nextElement();

			if (arc.isActive() && !dg.excludeReln(arc) ) {
				if (rc != null) {
					error("Only one relation must be selected");
					return;
				}
				rc = arc;
			}
		}

		System.out.println(rc.getLabel());

		DagGridLayout tree = new DagGridLayout(grp, rc);

		int numLevels = tree.numLevels();
		int maxBreadth = tree.maxBreadth();

		System.out.println(numLevels + " " + maxBreadth);

		if (numLevels == 0) {
			error("Group is not a DAG");
			return;
		}

		if (numLevels == 1) {
			error("Only single level found");
			return;
		}

		Layout plyt = 
			((EntityInstance) grp.elementAt(0)).getParent().getLayout();

		double sxpos = plyt.x + plyt.width/16;
		double dx	 = (plyt.width - plyt.width/8)/(maxBreadth+1);

		double ypos	 = plyt.y + plyt.height/16;
		double dy	 = (plyt.height - plyt.height/8)/(numLevels+1);

		for (int i=0; i<numLevels; i++) {
			double xpos = sxpos;

			for (int j=0; j<maxBreadth; j++) {
				EntityInstance e = tree.getGrid(i, j);

				if (e != null) {
					Layout lyt = e.getLayout();

					lyt.x = xpos;
					lyt.y = ypos;

					e.setLayout(lyt);
				}

				xpos += dx;
			}
		
			ypos += dy;
		}		

		doFeedback("Laid out group as tree");
		dg.rescaleDiagram();
		repaintDg();
	}

	// Alignment

	protected void align(int alignment) {
		Vector grp = startGroupOp();

		if (grp == null) {
			return;
		}

		// Align group based on alignment to key entity

		EntityInstance ke = dg.getKeyEntity();
		Layout klyt = ke.getLayout();

		double cx = klyt.x + klyt.width/2;
		double cy = klyt.y + klyt.height/2;

		Enumeration en = grp.elements();

		while (en.hasMoreElements()) {
			EntityInstance e = (EntityInstance) en.nextElement();

			if (e != ke) {
				Layout elyt = (Layout) e.getLayout().clone();

				switch(alignment)
				{
				case A_HORIZ_TOP:
					elyt.y = klyt.y;
					break;

				case A_HORIZ_CENTER:
					elyt.y = cy - elyt.height/2;
					break;

				case A_VERTICAL_LEFT:
					elyt.x = klyt.x;
					break;

				case A_VERTICAL_RIGHT:
					elyt.x += klyt.x + klyt.width - elyt.width;
					break; 

				case A_VERTICAL_CENTER:
					elyt.x = cx - elyt.width/2;
					break;

				default:
					MsgOut.println("Illegal allignment option");
					break;
				}

				e.setLayout(elyt);		
			}
		}

		dg.rescaleDiagram();
		repaintDg();
	}

	protected void sameSize(int sizeDim) {
		Vector grp = startGroupOp();

		if (grp == null)
			return;

		// Size group based on size of key entity

		EntityInstance ke = dg.getKeyEntity();
		Layout klyt = ke.getLayout();

		Enumeration en = grp.elements();

		while (en.hasMoreElements()) {
			EntityInstance e = (EntityInstance) en.nextElement();

			if (e != ke) {
				Layout elyt = (Layout) e.getLayout().clone();

				switch(sizeDim)
				{
				case SZ_WIDTH:
					elyt.width = klyt.width;
					break; 

				case SZ_HEIGHT:
					elyt.height = klyt.height;
					break;

				case SZ_WIDTH_HEIGHT:
					elyt.width = klyt.width;
					elyt.height = klyt.height;
					break;
		
				}

				e.setLayout(elyt);
			}
		}

		dg.rescaleDiagram();
		repaintDg();
	} 

	protected void equalSpacing(int mode) {
		Vector grp = startGroupOp();

		if (grp == null) {
			return;
		}

		if (grp.size() < 3) {
			error("Minimum group of three required");
			return;
		}

		Layout bb = dg.getGroupBoundingBox();

		if (bb == null) {
			error("Group not selected");
			return;
		}

		// Sort entities based on vertical or horizontal position

		sortVectorByLayout(grp, mode);

		// Calculate sep

		double tw, th;
		double sep = 0.0;

		Enumeration en = grp.elements();

		tw = th = 0;

		while(en.hasMoreElements()) {
			EntityInstance e = (EntityInstance) en.nextElement();

			Layout lyt = e.getLayout();

			tw += lyt.width;
			th += lyt.height;
		}

		if (mode == SPC_HORIZ) {
			sep = (bb.width - tw)/(grp.size()-1);
		}
		else {
			sep = (bb.height - th)/(grp.size()-1);
		}

/*
		// If overlapping would occur (-ve delta) abort

		if (sep) {
			error("
		}
*/

		// Apply new layouts to second through penultimate entitites

		EntityInstance ce = (EntityInstance) grp.elementAt(0);
		Layout clyt = ce.getLayout();

		for (int i = 1; i < grp.size()-1; i++) {
			EntityInstance e = (EntityInstance) grp.elementAt(i);

			Layout lyt = e.getLayout();

			if (mode == SPC_HORIZ) {
				lyt.x = clyt.x + clyt.width + sep;
			}
			else {
				lyt.y = clyt.y + clyt.height + sep;
			}

			clyt = (Layout) lyt.clone();
			e.setLayout(lyt);
		}

		dg.rescaleDiagram();
		repaintDg();
	}

	protected void createContainedGroup() {

		Vector grp = startGroupOp();

		if (grp == null) {
			return;
		}

		EntityInstance pe = ((EntityInstance) grp.firstElement()).getParent();

		Layout nlyt = dg.getGroupBoundingBox();

		nlyt.y -= RelationInstance.CLIENT_SUPPLIER_EL_LEN+5;
		nlyt.x -= 5;
		nlyt.width += 10;
		nlyt.height += RelationInstance.CLIENT_SUPPLIER_EL_LEN+10;

		String cname;
		int n = 0;

		do {
			cname = "Container#" + n;
			n++;
		} while(dg.entityExists(cname));

		EntityInstance ne = dg.addEntity(cname, nlyt, pe);

		Enumeration en = grp.elements();

		while (en.hasMoreElements()) {
			EntityInstance e = (EntityInstance) en.nextElement();

			Layout lyt = e.getLayout();

			e.moveEntityContainment(ne);

			e.setLayout(lyt);		
		}

		dg.rescaleDiagram();
		repaintDg();
	}
		
	protected Vector clipboard = null;
	protected ScrollableDiagram clipboardDg;
	protected boolean clipboardDump = false;

	protected void cutGroup() {
		clipboard = dg.getGroup();

		if (clipboard == null) {
			error("Group not selected");
		}
		else {
			clipboardDg = dg;
			doFeedback("Group copied to clipboard");
		}

		Enumeration en = clipboard.elements();

		while(en.hasMoreElements()) {
			EntityInstance e = (EntityInstance) en.nextElement();

			dg.removeEntity(e);
		}

		repaintDg();
	}

	protected void pasteGroup() {
		if (clipboard == null || clipboard.size() == 0) {
			error("Clipboard empty");
		}
		else {
			clipboardDump = true;
			setCursor(Frame.MOVE_CURSOR);
			doFeedback("Select parent entity for clipboard group");
		}
	}

	protected void newClipboardEntities(EntityInstance pe, int x, int y) {
		Enumeration en = clipboard.elements();

		Vector empty = new Vector();
		Layout plyt = pe.getLayout();
		int cnt = 0;

		int GAP = 4;
		int SIZE = 8;

		while(en.hasMoreElements()) {
			EntityInstance e = (EntityInstance) en.nextElement();

			Layout nlyt = new Layout(x + GAP + (SIZE+GAP)*cnt,
										y + GAP*2, SIZE, SIZE);

			EntityClass ec = e.getEntityClass();

			dg.addEntity(e.getId(), e.getLabel(), e.getDescription(),
						 nlyt, pe, ec, empty);
			cnt++;
		}
	}

	protected void dumpClipboard(EntityInstance pe, int x, int y) {

		// Make sure no item in clipboard isn't already in destination diagram.

		Enumeration en = clipboard.elements();

		while(en.hasMoreElements()) {
			EntityInstance ne = (EntityInstance) en.nextElement();

			if (dg.containsEntity(ne)) {
				error("Landscape already contains item in clipboard");
				return;
			}
		}

		EntityInstance e = (EntityInstance) clipboard.firstElement();

		if (dg == e.getDiagram()) {
			en = clipboard.elements();

			while(en.hasMoreElements()) {
				e = (EntityInstance) en.nextElement();

				pe.addContainment(e);
			}
		}
		else {
			newClipboardEntities(pe, x, y);
		}

		redrawDg();
	}

	public boolean mouseDown(Event ev, int x, int y) {

//		System.out.println("LandscapeEditorCore mousedown\n");

		if (dg == null)
			return false;

		if (clipboardDump) {
			int adjx = dg.adjustX(x);
			int adjy = dg.adjustY(y);

			EntityInstance e = dg.mouseOverEx(adjx, adjy);

			if (e == null) {
				error("Can't put entities there");
				return true;
			}

			dumpClipboard(e, adjx, adjy);

			clipboardDump = false;

			setCursor(Frame.DEFAULT_CURSOR);
			if (modeHandler != null)
				modeHandler.mouseMotion(e, adjx, adjy);

			return true;
		}
		else {
			return super.mouseDown(ev, x, y);
		}
	}

	public boolean mouseMove(Event ev, int x, int y) {
		if (dg == null)
			return false;

		int adjx = dg.adjustX(x);
		int adjy = dg.adjustY(y);

		EntityInstance e = dg.mouseOverEx(adjx, adjy);
		RelationInstance ri = dg.mouseOverEdge(e, adjx, adjy);

		if (clipboardDump) {
			showDescription(e, ri, false);
		}
		else {
			super.mouseMove(ev, x, y);
		}
		return true;
	}

	protected void saveStatus(String stat) {
		doFeedback("Save status: " + stat);
	}


	// Save the landscape in the URL

	protected String doSaveByURL(String lsSaveURL) {
		URL lsURL;

		try {
			lsURL = new URL(lsSaveURL);
		}
		catch(MalformedURLException e) {
		   return "Malformed URL on write";
		}

		URLConnection urlCon;

		try {
			urlCon = lsURL.openConnection();

			urlCon.setDoInput(false);
			urlCon.setDoOutput(true);
			urlCon.setAllowUserInteraction(false);
		}
		catch (IOException e) {
			return "Couldn't open connection";
		}

		OutputStream os;

		try {
			os = urlCon.getOutputStream();
		}
		catch (IOException e) {
			return "Couldn't openStream";
		}

		try {
			dg.saveDiagram(os, false);
		}
		catch (IOException e) {
			return "IOException on write";
		}

		return null;
	}

	public void saveByURL(String overridePath) {
		MsgOut.dprintln("Save by URL");

		URL lsURL;

		String str = doSaveByURL(lsSavePath);

		if (str != null) {
			saveStatus(str);
			return;
		}

		if (lsSaveCmd == null) {
			saveStatus("Success");
			return;
		}

		try {
			lsURL = new URL(lsSaveCmd);
		}
		catch(MalformedURLException e) {
		   saveStatus("Malformed URL on write");
		   return;
		}

		InputStream is;
		try {
			is = lsURL.openStream();
		}
		catch (IOException e) {
			saveStatus("Couldn't openStream for store");
			return;
		}

		try {
			DataInputStream dis = new DataInputStream(is);

			str = dis.readLine();

			is.close();

			saveStatus(str);
		}
		catch(IOException e) {
			saveStatus("IOexception on store");
		}
	}

	protected String saveByFile(String newPath) {
		try {
			File file = (File) dg.getContext();

			if (newPath == null) {
				String name = file.getPath();

				file.renameTo(new File(name + ".old"));
			}
			else {
				// New filename
				file = new File(newPath);
				dg.setContext(file);
			}

			FileOutputStream os = new FileOutputStream(file);

			dg.saveDiagram(os, false);
			return null;
		}
		catch(IOException e) {
			return "IOException on file open";
		}
	}

	protected String saveByFileRaw(String newPath) {
		try {
			File file = (File) dg.getContext();

			if (newPath == null) {
				String name = file.getPath();

				if (name.endsWith(".raw"))
					file.renameTo(new File(name + ".old"));
				else
					file = new File(name + ".raw");
			}
			else {
				// New filename
				file = new File(newPath + ".raw");
				dg.setContext(file);
			}

			FileOutputStream os = new FileOutputStream(file);

			dg.saveDiagramRaw(os);
			return null;
		}
		catch(IOException e) {
			return "IOException on file open";
		}
	}

	protected String saveWithCmd(String saveSuffix, String cmdTemplate)
	{
		File file = (File) dg.getContext();

		String path = file.getPath() + saveSuffix;

		File nfile = new File(path);

		// Save away the monolithic TA file

		try {

			FileOutputStream os = new FileOutputStream(nfile);

			dg.saveDiagram(os, true);
		}
		catch(IOException e) {
			return "IOException on file save";
		}
	
		// Run command after save

		String cmd = Util.expand(cmdTemplate, file.getName(), this);

		// System.out.println("Save command: " + cmd);

		String rc;

		try {
			Runtime rt = Runtime.getRuntime();

			Process p = rt.exec(cmd);

			DataInputStream es = 
				new DataInputStream(p.getErrorStream());

			rc = es.readLine();

			if (rc.equals("Done"))
				rc = "Save succeeded";

			p.destroy();
		}
		catch (Exception e) {
			rc = "Exec failed for: " + cmd;
		}

		return rc;
	}

	protected void doSaveLs(String overridePath) {
		doFeedback("Saving landscape...");

		if (dg.getContext() instanceof URL) {
			saveByURL(overridePath);
		}
		else {
			String rc;

			if (overridePath == null && lsSaveCmd != null) {
				MsgOut.dprintln("Save by app");
				rc = saveWithCmd(lsSaveSuffix, lsSaveCmd);
			}
			else {
				MsgOut.dprintln("Save by file");
				rc = saveByFile(overridePath); 
			}

			if (rc == null)
				rc = "Success";

			saveStatus(rc);
		}
	}

	protected void saveLs() {
		Object context = dg.getContext();
		String txt = null;

		if (context == null) {

			txt = filePrompt("Save Landscape", dg.getAbsolutePath(), FA_SAVE);
			if (txt.length() == 0) {
						return;
				}
		}

		doSaveLs(null);
	}

	protected void doSaveLsRaw(String overridePath) {
		doFeedback("Saving landscape in raw format...");

		if (dg.getContext() instanceof URL) {
			// saveByURL(overridePath);
		}
		else {
			MsgOut.dprintln("Save by file");

			String rc = saveByFileRaw(overridePath); 

			if (rc == null)
				rc = "Success";

			saveStatus(rc);
		}
	}

	protected void saveLsRaw() {
		Object context = dg.getContext();
		String txt = null;

		if (context == null) {
			txt = filePrompt("Save Landscape Raw", dg.getAbsolutePath(), FA_SAVE);

			if (txt.length() == 0)
				return;
		}

		doSaveLsRaw(null);
	}

	private static boolean _printing   = false;
	private static boolean _web		   = false;
	private static boolean _landscape  = false;
	private static boolean _blackwhite = false;

	private FileOutputStream pos;

	public static boolean isPrinting() {
		return _printing;
	}

	/**
	 * Create a printer graphics (for paper) using PSGr.
	 */
	private PSGr getPaperGraphics( String psPath ) throws IOException {
		Rectangle dr;	// diagram rectangle
		Rectangle pr;	// print rectangle
		Rectangle bb;	// bounding box
		float	  xScale;
		float	  yScale;
		float	  xyScale;
		PSGr	  g;

		dr = dg.bounds();
							
		// scale the printing to fit on a page;

		float wPage = PSGr.PAGEWIDTH - PSGr.PAGEMARGIN*2;
		float hPage = PSGr.PAGEHEIGHT - PSGr.PAGEMARGIN*2;

		if (_landscape) {
			xScale = hPage / (float)dr.width;
			yScale = wPage / (float)dr.height;
		}
		else {
			xScale = wPage / (float)dr.width;
			yScale = hPage / (float)dr.height;
		}

		xyScale = yScale;
	
		if (xScale < yScale) {
			xyScale = xScale;
		}

		if (xyScale > 1) {
			xyScale = 1;
		}

		pos = new FileOutputStream(psPath);

		g = new PSGr( pos, getGraphics());

		// Center it in page

		if (_landscape) {
			int mx = (int) (wPage - dr.height*xyScale)/2;
			int my = (int) (hPage - dr.width*xyScale)/2;

			int tx = (int) -(dr.x * xyScale);
			int ty = (int) -(dr.y * xyScale);

			// g.translate(tx, ty);
			g.translate(-71, 100);

			g.setBoundingBox( new Rectangle(mx, my, 
							(int)(dr.height * xyScale + 3), 
							(int)(dr.width * xyScale + 2) ) );
		}
		else {
			int mx = (int) (wPage - dr.width*xyScale)/2;
			int my = (int) (hPage - dr.height*xyScale)/2;

			int tx = (int) -(dr.x * xyScale) + mx;
			int ty = (int) -(dr.y * xyScale) + my;

			g.translate(tx, ty);

			g.setBoundingBox( new Rectangle(mx, my,
							(int)(dr.width * xyScale + 2),
							(int)(dr.height * xyScale + 3) ) );
		}

		g.setModes(_landscape, _blackwhite, -1);
		g.scale( xyScale, xyScale );

		return g;
	}

	private PSGr getBrowserGraphics( String psPath ) throws IOException {
		Rectangle dr;	// diagram rectangle
		Rectangle pr;	// print rectangle
		Rectangle bb;	// bounding box
		float	  xScale;
		float	  yScale;
		float	  xyScale;
		PSGr	  g;

		dr = dg.bounds();
		pr = new Rectangle(0, 0, PSGr.PAGEWIDTH, PSGr.PAGEHEIGHT);
							
		// scale the printing to fit on a page;
		// based on PSGr.scalePaint()

		xScale = pr.width / (float)dr.width;
		yScale = pr.height / (float)dr.height;
		xyScale = yScale;
	
		if (xScale < yScale) {
			xyScale = xScale;
		}

		if (xyScale > 1) {
			xyScale = 1;
		}

		pos = new FileOutputStream(psPath);

		g = new PSGr( pos, getGraphics(),
						0, //PSGr.PAGEMARGIN,
						PSGr.PAGEWIDTH, PSGr.PAGEHEIGHT);

		g.setModes(_landscape, _blackwhite, (int) (dr.height * xyScale - 4));

		// Push it to the top of page, with no margins

		int tx = (int) -(dr.x * xyScale);
		int ty = (int) -(dr.y * xyScale);

		g.translate(tx, ty);

		g.setBoundingBox( new Rectangle(0, 0,
							(int)(dr.width * xyScale + 2),
							(int)(dr.height * xyScale + 3) ) );

		g.scale( xyScale, xyScale );
		return g;
	}

	/**
	 * Print the diagram to a postscript file using the PSGr class
	 * PSGr is (C) 1996 E.J. Friedman-Hill and Sandia National Labs.
	 */
	protected boolean doPrint(String psPath) {
		PSGr	  g;
		int		  dx, dy;
		float	  scale;

		boolean success = true;

		BaseEntity.set3Dlook(!_blackwhite);

		try {
			_printing = true;

			if (_web)
				g = getBrowserGraphics(psPath);
			else
				g = getPaperGraphics(psPath);

			dg.draw( g );
			g.showpage();
			g.dispose();
			g.writeTrailer();

		} catch( Exception e ) {
			System.err.println("Printing..oops: " + e );
			e.printStackTrace();
			success = false;
		} finally {
			_printing = false;
		}

		try {
			pos.close();
		}
		catch( Exception e) {
			System.err.println("Printing..oops: " + e );
			success = false;
		}

		BaseEntity.set3Dlook(true);

		return success;
	}

	protected void printLs(String psPath) {
		doFeedback("Printing landscape to: " + psPath );

		boolean rc = doPrint(psPath);

		if (rc)
			doFeedback("Done printing (success)" );
		else
			error("Done printing (failed)" );
	}

	protected String getLsPath(EntityInstance e) {

		if (e == dg.getTopInstance()) {
			return Util.mungeName(e.getLabel());
		}

		Vector v = new Vector();

		EntityInstance ce = e;

		String path = "";

		do {
			v.addElement(Util.mungeName(ce.getLabel()));
			do {
				ce = ce.getParent();			
			} while (ce != null && !ce.isEnterable());

		} while (ce != null);

		for (int i = v.size()-2; i >= 0; i--) {
			String name = (String) v.elementAt(i);

			path += name;

			if (i > 0) {
				path += "#";
			}
		}

		return path;
	}

	protected EntityInstance 
	printAllLs(String dir, EntityInstance e, int minNumChildren)
	{
		if (e.numChildren() < minNumChildren)
			return null;

		if (e.isEnterable()) {
			dg.navigateTo(e);

			String psPath = dir + File.separatorChar + 
								getLsPath(dg.getRoot()) + ".eps";

			doFeedback("Printing: " + e.getLabel() );

			if (!doPrint(psPath))
				return e;
		}

		Enumeration en = e.getChildren();

		while (en.hasMoreElements()) {
			EntityInstance ce = (EntityInstance) en.nextElement();

			EntityInstance fe = printAllLs(dir, ce, minNumChildren);

			if (fe != null)
				return fe;
		}

		return null;
	}

	protected void printAllLs(String dir) {
		EntityInstance oldRoot = dg.getRoot();

		EntityInstance top = dg.getTopInstance();

		EntityInstance fe = printAllLs(dir, top, 2);

		if (fe != null) {
			error("Print to landscape failed on: " + fe.getLabel());
		}
		else {
			doFeedback("Done printing (success)" );
		}

		dg.navigateTo(oldRoot);
	}

	protected void printBook(String psPath) {
	}

	public void dialogAction(Object[] results, int mode) {
		String txt = (String) results[0];

		if (txt.length() != 0) {
			switch(mode)
			{
			case FA_SAVE:
				doSaveLs(txt);
				break;

			case FA_LOAD:
				loadLs(txt);
				break;

			case FA_PRINT:
				_web = ((Boolean) results[1]).booleanValue();
				_landscape = ((Boolean) results[2]).booleanValue();
				_blackwhite = ((Boolean) results[3]).booleanValue();

				printLs(txt);
				break;

			case FA_PRINT_ALL:
				_web = ((Boolean) results[1]).booleanValue();
				_landscape = ((Boolean) results[2]).booleanValue();
				_blackwhite = ((Boolean) results[3]).booleanValue();

				printAllLs(txt);
				break;
			}
		}
	}

	
	protected void initialLoad() {
		super.initialLoad();

		if (lsPath != null) {
			diagrams.addElement(dg);
			diagramNames.addElement(lsPath);
			registerLsName(dg.getName());
		}
		else {
			diagrams.addElement(dg);
			diagramNames.addElement("");
			registerLsName("unnamed");
		}
	}

	// --------------
	// Event handling 
	// --------------

	public boolean action(Event event, Object what) {
		if (super.action(event, what))
			return true;

		if (event.target == lsDropDown) {
			int ind = lsDropDown.getSelectedIndex();
			setDiagram(ind);
			this.requestFocus();
		}
		return false;
	}

	// Overrides LandscapeViewerCore.processKey() 
	public boolean processKey(int key, int modifiers) {

		if (super.processKey(key, modifiers))
			return true; 

		String str; 

		switch(key)
		{
		case CTRL.S:
			if (dg.isReadOnly()) {
				error("Diagram is read-only");
				return true;
			}
			saveLs();
			return true; 

		case SAVE_RAW:
			if (dg.isReadOnly()) {
				error("Diagram is read-only");
				return true;
			}
			saveLsRaw();
			return true;

		case SAVE_AS:
			{
			Object context = dg.getContext();
			String txt;

			if (context != null) {
				txt = filePrompt("Save Landscape As", dg.getAbsolutePath(), FA_SAVE);
			}
			else {
				txt = filePrompt("Save Landscape As", "", FA_SAVE);
			}

			if (txt == null || txt.length() == 0)
				return true;

			doSaveLs(txt);
			}
			return true;


		case PRINT_LS:
			{
			Object context = dg.getContext();
			String path, txt;

			String title = "Print Landscape Page to PostScript .eps File";

			if (context != null) {
				path = dg.getDir() + File.separatorChar + Util.mungeName(dg.getRoot().getLabel() + ".eps");
			} else {
				path = Util.mungeName(dg.getRoot().getLabel() + ".eps");
			}

			txt = filePrompt("Save to eps file:", path, FA_SAVE);

//			txt = printPrompt(title, "Print to file:", path, FA_PRINT);

			if (txt == null || txt.length() == 0)
				return true;

			printLs(txt);
			}
			return true;

		case PRINT_BOOK:
			{
			// Currently disabled
			Object context = dg.getContext();
			String path, txt;

			String title = "Print Landscape Book to PostScript .eps File";

			if (context != null) {
				path = dg.getDir() + File.separatorChar + 
						Util.mungeName(dg.getTopInstance().getLabel() + ".eps");
			}
			else {
				path = Util.mungeName(dg.getTopInstance().getLabel() + ".eps");
			}

			txt = printPrompt(title, "Print to file:", path, FA_PRINT_BOOK);

			if (txt == null || txt.length() == 0)
				return true;

			printBook(txt);
			}
			return true;

		case PRINT_ALL:
			{
			// Currently disabled
			Object context = dg.getContext();
			String path, txt;

			String title = "Print All Landscapes to PostScript .eps Files";

			if (context != null) {
				path = dg.getDir();
			}
			else {
				path = "";
			}

			txt = printPrompt(title, "Print in dir:", path, FA_PRINT_ALL);

			if (txt == null || txt.length() == 0)
				return true;

			printAllLs(txt);
			}
			return true;



		case 'V':
			if (modeHandler == editModeHandler) {
				doFeedback("Set to viewer mode");
				modeHandler = viewModeHandler;
				dg.setDrawBends(false);
				setMenuCheck('V', true);
			}
			else {
				doFeedback("Set to editor mode");
				modeHandler = editModeHandler;
				dg.setDrawBends(true);
				setMenuCheck('V', false);
			}
			modeHandler.select(dg);
			repaintDg();
			return true;

		case A_HORIZ_TOP:
		case A_HORIZ_CENTER:
		case A_VERTICAL_LEFT:
		case A_VERTICAL_RIGHT:
		case A_VERTICAL_CENTER:
			align(key);
			return true;

		case A_GROUP:
			createContainedGroup();
			return true;

		case A_FIT:
			fitChildren();
			return true;

		case A_FIT_LABEL:
			fitToLabel();
			return true;

		case SZ_WIDTH:
		case SZ_HEIGHT:
		case SZ_WIDTH_HEIGHT:
			sameSize(key);
			return true;

		case SPC_HORIZ:
		case SPC_VERTICAL:
			equalSpacing(key);
			return true;

		case ABOUT_HELP:
			OkMsgBox.Create(this, "About Landscape Editor", aboutStr);
			return true;

		case CTRL.X:
			cutGroup();
			return true;

		case CTRL.V:
			pasteGroup();
			return true;

		case 'g':
			{
			int grid = dg.getGrid();

			if (grid == 1) {
				error("Grid already set to minimum (1 pixel)");
			}
			else {
				int ng = grid-1;

				dg.setGrid(ng);
				doFeedback("Grid set to " + ng + " pixel(s)");
			}
			}
			return true;

		case 'G':
			{
			int grid = dg.getGrid();

			if (grid == GRID_MAX) {
				error("Grid already set to maximum (" + grid + " pixels)");
			}
			else {
				int ng = grid+1;

				dg.setGrid(ng);
				doFeedback("Grid set to " + ng + " pixels");
			}
			}
			return true;


		case CTRL.O:
			{
			Object context = dg.getContext();
			String txt;

			if (context != null) {
						txt = filePrompt( "Load Landscape", dg.getAbsolutePath(), FA_LOAD);
			}
			else {
						txt = filePrompt("Load Landscape", "", FA_LOAD);
			}

			loadLs(txt);

			}
			return true;
		}
		return false; 
	}

	// Overrides LandscapeViewer.isReadWrite()
	public boolean isreadWrite() {
		// Eventually this should be changed to return true
		// iff the landscape is definitely writeable
		// For now, assume true in editor

		return true;
	}

	public InternalBufferStream getInternalBufferStream() {
		return null;
	}
} 
