package lsedit;

import java.awt.*;
import java.util.*;
import java.io.*; 

public abstract class LandscapeObject extends Object {

	protected static int edge_mode = Diagram.BEST_EDGE;

	protected Diagram dg;		// Diagram which contains it

	public final static String COLOUR_ID =			 "color";
	public final static String LABEL_COLOUR_ID =	 "labelcolor";
	public final static String OPEN_COLOUR_ID =		 "opencolor";

	protected final static int MAX_RGB = 255; 

	protected String id, label, description, title;

	protected int m_nid = -1;

	private  Color		colourWhenOpen = null;
	private  Color		colour         = null;
	private  Color		labelColour    = null;
	private  int		style          = -1;		// No style

	// Attribute database

	protected Hashtable attributes = new Hashtable(5);

	protected void saveForUndo(LandscapeObject undoObject) 
	{
		undoObject.id = id;
		undoObject.label = label; 
		undoObject.dg = dg;
		undoObject.setNid(getNid());
		undoObject.attributes = (Hashtable) attributes.clone(); 
	}

	protected void undo(LandscapeObject undoObject) 
	{
		id = undoObject.id;
		label = undoObject.label; 
		dg = undoObject.dg;
		setNid(undoObject.getNid());
		attributes = undoObject.attributes; 
	}

	protected String qt(String str) 
	{
		return AttributeValueItem.qt(str);
	}

	abstract public boolean hasId(String id);

	// --------------
	// Public methods 
	// --------------

	public LandscapeObject derivedFrom(int i)
	{
		return(null);
	}

	public void regRawAttribute(Vector v, Hashtable st, String id, Object type, Object val)
	{
		Object[] ent = new Object[3];

		ent[0] = st.get(id);
		ent[1] = type;
		ent[2] = val;

		v.addElement(ent);
	}

	public void regRawAttribute(Vector v, Hashtable st, String id, int val) 
	{
		regRawAttribute(v, st, id, Attribute.INT, new Integer(val));
	}

	public void regRawAttribute(Vector v, Hashtable st, String id, double val) 
	{
		regRawAttribute(v, st, id, Attribute.DOUBLE, new Double(val));
	}

	public void regRawAttribute(Vector v, Hashtable st, String id, String val) 
	{
		regRawAttribute(v, st, id, Attribute.STRING, val);
	}

	public void regRawAttribute(Vector v, Hashtable st, String id, int[] val)
	{
		regRawAttribute(v, st, id, Attribute.INT_LIST, val);
	}

	public void regRawAttribute(Vector v, Hashtable st, String id, double[] val)
	{
		regRawAttribute(v, st, id, Attribute.DOUBLE_LIST, val);
	}

	public void regRawAttribute(Vector v, Hashtable st, String id, String[] val)
	{
		regRawAttribute(v, st, id, Attribute.STRING_LIST, val);
	}

	public void regAttributeRaw(Vector v, Hashtable st, Attribute attr, boolean classType)
	{
		if (attr.avi == null) {
			if (classType) {
				regRawAttribute(v, st, attr.id, Attribute.NULL, null);
			}
		}
		else if (classType || !attr.cloneOnAssign) {
			if (attr.avi.next != null) {
				AttributeValueItem avi = attr.avi;

				int num = 0;
				while(avi != null) {
					avi = avi.next;
					num++;
				}
				String[] sl = new String[num];
				avi = attr.avi;
				for (int i=0; i<num; i++) {
					sl[i] = avi.value;
				}
				regRawAttribute(v, st, attr.id, sl);
			} else {
				regRawAttribute(v, st, attr.id, attr.avi.value);
			}
		}
	}

	public void regRawAttribute(Vector v, Hashtable st, String id, Color c) 
	{
		int red = c.getRed();
		int green = c.getGreen();
		int blue = c.getBlue();

		double[] dl = new double[3];

		dl[0] = ((double) red)/((double) MAX_RGB);
		dl[1] = ((double) green)/((double) MAX_RGB);
		dl[2] = ((double) blue)/((double) MAX_RGB);

		regRawAttribute(v, st, id, dl);
	}

	// If a new attribute, add it to the attribute database,
	// otherwise merely update the value.

	public boolean processFirstOrder(Attribute attr) 
	{
		if (attr.id.equals(COLOUR_ID)) {
			colour = attr.parseColour();
			return true;
		}

		if (attr.id.equals(LABEL_COLOUR_ID)) {
			labelColour = attr.parseColour();
			return true;
		}

		if (attr.id.equals(OPEN_COLOUR_ID)) {
			colourWhenOpen = attr.parseColour();
			return true;
		}
		return false;
	}



	public void addAttribute(Attribute attr) {

		if (processFirstOrder(attr)) { 
			return;
		}

		Attribute curAttr = (Attribute) attributes.get(attr.id);

		if (curAttr == null) {
			// New attribute
			attributes.put(attr.id, attr);
		} else if (attr.avi != null) {

			// Replace any old attributes with overrides
			attributes.remove(curAttr);
			attributes.put(attr.id, attr);
		}
	}

	public boolean addRawAttribute(String id) 
	{
		Attribute attr = new Attribute(id, null);
		addAttribute(attr);
		return true;
	}

	public boolean addRawAttribute(String id, String val) 
	{
		AttributeValueItem avi = new AttributeValueItem(val);
		Attribute attr = new Attribute(id, avi);
		addAttribute(attr);
		return true;
	}

	public boolean addRawAttribute(String id, double val)
	{
		System.out.println("0");
		return false;
	}

	public boolean addRawAttribute(String id, int val) 
	{
		return false;
	}

	public boolean addRawAttribute(String id, int[] val) 
	{
		return false;
	}

	public boolean addRawAttribute(String id, double[] val) 
	{
		if (id.equals(COLOUR_ID)) {
			colour = new Color((float) val[0], (float) val[1], (float) val[2]);
			return true;
		}
		if (id.equals(LABEL_COLOUR_ID)) {
			labelColour = new Color((float) val[0], (float) val[1], (float) val[2]);
			return true;
		}
		if (id.equals(OPEN_COLOUR_ID)) {
			colourWhenOpen = new Color((float) val[0], (float) val[1], (float) val[2]);
			return true;
		}
		return false;
	}

	public boolean addRawAttribute(String id, String[] val) 
	{
		AttributeValueItem favi = null;
		AttributeValueItem lavi = null;
		int num = val.length;

		for (int i=0; i<num; i++) {
			AttributeValueItem avi = new AttributeValueItem(val[i]);

			if (lavi != null)
				lavi.next = avi;
			if (favi == null)
				favi = avi;
			lavi = avi;
		}
		Attribute attr = new Attribute(id, favi);
		addAttribute(attr);
		return true;
	}

	// Return the attribute with the associated id

	public Attribute getAttribute(String id) 
	{
		return (Attribute) attributes.get(id);
	}



	public void replaceAttribute(Attribute newAttr)
	{
		addAttribute(newAttr);
	}



	public void writeAttributes(PrintStream ps, boolean classType) throws IOException 
	{
		// Output the body of the attribute record 
		// The child method outputs the header and tail 

		if (hasObjectColor()) {
			ps.print(Attribute.indent + COLOUR_ID + " = " + taColor(getObjectColor()) + "\n"); 
		}

		if (hasLabelColor()) { 
			ps.print(Attribute.indent + LABEL_COLOUR_ID + " = " + taColor(getLabelColor()) + "\n");
		}

		if (hasObjectColorWhenOpen()) { 
			ps.print(Attribute.indent + OPEN_COLOUR_ID + " = " + taColor(getObjectColorWhenOpen()) + "\n");
		}

		Enumeration en = attributes.elements();
		while (en.hasMoreElements()) {
			Attribute attr = (Attribute) en.nextElement();
			attr.writeAttribute(ps, classType);
		}
	}

	public void getAttributesRaw(Vector v, Hashtable st, boolean classType) 
	{
		if (classType) {
			if (colour != null) {
				regRawAttribute(v, st, COLOUR_ID, colour);
			}
			if (labelColour != null) {
				regRawAttribute(v, st, LABEL_COLOUR_ID, labelColour);
			}
			if (colourWhenOpen != null) {
				regRawAttribute(v, st, OPEN_COLOUR_ID, labelColour);
			}
		}

		Enumeration en = attributes.elements();

		while (en.hasMoreElements()) {
			Attribute attr = (Attribute) en.nextElement();
			regAttributeRaw(v, st, attr, classType);
		}
	}

	public boolean hasAttributes(boolean classType) 
	{
		if (classType)
			return true;

		Enumeration en = attributes.elements();

		while (en.hasMoreElements()) {
			Attribute attr = (Attribute) en.nextElement();
			if (!attr.cloneOnAssign)
				return true;
		}
		return false;
	}

	public String toString() 
	{
		return id;
	}

	public String getId() 
	{
		return id;
	}

	public void setId(String newId) 
	{
		id = newId;
	}

	public String getLabel() 
	{
		return label;
	}

	public void setLabel(String newLabel) 
	{
		label = newLabel;
	}

	public String getDescription() 
	{
		return description;
	}



	public void setDescription(String desc) 
	{
		description = desc;
	}



	public String getTitle() 
	{
		return title;
	}



	public void setTitle(String title) 
	{
		this.title = title;
	}

	public int getNid() 
	{
		return m_nid;
	}

  

	public void setNid(int nid) 
	{
		this.m_nid = nid;
	}

	public boolean hasStyle()
	{
		return(style >= 0);
	}

	public boolean hasObjectColor()
	{
		return(colour != null);
	}

	public boolean hasObjectColorWhenOpen()
	{
		return(colourWhenOpen != null);
	}

	public boolean hasLabelColor()
	{
		return(labelColour != null);
	}

	public int getStyle() 
	{
		int	ret;

		ret = style;
		if (ret < 0) {
			LandscapeObject	superclass;

			for (int i = 0; (superclass = derivedFrom(i)) != null; ++i) {
				ret = superclass.getStyle();
				if (ret >= 0) {
					break;
		}	}	}
		return ret;
	}

	public Color getObjectColorWhenOpen() 
	{
		Color	ret;

		ret = colourWhenOpen;
		if (ret == null) {
			LandscapeObject	superclass;

			for (int i = 0; (superclass = derivedFrom(i)) != null; ++i) {
				ret = superclass.getObjectColorWhenOpen();
				if (ret != null) {
					break;
		}	}	}
		return ret;
	}

	public Color getObjectColor() 
	{
		Color	ret;

		ret = colour;
		if (ret == null) {
			LandscapeObject	superclass;

			for (int i = 0; (superclass = derivedFrom(i)) != null; ++i) {
				ret = superclass.getObjectColor();
				if (ret != null) {
					break;
		}	}	}
		return ret;
	}

	public Color getLabelColor() 
	{
		Color	ret;

		ret = labelColour;
		if (ret == null) {
			LandscapeObject	superclass;

			for (int i = 0; (superclass = derivedFrom(i)) != null; ++i) {
				ret = superclass.getLabelColor();
				if (ret != null) {
					break;
		}	}	}
		return ret;
	}

	public void setStyle(int value)
	{
		style          = value;
	}

	public void setObjectColor(Color color1) 
	{
		colour		   = color1;
	}

	public void setObjectColorWhenOpen(Color color1) 
	{
		colourWhenOpen = color1;
	}


	public void setLabelColor(Color color1) 
	{
		labelColour = color1;
	}

	public static void setEdgeMode(int mode) 
	{
		edge_mode = mode;
	}

	public static int getEdgeMode() 
	{
		return edge_mode;
	}

	public Attribute getAttributeAt(int index)
	{
		Enumeration	en;
		Attribute	attr;

		for (en = attributes.elements(); en.hasMoreElements(); --index) {
			attr = (Attribute) en.nextElement();
			if (index == 0) {
				return(attr);
		}	}
		return(null);
	}

	// The routines that follow hide the complexity of getting/setting attribute values
	// from EditAttributes

	public int getPrimaryAttributeCount()
	{
		return(0);
	}

	public int getAttributeCount()
	{
		return(getPrimaryAttributeCount() + attributes.size() + 1);
	}

	public boolean canEditName(int index)
	{
		int	primary = getPrimaryAttributeCount();

		if (index < primary) {
			return(false);
		}
		return(true);
	}

	public boolean canEditAttribute(int index)
	{
		int	primary;

		if (index == 0) {
			// This is the id of the object
			return(false);
		}

		primary =  getPrimaryAttributeCount();
		if (index < primary) {
			return(true);
		}
		index -= primary;
		if (index < attributes.size()) {
			return(true);
		}
		return(false);
	}

	public String getAttributeNameAt(int index)
	{
		Attribute	attr;

		index -= getPrimaryAttributeCount();
		if (index == attributes.size()) {
			// Dummy row to allow insertion
			return("");
		}
		attr   = getAttributeAt(index);
		if (attr == null) {
			return(null);
		}
		return(attr.id);
	}

	public Object getAttributeValueAt(int index)
	{
		Attribute	attr;

		index -= getPrimaryAttributeCount();
		if (index == attributes.size()) {
			// Dummy row to allow insertion
			return("");
		}
		attr   = getAttributeAt(index);
		if (attr == null) {
			return(null);
		}
		return(attr.avi);
	}

	private boolean unknownAttributeName(String name)
	{
		int			i, primary;
		String		name1;

		for (i = 0; ; ++i) {
			name1 = getAttributeNameAt(i);
			if (name1 == null) {
				break;
			}
			if (name.equals(name1)) {
				System.out.println("Can't rename generic attribute to '" + name + "': attribute already exists");
				return(false);
		}	} 
		return(true);
	}

	public boolean setAttributeNameAt(int index, Object value)
	{
		Attribute	attr;
		String		name;
		int			primary;

		primary = getPrimaryAttributeCount();

		if (index < primary) {
			// Can't change names of first order attributes
			return(false);
		}
		index -= primary;

		name = (String) value;
		if (index == attributes.size()) {
			if (name.equals("")) {
				// Dummy row hasn't changed
				return(false);
			}
			if (!unknownAttributeName(name)) {
				return(false);
			} 
			// insert the new attribute
			attr = new Attribute(name, null);
			attributes.put(attr.id, attr);
			return(true);
		}

		attr = getAttributeAt(index);
		if (attr == null) {
			return(false);
		}

		if (name.equals("")) {
			attributes.remove(attr.id);
			return(true);
		}
		if (name.equals(attr.id)) {
			return(false);
		}
		if (!unknownAttributeName(name)) {
			return(false);
		} 

		attributes.remove(attr.id);
		attr.id = name;
		attributes.put(attr.id, attr);
		return(true);
	}

	public void setAttributeValueAt(int index, Object value)
	{
		Attribute	attr;

		index -= getPrimaryAttributeCount();
		attr   = getAttributeAt(index);
		if (attr == null) {
			return;
		}
		attr.avi = (AttributeValueItem) value;
	}


	// Need to know the type in cases where value might be null
	// For example with some colors

	public int getAttributeTypeAt(int index)
	{
		if (index < getAttributeCount() - 1) {
			return(Attribute.AVI_TYPE);
		}
		// This attribute does not yet exist
		return(Attribute.NULL_TYPE);
	}

	public static String taColor(Color color1)
	{  
		int		red, green, blue; 

		red    = color1.getRed();
		green  = color1.getGreen();
		blue   = color1.getBlue();

		return("(" + ((double) red)/((double) MAX_RGB) + " " +	((double) green)/((double) MAX_RGB) + " " +	((double) blue)/((double) MAX_RGB) + ")"); 
	}
}

