package lsedit;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Event;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;

public class MapBox extends JComponent implements Scrollable
{
	protected static final int GAP = 5; 

	protected static final Font textFont = new Font("Helvetica", Font.PLAIN, 10); 
	protected static final int TY_CLEAR	 = 0;
	protected static final int TY_RAISED = 1;
	protected static final int TY_SUNK	 = 2;

	class Map extends JComponent implements MouseListener
	{
		protected int					m_style;
		protected MapBox				m_mapBox;
		protected JLabel				m_label;
		protected EntityInstance		m_entity;

		public Map(MapBox mapBox, EntityInstance entity)
		{
			EntityInstance	parent;

			setLayout(null);
			
			m_style  = MapBox.TY_CLEAR;
			m_mapBox = mapBox;
			parent   = entity.getContainedBy();
			if (parent != null) {
				m_label = new JLabel(parent.getLabel(), JLabel.CENTER);
			}
			m_label.setForeground(Color.black);
			m_label.setFont(MapBox.textFont);
			m_label.setLocation(0,0);
			add(m_label);

			m_entity = entity;
			setToolTipText(entity.getLabel());
			addMouseListener(this);
		}
		
		public void setBounds(int x, int y, int width, int height)
		{
			super.setBounds(x,y,width,height);
			m_label.setSize(width, 20);
		}
		
		public void paintComponent(Graphics g)
		{
			EntityInstance	parent;
			int	width, height, lheight;

			width   = getWidth();
			height  = getHeight();
			lheight = m_label.getHeight();

			if (m_style != MapBox.TY_CLEAR) {
				g.setColor(Diagram.boxColour);
				g.draw3DRect(0, 0, width-2, height-2, m_style == MapBox.TY_RAISED);
			} 

/*			// For debugging
			g.setColor(Color.red);
			g.drawLine(0, 0, width-1, height-1);
			g.drawLine(0, height-1, width-1, 0);
*/
			g.setColor(Color.black);
			g.drawLine(0, height-1, width-1, height-1);

			parent = m_entity.getContainedBy();
			if (parent != null) {
				parent.paintMap(g, (int) (width * 0.166), lheight+5, (int) (width * 0.66), height-lheight-10, m_entity, 0);
		}	}

		public void validate()
		{
			int			label_height;
			int			width;

			super.validate();

			width   = m_last_scrollPane_width - MapBox.GAP*2;		// Workable width
			label_height = m_label.getHeight();
			m_label.setBounds(0, 0, width, label_height);
		}

		public String toString()
		{
			return ("Map " + m_label.getText() + " " + getBounds());
		}

		// MouseListener interface

		public void mouseClicked(MouseEvent e)
		{
		}

		public void mouseEntered(MouseEvent e)
		{
			m_style = MapBox.TY_RAISED;
			repaint();
		}

		public void mouseExited(MouseEvent e)
		{
			m_style = MapBox.TY_CLEAR;
			repaint();
		}

		public void mousePressed(MouseEvent ev)
		{
			m_style = MapBox.TY_SUNK;
			repaint();
		}

		public void mouseReleased(MouseEvent ev)
		{
			m_style = MapBox.TY_CLEAR;
			m_ls.followLink(m_entity.getContainedBy(), false);
		}
	}

	protected LandscapeViewerCore   m_ls;
	protected JTabbedPane			m_tabbedPane;
	protected JScrollPane			m_scrollPane;
	protected String				m_title;

	protected int					m_last_scrollPane_width = 0;
	protected double				m_last_ratio            = 0;

	public    static final String m_helpStr = "This box shows stylized versions of landscapes representing the path from the top of the containment hierarchy to the current landscape, with green boxes highlighting steps in the path.";

	// --------------
	// Public methods 
	// --------------

	public MapBox(LandscapeViewerCore ls, JTabbedPane tabbedPane) 
	{
		m_ls         = ls;
		m_tabbedPane = tabbedPane;
		m_title      = "Map";
		setLayout(null);
		setBackground(Diagram.boxColour);
		setToolTipText(m_helpStr);

		m_scrollPane = new JScrollPane();
//		m_scrollPane.setLayout(null);
		m_scrollPane.setBounds(tabbedPane.getBounds());
		setBounds(tabbedPane.getBounds());
		m_scrollPane.setViewportView(this);

		tabbedPane.addTab(m_title, null, m_scrollPane, m_helpStr);
	}

	public void activate() 
	{
		m_tabbedPane.setSelectedComponent(m_scrollPane);
	}

	public boolean isActive() 
	{
		Component active;

		if (!isVisible()) {
			return(false);
		}
		active = m_tabbedPane.getSelectedComponent();
		return(active != null && this == active);
	}

/*
	public void setBounds(int x, int y, int width, int height)
	{
		super.setBounds(x,y,width,height);
	}
*/
	
	public Dimension getPreferredSize()
	{
		Diagram			diagram;
		
		diagram  = m_ls.getDiagram();
		if (diagram == null) {
			m_last_ratio = 0;
		} else {
			int			width, height, box_width, box_height;
			double		ratio;
			int			cnt, diagram_width, diagram_height;
			Insets		insets;

			insets		   = m_scrollPane.getInsets();
			width          = m_scrollPane.getWidth() - MapBox.GAP - insets.right - insets.left;
			diagram_width  = diagram.getWidth();
			diagram_height = diagram.getHeight();
			ratio          = ((double) diagram_height) / ((double) diagram_width);

			if (width != m_last_scrollPane_width || ratio != m_last_ratio) {
				Dimension		d;

				box_width  = width  - MapBox.GAP;		// Workable width
				if (box_width < 0) {
					box_width = 0;
				}
				box_height = (int) (box_width * ratio);

				cnt    = getComponentCount();
				height = cnt * (box_height + GAP);
				d = new Dimension(width, height);
				setPreferredSize(d);
				setSize(d);
				setMaximumSize(d);
				m_last_scrollPane_width = width;
				m_last_ratio            = ratio;
		}	}
		return(super.getPreferredSize());
	}

	public void addAll(EntityInstance e)
	{
		EntityInstance parent;
	
		if (e != null) {
			parent = e.getContainedBy();
			if (parent != null) {
				addAll(parent);
				if (e.isEnterable()) {
					add(new Map(this, e));
	}	}	}	}
	
	public void fill(EntityInstance e)
	{
		removeAll();
		addAll(e);
		m_last_scrollPane_width = -1;	// Force recalculation of preferred size
		getPreferredSize();
	}

	public void validate()
	{
		int				x, y, width, height;
		Map				map;
		double			ratio;
		Dimension		d;
		int				i, cnt;

		ratio   = m_last_ratio;
		if (ratio > 0) {
			x       = MapBox.GAP;
			y       = 0;
			width   = m_last_scrollPane_width - MapBox.GAP*2;	// Workable width
			height  = (int) (width * ratio);
			d       = new Dimension(width, height);
			
			cnt = getComponentCount();
			for (i = 0; i < cnt; ++i) {
				map = (Map) getComponent(i);
				map.setLocation(x, y);
				map.setSize(d);
				map.setMinimumSize(d);
				map.setPreferredSize(d);
				map.setMaximumSize(d);
				y  += height + GAP;
	}	}	}

	public Dimension getMaximumSize()
	{
		return(getSize());
	}


/*
	public void paintComponent(Graphics g)
	{
		int	width, height;

		width   = getWidth();
		height  = getHeight();
		
		// For debugging
		g.setColor(Color.green);
		g.drawLine(0, 0, width, height);
		g.drawLine(0, height, width, 0);
	}
*/
	// Scrollable interface
	// We need to implement this to stop iterative size changes..
	// The mapbox knows what size it wants to be and sets it on every validate
	// If we don't disable it the viewport tries to change the size to fit the window every time it validates

	public Dimension getPreferredScrollableViewportSize()
	{
		return(getSize());
	}
	 
	public int getScrollableBlockIncrement(Rectangle visibleRect, int orientation, int direction)
	{
		return(50);
	}
	 
	public boolean getScrollableTracksViewportHeight() 
	{
		return(false);
	}

	public boolean getScrollableTracksViewportWidth() 
	{
		return(false);
	}

	public int getScrollableUnitIncrement(Rectangle visibleRect, int orientation, int direction)
	{
		return(10);
	} 
}
