package lsedit;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.event.*;

import java.util.*;
import java.io.*;
import javax.swing.*;

import java.awt.geom.*;

public class RelationInstance extends LandscapeObject 
{
	// Constants

	public    final static int    CLIENT_SUPPLIER_EL_LEN = 16;
	protected final static int	  NEAR_PIXEL_SIZE		 = 6;

	// First order attribute ids

	protected final static String STYLE_ID	= "style";

	// Bits set in m_marks

	protected final static int LOOP_MARK	    	= 0x01;		// Repaint this edge
	protected final static int NORMAL_MARK			= 0x02;
	protected final static int ELISION_MARK         = 0x04;
	protected final static int OUT_ELIDED_MARK		= 0x10;
	protected final static int IN_ELIDED_MARK		= 0x20;
	protected final static int IN_DIAGRAM_MARK		= 0x40;

	protected final static int VALID_MARK           = 0x100;

	public final static int GROUP_FLAG_MARK         = 0x200;
	public final static int HIGHLIGHT_FLAG_MARK     = 0x400;
	public final static int	ADD_RESULTS_MARK		= 0x800;

	public final static int IN_OUT_ELIDED           = IN_ELIDED_MARK  | OUT_ELIDED_MARK;
	public final static int PRESENTATION_MARKS      = GROUP_FLAG_MARK | HIGHLIGHT_FLAG_MARK;

	// An edge is defined by a relation class and two entity instances

	protected EntityInstance  m_src,     m_dst;			// The real source and destination of this edge
	protected EntityInstance  m_drawSrc, m_drawDst;		// The derived source and destination of this edge given closed entities exist

	// Variables

	protected int			  m_mark = 0;				// A collection of bits

	// ---------------
	// Wrapper methods
	// ---------------

	public RelationComponent neededComponent()
	{
		RelationComponent relationComponent = (RelationComponent) getSwingObject();
		if (relationComponent == null) {
			relationComponent = new RelationComponent(this);
		}
		return(relationComponent);
	} 

	public void repaint()
	{
		JComponent relationComponent = getSwingObject();

		if (relationComponent != null) {
			relationComponent.repaint();
	}	}
		
	public void validate()
	{
		JComponent relationComponent = getSwingObject();

		if (relationComponent != null) {
			relationComponent.validate();
	}	}

	public EdgePoint mouseOverEdgePoint(int x, int y)
	{
		JComponent relationComponent = getSwingObject();

		if (relationComponent != null) {
			return ((RelationComponent) relationComponent).mouseOverEdgePoint(x, y);
		}
		return(null);
	}

	// First order attributes

	protected boolean processFirstOrderAttributes(Attribute attr, Hashtable entities) 
	{
		boolean hasVal = (attr.avi != null);

		if (attr.id.equals(COLOUR_ID)) {
			if (hasVal)
				setObjectColor(attr.parseColour());
			return true; 
		}
		if (attr.id.equals(STYLE_ID)) {
			if (hasVal) {
				setStyle(attr.parseInt());
			}
			return true;
		}
		return false;
	}

	// --------------
	// Public methods
	// --------------

	public RelationInstance(RelationClass parentClass, EntityInstance src, EntityInstance dst)
	{
		setParentClass(parentClass);
		this.m_src       = src;
		this.m_dst       = dst;
		this.m_drawSrc   = src;
		this.m_drawDst   = dst;
	}

	public void clearMark(int preserve_relation_marks)
	{
		RelationComponent relationComponent = (RelationComponent) getSwingObject();

		m_mark    &= preserve_relation_marks;
		m_drawSrc  = null;
		m_drawDst  = null;
		setSwingObject(null);		// IJD: Make this smarter

//		System.out.println("RelationInstance.clearMark " + this);
	}

	public void orMark(int value)
	{
		m_mark |= value;
	}

	public void nandMark(int value)
	{
		m_mark &= ~value;
	}

	public boolean isMarked(int value)
	{
		return((m_mark & value) != 0);
	}

	public boolean isSrcVisible()
	{
		return (m_src == m_drawSrc);
	}

	public boolean isDstVisible()
	{
		return (m_dst == m_drawDst);
	}

	/* Mark the edge as invalid so we paint it only once and then mark valid
	   -- not twice once from the source and once from the destination
	*/

	public void invalidateEdge()
	{
		nandMark(VALID_MARK);
	}

	public void setHighlightFlag() 
	{
		if (!isMarked(HIGHLIGHT_FLAG_MARK)) {
			orMark(HIGHLIGHT_FLAG_MARK);
			repaint();
	}	}

	public boolean getHighlightFlag() 
	{
		return isMarked(HIGHLIGHT_FLAG_MARK);
	}

	public void clearHighlightFlag() 
	{
		if (isMarked(HIGHLIGHT_FLAG_MARK)) {
			nandMark(HIGHLIGHT_FLAG_MARK);
			validate();
	}	}

	public void setGroupAndHighlightFlag() 
	{
		if ((m_mark & (GROUP_FLAG_MARK | HIGHLIGHT_FLAG_MARK)) != (GROUP_FLAG_MARK | HIGHLIGHT_FLAG_MARK)) {
			orMark(GROUP_FLAG_MARK | HIGHLIGHT_FLAG_MARK);
			validate(); 
	}	}

	public void drawHighlighted() 
	{
		if (getHighlightFlag() && getRelationClass().isActive()) {
			draw(false);
	}	}

	public void clearGroupFlag() 
	{
		if (isMarked(GROUP_FLAG_MARK)) {
			nandMark(GROUP_FLAG_MARK);
			validate();
	}	}

	public boolean getGroupFlag() 
	{
		return isMarked(GROUP_FLAG_MARK);
	}

	// Compute the rules for how an edge is to be drawn between two entities, handling elision settings.
	// Essentially just flag the relation appropriately to assist the verify process know what it is dealing with

	public void draw(boolean allowElision) 
	{
		int				marks;
		boolean			src_elided, dst_elided;
		RelationClass	parentClass;

		if (isMarked(VALID_MARK)) {
			return;
		}

		parentClass     = getRelationClass();
		Diagram	diagram = getDiagram();


	/*	System.out.println("RelationInstance:draw(" + allowElision + ") " + this);
		java.lang.Thread.dumpStack();
		System.out.println("-----");
	*/
		marks = 0;
		if (!allowElision || parentClass.isClassVisible()) {
			if (m_src == null || m_dst == null) {
				MsgOut.println("Landscape error: null src or dst");
				return;
			}
			if (m_src.getContainedBy() == null) {
				MsgOut.println("drawEdge: containedBy error");
				return;
			}
			if (m_dst.getContainedBy() == null) {
				MsgOut.println("drawEdge: containedBy error");
				return;
			}
					
			EntityInstance	src, dst, ancestor, drawRoot;

			src      = m_drawSrc;
			dst      = m_drawDst;

			if (src == null) {
				System.out.println("draw() " + this + " has no draw src");
				return;
			}
			if (dst == null) {
				System.out.println("draw() " + this + " has no draw dst");
				return;
			}

			drawRoot = diagram.getDrawRoot();
			if (drawRoot.hasDescendantOrSelf(src) || drawRoot.hasDescendantOrSelf(dst)) {
				// At least one edge addresses something in the diagram

				if (src != dst || src == m_src || dst == m_dst) {
					// Don't show internal edges beneath closed objects period
					// Such edges will have the same draw src and destination and both real edges lifted to this common closed ancestor

					marks = NORMAL_MARK;
					if (allowElision) {
						String			id        = parentClass.getId();
						EntityInstance	srcParent = src.getContainedBy();
						EntityInstance	dstParent = dst.getContainedBy();

						if (srcParent != null && srcParent == dstParent && srcParent.isInternalRelationElided(id)) {
							// Don't show -- this internal relation is elided
							marks = 0;
						} else {
							if (src.isSrcRelationElided(id)) {
								marks |= OUT_ELIDED_MARK;
							} else {
								for (; srcParent != null && !srcParent.hasDescendantOrSelf(dst); srcParent = srcParent.getContainedBy()) {
									// Source goes from inside srcParent to outside of it
									if (srcParent.isExitingRelationElided(id)) {
										marks |= OUT_ELIDED_MARK;
										break;
							}	}	}

							if (dst.isDstRelationElided(id)) {
								marks |= IN_ELIDED_MARK;
							} else {
								for (; dstParent != null && !dstParent.hasDescendantOrSelf(src); dstParent = dstParent.getContainedBy()) {
									// Destination is inside while src is outside
									if (dstParent.isEnteringRelationElided(id)) {
										marks |= IN_ELIDED_MARK;
										break;
					}	}	}	}	}

					if (marks == NORMAL_MARK && src == dst) {
						// Recursive loop
						marks = LOOP_MARK;
		}	}	}	}
			
		JComponent relationComponent;

		if (marks == 0) {
			if (isMarked(IN_DIAGRAM_MARK)) {
				relationComponent = getSwingObject();
				if (relationComponent != null) {
					diagram.remove(relationComponent);
					setSwingObject(null);
				}
				nandMark(IN_DIAGRAM_MARK);
			}
		} else {
			marks |= VALID_MARK;
			if (allowElision) {
				marks |= ELISION_MARK;
			}
			relationComponent = neededComponent();

			if (!isMarked(IN_DIAGRAM_MARK)) {
				marks |= IN_DIAGRAM_MARK;
				orMark(marks);
				diagram.add(relationComponent);	// Add this relation into the diagram
				relationComponent.setVisible(true);
			} else {
				orMark(marks);
			}
			relationComponent.validate();
	}	}	

	public void addAttribute(Attribute attr) 
	{
		processFirstOrderAttributes(attr, null);
	}

	public void assignAttributes(Attribute attr, Hashtable entities)
	{
		// Passed list of attributes

		for (; attr != null; attr = attr.next) {
			// For each attribute assignment, find the variable,
			// and assign the value.

			if (attr.avi == null) {
				MsgOut.println("Null attribute assignment");
			} else if (!processFirstOrderAttributes(attr, entities)) {
				Attribute attrVar = getLsAttribute(attr.id);

				if (attrVar == null) {
					MsgOut.println("Missing attribute '" + attr.toString() + "'");
				} else {
					if (attrVar.cloneOnAssign) {
						Attribute newAttr = (Attribute) attrVar.clone();
						replaceAttribute(newAttr);
					}
					attrVar.avi = attr.avi;
		}	}	}
	}

	public void writeRelation(PrintStream ps) throws IOException 
	{
		ps.print(qt(getParentClass().getId()) + " " + qt(m_src.getId()) + " " + qt(m_dst.getId()) + "\n");
	}

	public void writeAttributes(PrintStream ps) throws IOException 
	{
		ByteArrayOutputStream	byteStream = new ByteArrayOutputStream();
		PrintStream				ps1        = new PrintStream(byteStream);
		String					s;
		RelationClass			parentClass= getRelationClass();

		super.writeAttributes(ps1, parentClass, false);

		s = byteStream.toString();
		if (s.length() != 0) {
			ps.print("(" + qt(parentClass.getId()) + " " + qt(m_src.getId()) + " " + qt(m_dst.getId()) + ") {\n" + s + "}\n");
		}
	}

	public EntityInstance getSrc() 
	{
		return m_src;
	}

	public EntityInstance getDst() 
	{
		return m_dst;
	}

	public EntityInstance getDrawSrc() 
	{
		return m_drawSrc;
	}

	public EntityInstance getCurrentSrc()
	{
		if (m_drawSrc != null) {
			return(m_drawSrc);
		}
		return(m_src);
	}

	public EntityInstance getDrawDst() 
	{
		return m_drawDst;
	}
	
	public EntityInstance getCurrentDst()
	{
		if (m_drawDst != null) {
			return(m_drawDst);
		}
		return(m_dst);
	}

	public RelationClass getRelationClass() 
	{
		return (RelationClass) getParentClass();
	}

	public boolean matches(RelationInstance other)
	{
		return(m_drawSrc == other.m_drawSrc && m_drawDst == other.m_drawDst);
	}

	public String toString() 
	{
		return m_drawSrc + (m_drawSrc == m_src ? "" : "{" + m_src + "}") + "->" + m_drawDst + (m_drawDst == m_dst ? "" : "{" + m_dst + "}");
	}

	// The routines that follow hide the complexity of getting/setting attribute values
	// from EditAttributes

	public int getPrimaryAttributeCount()
	{
		return(3);
	}

	public String getLsAttributeNameAt(int index)
	{
		String	name;

		switch (index) {
		case 0:
			name  = "class";
			break;
		case 1:
			name  = COLOUR_ID;
			break;
		case 2:
			name  = STYLE_ID;
			break;	
		default:
			name  = super.getLsAttributeNameAt(index);
		}
		return(name);
	}

	public Object getLsAttributeValueAt(int index)
	{
		Object	value;

		switch (index) {
		case 0:
			value = getParentClass();
			break;
		case 1:
			if (hasObjectColor()) {
				value = getObjectColor();
			} else {
				value = null;
			}
			break;
		case 2:
			value = new Integer(getStyle());
			break;
		default:
			value = super.getLsAttributeValueAt(index);
		}
		return(value);
	}

	public void setAttributeValueAt(int index, Object value)
	{
		switch (index) {
		case 0:
			if (value != null) {
				LandscapeClassObject parentClass = getParentClass();
				String newId = (String) value;
				if (parentClass == null || !parentClass.getId().equals(newId)) {
					Enumeration		en;
					RelationClass	ec;

					for (en = getDiagram().enumRelationClasses(); en.hasMoreElements(); ) {
						ec = (RelationClass) en.nextElement();
						if (ec.getId().equals(newId)) {
							setParentClass(ec);
							break;
			}	}	}	}
			break;
		case 1:
			setObjectColor((Color) value);
			break;
		case 2:
			setStyle(((Integer) value).intValue());
			break;
		default:
			super.setAttributeValueAt(index, value);
	}	}

	// Need to know the type in cases where value might be null
	// For example with some colors

	public int getLsAttributeTypeAt(int index)
	{
		int		ret;
		
		switch (index) {
		case 0:
			ret = Attribute.RELATION_CLASS_TYPE;
			break;
		case 1:
			ret = Attribute.COLOR_OR_NULL_TYPE;
			break;
		case 2:
			ret = Attribute.STYLE_TYPE;
			break;
		default:
			ret = super.getLsAttributeTypeAt(index);
		}
		return(ret);
	}

	public void mouseEntered()
	{
		getDiagram().getLs().setCursor(Cursor.HAND_CURSOR);
	}

	public void mouseExited()
	{
		getDiagram().getLs().setCursor(Cursor.DEFAULT_CURSOR); 
	}

	public void mousePressed(MouseEvent ev, int diagramX, int diagramY)
	{
		getDiagram().relationPressed(ev, this, diagramX, diagramY);
	}

	public void mouseReleased(MouseEvent ev, int diagramX, int diagramY)
	{
		getDiagram().relationReleased(ev, this, diagramX, diagramY);
	}

	public void mouseDragged(MouseEvent ev, int diagramX, int diagramY)
	{
		getDiagram().relationDragged(ev, this, diagramX, diagramY);
	}

	public void mouseMoved(MouseEvent ev, int diagramX, int diagramY)
	{
		Diagram diagram = getDiagram();

		if (this != m_infoShown) {
			String message;

			m_infoShown = this;
			if (m_src == m_drawSrc) {
				message = "";
			} else {
				message = Util.quoted(m_src.getLabel()) + "->";
			}
			message += Util.quoted(m_drawSrc.getLabel()) +  " " + getRelationClass().getLabel() + " " + Util.quoted(m_drawDst.getLabel());
			if (m_dst != m_drawDst) {
				message += "->" + Util.quoted(m_src.getLabel());
			}
//			message += " " + (getX() + ev.getX()) + "x" + (getY() + ev.getY());
			
			diagram.getLs().showInfo(message);
		}
		diagram.movedOverThing(ev, this, diagramX, diagramY);
	}
}
