package lsedit;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Event;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;

public class ResultBox extends Box implements Scrollable 
{
	protected static final Color	m_titleColor = Color.red.darker();
	protected static final Font		m_titleFont  = new Font("Helvetica", Font.BOLD, 14);
	protected static final Font		m_textFont	 = new Font("Helvetica", Font.PLAIN, 11); 
	protected static final String	m_indent     = "  ";

	// <entity name> [<entity class name>]

	class ResultLabel extends JLabel
	{
		public ResultLabel()
		{
			super();
			setHorizontalAlignment(LEFT);
			setHorizontalTextPosition(LEFT);
			setFont(m_textFont);
		}

		public String toString()
		{
			return(getText());
		}
	}

	// <entity> [ {<class>} ]

	class ResultEntity extends ResultLabel implements MouseListener
	{
		protected LandscapeViewerCore	m_ls;
		protected EntityInstance		m_entity;

		public ResultEntity(LandscapeViewerCore ls, String indent, EntityInstance entity, boolean showClass)
		{
			super();

			String		text;
			
			m_ls     = ls;
			m_entity = entity;
			text     = indent + entity.getLabel();
			if (showClass) {
				text += " {" + entity.getEntityClass().getLabel() + "}";
			}
			setText(text);
			setForeground(Color.blue);
			setToolTipText(entity.getDescription());
			addMouseListener(this);
		}
		
		// MouseListener interface

		public void mouseClicked(MouseEvent e)
		{
		}

		public void mouseEntered(MouseEvent e)
		{
			setForeground(Color.red);
			repaint();
		}

		public void mouseExited(MouseEvent e)
		{
			setForeground(Color.blue);
			repaint();
		}

		public void mousePressed(MouseEvent ev)
		{
		}

		public void mouseReleased(MouseEvent ev)
		{
			setForeground(Color.blue);
			m_ls.followLink(m_entity, false);
		}
	}

	class HorizontalResult extends Box
	{
		public HorizontalResult()
		{	
			super(BoxLayout.X_AXIS);
		}

		public String toString()
		{
			int			i, cnt;
			Component	component;
			String		string;

			cnt      = getComponentCount();
			string   = "";
			for (i = 0; i < cnt; ++i) {
				component = getComponent(i);
				string += component.toString();
			}
			return string;
		}
	}

	class VerticalResult extends Box
	{
		public VerticalResult()
		{
			super(BoxLayout.Y_AXIS);
		}

		public String toString()
		{
			int			i, cnt;
			Component	component;
			String		string;

			cnt      = getComponentCount();
			string   = "";
			for (i = 0; i < cnt; ++i) {
				component = getComponent(i);
				string   += component.toString() + "\n";
			}
			return string;
		}
	}

	//  <Entity name> [<relation class>] <Entity name>

	class ResultRelation extends HorizontalResult
	{
		protected RelationInstance		m_relation;

		public ResultRelation(LandscapeViewerCore ls, String indent, RelationInstance relation, boolean hideRc)
		{	
			super();

			ResultEntity	resultEntity;
			ResultLabel		rc;

			m_relation = relation;
			resultEntity = new ResultEntity(ls, indent, relation.getSrc(), false);
			add(resultEntity);
			if (!hideRc) {
				rc = new ResultLabel();
				rc.setText(" " + relation.getRelationClass().getLabel());
				rc.setHorizontalAlignment(JLabel.CENTER);
				add(rc);
			}
			resultEntity = new ResultEntity(ls, " ", relation.getDst(), false);
			add(resultEntity);
		}

		public boolean matches(RelationInstance relation)
		{
			return(m_relation.matches(relation));
		}
	}

	// <ResultEntity> <rc> [*|?] 
	// [*|?] <rc> <ResultEntity>

	class ResultSetHeader extends HorizontalResult
	{
		public ResultSetHeader(LandscapeViewerCore ls, EntityInstance entity, RelationClass relationClass, boolean forward, boolean closure)
		{
			super();

			Enumeration			en;
			ResultEntity		resultEntity;
			ResultLabel			rc, label;
			Object				object;
			EntityInstance		e;
			RelationInstance	r;
			String				type;

			if (closure) {
				type = "*";
			} else {
				type = "?";
			}

			rc           = new ResultLabel();
			rc.setText(" " + relationClass.getLabel());
			rc.setHorizontalAlignment(JLabel.CENTER);
			label        = new ResultLabel();


			if (forward) {
				resultEntity = new ResultEntity(ls, "", entity, false);
				label.setText(" " + type);
				label.setHorizontalAlignment(JLabel.LEFT);
				add(resultEntity);
				add(rc);
				add(label);
			} else {
				label        = new ResultLabel();
				label.setText(type);
				label.setHorizontalAlignment(JLabel.LEFT);
				resultEntity = new ResultEntity(ls, " ", entity, false);
				add(label);
				add(rc);
				add(resultEntity);
			}
		}

		public ResultSetHeader(LandscapeViewerCore ls, EntityInstance entity, Vector list)
		{
			super();

			ResultEntity	resultEntity;
			ResultLabel		label;

			resultEntity = new ResultEntity(ls, "", entity, false);
			label        = new ResultLabel();
			label.setText(" contains (" + list.size() + " items):");
			label.setHorizontalAlignment(JLabel.LEFT);
			add(resultEntity);
			add(label);
		}
	}

	/*
		<entity> <relation class> ?
			<entity>
			...
			<entity>
	 */

	class ResultSet extends VerticalResult
	{
		public ResultSet(LandscapeViewerCore ls, EntityInstance entity, RelationClass relationClass, Vector entities, boolean forward, boolean closure)
		{
			super();

			Enumeration			en;
			ResultEntity		resultEntity;
			ResultRelation		resultRelation;
			Object				object;
			EntityInstance		e;
			RelationInstance	r;

			add(new ResultSetHeader(ls, entity, relationClass, forward, closure));
			for (en = entities.elements(); en.hasMoreElements(); ) {
				object = en.nextElement();
				if (object instanceof EntityInstance) {
					e = (EntityInstance) object;
					resultEntity = new ResultEntity(ls, m_indent, e, false);
					add(resultEntity);
				} else {
					r = (RelationInstance) object;
					resultRelation = new ResultRelation(ls, m_indent, r, true);
					add(resultRelation); 
			}	}
		}
	}

	/*
		<entity> <relation class> ?
			<entity>
			...
			<entity>
	 */

	class ResultContents extends VerticalResult
	{
		public ResultContents(LandscapeViewerCore ls, EntityInstance entity, Vector entities)
		{
			super();

			Enumeration			en;
			ResultEntity		resultEntity;
			Object				object;
			EntityInstance		e;

			add(new ResultSetHeader(ls,entity, entities));
			for (en = entities.elements(); en.hasMoreElements(); ) {
				e = (EntityInstance) en.nextElement();
				resultEntity = new ResultEntity(ls, m_indent, e, true);
				add(resultEntity);
			}
		}
	}

	protected static final int	horizontal_margin    = 10;
	protected static final int	vertical_indent      = 10;

	public    static final String m_helpStr = "This box shows the results of queries, and groupings";


	protected LandscapeViewerCore  m_ls;
	protected JTabbedPane			m_tabbedPane;
	protected JScrollPane			m_scrollPane;

	// --------------
	// Public methods 
	// --------------

	public ResultBox(LandscapeViewerCore ls, JTabbedPane tabbedPane) 
	{
		super(BoxLayout.Y_AXIS);

		Dimension d;

		m_ls         = ls;
		m_tabbedPane = tabbedPane;

		setBackground(Diagram.boxColour);
		setToolTipText(m_helpStr);

		m_scrollPane = new JScrollPane();
		m_scrollPane.setBounds(tabbedPane.getBounds());
		d            = m_scrollPane.getSize();
		setLocation(0,0);
		setSize(d);
		setPreferredSize(d);
		m_scrollPane.setViewportView(this);

		tabbedPane.addTab("Results", null, m_scrollPane, m_helpStr);
	}

	public void clear() 
	{
		removeAll();
		validate();
		repaint();
	}

	public void activate() 
	{
		m_tabbedPane.setSelectedComponent(m_scrollPane);
	}

	public boolean isActive() 
	{
		
		if (isVisible()) {
			Component active;

			active = m_tabbedPane.getSelectedComponent();
			return(active != null && this == active);
		}
		return(false);
	}

	public void setResultTitle(String title)
	{
		clear();
		if (title != null) {
			ResultLabel	label = new ResultLabel();
			label.setText(title);
			label.setHorizontalAlignment(JLabel.LEFT);
			label.setFont(m_titleFont);
			label.setForeground(m_titleColor);
			add(label);
			add(Box.createVerticalStrut(5));
	}	}

	public void addResultEntity(EntityInstance e)
	{
//		activate();
		add(new ResultEntity(m_ls, "", e, false));

	}

	public void addRelation(RelationInstance r)
	{
//		activate();
		add(new ResultRelation(m_ls, "", r, false));
	}

	public void addForwardRelation(EntityInstance e, RelationClass rc, Vector list)
	{
//		activate();
		add(new ResultSet(m_ls, e, rc, list, true, false /* Not closure */));
	}
		
	public void addBackRelation(EntityInstance e, RelationClass rc, Vector list)
	{
//		activate();
		add(new ResultSet(m_ls, e, rc, list, false, false));
	}
	
	protected void addRelations(EntityInstance e, RelationClass rc, Vector list, boolean src)
	{
//		activate();
		add(new ResultSet(m_ls, e, rc, list, src, true));
	}

	protected void addContents(EntityInstance e, Vector list)
	{
//		activate();
		add(new ResultContents(m_ls, e, list));
	}

	public void addText(String message)
	{
		if (message != null) {
//			activate();
			ResultLabel	label = new ResultLabel();
			label.setText(message);
			label.setHorizontalAlignment(JLabel.LEFT);
			label.setForeground(Color.black);
			add(label);
	}	}

	public void validate()
	{
		super.validate();
		setSize(getPreferredSize());
	}

/*
	public void paintComponent(Graphics g)
	{
		super.paintComponent(g);

		int	width, height;

		width   = getWidth();
		height  = getHeight();
		
		// For debugging
		g.setColor(Color.green);
		g.drawLine(0, 0, width, height);
		g.drawLine(0, height, width, 0);
	}
*/

	public String toString()
	{
		int			i, cnt;
		Component	component;
		String		string;

		cnt      = getComponentCount();
		string   = "";
		for (i = 0; i < cnt; ++i) {
			component = getComponent(i);
			string += component.toString() + "\n---\n";
		}
		return string;
	}
	
	// Scrollable interface
	// We need to implement this to stop iterative size changes..
	// The mapbox knows what size it wants to be and sets it on every validate
	// If we don't disable it the viewport tries to change the size to fit the window every time it validates

	public Dimension getPreferredScrollableViewportSize()
	{
		return(getSize());
	}
	 
	public int getScrollableBlockIncrement(Rectangle visibleRect, int orientation, int direction)
	{
		return(50);
	}
	 
	public boolean getScrollableTracksViewportHeight() 
	{
		return(false);
	}

	public boolean getScrollableTracksViewportWidth() 
	{
		return(false);
	}

	public int getScrollableUnitIncrement(Rectangle visibleRect, int orientation, int direction)
	{
		return(10);
	} 
}






