package lsedit;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.Rectangle;

import java.awt.event.*;
import java.net.*;
import java.io.*;
import java.util.*;
import javax.swing.*;

public class Diagram extends Ta implements MouseMotionListener, DiagramCoordinates  
{
	class ExitFlag extends JComponent implements MouseListener
	{
		protected final static int EXIT_FLAG_DIM = 8;

		protected Diagram m_diagram;

		public ExitFlag(Diagram diagram)
		{
			super();
			m_diagram = diagram;
			addMouseListener(this);
		}

		public void setLocation(int x, int y)
		{
			setBounds(x, y, EXIT_FLAG_DIM, EXIT_FLAG_DIM);
		}

		public void paintComponent(Graphics g)
		{
//			System.out.println("Diagram.paintComponent()");

			super.paintComponent(g);

			/*  Draw a small mark as shown in top left of object

				x---
				|
				|   --------
				|  |        |
				   |        |
				   |        |
	  			   |  ----  |
				   |        |
				   |        |
				   |        |
				   |________|
			*/

			g.setColor(boxColour.darker());
			g.drawRect(0, 0, EXIT_FLAG_DIM, EXIT_FLAG_DIM);
			// g.setColor(Color.black);
			g.drawLine(2, 4, 6, 4);
//			System.out.println("Diagram.paintComponent() done");
		}

		// MouseListener interface

		public void mouseClicked(MouseEvent ev)	
		{
			m_diagram.navigateToDrawRootParent();
		}

		public void mouseEntered(MouseEvent e)
		{
			getLs().setCursor(Cursor.HAND_CURSOR);
		}

		public void mouseExited(MouseEvent e)
		{
			getLs().setCursor(Cursor.DEFAULT_CURSOR); 
		}

		public void mousePressed(MouseEvent ev)
		{
		}

		public void mouseReleased(MouseEvent ev)
		{
		}
	}

	// Final values

	public final static float BG = 0.75f;	// Also change BG_STR if you change this
	public final static Color boxColour = Color.lightGray;

	protected final static int GAP = 5;
	protected final static int MARGIN = GAP*2;

	// Values

	protected double	m_zoom_x = 1.0;		//	The zoom factor
	protected double	m_zoom_y = 1.0;

	protected EntityInstance	m_drawRoot;					// The current entity being drawn
	
	/* By putting the edges in a separate container we simplify the management of painting
		(1) We ensure that edges are painted after entities. This is because the m_edges
			components is added first.
		(2) We ensure that adding and deleting edges is efficient.
			This is because we can add at end.
	 */

	protected Container			m_edges     = null;

	/* Likewise for cardinals */

	protected Container			m_cardinals = null;

	protected int				numVisibleRelationClasses = 0;

	protected boolean			persistentQuery = false;
	protected boolean			m_loaded        = false;

	protected int				mode; 

	protected int				gridPixels = 1;

	protected int				m_preserve_entity_marks   = 0;
	protected int				m_preserve_relation_marks = 0;

	protected Attribute			relVisibilityAttr; 

	protected boolean			m_visibleEdges = true;
	protected boolean			m_drawEdges = true;
	protected boolean			m_drawBends = false;

	protected int				newCnt = 0;					// Used to id new entities	
	protected boolean			undoValid = false;

	protected int				timeStamp = 0;

	protected EntityInstance	m_keyEntity;				// The currently highlight entity if any else null

	protected ExitFlag			m_exitFlag;
	protected SupplierSet		m_supplierSet;
	protected ClientSet			m_clientSet;	

	protected Vector			m_suppliers;
	protected Vector			m_dClients;
	protected Vector			m_dSuppliers;

	protected boolean			m_suppliersSorted = false;

	protected boolean			allClients = true;
	protected boolean			allSuppliers = true;

	protected boolean			m_modeHandlingActive = false;
	protected LandscapeModeHandler m_modeHandler;

	// -----------------
	// Protected methods
	// -----------------

	protected Diagram getDiagram()
	{
		// Hide the TA layer from having to know what the diagram is
		return(this);
	}

	protected void findClientsAndSuppliers() 
	{
		EntityInstance	drawRoot = m_drawRoot;
		boolean			showSuppliers;
		boolean			showClients;

		if (drawRoot == m_rootInstance) {
			showSuppliers = false;
			showClients   = false;
		} else {
			showSuppliers = m_ls.isShowSuppliers();
			showClients   = m_ls.isShowClients();
		}
		m_clientSet.removeAll();
		if (showClients) {
			m_clientSet.findClients(m_rootInstance, drawRoot);
		} 
		m_supplierSet.removeAll();
		if (showSuppliers) {
			m_supplierSet.findSuppliers(m_rootInstance, drawRoot);
		}
	}
		
	protected void setVisibilityFlags() 
	{
		Enumeration en;

		numVisibleRelationClasses = 0;

		for (en = enumRelationClasses(); en.hasMoreElements(); ) {
			RelationClass rc = (RelationClass) en.nextElement();
			if (!rc.excludeReln()) {
				if (rc.isClassVisible()) {
					rc.setOrdinal(numVisibleRelationClasses++);
				}
			}
		}
		m_visibleEdges = (numVisibleRelationClasses > 0);
	}

	// We are cutting this entity

	public void cutEntity(EntityInstance e)
	{
		if (m_drawRoot == e || m_drawRoot.hasAncestor(e)) {
			navigateTo(e.getContainedBy());
		}
		e.getContainedBy().removeContainment(e);
		m_entityInstances.remove(e.getId());
	}

	public void pasteEntity(EntityInstance parent, EntityInstance e)
	{
		parent.addContainment(e);
		m_entityInstances.put(e.getId(), e);
	}

	public void prepostorder()
	{
		m_rootInstance.prepostorder(1);
	}

/* 
	What a cautionary tale this is for a software engineering course..

	1. Somewhere in Java AWT's history someone had the bright idea of changing
	   the AWT framework so that every component could have a mouse cursor
	   associated with it instead of just the frame managing what the current
	   mouse cursor object should be at any given time.  The idea was that as
	   you moved the mouse over different objects each object would have the
	   capability of explicitly indicating what the cursor associated with the
	   mouse should be, while it hovered over that object.

	2. This means that when ever a new visible object is added to a container,
	   AWT must handle the possibility that the cursor happens to be over this
	   new object, in which case this new object should if appropriate change
	   the appearance of that mouse cursor immediately.

	3. So whenever an object is added to the AWT containment hierarchy, there
	   is an invocation to that object to change the system cursor immediately.

	   [This by the way is a less than smart implementation.. The smart thing
		to tell the component is change the cursor if the mouse pointer is over
		it and it actually cared what the cursor  should be in that unusual case]

	4. The object handles this request by walking all the way up the containment
	   heirarchy until it hits a component which actually cares what the cursor
	   looks like (in lsedit's case the topmost frame object, since tha ability
	   to associate cursors directly with objects is not used).

	5. This cursor isn't changed but the frame object doesn't know that. That
	   is because any component can over ride the frame's opinion as to what
	   the cursor should be (not that any does in lsedit).

	6. So the frame object starts off by asking itself what the mouse pointer
	   is over. To do this it walks back down through the containment heirarchy
	   asking each level if it contains the current point the mouse pointer is
	   over, recursively.  This comes close to search all components, if the
	   mouse pointer happens to be over the diagram, since entities and edges
	   dominate the number of components.

	7. Now containment is not a trivial exercise for edges.  At a minimum we
	   need to decide if the given point is within 'n' pixels of any part of
	   the line, the arrow head, etc.

	8. Having identified the deepest thing the mouse pointer is over, we walk up
	   finding the nearest component above which wishes to control the cursor
	   image, arriving back at the frame.  The frame then dutifully sets the
	   mouse cursor to what it already was.


	Now this is repeated every time we add a component.  So it is clearly an
	O(N^2) operation for N components.  So when displaying a large number of
	edges (order the number in linux or whatever) the cost of JAVA painting
	the cursor image associated with the mouse pointer, becomes prohibitive
	as the actual graph image is constructed line by line.

	I worked this out by narrowing down the performance problems when displaying
	a large ta file to: White cursor mouse arrow left on screen until lsedit
	draws = 40 seconds, while time taken was a mere 4 seconds if I cleverly fired
	up lsedit in a manner which left the cursor off the lsedit screen, somewhere
	down at the bottom of my actual screen.

	Needless to say I at first found this behaviour quite astonishing.

	Work around to correct this <<bug>> was to add to the diagram.java class
	the following special rule about what the diagram contains when not
	loaded -- nothing.

	Ian Davis
*/

	public boolean contains(int x, int y)
	{
		return (m_loaded && super.contains(x, y));
	}

	public void setPreserveEntityMarks(int value)
	{
		m_preserve_entity_marks = value;
	}

	public void setPreserveRelationMarks(int value)
	{
		m_preserve_relation_marks = value;
	}

	// For debugging
	
	
    protected void paintChildren(Graphics g) 
    {
    	super.paintChildren(g);
    }

	// For debugging
	    	
    public void repaint() 
    {
    	super.repaint();
    }
    
	// --------------
	// Public methods
	// --------------

	// This constructor is only called to create copies of diagrams
	// They are never active so don't need to assign them listeners

	public Diagram(LandscapeViewerCore ls) 
	{
		super(ls);
		setLayout(null);
		setLocation(0,0);
		setDoubleBuffered(true);
	}

	public Component add(Component c)
	{
		if (c instanceof RelationComponent) {
			return(m_edges.add(c));
		}
		if (c instanceof Cardinal) {
			return(m_cardinals.add(c));
		}
		return(super.add(c));
	}

	public void remove(Component c)
	{
		if (c instanceof RelationComponent) {
			m_edges.remove(c);
			return;
		}
		if (c instanceof Cardinal) {
			m_cardinals.remove(c);
			return;
		}
		super.remove(c);
	}

	public Diagram(LandscapeViewerCore ls, boolean makeBackup) 
	{
		super(ls);

		setLayout(null);
		setLocation(0, 0);

		m_exitFlag = new ExitFlag(this);
		m_exitFlag.setLocation(3, 3);

		m_supplierSet   = new SupplierSet(this);
		m_clientSet     = new ClientSet(this);	// Ignore suppliers when computing clients

		addMouseMotionListener(this);
		// The default empty instance

		// No marks are needed when the diagram is first drawn
		setPreserveEntityMarks(0);
		setPreserveRelationMarks(0);
		navigateTo(m_rootInstance);
	}

	// This logic attempts to assign a Rectangle to any component which lacks Rectangle information

	// Called from landscapeEditorCore.attach()
	// Called from landscapeEditorCore.loadLs()
	// Called from landscapeViewerCore.initialLoad()

	// Returns non-null string on error

	public String loadDiagram(String taPath, Vector bgPaths, Object context) {

		Attribute	attr;
		boolean		option;
		String		ret;

		ret = loadTA(taPath, bgPaths, context);

		attr = m_rootInstance.getLsAttribute(EDGEMODE_ID);
		if (attr.avi == null) { 
			attr.avi = new AttributeValueItem(String.valueOf(Do.DIRECT_EDGE_STATE));
		}

		LandscapeObject.setEdgeMode(attr.parseInt());

		attr = m_rootInstance.getLsAttribute(TOPCLIENTS_ID);
		if (attr.avi == null) {
			option   = Do.TOP_CLIENTS_DEFAULT;
			attr.avi = new AttributeValueItem(String.valueOf(option));
		} else {
			option   = attr.parseBoolean();
		}
		m_ls.setTopClients(option);

		attr = m_rootInstance.getLsAttribute(WANTCLIENTS_ID);
		if (attr.avi == null) {
			option   = Do.SHOW_CLIENTS_DEFAULT;
			attr.avi = new AttributeValueItem(String.valueOf(option));
		} else {
			option   = attr.parseBoolean();
		}
		m_ls.setShowClients(option);

		attr = m_rootInstance.getLsAttribute(WANTSUPPLIERS_ID);
		if (attr.avi == null) {
			option   = Do.SHOW_SUPPLIERS_DEFAULT;
			attr.avi = new AttributeValueItem(String.valueOf(option));
		} else {
			option   = attr.parseBoolean();
		}
		m_ls.setShowSuppliers(option);

		attr = m_rootInstance.getLsAttribute(WANTCARDINALS_ID);
		if (attr.avi == null) {
			option   = Do.SHOW_CARDINALS_DEFAULT;
			attr.avi = new AttributeValueItem(String.valueOf(option));
		} else {
			option   = attr.parseBoolean();
		}
		m_ls.setShowCardinals(option);

		attr = m_rootInstance.getLsAttribute(RELN_HIDDEN_ID);
		relVisibilityAttr = attr;

		AttributeValueItem avi = attr.avi;

		while(avi != null) {
			RelationClass rc = (RelationClass) m_relationClasses.get(avi.value);
			rc.setClassVisible(false); 
			avi = avi.next; 
		}

		setVisibilityFlags();

		return ret;
	}

	public boolean isUndoAvailable() 
	{
		return undoValid;
	}

	public Graphics getContext(Graphics gc) 
	{
		int vpwidth, vpheight;

		vpwidth = getWidth()-GAP*2;
		vpheight = getHeight() - GAP*2;
		return gc.create(getX()+GAP, getY()+GAP, vpwidth, vpheight);
	}

	public void rescaleDiagram() 
	{
//		System.out.println("Diagram.rescaleDiagram");
		m_drawRoot.rescaleChildren();
	}

	// 
	// Generates global coordinates for current scale factor
	//

	public void rescaleDiagram(int x, int y, int width, int height) 
	{
//		System.out.println("Diagram.rescaleDiagram {" + x + ", " + y + " " + width + "x" + height + "}");
		m_drawRoot.setBothBounds(0, 0, x, y, width, height);
	}

	public void deleteEdge(RelationInstance ri)
	{
		ri.getSrc().removeSrcRelation(ri);
		ri.getDst().removeDstRelation(ri);
	}

	public EntityInstance intersects(Rectangle lyt)
	{
		return m_drawRoot.intersects(lyt); 
	}

	public EntityInstance containing(Rectangle lyt) 
	{
		return m_drawRoot.containing(lyt);
	}

	public Vector setGroupRegion(Rectangle lyt) 
	{ 
		EntityInstance e = m_drawRoot.getMouseOver((int) lyt.x, (int) lyt.y);

		if (e == null) {
			return null;
		}
		return e.groupRegion(lyt);
	}

	public void setMode(int mode) 
	{
		this.mode = mode; 
	}


	public void setDrawBends(boolean state) 
	{
		MsgOut.dprintln("Draw bends set to: " + state);
		m_drawBends = state;
	}

	public boolean isDrawBends() 
	{
		return m_drawBends;
	}

	public void setEdgeMode(int mode) 
	{
		LandscapeObject.setEdgeMode(mode);
		Attribute attr = m_rootInstance.getLsAttribute(EDGEMODE_ID);
		attr.avi = new AttributeValueItem(String.valueOf(mode));
	}

	public int getEdgeMode() 
	{
		return LandscapeObject.getEdgeMode();
	}

	public EntityInstance getDrawRoot() 
	{
		return m_drawRoot; 
	}

	public void setDrawRoot(EntityInstance e)
	{
		if (e != m_drawRoot) {
			MapBox mapBox;

			m_drawRoot = e;
			e.setOpen();

			mapBox = m_ls.m_mapBox;
			if (mapBox != null) {
				mapBox.fill(e);
				mapBox.validate();
				mapBox.setVisible(true);
//				System.out.println("Diagram.setDrawRoot() map box added");
		}	}
	}

	public EntityInstance getRootInstance() 
	{
		return m_rootInstance;
	}

	public int getTimeStamp() 
	{
		return timeStamp;
	}

	public void toggleDstElision(EntityInstance e) 
	{
		Enumeration en;
		RelationClass rc;

		for (en = enumRelationClasses(); en.hasMoreElements(); ) {
			rc = (RelationClass) en.nextElement(); 
			if (!rc.excludeReln() && rc.isClassVisible()) {
				e.toggleDstElision(rc); 
			}
		}
	}

	public void toggleSrcElision(EntityInstance e) 
	{
		Enumeration en;
		RelationClass rc;

		for (en = enumRelationClasses(); en.hasMoreElements(); ) {
			rc = (RelationClass) en.nextElement();
			if (!rc.excludeReln() && rc.isClassVisible()) {
				e.toggleSrcElision(rc);
		}	}
	}

	public void toggleEnteringElision(EntityInstance e) 
	{

		Enumeration en;
		RelationClass rc;

		for (en = enumRelationClasses(); en.hasMoreElements(); ) {
			rc = (RelationClass) en.nextElement(); 
			if (!rc.excludeReln() && rc.isClassVisible()) {
				e.toggleEnteringElision(rc); 
		}	}
	}

	public void toggleExitingElision(EntityInstance e) 
	{
		Enumeration en;
		RelationClass rc;

		for (en = enumRelationClasses(); en.hasMoreElements();) {
			rc = (RelationClass) en.nextElement();
			if (!rc.excludeReln() && rc.isClassVisible()) {
				e.toggleExitingElision(rc);
		}	}
	}

	public void toggleInternalElision(EntityInstance e) 
	{
		Enumeration en;
		RelationClass rc;

		for (en = enumRelationClasses(); en.hasMoreElements(); ) {
			rc = (RelationClass) en.nextElement();
			if (!rc.excludeReln() && rc.isClassVisible())
				e.toggleInternalElision(rc);
		}
	}

	public void toggleContainElision(EntityInstance e) 
	{
		MsgOut.dprintln("Toggle contains elision on " + e.getLabel());
		e.toggleContainElision();
	}

	public void setVisible(boolean value)
	{
		if (m_drawRoot != null) {
			m_drawRoot.setVisible(value);
	}	}
	
	public void setEdgeVisibilityState(RelationClass rc, boolean state) 
	{
		rc.setClassVisible(state);

		// List is HIDDEN relations

		if (state) {
			relVisibilityAttr.removeFromList(rc.getId());
		} else {
			relVisibilityAttr.addToList(rc.getId());
		}
		setVisibilityFlags();
	}

	public void toggleEdgeVisibility() 
	{
		Enumeration en;
		RelationClass rc;

		for (en = enumRelationClasses(); en.hasMoreElements();) {
			rc = (RelationClass) en.nextElement();
			if (!rc.excludeReln() && rc.isClassVisible()) {
				setEdgeVisibilityState(rc, false);
			}
		}
	}

	// Only draw the edges found

	public void clearDrawEdges() 
	{
//		System.out.println("clearDrawEdges");
		m_drawEdges = false;
//		java.lang.Thread.dumpStack();
//		System.out.println("-----");
	}

	// Draw all edges

	public void setDrawEdges() 
	{
//		System.out.println("setDrawEdges");
		m_drawEdges = true;
//		java.lang.Thread.dumpStack();
//		System.out.println("-----");
	}

	public boolean getDrawEdges()
	{
		return(m_drawEdges);
	}

	public Vector getHighlightedEdges() 
	{
		Vector v = new Vector();

		m_drawRoot.getHighlightedEdges(v);
		if (v.size() == 0) {
			return null;
		}
		return v;
	}

	public void clearKeyEntity()
	{
		m_keyEntity = null;
	}

	public boolean clearFlags() 
	{
		boolean ret;

		clearKeyEntity();

		ret = false;
		if (!getDrawEdges()) {
			ret = true;
			setDrawEdges();
		}
		m_rootInstance.clearAllFlags();
		if (m_drawRoot != null) {
			m_drawRoot.setOpen();
		}
		return(ret);
	}

	// Remove all RedBox and HighLight flags under the root

	public void clearQueryFlags() 
	{
		m_rootInstance.clearQueryFlags();
	}

	public void clearGroupFlags() 
	{
		clearKeyEntity();
		setDrawEdges();
		m_rootInstance.clearGroupFlags();
	}

	public boolean toggleQueryPersistence() 
	{
		persistentQuery = !persistentQuery;
		return persistentQuery;
	}

	public boolean getQueryPersistance() 
	{
		return persistentQuery;
	}

	/* Programatically create a new entity */

	public EntityInstance newEntity(EntityInstance container)
	{
		EntityInstance	e;
		EntityClass		ec;
		int				n;
		String			ename;

		for (n = 0; ; ++n) {
			ename = "Entity#" + n;
			if (!entityExists(ename)) {
				break;
		}	}

		e = new EntityInstance(defaultEntityClass, ename);
		putEntity(e);
		ec = e.getEntityClass();
		if (ec != null) {
			ec.makeInstanceOfUs(e);
		}
		e.setxRelLocal(0.25);
		e.setyRelLocal(0.25);
		e.setwidthRelLocal(0.5);
		e.setheightRelLocal(0.5);
		e.rescale(container.getDiagramX(), container.getDiagramY(), container.getWidth(), container.getHeight());

		container.addContainment(e);
		prepostorder();
		EditAttribute.Create(m_ls, e);
		return(e);
	}

	// Programatically create a new entity

	public EntityInstance addEntity(String initId, Rectangle lyt, EntityInstance container)
	{
		EntityInstance	e;
		EntityClass		parent;

		e = new EntityInstance(defaultEntityClass, initId);
		e.setDiagramBounds(lyt);
		putEntity(e);

		EditAttribute.Create(m_ls, e);
		parent = e.getEntityClass();

		if (parent != null) {
			parent.makeInstanceOfUs(e);
		}
		container.addContainment(e);
		prepostorder();
		m_rootInstance.rescaleChildren();
		return e;
	}

	// Programatically delete an entry

	public EntityInstance DeleteEntity(EntityInstance e) 
	{
		if (e != null) {
			if (e == getDrawRoot()) {
				m_ls.doFeedback("Can't delete the draw root");
				return(null);
			}
			e.deleteMe();
			m_ls.doFeedback("Entity " + e.getLabel() + " and its contents have been deleted.");
		}
		return(e);
	}

	public int numVisibleRelationClasses() 
	{
		return this.numVisibleRelationClasses;
	}

	public Vector getGroup() 
	{
		Vector grp = new Vector();

		m_rootInstance.getGroup(grp);
		if (grp.isEmpty()) {
			return null;
		}
		return grp;
	}

	public Rectangle getGroupBoundingBox() 
	{
		Vector grp = getGroup();

		if (grp == null || grp.isEmpty()) {
			return null;
		}

		// Determine the bounding box of whole group

		Enumeration en0;
		double x1, y1, x2, y2;


		x1 = Double.MAX_VALUE;
		y1 = Double.MAX_VALUE;
		x2 = Double.MIN_VALUE;
		y2 = Double.MIN_VALUE;

		for (en0 = grp.elements(); en0.hasMoreElements(); ) {
			EntityInstance e = (EntityInstance) en0.nextElement();
			Rectangle lyt = e.getDiagramBounds();

			if (lyt.x < x1) 
				x1 = lyt.x;

			if (lyt.y < y1)
				y1 = lyt.y;

			if (lyt.x + lyt.width > x2)
				x2 = lyt.x + lyt.width;

			if (lyt.y + lyt.height > y2)
				y2 = lyt.y + lyt.height;
		}
		return new Rectangle((int) x1, (int) y1, (int) (x2-x1), (int) (y2-y1));
	}

	public EntityInstance getKeyEntity() 
	{
		return m_keyEntity;
	}

	public void setKeyEntity(EntityInstance e) {

		if (m_keyEntity != null) {
			m_keyEntity.clearGroupKeyFlag();		// Must do this first
			m_keyEntity.setGroupFlag();
		}
		m_keyEntity = e;
		m_keyEntity.setGroupKeyFlag();
		m_keyEntity.setGroupFlag();
	}

	public Vector getRedBoxGroup() 
	{
		Vector grp = new Vector();

		m_rootInstance.getRedBoxGroup(grp);
		if (grp.isEmpty()) {
			return null;
		}
		return grp;
	}

	public void selectEdges(Vector edges) 
	{
		Enumeration en;
		RelationInstance r;

		for (en = edges.elements(); en.hasMoreElements(); ) {
			r = (RelationInstance) en.nextElement();
			r.setGroupAndHighlightFlag();
		}
		clearDrawEdges();
	}

	public boolean isOverGroupRegion(int x, int y) 
	{
		Rectangle lyt = getGroupBoundingBox();

		if (lyt != null) {
			return lyt.contains(x, y);
		}
		return false;
	}

	public int getGrid() 
	{
		return gridPixels;
	}

	public void setGrid(int val) 
	{
		gridPixels = val;
	}

	public void flagChanged() {

		changedFlag = true;

	}

	public boolean getChangedFlag()
	{
		return changedFlag;
	}

	public void resetChangedFlag() 
	{
		changedFlag = false;
	}

	public void setToViewport() 
	{
		navigateTo();
	}

	public void fitTo(EntityInstance e) 
	{
		e.fitTo(true);
		rescaleDiagram();
	}

	// Mark all edges as needing to be repainted

	public void invalidateAllEdges(EntityInstance root)
	{
		Enumeration en;

		/* Draw all the visible objects */

		root.invalidateAllEdges(); 

		if (root.isOpen()) {
			for (en = root.getChildren(); en.hasMoreElements(); ) {
				invalidateAllEdges((EntityInstance) en.nextElement());
		}	}
	}

	// Draw all edges marked as needing to be repainted at or under e

	protected void drawHighlightEdges(EntityInstance e)
	{
		Enumeration en;

		/* Draw all the highlight edges */

		e.drawHighlightedEdges();
		if (e.isOpen()) {
			for (en = e.getChildren(); en.hasMoreElements(); ) {
				drawHighlightEdges((EntityInstance) en.nextElement());
		}	}
	}
	// Draw all edges marked as needing to be repainted at or under e

	protected void drawAllEdges(EntityInstance e, EntityInstance under)
	{
		Enumeration en;

		/* Draw all the visible edges */

		e.drawAllEdges(under); 
		if (e.isOpen()) {
			for (en = e.getChildren(); en.hasMoreElements(); ) {
				drawAllEdges((EntityInstance) en.nextElement(), under);
		}	}
	}

	protected void doDrawEdges(EntityInstance e, EntityInstance under)
	{
		/* Draw all the visible objects */

		if (getDrawEdges()) {
			if (m_visibleEdges) {
				drawAllEdges(e, under); 
			}
		} else {
			drawHighlightEdges(e);
	}	}

	/* If this isn't changed to return true the revalidate() becomes a
	   request to revalidate the JScrollPane containing us and that for
	   some reason doesn't attempt to validate() the thing it contains.
	 */

	public boolean isValidateRoot()
	{
		return(true);
	}

	public Dimension getPreferredSize()
	{
		Dimension	preferredSize;
		JComponent	container;
		Insets		insets;
		int			mw, mh;

		container = (JComponent) getParent();
		if (container == null) {
			mw = 0;
			mh = 0;
		} else {
			insets = container.getInsets();

			mw = container.getWidth()-insets.right-insets.left;
			if (mw <= 0) {
				mw = 0;
			}
			mh = container.getHeight()-insets.bottom-insets.top;
			if (mh <= 0) {
				mw = 0;
			}
			mw *= m_zoom_x;
			mh *= m_zoom_y;
		}
		preferredSize = super.getPreferredSize();
		if (preferredSize.width != mw || preferredSize.height != mh) {
			preferredSize.setSize(mw, mh);
			setPreferredSize(preferredSize);
			container.revalidate();
		}
		return(preferredSize);
	}


/* All complex stuff is delayed until we validate to ensure that we have a
   graphic object so that we can compute sizes of labels etc. Much of this
   stuff doesn't really belong in validate -- which is supposed to be called
   only whenever the diagram changes its shape.
 */

	public void validate()
	{
		Dimension		size;
		int				mw, mh, yshift;
		Vector			clients, suppliers;
		Enumeration		en;
		Container		container;
		EntityInstance	e;
		boolean			isTopClients, liftEdges;
		int				temp;

		m_loaded = false;

		size = getSize();
		mw   = size.width;
		mh   = size.height;
//		System.out.println("Diagram.validate Started validation mw=" + mw + " mh=" + mh);

		if (mw < 1 || mh < 1) {
			return;
		}

		m_ls.waitCursorOn();

/*
		System.out.println("Diagram.validate() " + getBounds() + ".v." + "[" + m_zoom_x + "," + m_zoom_y + "]");
		java.lang.Thread.dumpStack();
		System.out.println("-----");
*/

		if (m_cardinals == null) {
			m_cardinals = new Container();
			m_cardinals.setLayout(null);
		}
		if (m_edges == null) {
			m_edges = new Container();
			m_edges.setLayout(null);
		}

		removeAll();
		m_cardinals.removeAll();

		// These go on top
		if (getLs().isShowCardinals()) {
			m_cardinals.setBounds(0, 0, mw, mh);
			add(m_cardinals);
		}
		// This goes under cardinals if present
		m_edges.removeAll();
		m_edges.setBounds(0,0,mw, mh);
		add(m_edges);		// Add the set of all edges


		// Clear all but the indicated marks

		if (!getDrawEdges()) {
			m_preserve_entity_marks   |= EntityInstance.REDBOX_MARK;
			m_preserve_relation_marks |= RelationInstance.HIGHLIGHT_FLAG_MARK;
		}

		m_rootInstance.clearAllMarks(m_preserve_entity_marks, m_preserve_relation_marks);
		m_drawRoot.setOpen();	// The draw root is always open

//		System.out.println("Diagram.validate() cleared all marks");

		/* This has to be delayed until we have resized things appropriately since the adding of
		   the drawRoot to the diagram causes the edges to also be added, and these need to know
		   displayable dimensions of entities
		 */

		mw       -= GAP*6;		// Maximum width  of e
		mh	     -= GAP*4;		// Maximum height of e
//		System.out.println("mw=" + mw + " mh=" + mh);
		yshift    = 0;
		clients   = null;
		suppliers = null;
		isTopClients = m_ls.isTopClients();

		// Make entity take a full footprint in the display.

		if (m_drawRoot != m_rootInstance) {
			add(m_exitFlag);
			m_exitFlag.setVisible(true);
		} else {
			// If we don't do a repaint the [-] box doesn't vanish
			repaint();
		}

//		System.out.println("Diagram.validate() added to Diagram");

		/* This performs a first cut at finding clients and suppliers
		 * It essentially collects into a vector topmost clients and suppliers
		 */

		findClientsAndSuppliers();

//		System.out.println("Diagram.validate() found clients and suppliers");

		if (m_clientSet.getFullSetSize() != 0) {
			// we will either show or report not showing
			// Reduce height to allow clients
			temp = (ClientSupplierSet.CLIENT_SUPPLIER_HEIGHT + GAP*4);
			mh  -= temp;
			if (isTopClients) {
				m_clientSet.setBounds(GAP*3, GAP*2, mw, ClientSupplierSet.CLIENT_SUPPLIER_HEIGHT);
				yshift = temp; 
			} else {
				m_clientSet.setBounds(GAP*3, getHeight() - ClientSupplierSet.CLIENT_SUPPLIER_HEIGHT - GAP*2, mw, ClientSupplierSet.CLIENT_SUPPLIER_HEIGHT);
			}
//			System.out.println("Diagram.validate() m_clientSet=" + m_clientSet.getBounds());
		}

		if (m_supplierSet.getFullSetSize() != 0) {			// Reduce height to allow suppliers
			temp = (ClientSupplierSet.CLIENT_SUPPLIER_HEIGHT + GAP*4);
			mh  -= temp;
			if (!isTopClients) {
				m_supplierSet.setBounds(GAP*3, GAP*2, mw, ClientSupplierSet.CLIENT_SUPPLIER_HEIGHT);
				yshift = temp; 
			} else {
				m_supplierSet.setBounds(GAP*3, getHeight() - ClientSupplierSet.CLIENT_SUPPLIER_HEIGHT - GAP*2, mw, ClientSupplierSet.CLIENT_SUPPLIER_HEIGHT);

//				System.out.println("Supplier=" + (getY() + m_supplierSet.getY() + m_supplierSet.getHeight()) + " actual=" + m_ls.getHeight());
			}
//			System.out.println("Diagram.validate() m_supplierSet=" + m_supplierSet.getBounds());
		}

		ClientSupplierSet.compact(m_clientSet, m_supplierSet);

		clients = m_clientSet.getFullSet();
		if (clients != null && clients.size() == 0) {
			clients = null;
		}

		suppliers = m_supplierSet.getFullSet();
		if (suppliers != null && suppliers.size() == 0) {
			suppliers = null;
		}
		rescaleDiagram(GAP*3, GAP*2+yshift, mw, mh);
//		System.out.println("Diagram.validate() rescaled Diagram");

		m_drawRoot.addUnder(this);

		liftEdges = m_ls.isLiftEdges();

		for (en = m_drawRoot.getChildren(); en.hasMoreElements(); ) {
			e = (EntityInstance) en.nextElement();
			e.computeAllDiagramEdges(m_drawRoot, clients, suppliers, null /* No closed entity yet */, liftEdges);
		}
//		System.out.println("Diagram.validate() computed all diagram edges");

		for (en = m_drawRoot.getChildren(); en.hasMoreElements(); ) {
			e = (EntityInstance) en.nextElement();
			e.liftAllDiagramEdges(m_drawRoot, liftEdges);
		}

//		System.out.println("Diagram.validate() lifted all edges");

		// Do after lifted edges since ordering based on where edges go to/come from

		if (m_supplierSet.getFullSetSize() != 0) {	
			m_supplierSet.order();
			add(m_supplierSet);		// Add the quasi supplier box to diagram
			m_supplierSet.setVisible(true);
//			System.out.println("Diagram.validate() added Suppliers");
		}

		if (m_clientSet.getFullSetSize() != 0) {
			m_clientSet.order();
			add(m_clientSet);		// Add the quasi client box to the diagram
			m_clientSet.setVisible(true);
//			System.out.println("Diagram.validate() added Clients");
		}

		doDrawEdges(m_drawRoot, m_drawRoot);	// Marks have been previously cleared so do not need to invalidate

		if (getLs().isShowCardinals()) {
			EntityInstance	drawInstance;
		
			drawInstance = (EntityInstance) m_drawRoot;
			drawInstance.resetCardinals(m_numRelationClasses);
			drawInstance.calcEdgeCardinals(null, !getDrawEdges());
			drawInstance.showCardinals();
//			System.out.println("Show cardinals " + m_cardinals.getComponentCount());
		}

//		System.out.println("Diagram.validate() processed edges");

		m_ls.waitCursorOff();

		m_loaded = true;
		repaint();

//		System.out.println("Diagram.validate() done");
	}

	public void redrawDiagram()
	{
		revalidate();
	}

	public boolean shouldScrollDiagram()
	{
		return(m_zoom_x > 1.0 || m_zoom_y > 1.0);
	}

	public boolean zoom(double xfactor, double yfactor) 
	{
		if (xfactor <= 1.0 && yfactor <= 1.0 && m_zoom_x == 1.0 && m_zoom_y == 1.0) {
			return(false);
		}
		m_zoom_x = m_zoom_x * xfactor;
		m_zoom_y = m_zoom_y * yfactor;

		if (m_zoom_x < 1.0) {
			m_zoom_x = 1.0;
		}
		if (m_zoom_y < 1.0) {
			m_zoom_y = 1.0;
		}
		// Update our preferred size
		getPreferredSize();	

//		System.out.println("Zoom");
		redrawDiagram();
		return true;
	}

	public boolean set_to_viewport()
	{
		if (!shouldScrollDiagram()) {
			return(false);
		}
		m_zoom_x = 1.0;
		m_zoom_y = 1.0;
		// Update our preferred size
		getPreferredSize();	
		redrawDiagram();
		return true;
	}

	// Carefully decide on values for preserveEntityMarks and preserveRelationMarks before calling this

	public void navigateTo(EntityInstance e) 
	{
		// Make entity take a full footprint in the display.

		setDrawRoot(e);
		redrawDiagram();
	}

	public void navigateTo(String eid) 
	{
		EntityInstance e = getEntity(eid);

		if (e != null) {
			navigateTo(e);
	}	}

	// Navigate to default entity

	public void navigateTo()
	{
		Enumeration		en;
		EntityInstance	e;
			
		en = m_rootInstance.getChildren();
		e  = m_rootInstance; 
		if (en.hasMoreElements()) {
			e = (EntityInstance) en.nextElement();
			if (en.hasMoreElements()) {
				e = m_rootInstance;
		}	}

		setPreserveEntityMarks(0);
		setPreserveRelationMarks(0);
		navigateTo(e);
	}

	public void navigateToDrawRootParent()
	{
		EntityInstance e;

		e = m_drawRoot.getEnterableParent();

		if (e == null) {
			m_ls.error("At topmost landscape");
		} else {
			// Going up
			setPreserveEntityMarks(0);
			setPreserveRelationMarks(0);
			navigateTo(e);
	}	}

	public boolean entityExists(String name) 
	{
		return (getEntity(name) != null);
	}

	public boolean allowElision() 
	{
		return getDrawEdges();
	}

	public void setModeHandler(LandscapeModeHandler modeHandler)
	{
//		System.out.println("Diagram.setModeHandler " + modeHandler);
		if (modeHandler != m_modeHandler) {
			if (modeHandler != null) {
				// Make sure mode handler knows it is talking to us before
				// we use it.
				modeHandler.select(this);
			}
			m_modeHandler = modeHandler;
	}	}

	public LandscapeModeHandler getModeHandler()
	{
		return(m_modeHandler);
	}

	public boolean isModeHandlingActive()
	{
		return(m_modeHandlingActive);
	}

	public void setModeHandlingActive(boolean value)
	{
		m_modeHandlingActive = value;
	}

	public RelationInstance targetRelation(Object object)
	{
		if (object != null) {
			if (object instanceof Vector) {
				Vector v = (Vector) object;
				switch (v.size()) {
				case 0:
					m_ls.doFeedback("Target relation set is empty");
					break;
				case 1:
					return((RelationInstance) v.firstElement());
				default:
					m_ls.doFeedback("Target relation set has multiple objects in it");
				}
				return(null);
			}
			return((RelationInstance) object);
		}

		Vector v = getHighlightedEdges();

		if (v == null) {
			m_ls.doFeedback("Highlight the edge you wish to work on");
		} else {
			if (v.size() == 1) {
				return((RelationInstance) v.firstElement());
			}
			m_ls.doFeedback("Can't operate on " + v.size() + " highlight edges");
		}
		return(null);
	}

	public EntityInstance targetEntity(Object object)
	{
		if (object != null) {
			if (object instanceof Vector) {
				Vector v = (Vector) object;
				switch (v.size()) {
				case 0:
					m_ls.doFeedback("Target entity set is empty");
					break;
				case 1:
					return((EntityInstance) v.firstElement());
				default:
					m_ls.doFeedback("Target entity set has multiple objects in it");
				}
				return(null);
			}
			return((EntityInstance) object);
		}

		EntityInstance ke;

		ke = getKeyEntity();
		if (ke != null) {
			return(ke);
		}
		return(getDrawRoot());
	}

	public Vector targetRelations(Object object)
	{
		Vector v;

		if (object != null) {
			if (object instanceof Vector) {
				v = (Vector) object;
				if (v.size() > 0) {
					return(v);
				}
				m_ls.doFeedback("Target relation set is empty");
				return(null);
			}
			v = new Vector();
			v.addElement(object);
			return(v);
		}

		v = getHighlightedEdges();

		if (v == null) {
			m_ls.doFeedback("Select the edges you wish to work on");
		}
		return(v);
	}

	public Vector targetEntities(Object object)
	{
		Vector v;

		if (object != null) {
			if (object instanceof Vector) {
				v = (Vector) object;
				if (v.size() > 0) {
					return(v);
				}
				m_ls.doFeedback("Target entity set is empty");
				return(null);
			}
			v = new Vector();
			v.addElement(object);
			return(v);
		}
		return(getGroup());
	}

	public void entityPressed(MouseEvent ev, EntityInstance e, int x, int y)
	{
		m_ls.showDescription(e, true);
		if (m_modeHandler != null) {
			setModeHandlingActive(true);
			if (!m_modeHandler.entityPressed(ev, e, x, y)) {
				m_modeHandler.cleanup();
				setModeHandlingActive(false);
	}	}	}

	public void entityDragged(MouseEvent ev, EntityInstance e, int x, int y)
	{		
		if (isModeHandlingActive()) {
			if (m_modeHandler.entityDragged(ev, e, x, y)) {
				return;
		}	}
		movedOverThing(ev, e, x, y); // Must remain as unadjusted values!
	}

	public void entityReleased(MouseEvent ev, EntityInstance e, int x, int y)
	{
//		System.out.println("Diagram.entityReleased=" + e);

		if (isModeHandlingActive()) {
			LandscapeModeHandler modeHandler = getModeHandler();

			modeHandler.entityReleased(ev, e, x, y);
			modeHandler.cleanup();
			setModeHandlingActive(false);
		}

		if (y > 0) {
			getLs().toolButton[0].requestFocus();
			getLs().requestFocus();
	}	}		

	public void relationPressed(MouseEvent ev, RelationInstance ri, int x, int y)
	{
//		System.out.println("Diagram.relationPressed=" + ri);

		m_ls.showDescription(ri, true);
		if (m_modeHandler != null) {
			setModeHandlingActive(true);
			if (!m_modeHandler.relationPressed(ev, ri, x, y)) {
				m_modeHandler.cleanup();
				setModeHandlingActive(false);
	}	}	}
	
	public void relationDragged(MouseEvent ev, RelationInstance ri, int x, int y)
	{		
		if (isModeHandlingActive()) {
			if (m_modeHandler.relationDragged(ev, ri, x, y)) {
				return;
		}	}
		movedOverThing(ev, ri, x, y); // Must remain as unadjusted values!
	}
		
	public void relationReleased(MouseEvent ev, RelationInstance ri, int x, int y)
	{
//		System.out.println("Diagram.relationReleased=" + ri);

		if (isModeHandlingActive()) {
			LandscapeModeHandler modeHandler = getModeHandler();

			modeHandler.relationReleased(ev, ri, x, y);
			modeHandler.cleanup();
			setModeHandlingActive(false);
		}

		if (y > 0) {
			getLs().toolButton[0].requestFocus();
			getLs().requestFocus();
	}	}

	public void movedOverThing(MouseEvent ev, Object thing, int x, int y)
	{
		m_ls.resetAnticipateCursor();
		if (m_modeHandler != null) {
			m_modeHandler.movedOverThing(ev, thing, x, y);
		}
		m_ls.useAnticipateCursor();
	}

	// MouseMotionListener interface

	public void mouseDragged(MouseEvent ev)
	{
	}

	public void mouseMoved(MouseEvent ev)
	{
//		System.out.println("Diagram.mouseMoved");
		movedOverThing(ev, this, ev.getX(), ev.getY());
	}

	// DiagramCoordinates interface

	public int getDiagramX()
	{
		return(0);
	}

	public int getDiagramY()
	{
		return(0);
	}
}

