package lsedit;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Polygon;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.*;

import java.util.*;
import java.io.*; 
import javax.swing.*;

// Entities in a landscape 

public class EntityComponent extends JComponent implements Icon {

	// Final values
	
	private final static int MARGIN            = 5; 
	
	/* Used in computing rules for shape of a folder */

	public  final static int MIN_FLAP_HT       = 3;
	public  final static int MAX_FLAP_HT       = 8;
	public  final static int FLAP_MARGIN       = 4;
	public  final static int TINY_FLAP_WD      = 10;
	public  final static int MIN_FLAP_WD       = 75;

	/* Position and size of the contents flag when entity closed */

	public final static int CONTENTS_FLAG_X   = 3;
	public final static int CONTENTS_FLAG_Y   = 9;
	public final static int CONTENTS_FLAG_DIM = 8;

	// These are the top left coordinates of the object on the screen

	private EntityInstance m_entityInstance; 

	// Sorting uses this value and it is really expensive to recompute for each key compare

	private double		   m_avgX;

	// --------------
	// Public methods
	// --------------

 	public EntityComponent(EntityInstance entityInstance) 
	{
		m_entityInstance = entityInstance;

		setLayout(null);
		entityInstance.setSwingObject(this);
		setBounds(entityInstance.getX(), entityInstance.getY(), entityInstance.getWidth(), entityInstance.getHeight());
	}

	public String toString() 
	{
		return "EntityComponent: " + m_entityInstance.toString();
	}

	public EntityInstance getEntityInstance()
	{
		return(m_entityInstance);
	}

	public void setAvgX(double avgX)
	{
		m_avgX = avgX;
	}

	public double getAvgX()
	{
		return m_avgX;
	}

	protected void setGraphicsColor(Graphics g, Color explicit, EntityInstance entityInstance) 
	{
		if (explicit != null) {
			g.setColor(explicit);
		} else {
			if (entityInstance.red_closed()) {
				g.setColor(Color.red.darker());
			} else {
				g.setColor(entityInstance.getCurrentObjectColor());
		}	}
	}

	public void paintShape(Graphics g, int x, int y, int width, int height, boolean fill, Color fillColor) 
	{
		if (width > 0 && height > 0) {
			EntityInstance entityInstance = m_entityInstance;

			switch (entityInstance.getStyle()) {
				case EntityClass.ENTITY_STYLE_3DBOX:
				{
					if (fill) {
						setGraphicsColor(g, fillColor, entityInstance);
						g.fill3DRect(x+1, y+1, width-1, height-1, true);
					}
					setGraphicsColor(g, null, entityInstance);
					g.draw3DRect(x, y, width, height, true);
					break;
				}

				case EntityClass.ENTITY_STYLE_2DBOX:
				{
					if (fill) {
						setGraphicsColor(g, fillColor, entityInstance);
						g.fillRect(x, y, width, height);
					}
					g.setColor(Color.black);
					g.drawRect(x, y, width, height);
					break;
				}

				case EntityClass.ENTITY_STYLE_FILE:
				{
					int fd = Math.min(Math.min(width, height)/2, 8 /* FLAP_DIM */);

					int[] xp = new int[6];
					int[] yp = new int[6];

					/*
					   0--(width-fd)----1 \        
					   |                     \    (fd)
					   |				        \ 
					   |                           2
					   | 						    |
					   |						 (height)
					   |							|
					   4 -------------------------- 3
					 */

					xp[0] = x;
					yp[0] = y;
					xp[1] = x + width - fd;
					yp[1] = y;
					xp[2] = x + width - 1;
					yp[2] = y + fd;
					xp[3] = xp[2];
					yp[3] = y + height - 1;
					xp[4] = x;
					yp[4] = yp[3];
					xp[5] = xp[0];
					yp[5] = yp[0];

					if (fill) {
						setGraphicsColor(g, fillColor, entityInstance);
						g.fillPolygon(xp, yp, 6);
					}
					g.setColor(Color.darkGray);
					g.drawPolygon(xp, yp, 6);

					xp[0] = xp[1];
					yp[0] = yp[2];
					xp[3] = xp[0];
					yp[3] = yp[0];

					g.drawPolygon(xp, yp, 4);
					break;
				}

				case EntityClass.ENTITY_STYLE_DISK:
				{
					int ad = Math.min(Math.min(width, height)/2, 8 /* ARC_DIM */);

					// Draw a cylinder

					if (fill) {
						setGraphicsColor(g, fillColor, entityInstance);
						g.fillRect(x, y+ad/2, width, height-ad);
						g.fillOval(x, y, width, ad);
						g.fillOval(x, y+height-ad, width, ad);
					}
					g.setColor(Color.black);

					g.drawOval(x, y, width, ad);
					g.drawArc(x, y+height-ad-1, width, ad, 180, 180);
					g.drawLine(x, y+ad/2, x, y+height-ad/2);
					g.drawLine(x+width-1, y+ad/2, x+width-1, y+height-ad/2);
					break;
				}

				case EntityClass.ENTITY_STYLE_FOLDER:
				{
					// Compute flap size

					int fw = ((int) (((double) width) * .4));
					int fh = Math.max(MIN_FLAP_HT, Math.min(MAX_FLAP_HT, ((int) (height * .2))));
					int fm = FLAP_MARGIN;

					if (fw < MIN_FLAP_WD) {
					   fw += fw/2;
					}
					if (fw < TINY_FLAP_WD) {
						fw = Math.min(fw + FLAP_MARGIN, width - width/3);
						fm = 0;
					}

				/*   
                     <--------------fw-------------------------->

                                     2--------------------------3
                   (fh)             /                            \
				     0<--- fm---->1 fh/2                      fh/2  4---------------------------5
				     |                                                                         |
					 |																		   |
					 |																		   |
					 7------------------------------------------------------------------------ 6
			     */

					int[] xp = new int[9];
					int[] yp = new int[9];

					xp[0] = x;
					yp[0] = y+fh;
					xp[1] = x+fm;
					yp[1] = y+fh;
					xp[2] = xp[1]+fh/2;
					yp[2] = y;
					xp[3] = x + fw;
					yp[3] = yp[2];
					xp[4] = xp[3]+fh/2;
					yp[4] = yp[0];
					xp[5] = x + width-1;
					yp[5] = yp[0];
					xp[6] = xp[5];
					yp[6] = y + height-1;
					xp[7] = xp[0];
					yp[7] = yp[6];
					xp[8] = xp[0];
					yp[8] = yp[0];

					if (fill) {
						setGraphicsColor(g, fillColor, entityInstance);
						g.fillPolygon(xp, yp, 9);
					}

					g.setColor(entityInstance.red_open() ? Color.red : Color.black);
					g.drawPolygon(xp, yp, 9);

					if (!entityInstance.isOpen()) {
						g.drawLine(xp[1], yp[1], xp[4], yp[4]);
					}
					break;
				}

				case EntityClass.ENTITY_STYLE_SOURCEOBJ:
				{
					if (fill) {
						setGraphicsColor(g, fillColor, entityInstance);
						g.fillOval(x, y, width, height);
					}
					g.setColor(m_entityInstance.red_open() ? Color.red : Color.black);
					g.drawOval(x, y, width, height);
					break;
				}

				case EntityClass.ENTITY_STYLE_CLASS:
				{
					int arc_w = width/5;
					int arc_h = height/5;
					int arc   = Math.min(arc_w, arc_h);

					if (fill) {
						setGraphicsColor(g, fillColor, entityInstance);
						g.fillRoundRect(x, y, width, height, arc, arc);
					}
					g.setColor(Color.black);
					g.drawRoundRect(x, y, width, height, arc, arc);
					break;
				}

				case EntityClass.ENTITY_STYLE_GROUP:
				{
					Util.drawOutlineBox(g, x, y, width, height, true /* Has 3D look */);
					break;
				}

				case EntityClass.ENTITY_STYLE_LABELLED_GROUP:
				{
					g.setFont(EntityInstance.smallFont);
					Util.drawGroupBox(g, x, y, width, height, entityInstance.getLabel(), entityInstance.getCurrentLabelColor(), true /* Has 3D look */);
					break;
				}
/*
				default:
				{
					// For debugging (This shows the root)
					g.setColor(Color.red);
					g.drawLine(x, y,        x+width-1, y+height-1);
					g.drawLine(x, y+height-1, x+width-1, y);
				}
*/
	}	}	}	

	public void paintMap(Graphics g, int x, int y, int width, int height, EntityInstance onPath, int depth)
	{
		if (width > 0 && height > 0) {
			EntityInstance	entityInstance = m_entityInstance;
			Enumeration		children;
			EntityInstance	child;
			int				x1, y1,width1, height1;
			Color			color;

			if (entityInstance == onPath) {
				color = Color.green;
			} else {
				color = entityInstance.getObjectColor();
			} 
			paintShape(g, x, y, width, height, !entityInstance.hasChildren() || depth == 1 /* then fill */, color);

			if (depth < 1) {
				for (children = entityInstance.getChildren(); children.hasMoreElements(); ) {
					child = (EntityInstance) children.nextElement();
					x1      = x + (int) (width  * child.xRelLocal());
					y1      = y + (int) (height * child.yRelLocal()); 
					width1  = (int) (width  * child.widthRelLocal());
					height1 = (int) (height * child.heightRelLocal());
					child.paintMap(g, x1, y1, width1, height1, onPath, depth+1);
	}	}	}	}

	protected void drawTopLeftLabel(Graphics g) 
	{
		EntityInstance entityInstance = m_entityInstance;

		switch (entityInstance.getStyle()) {
		case EntityClass.ENTITY_STYLE_FOLDER:

			g.setFont(EntityInstance.smallFont);
			// Compute flap size
			int fw = (int) (((double) getWidth()) * .4);
			int fh = Math.max(MIN_FLAP_HT, Math.min(MAX_FLAP_HT, ((int) (getHeight() * .2))));

			if (fw < MIN_FLAP_WD) {
			   fw += fw/2;
			}
			if (fw < TINY_FLAP_WD) {
				fw = Math.min(fw + FLAP_MARGIN, getWidth() - getWidth()/3);
			}
			Util.drawStringClipped(g, entityInstance.getLabel(), FLAP_MARGIN+fh/2+2, 0, (double) fw, getHeight()-MARGIN*2);
			break;
		case EntityClass.ENTITY_STYLE_CLASS:
			g.setFont(EntityInstance.openFont);
			Util.drawStringClipped(g, entityInstance.getLabel(), MARGIN*3, MARGIN, getWidth()-MARGIN*2, getHeight()-MARGIN*2);
			break;
		default:
			g.setFont(EntityInstance.smallFont);
//			g.setColor(Color.black); 
			Util.drawStringClipped(g, entityInstance.getLabel(), MARGIN, MARGIN, getWidth()-MARGIN*2, getHeight()-MARGIN*2);
		}
	}

	public void paintComponent(Graphics g) 
	{
		EntityInstance	entityInstance = m_entityInstance;
		int	width                      = getWidth();
		int height                     = getHeight();
		
//		System.out.println("EntityInstance.paintComponent: " + this + " bounds=" + getBounds() + " clip=" + g.getClipBounds() + " style=" + getStyle() + " parentClass=" + getParentClass());		// IJD
/*		java.lang.Thread.dumpStack();
		System.out.println("-----");
*/

		paintShape(g, 0, 0, width, height, true /* fill */, null /* No given fill color */); 

		if (entityInstance.isMarked(EntityInstance.HIGHLIGHT_EDGE_MARK)) {
			g.setColor(Color.white);
			g.drawRect(0, 0, width, height);
			g.drawRect(1, 1, width-1, height-1);
		}

		if (entityInstance.close_with_children_under_drawroot()) {

			/*  Draw a small mark as shown in top left of object

				x---
				|
				|   --------
				|  |        |
				   |    |   |
				   |    |   |
	  			   |  ----  |
				   |    |   |
				   |    |   |
				   |        |
				   |________|
			*/

			if (width > CONTENTS_FLAG_X+CONTENTS_FLAG_DIM && height > CONTENTS_FLAG_Y+CONTENTS_FLAG_DIM) {
				int x1, y1;
		
			//	g.setColor(entityInstance.getObjectColor().darker());
				g.setColor(new Color(entityInstance.getObjectColor().getRGB() ^ 0xFFFFFF));

				g.drawRect(CONTENTS_FLAG_X, CONTENTS_FLAG_Y, CONTENTS_FLAG_DIM /* 8 */, CONTENTS_FLAG_DIM);

				g.drawLine(CONTENTS_FLAG_X+1,                     CONTENTS_FLAG_Y+(CONTENTS_FLAG_DIM/2), CONTENTS_FLAG_X+CONTENTS_FLAG_DIM-1,   CONTENTS_FLAG_Y+(CONTENTS_FLAG_DIM/2));
				g.drawLine(CONTENTS_FLAG_X+(CONTENTS_FLAG_DIM/2), CONTENTS_FLAG_Y+1,                     CONTENTS_FLAG_X+(CONTENTS_FLAG_DIM/2), CONTENTS_FLAG_Y+CONTENTS_FLAG_DIM-1);
		}	}

		if (entityInstance.isMarked(EntityInstance.DRAW_CLIENT_MARK)) {
			// Draw flag that shows client exists
			Util.drawArrow(g, width - 5, (int) Util.ARROW_L, (int) (width - Util.ARROW_L*2 - 5), (int) Util.ARROW_L);
		}

		if (entityInstance.isMarked(EntityInstance.DRAW_SUPPLIER_MARK)) {
			Util.drawArrow(g, (int) (width - Util.ARROW_L*2 - 5), (int) (Util.ARROW_L), (int) (width - 5), (int) Util.ARROW_L);
		}

		if (entityInstance.getContainedBy() != null) {
			// Draw our own label

			if (entityInstance.isLabelDrawable()) {
				g.setColor(entityInstance.getCurrentLabelColor());
				if (entityInstance.isOpen()) {
					drawTopLeftLabel(g);
				} else {
					// centered, perhaps multi-line
					String label1 = entityInstance.getLabel();
					if (!entityInstance.isMarked(EntityInstance.CLIENT_SUPPLIER)) {
						g.setFont(entityInstance.closedFont[EntityInstance.START_FONT_NUM + entityInstance.getFontDelta()]);
					} else {
						g.setFont(EntityInstance.smallFont);
						EntityInstance pe = entityInstance.getEnterableParent();
						if (pe != null) {
							label1 = pe.getLabel() + " .\n" + label1;
					}	}
					Util.drawStringWrapped(g, label1, MARGIN, MARGIN, getWidth()-MARGIN*2, getHeight()-MARGIN*2, true, false);
				}
		}	} 
	
		// Put flags around the edges of the box

		if (entityInstance.getGroupFlag()) {
			g.setColor(entityInstance.getLabelColor());

			int pdim = (height < 20) ? 4 : 6;

			// Draw resize points

			if (entityInstance.getGroupKeyFlag()) {
				g.fillRect(1,			   1,				pdim, pdim);
				g.fillRect(width/2-pdim/2, 1,				pdim, pdim);
				g.fillRect(width-pdim,	   1,				pdim, pdim);
				g.fillRect(1,			   height/2-pdim/2, pdim, pdim);
				g.fillRect(width-pdim,	   height/2-pdim/2, pdim, pdim);
				g.fillRect(1,			   height-pdim,		pdim, pdim);
				g.fillRect(width/2-pdim/2, height-pdim,		pdim, pdim);
				g.fillRect(width-pdim,	   height-pdim,		pdim, pdim);
			}  else {
				// Stupidity here.. An outline is one extra byte wide and high because the two
				// edges of a one pixel box occupy 2 pixels.

				g.drawRect(1,			   1,				pdim, pdim);
				g.drawRect(width/2-pdim/2, 1,				pdim, pdim);
				g.drawRect(width-pdim-1,   1,				pdim, pdim);
				g.drawRect(1,			   height/2-pdim/2, pdim, pdim);
				g.drawRect(width-pdim-1,   height/2-pdim/2, pdim, pdim);
				g.drawRect(1,			   height-pdim-1,	pdim, pdim);
				g.drawRect(width/2-pdim/2, height-pdim-1,	pdim, pdim);
				g.drawRect(width-pdim-1,   height-pdim-1,	pdim, pdim);
			}
		}
	}

	// Icon interface (used to paint legends)

	public int getIconWidth()
	{
		return(getWidth());
	}

	public int getIconHeight()
	{
		return(getHeight());
	}

	public void paintIcon(Component c, Graphics g, int x, int y)
	{
		g.translate(x, y);
		paintComponent(g);
	}
}
