package lsedit;

import java.awt.Container;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Event;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.Rectangle;

import java.awt.event.*;
import java.net.*;
import java.io.*;
import java.util.*;
import javax.swing.*;

public class LandscapeEditorCore extends LandscapeViewerCore /* extends JPanel */ implements ItemListener 
{
	protected final static int GRID_MAX			 = 10;

	protected final static int NUM_TOOL_BUTTONS	 = 10;

	protected final static String TITLE = "Software Landscape Editor";

	protected final static int FA_SAVE		 = FA_LOAD+1;
	protected final static int FA_PRINT		 = FA_LOAD+2;
	protected final static int FA_PRINT_ALL	 = FA_LOAD+3;
	protected final static int FA_PRINT_BOOK = FA_LOAD+4;

	protected EditModeHandler m_editModeHandler;
	protected ViewModeHandler m_viewModeHandler;

	protected Vector diagrams = new Vector();
	protected Vector diagramNames = new Vector();

	public String getTitle() 
	{
		return TITLE;
	}

	public JMenuBar genMenu() 
	{ 
		JMenuBar mb = new JMenuBar();
		JMenu	 m, m1;
		MyMenuItem mi;
		MyCheckBoxMenuItem cmi;


		// Build File Menu

		m = new JMenu("File");
		Do.landscapeMenuItem(m, this);
		m.addSeparator();
		Do.printLandscapeMenuItem(m, this);
		m.addSeparator();
		Do.quitMenuItem(m, this);

		mb.add(m);

		m = new JMenu("Edit");

		Do.refreshMenuItem(m, this);

		m1 = new JMenu("Draw");
		Do.cutMenuItem(m1, this);
		Do.pasteMenuItem(m1, this);
		Do.drawMenuItem(m1, this);
		Do.deleteEdgeMenuItem(m1, this);

		Do.editEntityMenuItem(m1, this);
		Do.editRelationMenuItem(m1, this);
		m.add(m1);

		m1 = new JMenu("Find");
		Do.findMenuItem(m1, this);
		m.add(m1);

		mb.add(m);

		m = new JMenu("Arrange");
		m1 = new JMenu("Align");
		Do.alignMenuItem(m1, this);
		Do.gridMenuItem(m1, this);

		m.add(m1);

		m1 = new JMenu("Scale");
		Do.scaleMenuItem(m1, this);
		m.add(m1);

		m1 = new JMenu("Group");
		Do.fitMenuItem(m1, this);
		Do.groupMenuItem(m1, this);
		Do.groupAllMenuItem(m1, this);
		m.add(m1);

		m1 = new JMenu("Label");
		fontMenuItem(m1, this);
		Do.fitLabelMenuItem(m1, this);
		m.add(m1);
	
		mb.add(m);

		m = new JMenu("Layout");
		LandscapeLayouter layout;
		layout = new FlipLayoutHorizontally(this);
		new MyMenuItem(m, layout.getMenuLabel(), layout, 0, 0);
		layout = new FlipLayoutVertically(this);
		new MyMenuItem(m, layout.getMenuLabel(), layout, 0, 0);
		m.addSeparator();
		layout = new GridGroupHorizontally(this);
		new MyMenuItem(m, layout.getMenuLabel(), layout, 0, 0);
		layout = new GridGroupVertically(this);
		new MyMenuItem(m, layout.getMenuLabel(), layout, 0, 0);
		m.addSeparator();
		layout = new SugiyamaLayout(this);
		new MyMenuItem(m, layout.getMenuLabel(), layout, 0, 0);
		mb.add(m);

		m = new JMenu("Options");
		viewModeMenuItem(m, this);
		m.addSeparator();
		bestEdgeMenuItem(m, this);
		m.addSeparator();
		showMenuItem(m, this);
		mb.add(m);

		m  = new JMenu("Other");
		m1 = new JMenu("Tab box");
		Do.tabMenuItem(m1, this);
		m.add(m1);
		m1 = new JMenu("Visible relations");
		Do.visibleMenuItem(m1, this);
		m.add(m1);
		m1 = new JMenu("Active relations");
		Do.activeMenuItem(m1, this);
		m.add(m1);
		mb.add(m);
		m1 = new JMenu("Navigate");
		Do.navigateEdgeMenu(m1, this);
		Do.navigateEntityMenu(m1, this);
		tocMenuItem(m1, this);
		m.add(m1);
		m1 = new JMenu("Query");
		Do.queryMenuItem(m1, this);
		m.add(m1);

		m1 = new JMenu("Elide");
		Do.hideMenuItem(m1, this);
		m.add(m1);

		mb.add(m);

		m = new JMenu("Help");
		Do.helpMenuItem(m, this, "Editor");
		mb.add(m);
		return mb;
	}

	protected void genDropDowns() 
	{
		Dimension	dim;

		dim = new Dimension(100, ToolBarButton.HEIGHT);
		m_lsDropDown = new JComboBox();
		m_lsDropDown.setMaximumSize(dim);
		m_lsDropDown.setMinimumSize(dim);
		m_lsDropDown.setPreferredSize(dim);
		m_lsDropDown.setSize(dim);
		m_lsDropDown.addItemListener(this);
	}

	// Overrides LandscapeViewerCore.genGUI() completely

	protected void genGUI(int diagramPercentWidth, int diagramPercentHeight) 
	{
		MsgOut.dprintln("App: gen GUI");

		genDropDowns();
		genMainGUI(diagramPercentWidth, diagramPercentHeight);
	}

	protected void genModeHandlers() 
	{
		m_editModeHandler = new EditModeHandler();
		m_editModeHandler.init(this);

		m_viewModeHandler = new ViewModeHandler();
		m_viewModeHandler.init(this);
		m_modeHandler = m_editModeHandler;
	}

	protected int getNumToolButtons() 
	{
		return toolButton.length;
	}

	protected void registerLsName(String name) 
	{
		if (m_lsDropDown != null) {
			m_lsDropDown.addItem(name);

			if (m_lsDropDown.getItemCount() > 1) {
				m_lsDropDown.enable();
	}	}	}


	protected void setDiagram(Diagram value) 
	{
		String	name, diagramName;
		int		i;

		super.setDiagram(value);

		if (m_diagram != null) {
			m_diagram.setDrawBends(!(m_modeHandler instanceof ViewModeHandler));
			diagramName = m_diagram.getName();
			if (diagramName != null) {
				for (i = 0; i < m_lsDropDown.getItemCount(); ++i) {
					name = (String) m_lsDropDown.getItemAt(i);
					if (diagramName.equals(name)) {
						m_lsDropDown.setSelectedIndex(i);
						break;
		}	}	}	}

		if (m_diagram != null) {
			doFeedback("Set to: " + m_diagram.getName());
		}
		setLeftBox();
	}

	protected void setDiagram(int ind) 
	{
		Diagram newDg = (Diagram) diagrams.elementAt(ind);

		if (newDg != m_diagram) {
			setDiagram(newDg);
	}	}

	protected void addDiagram(Diagram newDg, String path) 
	{
		diagrams.addElement(newDg);
		diagramNames.addElement(path);

		registerLsName(newDg.getName());

		setDiagram(newDg);
		newDg.setToViewport();
	}

	public void attach(String lsPath) 
	{
		int		ind;
		Diagram	dgLink;

		ind = diagramNames.indexOf(lsPath);
		if (ind < 0) {
			dgLink = new Diagram(this, true);

			String rc = dgLink.loadDiagram(lsPath, null, null);
		  
			if (rc != null) {
				error("Attach failed: " + rc);
				return;
			}
			dgLink.prepostorder();
			addDiagram(dgLink, lsPath);
		}
		else {
			setDiagram(ind);
		}
	}

	public void followLink(String url, int target) 
	{
//		System.out.println("followLink: " + target);

		switch (target) {
		case LsLink.TARGET_APP:
			attach(url);
			break;
		case LsLink.TARGET_NEW:
			if (af != null) {
				// Running under a frame -- not an applet
				LandscapeEditorFrame af = LandscapeEditorFrame.create();
				af.setLsPath(url);
				af.launch();
				m_openFrames++;
				break;
			}
		default:
			super.followLink(url, target);
	}	}

	protected void loadLs(String file) 
	{
		if (file != null && file.length() > 0) {
			lsPath = file;

			doFeedback("Reading: " + lsPath);

			Diagram newDg = new Diagram(this, true);

			String rc = newDg.loadDiagram(lsPath, null, null);

			if (rc != null) {
				error("Failed to load (" + rc + ")");
			} else {
				newDg.prepostorder();
				addDiagram(newDg, lsPath);
			}
		}

		this.requestFocus();
		repaint();
	}

	protected void fitChildren() 
	{
		m_diagram.fitTo(m_diagram.getDrawRoot());
		doFeedback("Children fitted to: " + m_diagram.getDrawRoot().getLabel());
		m_diagram.rescaleDiagram();
		repaintDg();
	}

	protected void fitToLabel() 
	{
		Vector v = startGroupOp();

		if (v == null)
			return;
		
		int H_MARGIN = 8; 
		int V_MARGIN = 4;

		Enumeration en;

		for (en = v.elements(); en.hasMoreElements(); ) {
			EntityInstance e = (EntityInstance) en.nextElement();

			Dimension ld = e.getLabelDim(getGraphics(), EntityInstance.REG_FONT);

			Rectangle curLayout = e.getDiagramBounds();

			curLayout.width  = ld.width+H_MARGIN*2;
			curLayout.height = ld.height+V_MARGIN*2;

			e.setDiagramBounds(curLayout);
		}

		m_diagram.rescaleDiagram();
		redrawDg();
	}

	protected Vector startGroupOp() 
	{
		Vector grp = m_diagram.getGroup();

		if (grp == null) {
			error("Group not selected");
			return null;
		}
		return grp;
	}

	// Alignment

	protected void align(int alignment) 
	{
		Vector grp = startGroupOp();

		if (grp == null) {
			return;
		}

		// Align group based on alignment to key entity

		EntityInstance ke = m_diagram.getKeyEntity();
		Rectangle klyt = ke.getDiagramBounds();

		int cx = klyt.x + klyt.width/2;
		int cy = klyt.y + klyt.height/2;

		Enumeration en;

		for (en = grp.elements(); en.hasMoreElements(); ) {
			EntityInstance e = (EntityInstance) en.nextElement();

			if (e != ke) {
				Rectangle elyt = (Rectangle) e.getDiagramBounds().clone();

				switch(alignment)
				{
				case /* ALT */ Do.A_HORIZ_TOP:
					elyt.y = klyt.y;
					break;

				case /* ALT */ Do.A_HORIZ_CENTER:
					elyt.y = cy - elyt.height/2;
					break;

				case /* ALT */ Do.A_VERTICAL_LEFT:
					elyt.x = klyt.x;
					break;

				case /* ALT */ Do.A_VERTICAL_RIGHT:
					elyt.x += klyt.x + klyt.width - elyt.width;
					break; 

				case /* ALT */ Do.A_VERTICAL_CENTER:
					elyt.x = cx - elyt.width/2;
					break;

				default:
					MsgOut.println("Illegal allignment option");
					break;
				}
				e.setDiagramBounds(elyt);		
			}
		}

		m_diagram.rescaleDiagram();
		repaintDg();
	}

	protected void sameSize(int sizeDim) 
	{
		Vector grp = startGroupOp();

		if (grp == null)
			return;

		// Size group based on size of key entity

		EntityInstance ke = m_diagram.getKeyEntity();
		Rectangle klyt = ke.getDiagramBounds();

		Enumeration en = grp.elements();

		while (en.hasMoreElements()) {
			EntityInstance e = (EntityInstance) en.nextElement();

			if (e != ke) {
				Rectangle elyt = (Rectangle) e.getDiagramBounds().clone();

				switch(sizeDim) {
				case /* ALT */ Do.SZ_WIDTH:
					elyt.width = klyt.width;
					break; 

				case /* ALT */ Do.SZ_HEIGHT:
					elyt.height = klyt.height;
					break;

				case /* ALT */ Do.SZ_WIDTH_HEIGHT:
					elyt.width = klyt.width;
					elyt.height = klyt.height;
					break;
				}
				e.setDiagramBounds(elyt);
			}
		}

		m_diagram.rescaleDiagram();
		repaintDg();
	} 

	// mode is either Do.SPC_HORIZ or Do.SPC_VERTICAL:

	protected void equalSpacing(int mode) 
	{
		Vector grp = startGroupOp();

		if (grp == null) {
			return;
		}

		if (grp.size() < 3) {
			error("Minimum group of three required");
			return;
		}

		Rectangle bb = m_diagram.getGroupBoundingBox();

		if (bb == null) {
			error("Group not selected");
			return;
		}

		// Sort entities based on vertical or horizontal position
		switch (mode) {
		case Do.SPC_HORIZ:
			SortVector.byDiagramX(grp);
			break;
		case Do.SPC_VERTICAL: 
			SortVector.byDiagramY(grp);
			break;
		}

		// Calculate sep

		double tw, th;
		double sep = 0.0;

		Enumeration en = grp.elements();

		tw = th = 0;

		while(en.hasMoreElements()) {
			EntityInstance e = (EntityInstance) en.nextElement();

			Rectangle lyt = e.getDiagramBounds();

			tw += lyt.width;
			th += lyt.height;
		}

		if (mode == Do.SPC_HORIZ) {
			sep = (bb.width - tw)/(grp.size()-1);
		}
		else {
			sep = (bb.height - th)/(grp.size()-1);
		}

		// Apply new layouts to second through penultimate entitites

		EntityInstance ce = (EntityInstance) grp.elementAt(0);
		Rectangle clyt = ce.getDiagramBounds();

		for (int i = 1; i < grp.size()-1; i++) {
			EntityInstance e = (EntityInstance) grp.elementAt(i);

			Rectangle lyt = e.getDiagramBounds();

			if (mode == Do.SPC_HORIZ) {
				lyt.x = (int) (clyt.x + clyt.width + sep);
			}
			else {
				lyt.y = (int) (clyt.y + clyt.height + sep);
			}

			clyt = (Rectangle) lyt.clone();
			e.setDiagramBounds(lyt);
		}

		m_diagram.rescaleDiagram();
		repaintDg();
	}

	protected void createContainedGroup() 
	{
		Vector grp = startGroupOp();

		if (grp == null) {
			return;
		}

		EntityInstance pe = ((EntityInstance) grp.firstElement()).getContainedBy();

		Rectangle nlyt = m_diagram.getGroupBoundingBox();

		nlyt.y -= RelationInstance.CLIENT_SUPPLIER_EL_LEN+5;
		nlyt.x -= 5;
		nlyt.width += 10;
		nlyt.height += RelationInstance.CLIENT_SUPPLIER_EL_LEN+10;

		String cname;
		int n = 0;

		do {
			cname = "Container#" + n;
			n++;
		} while(m_diagram.entityExists(cname));

		EntityInstance ne = addEntity(cname, nlyt, pe);

		Enumeration en = grp.elements();

		while (en.hasMoreElements()) {
			EntityInstance e = (EntityInstance) en.nextElement();

			Rectangle lyt = e.getDiagramBounds();

			moveEntityContainment(ne, e);

			e.setDiagramBounds(lyt);		
		}

		m_diagram.rescaleDiagram();
		repaintDg();
	}
		
	protected void cutGroup(Object object) 
	{
		Enumeration en;
		EntityInstance e;

		if (m_clipboard != null && m_clipboard.size() != 0) {
			error("Clipboard not empty");
		} else {
			if (object != null && object == m_tocBox) {
				m_clipboard = m_tocBox.getGroup();
			} else {
				m_clipboard = m_diagram.getGroup();
			}

			if (m_clipboard == null) {
				error("Group not selected");
			} else {
				for (en = m_clipboard.elements(); en.hasMoreElements(); ) {
					e = (EntityInstance) en.nextElement();
					if (e.getContainedBy() == null) {
						doFeedback("Can't cut the root node in the diagram");
						m_clipboard = null;
						return;
				}	}
				// Don't want a subsequent paste to try and paste X into X
				m_diagram.clearKeyEntity();
				doFeedback("Group copied to clipboard");
			
				// Make sure we cut descendants before ancestors so that we can still find them
				SortVector.byPostorder(m_clipboard);

				for (en = m_clipboard.elements(); en.hasMoreElements(); ) {
					e = (EntityInstance) en.nextElement();
					cutEntity(e);
			}	}
			redrawDg();
	}	}

	protected void pasteGroup(Object object) 
	{
		if (m_clipboard == null || m_clipboard.size() == 0) {
			error("Clipboard empty");
		} else {
			EntityInstance pe;

			if (object != null && object == m_tocBox) {
				pe = m_tocBox.targetEntity();
			} else {
				pe = m_diagram.targetEntity(object);
			}
			if (pe != null) {
				Enumeration en;
				EntityInstance e;
				int		errs   = 0;
				int		cnt    = 0;

				for (en = m_clipboard.elements(); en.hasMoreElements(); ) {
					e = (EntityInstance) en.nextElement();
					if (m_diagram != e.getDiagram()) {
						++errs;
						error("Attempting to paste entity into a different diagram");
						continue;
					}

//					System.out.println("pasteGroup: " + e + " into " + pe);
					pasteEntity(pe, e);
					++cnt;
				}
				if (cnt != 0) {
					m_diagram.prepostorder();
					doFeedback("Pasted " + cnt + " entities into " + pe.getLabel());
					redrawDg();
		}	}	}
		m_clipboard = null;
	}

	protected void saveStatus(String stat) 
	{
		doFeedback("Save status: " + stat);
	}


	// Save the landscape in the URL

	protected String doSaveByURL(String lsSaveURL) 
	{
		URL lsURL;

		try {
			lsURL = new URL(lsSaveURL);
		}
		catch(MalformedURLException e) {
		   return "Malformed URL on write";
		}

		URLConnection urlCon;

		try {
			urlCon = lsURL.openConnection();

			urlCon.setDoInput(false);
			urlCon.setDoOutput(true);
			urlCon.setAllowUserInteraction(false);
		}
		catch (IOException e) {
			return "Couldn't open connection";
		}

		OutputStream os;

		try {
			os = urlCon.getOutputStream();
		}
		catch (IOException e) {
			return "Couldn't openStream";
		}

		try {
			m_diagram.saveDiagram(os, false);
		}
		catch (IOException e) {
			return "IOException on write";
		}

		return null;
	}

	public void saveByURL(String overridePath) 
	{
		MsgOut.dprintln("Save by URL");

		URL lsURL;

		String str = doSaveByURL(lsSavePath);

		if (str != null) {
			saveStatus(str);
			return;
		}

		if (lsSaveCmd == null) {
			saveStatus("Success");
			return;
		}

		try {
			lsURL = new URL(lsSaveCmd);
		}
		catch(MalformedURLException e) {
		   saveStatus("Malformed URL on write");
		   return;
		}

		InputStream is;
		try {
			is = lsURL.openStream();
		}
		catch (IOException e) {
			saveStatus("Couldn't openStream for store");
			return;
		}

		try {
			DataInputStream dis = new DataInputStream(is);

			str = dis.readLine();

			is.close();

			saveStatus(str);
		}
		catch(IOException e) {
			saveStatus("IOexception on store");
		}
	}

	protected String saveByFile(String newPath) 
	{
		try {
			File file = (File) m_diagram.getContext();

			if (newPath == null) {
				String name = file.getPath();

				file.renameTo(new File(name + ".old"));
			}
			else {
				// New filename
				file = new File(newPath);
				m_diagram.setContext(file);
			}

			FileOutputStream os = new FileOutputStream(file);

			m_diagram.saveDiagram(os, false);
			return null;
		}
		catch(IOException e) {
			return "IOException on file open";
		}
	}

	protected String saveByFileRaw(String newPath) 
	{
		try {
			File file = (File) m_diagram.getContext();

			if (newPath == null) {
				String name = file.getPath();

				if (name.endsWith(".raw"))
					file.renameTo(new File(name + ".old"));
				else
					file = new File(name + ".raw");
			}
			else {
				// New filename
				file = new File(newPath + ".raw");
				m_diagram.setContext(file);
			}

			FileOutputStream os = new FileOutputStream(file);

			m_diagram.saveDiagramRaw(os);
			return null;
		}
		catch(IOException e) {
			return "IOException on file open";
		}
	}

	protected String saveWithCmd(String saveSuffix, String cmdTemplate)
	{
		File file = (File) m_diagram.getContext();

		String path = file.getPath() + saveSuffix;

		File nfile = new File(path);

		// Save away the monolithic TA file

		try {

			FileOutputStream os = new FileOutputStream(nfile);

			m_diagram.saveDiagram(os, true);
		}
		catch(IOException e) {
			return "IOException on file save";
		}
	
		// Run command after save

		String cmd = Util.expand(cmdTemplate, file.getName(), this);

		// System.out.println("Save command: " + cmd);

		String rc;

		try {
			Runtime rt = Runtime.getRuntime();

			Process p = rt.exec(cmd);

			DataInputStream es = 
				new DataInputStream(p.getErrorStream());

			rc = es.readLine();

			if (rc.equals("Done"))
				rc = "Save succeeded";

			p.destroy();
		}
		catch (Exception e) {
			rc = "Exec failed for: " + cmd;
		}

		return rc;
	}

	protected void doSaveLs(String overridePath) 
	{
		doFeedback("Saving landscape...");

		if (m_diagram.getContext() instanceof URL) {
			saveByURL(overridePath);
		}
		else {
			String rc;

			if (overridePath == null && lsSaveCmd != null) {
				MsgOut.dprintln("Save by app");
				rc = saveWithCmd(lsSaveSuffix, lsSaveCmd);
			}
			else {
				MsgOut.dprintln("Save by file");
				rc = saveByFile(overridePath); 
			}

			if (rc == null)
				rc = "Success";

			saveStatus(rc);
		}
	}

	protected void saveLs() 
	{
		Object context = m_diagram.getContext();
		String txt = null;

		if (context == null) {

			txt = filePrompt("Save Landscape", m_diagram.getAbsolutePath(), FA_SAVE);
			if (txt.length() == 0) {
						return;
				}
		}

		doSaveLs(null);
	}

	protected void doSaveLsRaw(String overridePath) 
	{
		doFeedback("Saving landscape in raw format...");

		if (m_diagram.getContext() instanceof URL) {
			// saveByURL(overridePath);
		}
		else {
			MsgOut.dprintln("Save by file");

			String rc = saveByFileRaw(overridePath); 

			if (rc == null)
				rc = "Success";

			saveStatus(rc);
		}
	}

	protected void saveLsRaw() 
	{
		Object context = m_diagram.getContext();
		String txt = null;

		if (context == null) {
			txt = filePrompt("Save Landscape Raw", m_diagram.getAbsolutePath(), FA_SAVE);

			if (txt.length() == 0)
				return;
		}

		doSaveLsRaw(null);
	}

	protected void initialLoad() 
	{
		super.initialLoad();

		if (lsPath != null) {
			diagrams.addElement(m_diagram);
			diagramNames.addElement(lsPath);
			registerLsName(m_diagram.getName());
		} else {
			diagrams.addElement(m_diagram);
			diagramNames.addElement("");
			registerLsName("unnamed");
		}
	}

	public boolean isViewer() 
	{
		return false;
	}

	public void newEntity(EntityInstance container)
	{
		EntityInstance	e;

		e = m_diagram.newEntity(container);
		insertTOC(container, e);
	}

	public EntityInstance addEntity(String initId, Rectangle lyt, EntityInstance container)
	{
		EntityInstance	e;

		e = m_diagram.addEntity(initId, lyt, container);
		insertTOC(container, e);
		return(e);
	}

	// This should permanently delete the entity and remove all edges to from it

	public void DeleteEntity(Object object) 
	{
		EntityInstance e = m_diagram.targetEntity(object);

		e = m_diagram.DeleteEntity(e);
		if (e != null) {
			deleteTOC(e);
	}	}

	// This should just take the entity and edges to/from it out of the diagram

	public void cutEntity(EntityInstance e)
	{
		m_diagram.cutEntity(e);
		deleteTOC(e);
	}

	public void pasteEntity(EntityInstance parent, EntityInstance e)
	{
		m_diagram.pasteEntity(parent, e);
		insertTOC(parent, e);
	}

	public void moveEntityContainment(EntityInstance newparent, EntityInstance e)
	{
		EntityInstance oldparent;

		oldparent = e.getContainedBy();
		if (oldparent != newparent) {
			e.moveEntityContainment(newparent);
			deleteTOC(e);
			insertTOC(newparent, e);
	}	}

	public void deleteContainer(EntityInstance e)
	{
		e.deleteJustMe();
		deleteJustMeTOC(e);

		doFeedback("Entity " + e.getLabel() + " has been deleted.");
	}

	protected boolean testForClose() 
	{
		if (af == null) {
			return false;
		}
		if (!m_diagram.getChangedFlag())
			return true;

		System.out.print("\007");	// Bell

		int rc = JOptionPane.showConfirmDialog(m_frame, "Landscape has been changed.\nShould it be saved?", "Landscape Changed", JOptionPane.YES_NO_CANCEL_OPTION);

		switch(rc)
		{
		case JOptionPane.YES_OPTION:
			saveLs();
			// Drop through
		case JOptionPane.NO_OPTION:
			return true;
		}
		return false;
	}


	// --------------
	// Event handling 
	// --------------

	// Overrides LandscapeViewerCore.processKey() 

	public void processKey(int key, int modifiers, Object object) 
	{
//		System.out.println("LandscapeEditorCore:processKey(" + key + ", " + ", " + modifiers + ", " + object +")");

		String str; 

		if ((modifiers & Event.CTRL_MASK) != 0) {
			switch(key) {
			case Do.OPEN_LANDSCAPE:
			{
				Object context = m_diagram.getContext();
				String txt;

				if (context != null) {
					txt = filePrompt( "Load Landscape", m_diagram.getAbsolutePath(), FA_LOAD);
				} else {
					txt = filePrompt("Load Landscape", "", FA_LOAD);
				}
				loadLs(txt);
				return;
			}
			case Do.PRINT_LANDSCAPE:
			{
				Vista vista = new Vista(this); 
				return;
			}
			case Do.SAVE_AS:
			{
				Object context = m_diagram.getContext();
				String txt;

				if (context != null) {
					txt = filePrompt("Save Landscape As", m_diagram.getAbsolutePath(), FA_SAVE);
				} else {
					txt = filePrompt("Save Landscape As", "", FA_SAVE);
				}
				if (txt == null || txt.length() == 0) {
					return;
				}
				doSaveLs(txt);
				return;
			}
			case Do.SAVE:
				if (m_diagram.isReadOnly()) {
					error("Diagram is read-only");
					return;
				}
				if ((modifiers & Event.ALT_MASK) != 0) {
					saveLsRaw();
				} else {
					saveLs();
				}
				return;
			case Do.PASTE:
				pasteGroup(object);
				return;
			case Do.CUT:
				cutGroup(object);
				return;
			}
		} else if ((modifiers & Event.ALT_MASK) != 0) {
			switch (key) {
			case Do.A_HORIZ_TOP:
			case Do.A_HORIZ_CENTER:
			case Do.A_VERTICAL_LEFT:
			case Do.A_VERTICAL_RIGHT:
			case Do.A_VERTICAL_CENTER:
				align(key);
				return;
			case Do.A_FIT_LABEL:
				fitToLabel();
				return;
			case Do.A_FIT:
				fitChildren();
				return;
			case Do.A_GROUP:
				createContainedGroup();
				return;
			case Do.SZ_WIDTH:
			case Do.SZ_HEIGHT:
			case Do.SZ_WIDTH_HEIGHT:
				sameSize(key);
				return;
			case Do.SPC_HORIZ:
			case Do.SPC_VERTICAL:
				equalSpacing(key);
				return;
			case Do.ARROW_DIMENSIONS:
				ArrowDimensions.create(this);
				repaint();
				return;
			}

		} else if (modifiers == 0) { 
			switch (key) {
			case Do.VIEW_MODE:
			{
				if (m_modeHandler == m_editModeHandler) {
					doFeedback("Set to viewer mode");
					m_modeHandler = m_viewModeHandler;
					m_diagram.setDrawBends(false);
					setViewMode(true);
				} else {
					doFeedback("Set to editor mode");
					m_modeHandler = m_editModeHandler;
					m_diagram.setDrawBends(true);
					setViewMode(false);
				}
				m_diagram.setModeHandler(m_modeHandler);

				repaintDg();
				return;
			}
			case Do.GRID_INCREASE:
			{
				int grid = m_diagram.getGrid();

				if (grid == GRID_MAX) {
					error("Grid already set to maximum (" + grid + " pixels)");
				} else {
					int ng = grid+1;

					m_diagram.setGrid(ng);
					doFeedback("Grid set to " + ng + " pixels");
				}
				return;
			}
			case Do.GRID_DECREASE:
			{
				int grid = m_diagram.getGrid();

				if (grid == 1) {
					error("Grid already set to minimum (1 pixel)");
				} else {
					int ng = grid-1;

					m_diagram.setGrid(ng);
					doFeedback("Grid set to " + ng + " pixel(s)");
				}
				return;
			}}
		}
		super.processKey(key, modifiers, object);
	}

	// Overrides LandscapeViewer.isReadWrite()
	public boolean isreadWrite() {
		// Eventually this should be changed to return true
		// iff the landscape is definitely writeable
		// For now, assume true in editor

		return true;
	}

	public InternalBufferStream getInternalBufferStream() {
		return null;
	}

	// ItemListener interface

	public void itemStateChanged(ItemEvent ev)
	{
		int ind = m_lsDropDown.getSelectedIndex();
		setDiagram(ind);
		requestFocus();

	}
} 
