package lsedit;

import java.awt.Color;
import java.awt.Rectangle;
import java.awt.Graphics;
import java.awt.Event;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;



public class ResizeModeHandler extends LandscapeModeHandler
{
	// A small JComponent so that the whole handler doesn't need to be one
	// This object just draws the outline of the thing(s) being moved

	protected class DrawOutline extends JComponent
	{
		private ResizeModeHandler	m_handler;

		public DrawOutline(ResizeModeHandler handler)
		{
			super();
			m_handler = handler;
			setForeground(Color.BLACK);
		}

		public void paintComponent(Graphics g)
		{
			super.paintComponent(g);

			Rectangle	curLayout = m_handler.getCurLayout();

			if (curLayout != null) {
				g.drawRect(curLayout.x, curLayout.y, curLayout.width, curLayout.height);
		}	}
	}

	// Variables used by modes

	protected static final int RM_NONE	 = 0;
	protected static final int RM_LEFT	 = 1;
	protected static final int RM_RIGHT	 = 2;
	protected static final int RM_TOP	 = 4;
	protected static final int RM_BOTTOM = 8;

	protected static final int LR_MASK = RM_LEFT | RM_RIGHT;
	protected static final int TB_MASK = RM_TOP  | RM_BOTTOM;

	protected static final int RZONE_SIZE = 5;
	protected static final double MIN_DIM	 = 10.0;

	protected DrawOutline	m_drawOutline;
	protected Rectangle	m_curLayout;
	protected Rectangle	bb;				// bounding box of children 
	private Rectangle		m_plyt = null;


	protected int resizeMode = RM_NONE;

	protected int[] startMode = 

		{ RM_TOP    | RM_LEFT, 
		  RM_TOP, 
		  RM_TOP    | RM_RIGHT, 
		  RM_RIGHT,
		  RM_BOTTOM | RM_RIGHT,
		  RM_BOTTOM, 
		  RM_BOTTOM | RM_LEFT, 
		  RM_LEFT
		};

	protected void setCurLayout(int x, int y, int width, int height)
	{
		if (m_curLayout == null) {
			m_curLayout    = new Rectangle(x, y, width, height);
			m_drawOutline.setBounds(m_dg.getBounds());
			m_drawOutline.setVisible(true);
			m_dg.add(m_drawOutline /* , JLayeredPane.PALETTE_LAYER */, 0);
		} else {
			m_curLayout.x      = x;
			m_curLayout.y      = y;
			m_curLayout.width  = width;
			m_curLayout.height = height;
			m_drawOutline.repaint();
		}
	}

	protected void clearCurLayout()
	{
		if (m_curLayout != null) {
			m_dg.remove(m_drawOutline);
			m_curLayout = null;
	}	}

	protected void addMode(int newMode) 
	{
		int omode = 0;

		switch(newMode) {
		case RM_TOP:
			omode = RM_BOTTOM;
			break;
		case RM_BOTTOM:
			omode = RM_TOP;
			break;
		case RM_LEFT:
			omode = RM_RIGHT;
			break;
		case RM_RIGHT:
			omode = RM_LEFT;
			break;
		} 

		if ((resizeMode & omode) == 0) {
			resizeMode |= newMode;
		}
	}

	private String resizeDescription() 
	{
		String ret;

		if (resizeMode == RM_NONE) {
			ret = "NONE";
		} else {
			ret = "";
			if ((resizeMode & RM_LEFT) != 0) {
				ret += "LEFT ";
			}
			if ((resizeMode & RM_RIGHT) != 0) {
				ret += "RIGHT ";
			}
			if ((resizeMode & RM_TOP) != 0) {
				ret += "TOP ";
			}
			if ((resizeMode & RM_BOTTOM) != 0) {
				ret += "BOTTOM";
			}
		}
		return(ret);
	}

	protected void doResizeAdjust(int mode, int xpos, int ypos) 
	{
		int gap = 20;
		int x1, y1, width1, height1;

		x1		= (int) m_curLayout.x;
		y1		= (int) m_curLayout.y;
		width1	= (int) m_curLayout.width;
		height1 = (int) m_curLayout.height;

		if (xpos-x1 < 4) {
			addMode(RM_LEFT);
		}
		if (x1 + width1 - xpos < 4) {
			addMode(RM_RIGHT);
		}
		if (ypos-y1 < 4) {
			addMode(RM_TOP);
		}
		if (y1 + height1 - ypos < 4) {
			addMode(RM_BOTTOM);
		}

		if ((mode & RM_TOP) != 0) {

			if (ypos >= m_plyt.y + gap) {
				height1 += y1 - ypos;
				y1		 = ypos;
			}
		} else if ((mode & RM_BOTTOM) != 0) {
			if (ypos <= m_plyt.y + m_plyt.height - gap) {
				height1 = ypos - y1;
		}	 }

		if ((mode & RM_LEFT) != 0) {
			if (xpos >= m_plyt.x + gap) {
				width1 += x1 - xpos;
				x1		= xpos;
			}
		} else if ((mode & RM_RIGHT) != 0) {
			if (xpos <= m_plyt.x + m_plyt.width - gap) {
				width1 = xpos - x1;
		}	}	

//		System.out.println("1: x=" + xpos + " y=" + ypos + " m_curLayout=" + m_curLayout + " plyt=" + m_plyt + " new={" + x1 +"," + y1 + "," + width1 + "," + height1);

		if (width1 < MIN_DIM) {
			x1      = m_curLayout.x;
			width1  = m_curLayout.width;
		}
		if (height1 < MIN_DIM) {
			y1      = m_curLayout.y;
			height1 = m_curLayout.height;
		}

		setCurLayout(x1,y1, width1, height1);
	}


	//
	// Public methods
	//

	public ResizeModeHandler() 
	{
		m_drawOutline = new DrawOutline(this);
	}


	//
	//	 ++-----------++
	//	 +			   +   
	//	 |			   |  Corner or edge grab
	//	 |			   | 
	//	 +			   +
	//	 ++-----------++
	//

	public void select(Diagram dg) 
	{
		if (m_dg != dg) {
			clearCurLayout();
			super.select(dg);
	}	}

	public Rectangle getCurLayout()
	{
		return(m_curLayout);
	}

	public boolean entityPressed(MouseEvent ev, EntityInstance e, int x, int y) 
	{
		if (m_dg == null) {
			return false;
		}

		if (e == null || !e.getGroupKeyFlag()) {
			return false;
		} 

		int rpoint = e.overResizeTab(x, y);
		if (rpoint < 0) {
			return false;
		}

		resizeMode = startMode[rpoint];
		setCurLayout(e.getDiagramX(), e.getDiagramY(), e.getWidth(), e.getHeight());

		if (m_plyt == null) {
			m_plyt = new Rectangle();
		} 

		EntityInstance parent;

		parent = e.getContainedBy();
		if (parent != null && parent.getContainedBy() != null) {
			Rectangle plyt;

			plyt          = parent.getDiagramBounds();
			m_plyt.x	  = plyt.x;
			m_plyt.y	  = plyt.y;
			m_plyt.width  = plyt.width;
			m_plyt.height = plyt.height;
		} else {
			m_plyt	      = m_dg.getBounds();
			m_plyt.x      = 0;
			m_plyt.y      = 0;
		}

		m_ls.doFeedback("Size: " + m_curLayout.width + " x " + m_curLayout.height);
		return true; 
	}

	public boolean entityDragged(MouseEvent ev, EntityInstance e, int x, int y) 
	{
		double xpos = x;
		double ypos = y;

		if (resizeMode == RM_NONE) {

			// See if we have reached an edge

			boolean l = (xpos <= m_curLayout.x);
			boolean r = (xpos >= m_curLayout.x + m_curLayout.width);
			boolean t = (ypos <= m_curLayout.y);
			boolean b = (ypos >= m_curLayout.y + m_curLayout.height);


			if (e == m_dg.getDrawRoot()) {
				l = false;		// Can't change left
				t = false;		// Can't change top
			}

			if (l) {
				if (t || ypos-m_curLayout.y < 4) {
					resizeMode = RM_LEFT | RM_TOP;
				} else if (b || (m_curLayout.y + m_curLayout.height - ypos < 4)) {
					resizeMode = RM_LEFT | RM_BOTTOM;
				} else {
					resizeMode = RM_LEFT;
				}
			} else if (r) {
				if (t || ypos-m_curLayout.y < 4) {
					resizeMode = RM_RIGHT | RM_TOP;
				} else if (b || (m_curLayout.y + m_curLayout.height - ypos < 4)) {
					resizeMode = RM_RIGHT | RM_BOTTOM;
				} else {
					resizeMode = RM_RIGHT;
				}
			} else if (t) {
				if (xpos-m_curLayout.x < 4) {
					resizeMode = RM_TOP | RM_LEFT;
				} else if (m_curLayout.x + m_curLayout.width - xpos < 4) {
					resizeMode = RM_TOP | RM_RIGHT;
				} else {
					resizeMode = RM_TOP;
				}
			} else if (b) {
				if (xpos-m_curLayout.x < 4) {
					resizeMode = RM_BOTTOM | RM_LEFT;
				} else if (m_curLayout.x + m_curLayout.width - xpos < 4) {
					resizeMode = RM_BOTTOM | RM_RIGHT;
				} else {
					resizeMode = RM_BOTTOM;
			}	}
		}

		if (resizeMode != RM_NONE) {
			doResizeAdjust(resizeMode, x, y);
			m_ls.doFeedback("Size: " + m_curLayout.width + " x " + m_curLayout.height);
		}
		return true;
	}

	public void entityReleased(MouseEvent ev, EntityInstance e, int x, int y) 
	{
		if (e != null) {
			e.setDiagramBounds(m_curLayout);
			m_dg.rescaleDiagram();
			e.validateEdges();
			e.fitTo(false);
		}
		m_ls.clearFeedback();
		clearCurLayout();
	}


}

