package lsedit;

import java.util.Enumeration;
import java.util.Vector;
import java.awt.Cursor;
import java.awt.Toolkit;
import java.awt.FontMetrics;

public class SimplexLayout extends LandscapeLayouter  implements ToolBarEventHandler {

	protected final static int BORDER = 30;	// 1/30 of graph all round

	public SimplexLayout(LandscapeEditorCore ls) 
	{
		super(ls);
	}

/****************************************************************************/

	public String getName()
	{
		return "Simplex";
	}

	public String getMenuLabel() 
	{
		return "Layout using Network Simplex algorithm";
	} 

/***************************************************************************/

	/* While seemingly somewhat inefficient this may be more efficient than lifting
	 * edges twice. It has the advantage of being quick when edges are found
	 */

	protected static int edgesBetween(EntityInstance from, EntityInstance to)
	{
		Enumeration			en;
		RelationInstance	relation;
		RelationClass		rc;
		EntityInstance		e;
		int					ret;

		ret = 0;
		for (en = from.srcLiftedRelationElements(); en.hasMoreElements(); ) {
			relation = (RelationInstance) en.nextElement();
			rc       = relation.getRelationClass();
			// Consider only visible edges when drawing layout
			if (rc.isClassVisible()) {
				e        = relation.getDrawDst();
				if (to.hasDescendantOrSelf(e)) {
					++ret;
			}	}
		}

		if (from.isOpen()) {
			for (en = from.getChildren(); en.hasMoreElements(); ) {
				e = (EntityInstance) en.nextElement();
				ret += edgesBetween(e, to);
		}	}
//		System.out.println("SimplexLayout: Testing for edge from " + from + (from.isOpen() ? "[open]" : "[notopen]") + " to at/under " + to + "=" + ret);
		return(ret);
	}

	protected static int edgesFromClient(EntityInstance to)
	{
		Enumeration			en;
		RelationInstance	relation;
		EntityInstance		e;
		int					ret;

		ret = 0;
		for (en = to.dstLiftedRelationElements(); en.hasMoreElements(); ) {
			relation = (RelationInstance) en.nextElement();
			e        = relation.getDrawSrc();
			if (e.isMarked(EntityInstance.CLIENT_MARK)) {
				++ret;
		}	}

		if (to.isOpen()) {
			for (en = to.getChildren(); en.hasMoreElements(); ) {
				e = (EntityInstance) en.nextElement();
				ret += edgesFromClient(e);
		}	}
		return(ret);
	}

	protected static int edgesToSupplier(EntityInstance from)
	{
		Enumeration			en;
		RelationInstance	relation;
		EntityInstance		e;
		int					ret;

		ret = 0;
		for (en = from.srcLiftedRelationElements(); en.hasMoreElements(); ) {
			relation = (RelationInstance) en.nextElement();
			e        = relation.getDrawDst();
			if (e.isMarked(EntityInstance.SUPPLIER_MARK)) {
				++ret;
		}	}

		if (from.isOpen()) {
			for (en = from.getChildren(); en.hasMoreElements(); ) {
				e = (EntityInstance) en.nextElement();
				ret += edgesToSupplier(e);
		}	}
		return(ret);
	}

	public boolean isConfigurable()
	{
		return true;
	}

	public void configure(LandscapeEditorCore ls)
	{
		HiGraphLayout.configure(ls);
	}

	// The doLayout method executes the Network Simplex algorithm
	// Assumption: All boxes selected are in the same container.

	public void doLayout1(Vector selectedBoxes, EntityInstance container, boolean update) 
	{
		HiGraph			root, from, to, client, supplier;
		HiArc			arc1, arc2;
		Vector			children;
		Enumeration		en, en1;
		EntityInstance	e, other;

		int		width, height;
		int		weight;
		int		min_x,  max_x,  min_y,  max_y;
		int		min_x1, max_x1, min_y1, max_y1;
		boolean	seen;
		double	xrel, yrel, widthrel, heightrel;
		double	scaleX, scaleY, scaleWidth, scaleHeight;
		int		graph_width, graph_height;
		double	avg_relheight, d;

//		System.out.println("Simplex started on " + container + "(" + selectedBoxes.size() + ")" + ": " + selectedBoxes);

		switch (selectedBoxes.size()) {
		case 0:
			return;
		case 1:
			e = (EntityInstance) selectedBoxes.firstElement();
			widthrel  = e.widthRelLocal();
			heightrel = e.heightRelLocal();
			xrel      = (1 - widthrel )/2;
			yrel     = (1 - heightrel)/2;

			if (update) {
				e.updateRelLocal(xrel, yrel, widthrel, heightrel);
			} else {
				e.setXRelLocal(xrel);
				e.setYRelLocal(yrel);
			}
			return;
		}
		
		HiGraphLayout	hiGraphLayout = new HiGraphLayout();

		width  = container.getWidth();
		height = container.getHeight();
	
		// create graph to store info on selected boxes and their relationships
		root     = new HiGraph(null, "root", 0, height);

		// Simplex algorithm assumes all boxes same height

		avg_relheight = 0;
		for (en = selectedBoxes.elements(); en.hasMoreElements(); ) {
			e              = (EntityInstance) en.nextElement();
			avg_relheight += e.heightRelLocal();

		}
		avg_relheight /= selectedBoxes.size();

		for (en = selectedBoxes.elements(); en.hasMoreElements(); ) {
			e              = (EntityInstance) en.nextElement();
			// N.B. don't use getWidth() since may not yet be known
//			System.out.println(e + " width=" + e.getWidth() + " computedWidth=" + (int) (e.widthRelLocal() * (double) width));
			root.newChild(e, e.getEntityLabel(),(int) (e.widthRelLocal() * (double) width), (int) (avg_relheight * height));
		} 

		seen     = false;
		children = root.children();

		// get (directed) relationships between the boxes

		for (en = children.elements(); en.hasMoreElements(); ) {
			arc1        = (HiArc) en.nextElement();
			from        = arc1.to();
			e           = from.getReferencedObject();
			for (en1 = children.elements(); en1.hasMoreElements(); ) {
				arc2    = (HiArc) en1.nextElement();
				if (arc2 == arc1) {
					continue;
				}
				to      = arc2.to();
				other   = to.getReferencedObject();
				weight  = edgesBetween(e, other);
//				System.out.println("SimplexLayout: " + from + "->" + to + "[" + weight + "]");
				if (weight > 0) {
//					System.out.println(e + "=>" + other);
					seen = true;
					arc1 = to.newInputArc(from);
					arc1.weight(weight);
		}	}	}

//		System.out.println("Computed inner edges");

		client   = null;
		supplier = null;
		if (seen  && container != m_ls.getDiagram().getRootInstance()) {

			if (m_ls.isShowClients()) {
				for (en = children.elements(); en.hasMoreElements(); ) {
					arc1        = (HiArc) en.nextElement();
					to          = arc1.to();
					e           = to.getReferencedObject();
					if (e == null) {
						continue;
					}
					weight = edgesFromClient(e);
					if (weight > 0) {
						if (client == null) {
							client = root.newChild(null, "client", 0, 0);
						}
//						System.out.println("SimplexLayout: client=>" + to + "[" + weight + "]");
						if (m_ls.isTopClients()) {
							arc1 = to.newInputArc(client);
						} else {
							arc1 = client.newInputArc(to);
						}
						arc1.weight(weight);
			}	}	}

//			System.out.println("Computed client edges");

			if (m_ls.isShowSuppliers()) {
				for (en = children.elements(); en.hasMoreElements(); ) {
					arc1        = (HiArc) en.nextElement();
					from        = arc1.to();
					e           = from.getReferencedObject();
					if (e == null) {
						continue;
					}
					weight = edgesToSupplier(e);
					if (weight > 0) {
						if (supplier == null) {
							supplier = root.newChild(null, "supplier", 0, 0);
						}
//						System.out.println("SimplexLayout: " + from + "=>supplier[" + weight + "]");
						if (m_ls.isTopClients()) {
							arc1 = supplier.newInputArc(from);
						} else {
							arc1 = from.newInputArc(supplier);
						}
						arc1.weight(weight);
			}	}	}
//			System.out.println("Computed supplier edges");
		}

//		System.out.println("Beginning layout");

//		root.dump();
		hiGraphLayout.layout(root);
//		System.out.println("Done layout");

		min_x = max_x = min_y = max_y = 0;
		seen  = false;

		children = root.children();

		for (en = children.elements(); en.hasMoreElements(); ) {
			arc1       = (HiArc) en.nextElement();
			from       = arc1.to();
			e          = from.getReferencedObject();
			if (e == null) {
				continue;
			}
//			System.out.println(e + "{" + from.x() + "," + from.y() + "/" + e.getWidth() + "," + e.getHeight() + "}");

			min_x1  = from.x() - from.width()/2;
			max_x1  = min_x1   + from.width();
			min_y1  = from.y() - from.height()/2;
			max_y1  = min_y1   + from.height();

			if (!seen) {
				min_x = min_x1;
				max_x = max_x1;
				min_y = min_y1;
				max_y = max_y1;
				seen  = true;
			} else {
				if (min_x1 < min_x) {
					min_x = min_x1;
				}
				if (max_x1 > max_x) {
					max_x = max_x1;
				}
				if (min_y1 < min_y) {
					min_y = min_y1;
				}
				if (max_y1 > max_y) {
					max_y = max_y1;
		}	}	} 

		graph_width  = max_x - min_x;
		min_x       -= (graph_width/BORDER);
		graph_width += (2*graph_width)/BORDER;
		graph_height = max_y - min_y;
		min_y       -= (graph_height/BORDER);
		graph_height+= (2*graph_height)/BORDER;

//		System.out.println("Graph " + graph_width + "x" + graph_height + " Actual " + width + "x" + height);
		

		if (graph_width <= 0 || graph_height <= 0) {
			return;
		}

		if (graph_width > width) {
			scaleX     = ((double) width) / ((double) graph_width);
			scaleWidth = scaleX;
		} else {
			scaleX     = ((double) width) / ((double) graph_width);
			scaleWidth = 1.0;
		}
		if (graph_height > height) {
			scaleY      = ((double) height) / ((double) graph_height);
			scaleHeight = scaleY;
		} else {
			scaleY      = ((double) height) / ((double) graph_height);
			scaleHeight = 1.0;
		}

		for (en = root.children().elements(); en.hasMoreElements(); ) {
			arc1       = (HiArc) en.nextElement();
			from       = arc1.to();
			e          = from.getReferencedObject();
			if (e == null) {
				continue;
			}
			widthrel = e.widthRelLocal() * scaleWidth;
			xrel     = (( ((double) (from.x() - min_x)) * scaleX ) / ((double) width)) - (widthrel/2.0);
			if (xrel + widthrel > 1.0) {
				// (e.widthRelLocal() * width / graph_width) + (fromx - min_x) * width / graph_width)/ width  - e.widthRelLocal()*width/(2*graph_width)
				// (e.widthRelLocal()/2) * width/graph_width + (fromx - min_x)/graph_width

				System.out.println("Simplex layout error: " + e + " xrel=" + xrel + " widthrel=" + widthrel + " xrel+widthrel=" + (xrel+widthrel) + " > 1");
				xrel = 1.0 - widthrel;
			} 
			heightrel = avg_relheight * scaleHeight;
			yrel      = (( ((double) (from.y() - min_y)) * scaleY ) / ((double) height)) - (heightrel/2.0);

//			System.out.println(e + " xrel=" + xrel + " hx=" + from.x() + " hwidth=" + from.width());
			if (yrel + heightrel > 1.0) {
				System.out.println("Simplex layout error: yrel + heightrel > 1");
				yrel = 1 - heightrel;
			}
			if (update) {
				e.updateRelLocal(xrel, yrel, widthrel, heightrel);
			} else {
				e.setRelLocal(xrel, yrel, widthrel, heightrel);
		}	}

//		System.out.println("Simplex done");

	} // doLayout


  // The doLayout method executes the Coffman-Graham Layer Assignment
  // algorithm and the Sugiyama algorithm on the boxes selected.
  // Assumption: All boxes selected are in the same container.

	public String doLayout(Diagram dg) 
	{
		EntityInstance parent;

		// get user's selection of boxes to be laid out

		m_ls.setLayouter(this);
		m_ls.setCursor(Cursor.WAIT_CURSOR);

		Vector selectedBoxes = dg.getGroup();
		if (selectedBoxes == null) {
			  beep();
			  return "No group selected";
		}

		parent = parentOfSet(selectedBoxes);
		if (parent == null) {
			return	"Simplex algorithm requires that all things laid out share same parent";
		}
		dg.beginUndoRedo("Simplex layout");
		doLayout1(selectedBoxes, parent, true /* Updating */);
		dg.endUndoRedo();

		m_ls.setCursor(Cursor.DEFAULT_CURSOR);

		return "Graph redrawn using Network Simplex algorithm";
	} // doLayout

	public void processKeyEvent(int key, int modifiers, Object object) 
	{
		Diagram	dg;
		String	rmsg;

		dg = m_ls.getDiagram();
		if (dg != null) {
			rmsg = doLayout(dg);
			m_ls.doFeedback(rmsg);
			dg.rescaleDiagram();
			m_ls.repaintDg();
	}	}

} 





