package lsedit;
import java.io.IOException;
import java.io.InputStream;

/**
 * A class to turn an TA input stream into a stream of tokens.
 * @version 1.0, 22/07/98
 * @author	Gary Farmaner
 */

public

class TA_StreamTokenizer extends Object {

	private static final byte CT_DELIMIT	= 2;

	private InputStream m_is;
	private char		m_buf[];
	private int			m_peekc = ' ';	// Will appear to be first character but ignored so ok
	private boolean		m_pushedBack;

	/** The line number of the last token read */

	private int			m_lineno = 1;
	private byte		m_ctype[] = new byte[256];

	/**
	 * The type of the last token returned.	 It's value will either
	 * be one of the following TT_* constants, or a single
	 * character.  For example, if '+' is encountered and is
	 * not a valid word character, ttype will be '+'
	 */

	public int ttype;

	public static final int TT_EOF = -1;				// The End-of-file token.
	public static final int TT_EOL = '\n';				// The End-of-line token.				
	public static final int TT_NUMBER = -2;				// The number token.  This value is in nval.
	public static final int TT_WORD = -3;				// The word token.	This value is in sval.

	/**
	 * The Stream value.
	 */

	public String sval;

	/**
	 * The number value.
	 */

	public double nval;

	private static final int IBUF_SIZE = 8 * 1024;

	protected byte[] m_ibuf = new byte[IBUF_SIZE];
	protected int	m_cnt = 0;
	protected int	m_pos = 0;


	public TA_StreamTokenizer (InputStream is) 
	{
		m_is          = is;
		m_buf         = new char[20*1024];

		m_ctype[' ']  = CT_DELIMIT;
		m_ctype['\t'] = CT_DELIMIT;
		m_ctype['\r'] = CT_DELIMIT;
		m_ctype['\n'] = CT_DELIMIT;
		m_ctype['=']  = CT_DELIMIT;
		m_ctype[':']  = CT_DELIMIT;
		m_ctype['{']  = CT_DELIMIT;
		m_ctype['}']  = CT_DELIMIT;
		m_ctype['(']  = CT_DELIMIT;
		m_ctype[')']  = CT_DELIMIT;
	}

	// What a pain -- can't add a next pointer to subclass of string since String is final
	// We want to reuse references to same strings

	class LinkedString {
		public String			m_string;
		public	LinkedString	m_next;
		
		public LinkedString(String string, LinkedString next)
		{
			m_string = string;
			m_next   = next;
		}
	}
	
	protected final int cread() throws IOException 
	{
		int	c;

		c = m_peekc;
		if (m_pos == m_cnt) {
			if (c == TT_EOF || (m_cnt = m_is.read(m_ibuf, 0, IBUF_SIZE)) < 1) {
				m_pos   = m_cnt;
				m_peekc = TT_EOF;
				return c;
			}
			m_pos = 0;
		}
		m_peekc = (int) m_ibuf[m_pos++];
		switch (c) {
		case '\r':
			if (m_peekc == '\n') {
				break;
			}
		case '\n':
			++m_lineno;
		}
		return c;
	}

	/**
	 * Parses a token from the input stream.  The return value is
	 * the same as the value of ttype.	Typical clients of this
	 * class first set up the syntax tables and then sit in a loop
	 * calling nextToken to parse successive tokens until TT_EOF
	 * is returned.
	 */

	public final int nextToken() throws IOException {

		if (m_pushedBack) {
			m_pushedBack = false;
			return ttype;
		}

		int c;
		int pos = 0;

		for (sval = null; ; ) {
			// This loop is repeated while initial whitespace
			c = cread();

			switch(c) {
			case TT_EOF:
				sval = "EOF";
				return ttype = TT_EOF;
			case '\r':
			case '\n':
			case '\t':
			case ' ':
				continue;
			case '"':
				for (;;) {
					c = cread();
					switch (c) {
					case TT_EOF:
						System.out.println("Unterminated \" in input");
					case '"':
						break;
					case '\\':
						m_buf[pos++] = (char) c;
						c = cread();
						if (c == TT_EOF) {
							break;
						}
					default:	
						m_buf[pos++] = (char) c;
						continue;
					}
					break;
				} 
				sval = StringCache.get(m_buf, pos);
				return ttype = TT_WORD;
			case '\'':
				for (;;) {
					c = cread();
					switch (c) {
					case TT_EOF:
						System.out.println("Unterminated ' in input");
					case '\'':
						break;
					case '\\':
						m_buf[pos++] = (char) c;
						c = cread();
						if (c == TT_EOF) {
							break;
						}
					default:	
						m_buf[pos++] = (char) c;
						continue;
					}
					break;
				} 
				sval = StringCache.get(m_buf, pos);
				return ttype = TT_WORD;
			case '/':
				switch (m_peekc) {
				case '/':	
					// Toss the remainder of the line
					for (;;) {
						c = cread();
						switch (c) {
						case '\r':
							if (m_peekc == '\n') {
								continue;
							}
						case TT_EOF:
						case '\n':
							break;
						default:
							continue;
						}
						break;
					}
					continue;
				case '*':
					// Toss the remainder of the comment
					for (;;) {
						c = cread();
						switch (c) {
						case '*':
							if (m_peekc != '/') {
								continue;
							}
							c = cread();
						case TT_EOF:
							break;
						default:
							continue;
						}
						break;
					}
					continue;
				}
			default:
				if (m_ctype[c] == CT_DELIMIT) {
					sval = String.valueOf((char) c);
					return ttype = c;
				}

				// Read characters until delimitter

				for (;;) {
					m_buf[pos++] = (char) c;
					if (m_peekc == TT_EOF) {
						break;
					}
					if (c == '\\') {
						m_buf[pos++] = (char) cread();
					} 
					if (m_ctype[m_peekc] == CT_DELIMIT) {
						break;
					}
					c = cread();
				}	
				sval = StringCache.get(m_buf, pos);
				return ttype = TT_WORD;
			}
		}
	}

	/** Does an unget of the last token */

	public final void pushBack() {
		m_pushedBack = true;
	} 

	/** Return the current line number. */
	public int lineno() {
		return m_lineno;
	}
}

