package lsedit;
import java.util.Vector;
import java.util.Enumeration;

/**
 * Developed by: Ian Davis in Summer 1997 for Grant Weddell
 * Provides a static method to associate an ordering of nodes at the same rank.. In chess parlance
 * on the same row.. thus the name.  This code is conscious of the fact that any attempt to reorder
 * nodes on a rank must be careful not to violate the orderings enforced on a global higraph order
 * imposed by HiChildren (which is used to determine whether nodes not in our subtree are to our
 * left or right.
 *
 * N.B.  The problem of optimising the ordering of nodes on two rows so that edge crossings are
 * minimized is NP-complete.  Therefore this is a rattle and shake algorithm, which attempts to
 * reduce edge crossings by placing nodes at the median position within the nodes which then 
 * address/ are addressed by.
 */

 
/* N.B. We assume below that all edges are directed strictly downwards according to rank one level
 * at a time.  
 */

class HiRow {
	
	static final boolean debug  = false;

	/* Create the structure used to hold the total collection of HiGraph nodes on the board
	 * We assume that the minimal rank or all children is node.m_rank that no child node has
	 * a rank > node.m_sink_rank
	 */

	static HiGraph[][] buildBoard(HiGraph node)  {

		int			count[]  = new int[node.m_sink_rank-node.m_rank + 1];
		int			ranks[]  = new int[node.m_sink_rank-node.m_rank + 1];
		Vector		children = node.m_children;
		int			size     = children.size();
		int			i, j;
		HiArc		arc;
		HiGraph		child;
		
		/* Count the children at each possible rank */

		for (i = 0; i < size; ++i) {
			arc   = (HiArc) children.elementAt(i);
			child = arc.to();
			// Ranks are with respect to parent rank
			++count[child.m_rank-node.m_rank];
		}

		/* Compute the real board ranks ignoring empty ranks */
	
		for (i = j = 0; i < count.length; ++i) {
			ranks[i] = j;
			if (count[i] != 0) ++j;
		}

		/* Build the board of the correct shape */

		HiGraph	board[][] = new HiGraph[j][];

		for (i = j = 0; i < count.length; ++i) {
			if (count[i] != 0) {
				board[j++] = new HiGraph[count[i]];
				count[i] = 0;
		}	}

		/* Fill the board */

		int	rank, position;

		for (i = 0; i < size; ++i) {
			arc                   = (HiArc) children.elementAt(i);
			child                 = arc.to();
			rank                  = ranks[child.m_rank-node.m_rank]; // rank on board to use for this rank.
			position              = count[rank];
			child.m_position      = position;
			board[rank][position] = child;
			++count[rank];
		}
		return(board);
	}

	/* Copy the board1 to board2 - create board2 if null */

	static void dumpBoard(HiGraph node, HiGraph board[][]) {
		System.out.println("Board for " + node);

		int		i, j;
		HiGraph	row[];
	
		for (i = 0; i < board.length; ++i) {
			row = board[i];
			System.out.print(i + ":");
			for (j = 0; j < row.length; ++j) {
				System.out.print("{"+j);
				if (row[j].m_position != j) {
					System.out.print("??");
				}
				System.out.print(":" + row[j].m_outside + "}" + row[j]);

			}
			System.out.println("");
	}	}

	private static int rowWeight(HiGraph higraph)
	{
		Vector		arcs;
		int			weight, size, i, position, diff;
		HiArc		arc;
		HiGraph		other;

		arcs     = higraph.m_out;
		size     = arcs.size();
		weight   = 0;		
		position = higraph.m_position;

		for (i = 0; i < size; ++i) {
			arc     = (HiArc) arcs.elementAt(i);
			other   = arc.to();
			diff    = other.m_position - position;  
			weight += diff * diff;
		}

		arcs     = higraph.m_in;
		size     = arcs.size();

		for (i = 0; i < size; ++i) {
			arc     = (HiArc) arcs.elementAt(i);
			other   = arc.from();
			diff    = other.m_position - position;
			weight += diff * diff;
		}
		return(weight);
	}

	private static void initBoard(HiGraph board[][])
	{
		HiGraph row[];
		HiGraph higraph;
		int		i, j;

		for (i = board.length; i > 0; ) {
			row = board[--i];
			for (j = row.length; j > 0; ) {
				higraph = row[--j];
				higraph.m_rowweight = rowWeight(higraph);
	}	}	}

	private static boolean shake(HiGraph row[])
	{
		HiGraph	higraph, other;
		int		i, j, best, bestreduction, weight, otherweight, weight1, otherweight1, reduction;
		boolean	ret;

		ret = false;
		for (i = row.length; i > 1; ) {
			higraph       = row[--i];
			weight        = higraph.m_rowweight;
			best          = -1;
			bestreduction = 0;
			for (j = i; j > 0; ) {
				other = row[--j];

				otherweight = other.m_rowweight;
				higraph.m_position = j;
				other.m_position   = i;
				weight1            = rowWeight(higraph);
				otherweight1       = rowWeight(other);
				higraph.m_position = i;
				other.m_position   = j;
				reduction          = weight + otherweight - weight1 - otherweight1;
				if (reduction > bestreduction) {
					best          = j;
					bestreduction = reduction;
			}	}
			if (bestreduction > 0) {
				other              = row[best];
				row[i]             = other;
				row[best]          = higraph;
				other.m_position   = i;
				higraph.m_position = best;
				higraph.m_rowweight= rowWeight(higraph);
				other.m_rowweight  = rowWeight(other);
				ret                = true;
		}	}
		return(ret);
	}

	// Returns true if children have been considered for reordering

	private static boolean orderchildren(HiGraphLayout options, HiGraph node) throws HiGraphException {

		HiArc		arc;
		HiGraph		child;
		Vector		children = node.m_children;
		int			size     = children.size();
		int			i;
		boolean		ret      = false;
		
		node.m_position = 0;
		if (size == 0) {
			return(ret);
		}

		// Order children from the bottom up
		for (i = 0; i < size; ++i) {
			arc = (HiArc) children.elementAt(i);
			child = arc.to();
			ret |= orderchildren(options, child);
		}
		
		// Only reorder at the lowest reorderable level
		if (ret) {
			return(ret);
		}
		
		if (node.dontReorder()) {
			return(true);
		}

		HiGraph board[][]      = buildBoard(node);		// [rank][offset within rank]
		int		cycles         = options.crossing();
		int		to;
		boolean	improved;

		initBoard(board);

		to = board.length - 1;

		for (;;) {
			improved = false;
			for (i = 1; i < to; ++i) {
				improved |= shake(board[i]);
			}
			if (!improved) {
				break;
			}
			improved = false;
			for (i = to; i > 0; --i) {
				improved |= shake(board[i]);
			}
			if (!improved) {
				break;
			}
			// Count each row as a cycle to balance load on large graphs
			cycles -= to;
			if (cycles < 0) {
				break;
		}	}

		if (debug) {
			System.out.println("After ordering lowest level of descendant");
			dumpBoard(node, board);
			node.dump();
		}
		return(true);
	}
	
	static void order(HiGraphLayout options, HiGraph root) throws HiGraphException 
	{
		orderchildren(options, root);

		if (debug) {
			System.out.println("After ordering children");
			root.dump();
		}
	}
}
