package lsedit;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Container;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JDialog;
import javax.swing.JButton;
import javax.swing.JTextField;
import javax.swing.JPanel;
import javax.swing.JLabel;

/** 
 *   Developed by: Ian Davis in Summer 1997 for Grant Weddell
 *   Class which hides all of the various display options which may be passed (as a structure)
 *   to Layouter.   
 */

class HiGraphConfigure extends JDialog implements ActionListener {

	protected JTextField		m_xgap_textField;				// specify horizontal gap between nodes
	protected JTextField		m_ygap_textField;				// specify vertical gap between nodes
	protected JTextField    	m_simplex_textField;			// specify the maximum number of simplex pivots
	protected JTextField		m_crossing_textField;			// specify the maximum number of crossing cycles

	protected JLabel			m_message;

	protected JButton			m_ok;
	protected JButton			m_default;
	protected JButton			m_cancel;
	
	public HiGraphConfigure(LandscapeEditorCore ls)
	{
		super(ls.getFrame(), "Set Simplex Algorithm parameters", true);

		Container			contentPane;
		Font				font, bold;
		JLabel				label;

		font   = FontCache.getDialogFont();
		bold   = font.deriveFont(Font.BOLD);

		setForeground(ColorCache.get(0,0,0));
		setBackground(ColorCache.get(192,192,192));
		setFont(font);

		JPanel topPanel    = new JPanel();
		JPanel leftPanel   = new JPanel();
		JPanel rightPanel  = new JPanel();

		GridLayout gridLayout;

		topPanel.setLayout( new BorderLayout() );
		gridLayout = new GridLayout(4,1);
		gridLayout.setVgap(10);
		leftPanel.setLayout(gridLayout);

		gridLayout = new GridLayout(4,1);
		gridLayout.setVgap(10);
		rightPanel.setLayout( gridLayout);

		label = new JLabel("Minimum horizontal gap between nodes:", JLabel.RIGHT);
		label.setFont(bold);
		leftPanel.add(label);

		m_xgap_textField = new JTextField("" + HiGraphLayout.xgap(), 6);
		m_xgap_textField.setFont(font);
		rightPanel.add(m_xgap_textField);

		label = new JLabel("Minimum vertical gap between nodes:", JLabel.RIGHT);
		label.setFont(bold);
		leftPanel.add(label);

		m_ygap_textField = new JTextField("" + HiGraphLayout.ygap(), 6);
		m_ygap_textField.setFont(font);
		rightPanel.add(m_ygap_textField);

		label = new JLabel("Maximum allowed simplex pivots:", JLabel.RIGHT);
		label.setFont(bold);
		leftPanel.add(label);

		m_simplex_textField = new JTextField("" + HiGraphLayout.simplex(), 6);
		m_simplex_textField.setFont(font);
		rightPanel.add(m_simplex_textField);

		label = new JLabel("Maximum reorderings of a row:", JLabel.RIGHT);
		label.setFont(bold);
		leftPanel.add(label);

		m_crossing_textField = new JTextField("" + HiGraphLayout.crossing(), 6);
		m_crossing_textField.setFont(font);
		rightPanel.add(m_crossing_textField);

		topPanel.add( BorderLayout.WEST, leftPanel);
		topPanel.add( BorderLayout.EAST, rightPanel);

		contentPane = getContentPane();
		
		contentPane.add( BorderLayout.NORTH, topPanel );

		m_message = new JLabel(" ", JLabel.CENTER);
		m_message.setFont(font);
		m_message.setForeground(Color.RED);

		m_message.setSize(400,50);
		contentPane.add( BorderLayout.CENTER, m_message);

		// --------------
		// Use a FlowLayout to center the button and give it margins.

		JPanel bottomPanel = new JPanel();

		bottomPanel.setLayout(new FlowLayout(FlowLayout.CENTER, 15, 15));

		m_ok = new JButton("Ok");
		m_ok.setFont(bold);
		bottomPanel.add(m_ok);
		m_ok.addActionListener(this);

		m_default = new JButton("Default");
		m_default.setFont(bold);
		bottomPanel.add(m_default);
		m_default.addActionListener(this);

		m_cancel = new JButton("Cancel");
		m_cancel.setFont(bold);
		bottomPanel.add(m_cancel);
		m_cancel.addActionListener(this);

		contentPane.add( BorderLayout.SOUTH, bottomPanel);

		// Resize the window to the preferred size of its components
		pack();
		show();
	}

	// ActionListener interface

	public void actionPerformed(ActionEvent ev)
	{
		Object			source;

		source = ev.getSource();

		if (source == m_default) {
			m_xgap_textField.setText("" + HiGraphLayout.horizVertexGap);
			m_ygap_textField.setText("" + HiGraphLayout.vertVertexGap);
			m_simplex_textField.setText("" + HiGraphLayout.simplexDefault);
			m_crossing_textField.setText("" + HiGraphLayout.crossingDefault);
			return;
		}

		if (source != m_cancel && source != m_ok) {
			return;
		}

		if (source == m_ok) {
			String	string;
			int		xgap, ygap, simplex, crossing;

			string = m_xgap_textField.getText();
			try {
				xgap = Integer.parseInt(string);
				if (xgap < 0) {
					m_message.setText("Horizontal gap negative");
					return;
				}
			} catch (Throwable exception) {
				m_message.setText("Horizontal gap not an integer string value");
				return;
			}

			string = m_ygap_textField.getText();
			try {
				ygap = Integer.parseInt(string);
				if (ygap < 0) {
					m_message.setText("Vertical gap negative");
					return;
				}
			} catch (Throwable exception) {
				m_message.setText("Vertical gap not an integer string value");
				return;
			}

			string = m_simplex_textField.getText();
			try {
				simplex = Integer.parseInt(string);
				if (simplex < 0) {
					m_message.setText("Maximum pivots negative");
					return;
				}
			} catch (Throwable exception) {
				m_message.setText("Maximum pivots not an integer string value");
				return;
			}

			string = m_crossing_textField.getText();
			try {
				crossing = Integer.parseInt(string);
				if (crossing < 0) {
					m_message.setText("Maximum crossing cycles negative");
					return;
				}
			} catch (Throwable exception) {
				m_message.setText("Maximum crossing cycles not an integer string value");
				return;
			}
			HiGraphLayout.xgap(xgap);
			HiGraphLayout.ygap(ygap);
			HiGraphLayout.simplex(simplex);
			HiGraphLayout.crossing(crossing);
		}
		setVisible(false);
		return;
	}
}

public class HiGraphLayout {

	public final static int		horizVertexGap = 20;		// default horizontal gap between boxes	 
	public final static int		vertVertexGap  = 20;		// default vertical gap between boxes
	public final static int		horizMargin    = 2;			// horizontal margin for the graph
	public final static int		vertMargin     = 2;			// vertical margin for graph
	public final static int		simplexDefault = 10000;		// Number of pivots
	public final static int		crossingDefault= 125;		// Number of reorderings of rows

	/* Global display parameters */

	private static int			m_xgap     = vertVertexGap;	// specify horizontal gap between nodes
	private static int			m_ygap     = horizVertexGap;// specify vertical gap between nodes
	private static int     		m_simplex  = simplexDefault;// specify the maximum number of simplex pivots
	private static int			m_crossing = crossingDefault;// specify the maximum number of crossing cycles
	private static int			m_weight_1dummy = 0;		// Weight of edge with one dummy node in it (2 if showing such nodes)
	private static int			m_weight_2dummy = 0;		// Weight of edge with two dummy nodes in it (8 if showing such nodes)
	private static int			m_outside_bias  = 1;		// Bias for outside edges in layout
	private static int			m_group_children = 0;		// Group children above/below higraphs

	/* Other */

	private	static boolean		m_bends    = false;			// True if will be using the bends in arcs
	
	/**
	 * Constructor.
	 */

	public HiGraphLayout() 
	{
	}

	public static void configure(LandscapeEditorCore ls)
	{
		HiGraphConfigure configure = new HiGraphConfigure(ls);

		configure.dispose();
	}
		
	public static void setBends(boolean value)
	{
		if (value) {
			m_weight_1dummy = 2;
			m_weight_2dummy = 8;
		} else {
			m_weight_1dummy = 0;
			m_weight_2dummy = 0;
		}
		m_bends = value;
	}
	
	public static boolean hasBends()
	{
		return m_bends;
	}

	public static int	xgap() 
	{
		return (m_xgap);
	}

	public static boolean xgap(int value) 
	{
		if (value >= 1) {
			m_xgap = value;
			return(true);
		}
		return(false);
	}

	public static int ygap() 
	{
		return (m_ygap);
	}

	public static boolean ygap(int value) 
	{
		if (value >= 1) {
			m_ygap = value;
			return(true);
		}
		return(false);
	}

	public static int	simplex() 
	{
		return(m_simplex);
	}

	public static boolean simplex(int value) 
	{
		if (value >= 0) {
			m_simplex = value;
			return(true);
		}
		return(false);
	}

	public static int crossing() 
	{
		return(m_crossing);
	}

	public static boolean crossing(int value) 
	{
		if (value >= 0) {
			m_crossing = value;
			return(true);
		}
		return(false);
	}

	public static int weight_1dummy() 
	{
		return(m_weight_1dummy);
	}

	public static boolean weight_1dummy(int value) 
	{
		if (value >= 1) {
			m_weight_1dummy = value;
			return(true);
		}
		return(false);
	}

	public static int weight_2dummy() 
	{
		return(m_weight_2dummy);
	}

	public static boolean weight_2dummy(int value) 
	{
		if (value >= 1) {
			m_weight_2dummy = value;
			return(true);
		}
		return(false);
	}

	public static int outside_bias() 
	{
		return(m_outside_bias);
	}

	public static boolean outside_bias(int value) 
	{
		if (value >= 0) {
			m_outside_bias = value;
			return(true);
		}
		return(false);
	}

	public static int group_children() 
	{
		return(m_group_children);
	}

	public static boolean group_children(int value) 
	{
		if (value >= -1 && value <= 1) {
			m_group_children = value;
			return(true);
		}
		return(false);
	}

	public static int xmargin() {
		return (horizMargin);
	}

	public static int ymargin() {
		return (vertMargin);
	}

	// --- Main processing logic

	// Once the highgraph has been laid out this may be called separately if the fonts change

	public boolean coordinates(HiGraph root) 
	{
		try {
//			System.out.println("Laying out");
			HiGraphCoordinates.coordinates(this, root);
//			System.out.println("Layout done");
			return(true);
		}
		catch (HiGraphException e) {
			System.out.println("HiGraph error in HiGraphCoordinates: " + e.getMessage());
			e.printStackTrace();
		}
		catch (Exception e) {
			System.out.println("Error during HiGraph layout in HiGraphCoordinates: " + e.getMessage());
			e.printStackTrace();
		}
		return(false);
	}

	public boolean layout(HiGraph root)
	{
		boolean ret;

		try {
			
//			System.out.println("HiGraphLayout: beginning group");
			HiGroup.group(root, group_children());

			// Compute an optimal vertical ranking 
//			System.out.println("HiGraphLayout: beginning ranking");
			HiRank.compute(root, simplex());

			// Compute a good horizontal ordering for higraphs based on edges between higraphs
//			System.out.println("HiGraphLayout: beginning child ordering");
			HiChildren.order(root, outside_bias());

			// Compute a good horizontal ordering within each higraph by exchanging positions
			// System.out.println("Before interchange\n");
			// root.dump();

//			System.out.println("HiGraphLayout: beginning row ordering");
			HiRow.order(this, root);
			// System.out.println("After interchange\n");
			// root.dump();
//			System.out.println("HiGraphLayout: beginning coordinates");
			ret = coordinates(root);
//			System.out.println("HiGraphLayout: success");
			return(ret);
		}
		catch (HiGraphException e) {
			System.out.println("HiGraph error in " + root + ": " + e.getMessage());
			e.printStackTrace();
		}
		catch (Exception e) {
			System.out.println("Error during HiGraph layout in " + root + ": " + e.getMessage());
			e.printStackTrace();
		}
		return(false);
	}
}

