package lsedit;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;

import javax.swing.JFrame;
import javax.swing.JMenuBar;

class LandscapeEditorFrame extends JFrame implements WindowListener
{
	protected static final int MAX_WIDTH  = 1280;
	protected static final int MAX_HEIGHT = 1024;

	protected static Dimension editorDim = null;			// Set by -g option
	protected static Dimension diagDim = null;				// Set by -G option
	
	protected boolean				m_landscapeGeo         = false;			// Set by -L option
	protected int					m_frameWidth           = 0;
	protected int					m_frameHeight          = 0;
	protected int					m_diagramPercentWidth  = 0;
	protected int					m_diagramPercentHeight = 0;
	protected String				m_use_layouter         = null;		// Set by -l option
	protected int					m_handicapped          = 0;			// Set by -b option

	protected LandscapeEditorCore	m_app;

	// This just gives back something we can then set parameters on

	public LandscapeEditorFrame() 
	{
		super();

		MsgOut.setVerboseFlag(false); 
		MsgOut.setDebugFlag(false);
	}

	public static LandscapeEditorFrame create() 
	{
		return new LandscapeEditorFrame();
	}

	public LandscapeEditorCore getApp()
	{
		if (m_app == null) {
			m_app = new LandscapeEditorCore(this);

			if (m_use_layouter != null) {
				m_app.defaultToLayouter(m_use_layouter);
		}	}
		return(m_app);
	}

	public void launch()
	{
		LandscapeEditorCore	app;
		JMenuBar			mb;

		if (m_frameWidth <= 0 || m_frameHeight <= 0) {
			Toolkit tk = Toolkit.getDefaultToolkit();

			Dimension dim = tk.getScreenSize();

			double f = 0.75;

			if (m_landscapeGeo) {
				m_frameWidth  = Math.min(MAX_WIDTH, dim.width);
				m_frameHeight = Math.min(MAX_HEIGHT, dim.height);
			}
			else {
				m_frameWidth  = (int) (dim.width/f);
				m_frameHeight = dim.height-50;
			}

			m_frameWidth  = Math.min(m_frameWidth,  dim.width);
			m_frameHeight = Math.min(m_frameHeight, dim.height);
		}
		setBounds(0, 0, m_frameWidth, m_frameHeight); 

		app = getApp();

		setTitle(app.getTitle());
		setVisible(true);
		mb = app.genMenu();
		setJMenuBar(mb);
		
		app.init_core(m_diagramPercentWidth, m_diagramPercentHeight);

		if (m_handicapped > 0) {
			app.handicapped(m_handicapped);
		}

		addWindowListener(this);				// Want to be able to detect x pressed in Top RH corner
		setBackground(Color.lightGray);
		setVisible(true);
	}

	public void setDebugOn()
	{
		// Turn on debugging output 
		MsgOut.setDebugFlag(true);
		MsgOut.dprintln("Start app: ");
	}

	public void setVerboseOn()
	{
		// Turn on verbose parsing output 
		MsgOut.setVerboseFlag(true); 
	}

	public void setLandscapeGeo()
	{
		m_landscapeGeo = true;
	}

	public boolean setFrameDimension(int width, int height) 
	{
		if (width > 0 && height > 0) {
			m_frameWidth  = width;
			m_frameHeight = height;
			return true;
		}
		return false;
	}
	// Returns the dimensions for input of form <x>x<y>

	public boolean setFrameDimension(String geo) 
	{
		int ind = geo.indexOf('x');

		if (ind > 0) {
			int	w, h;

			try {
				w = Util.parseInt(geo.substring(0, ind));
				h = Util.parseInt(geo.substring(ind+1));
				return(setFrameDimension(w, h));
			} catch (Exception e) {
			}
		}
		return false;
	}

	public boolean setDiagramPercent(int width, int height) 
	{
		if (width > 0 && width <= 100 && height > 0 && height <= 100) {
			m_diagramPercentWidth  = width;
			m_diagramPercentHeight = height;
			return true;
		}
		return false;
	}

	// Returns the dimensions for input of form <x>x<y>

	public boolean setDiagramPercent(String geo) 
	{
		int ind = geo.indexOf('x');

		if (ind > 0) {
			int	w, h;

			try {
				w = Util.parseInt(geo.substring(0, ind));
				h = Util.parseInt(geo.substring(ind+1));
				return(setDiagramPercent(w, h));
			} catch (Exception e) {
			}
		}
		return false;
	}

	public void setHandicapped(String fontsize) 
	{
		int	size;

		try {
			size = Util.parseInt(fontsize);
		} catch (Exception e) {
			size = 20;
		}
		m_handicapped = size;
	}

	public void showToc()
	{
		LandscapeEditorCore	app;

		app = getApp();
		app.setTocHiddenState(false);
	}

	public boolean setLayouter(String layouter)
	{
		m_use_layouter = layouter;
		return(true);
	}

	public boolean setStartEntity(String entityName)
	{
		LandscapeEditorCore	app;

		app = getApp();
		app.m_startEntity = entityName;
		return(true);
	}

	public boolean setSavePath(String path)
	{
		LandscapeEditorCore	app;

		app = getApp();
		app.lsSavePath = path;
		return(true);
	}

	public boolean setSaveSuffix(String suffix)
	{
		LandscapeEditorCore	app;

		app = getApp();
		app.lsSaveSuffix = suffix;
		return(true);
	}

	public boolean setSaveCommand(String command)
	{
		LandscapeEditorCore	app;

		app = getApp();
		app.lsSaveCmd = command;
		return(true);
	}

	public boolean setLsPath(String file)
	{
		LandscapeEditorCore	app;

		app = getApp();
		if (app.m_lsPath == null) {
			app.m_lsPath = file;
		} else {
			app.addLseditHistory(file);
		}
		return(true);
	}

	public static void usage() 
	{
		System.out.println("\nLandscape Editor " + Version.MAJOR + "." + Version.MINOR + "." + Version.BUILD + "\n");
		System.out.println("<executor> LandscapeEditorFrame [options] <HTTP or file path>\n");
		System.out.println("-h	 This message");
		System.out.println("-v	 Verbose parsing output");
		System.out.println("");

		System.out.println("-g<width>x<height> Geometry of editor");
		System.out.println("-G<width>x<height> Geometry of diagram");
		System.out.println("");

		System.out.println("-l \"matrix\"   Use the matrix layout algorithm in any initial layout ");
  		System.out.println("-l \"simplex\"  Use the network simplex layout algorithm in any initial layout ");
		System.out.println("-l \"sugiyama\" Use the sugiyama layout algorithm in any initial layout ");

		System.out.println("-L	 Start with landscape geometry");
		System.out.println("-V	 Start in viewer mode");
		System.out.println("");

		System.out.println("-P <path>  Optional save path");
		System.out.println("-S <path>  Optional save suffix");
		System.out.println("-X <path>  Optional app to exec on save");
		System.out.println("");
	}

	// A program can pass arguments this way without invoking main if it wishes
	// However in general it would probably prefer to just make the necessary calls

	public boolean setOptions(String args[]) 
	{
		int		n, i, len;
		char	c;
		boolean	ok;

		ok = true;
		
		// Decide what we are running

		for (n= 0; n < args.length && args[n].charAt(0) == '-'; ++n) {
			c = args[n].charAt(1);

			switch(c) {
			case 'h':
				// Show usage info
				usage();
				return(false);
		}	}

		// Process command line switches

		for (n= 0; n < args.length && args[n].charAt(0) == '-'; ++n) {
			len = args[n].length();
			for (i = 1; i < len; i++) {
				c = args[n].charAt(i);

				switch(c) {
				case 'd':
					setDebugOn();
					break;
				case 'b':	// big
					setHandicapped(args[n].substring(i+1));
					break;
				case 't':
					showToc();
					break;

				case 'v':
					setVerboseOn();
					break;

				case 'g':
					if (!setFrameDimension(args[n].substring(i+1))) {
						System.out.println("Illegal -g option [ignored]");
						ok = false;
					}
					i = len;
					break;

				case 'G':
					if (!setDiagramPercent(args[n].substring(i+1))) {
						System.out.println("Illegal -G option [must be percentages]");
						ok = false;
					}
					i = len;
					break;

				case 'L':
					setLandscapeGeo();
					break;
				
				case 'l':
					++n;
					if (n >= args.length || !setLayouter(args[n])) {
						System.out.println("\n-l <string> missing");
						ok = false;
					} 
					i = len;
					break;
				case 'N':
					++n;
					if (n >= args.length || !setStartEntity(args[n])) {
						System.out.println("\nInvalid use of -N");
						ok = false;
					}
					i = len;	
					break;

				case 'P':
					n++;
					if (n >= args.length || !setSavePath(args[n])) {
						System.out.println("\nInvalid use of -P");
						ok = false;
					}
					i = len;	
					break;

				case 'S':
					n++;
					if (n >= args.length || !setSaveSuffix(args[n])) {
						System.out.println("\nInvalid use of -S");
						ok = false;
					}
					i = len;	
					break;
		
				case 'X':
					n++;
					if (n >= args.length || !setSaveCommand(args[n])) {
						ok = false;
					}
					i = len;
					break;
		}	}	}

		// Process command line args
		while (n < args.length) {
			setLsPath(args[n++]);
		}
		return(ok);
	}

	public static void main(String args[]) 
	{
		LandscapeEditorFrame m_af = new LandscapeEditorFrame();

		if (!m_af.setOptions(args)) {
			usage();
			System.exit(0);
		}
		m_af.launch();

		MsgOut.dprintln("exit main");
	}

	// Window Listener implementation follows

	public void windowOpened(WindowEvent evt) 
	{
	}

	public void windowActivated(WindowEvent evt) 
	{
	}

	public void windowClosed(WindowEvent evt) 
	{
	}

	public void windowClosing(WindowEvent evt) 
	{
//		System.out.println("LandscapeEditorFrame.windowClosing");

		if (m_app != null) {
			m_app.testForClose(true);
		}
		removeWindowListener(this);
		dispose();

	}

	public void windowDeactivated(WindowEvent evt) 
	{
	}

	public void windowDeiconified(WindowEvent evt) 
	{
	}

	public void windowIconified(WindowEvent evt) 
	{
	}
}
