package lsedit;

// Layout method using Coffman-Graham Layer Assignment algorithm and the
// Sugiyama algorithm.

// John S. Y. Lee January 2000

import java.util.Enumeration;
import java.util.Vector;

public class SugiyamaLayout extends MatrixLayout implements ToolBarEventHandler {

	public final static double BORDER = 0.0333;	// 1/30 of graph all round
	public final static double XGAP   = 0.2;		// Make gap 1/5 of space
	public final static double YGAP   = 0.2;		// Make gap 1/5 of space

	protected static double m_border  = BORDER;
	protected static double m_xgap    = XGAP;
	protected static double m_ygap    = YGAP;

	public SugiyamaLayout(LandscapeEditorCore ls) 
	{
		super(ls);
	}

	public String getName()
	{
		return "Sugiyama";
	}

	public String getMenuLabel() 
	{
		return "Layout using Sugiyama algorithm";
	} // getMenuLabel


	public static double getBorder()
	{
		return m_border;
	}

	public void setBorder(double value)
	{
		m_border = value;
	}

	public static double getXGap()
	{
		return m_xgap;
	}

	public void setXGap(double value)
	{
		m_xgap = value;
	}

	public static double getYGap()
	{
		return m_ygap;
	}

	public void setYGap(double value)
	{
		m_ygap = value;
	}

	public boolean isConfigurable()
	{
		return true;
	}

	public void configure(LandscapeEditorCore ls)
	{
		MatrixConfigure configure = new MatrixConfigure(this, BORDER, XGAP, YGAP, m_border, m_xgap, m_ygap);

		configure.dispose();
	}

  // The doLayout method executes the Coffman-Graham Layer Assignment
  // algorithm and the Sugiyama algorithm on the boxes selected.
  // Assumption: All boxes selected are in the same container.

  // This is called directly to layout incoming TA

	public void doLayout1(Vector selectedBoxes, EntityInstance container, boolean update) 
	{
		int				width, height;
		Enumeration		en, edge, f;
		EntityInstance	e, child;
		int				vertexID = 0;
		int				groupSize = selectedBoxes.size();
		RelationInstance relation;
		RelationClass	 rc;
		int				index;

		width  = container.getWidth();
		height = container.getHeight();
		
		// create graph to store info on selected boxes and their relationships
		Graph graph = new Graph(groupSize);

		/* Graph is an array of vertex's */

		for (en = selectedBoxes.elements(); en.hasMoreElements(); ) {
			e = (EntityInstance) en.nextElement();
			graph.addVertex(e.widthRelLocal());
			e.orMark(EntityInstance.IN_GRAPH_MARK);
		}

		// get (directed) relationships between the boxes

		for (en = selectedBoxes.elements(); en.hasMoreElements(); ++vertexID) {
			e = (EntityInstance) en.nextElement();

			// Need only look at src since will be src for one or other node in diagram

			for (edge = e.srcRelationElements(); edge.hasMoreElements(); ) {
				relation = (RelationInstance) edge.nextElement();
				rc       = (RelationClass)    relation.getParentClass();
				if (rc == null || rc.isClassVisible()) {
					child = relation.getDst();
					if (child != e && child.isMarked(EntityInstance.IN_GRAPH_MARK)) {
						// store relationship only if children is in the same container
						for (index = 0; index < selectedBoxes.size(); index++) {
							if (child == selectedBoxes.elementAt(index)) {
								break;
							}
						} // for
						graph.addEdge(vertexID, index);
					}
				}	
			} // for
		} // for

		for (en = selectedBoxes.elements(); en.hasMoreElements(); ) {
			e = (EntityInstance) en.nextElement();
			e.nandMark(EntityInstance.IN_GRAPH_MARK);
		}

/*
		int index = 0;
	  
		System.out.println("Graph read:");
		for (en = selectedBoxes.elements(); en.hasMoreElements(); index++) {
			System.out.println(index + " " + en.nextElement());
		}
		graph.print();
*/

		Vector layers = graph.doCoffmanGrahamSugiyama();

/*
		System.out.println("RESULTS");
		int level = 0;
		for (en = layers.elements(); en.hasMoreElements(); ) {
			System.out.println("level " + level);
			for (Enumeration f = ((Vector)en.nextElement()).elements(); f.hasMoreElements(); ) {
				System.out.print(f.nextElement() + " ");
			}
			System.out.println();
		}
 */

		// Assign coordinates to each box

		Vector	row;
		int		row_across, max_across;

		max_across       = 0;
		for (en = layers.elements(); en.hasMoreElements(); ) {
			row        = (Vector) en.nextElement();
			row_across = row.size();
			if (row_across > max_across) {
				max_across = row_across;
		}	}

		int		available_width, available_height;
		int		row_gap, column_gap;
		double  widthrel, heightrel;

		available_width  = (int) ((1.0 - m_border) * (double) width);
		available_height = (int) ((1.0 - m_border) * (double) height);

		if (max_across < 2) {
			row_gap = 0;
		} else {
			row_gap = (int) ((m_xgap * (double) available_width)/(max_across - 1)); 
		}

		int		numLayers   = layers.size();

		if (numLayers < 2) {
			column_gap = 0;
		} else {
			column_gap = (int) ((m_ygap * (double) available_height)/(numLayers - 1)); 
		}


		int		graph_width, graph_height, row_width, row_height, entity_height;

		graph_width      = 0;
		graph_height     = 0;

		for (en = layers.elements(); en.hasMoreElements(); ) {
			row = (Vector) en.nextElement();
			row_width  = 0;
			row_height = 0;
			for (f = row.elements(); f.hasMoreElements(); ) {
				int curVertex = ((Integer)f.nextElement()).intValue();

				// Process non-dummy boxes only
				if (curVertex < groupSize) {
					e          = (EntityInstance) selectedBoxes.elementAt(curVertex);
					widthrel   = e.widthRelLocal();
					heightrel  = e.heightRelLocal();

					row_width += row_gap + (int) (widthrel * (double) width);
					entity_height = (int) (heightrel * (double) height);
					if (entity_height > row_height) {
						row_height = entity_height;
					}

				} // if
			} // for
			row_width -= row_gap;	// No gap before first element

			if (row_width > graph_width) {
				graph_width = row_width;
			}
			graph_height += column_gap + row_height;
		}
		graph_height -= column_gap;	// No gap before first column

		double	scaleX, scaleY, xmargin, ymargin;

		scaleX = ((double) available_width) / ((double) graph_width);
		if (scaleX < 1.0) {
			xmargin = m_border/2.0;
		} else {
			xmargin = ((double) (width-graph_width))/(2.0 * ((double) width));
			scaleX  = 1.0;
		} 

		scaleY = ((double) available_height) / ((double) graph_height);
		if (scaleY < 1.0) {
			ymargin = m_border/2.0;
		} else {
			ymargin = ((double) (height-graph_height))/(2.0 * ((double) height));	// Center on Y axis
			scaleY  = 1.0;
		} 

		double	curX, curY, xrel, yrel, maxheightrel;
		double	xgap    = scaleX * ((double) row_gap)/((double) width);
		double  ygap    = scaleY * ((double) column_gap)/((double) height);

		curY = ymargin;

		for (index = layers.size(); index > 0; ) {
			row          = (Vector) layers.elementAt(--index);
			curX         = xmargin;
			maxheightrel = 0;
			for (f = row.elements(); f.hasMoreElements(); ) {
				int curVertex = ((Integer)f.nextElement()).intValue();

				// Process non-dummy boxes only
				if (curVertex < groupSize) {
					e         = (EntityInstance) selectedBoxes.elementAt(curVertex);

					widthrel  = e.widthRelLocal()  * scaleX;
					heightrel = e.heightRelLocal() * scaleY;
					if (heightrel > maxheightrel) {
						maxheightrel = heightrel;
					}
					if (update) {
						e.updateRelLocal(curX, curY, widthrel, heightrel);
					} else {
						e.setRelLocal(curX, curY, widthrel, heightrel);
					}
					curX += widthrel + xgap;
				} // if
			} // for
			curY += maxheightrel + ygap;
		} // for

	} // doLayout


  // The doLayout method executes the Coffman-Graham Layer Assignment
  // algorithm and the Sugiyama algorithm on the boxes selected.
  // Assumption: All boxes selected are in the same container.

	public String doLayout(Diagram dg) 
	{
		EntityInstance	parent;

		m_ls.setLayouter(this);
		// get user's selection of boxes to be laid out

		Vector selectedBoxes = dg.getGroup();
		if (selectedBoxes == null) {
			  beep();
			  return "No group selected";
		}

		parent = parentOfSet(selectedBoxes);
		if (parent == null) {
			return	"Sugiyama requires that all things laid out share same parent";
		}
		dg.beginUndoRedo("Sugiyama layout");
		doLayout1(selectedBoxes, parent, true /* Updating */);
		dg.endUndoRedo();
		return "Graph redrawn with Coffman-Graham-Sugiyama method";

	} // doLayout

	public void processKeyEvent(int key, int modifiers, Object object) 
	{
		Diagram	dg;
		String	rmsg;

		dg = m_ls.getDiagram();
		if (dg != null) {
			rmsg = doLayout(dg);
			m_ls.doFeedback(rmsg);
			dg.rescaleDiagram();
			m_ls.repaintDg();
	}	}

} // SugiyamaLayout






