package lsedit;

import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.util.Enumeration;
import java.util.Vector;

import javax.swing.JLabel;
import javax.swing.JPanel;

// This object has the responsibility for laying out clients and supplier lists

abstract public class ClientSupplierSet extends JPanel implements DiagramCoordinates 
{
	public final static int CLIENT_SUPPLIER_HEIGHT = 50;
	public final static int GAP = 5;

	public final static String DEFAULT_CLIENT_FONT_NAME  = FontCache.DEFAULT_FONT_NAME;
	public final static int    DEFAULT_CLIENT_FONT_STYLE = Font.PLAIN;
	public final static int    DEFAULT_CLIENT_FONT_SIZE  = 10;

	protected     static Font   m_clientFont    = FontCache.get(DEFAULT_CLIENT_FONT_NAME, DEFAULT_CLIENT_FONT_STYLE, DEFAULT_CLIENT_FONT_SIZE);

	// Established by constructor

	protected LandscapeEditorCore		m_ls;
	protected Diagram					m_diagram;
	protected JLabel                    m_label = null;


	// Working variables

	protected EntityInstance	m_drawRoot;					// The current entity for which clients/services are being computed
	protected Vector			m_set;						// Full set identified
	protected int				m_fullSize;					// Number of members in m_set
	protected int				m_displayedSize;

	// Calculate the width needed to display the clients/suppliers

	protected int calcWidth(Graphics g) 
	{
		Enumeration		en;
		Dimension		dim;
		EntityInstance	e;
		int				tw;

		tw = 0;
		for (en = m_set.elements(); en.hasMoreElements(); ) {
			e = (EntityInstance) en.nextElement();
			dim = e.getFitDim(g, EntityInstance.SMALL_FONT, true);
			tw += dim.width;
		}
		return tw;
	}

	// Returns the entity which is the parent of the most entities in the vector m_set

	protected EntityInstance mostFrequentParent() 
	{
		Vector			p   = new Vector();
		int[]			cnt = new int[m_set.size()];
		Enumeration		en;
		EntityInstance	e, pe;
		int				ind, mVal, mInd;

		for (en = m_set.elements(); en.hasMoreElements(); ) {
			e  = (EntityInstance) en.nextElement();
			pe = e.getContainedBy();
			if (pe.hasDescendantOrSelf(m_drawRoot)) {
				// m_drawRoot is a descendant of pe
				continue;
			}
			 ind = p.indexOf(pe);

			if (ind < 0) {
				// Haven't yet seen this parent
				cnt[p.size()] = 1;
				p.addElement(pe);
			} else {
				// Add one to this parents count
				cnt[ind]++;
		}	}

		mVal = 0;
		mInd = -1;

		for (int i = 0; i < p.size(); i++) {
			if (cnt[i] > mVal) {
				mVal = cnt[i];
				mInd = i;
		}	}

		if (mInd < 0) {
			return(null);
		}
		return (EntityInstance) p.elementAt(mInd);
	}

	// Remove all members of v which are descendants of pe
	// Returns the number of items eliminated

/*
	protected void showDescendents(EntityInstance pe)
	{
		Vector			v = m_set; 
		int				i;
		EntityInstance	ei;

		for (i = v.size(); i > 0; ) {
			ei = (EntityInstance) v.elementAt(--i);
			if (pe.hasDescendantOrSelf(ei)) {
				System.out.print(" " + ei);
	}	}	}

	protected void showMembers()
	{
		Vector			v = m_set; 
		int				i;
		EntityInstance	ei;

		for (i = v.size(); i > 0; ) {
			ei = (EntityInstance) v.elementAt(--i);
			System.out.print(" " + ei);
	}	}
*/

	protected boolean elimDescendents(EntityInstance pe) 
	{
		Vector			v = m_set; 
		int				i;
		EntityInstance	ei;
		boolean			ret = false;

		for (i = v.size(); i > 0; ) {
			ei = (EntityInstance) v.elementAt(--i);
			if (pe.hasDescendantOrSelf(ei)) {
				ei.nandMark(EntityInstance.CLIENT_MARK|EntityInstance.SUPPLIER_MARK);
				v.remove(i);
				ret = true;
		}	}
		return(ret);
	}

	protected void sort() 
	{
		if (m_set.size() > 1) {
			SortVector.byAvgX(m_set, true);
	}	}

	/* Validation is a problem here.  The problem is that while components are painted from
	 * high order element down, they are validated from low order up.  This means that the
	 * edges get validated in the diagram before the client and server gets validated.  Thus
	 * the edges end up pointing at where entities were -- not where they are moved to by
	 * the validation process. So we set the bounds on the entities in the client/supplier
	 * set when we add into that set, and there after don't change these values.
	 */

	protected void addSet() 
	{
		int	components = m_set.size();
		
		if (components != 0) {
			int		diagramX, diagramY, w, width, width1, x, height, tw, gaps;
			double	xpos, scale, width2;
			Enumeration en;
			EntityInstance e;
			Dimension dim;

			gaps  = ((components+1) * GAP);	// Allow for a gap on both sides
			w     = getWidth();
			width = w - gaps;
			
/*
			if (width < components) {		// Each component needs at least one pixel
				int	h;

				if (m_label == null) {
					m_label = new JLabel();
					m_label.setFont(ClientSupplierSet.getClientSupplierFont());
				}
				m_label.setText(components + " entities is too many to show");
				m_label.setHorizontalAlignment(JLabel.CENTER);

				h = getHeight();
				m_label.setBounds(0, 0, w, h);
				add(m_label);		// Add this message to the diagram
				return;
			}
*/			
			Graphics g = m_ls.getGraphics();

			// Total widths of all entities
			tw    = calcWidth(g);
		
			scale = ((double) width)/((double) tw);
			xpos  = GAP;
			
			if (scale > 2) {
				scale = 2;
				xpos += ((double) (width - tw*2))/2.0;
			}

			diagramX = getX();
			diagramY = getY();
			height   = getHeight();

			for (en = m_set.elements(); en.hasMoreElements(); ) {
				e = (EntityInstance) en.nextElement();
				e.removeAll();

				dim     = e.getFitDim(g, EntityInstance.SMALL_FONT, true);
				width2  = scale * (double) dim.width;
				width1  = (int) width2;
				
//				System.out.println("ClientSupplierSet.addSet " + e.m_label + "{" + (getX() + x) + "," + (getY() + y) + "/" + x + "," + y + "," + width1 + "," + dim.height + "}");
				e.setBothBounds(diagramX, diagramY, (int) xpos, 0, width1, /* dim.height */ height);
				e.setVisible(true);
				xpos  += width2 + GAP;
				e.addUnder(this);				// Add entities under client/suppliers
	}	}	}

	// --------------
	// Public methods
	// --------------

	public ClientSupplierSet(Diagram diagram) 
	{
		setLayout(null);
		m_diagram = diagram;
		m_ls      = diagram.getLs();
		m_set     = new Vector();
	}

	public static Font getClientSupplierFont() 
	{
		return m_clientFont;
	}

	public static void setClientSupplierFont(Font font)
	{
		m_clientFont = font;
	}

	// Return the full set of entities in our set

	public Vector getFullSet()
	{
		return(m_set);
	}

	public int getFullSetSize()
	{
		return(m_fullSize);
	}

	public int getDisplaySetSize()
	{
		return(m_displayedSize);
	}		

	public void removeAll()
	{
		super.removeAll();
		m_set.removeAllElements();
		m_fullSize      = 0;
		m_displayedSize = 0;	
	}

	public static void compact(ClientSet clientSet, SupplierSet supplierSet)
	{
		EntityInstance	e, pe;
		boolean			seen;
		Vector			clients   = clientSet.getFullSet();
		Vector			suppliers = supplierSet.getFullSet();
		int				clientWidth, supplierWidth, cwidth, swidth, i;
		Graphics		g;

		/* Strictly for debugging

		Object			debugClients[]   = clients.toArray();
		Object			debugSuppliers[] = suppliers.toArray(); 

		*/

		/* Two problems are addressed here, which are inter-related.
		 * (1) We can't physically show an entity as both client and supplier.. since
		 *     and EntityInstance has only one EntityComponent.  For safety we don't
		 *     even want to display X and Y where X contains Y since the problem of
		 *     duplication still arises if X is opened.  The solution here is to
		 *     throw away conflicts.
		 * (2) We don't want to display too many clients/suppliers if compacting.
		 *     The solution here is to aggregate
		 *
		 * At this point we know that clients and suppliers separately do not overlap
		 * because of the way they were initially found.  Specifically we don't look
		 * for clients under clients or suppliers under suppliers but the one set can 
		 * collide with the other
		 */

		// Move any clients that are suppliers or under suppliers across to the supplier side
		// Delete them from the client side

/*
		System.out.println("------------------------------------------------------------------------------------");

		System.out.print("ClientSupplierSet.compact() Clients =");
		clientSet.showMembers();
		System.out.println("");
		System.out.print("ClientSupplierSet.compact() Suppliers =");
		supplierSet.showMembers();
		System.out.println("");
*/

		for (i = clients.size(); i > 0; ) {
			e = (EntityInstance) clients.elementAt(--i);
			for (pe = e; pe != null; pe = pe.getContainedBy()) { 
				if (pe.isMarked(EntityInstance.SUPPLIER_MARK)) {
					e.nandMark(EntityInstance.CLIENT_MARK);		// This is no longer considered a client
					pe.orMark(EntityInstance.CLIENT_MARK);		// This is now considered a client as well as a supplier
					clients.remove(i);							// Remove e from the client side 
//					System.out.println("ClientSupplierSet.compact() Removing client " + e + ". At or under supplier " + pe);
		}	}	}

		// Remove any suppliers under clients.  We can start above each supplier
		// since we know no supplier is also in the clientset.  If it had been it 
		// would by now have been removed from the client set 

		seen = false;
		for (i = suppliers.size(); i > 0; ) {
			e = (EntityInstance) suppliers.elementAt(--i);
			for (pe = e.getContainedBy(); pe != null; pe = pe.getContainedBy()) { 
				if (pe.isMarked(EntityInstance.CLIENT_MARK)) {
					e.nandMark(EntityInstance.SUPPLIER_MARK);
					pe.orMark(EntityInstance.SUPPLIER_MARK);
					suppliers.remove(i);
//					System.out.println("ClientSupplierSet.compact() Removing supplier " + e + ". Under client " + pe);
					seen = true;
		}	}	}

		// Clients and suppliers are now quite separate
		// No client   is under a client or supplier
		// No supplier is under a client or supplier

		if (seen) {
			for (i = clients.size(); i > 0; ) {
				e = (EntityInstance) clients.elementAt(--i);
				if (e.isMarked(EntityInstance.SUPPLIER_MARK)) {
					clients.remove(i);
					suppliers.addElement(e);
		}	}	}

/*
		System.out.print("ClientSupplierSet.compact() Clients =");
		clientSet.showMembers();
		System.out.println("");
		System.out.print("ClientSupplierSet.compact() Suppliers =");
		supplierSet.showMembers();
		System.out.println("");
*/
		// All suppliers are now under suppliers even if they are also clients

		if (!clientSet.m_ls.isUseCompaction()) {
			return;
		}

		cwidth       = clientSet.getWidth();	
		swidth       = supplierSet.getWidth();

		if (cwidth > 0 || swidth > 0) {
			g             = clientSet.m_ls.getGraphics();
			clientWidth   = -1;
			supplierWidth = -1;

			for (;;) {
				seen = false;
				if (cwidth > 0) {
					if (clientWidth < 0) {
						clientWidth   = clientSet.calcWidth(g) + (GAP * (clients.size() + 1));	// Allow a gap both sides
					}
					if (clientWidth > cwidth) {
	//					System.out.println("ClientSupplierSet.compact clientWidth=" + clientWidth + " width=" + width);
						pe = clientSet.mostFrequentParent();
						if (pe == null) {
							// Cant do any more
							clientWidth = cwidth;
						} else {
							seen = true;
							// Eliminate the descendants of pe
/*
							System.out.print("ClientSupplierSet.compact() Using client " + pe + ". Deleting clients =");
							clientSet.showDescendents(pe);
							System.out.print(" and suppliers =");
							supplierSet.showDescendents(pe);
							System.out.println("");
 */
							clientSet.elimDescendents(pe);
							pe.orMark(EntityInstance.CLIENT_MARK);
							if (supplierSet.elimDescendents(pe)) {
								pe.orMark(EntityInstance.SUPPLIER_MARK);
								suppliers.addElement(pe);
								supplierWidth = -1;
							} else {
								clients.addElement(pe);
							}
							clientWidth = -1;
				}	}	}

				if (swidth > 0) {
					if (supplierWidth < 0) {
						supplierWidth   = supplierSet.calcWidth(g)   + (GAP * (suppliers.size() + 1));	// Allow a gap both sides
					}
					if (supplierWidth > swidth) {
	//					System.out.println("ClientSupplierSet.compact supplierWidth=" + supplierWidth + " width=" + width);
						pe = supplierSet.mostFrequentParent();
						if (pe == null) {
							// Cant do any more
							supplierWidth = swidth;
						} else {
							seen = true;
							// Eliminate the descendants of pe
/*
							System.out.print("ClientSupplierSet.compact() Using supplier " + pe + ". Deleting clients =");
							clientSet.showDescendents(pe);
							System.out.print(" and suppliers =");
							supplierSet.showDescendents(pe);
							System.out.println("");
 */

							supplierSet.elimDescendents(pe);
							pe.orMark(EntityInstance.SUPPLIER_MARK);
							if (clientSet.elimDescendents(pe)) {
								pe.orMark(EntityInstance.CLIENT_MARK);
								clientWidth = -1;
							}
							suppliers.addElement(pe);
							supplierWidth = -1;
				}	}	}

				if (!seen) {
					break;
		}	}	}

		/* Strictly for debugging..
		 * Make sure that our client and supplier marks are right

		for (i = 0; i < debugClients.length; ++i) {
			e = (EntityInstance) debugClients[i];
			if (e.isMarked(EntityInstance.SUPPLIER_MARK)) {
				if (!suppliers.contains(e)) {
					System.out.println("Error: client supplier mark not cleared");
			}	}
			if (e.isMarked(EntityInstance.CLIENT_MARK)) {
				if (!clients.contains(e) && !suppliers.contains(e)) {
					System.out.println("Error: client client mark not cleared");
		}	}	}

		for (i = 0; i < debugSuppliers.length; ++i) {
			e = (EntityInstance) debugSuppliers[i];
			if (e.isMarked(EntityInstance.SUPPLIER_MARK)) {
				if (!suppliers.contains(e)) {
					System.out.println("Error: supplier supplier mark not cleared");
			}	}
			if (e.isMarked(EntityInstance.CLIENT_MARK)) {
				if (!clients.contains(e) && !suppliers.contains(e)) {
					System.out.println("Error: supplier client mark not cleared");
		}	}	}

		*/

	}

	public void order()
	{
		m_displayedSize = m_set.size();
		sort();
		addSet();
	}

	/* 

	// For debugging

	public void paintComponent(Graphics g)
	{
		super.paintComponent(g);
		g.drawRect(0, 0, getWidth()-1, getHeight()-1);

	}
	*/

	// DiagramCoordinates interface

	public int getDiagramX()
	{
		return(getX());
	}

	public int getDiagramY()
	{
		return(getY());
	}
}
