package lsedit;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;

import java.util.Enumeration;
import java.util.Vector;

import javax.swing.ButtonGroup;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.JSlider;
import javax.swing.JViewport;
import javax.swing.event.UndoableEditListener;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.undo.UndoableEdit;

public class Diagram extends TemporalTa /* extends UndoableTa extends Ta extends JPanel */ implements MouseMotionListener, DiagramCoordinates  
{
	class ExitFlag extends JComponent implements MouseListener
	{
		protected final static int EXIT_FLAG_DIM = 8;
		protected int	m_priorcursor = -1;

		public ExitFlag()
		{
			super();
			addMouseListener(this);
		}

		public void setLocation(int x, int y)
		{
			setBounds(x, y, EXIT_FLAG_DIM, EXIT_FLAG_DIM);
		}

		public void activate()
		{
			setVisible(true);
		}

		public void paintComponent(Graphics g)
		{
//			System.out.println("Diagram.ExitFlag.paintComponent()");

			super.paintComponent(g);

			/*  Draw a small mark as shown in top left of object

				x---
				|
				|   --------
				|  |        |
				   |        |
				   |        |
	  			   |  ----  |
				   |        |
				   |        |
				   |        |
				   |________|
			*/

			g.setColor(boxColor.darker());
			g.drawRect(0, 0, EXIT_FLAG_DIM, EXIT_FLAG_DIM);
			// g.setColor(Color.black);
			g.drawLine(2, 4, 6, 4);
//			System.out.println("Diagram.paintComponent() done");
		}

		// MouseListener interface

		public void mouseClicked(MouseEvent ev)	
		{
			if (m_priorcursor != -1) {
				getLs().setCursor(m_priorcursor); 
				m_priorcursor = -1;
			}
			navigateToDrawRootParent();
		}

		public void mouseEntered(MouseEvent e)
		{
			m_priorcursor = getLs().setCursor(Cursor.HAND_CURSOR);
//			System.out.println("m_priorcursor=" + m_priorcursor);
		}

		public void mouseExited(MouseEvent e)
		{
			if (m_priorcursor != -1) {
//				System.out.println("MouseExited " + m_priorcursor);
				getLs().setCursor(m_priorcursor); 
				m_priorcursor = -1;
		}	}

		public void mousePressed(MouseEvent ev)
		{
		}

		public void mouseReleased(MouseEvent ev)
		{
		}
	}

	// Final values

	// Options on the root entity specific to Diagram
	// Non-first order attributes of $ROOT

	public final static String RELN_HIDDEN_ID			= "reln_hidden";
	public final static String EDGEMODE_ID				= "edgemode"; 
	public final static String TOPCLIENTS_ID			= "topclients"; 
	public final static String WANTCLIENTS_ID			= "wantclients"; 
	public final static String WANTSUPPLIERS_ID			= "wantsuppliers"; 
	public final static String WANTCARDINALS_ID			= "wantcardinals"; 
	public final static String WANTOUTCARDINALS_ID      = "wantoutcardinals";


	public final static int BG = 191;				// Was 0.75 -> 255 * 0.75 = 191.25
	public final static Color boxColor        = Color.lightGray;
	public final static Color lighterBoxColor = new Color(0xe0e0e0);

	protected final static int GAP = 5;
	protected final static int MARGIN = GAP*2;

	protected final static double SMALL_SCALE_UP		= 1.2; 
	protected final static double SMALL_SCALE_DOWN		= 0.8; 
	protected final static String SMALL_SCALE_STRING	= "20%";

	// Values

	protected LandscapeEditorCore m_ls;

	protected boolean	m_visited = false;
	protected double	m_zoom_x = 1.0;		//	The zoom factor
	protected double	m_zoom_y = 1.0;

	protected EntityInstance	m_drawRoot;					// The current entity being drawn
	
	/* By putting the edges in a separate container we simplify the management of painting
		(1) We ensure that edges are painted after entities. This is because the m_edges
			components is added first.
		(2) We ensure that adding and deleting edges is efficient.
			This is because we can add at end.
	 */

	protected Container			m_edges     = null;

	/* Likewise for cardinals */

	protected Container			m_cardinals = null;

	protected int				numVisibleRelationClasses = 0;

	protected boolean			persistentQuery = false;
	protected boolean			m_loaded        = false;

	protected int				mode; 

	protected static int		m_gridPixels = 1;
	protected static Color		m_gridColor  = Color.WHITE;

	protected int				m_preserve_entity_marks   = 0;
	protected int				m_preserve_relation_marks = 0;

	protected Attribute			m_relVisibilityAttr; 

	protected boolean			m_visibleEdges = true;
	protected boolean			m_drawEdges = true;

	protected int				newCnt = 0;					// Used to id new entities	
	protected boolean			undoValid = false;

	protected EntityInstance	m_keyEntity;				// The currently highlight entity if any else null

	protected ExitFlag			m_exitFlag;
	protected SupplierSet		m_supplierSet;
	protected ClientSet			m_clientSet;	

	protected Vector			m_suppliers;
	protected Vector			m_dClients;
	protected Vector			m_dSuppliers;

	protected boolean			m_suppliersSorted = false;

	protected boolean			allClients = true;
	protected boolean			allSuppliers = true;

	protected boolean			m_modeHandlingActive = false;

	protected Clipboard			m_clipboard = null;

	protected Vector			m_oldChildren;


	// -----------------
	// Protected methods
	// -----------------

	protected Diagram getDiagram()
	{
		// Hide the TA layer from having to know what the diagram is
		// While still being able to pass it into its own functions
		return(this);
	}

	protected void findClientsAndSuppliers(boolean liftEdges) 
	{
		EntityInstance	drawRoot = m_drawRoot;
		boolean			showSuppliers;
		boolean			showClients;

		if (drawRoot == m_rootInstance) {
			showSuppliers = false;
			showClients   = false;
		} else {
			showSuppliers = m_ls.isShowSuppliers();
			showClients   = m_ls.isShowClients();
		}
		m_clientSet.removeAll();
		if (showClients) {
			m_clientSet.findClients(m_rootInstance, drawRoot, liftEdges);
		} 
		m_supplierSet.removeAll();
		if (showSuppliers) {
			m_supplierSet.findSuppliers(m_rootInstance, drawRoot, liftEdges);
		}
	}
		
	protected void setVisibilityFlags() 
	{
		Enumeration en;

		numVisibleRelationClasses = 0;

		for (en = enumRelationClasses(); en.hasMoreElements(); ) {
			RelationClass rc = (RelationClass) en.nextElement();
			if (!rc.isContainsClass()) {
				if (rc.isClassVisible()) {
					rc.setOrdinal(numVisibleRelationClasses++);
				}
			}
		}
		m_visibleEdges = (numVisibleRelationClasses > 0);
	}

/* 
	What a cautionary tale this is for a software engineering course..

	1. Somewhere in Java AWT's history someone had the bright idea of changing
	   the AWT framework so that every component could have a mouse cursor
	   associated with it instead of just the frame managing what the current
	   mouse cursor object should be at any given time.  The idea was that as
	   you moved the mouse over different objects each object would have the
	   capability of explicitly indicating what the cursor associated with the
	   mouse should be, while it hovered over that object.

	2. This means that when ever a new visible object is added to a container,
	   AWT must handle the possibility that the cursor happens to be over this
	   new object, in which case this new object should if appropriate change
	   the appearance of that mouse cursor immediately.

	3. So whenever an object is added to the AWT containment hierarchy, there
	   is an invocation to that object to change the system cursor immediately.

	   [This by the way is a less than smart implementation.. The smart thing
		to tell the component is change the cursor if the mouse pointer is over
		it and it actually cared what the cursor  should be in that unusual case]

	4. The object handles this request by walking all the way up the containment
	   heirarchy until it hits a component which actually cares what the cursor
	   looks like (in lsedit's case the topmost frame object, since tha ability
	   to associate cursors directly with objects is not used).

	5. This cursor isn't changed but the frame object doesn't know that. That
	   is because any component can over ride the frame's opinion as to what
	   the cursor should be (not that any does in lsedit).

	6. So the frame object starts off by asking itself what the mouse pointer
	   is over. To do this it walks back down through the containment heirarchy
	   asking each level if it contains the current point the mouse pointer is
	   over, recursively.  This comes close to search all components, if the
	   mouse pointer happens to be over the diagram, since entities and edges
	   dominate the number of components.

	7. Now containment is not a trivial exercise for edges.  At a minimum we
	   need to decide if the given point is within 'n' pixels of any part of
	   the line, the arrow head, etc.

	8. Having identified the deepest thing the mouse pointer is over, we walk up
	   finding the nearest component above which wishes to control the cursor
	   image, arriving back at the frame.  The frame then dutifully sets the
	   mouse cursor to what it already was.


	Now this is repeated every time we add a component.  So it is clearly an
	O(N^2) operation for N components.  So when displaying a large number of
	edges (order the number in linux or whatever) the cost of JAVA painting
	the cursor image associated with the mouse pointer, becomes prohibitive
	as the actual graph image is constructed line by line.

	I worked this out by narrowing down the performance problems when displaying
	a large ta file to: White cursor mouse arrow left on screen until lsedit
	draws = 40 seconds, while time taken was a mere 4 seconds if I cleverly fired
	up lsedit in a manner which left the cursor off the lsedit screen, somewhere
	down at the bottom of my actual screen.

	Needless to say I at first found this behaviour quite astonishing.

	Work around to correct this <<bug>> was to add to the diagram.java class
	the following special rule about what the diagram contains when not
	loaded -- nothing.

	Ian Davis
*/

	public boolean contains(int x, int y)
	{
		return (m_loaded && super.contains(x, y));
	}

	public void setPreserveEntityMarks(int value)
	{
//		System.out.println("Set preserve entity marks " + value);
		m_preserve_entity_marks = value;
	}

	public void setPreserveRelationMarks(int value)
	{
		m_preserve_relation_marks = value;
	}

	// For debugging
	
	
    protected void paintChildren(Graphics g) 
    {
    	super.paintChildren(g);
    }

	// For debugging
	    	
    public void repaint() 
    {
    	super.repaint();
    }
    
	// --------------
	// Public methods
	// --------------

	// This constructor is only called to create copies of diagrams
	// They are never active so don't need to assign them listeners

	public Component add(Component c)
	{
		if (c instanceof RelationComponent) {
			return(m_edges.add(c));
		}
		if (c instanceof Cardinal) {
			return(m_cardinals.add(c));
		}
		return(super.add(c));
	}

	public void remove(Component c)
	{
		if (c instanceof RelationComponent) {
			m_edges.remove(c);
			return;
		}
		if (c instanceof Cardinal) {
			m_cardinals.remove(c);
			return;
		}
		super.remove(c);
	}

	public Diagram(LandscapeEditorCore ls) 
	{
		super(ls, ls);
		m_ls = ls;
		setDiagram(this);

		setLayout(null);
		setLocation(0, 0);

		m_exitFlag = new ExitFlag();
		m_exitFlag.setLocation(3, 3);

		m_supplierSet   = new SupplierSet(this);
		m_clientSet     = new ClientSet(this);	// Ignore suppliers when computing clients

		addMouseMotionListener(this);
		// The default empty instance

		// No marks are needed when the diagram is first drawn
		setPreserveEntityMarks(0);
		setPreserveRelationMarks(0);
	}

	public LandscapeEditorCore getLs() 
	{
		return m_ls;
	}

	public void entityCut(EntityInstance e)
	{
		EntityInstance parent    = e.getContainedBy();
		EntityInstance keyEntity = m_diagram.getKeyEntity();

		if (e.hasDescendantOrSelf(getDrawRoot())) {
			// Don't allow us to remain under something cut
			navigateTo(parent);
		}

		if (keyEntity != null) {
			if (e.hasDescendantOrSelf(keyEntity)) {
				clearKeyEntity();
		}	}
	}

	public void containerCut(EntityInstance parent, EntityInstance e)
	{
		if (getKeyEntity() == e) {
			clearKeyEntity();
		}

		if (e == getDrawRoot()) {
			navigateTo(parent);
		}
		redrawDiagram();
	}

	public void containerUncut(EntityInstance e)
	{
		redrawDiagram();
	}

	// Programatically delete marked things

	public void doDelete(Object object) 
	{
		Vector				objects;
		Object				o;
		Enumeration			en;
		RelationInstance	ri;
		EntityInstance		e;
		int					mode = 0;
		String				msg;


		objects = targetEntityRelations(object);
		if (objects != null) {
			beginUndoRedo("Delete");
			for (en = objects.elements(); en.hasMoreElements(); ) {
				o = en.nextElement();
				if (o != null) {
					if (o instanceof RelationInstance) {
						ri = (RelationInstance) o;
						mode |= 1;
						ri.updateDeleteEdge();
						continue;
					}
					if (o instanceof EntityInstance) {
						e = (EntityInstance) o;
						if (e.getContainedBy() == null) {
							continue;
						}
						updateCutEntity(e);
						mode |= 2;
						continue;
					}
					System.out.println("Can't delete object of type " + o.getClass());
			}	}
			endUndoRedo();
			if (mode != 0) {
				if ((mode & 2) != 0) {
					clearGroupFlags();
				}
				redrawDiagram();
		}	}

		switch (mode) {
		case 0:	
			msg = "Nothing selected to delete";
			break;
		case 1:
			msg = "Edge(s) deleted";
			break;
		case 2:
			msg = "Entity(s) deleted";
			break;
		case 3:
			m_diagram.clearGroupFlags();
			msg = "Entity(s) and Edge(s) deleted";
			break;
		default:
			msg = "???";
		}
		m_ls.doFeedback(msg);
	}

	public boolean updateContainsRelation(RelationClass relationClass)
	{
		Vector newForest = establishForest(relationClass);
		if (newForest != null) {
			updateSwitchContainsClass(relationClass, newForest); 
			redrawDiagram();
			return true;
		}
		return false;
	}

	// This logic attempts to assign a Rectangle to any component which lacks Rectangle information

	// Called from landscapeEditorCore.attach()
	// Called from landscapeEditorCore.loadLs()
	// Called from landscapeEditorCore.initialLoad()

	// Returns non-null string on error

	public String loadDiagram(String taPath, Object context) 
	{

		Attribute		attr;
		boolean			option;
		String			ret;
		EntityInstance	optionsInstance;
		int				value;

		ret = loadTA(taPath, context);
		if (ret != null) {
			return ret;
		}
		navigateToRoot();

		m_ls.addLseditHistory(taPath);

		optionsInstance = m_rootInstance;
		if (optionsInstance != null) {
			value = Do.EDGE_STATE_DEFAULT;
			attr  = optionsInstance.getLsAttribute(EDGEMODE_ID);
			if (attr != null && attr.avi != null && attr.avi.value != null) {
				value = attr.parseInt();
			}
			m_ls.setEdgeMode(value);

			option = Do.TOP_CLIENTS_DEFAULT;
			attr = optionsInstance.getLsAttribute(TOPCLIENTS_ID);
			if (attr != null && attr.avi == null && attr.avi.value != null) {
				option   = attr.parseBoolean();
			}
			m_ls.setTopClients(option);

			option = Do.SHOW_CLIENTS_DEFAULT;
			attr = optionsInstance.getLsAttribute(WANTCLIENTS_ID);
			if (attr != null && attr.avi == null && attr.avi.value != null) {
				option   = attr.parseBoolean();
			}
			m_ls.setShowClients(option);

			option = Do.SHOW_SUPPLIERS_DEFAULT;
			attr = optionsInstance.getLsAttribute(WANTSUPPLIERS_ID);
			if (attr != null && attr.avi == null && attr.avi.value != null) {
				option   = attr.parseBoolean();
			}
			m_ls.setShowSuppliers(option);

			option = Do.SHOW_SUPPLIERS_DEFAULT;
			attr = optionsInstance.getLsAttribute(WANTSUPPLIERS_ID);
			if (attr != null && attr.avi == null && attr.avi.value != null) {
				option   = attr.parseBoolean();
			}
			m_ls.setShowSuppliers(option);

			option = Do.SHOW_DST_CARDINALS_DEFAULT;
			attr = optionsInstance.getLsAttribute(WANTCARDINALS_ID);
			if (attr != null && attr.avi == null && attr.avi.value != null) {
				option   = attr.parseBoolean();
			}
			m_ls.setShowDstCardinals(option);

			option = Do.SHOW_SRC_CARDINALS_DEFAULT;
			attr = optionsInstance.getLsAttribute(WANTOUTCARDINALS_ID);
			if (attr != null && attr.avi == null && attr.avi.value != null) {
				option   = attr.parseBoolean();
			}
			m_ls.setShowSrcCardinals(option);

			m_relVisibilityAttr = attr = optionsInstance.getLsAttribute(RELN_HIDDEN_ID);
			if (attr != null) {
				AttributeValueItem avi = attr.avi;

				for (; avi != null; avi = avi.next) {
					if (avi.value != null) {
						RelationClass rc = getRelationClass(avi.value);

						if (rc != null) {
							rc.setClassVisible(false); 
				}	}	}
			}	
		}
		setVisibilityFlags();

		return ret;
	}

	public void cutGroup(Object object) 
	{
		Vector			vector;
		Clipboard		old_clipboard, new_clipboard;
		String			msg;
		boolean			addPriorCuts = m_ls.isAddToClipboard();

		TextTree	tocBox = m_ls.getTocBox();
		
		if (object != null && object == tocBox) {
			vector = tocBox.getGroup();
		} else {
			vector = getGroup();
		}

		if (vector == null || vector.size() == 0) {
			m_ls.error("Group not selected");
			return;
		}

		if (vector.contains(m_rootInstance)) {
			m_ls.error("Can't cut the root node in the diagram");
		}

		// Don't want a subsequent paste to try and paste X into X
		clearKeyEntity();

		new_clipboard = new Clipboard(vector);

		msg = "Group copied to clipboard";
		
		old_clipboard = getClipboard();
		if (old_clipboard != null && old_clipboard.size() != 0) {
			if (addPriorCuts) {
				new_clipboard.setExtendsClipboard(old_clipboard);
				msg += " - old cuts preserved";
			} else {
				msg += " - old cuts discarded";
		}	}
		updateCutClipboard(old_clipboard, new_clipboard);
		m_ls.doFeedback(msg);
	}
	
	public void pasteGroup(Object object) 
	{
		Clipboard clipboard = getClipboard();

		if (clipboard == null) {
			m_ls.error("Clipboard empty");
			return;
		}

		TextTree		tocBox = m_ls.getTocBox();
		EntityInstance	pe;

		if (object != null && object == tocBox) {
			pe = tocBox.targetEntity();
		} else {
			pe = targetEntity(object);
		}
		if (pe != null) {
			updatePasteClipboard(clipboard, pe);
			m_ls.doFeedback("Pasted entities into " + pe.getEntityLabel());
	}	}	

	public boolean isUndoAvailable() 
	{
		return undoValid;
	}

	/* Some relative positioning/sizing information has changed
	 * Recompute the absolute positions and sizes of all visible entities 
	 * in the diagram
	 */

	public void rescaleDiagram() 
	{
//		System.out.println("Diagram.rescaleDiagram");
		// Resize first so that sizes known when calling postprocess() during relocation
		m_drawRoot.resizeChildren();
		m_drawRoot.relocateChildren();
	}

	public EntityInstance containing(Rectangle lyt) 
	{
		return m_drawRoot.containing(lyt);
	}

	public Vector setGroupRegion(Rectangle lyt) 
	{ 
		EntityInstance e = m_drawRoot.getMouseOver((int) lyt.x, (int) lyt.y);

		if (e == null) {
			return null;
		}
		return e.groupRegion(lyt);
	}

	public void setMode(int mode) 
	{
		this.mode = mode; 
	}

	public void setEdgeMode(int mode) 
	{
		Attribute			attr = m_rootInstance.getLsAttribute(EDGEMODE_ID);
		AttributeValueItem	avi  = new AttributeValueItem(String.valueOf(mode));
		if (attr == null) {
			attr = new Attribute(EDGEMODE_ID, avi); 
			m_rootInstance.putLsAttribute(attr);
		} else {
			attr.avi = avi;
	}	}

	public void setOption(String id, boolean value)
	{
		Attribute			attr = m_rootInstance.getLsAttribute(id);
		AttributeValueItem	avi  = new AttributeValueItem(value);
		if (attr == null) {
			attr = new Attribute(id, avi); 
			m_rootInstance.putLsAttribute(attr);
		} else {
			attr.avi = avi;
	}	}

	public void setTopClients(boolean value)
	{
		setOption(TOPCLIENTS_ID, value);
	}

	public void setShowClients(boolean value)
	{
		setOption(WANTCLIENTS_ID, value);
	}

	public void setShowSuppliers(boolean value)
	{
		setOption(WANTSUPPLIERS_ID, value);
	}

	public void setShowDstCardinals(boolean value)
	{
		setOption(WANTCARDINALS_ID, value);
	}

	public void setShowSrcCardinals(boolean value)
	{
		setOption(WANTOUTCARDINALS_ID, value);
	}

	public int getEdgeMode() 
	{
		return m_ls.getEdgeMode();
	}

	public EntityInstance getDrawRoot() 
	{
		return m_drawRoot; 
	}

	public void setDrawRoot(EntityInstance e)
	{
		EntityInstance old = m_drawRoot;

		if (e != old) {
			m_drawRoot = e;
			if (old != null) {
				old.setToolTipText();
			}
			if (e != null) {
				MapBox		mapBox;
				HistoryBox	historyBox;

				e.setToolTipText();
				e.setOpen();

				mapBox = m_ls.m_mapBox;
				if (mapBox != null) {
					mapBox.fill(e);
//					System.out.println("Diagram.setDrawRoot() map box added");
				}
				m_ls.addHistoryEntity(e);
		}	}
	}

	public void toggleDstElision(EntityInstance e) 
	{
		Enumeration en;
		RelationClass rc;

		for (en = enumRelationClasses(); en.hasMoreElements(); ) {
			rc = (RelationClass) en.nextElement(); 
			if (!rc.isContainsClass() && rc.isClassVisible()) {
				e.toggleDstElision(rc); 
			}
		}
	}

	public void toggleSrcElision(EntityInstance e) 
	{
		Enumeration en;
		RelationClass rc;

		for (en = enumRelationClasses(); en.hasMoreElements(); ) {
			rc = (RelationClass) en.nextElement();
			if (!rc.isContainsClass() && rc.isClassVisible()) {
				e.toggleSrcElision(rc);
		}	}
	}

	public void toggleEnteringElision(EntityInstance e) 
	{

		Enumeration en;
		RelationClass rc;

		for (en = enumRelationClasses(); en.hasMoreElements(); ) {
			rc = (RelationClass) en.nextElement(); 
			if (!rc.isContainsClass() && rc.isClassVisible()) {
				e.toggleEnteringElision(rc); 
		}	}
	}

	public void toggleExitingElision(EntityInstance e) 
	{
		Enumeration en;
		RelationClass rc;

		for (en = enumRelationClasses(); en.hasMoreElements();) {
			rc = (RelationClass) en.nextElement();
			if (!rc.isContainsClass() && rc.isClassVisible()) {
				e.toggleExitingElision(rc);
		}	}
	}

	public void toggleInternalElision(EntityInstance e) 
	{
		Enumeration en;
		RelationClass rc;

		for (en = enumRelationClasses(); en.hasMoreElements(); ) {
			rc = (RelationClass) en.nextElement();
			if (!rc.isContainsClass() && rc.isClassVisible())
				e.toggleInternalElision(rc);
		}
	}

	public void setVisible(boolean value)
	{
		if (m_drawRoot != null) {
			m_drawRoot.setVisible(value);
	}	}
	
	public void setEdgeVisibilityState(RelationClass rc, boolean state) 
	{
		rc.setClassVisible(state);

		// List is HIDDEN relations

		if (state) {
			if (m_relVisibilityAttr != null) {
				m_relVisibilityAttr.removeFromList(rc.getId());
			}
		} else {
			if (m_relVisibilityAttr == null) {
				m_rootInstance.addAttribute( new Attribute(RELN_HIDDEN_ID, null));
				m_relVisibilityAttr = m_rootInstance.getLsAttribute(RELN_HIDDEN_ID);
			}
			m_relVisibilityAttr.addToList(rc.getId());
		}
		setVisibilityFlags();
	}

	public void toggleEdgeVisibility() 
	{
		Enumeration en;
		RelationClass rc;

		for (en = enumRelationClasses(); en.hasMoreElements();) {
			rc = (RelationClass) en.nextElement();
			if (!rc.isContainsClass() && rc.isClassVisible()) {
				setEdgeVisibilityState(rc, false);
			}
		}
	}

	// Only draw the edges found

	public void clearDrawEdges() 
	{
//		System.out.println("clearDrawEdges");
		m_drawEdges = false;
//		java.lang.Thread.dumpStack();
//		System.out.println("-----");
	}

	// Draw all edges

	public void setDrawEdges() 
	{
//		System.out.println("setDrawEdges");
		m_drawEdges = true;
//		java.lang.Thread.dumpStack();
//		System.out.println("-----");
	}

	public boolean getDrawEdges()
	{
		return(m_drawEdges);
	}

	protected void getHighlightedEdges(EntityInstance me, EntityInstance drawRoot, Vector v) 
	{ 
		Enumeration en;
		RelationInstance ri;
		EntityInstance src, e;


		for (en = me.srcRelationElements(); en.hasMoreElements();) {
			ri = (RelationInstance) en.nextElement();
			if (ri.getHighlightFlag() && ri.getRelationClass().isActive()) {
				v.addElement(ri);
			}
		}

		for (en = me.dstRelationElements(); en.hasMoreElements(); ) {
			ri  = (RelationInstance) en.nextElement();
			src = ri.getSrc();

			// If has descendant already added if going to
			if (!drawRoot.hasDescendant(src) && ri.getHighlightFlag() && ri.getRelationClass().isActive()) {
				v.addElement(ri);
			}
		}

		// Call method on each child

		for (en = me.getChildren(); en.hasMoreElements();) {
			e = (EntityInstance) en.nextElement();
			getHighlightedEdges(e, drawRoot, v);
		}
	}

	public Vector getHighlightedEdges() 
	{
		Vector			v        = new Vector();
		EntityInstance	drawRoot = m_drawRoot;

		getHighlightedEdges(drawRoot, drawRoot, v);
		if (v.size() == 0) {
			return null;
		}
		return v;
	}

	public boolean clearFlags() 
	{
		boolean ret;

		clearKeyEntity();

		ret = false;
		if (!getDrawEdges()) {
			ret = true;
			setDrawEdges();
		}
		m_rootInstance.clearAllFlags();
		if (m_drawRoot != null) {
			m_drawRoot.setOpen();
		}
		return(ret);
	}

	// Remove all RedBox and HighLight flags under the root

	public void clearQueryFlags() 
	{
		m_rootInstance.clearQueryFlags();
	}

	public void clearGroupFlags() 
	{
		clearKeyEntity();
		setDrawEdges();
		m_rootInstance.clearGroupFlags();
	}

	public boolean toggleQueryPersistence() 
	{
		persistentQuery = !persistentQuery;
		return persistentQuery;
	}

	public boolean getQueryPersistance() 
	{
		return persistentQuery;
	}

	public int numVisibleRelationClasses() 
	{
		return this.numVisibleRelationClasses;
	}

	public Vector getGroup() 
	{
		Vector grp = new Vector();

		m_rootInstance.getGroup(grp);
		if (grp.isEmpty()) {
			return null;
		}
		return grp;
	}

	public Rectangle getGroupBoundingBox() 
	{
		Vector grp = getGroup();

		if (grp == null || grp.isEmpty()) {
			return null;
		}

		// Determine the bounding box of whole group

		Enumeration en;
		double x1, y1, x2, y2;


		x1 = Double.MAX_VALUE;
		y1 = Double.MAX_VALUE;
		x2 = Double.MIN_VALUE;
		y2 = Double.MIN_VALUE;

		for (en = grp.elements(); en.hasMoreElements(); ) {
			EntityInstance e = (EntityInstance) en.nextElement();
			Rectangle lyt = e.getDiagramBounds();

			if (lyt.x < x1) 
				x1 = lyt.x;

			if (lyt.y < y1)
				y1 = lyt.y;

			if (lyt.x + lyt.width > x2)
				x2 = lyt.x + lyt.width;

			if (lyt.y + lyt.height > y2)
				y2 = lyt.y + lyt.height;
		}
		return new Rectangle((int) x1, (int) y1, (int) (x2-x1), (int) (y2-y1));
	}

	public EntityInstance getKeyEntity() 
	{
		return m_keyEntity;
	}

	public void clearKeyEntity()
	{
		if (m_keyEntity != null) {
			m_keyEntity.clearGroupKeyFlag();		// Must do this first
			m_keyEntity.clearGroupFlag();
			m_keyEntity = null;
	}	}

	public void setKeyEntity(EntityInstance e) {

		if (m_keyEntity != null) {
			m_keyEntity.clearGroupKeyFlag();		// Must do this first
			m_keyEntity.setGroupFlag();
		}
		m_keyEntity = e;
		m_keyEntity.setGroupKeyFlag();
		m_keyEntity.setGroupFlag();
	}

	public Vector getRedBoxGroup() 
	{
		Vector grp = new Vector();

		m_rootInstance.getRedBoxGroup(grp);
		if (grp.isEmpty()) {
			return null;
		}
		return grp;
	}

	public boolean isOverGroupRegion(int x, int y) 
	{
		Rectangle lyt = getGroupBoundingBox();

		if (lyt != null) {
			return lyt.contains(x, y);
		}
		return false;
	}

	public String show_groupList(ResultBox resultBox) 
	{
		Vector grp      = getGroup();
		String msg      = null;

		if (grp != null) {
			SortVector.byString(grp);
			resultBox.showResults("GROUP:", grp, "-- End of group --");
		} else {
			msg = "No entities selected";
		}
		return(msg);
	}

	public String groupAll(ResultBox resultBox) 
	{
		boolean ret;

		EntityInstance	ke      = getKeyEntity();
		EntityInstance	ge      = ke;
		Enumeration		en;
		EntityInstance	ce;
		String			msg     = null;

		if (ke == null) {
			ke = getDrawRoot();
			ge = ke;
		}

		ret = clearFlags();

		if (!ge.isOpen()) {
			ge = ge.getContainedBy();
		}

		for (en = ge.getChildren(); en.hasMoreElements(); ) {
			ce = (EntityInstance) en.nextElement();
			ce.setGroupFlag();
		}

		Vector grp = getGroup();

		if (grp != null) {
			if (!grp.contains(ke))
				ke = (EntityInstance) grp.firstElement();

			setKeyEntity(ke);
			setPreserveEntityMarks(EntityInstance.GROUP_MARK|EntityInstance.GROUPKEY_MARK);
			setPreserveRelationMarks(0);

			if (ret) {
				redrawDiagram();
			}
			msg = show_groupList(resultBox);
		} else {
			msg = "No entities selected";
		}
		return(msg);
	}

	public static int getGrid() 
	{
		return m_gridPixels;
	}

	public static void setGrid(int val) 
	{
		m_gridPixels = val;
	}

	public static Color getGridColor()
	{
		return m_gridColor;
	}

	public static void setGridColor(Color color)
	{
		m_gridColor = color;
	}

	class SetNewGridDialog extends JDialog implements ChangeListener, ActionListener
	{
		class GridImage extends JComponent
		{
			public GridImage()
			{
				super();
				setPreferredSize(new Dimension(400, 100));
			}

			public void paintComponent(Graphics g)
			{
				int	grid = getGrid();
				int	w    = getWidth();
				int h    = getHeight();
				int	i;

				g.drawRect(0,0, w-1, h-1);

				if (grid > 1) {
					Color color   = m_gridColor;
					Color inverse = ColorCache.getInverse(color.getRGB());

					g.setColor(inverse);
					g.fillRect(1,1,w-2,w-2);

					g.setColor(color);
					for (i = grid; i < h; i += grid) {
						g.drawLine(0,i,w-1,i);
					}
					for (i = grid; i < w; i += grid) {
						g.drawLine(i,0,i, h-1);
			}	}	}
		}

		JFrame		m_frame;
		JSlider		m_gridSlider;
		GridImage	m_gridImage;
		JLabel		m_feedback;
		JButton		m_color;


		public SetNewGridDialog(JFrame frame)
		{
			super(frame, "Set grid size", true);

			Container	contentPane = getContentPane();
			Font		font, bold;

			m_frame      = frame;
			font         = FontCache.getDialogFont();
			bold         = font.deriveFont(Font.BOLD);

			JLabel sliderLabel = new JLabel("Grid size when moving entities", JLabel.CENTER);
			sliderLabel.setAlignmentX(Component.CENTER_ALIGNMENT);
			sliderLabel.setFont(bold);

			JLabel feedback  = new JLabel("Current grid size " + getGrid(), JLabel.CENTER);
			feedback.setAlignmentX(Component.CENTER_ALIGNMENT);
			feedback.setFont(font);
			feedback.setForeground(Color.BLUE);
			m_feedback = feedback;

			m_gridImage = new GridImage();

			JSlider gridSlider = new JSlider(JSlider.HORIZONTAL, 0, 100, m_gridPixels);

			m_gridSlider = gridSlider;

			gridSlider.addChangeListener(this);

			gridSlider.setMajorTickSpacing(10);
			gridSlider.setMinorTickSpacing(2);
			gridSlider.setPaintTicks(true);
			gridSlider.setPaintLabels(true);
			gridSlider.setFont(font);
			gridSlider.setPreferredSize(new Dimension(400, 50));

			gridSlider.setBorder(BorderFactory.createEmptyBorder(0,0,10,0));

			JPanel	panel = new JPanel();
			panel.setLayout(new BorderLayout());
			panel.add(feedback, BorderLayout.NORTH);
			panel.add(m_gridImage,BorderLayout.CENTER);

			m_color = new JButton("Colour");
			m_color.addActionListener(this);
			panel.add(m_color, BorderLayout.SOUTH);

			contentPane.add(sliderLabel, BorderLayout.NORTH);
			contentPane.add(gridSlider,  BorderLayout.CENTER);
			contentPane.add(panel,       BorderLayout.SOUTH);

			if (frame != null) {
				setLocation(frame.getX()+200, frame.getY()+300);
			}
			pack();
			setVisible(true);
		}

		public void finished()
		{
			m_gridSlider = null;
			m_gridImage  = null;
			m_feedback   = null;
		}

		// ActionListener interface

		public void actionPerformed(ActionEvent ev)
		{
			Object source = ev.getSource();

			if (source == m_color) {
				ColorChooser colorChooser = new ColorChooser(m_frame, "Pick a grid color", m_gridColor, true /* include alpha */, false /* Don't allow null */);
				m_gridColor = colorChooser.getColor();
				colorChooser.dispose();
				m_feedback.setText("Color changed");
				m_gridImage.repaint();
				if (m_ls.isShowGrid()) {
					Diagram.this.repaint();
			}	}
		}

		public void stateChanged(ChangeEvent ev)
		{
			Object source = ev.getSource();

			if (source == m_gridSlider) {
				int value = (int) m_gridSlider.getValue();

				if (value < 1) {
					value = 1;
				}
				setGrid(value);
				m_feedback.setText("Grid size set to " + value);
				m_gridImage.repaint();
			}
		}
	}

	public int setNewGridValue()
	{
		int					oldgrid = getGrid();
		SetNewGridDialog	dialog  = new SetNewGridDialog(m_ls.getFrame());
		int					newgrid;

		dialog.finished();
		dialog.dispose();
		dialog = null;
		newgrid = getGrid();
		if (oldgrid != newgrid) {
			if (m_ls.isShowGrid()) {
				repaint();
		}	}
		return newgrid;
	}

	public void setToViewport() 
	{
		navigateToRoot();
	}

	// Mark all edges as needing to be repainted

	public void invalidateAllEdges(EntityInstance root)
	{
		Enumeration en;

		/* Draw all the visible objects */

		root.invalidateAllEdges(); 

		if (root.isOpen()) {
			for (en = root.getChildren(); en.hasMoreElements(); ) {
				invalidateAllEdges((EntityInstance) en.nextElement());
		}	}
	}

	// Draw all edges marked as needing to be repainted at or under e

	protected void drawHighlightEdges(EntityInstance e)
	{
		Enumeration en;

		/* Draw all the highlight edges */

		e.drawHighlightedEdges();
		if (e.isOpen()) {
			for (en = e.getChildren(); en.hasMoreElements(); ) {
				drawHighlightEdges((EntityInstance) en.nextElement());
		}	}
	}
	// Draw all edges marked as needing to be repainted at or under e

	protected void drawAllEdges(EntityInstance e, EntityInstance under)
	{
		Enumeration en;

		/* Draw all the visible edges */

		e.drawAllEdges(under); 
		if (e.isOpen()) {
			for (en = e.getChildren(); en.hasMoreElements(); ) {
				drawAllEdges((EntityInstance) en.nextElement(), under);
		}	}
	}

	protected void doDrawEdges(EntityInstance e, EntityInstance under)
	{
		/* Draw all the visible objects */

		if (getDrawEdges()) {
			if (m_visibleEdges) {
				drawAllEdges(e, under); 
			}
		} else {
			drawHighlightEdges(e);
	}	}

	/* If this isn't changed to return true the revalidate() becomes a
	   request to revalidate the JScrollPane containing us and that for
	   some reason doesn't attempt to validate() the thing it contains.
	 */

	public boolean isValidateRoot()
	{
		return(true);
	}

	public Dimension getPreferredSize()
	{
		Dimension	preferredSize;
		JComponent	container;
		Insets		insets;
		int			mw, mh;

		container = (JComponent) getParent();
		if (container == null) {
			mw = 0;
			mh = 0;
		} else {
			insets = container.getInsets();

			mw = container.getWidth()-insets.right-insets.left;
			if (mw <= 0) {
				mw = 0;
			}
			mh = container.getHeight()-insets.bottom-insets.top;
			if (mh <= 0) {
				mw = 0;
			}
			mw = (int) (((double) mw) * m_zoom_x);
			mh = (int) (((double) mh) * m_zoom_y);
		}
		preferredSize = super.getPreferredSize();
		if (preferredSize.width != mw || preferredSize.height != mh) {
			preferredSize.setSize(mw, mh);
			setPreferredSize(preferredSize);
			container.revalidate();
		}
		return(preferredSize);
	}

	public void closeEntities()
	{
		String			containsId = getContainsId();
		EntityInstance	e;
		Enumeration		en;
			
		for (e = m_entityCache.getFirst(); e != null; e = m_entityCache.getNext()) {
			if (e.isDstRelationElided(containsId)) {
				return;
		}	}
		for (e = m_entityCache.getFirst(); e != null; e = m_entityCache.getNext()) {
			if (e == m_drawRoot || e == m_rootInstance) {
				continue;
			}
			e.toggleContainElision(containsId);
		}

		// Since the root is a forest always open the roots children
		// These are the real highest level nodes
		for (en = m_rootInstance.getChildren(); en.hasMoreElements(); ) {
			e = (EntityInstance) en.nextElement();
			if (e == m_drawRoot) {
				continue;
			}
			e.toggleContainElision(containsId);
		}
	}

/* All complex stuff is delayed until we validate to ensure that we have a
   graphic object so that we can compute sizes of labels etc. Much of this
   stuff doesn't really belong in validate -- which is supposed to be called
   only whenever the diagram changes its shape.

	synchronized(getTreeLock())  -- locks against multithreading

 */

	public void validate()
	{
		Dimension		size;
		int				mw, mh, yshift;
		Vector			clients, suppliers;
		Enumeration		en;
		Container		container;
		EntityInstance	e;
		boolean			isTopClients, liftEdges;
		Cardinal		cardinal;
		int				temp, i, cnt;

		m_loaded = false;

		size = getSize();
		mw   = size.width;
		mh   = size.height;

//		System.out.println("Diagram.validate Started validation mw=" + mw + " mh=" + mh);

		if (mw < 1 || mh < 1) {
//			System.out.println("Diagram.validate done mw=" + mw + " mh=" + mh);
			return;
		}


/*
		System.out.println("Diagram.validate() " + getBounds() + ".v." + "[" + m_zoom_x + "," + m_zoom_y + "]");
		java.lang.Thread.dumpStack();
		System.out.println("-----");
*/

		if (m_cardinals == null) {
			m_cardinals = new Container();
			m_cardinals.setLayout(null);
		}
		if (m_edges == null) {
			m_edges = new Container();
			m_edges.setLayout(null);
		}

		removeAll();

		if (m_drawRoot == null) {
//			System.out.println("Diagram.validate done m_drawRoot=null");
			return;
		}

		m_ls.setCursor(Cursor.WAIT_CURSOR);

		m_cardinals.removeAll();

		// These go on top
		if (getLs().isShowDstCardinals() || getLs().isShowSrcCardinals()) {
			m_cardinals.setBounds(0, 0, mw, mh);
			add(m_cardinals);
		}
		// This goes under cardinals if present
		m_edges.removeAll();
		m_edges.setBounds(0,0,mw, mh);
		add(m_edges);		// Add the set of all edges


		// Clear all but the indicated marks

		if (!getDrawEdges()) {
			m_preserve_entity_marks   |= EntityInstance.REDBOX_MARK;
			m_preserve_relation_marks |= RelationInstance.HIGHLIGHT_FLAG_MARK;
		}

		m_rootInstance.clearAllMarks(m_preserve_entity_marks, m_preserve_relation_marks);

		m_drawRoot.setOpen();	// The draw root is always open

//		System.out.println("Diagram.validate() cleared all marks");

		/* This has to be delayed until we have resized things appropriately since the adding of
		   the drawRoot to the diagram causes the edges to also be added, and these need to know
		   displayable dimensions of entities
		 */

		mw       -= GAP*6;		// Maximum width  of e
		mh	     -= GAP*4;		// Maximum height of e
//		System.out.println("mw=" + mw + " mh=" + mh);
		yshift    = 0;
		clients   = null;
		suppliers = null;
		isTopClients = m_ls.isTopClients();

		// Make entity take a full footprint in the display.

		if (m_drawRoot != m_rootInstance) {
			add(m_exitFlag);
			m_exitFlag.activate();
		} else {
			// If we don't do a repaint the [-] box doesn't vanish
			repaint();
		}

//		System.out.println("Diagram.validate() added to Diagram");

		/* This performs a first cut at finding clients and suppliers
		 * It essentially collects into a vector topmost clients and suppliers
		 */

		liftEdges = m_ls.isLiftEdges();

		findClientsAndSuppliers(liftEdges);

//		System.out.println("Diagram.validate() found clients and suppliers");

		if (m_clientSet.getFullSetSize() != 0) {
			// we will either show or report not showing
			// Reduce height to allow clients
			temp = (ClientSupplierSet.CLIENT_SUPPLIER_HEIGHT + GAP*4);
			mh  -= temp;
			if (isTopClients) {
				m_clientSet.setBounds(GAP*3, GAP*2, mw, ClientSupplierSet.CLIENT_SUPPLIER_HEIGHT);
				yshift = temp; 
			} else {
				m_clientSet.setBounds(GAP*3, getHeight() - ClientSupplierSet.CLIENT_SUPPLIER_HEIGHT - GAP*2, mw, ClientSupplierSet.CLIENT_SUPPLIER_HEIGHT);
			}
//			System.out.println("Diagram.validate() m_clientSet=" + m_clientSet.getBounds());
		}

		if (m_supplierSet.getFullSetSize() != 0) {			// Reduce height to allow suppliers
			temp = (ClientSupplierSet.CLIENT_SUPPLIER_HEIGHT + GAP*4);
			mh  -= temp;
			if (!isTopClients) {
				m_supplierSet.setBounds(GAP*3, GAP*2, mw, ClientSupplierSet.CLIENT_SUPPLIER_HEIGHT);
				yshift = temp; 
			} else {
				m_supplierSet.setBounds(GAP*3, getHeight() - ClientSupplierSet.CLIENT_SUPPLIER_HEIGHT - GAP*2, mw, ClientSupplierSet.CLIENT_SUPPLIER_HEIGHT);

//				System.out.println("Supplier=" + (getY() + m_supplierSet.getY() + m_supplierSet.getHeight()) + " actual=" + m_ls.getHeight());
			}
//			System.out.println("Diagram.validate() m_supplierSet=" + m_supplierSet.getBounds());
		}

		ClientSupplierSet.compact(m_clientSet, m_supplierSet);

		clients = m_clientSet.getFullSet();
		if (clients != null && clients.size() == 0) {
			clients = null;
		}

		suppliers = m_supplierSet.getFullSet();
		if (suppliers != null && suppliers.size() == 0) {
			suppliers = null;
		}

		// Must do this before calling computeAllDiagramEdges because zero sized entities are presumed closed
		// This will assign -l width/height values

		m_drawRoot.setSize(mw, mh);

//		System.out.println("Diagram.validate() resized Diagram");

		if (!m_visited) {
			closeEntities();
			m_visited = true;
		}

		m_drawRoot.computeAllDiagramEdges(m_drawRoot, clients, suppliers, null /* No closed entity yet */, liftEdges);
//		System.out.println("Diagram.validate() computed all diagram edges");

		m_drawRoot.liftAllDiagramEdges(liftEdges);

//		System.out.println("Diagram.validate() lifted all edges");

		// This will assign -1 xrel/yrel values
		m_drawRoot.setBothLocations(0, 0, GAP*3, GAP*2+yshift);

//		System.out.println("Diagram.validate() setBothLocations");

		m_drawRoot.addUnder(this);

		// Do after lifted edges since ordering based on where edges go to/come from

		if (m_supplierSet.getFullSetSize() != 0) {	
			m_supplierSet.order();
			add(m_supplierSet);		// Add the quasi supplier box to diagram
			m_supplierSet.setVisible(true);
//			System.out.println("Diagram.validate() added Suppliers");
		}

		if (m_clientSet.getFullSetSize() != 0) {
			m_clientSet.order();
			add(m_clientSet);		// Add the quasi client box to the diagram
			m_clientSet.setVisible(true);
//			System.out.println("Diagram.validate() added Clients");
		}

		doDrawEdges(m_drawRoot, m_drawRoot);	// Marks have been previously cleared so do not need to invalidate

		if (getLs().isShowDstCardinals()) {
			EntityInstance	drawInstance;
		
			drawInstance = (EntityInstance) m_drawRoot;
			drawInstance.resetDstCardinals(m_numRelationClasses);
			drawInstance.calcDstEdgeCardinals(!getDrawEdges());
			drawInstance.showDstCardinals();
//			System.out.println("Show dst cardinals " + m_cardinals.getComponentCount());
		}

		if (getLs().isShowSrcCardinals()) {
			EntityInstance	drawInstance;
		
			drawInstance = (EntityInstance) m_drawRoot;
			drawInstance.resetSrcCardinals(m_numRelationClasses);
			drawInstance.calcSrcEdgeCardinals(!getDrawEdges());
			drawInstance.showSrcCardinals();
//			System.out.println("Show src cardinals " + m_cardinals.getComponentCount());
		}

		container = m_cardinals;
		cnt = container.getComponentCount();
		for (i = 0; i < cnt; ++i) {
			cardinal = (Cardinal) container.getComponent(i);
			cardinal.known();
		}

//		System.out.println("Diagram.validate() processed edges");

		m_ls.setCursor(Cursor.DEFAULT_CURSOR);

		m_loaded = true;
		repaint();

//		System.out.println("Diagram.validate() done");
	}

	public void redrawDiagram()
	{
		revalidate();
	}

	public boolean shouldScrollDiagram()
	{
		return(m_zoom_x > 1.0 || m_zoom_y > 1.0);
	}

	public void setZoom(double xfactor, double yfactor)
	{
		m_zoom_x = xfactor;
		m_zoom_y = yfactor;
		getPreferredSize();	
		redrawDiagram();
	}

	class Zoom extends MyUndoableEdit implements UndoableEdit
	{
		double				 m_oldX, m_oldY;
		double				 m_newX, m_newY;

		Zoom(double oldx, double oldy)
		{
			m_oldX      = oldx;
			m_oldY      = oldy;
			m_newX      = m_zoom_x;
			m_newY      = m_zoom_y;
			logEdit(this);
		}

		public String getPresentationName() 
		{
			return " Zoom diagram";
		}

		public void undo()
		{
			setZoom(m_oldX, m_oldY);
		}

		public void redo()
		{
			setZoom(m_newX, m_newY);
	}
	}	

	public boolean updateZoom(double xfactor, double yfactor) 
	{
		double oldx = m_zoom_x;
		double oldy = m_zoom_y;
		double x, y;

		if (xfactor <= 1.0 && yfactor <= 1.0 && oldx == 1.0 && oldy == 1.0) {
			return(false);
		}
		x = oldx * xfactor;
		y = oldy * yfactor;

		if (x < 1.0) {
			x = 1.0;
		}
		if (y < 1.0) {
			y = 1.0;
		}

		if (x != oldx || y != oldy) {
			setZoom(x, y);
			if (undoEnabled()) {
				new	Zoom(oldx,  oldy);
		}	}
		return true;
	}

	public boolean set_to_viewport()
	{
		if (!shouldScrollDiagram()) {
			return(false);
		}
		double oldx = m_zoom_x;
		double oldy = m_zoom_y;

		setZoom(1.0, 1.0);
		if (undoEnabled()) {
			new Zoom(oldx, oldy);
		}
		return true;
	}

	// Carefully decide on values for preserveEntityMarks and preserveRelationMarks before calling this

	public void navigateTo(EntityInstance e) 
	{
		LandscapeEditorCore ls = getLs();

		// Make entity take a full footprint in the display.

		setDrawRoot(e);
		ls.setLeftBox();
		ls.doFeedback("Now showing: " + e.getEntityLabel());

		redrawDiagram();
	}

	// Not current used

	public void navigateToId(String eid) 
	{
		EntityInstance e = getCache(eid);

		if (e != null) {
			navigateTo(e);
	}	}

	// Navigate to default entity

	public void navigateToRoot()
	{
		setPreserveEntityMarks(0);
		setPreserveRelationMarks(0);
		navigateTo(m_rootInstance);
	}

	public void navigateToDrawRootParent()
	{
		EntityInstance e;

		e = m_drawRoot.getContainedBy();

		if (e == null) {
			m_ls.error("At topmost landscape");
		} else {
			// Going up
			setPreserveEntityMarks(0);
			setPreserveRelationMarks(0);
			navigateTo(e);
	}	}

	public boolean allowElision() 
	{
		return getDrawEdges();
	}

	public boolean isModeHandlingActive()
	{
		return(m_modeHandlingActive);
	}

	public void setModeHandlingActive(boolean value)
	{
		m_modeHandlingActive = value;
	}

	public RelationInstance targetRelation(Object object)
	{
		if (object != null) {
			if (object instanceof Vector) {
				Vector v = (Vector) object;
				switch (v.size()) {
				case 0:
					m_ls.doFeedback("Target relation set is empty");
					return(null);
				case 1:
					object = v.firstElement();
					break;
				default:
					m_ls.doFeedback("Target relation set has multiple objects in it");
					return(null);
				}
			}
			if (object instanceof RelationInstance) {
				return((RelationInstance) object);
			}
			return(null);
		}

		Vector v = getHighlightedEdges();

		if (v == null) {
			m_ls.doFeedback("Highlight the edge you wish to work on");
		} else {
			if (v.size() == 1) {
				return((RelationInstance) v.firstElement());
			}
			m_ls.doFeedback("Can't operate on " + v.size() + " highlight edges");
		}
		return(null);
	}

	public EntityInstance targetEntity(Object object)
	{
		if (object != null) {
			if (object instanceof Vector) {
				Vector v = (Vector) object;
				switch (v.size()) {
				case 0:
					m_ls.doFeedback("Target entity set is empty");
					return(null);
				case 1:
					object = v.firstElement();
					break;
				default:
					m_ls.doFeedback("Target entity set has multiple objects in it");
					return(null);
			}	}
			if (object instanceof EntityInstance) {
				return((EntityInstance) object);
			}
			return(null);
		}

		EntityInstance ke;

		ke = getKeyEntity();
		if (ke != null) {
			return(ke);
		}
		return(getDrawRoot());
	}

	public Vector targetEntityRelations(Object object)
	{
		Vector v, v1;

		if (object != null) {
			if (object instanceof Vector) {
				v = (Vector) object;
				if (v.size() > 0) {
					return(v);
				}
				m_ls.doFeedback("Target set is empty");
				return(null);
			}
			v = new Vector();
			v.addElement(object);
			return(v);
		}

		v  = getHighlightedEdges();
		v1 = getGroup();
		if (v == null) {
			v = v1;
		} else if (v1 != null) {
			v.addAll(v1);
		}
		if (v == null) {
			m_ls.doFeedback("Select the entity/relations you wish to work on");
		}
		return(v);
	}

	public LandscapeObject targetEntityRelation(Object object)
	{
		Vector v = targetEntityRelations(object);

		LandscapeObject	landscapeObject = null;

		if (v != null) {
			if (v.size() != 1) {
				m_ls.doFeedback("Select a single entity or relation that you wish to work with");
			} else {
				landscapeObject = (LandscapeObject) v.firstElement();
			}
			v = null;
		}	
		return(landscapeObject);
	}

	public Vector targetRelations(Object object)
	{
		Vector v;

		if (object != null) {
			if (object instanceof Vector) {
				v = (Vector) object;
				if (v.size() > 0) {
					return(v);
				}
				m_ls.doFeedback("Target relation set is empty");
				return(null);
			}
			v = new Vector();
			v.addElement(object);
			return(v);
		}

		v = getHighlightedEdges();

		if (v == null) {
			m_ls.doFeedback("Select the edges you wish to work on");
		}
		return(v);
	}

	public Vector targetEntities(Object object)
	{
		Vector v;

		if (object != null) {
			if (object instanceof Vector) {
				v = (Vector) object;
				if (v.size() > 0) {
					return(v);
				}
				m_ls.doFeedback("Target entity set is empty");
				return(null);
			}
			v = new Vector();
			v.addElement(object);
			return(v);
		}
		return(getGroup());
	}

	public Clipboard getClipboard()
	{
		return m_clipboard;
	}

	protected void setClipboard(Clipboard value)
	{
		m_clipboard = value;
		m_ls.clipboardChanged();
	}

	protected void relayoutSubtree(EntityInstance container)
	{
		Enumeration		en;
		EntityInstance	e;

		for (en = container.getChildren(); en.hasMoreElements(); ) {
			e = (EntityInstance) en.nextElement();
			e.setRelLocal(-1, -1, e.widthRelLocal(), e.heightRelLocal());
			relayoutSubtree(e);
	}	}

	class RelayoutSubtree extends MyUndoableEdit implements UndoableEdit
	{
		// This is probably the cheapest storage cost
		// We have to cache and uncache the whole operation without
		// intermediate redraws since the -1's will be corrected by a redrawer
		// before we are finished undoing..

		EntityInstance[]	m_entities;
		double[]			m_xRel;
		double[]			m_yRel;
		double[]			m_widthRel;
		double[]			m_heightRel;
		LandscapeLayouter	m_layouter;

		protected void saveInfo(EntityInstance e, int basePreorder)
		{
			{
				int	index          = e.getPreorderNumber() - basePreorder;
			
				m_entities[index]  = e;
				m_xRel[index]      = e.xRelLocal();
				m_yRel[index]      = e.yRelLocal();
				m_widthRel[index]  = e.widthRelLocal();
				m_heightRel[index] = e.heightRelLocal();
			}

			Enumeration		en;
			EntityInstance	e1;

			for (en = e.getChildren(); en.hasMoreElements(); ) {
				e1 = (EntityInstance) en.nextElement();
				saveInfo(e1, basePreorder);
		}	}

		RelayoutSubtree(EntityInstance container)
		{
			int	need = container.nodesInSubtree();

			m_entities  = new EntityInstance[need];
			m_xRel      = new double[need];
			m_yRel      = new double[need];
			m_widthRel  = new double[need];
			m_heightRel = new double[need];
			m_layouter  = m_ls.getLayouter();

			saveInfo(container, container.getPreorderNumber());
			logEdit(this);
		}

		public String getPresentationName() 
		{
			return "Relayout subtree " + m_entities[0];
		}

		public void undo()
		{
			EntityInstance[]	entities  = m_entities;
			double[]			xRel      = m_xRel;
			double[]			yRel      = m_yRel;
			double[]			widthRel  = m_widthRel;
			double[]			heightRel = m_heightRel;
			int					length    = entities.length;
			int					i;
			EntityInstance		e;

			for (i = 1; i < length; ++i) {
				e = entities[i];
				// Not all entities exist if we do a lazy delete of an entity
				// This is because be don't preorder in such situations
				if (e != null) {
					e.setRelLocal(xRel[i], yRel[i], widthRel[i], heightRel[i]);
			}	}
			m_ls.setLayouter(m_layouter);
			redrawDiagram();
		}

		public void redo()
		{
			EntityInstance[]	entities  = m_entities;
			double[]			xRel      = m_xRel;
			double[]			yRel      = m_yRel;
			double[]			widthRel  = m_widthRel;
			double[]			heightRel = m_heightRel;
			int					length    = entities.length;
			int					i;
			EntityInstance		e;

			for (i = 1; i < length; ++i) {
				e = entities[i];
				if (e != null) {
					// Not all entities exist if we do a lazy delete of an entity
					// This is because be don't preorder in such situations
					e.setRelLocal(-1, -1, widthRel[i], heightRel[i]);
			}	}
			m_ls.setLayouter(m_layouter);
			redrawDiagram();		
		}
	}	
	
	protected void doRelayoutAll(EntityInstance e)
	{
		if (undoEnabled()) {
			new RelayoutSubtree(e);
		}
		relayoutSubtree(e);
	} 

	protected void doRelayoutAll()
	{
		doRelayoutAll(m_drawRoot);
		redrawDiagram();
	} 

	public class RelayoutDialog extends JDialog implements ActionListener {

		LandscapeLayouter[]			m_layouters;

		protected ButtonGroup		m_buttonGroup;
		protected JRadioButton[]	m_radioButtons;
		protected JButton[]			m_buttons;

		protected JButton			m_ok;
		protected JButton			m_cancel;
		protected String			m_msg = null;
		
		public RelayoutDialog()
		{
			super(m_ls.getFrame(), "Reconfigure layouters", true);

			Container			contentPane;
			JPanel				grid;
			JPanel				panel;
			int					x, y;
			JRadioButton		radioButton;
			JButton				button;
			int					i;
			Font				font, bold;
			LandscapeLayouter   layouter;
			int				    layouts;

			m_layouters = m_ls.getLayouters();
			layouts     = m_layouters.length;
			layouter    = m_ls.getLayouter();

			font   = FontCache.getDialogFont();
			bold   = font.deriveFont(Font.BOLD);

			x      = Diagram.this.getX() + 20;
			y      = Diagram.this.getY() + 20;
			setLocation(x, y);
			setForeground(ColorCache.get(0,0,0));
			setBackground(ColorCache.get(192,192,192));
			setFont(font);

			contentPane = getContentPane();
	
			grid           = new JPanel();
			grid.setLayout(new GridLayout(layouts, 1, 10, 10));

			m_buttonGroup  = new ButtonGroup();
			m_radioButtons = new JRadioButton[layouts];
			m_buttons      = new JButton[layouts];

			for (i = 0; i < layouts; ++i) {
				m_radioButtons[i] = radioButton = new JRadioButton(m_layouters[i].getMenuLabel());
				radioButton.setFont(bold);
				if (!m_layouters[i].isLayouter()) {
					radioButton.setEnabled(false);
				} else {
					if (m_layouters[i] == layouter) {
						radioButton.setSelected(true);
					}
					m_buttonGroup.add(radioButton);
				}
				grid.add(radioButton);
			}
			contentPane.add(grid, BorderLayout.WEST);
			 
			grid           = new JPanel();
			grid.setLayout(new GridLayout(layouts, 1, 10, 10));

			for (i = 0; i < layouts; ++i) {
				if (m_layouters[i].isConfigurable()) {
					button = new JButton("configure");
				} else {
					button = new JButton("");
				}
				m_buttons[i] = button; 
				grid.add(button);
				button.addActionListener(this);
			}
			contentPane.add(grid, BorderLayout.EAST);

			panel = new JPanel();
			panel.setLayout(new FlowLayout());

			m_ok = new JButton("Relayout subtree " + m_drawRoot);
			panel.add(m_ok);
			m_ok.addActionListener(this);
			m_cancel = new JButton("Don't Relayout");
			panel.add(m_cancel);
			m_cancel.addActionListener(this);

			contentPane.add(panel, BorderLayout.SOUTH);

			// Resize the window to the preferred size of its components

			this.pack();
			setVisible(true);

		}

		public String msg()
		{
			return m_msg;
		}

		// ActionListener interface

		public void actionPerformed(ActionEvent ev)
		{
			Object			source;
			JRadioButton	radioButton;
			int				i, cnt;

			source = ev.getSource();
			cnt    = m_radioButtons.length;

			if (source == m_ok || source == m_cancel) {

				for (i = 0; i < cnt; ++i) {
					radioButton = m_radioButtons[i];
					if (radioButton != null && radioButton.isSelected()) {
						m_ls.defaultToLayouter(i);
						if (source == m_ok) {
							doRelayoutAll();
							m_msg = "Prior graph layout deleted";
						}
						break;
				}	}
				setVisible(false);
				return;
			}

			for (i = 0; i < cnt; ++i) {
				if (source == m_buttons[i]) {
					if (m_radioButtons[i].isEnabled()) {
						m_radioButtons[i].setSelected(true);
					}
					m_layouters[i].configure(m_ls);
					return;
			}	}
		}
	}

	public String relayoutAll()
	{
		RelayoutDialog relayoutDialog = new RelayoutDialog();
		String		   ret            = relayoutDialog.msg();

		relayoutDialog.dispose();
		return ret;
	}

	public String closeAll()
	{
		m_drawRoot.closeAll(getContainsId());
		redrawDiagram();
		return "Entities closed";
	}

	public String openAll()
	{
		m_drawRoot.openAll(getContainsId());
		return "Entities opened";
	}

	/* This doesn't work at all as expected and I have little idea why 
	   It only approximately centers on an entity e as the zoom factor changes 
	 */

	protected void centerDiagramOn(EntityInstance e)
	{
		double		zoom_x     = m_zoom_x;
		double		zoom_y     = m_zoom_y;

		if (zoom_x > 1.0 || zoom_y > 1.0) {

			JScrollPane	scrollPane = m_ls.m_scrollDiagram;
			int			x          = 0;
			int			y          = 0;

			scrollPane.validate();		// Call to fix a bug in Swing http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=5066771

			JViewport	viewport      = scrollPane.getViewport();
			Dimension	extent        = viewport.getExtentSize();

			if (zoom_x > 1.0) {
				int			diagramWidth  = getWidth();
				int			minX          = (int) (((double) extent.width) / zoom_x);
				int         maxX          = diagramWidth - minX;
				
				x = e.getDiagramX()  + (e.getWidth()/2) - minX;
				if (x >= maxX) {
					x = maxX;
				} 
				if (x <= 0) {
					x = 0;
			}	} 

			if (zoom_y > 1.0) {
				int			diagramHeight = getHeight();
				int			minY          = (int) (((double) extent.height) / zoom_y);
				int			maxY          = diagramHeight - minY;
				
				y = e.getDiagramY()  + (e.getHeight()/2) - minY;
				if (y >= maxY) {
					y = maxY;
				} 
				if (y <= 0) {
					y = 0;
			}	} 

			// Source code "suggests that this changes the Location() of the diagram to -x,-y
			// making the x,y point the origin on which the view is positioned.  These seems
			// solid enough to suggest that I've the units of scale right.

			viewport.setViewPosition(new Point(x, y));

			// System.out.println(x + " " + y + " .v. " + getLocation() + " " + extent);
	}	}

	public String scaleEntity(int scale, boolean incFlag) 
	{
		Vector			grp     = getGroup();
		String			msg     = null;
		EntityInstance	e       = null;

		if (grp != null) {
			Enumeration en;

			switch (grp.size()) {
			case 0:
				break;
			case 1:
				e = (EntityInstance) grp.firstElement();
				if (e == getDrawRoot()) {
					break;
				}
			default:
				switch (scale) {
				case Do.DECREASE_MAG:
				case Do.INCREASE_MAG:
					if (e == null) {
						e = (EntityInstance) grp.firstElement();
					}
					break;
				default:
					for (en = grp.elements(); en.hasMoreElements(); ) {
						EntityInstance ge = (EntityInstance) en.nextElement();
						switch(scale) {
						case Do.DECREASE_WIDTH:
							ge.scaleX(SMALL_SCALE_DOWN, incFlag);
							break; 

						case Do.INCREASE_WIDTH:
							ge.scaleX(SMALL_SCALE_UP, incFlag);
							break; 

						case Do.DECREASE_HEIGHT:
							ge.scaleY(SMALL_SCALE_DOWN, incFlag);
							break; 

						case Do.INCREASE_HEIGHT:
							ge.scaleY(SMALL_SCALE_UP, incFlag);
							break; 

						case Do.DECREASE_SIZE:
							ge.scale(SMALL_SCALE_DOWN, incFlag);
							break;

						case Do.INCREASE_SIZE:
							ge.scale(SMALL_SCALE_UP, incFlag);
							break;
						}
					}
					return(msg);
			}	}
		}

		// Do it to root

		switch (scale) {
		case Do.INCREASE_MAG:
			if (e != null) {
				if (updateZoom(SMALL_SCALE_UP, SMALL_SCALE_UP)) {
					msg = "Magnified " + e.getEntityLabel();
				}
				centerDiagramOn(e);
				break;
			}
		case Do.INCREASE_SIZE:
			// Zoom in
			if (updateZoom(SMALL_SCALE_UP, SMALL_SCALE_UP)) {
				msg = "Zoomed in";
			}
			break;
		case Do.DECREASE_MAG:
			if (e != null) {
				if (updateZoom(SMALL_SCALE_DOWN, SMALL_SCALE_DOWN)) {
					msg = "Reduced " + e.getEntityLabel();
				}
				centerDiagramOn(e);
				break;
			}
		case Do.DECREASE_SIZE:
			// Zoom out
			if (updateZoom(SMALL_SCALE_DOWN, SMALL_SCALE_DOWN)) {
				msg = "Zoomed out";
			}
			break;

		case Do.INCREASE_WIDTH:
			if (updateZoom(SMALL_SCALE_UP, 1.0)) {
				msg = "Zoomed in X direction";
			}
			break;

		case Do.DECREASE_WIDTH:
			if (updateZoom(SMALL_SCALE_DOWN, 1.0)) {
				msg = "Zoomed out X direction";
			}
			break;

		case Do.INCREASE_HEIGHT:
			if (updateZoom(1.0, SMALL_SCALE_UP)) {
				msg = "Zoomed in Y direction";
			}
			break;

		case Do.DECREASE_HEIGHT:
			if (updateZoom(1.0, SMALL_SCALE_DOWN)) {
				msg = "Zoomed out Y direction";
			}
			break;
		}
		msg += " (" + SMALL_SCALE_STRING + ") to " + m_zoom_x + "x" + m_zoom_y;
		return(msg);
	}


	public String handleElision(int key, Object object) 
	{
		String			em     = "";
		Vector			grp    = targetEntities(object);
		boolean			closed = false;
		EntityInstance	ge, ke = null;
		Enumeration		en;
		String			containsId = getContainsId();

		if (grp == null  || grp.size() == 0) {
			return "Nothing selected";
		}

		switch(key)	{
		case Do.SHOW_CONTENTS:
			 // Toggle contain elision
			em = "Containment";

			if (grp.size() == 1) {
				ke = (EntityInstance) grp.elementAt(0);
			} else {
				ke = getKeyEntity();
			}
			if (ke == null) {
				return "Unable to identify key entity to open/close";
			}
			closed = ke.isDstRelationElided(containsId);
			break;
		case Do.DST_EDGES:
			em = "Target ";
			break;

		case Do.SRC_EDGES:
			em = "Source "; 
			break;

		case Do.ENTERING_EDGES:
			em = "Entering";
			break; 

		case Do.EXITING_EDGES:
			em = "Exiting";
			break;

		case Do.INTERNAL_EDGES:
			em = "Internal edges";
			break;
		default:
			return(null);
		}

		for (en = grp.elements(); en.hasMoreElements(); ) {
			ge = (EntityInstance) en.nextElement();

			switch(key)	{
			case Do.SHOW_CONTENTS:
				if (ge == ke || closed == ge.isDstRelationElided(containsId)) {
					ge.toggleContainElision(containsId);
				}
				break;

			case Do.DST_EDGES:
				toggleDstElision(ge);
				break;

			case Do.SRC_EDGES:
				toggleSrcElision(ge);
				break;

			case Do.ENTERING_EDGES:
				if (ge.isOpen()) {
					toggleEnteringElision(ge);
				}
				break;

			case Do.EXITING_EDGES:
				if (ge.isOpen()) {
					toggleExitingElision(ge);
				}
				break;
			
			case Do.INTERNAL_EDGES:
				if (ge.isOpen()) {
					toggleInternalElision(ge);
				}
				break;
			}
		}
		redrawDiagram();
		return(em + " elision toggled for group");
	}

	protected boolean handleEdgeExpansion(int key, Object object, ResultBox resultBox) 
	{
		Vector				edges;
		Enumeration			en;
		RelationInstance	ri;
		EntityInstance		src, dst, drawRoot;
		String				msg, msg1;
		String				containsId = getContainsId();

		edges = targetRelations(object);
		if (edges == null) {
			return false;
		}
		msg = null;

		switch (key) {
		case Do.EDGE_OPEN_LOW:
			msg1 = "Opened sources and destinations of edges.";
			break;
		case Do.EDGE_OPEN_SRC:
			msg1 = "Opened sources of edges.";
			break;
		case Do.EDGE_OPEN_DST:
			msg1 = "Opened destination of edges.";
			break;
		case Do.EDGE_CLOSE_LOW:
			msg1 = "Closed source and destination of edges";
			break;
		case Do.EDGE_CLOSE_SRC:
			msg1 = "Closed source of edges";
			break;
		default:
			msg1 = "Closed source and destination of edges";
			break;
		}

		resultBox.setResultTitle("SELECTED RELATIONS:");

		drawRoot = getDrawRoot();

		for (en = edges.elements(); en.hasMoreElements(); ) {
			ri = (RelationInstance) en.nextElement();
			ri.setGroupAndHighlightFlag();

			src = ri.getDrawSrc();
			dst = ri.getDrawDst();
			switch (key) {
			case Do.EDGE_OPEN_LOW:
				if (src.openEdgeExpansion(src, dst, containsId, "", resultBox)) {
					msg = msg1;
				}
				continue;
			case Do.EDGE_OPEN_SRC:
				if (src.openSrcEdgeExpansion(src, dst, containsId, "", resultBox)) {
					msg = msg1;
				}
				continue;
			case Do.EDGE_OPEN_DST:
				if (dst.openDstEdgeExpansion(src, dst, containsId, "", resultBox)) {
					msg = msg1;
				}
				continue;
			case Do.EDGE_CLOSE_LOW:
				dst = ri.getDst();
				src = ri.getSrc();
				for (;;) {
					src = src.getContainedBy();
					if (src == null) {
						break;
					}
					if (!drawRoot.hasDescendant(src)) {
						break;
					}
					if (src == dst || src.hasDescendant(dst)) {
						break;
					}
					if (src.setContainElision(containsId)) {
						msg = msg1;
				}	}
				for (;;) {
					dst = dst.getContainedBy();
					if (dst == null) {
						break;
					}
					if (!drawRoot.hasDescendant(dst)) {
						break;
					}
					if (src == dst || dst.hasDescendant(src)) {
						break;
					}
					if (dst.setContainElision(containsId)) {
						msg = msg1;
				}	}
				break;
			case Do.EDGE_CLOSE_SRC:
				dst = ri.getDst();
				src = ri.getSrc();
				for (;;) {
					src = src.getContainedBy();
					if (src == null) {
						break;
					}
					if (!drawRoot.hasDescendant(src)) {
						break;
					}
					if (src == dst || src.hasDescendant(dst)) {
						break;
					}
					if (src.setContainElision(containsId)) {
						msg = msg1;
				}	}
				break;
			case Do.EDGE_CLOSE_DST:
				dst = ri.getDst();
				src = ri.getSrc();
				for (;;) {
					dst = dst.getContainedBy();
					if (dst == null) {
						break;
					}
					if (!drawRoot.hasDescendant(dst)) {
						break;
					}
					if (src == dst || dst.hasDescendant(src)) {
						break;
					}
					if (dst.setContainElision(containsId)) {
						msg = msg1;
				}	}
				break;
			}
			resultBox.addRelation(ri);
		}

//		resultBox.activate();
		resultBox.done("-- End --");

		clearDrawEdges();

		if (msg != null) {
			m_ls.doFeedback(msg);
			return true;
		}

		switch (key) {
		case Do.EDGE_OPEN_LOW:
		case Do.EDGE_OPEN_SRC:
		case Do.EDGE_OPEN_DST:
			msg = "No further expansion is possible";
			break;
		default:
			msg = "No further contraction is possible";
		}	
		m_ls.error(msg);
		return false;
	}

	protected boolean goodQueryEdge(EntityInstance src, EntityInstance dst, EntityInstance root)
	{
		if (src.hasDescendant(dst) || dst.hasDescendant(src))
			return false;

		if (src.isClientOrSupplier()) { 
			return root.hasDescendant(dst);
		}
		return true;
	}

	protected void addToForwardEntityList(EntityInstance te, EntityInstance e, RelationClass rc, Vector list, boolean groupingFlag, boolean showInDiagram)
	{
		Enumeration		 en;
		RelationInstance ri;
		EntityInstance	 root    = getDrawRoot();
		EntityInstance	 dst, vdst, ne;

//		System.out.println("Diagram.addToForwardEdgeList " + e);
		for (en = e.srcLiftedRelationElements(); en.hasMoreElements(); ) {
			ri = (RelationInstance) en.nextElement();
//			System.out.println("Diagram.addToForwardEdgeList " + ri);
			if (ri.getRelationClass() == rc) {
				vdst = ri.getDrawDst();
				if (vdst != null) {

					ri.setHighlightFlag();
					vdst.setRedBoxFlag();
					if (groupingFlag) {
						vdst.setGroupFlag();
		}	}	}	}

//		System.out.println("Diagram.addToForwardEdgeList " + e);
		for (en = e.srcRelationElements(); en.hasMoreElements(); ) {
			ri = (RelationInstance) en.nextElement();
//			System.out.println("Diagram.addToForwardEdgeList " + ri);
			if (ri.getRelationClass() == rc) {
				dst  = ri.getDst();
				if (showInDiagram && !m_drawRoot.hasDescendantOrSelf(dst)) {
					continue;
				}
				if (list.indexOf(dst) < 0) {
					list.addElement(dst);
		}	}	}
	}

	protected int addForwardRelation(EntityInstance e, RelationClass rc, ResultBox resultBox, boolean groupingFlag, boolean showInDiagram)
	{
		Vector list = new Vector();

		addToForwardEntityList(e, e, rc, list, groupingFlag, showInDiagram);

		int num = list.size();

		if (num > 0) {
			resultBox.addForwardRelation(e, rc, list);
		}
		return num;
	}

	protected int showForwardEdges(EntityInstance e, ResultBox resultBox, boolean groupingFlag) 
	{
		Enumeration en;
		boolean		showInDiagram = e.isMarked(EntityInstance.CLIENT_SUPPLIER);
		int			num = 0;

		for (en = enumRelationClasses(); en.hasMoreElements(); ) {
			RelationClass rc = (RelationClass) en.nextElement();
//			System.out.println("Diagram.showForwardEdges class " + rc);
			if (rc.isActive()) {
				num += addForwardRelation(e, rc, resultBox, groupingFlag, showInDiagram);
//				System.out.println("Diagram.showForwardEdges num " + num);
		}	}
		return num;
	}

	// Backtrace

	public void addToBackEntityList(EntityInstance te, EntityInstance e,	RelationClass rc, Vector list, boolean groupingFlag, boolean showInDiagram)
	{
		Enumeration      en;
		RelationInstance ri;
		EntityInstance   child, src, vsrc;

		EntityInstance root = getDrawRoot();

		for (en = e.dstLiftedRelationElements(); en.hasMoreElements(); ) {
			ri = (RelationInstance) en.nextElement();

			if (ri.getRelationClass() == rc) {
				vsrc = ri.getDrawSrc();
				if (vsrc != null) {
					ri.setHighlightFlag();
					vsrc.setRedBoxFlag();
					if (groupingFlag) {
						vsrc.setGroupFlag();
		}	}	}	}

		for (en = e.dstRelationElements(); en.hasMoreElements(); ) {
			ri = (RelationInstance) en.nextElement();

			if (ri.getRelationClass() == rc) {
				src = ri.getSrc();
				if (showInDiagram && !m_drawRoot.hasDescendantOrSelf(src)) {
					continue;
				}
				if (list.indexOf(src) < 0) {
					list.addElement(src);
	}	}	}	}

	public int addBackRelation(EntityInstance e, RelationClass rc, ResultBox resultBox, boolean groupingFlag, boolean showInDiagram)
	{
		Vector list = new Vector();

		addToBackEntityList(e, e, rc, list, groupingFlag, showInDiagram);
		int num = list.size();
		if (num > 0) {
			resultBox.addBackRelation(e, rc, list);
		}
		return num;
	}

	public int showBackEdges(EntityInstance e, ResultBox resultBox, boolean groupingFlag) 
	{
		Enumeration		en;
		RelationClass	rc;
		boolean			showInDiagram = e.isMarked(EntityInstance.CLIENT_SUPPLIER);
		int				num = 0;


		for (en = enumRelationClasses(); en.hasMoreElements(); ) {
			rc = (RelationClass) en.nextElement();
			if (rc.isActive()) {
				num += addBackRelation(e, rc, resultBox, groupingFlag, showInDiagram);
		}	}	
		return num;
	}

	// Performed when FORWARD_CLOSURE is pressed

	// Get all of the relationships from visited destinations to earlier sources

	protected void getSrcRelationWithClosure(EntityInstance e, Vector list, boolean groupingFlag)
	{
		Enumeration en;
		EntityInstance root = getDrawRoot();

		for (en = e.srcRelationElements(); en.hasMoreElements(); ) {
			RelationInstance ri = (RelationInstance) en.nextElement();

			EntityInstance dst = ri.getDst();

			// Using highlight flag to mark visitation

			boolean visited = dst.isMarked(EntityInstance.REDBOX_MARK);

			ri.setHighlightFlag();

			if (list.indexOf(ri) < 0) {
				list.addElement(ri);
			}
			if (!visited) {
				dst.setRedBoxFlag();
				if (groupingFlag) {
					dst.setGroupFlag();
				}
				getSrcRelationWithClosure(dst, list, groupingFlag);
			}
		}
	}

	protected int addRelations(EntityInstance e, RelationClass rc, Vector list, ResultBox resultBox, boolean src)
	{
		int num = 0;

		if (list.size() > 0) {
			Vector	sublist = new Vector();
			Enumeration en;

			for (en = list.elements(); en.hasMoreElements(); ) {
				RelationInstance ri = (RelationInstance) en.nextElement();

				if (ri.getRelationClass() == rc) {
					ri.setHighlightFlag();
					sublist.add(ri);
					num++;
			}	}

			if (num > 0) {
				resultBox.addRelations(e, rc, sublist, src);
			}
		}

		return num;
	}

	protected int showSrcEdgesWithClosure(EntityInstance e, ResultBox resultBox, boolean groupingFlag) 
	{
		Enumeration   en;
		RelationClass rc;
		Diagram		  diagram = getDiagram();
		Vector        list    = new Vector();

		int num = 0;

		diagram.clearQueryFlags();
		getSrcRelationWithClosure(e, list, groupingFlag);
		diagram.clearQueryFlags();

		for (en = diagram.enumRelationClasses(); en.hasMoreElements(); ) {
			rc = (RelationClass) en.nextElement();

			if (rc.isActive()) {
				num += addRelations(e, rc, list, resultBox, true);
		}	}
		return num;
	}

	// Backward closure

	protected void getDstRelationWithClosure(EntityInstance e, Vector list, boolean groupingFlag)
	{
		Enumeration			en;
		RelationInstance	ri;
		EntityInstance		src;
		boolean				visited;

		for (en = e.dstRelationElements(); en.hasMoreElements(); ) {
			ri = (RelationInstance) en.nextElement();
			src = ri.getSrc();

			visited = src.isMarked(EntityInstance.REDBOX_MARK);
			ri.setHighlightFlag();
			if (list.indexOf(ri) < 0) {
				list.addElement(ri);
			}

			if (!visited) {
				src.setRedBoxFlag();
				if (groupingFlag) {
					src.setGroupFlag();
				}
				getDstRelationWithClosure(src, list, groupingFlag);
			}
		}
	}

	protected int showDstEdgesWithClosure(EntityInstance e, ResultBox resultBox, boolean groupingFlag) 
	{
		Enumeration en;
		Vector		list    = new Vector();
		Diagram		diagram = getDiagram();

		int num = 0;

		clearQueryFlags();
		getDstRelationWithClosure(e, list, groupingFlag);
		clearQueryFlags();


		for (en = diagram.enumRelationClasses(); en.hasMoreElements(); ) {
			RelationClass rc = (RelationClass) en.nextElement();

			if (rc.isActive()) {
				num += addRelations(e, rc, list, resultBox, false);
		}	}
		return num;
	}

	protected int showContents(EntityInstance e, ResultBox resultBox) 
	{
		Enumeration en;
		Vector list = new Vector();
		int n = 0;

		for (en = e.getChildren(); en.hasMoreElements(); ) {
			list.add(en.nextElement());
			n++;
		}
		if (n > 0) {
			SortVector.byString(list);
		}
		resultBox.addContents(e, list);
		return n;
	}

	protected int showContentsWithClosure(EntityInstance e, ResultBox resultBox)
	{ 
		Enumeration en;
		int			n;

		n = showContents(e, resultBox);

		for (en = e.getChildren(); en.hasMoreElements(); ) {
			n += showContentsWithClosure((EntityInstance) en.nextElement(), resultBox);
		}
		return n;
	}

	protected int doQueryEntity(EntityInstance e, int query, ResultBox resultBox, boolean groupingFlag) 
	{
		switch(query) {
		case Do.FORWARD_QUERY:
			return showForwardEdges(e, resultBox, groupingFlag);

		case Do.BACKWARD_QUERY:
			return showBackEdges(e, resultBox, groupingFlag);

		case Do.FORWARD_CLOSURE:
			return showSrcEdgesWithClosure(e, resultBox, groupingFlag);

			case Do.BACKWARD_CLOSURE:
			return showDstEdgesWithClosure(e, resultBox, groupingFlag);

		case Do.CONTENTS_QUERY:
			return showContents(e, resultBox);

		case Do.CONTENT_CLOSURE:
			return showContentsWithClosure(e, resultBox);

		default:
			return 0;
		}
	}

	public void queryEntity(int query, Object object, ResultBox resultBox, boolean groupingFlag) 
	{
		Enumeration		en;
		Vector			grp        = targetEntities(object);
		String			title, footer;
		EntityInstance	ge;

		if (!getQueryPersistance()) {
			clearQueryFlags();
		}

		switch(query) {
		case Do.FORWARD_QUERY:
		case Do.BACKWARD_QUERY:
		{
			int	mode;

			if (query == Do.FORWARD_QUERY) {
				title = "FORWARD QUERY";
			} else {
				title = "BACKWARD QUERY";
			}
			mode = 0;
			if (grp != null) {
				for (en = grp.elements(); en.hasMoreElements(); ) {
					ge = (EntityInstance) en.nextElement();
					if (ge.isMarked(EntityInstance.CLIENT_SUPPLIER)) {
						mode |= 2;
					} else {
						mode |= 1;
			}	}	}
			switch (mode) {
			case 2:
				title += " IN DIAGRAM";
				break;
			case 3:
				title += " IN DIAGRAM WHEN CLIENT/SUPPLIER";
				break;
			}
			break;
		}
		case Do.FORWARD_CLOSURE:
			title = "FORWARD QUERY (closure)";
			break;
		case Do.BACKWARD_CLOSURE:
			title = "BACKWARD QUERY (closure)";
			break;
		default:
			title = null;
		}

		resultBox.setResultTitle(title);

		if (grp == null || grp.isEmpty()) {
			footer = "No entity or group is selected";
			m_ls.error(footer);
		} else {
			int num = 0;

			for (en = grp.elements(); en.hasMoreElements(); ) {
				ge = (EntityInstance) en.nextElement();

				num += doQueryEntity(ge, query, resultBox, groupingFlag);
			}

			if (num == 0) {
				footer = "NO ENTITIES";
			} else {
				footer = "-- End of Query --";
			}
			clearDrawEdges();
		}
		resultBox.done(footer);
	}

	protected void queryContents(int key, ResultBox resultBox, boolean groupingFlag) 
	{
		Enumeration		en;
		Vector			grp        = getGroup();
		EntityInstance  ge;
		String			title, footer;

		switch (key) {
			case Do.CONTENT_CLOSURE:
				title = "CONTENTS (CLOSURE)";
				break;
			case Do.CONTENTS_QUERY:
				title = "CONTENTS";
				break;
			default:
				title = null;
		}

		resultBox.setResultTitle(title);
		footer = "-- End of content --";

		if (grp == null || grp.isEmpty()) {
			doQueryEntity(getDrawRoot(), key, resultBox, groupingFlag);
		} else {
			int num = 0;

			for (en = grp.elements(); en.hasMoreElements(); ) {
				ge = (EntityInstance) en.nextElement();
				num += doQueryEntity(ge, key, resultBox, groupingFlag);
			}

			if (num == 0) {
				footer = "NO ENTITIES";
		}	}
		resultBox.done(footer);
	}

	public void containsClassChanging()
	{
		RelationClass	oldContainsClass = getContainsClass();
		LandscapeEditorCore	ls           = getLs();
		Vector			oldChildren      = null;

		if (oldContainsClass != null && m_ls.isFocusAncestor()) {
			Enumeration en;
			
			// This step doesn't depend on the contains hierarchy
			oldChildren = getGroup();
			if (oldChildren == null) {
				oldChildren = new Vector();
			}
			if (oldChildren.size() == 0) {
				for (en = m_drawRoot.getChildren(); en.hasMoreElements(); ) {
					oldChildren.add(en.nextElement());
				}
				if (oldChildren.size() == 0) {
					oldChildren = null;
		}	}	}
		m_oldChildren = oldChildren;
	}

	public void containsClassChanged()
	{
		Vector	oldChildren = m_oldChildren;

		if (oldChildren != null) {
			EntityInstance	drawRoot = getDrawRoot();
			EntityInstance	newDrawRoot;

			for (newDrawRoot = drawRoot; newDrawRoot != null && !newDrawRoot.hasDescendantsOrSelf(oldChildren); newDrawRoot = newDrawRoot.getContainedBy());
			if (newDrawRoot != drawRoot && newDrawRoot != null) {
				navigateTo(newDrawRoot);
			}
			m_oldChildren = null;
		}
		redrawDiagram();
	}

	public boolean newEntityClass()
	{
		String		id;
		String		message;
		EntityClass	ec;

		message = "Enter new entry class name";
		for (;;) {
			id = JOptionPane.showInputDialog(message); 
			if (id == null) {
				return false;
			}
			ec = getEntityClass(id);
			if (ec == null) {
				ec = addEntityClass(id);
				ec.addParentClass(m_entityBaseClass);

				return true;
			}
			message = id + " in use. Enter new name";
		}
	}

	public boolean newRelationClass()
	{
		String			id;
		String			message;
		RelationClass	rc;

		message = "Enter new relation class name";
		for (;;) {
			id = JOptionPane.showInputDialog(message); 
			if (id == null) {
				return false;
			}
			rc = getRelationClass(id);
			if (rc == null) {
				rc = addRelationClass(id);
				rc.addParentClass(m_relationBaseClass);
				return true;
			}
			message = id + " in use. Enter new name";
		}
	}

	public void alsoValidateSubclasses(Vector v, String title)
	{
		String					message;
		LandscapeClassObject	first, o;
		int						i, size;

		size = v.size();
		switch (size) {
		case 1:
			return;
		case 2:
			o = (LandscapeClassObject) v.elementAt(1);
			message = "Also validate subclass " + o.getLabel();
			break;
		default:
			o = (LandscapeClassObject) v.elementAt(1);
			message = "Also validate subclasses " + o.getLabel();
			for (i = 2; i < size; ) {
				o = (LandscapeClassObject) v.elementAt(i);
				++i;
				if (i == size) {
					message += " and ";
				} else {
					message += ", ";
				}
				message += o.getLabel();
		}	}

		first = (LandscapeClassObject) v.elementAt(0);

		switch (JOptionPane.showConfirmDialog(null, message, "Validating " + first.getLabel() + title, JOptionPane.YES_NO_CANCEL_OPTION)) {
		case JOptionPane.YES_OPTION:
			break;
		case JOptionPane.NO_OPTION:
			v.removeAllElements();
			v.add(first);
			break;
		default:
			v.removeAllElements();
		}
		return;
	}

	public void showValidAttributes(LandscapeClassObject o, ResultBox resultBox)
	{
		Vector		v;
		int			i, size;
		Attribute	attribute;

		resultBox.addResultTitle("Valid attributes of " + o.getLabel() );
		resultBox.activate();
		v = o.getValidAttributes();
		size = v.size();
		for (i = 0; i < size; ++i) {
			attribute = (Attribute) v.elementAt(i);
			resultBox.addResultAttribute(attribute);
		}

		resultBox.addResultTitle("Lsedit class attributes" );
		o.reportClassAttributes(resultBox);
		resultBox.addResultTitle("Lsedit first order attributes");
		if (o instanceof EntityClass) {
			EntityInstance.reportFirstOrderAttributes(resultBox);
		} else {
			RelationInstance.reportFirstOrderAttributes(resultBox);
		}
		resultBox.done("End of report");

	}

	public void validateEntityAttributes(EntityClass ec, ResultBox resultBox, boolean query)
	{
		Vector		v;
		Vector		a[];
		int			errors;
		String		message;

		v      = getClassAndSubclasses(ec);

		if (query) {
			alsoValidateSubclasses(v, " attributes");
			if (v.size() == 0) {
				return;
		}	}

		a      = new Vector[v.size()];
		resultBox.addResultTitle("Validating attributes of " + ec.getLabel() + ((v.size() <= 1) ? "" : " and subclasses") + ((m_drawRoot == m_rootInstance) ? "" : " under " + m_drawRoot.getEntityLabel()) );
		resultBox.activate();

		// Never validate the root instance 
		// This instance contains all sorts of attributes that carry flags

		if (m_drawRoot != m_rootInstance) {
			errors = m_drawRoot.validateEntityAttributes(v, a, resultBox);
		} else {
			Enumeration		en;
			EntityInstance	child;

			errors = 0;
			for (en = m_rootInstance.getChildren(); en.hasMoreElements(); ) {
				child = (EntityInstance) en.nextElement();
				errors += child.validateEntityAttributes(v, a, resultBox);
		}	}


		switch (errors) {
		case 0:
			message = "No errors";
			break;
		case 1:
			message = "1 erroneous entity encountered during validation";
			break;
		default:
			message = errors + " erroneous entities encountered during validation";
		}
		resultBox.done(message);
	}

	public void validateRelationAttributes(RelationClass rc, ResultBox resultBox, boolean query)
	{
		Vector	v;
		Vector	a[];
		int		errors;
		String	message;

		v      = getClassAndSubclasses(rc);
		if (query) {
			alsoValidateSubclasses(v, " attributes");
			if (v.size() == 0) {
				return;
		}	}

		a      = new Vector[v.size()];
		resultBox.addResultTitle("Validating attributes of " + rc.getLabel() + ((v.size() <= 1) ? "" : " and subclasses") + ((m_drawRoot == m_rootInstance) ? "" : " under " + m_drawRoot.getEntityLabel()) );
		resultBox.activate();

		// Never validate the root instance 
		// This instance contains all sorts of attributes that carry flags

		errors = m_drawRoot.validateRelationAttributes(v, a, resultBox);

		switch (errors) {
		case 0:
			message = "No errors";
			break;
		case 1:
			message = "1 erroneous relation encountered during validation";
			break;
		default:
			message = errors + " erroneous relations encountered during validation";
		}
		resultBox.done(message);
	}

	public void validateRelations(RelationClass rc, ResultBox resultBox, boolean query)
	{
		Vector	v;
		boolean	a[][][];
		int		errors;
		String	message;

		v      = rc.getClassAndSubclasses(m_relationClasses);
		if (query) {
			alsoValidateSubclasses(v, " constraints");
			if (v.size() == 0) {
				return;
		}	}

		a      = new boolean[v.size()][][];
		resultBox.addResultTitle("Validating " + rc.getLabel() + " relations" + ((v.size() <= 1) ? "" : " and subclasses") + ((m_drawRoot == m_rootInstance) ? "" : " under " + m_drawRoot.getEntityLabel()) );
		resultBox.activate();

		// Never validate the root instance 
		// This instance contains all sorts of attributes that carry flags

		m_drawRoot.clearValidatedMark();
		errors = m_drawRoot.validateRelations(v, a, resultBox, m_rootInstance);

		switch (errors) {
		case 0:
			message = "No errors";
			break;
		case 1:
			message = "1 erroneous relation encountered during validation";
			break;
		default:
			message = errors + " erroneous relations encountered during validation";
		}
		resultBox.done(message);
	}

	public void validateAll(ResultBox resultBox)
	{
		validateEntityAttributes(m_entityBaseClass, resultBox, false);
		validateRelationAttributes(m_relationBaseClass, resultBox, false);
		validateRelations(m_relationBaseClass, resultBox, false);
	}

	//  Event handling

	public void entityPressed(MouseEvent ev, EntityInstance e, int x, int y)
	{
		LandscapeModeHandler	modeHandler = m_ls.getModeHandler();

		m_ls.showDescription(e, true);
		modeHandler.entityPressed(ev, e, x, y);
	}

	public void entityDragged(MouseEvent ev, EntityInstance e, int x, int y)
	{		
		LandscapeModeHandler	modeHandler = m_ls.getModeHandler();
		modeHandler.entityDragged(ev, e, x, y);
	}

	public void entityReleased(MouseEvent ev, EntityInstance e, int x, int y)
	{
		LandscapeModeHandler modeHandler = m_ls.getModeHandler();

//		System.out.println("Diagram.entityReleased=" + e);

		modeHandler.entityReleased(ev, e, x, y);
		if (y > 0) {
			m_ls.toolButton[0].requestFocus();
			m_ls.requestFocus();
	}	}		

	public void relationPressed(MouseEvent ev, RelationInstance ri, int x, int y)
	{
		LandscapeModeHandler modeHandler = m_ls.getModeHandler();

//		System.out.println("Diagram.relationPressed=" + ri);

		m_ls.showDescription(ri, true);
		modeHandler.relationPressed(ev, ri, x, y);
	}
	
	public void relationDragged(MouseEvent ev, RelationInstance ri, int x, int y)
	{		
		LandscapeModeHandler modeHandler = m_ls.getModeHandler();
		modeHandler.relationDragged(ev, ri, x, y);
	}
		
	public void relationReleased(MouseEvent ev, RelationInstance ri, int x, int y)
	{
		LandscapeModeHandler modeHandler = m_ls.getModeHandler();

//		System.out.println("Diagram.relationReleased=" + ri);

		modeHandler.relationReleased(ev, ri, x, y);
		if (y > 0) {
			m_ls.toolButton[0].requestFocus();
			m_ls.requestFocus();
	}	}

	public void movedOverThing(MouseEvent ev, Object thing, int x, int y)
	{
		LandscapeModeHandler modeHandler = m_ls.getModeHandler();
		
		modeHandler.movedOverThing(ev, thing, x, y);
	}

	// MouseMotionListener interface

	public void mouseDragged(MouseEvent ev)
	{
	}

	public void mouseMoved(MouseEvent ev)
	{
//		System.out.println("Diagram.mouseMoved");
		movedOverThing(ev, this, ev.getX(), ev.getY());
	}

	// DiagramCoordinates interface

	public int getDiagramX()
	{
		return(0);
	}

	public int getDiagramY()
	{
		return(0);
	}
}

