package lsedit;

import java.lang.Math;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Graphics;
import java.util.Enumeration;

import javax.swing.Icon;
import javax.swing.JComponent;

// Entities in a landscape 

public class EntityComponent extends JComponent implements Icon {

	// Final values
	
	public final static int MARGIN            = 5; 
	
	/* Used in computing rules for shape of a folder */

	public  final static int MIN_FLAP_HT       = 3;
	public  final static int MAX_FLAP_HT       = 8;
	public  final static int FLAP_MARGIN       = 4;
	public  final static int TINY_FLAP_WD      = 10;
	public  final static int MIN_FLAP_WD       = 75;

	/* Position and size of the contents flag when entity closed */

	public final static int CONTENTS_FLAG_X   = 3;
	public final static int CONTENTS_FLAG_Y   = 9;
	public final static int CONTENTS_FLAG_DIM = 8;
	public final static int CONTENTS_FLAG_X_RESERVE = CONTENTS_FLAG_X + CONTENTS_FLAG_DIM;
	public final static int CONTENTS_FLAG_Y_RESERVE = CONTENTS_FLAG_Y + CONTENTS_FLAG_DIM;

	// These are the top left coordinates of the object on the screen

	private EntityInstance m_entityInstance; 

	// Sorting uses this value and it is really expensive to recompute for each key compare

	private double		   m_avgX;

 	private   Cardinal[]    m_dstCardinals;
	private   SrcCardinal[] m_srcCardinals;

	private	static int[]   m_xp = null;
	private static int[]   m_yp = null;

	private static int[] getXp()
	{
		if (m_xp == null) {
			m_xp = new int[21];
		}
		return m_xp;
	}

	private static int[] getYp()
	{
		if (m_yp == null) {
			m_yp = new int[21];
		}
		return m_yp;
	}

	// --------------
	// Public methods
	// --------------

 	public EntityComponent(EntityInstance entityInstance) 
	{
		m_entityInstance = entityInstance;

		setLayout(null);
		entityInstance.setSwingObject(this);
		setBounds(entityInstance.getX(), entityInstance.getY(), entityInstance.getWidth(), entityInstance.getHeight());
		if (m_xp == null) {
			m_xp = new int[21];
			m_yp = new int[21];
		}
	}

	public String toString() 
	{
		return "EntityComponent: " + m_entityInstance.toString();
	}

	public EntityInstance getEntityInstance()
	{
		return(m_entityInstance);
	}

	public boolean shades(EntityComponent base)
	{
		int	x     = getX();
		int	basex = base.getX();

		if (x >= basex) {
			int	y     = getY();
			int basey = base.getY();
			if (y >= basey) {
				x     += getWidth();
				basex += base.getWidth();
				if (x <= basex) {
					y     += getHeight();
					basey += base.getHeight();
					if (y <= basey) {
						return true;
		}	}	}	}
		return false; 
	}

	public void computeShading()
	{
		Container parent = getParent();

		if (parent instanceof EntityComponent) {
			int				i, size;
			EntityComponent	ec1;
			EntityInstance	e1;

			e1    = getEntityInstance();
			for (i = parent.getComponentCount(); i > 0; ) {
				ec1   = (EntityComponent) parent.getComponent(--i);
				if (ec1 == this) {
					e1.orMark(EntityInstance.SHADING_KNOWN_MARK);
					return;
				}
				if (shades(ec1)) {
					e1.orMark(EntityInstance.SHADING_KNOWN_MARK | EntityInstance.SHADES_MARK);
//					System.out.println("EntityComponent;computeShading " + this + " shades " + ec1);
					return;
	}	}	}	}

	public Component add(Component child)
	{
//		System.out.println("Adding " + child + " under " + this);

		EntityComponent ec    = (EntityComponent) child;
		EntityInstance	e     = ec.getEntityInstance();
		double			area  = e.widthRelLocal() * e.heightRelLocal();
		int				size  = getComponentCount();
		EntityComponent	ec1;
		EntityInstance	e1;
		double			area1;
		int		        i;

		e.nandMark(EntityInstance.SHADING_KNOWN_MARK);
		for (i = 0; i < size; ++i) {
			ec1   = (EntityComponent) getComponent(i);
			e1    = ec1.getEntityInstance();
			area1 = e1.widthRelLocal() * e1.heightRelLocal();
			if (area1 > area) {
				// Put smaller areas before larger ones so they get painted later
				return add(child, i);
			}
			e1.nandMark(EntityInstance.SHADING_KNOWN_MARK);
		}		
		return super.add(child); 
	}

	// Clear the shading known flags for everything less than or equal area of this

	public void areaMoved()
	{
		Container parent = getParent();

		if (parent instanceof EntityComponent) {
			int				i, size;
			EntityComponent	ec1;
			EntityInstance	e1;

			size = parent.getComponentCount();
			for (i = 0; i < size; ++i) {
				ec1   = (EntityComponent) parent.getComponent(i);
				e1    = ec1.getEntityInstance();
				if (e1.isMarked(EntityInstance.SHADING_KNOWN_MARK)) {
					e1.nandMark(EntityInstance.SHADING_KNOWN_MARK);
					if (e1.isMarked(EntityInstance.SHADES_MARK)) {
						e1.repaint();
				}	}
				if (ec1 == this) {
					break;
	}	}	}	}

	// Keep the Z ordering satisfying the fact that smaller areas get painted after later ones

	public void areaChanged()
	{
		Container parent = getParent();

		if (!(parent instanceof EntityComponent)) {
			return;
		}

		int				size  = parent.getComponentCount();
		int				i, j;
		EntityComponent	ec;
		EntityInstance	e;
		double			area, area1;

		for (i = 0;; ++i) {
			if (i == size) {
				return;
			}
			ec = (EntityComponent) parent.getComponent(i);
			e  = ec.getEntityInstance();
			e.nandMark(EntityInstance.SHADING_KNOWN_MARK);
			if (this == ec) {
				break;
		}	}

		area  = e.widthRelLocal() * e.heightRelLocal();

		for (j = i-1; j >= 0; --j) {
			ec    = (EntityComponent) parent.getComponent(j);
			e     = ec.getEntityInstance();
			area1 = e.widthRelLocal() * e.heightRelLocal();
			if (area1 <= area) {
				break;
		}	}
		if (++j != i) {
//			System.out.println("Moving " + i + " down to before " + j);
			parent.remove(i);
			parent.add(this, j);	// Add after the entity with smaller area 
			return;
		}
		for (j = i+1; j < size; ++j) {
			ec    = (EntityComponent) parent.getComponent(j);
			e     = ec.getEntityInstance();
			area1 = e.widthRelLocal() * e.heightRelLocal();
			if (area1 >= area) {
				break;
		}	}
		if (--j != i) {
//			System.out.println("Moving " + i + " up to before " + j);
			parent.remove(i);
			parent.add(this, j);	// Add before the entity with the larger area
	}	}


	// Reset all cardinals associated with this entity and things it contains

	public void resetDstCardinals(int numRelations) 
	{
		Enumeration en;

		if (m_dstCardinals == null) {
			if (numRelations > 0) {
				m_dstCardinals = new Cardinal[numRelations];
			}
		} else {
			int		 i;
			Cardinal cardinal;
			
			for (i = 0; i < numRelations; ++i) {
				cardinal = m_dstCardinals[i];
				if (cardinal != null) {
					cardinal.reset();
	}	}	}	}

	public Cardinal[] getDstCardinals()
	{
		return m_dstCardinals;
	}

	public void resetSrcCardinals(int numRelations) 
	{
		Enumeration en;

		if (m_srcCardinals == null) {
			if (numRelations > 0) {
				m_srcCardinals = new SrcCardinal[numRelations];
			}
		} else {
			int		 i;
			Cardinal cardinal;
			
			for (i = 0; i < numRelations; ++i) {
				cardinal = m_srcCardinals[i];
				if (cardinal != null) {
					cardinal.reset();
	}	}	}	}

	public Cardinal[] getSrcCardinals()
	{
		return m_srcCardinals;
	}

	public void setAvgX(double avgX)
	{
		m_avgX = avgX;
	}

	public double getAvgX()
	{
		return m_avgX;
	}

	/* N.B. Must set graphics color prior to entry */

	public static void paintImage(Graphics g, int image, int x, int y, int width, int height)
	{
		int	i;

		for (i = EntityClass.ENTITY_IMAGE_LAST; i != 0; i >>= 1) {
			if ((i & image) == 0) {
				continue;
			}
			switch (i) {
				case EntityClass.ENTITY_IMAGE_ACTOR:
				{
					int	min, centerx, centery;

					min = width;
					if (height < min) {
						min = height;
					}
					centerx = x+width/2;
					centery = y+height/2;

					g.drawOval(centerx-(min/10), centery-((7*min)/20), min/5, min/5);						// Head
					g.drawLine(centerx, centery-((3*min)/20), centerx, centery+((3*min)/20));				// Body
					g.drawLine(centerx-(min/4), centery-(min/20), centerx+(min/4), centery-(min/20));		// Arms
					g.drawLine(centerx, centery+((3*min)/20), centerx-(min/5), centery+((2*min)/5));		// Left leg
					g.drawLine(centerx, centery+((3*min)/20), centerx+(min/5), centery+((2*min)/5));		// Right leg
					break;
				}
				case EntityClass.ENTITY_IMAGE_OVAL:
				{
					g.drawOval(x, y, width, height);
					break;
				}
				case EntityClass.ENTITY_IMAGE_FRAME:
				{
					g.drawRect(x, y, width-1, height-1);
					break;
				}
				case EntityClass.ENTITY_IMAGE_ROUNDED_FRAME:
				{
					int arc_w = width/5;
					int arc_h = height/5;
					int arc   = Math.min(arc_w, arc_h);

					g.drawRoundRect(x, y, width-1, height-1, arc, arc);
					break;

				}
				case EntityClass.ENTITY_IMAGE_X:
				{
					int	min, centerx, centery;

					min = width;
					if (height < min) {
						min = height;
					}
					centerx = x+width/2;
					centery = y+height/2;

					g.drawLine(centerx-(min/4), centery-(min/4), centerx+(min/4), centery+(min/4));	
					g.drawLine(centerx+(min/4), centery-(min/4), centerx-(min/4), centery+(min/4));	
					break;
	}	}	}	}

	private static void draw3DRectRaised(Graphics g, Color c, Color fill, int x, int y, int width, int height)
	{
		Color brighter = Color.white;
		Color darker   = Color.darkGray;

		if (!ArrowDimensions.isBlackWhite3D()) {
			brighter = c.brighter();
			darker   = c.darker();
		}

		if (fill != null) {
			g.setColor(fill);
			g.fillRect(x+ArrowDimensions.m_pixels_3d, y+ArrowDimensions.m_pixels_3d, width-2*ArrowDimensions.m_pixels_3d, height-2*ArrowDimensions.m_pixels_3d);
		}
		g.setColor(brighter);
		g.fillRect(x, y, ArrowDimensions.m_pixels_3d, height);																									// Band down left side
		g.fillRect(x + ArrowDimensions.m_pixels_3d, y, width - ArrowDimensions.m_pixels_3d, ArrowDimensions.m_pixels_3d);										// Band across top
		g.setColor(darker);
		g.fillRect(x + ArrowDimensions.m_pixels_3d, y + height-ArrowDimensions.m_pixels_3d, width-ArrowDimensions.m_pixels_3d, ArrowDimensions.m_pixels_3d);	// Band across bottom
		g.fillRect(x + width-ArrowDimensions.m_pixels_3d, y, ArrowDimensions.m_pixels_3d, height);																// Band down right side
		g.setColor(Color.black);
		g.drawLine(x,y, x+ArrowDimensions.m_pixels_3d-1, y+ArrowDimensions.m_pixels_3d-1);
    }    

	private static void draw3DRectSunken(Graphics g, Color c, Color fill, int x, int y, int width, int height)
	{
		Color brighter = Color.white;
		Color darker   = Color.darkGray;

		if (!ArrowDimensions.isBlackWhite3D()) {
			brighter = c.brighter();
			darker   = c.darker();
		}

		if (fill != null) {
			g.setColor(fill);
			g.fillRect(x+ArrowDimensions.m_pixels_3d, y+ArrowDimensions.m_pixels_3d, width-2*ArrowDimensions.m_pixels_3d, height-2*ArrowDimensions.m_pixels_3d);
		}
		g.setColor(darker);
		g.fillRect(x, y, ArrowDimensions.m_pixels_3d, height);																									// Band down left side
		g.fillRect(x + ArrowDimensions.m_pixels_3d, y, width - ArrowDimensions.m_pixels_3d, ArrowDimensions.m_pixels_3d);										// Band across top
		g.setColor(brighter);
		g.fillRect(x + ArrowDimensions.m_pixels_3d, y + height-ArrowDimensions.m_pixels_3d, width-ArrowDimensions.m_pixels_3d, ArrowDimensions.m_pixels_3d);	// Band across bottom
		g.fillRect(x + width-ArrowDimensions.m_pixels_3d, y, ArrowDimensions.m_pixels_3d, height);																// Band down right side
		g.setColor(Color.black);
		g.drawLine(x+width-ArrowDimensions.m_pixels_3d ,y+height-ArrowDimensions.m_pixels_3d, x+width-1, y+height-1);
    }    

	/* N.B. Any change to shapes requires potential correction to EdgePoint.adjustEdgePoint() */

	public static void paintShape(Graphics g, EntityInstance entityInstance, int x, int y, int width, int height, boolean fill, Color fillColor) 
	{
		if (width > 0 && height > 0) {
			int			   style, direction, image;
			int[]		   xp = m_xp;
			int[]		   yp = m_yp;
			int			   polygon_dimension = 0;
			int			   regular_dimension = 0;
			Color		   color, foregroundColor, backgroundColor;

			if (entityInstance.isDrawRoot()) {
				style = EntityClass.ENTITY_STYLE_3DBOX;
			} else {
				style = entityInstance.getInheritedStyle();
			}

			if (entityInstance.isDrawRoot()) {
				direction = 0;
			} else {
				direction = entityInstance.getClassDirection();
			}

			if (!fill) {
				backgroundColor = null;
			} else {
				if (fillColor != null) {
					backgroundColor = fillColor;
				} else {
					if (entityInstance.red_closed()) {
						backgroundColor = Color.red.darker();
					} else {
						backgroundColor = entityInstance.getCurrentObjectColor();
			}	}	}
			foregroundColor = (entityInstance.red_open() ? Color.red.darker() : Color.black);

			switch (style) {
				case EntityClass.ENTITY_STYLE_3DBOX:
				{
					draw3DRectRaised(g, foregroundColor, backgroundColor, x, y, width-1, height-1);
					break;
				}
				case EntityClass.ENTITY_STYLE_SUNK_BOX:
				{
					draw3DRectSunken(g, foregroundColor, backgroundColor, x, y, width-1, height-1);
					break;
				}
				case EntityClass.ENTITY_STYLE_2DBOX:
				{
					if (fill) {
						g.setColor(backgroundColor);
						g.fillRect(x, y, width, height);
					}
					g.setColor(foregroundColor);
					g.drawRect(x, y, width-1, height-1);
					break;
				}

				case EntityClass.ENTITY_STYLE_FILE:
				{
					int fd = Math.min(Math.min(width, height)/2, 16 /* FLAP_DIM */);

					/*
					   0--(width-fd)----1 \        
					   |                     \    (fd)
					   |				        \ 
					   |                           2
					   | 						    |
					   |						 (height)
					   |							|
					   4 -------------------------- 3
					 */

					xp[0] = x;
					yp[0] = y;
					xp[1] = x + width - fd;
					yp[1] = y;
					xp[2] = x + width - 1;
					yp[2] = y + fd;
					xp[3] = xp[2];
					yp[3] = y + height - 1;
					xp[4] = x;
					yp[4] = yp[3];
					xp[5] = xp[0];
					yp[5] = yp[0];

					if (fill) {
						g.setColor(backgroundColor);
						g.fillPolygon(xp, yp, 6);
					}
					g.setColor(foregroundColor);
					g.drawPolygon(xp, yp, 6);

					xp[0] = xp[1];
					yp[0] = yp[2];
					xp[3] = xp[0];
					yp[3] = yp[0];

					g.drawPolygon(xp, yp, 4);
					break;
				}

				case EntityClass.ENTITY_STYLE_DISK:
				{
					int ad = Math.min(Math.min(width, height)/2, 8 /* ARC_DIM */);

					// Draw a cylinder

					if (fill) {
						g.setColor(backgroundColor);
						g.fillRect(x, y+ad/2, width, height-ad);
						g.fillOval(x, y, width, ad);
						g.fillOval(x, y+height-ad, width, ad);
					}
					g.setColor(foregroundColor);

					g.drawOval(x, y, width, ad);
					g.drawArc(x, y+height-ad-1, width, ad, 180, 180);
					g.drawLine(x, y+ad/2, x, y+height-ad/2);
					g.drawLine(x+width-1, y+ad/2, x+width-1, y+height-ad/2);
					break;
				}

				case EntityClass.ENTITY_STYLE_FOLDER:
				{
					// Compute flap size

					int fw = ((int) (((double) width) * .4));
					int fh = Math.max(MIN_FLAP_HT, Math.min(MAX_FLAP_HT, ((int) (height * .2))));
					int fm = FLAP_MARGIN;

					if (fw < MIN_FLAP_WD) {
					   fw += fw/2;
					}
					if (fw < TINY_FLAP_WD) {
						fw = Math.min(fw + FLAP_MARGIN, width - width/3);
						fm = 0;
					}

				/*   
                     <--------------fw-------------------------->

                                     2--------------------------3
                   (fh)             /                            \
				     0<--- fm---->1 fh/2                      fh/2  4---------------------------5
				     |                                                                         |
					 |																		   |
					 |																		   |
					 7------------------------------------------------------------------------ 6
			     */

					int x0, y0, x1, y1, x2, y2, x3, y3, x4, y4, x5, y5;


					if (fill) {
						g.setColor(backgroundColor);
						g.fillRect(x, y, width-1, height-1);
					}

					g.setColor(foregroundColor);
					g.drawRect(x,y,width-1, height-1);

					x0 = x;
					y0 = y+fh;
					x5 = x + width - 1;
					y5 = y0;

					if (entityInstance.isOpen()) {
						g.drawLine(x0, y0, x5, y5);
					} else {
						x1 = x+fm;
						y1 = y+fh;

						g.drawLine(x0, y0, x1, y1);

						x2 = x1+fh/2;
						y2 = y;

						g.drawLine(x1, y1, x2, y2);

						x3 = x + fw;
						y3 = y2;

						g.drawLine(x2, y2, x3, y3);

						x4 = x3 + fh/2;
						y4 = y0;

						g.drawLine(x3, y3, x4, y4);
						g.drawLine(x4, y4, x5, y5);
					}
/*
					xp[0] = x;
					yp[0] = y+fh;
					xp[1] = x+fm;
					yp[1] = y+fh;
					xp[2] = xp[1]+fh/2;
					yp[2] = y;
					xp[3] = x + fw;
					yp[3] = yp[2];
					xp[4] = xp[3]+fh/2;
					yp[4] = yp[0];
					xp[5] = x + width-1;
					yp[5] = yp[0];
					xp[6] = xp[5];
					yp[6] = y + height-1;
					xp[7] = xp[0];
					yp[7] = yp[6];
					xp[8] = xp[0];
					yp[8] = yp[0];

					if (fill) {
						g.setColor(backgroundColor);
						g.fillPolygon(xp, yp, 9);
					}

					g.setColor(foregroundColor);
					g.drawPolygon(xp, yp, 9);

					if (!entityInstance.isOpen()) {
						g.drawLine(xp[1], yp[1], xp[4], yp[4]);
					}
 */
					break;
				}
				case EntityClass.ENTITY_STYLE_SOURCEOBJ:
				{
					if (fill) {
						g.setColor(backgroundColor);
						g.fillOval(x, y, width, height);
					}
					g.setColor(foregroundColor);
					g.drawOval(x, y, width, height);
					break;
				}

				case EntityClass.ENTITY_STYLE_CLASS:
				{
					int arc_w = width/4;
					int arc_h = height/4;
					int arc   = Math.min(arc_w, arc_h);

					if (fill) {
						g.setColor(backgroundColor);
						g.fillRoundRect(x, y, width-1, height-1, arc, arc);
					}
					g.setColor(foregroundColor);
					g.drawRoundRect(x, y, width-1, height-1, arc, arc);
					break;
				}

				case EntityClass.ENTITY_STYLE_GROUP:
				{
					g.setColor(backgroundColor);
					Util.drawOutlineBox(g, x, y, width, height);
					break;
				}

				case EntityClass.ENTITY_STYLE_LABELLED_GROUP:
				{
					g.setColor(backgroundColor);
					Util.drawGroupBox(g, x, y, width, height, entityInstance.getEntityLabel() /* Label is used only for sizing info */);
					break;
				}
				case EntityClass.ENTITY_STYLE_TRIANGLE:
				{
					switch (direction) {
					case 0:
						xp[0] = x;
						yp[0] = y+height-1;
						xp[1] = x+width/2;
						yp[1] = y;
						xp[2] = x+width-1;
						yp[2] = yp[0];
						break;
					case 1:
						xp[0] = x;
						yp[0] = y;
						xp[1] = x+width-1;
						yp[1] = y+height/2;
						xp[2] = xp[0];
						yp[2] = y+height-1;
						break;
					case 2:
						xp[0] = x;
						yp[0] = y;
						xp[1] = x + width - 1;
						yp[1] = yp[0];
						xp[2] = x + width/2;
						yp[2] = y + height - 1;
						break;
					case 3:
						xp[0] = x + width - 1;
						yp[0] = y;
						xp[1] = x;
						yp[1] = y + height/2;
						xp[2] = xp[0];
						yp[2] = y + height - 1;
						break;
					}
					polygon_dimension = 4;
					break;
				}
				case EntityClass.ENTITY_STYLE_ROMBUS:
				{
					switch (direction) {
					case 0:
						xp[0] = x;
						yp[0] = y+height-1;
						xp[1] = x+width/5;
						yp[1] = y;
						xp[2] = x+width-1;
						yp[2] = y;
						xp[3] = x+(width*4)/5;
						yp[3] = y+height-1;
						break;
					case 1:
						xp[0] = x;
						yp[0] = y;
						xp[1] = x+width-1;
						yp[1] = y+height/5;
						xp[2] = xp[1];
						yp[2] = y+height-1;
						xp[3] = xp[0];
						yp[3] = y+(4*height)/5;
						break;
					case 2:
						xp[0] = x+width/5;
						yp[0] = y+height-1;
						xp[1] = x;
						yp[1] = y;
						xp[2] = x+(4 *width)/5;
						yp[2] = y;
						xp[3] = x+width-1;
						yp[3] = y+height-1;
						break;
					case 3:
						xp[0] = x;
						yp[0] = y+height/5;
						xp[1] = x+width-1;
						yp[1] = y;
						xp[2] = xp[1];
						yp[2] = y+(4*height)/5;
						xp[3] = xp[0];
						yp[3] = y+height-1;
						break;
					}
					polygon_dimension = 5;
					break;
				}
				case EntityClass.ENTITY_STYLE_TRAPEZOID:
				{
					switch (direction) {
					case 0:
						xp[0] = x;
						yp[0] = y+height-1;
						xp[1] = x+width/5;
						yp[1] = y;
						xp[2] = x+(4*width)/5;
						yp[2] = yp[1];
						xp[3] = x+width-1;
						yp[3] = yp[0];
						break;
					case 1:
						xp[0] = x;
						yp[0] = y;
						xp[1] = x + width - 1;
						yp[1] = y + height/5;
						xp[2] = xp[1];
						yp[2] = y + (4*height)/5;
						xp[3] = xp[0];
						yp[3] = y + height - 1;
						break;
					case 2:
						xp[0] = x;
						yp[0] = y;
						xp[1] = x+width-1;
						yp[1] = yp[0];
						xp[2] = x+(4*width)/5;
						yp[2] = y+height-1;
						xp[3] = x+width/5;
						yp[3] = yp[2];
						break;
					case 3:
						xp[0] = x + width - 1;
						yp[0] = y;
						xp[1] = x;
						yp[1] = y + height/5;
						xp[2] = xp[1];
						yp[2] = y + (4*height)/5;
						xp[3] = xp[0];
						yp[3] = y + height - 1;
						break;
					}
					polygon_dimension = 5;
					break;
				}
				case EntityClass.ENTITY_STYLE_TRIANGLE2:
				{
					regular_dimension = 3;
					break;
				}
				case EntityClass.ENTITY_STYLE_RECTANGLE:
				{
					regular_dimension = 4;
					break;
				}

				case EntityClass.ENTITY_STYLE_PENTAGON:
				{
					regular_dimension = 5;
					break;
				}
				case EntityClass.ENTITY_STYLE_HEXAGON:
				{
					regular_dimension = 6;
					break;
				}
				case EntityClass.ENTITY_STYLE_OCTAGON:
				{
					regular_dimension = 8;
					break;
				}
				case EntityClass.ENTITY_STYLE_DECAHEDRON:
				{
					regular_dimension = 10;
					break;
				}
				case EntityClass.ENTITY_STYLE_12SIDED:
				{
					regular_dimension = 12;
					break;
				}
				case EntityClass.ENTITY_STYLE_14SIDED:
				{
					regular_dimension = 14;
					break;
				}
				case EntityClass.ENTITY_STYLE_16SIDED:
				{
					regular_dimension = 16;
					break;
				}
				case EntityClass.ENTITY_STYLE_18SIDED:
				{
					regular_dimension = 18;
					break;
				}
				case EntityClass.ENTITY_STYLE_20SIDED:
				{
					regular_dimension = 20;
					break;
				}
				case EntityClass.ENTITY_STYLE_PAPER:
				{
					int	baseline = y+(height*4)/5;

					xp[0] = x+(3*width)/8;			// x-
					yp[0] = baseline;				// | /
					xp[1] = xp[0];					// |/
					yp[1] = y+((193*height)/200);	// x
					xp[2] = x+(2*width)/3;			//    Diagonal up
					yp[2] = baseline;				//    Right top point			
					xp[3] = xp[0];
					yp[3] = yp[0];

					if (fill) {
						g.setColor(backgroundColor);
						g.fillRect(x,y, width, (height*4)/5);
						g.fillArc(x, y+height/2, width/2, height/2, -180, 180);
						g.fillPolygon(xp,yp, 4);								// This triangle connects the arc with the baseline

					}

					g.setColor(foregroundColor);
					g.drawLine(x, y, x, y+(height*3)/4);						// Left vertical edge
					g.drawLine(x, y, x+width-1, y);								// Top horizontal edge
					g.drawLine(x+width-1, y, x+width-1, baseline);				// Right vertical edge
					g.drawLine(x+(2*width)/3, baseline, x+width-1, baseline);

					g.drawArc(x, y+(height/2)-1, (width/2), (height/2)-1, -180, 120);
					g.drawLine( xp[1],yp[1], xp[2], yp[2]);

//					g.drawArc(x+(width/12),  baseline, x+((7*width)/6), y+((5*height)/2)-1, 120, -30);
					break;
				}
/*
				default:
				{
					// For debugging (This shows the root)
					g.setColor(Color.red);
					g.drawLine(x, y,        x+width-1, y+height-1);
					g.drawLine(x, y+height-1, x+width-1, y);
				}
*/
			}
			if (regular_dimension != 0) {
				polygon_dimension = regular_dimension + 1;

				double	angle = Math.toRadians(entityInstance.getClassAngle() - 90.0);		// -90.0 to point up N.B. coordinate at (0,-.5)
				double	shift = 2.0 * Math.PI/((double) regular_dimension);
				double	dwidth, dheight;
				int		i;

				dwidth  = (double) width;
				dheight = (double) height;
				for (i = 0; i < regular_dimension; ++i) {
					xp[i] = x + (int) (dwidth  * (1.0 + Math.cos(angle)) * 0.5);
					yp[i] = y + (int) (dheight * (1.0 + Math.sin(angle)) * 0.5);
					angle -= shift;
				}
			}

			if (polygon_dimension != 0) {
				xp[polygon_dimension-1] = xp[0];
				yp[polygon_dimension-1] = yp[0];
				if (fill) {
					g.setColor(backgroundColor);
					g.fillPolygon(xp, yp, polygon_dimension);
				}

				g.setColor(foregroundColor);
				g.drawPolygon(xp, yp, polygon_dimension);
			}

			if (!entityInstance.isDrawRoot() && !entityInstance.isOpen()) {
				image = entityInstance.getClassImage();
				if (image != EntityClass.ENTITY_IMAGE_NONE) {
					Color	inverse;

					g.setColor(backgroundColor);
					inverse = ColorCache.getInverse(backgroundColor.getRGB());
					g.setColor(inverse);

					paintImage(g, image, x, y, width, height);
			}	}
	}	}		

	public void paintMap(Graphics g, int x, int y, int width, int height, EntityInstance onPath, int depth)
	{
		if (width > 0 && height > 0) {
			EntityInstance	entityInstance = m_entityInstance;
			Enumeration		children;
			EntityInstance	child;
			int				x1, y1,width1, height1;
			Color			color;

			if (entityInstance == onPath) {
				color = Color.green;
			} else {
				color = entityInstance.getInheritedObjectColor();
			} 
			paintShape(g, m_entityInstance, x, y, width, height, !entityInstance.hasChildren() || depth == 1 /* then fill */, color);

			if (depth < 1) {
				for (children = entityInstance.getChildren(); children.hasMoreElements(); ) {
					child = (EntityInstance) children.nextElement();
					x1      = x + (int) (width  * child.xRelLocal());
					y1      = y + (int) (height * child.yRelLocal()); 
					width1  = (int) (width  * child.widthRelLocal());
					height1 = (int) (height * child.heightRelLocal());
					child.paintMap(g, x1, y1, width1, height1, onPath, depth+1);
	}	}	}	}

	protected void drawTopLeftLabel(Graphics g) 
	{
		EntityInstance entityInstance = m_entityInstance;

		switch (entityInstance.getInheritedStyle()) {
		case EntityClass.ENTITY_STYLE_FOLDER:

			g.setFont(EntityInstance.getSmallFont());
			// Compute flap size
			int fw = (int) (((double) getWidth()) * .4);
			int fh = Math.max(MIN_FLAP_HT, Math.min(MAX_FLAP_HT, ((int) (getHeight() * .2))));

			if (fw < MIN_FLAP_WD) {
			   fw += fw/2;
			}
			if (fw < TINY_FLAP_WD) {
				fw = Math.min(fw + FLAP_MARGIN, getWidth() - getWidth()/3);
			}
//			Util.drawStringClipped(g, entityInstance.getEntityLabel(), FLAP_MARGIN+fh/2+2, 0, (double) fw, getHeight()-MARGIN*2);
			Util.drawStringWrapped(g, entityInstance.getEntityLabel(), FLAP_MARGIN+fh/2+2, 0, (double) fw, getHeight()-MARGIN*2, false, false);

			break;
		case EntityClass.ENTITY_STYLE_CLASS:
			g.setFont(EntityInstance.getOpenClassFont());
//			Util.drawStringClipped(g, entityInstance.getEntityLabel(), MARGIN*3, MARGIN, getWidth()-MARGIN*2, getHeight()-MARGIN*2);
			Util.drawStringWrapped(g, entityInstance.getEntityLabel(), MARGIN*3, MARGIN, getWidth()-MARGIN*2, getHeight()-MARGIN*2, false, false);

			break;
		default:
			g.setFont(EntityInstance.getSmallFont());
//			g.setColor(Color.black); 
//			Util.drawStringClipped(g, entityInstance.getEntityLabel(), MARGIN, MARGIN, getWidth()-MARGIN*2, getHeight()-MARGIN*2);
			Util.drawStringWrapped(g, entityInstance.getEntityLabel(), MARGIN, MARGIN, getWidth()-MARGIN*2, getHeight()-MARGIN*2, false, false);
		}
	}

	public void paintComponent(Graphics g) 
	{
		EntityInstance	entityInstance = m_entityInstance;
		int		x                      = 0;
		int		y                      = 0;
		int		width                  = getWidth();
		int		height                 = getHeight();
		int		margin                 = MARGIN;
		String	label1                 = entityInstance.getEntityLabel();
		int		labelWidth;
		boolean	center;

		
//		System.out.println("EntityInstance.paintComponent: " + entityInstance + " bounds=" + getBounds() + " clip=" + g.getClipBounds() + " style=" + entityInstance.getInheritedStyle() + " parentClass=" + entityInstance.getParentClass());		// IJD
//		System.out.println("EntityInstance.paintComponent: " + entityInstance + " redBox=" + entityInstance.isMarked(EntityInstance.REDBOX_MARK));		// IJD
/*		{
			int	cnt, i;

			cnt = getComponentCount();
			for (i = 0; i < cnt; ++i) {
				System.out.println("EntityInstance.paintComponent: child " + getComponent(i));
		}	}
*/

/*		java.lang.Thread.dumpStack();
		System.out.println("-----");
*/

		if (!entityInstance.isMarked(EntityInstance.SHADING_KNOWN_MARK)) {
			computeShading();
		}
		if (entityInstance.isMarked(EntityInstance.SHADES_MARK)) {
			int	shadow_size = ArrowDimensions.m_shadow_size;
			int	max         = ((width > height) ? height : width) >> 1;
			if (shadow_size > max) {
				shadow_size = max;
			}
			if (shadow_size > 0) {
				Color backgroundColor;

				if (entityInstance.red_closed()) {
					backgroundColor = Color.red.darker();
				} else {
					backgroundColor = entityInstance.getCurrentObjectColor();
				}
				backgroundColor = backgroundColor.darker();
				width  -= shadow_size;
				height -= shadow_size;
				paintShape(g, m_entityInstance, shadow_size, shadow_size, width, height, true /* fill */, backgroundColor);
		}	}

		paintShape(g, m_entityInstance, 0, 0, width, height, true /* fill */, null /* No given fill color */); 

		Diagram				diagram = entityInstance.getDiagram();
		LandscapeEditorCore ls      = diagram.getLs();

		if (ls.isShowGrid()) {

			if (entityInstance == diagram.getDrawRoot()) {
				int	grid = diagram.getGrid();

				if (grid > 1) {
					int i;
					g.setColor(diagram.getGridColor());
					for (i = grid; i < height; i += grid) {
						g.drawLine(0,i,width-1,i);
					}
					for (i = grid; i < width; i += grid) {
						g.drawLine(i,0,i, height-1);
		}	}	}	}

		if (entityInstance.isMarked(EntityInstance.HIGHLIGHT_EDGE_MARK)) {
			g.setColor(Color.white);
			g.drawRect(0, 0, width, height);
			g.drawRect(1, 1, width-1, height-1);
		}

		if (entityInstance.close_with_children_under_drawroot()) {

			/*  Draw a small mark as shown in top left of object

				x---
				|
				|   --------
				|  |        |
				   |    |   |
				   |    |   |
	  			   |  ----  |
				   |    |   |
				   |    |   |
				   |        |
				   |________|
			*/

			if (width > CONTENTS_FLAG_X_RESERVE && height > CONTENTS_FLAG_Y_RESERVE) {
				int x1, y1;
		
				g.setColor(ColorCache.getInverse(entityInstance.getInheritedObjectColor().getRGB()));

				g.drawRect(CONTENTS_FLAG_X, CONTENTS_FLAG_Y, CONTENTS_FLAG_DIM /* 8 */, CONTENTS_FLAG_DIM);

				g.drawLine(CONTENTS_FLAG_X+1,                     CONTENTS_FLAG_Y+(CONTENTS_FLAG_DIM/2), CONTENTS_FLAG_X_RESERVE-1,             CONTENTS_FLAG_Y+(CONTENTS_FLAG_DIM/2));
				g.drawLine(CONTENTS_FLAG_X+(CONTENTS_FLAG_DIM/2), CONTENTS_FLAG_Y+1,                     CONTENTS_FLAG_X+(CONTENTS_FLAG_DIM/2), CONTENTS_FLAG_Y_RESERVE-1);
				margin += CONTENTS_FLAG_X_RESERVE;
		}	}

		// Draw our own label

		g.setColor(entityInstance.getCurrentLabelColor());

		switch (entityInstance.getInheritedStyle()) {
		case EntityClass.ENTITY_STYLE_LABELLED_GROUP:
			Util.drawGroupBoxLabel(g, 0, 0, width, label1);
			break;
		default:
			labelWidth = getWidth() - MARGIN - margin;
			if (entityInstance.isOpen()) {
				center     = false;
				switch (entityInstance.getInheritedStyle()) {
				case EntityClass.ENTITY_STYLE_FOLDER:

					g.setFont(EntityInstance.getSmallFont());
					// Compute flap size
					int fw = (int) (((double) getWidth()) * .4);
					int fh = Math.max(MIN_FLAP_HT, Math.min(MAX_FLAP_HT, ((int) (getHeight() * .2))));

					if (fw < MIN_FLAP_WD) {
					   fw += fw/2;
					}
					if (fw < TINY_FLAP_WD) {
						fw = Math.min(fw + FLAP_MARGIN, getWidth() - getWidth()/3);
					}
					margin     = FLAP_MARGIN+fh/2+2;
//					Util.drawStringClipped(g, label1, FLAP_MARGIN+fh/2+2, 0, (double) fw, getHeight()-MARGIN*2);
					break;
				case EntityClass.ENTITY_STYLE_CLASS:
					g.setFont(EntityInstance.getOpenClassFont());
				
//					Util.drawStringClipped(g, label1, MARGIN*3, MARGIN, getWidth()-MARGIN*2, getHeight()-MARGIN*2);
					break;
				default:
					g.setFont(EntityInstance.getSmallFont());
//					g.setColor(Color.black); 
//					Util.drawStringClipped(g, label1, MARGIN, MARGIN, getWidth()-MARGIN*2, getHeight()-MARGIN*2);
				}
			} else {
				center = true;
				// centered, perhaps multi-line
				if (!entityInstance.isMarked(EntityInstance.CLIENT_SUPPLIER)) {
					g.setFont(entityInstance.getAdjustedClosedFont());
				} else {
					g.setFont(ClientSupplierSet.getClientSupplierFont());
					EntityInstance pe = entityInstance.getContainedBy();
					if (pe != null) {
						label1 = pe.getEntityLabel() + " .\n" + label1;
			}	}	}
			Util.drawStringWrapped(g, label1, margin, MARGIN, labelWidth, getHeight()-MARGIN*2, center, false);
			
		} 
	
		// Put flags around the edges of the box

		if (entityInstance.getGroupFlag()) {
			g.setColor(entityInstance.getInheritedLabelColor());

			int pdim = (height < 20) ? 4 : 6;

			// Draw resize points

			if (entityInstance.getGroupKeyFlag()) {
				g.fillRect(1,			   1,				pdim, pdim);
				g.fillRect(width/2-pdim/2, 1,				pdim, pdim);
				g.fillRect(width-pdim,	   1,				pdim, pdim);
				g.fillRect(1,			   height/2-pdim/2, pdim, pdim);
				g.fillRect(width-pdim,	   height/2-pdim/2, pdim, pdim);
				g.fillRect(1,			   height-pdim,		pdim, pdim);
				g.fillRect(width/2-pdim/2, height-pdim,		pdim, pdim);
				g.fillRect(width-pdim,	   height-pdim,		pdim, pdim);
			}  else {
				// Stupidity here.. An outline is one extra byte wide and high because the two
				// edges of a one pixel box occupy 2 pixels.

				g.drawRect(1,			   1,				pdim, pdim);
				g.drawRect(width/2-pdim/2, 1,				pdim, pdim);
				g.drawRect(width-pdim-1,   1,				pdim, pdim);
				g.drawRect(1,			   height/2-pdim/2, pdim, pdim);
				g.drawRect(width-pdim-1,   height/2-pdim/2, pdim, pdim);
				g.drawRect(1,			   height-pdim-1,	pdim, pdim);
				g.drawRect(width/2-pdim/2, height-pdim-1,	pdim, pdim);
				g.drawRect(width-pdim-1,   height-pdim-1,	pdim, pdim);
			}
		}

		/* For debugging 
		g.setColor(Color.black);
		g.drawLine(0,0,width,height);
		g.drawLine(0,height, width, 0);
		*/
	}

	// Icon interface (used to paint legends)

	public int getIconWidth()
	{
		return(getWidth());
	}

	public int getIconHeight()
	{
		return(getHeight());
	}

	public void paintIcon(Component c, Graphics g, int x, int y)
	{
//		System.out.println("EntityComponent.paintIcon");
		g.translate(x, y);
		paintComponent(g);
	}
}
