package lsedit;
import java.util.Vector;
import java.util.Enumeration;

/**
 * Developed by: Ian Davis in Summer 1997 for Grant Weddell
 * Provides a static method to associate an ordering of nodes at the same rank.. In chess parlance
 * on the same row.. thus the name.  This code is conscious of the fact that any attempt to reorder
 * nodes on a rank must be careful not to violate the orderings enforced on a global higraph order
 * imposed by HiChildren (which is used to determine whether nodes not in our subtree are to our
 * left or right.
 *
 * N.B.  The problem of optimising the ordering of nodes on two rows so that edge crossings are
 * minimized is NP-complete.  Therefore this is a rattle and shake algorithm, which attempts to
 * reduce edge crossings by placing nodes at the median position within the nodes which then 
 * address/ are addressed by.
 */

 
/* N.B. We assume below that all edges are directed strictly downwards according to rank one level
 * at a time.  
 */

class HiRow {
	
	/* Create the structure used to hold the total collection of HiGraph nodes on the board
	 * We assume that the minimal rank or all children is node.m_rank that no child node has
	 * a rank > node.m_sink_rank
	 */

	static HiGraph[][] buildBoard(HiGraph node)  {

		int			count[]  = new int[node.m_sink_rank-node.m_rank + 1];
		int			ranks[]  = new int[node.m_sink_rank-node.m_rank + 1];
		Vector		children = node.m_children;
		int			size     = children.size();
		int			i, j;
		HiArc		arc;
		HiGraph		child;
		
		/* Count the children at each possible rank */

		for (i = 0; i < size; ++i) {
			arc   = (HiArc) children.elementAt(i);
			child = arc.to();
			// Ranks are with respect to parent rank
			++count[child.m_rank-node.m_rank];
		}

		/* Compute the real board ranks ignoring empty ranks */
	
		for (i = j = 0; i < count.length; ++i) {
			ranks[i] = j;
			if (count[i] != 0) ++j;
		}

		/* Build the board of the correct shape */

		HiGraph	board[][] = new HiGraph[j][];

		for (i = j = 0; i < count.length; ++i) {
			if (count[i] != 0) {
				board[j++] = new HiGraph[count[i]];
				count[i] = 0;
		}	}

		/* Fill the board */

		int	rank, position;

		for (i = 0; i < size; ++i) {
			arc                   = (HiArc) children.elementAt(i);
			child                 = arc.to();
			rank                  = ranks[child.m_rank-node.m_rank]; // rank on board to use for this rank.
			position              = count[rank];
			child.m_position      = position;
			board[rank][position] = child;
			++count[rank];
		}
		return(board);
	}

	static void dumpRow(int number, HiGraph row[]) 
	{
		int j;

		System.out.print(number + ":");
		for (j = 0; j < row.length; ++j) {
			System.out.print(row[j].m_position + ":" + row[j] + " ");
		}
		System.out.println("");
	}

	static void dumpBoard(HiGraph board[][]) 
	{
		int		i;
	
		for (i = 0; i < board.length; ++i) {
			dumpRow(i, board[i]);
	}	}

	private static void rowWeight(HiGraph higraph)
	{
		Vector		arcs;
		int			weight, i, in_size, out_size;
		HiArc		arc;
		HiGraph		other;

		weight       = 0;

		arcs       = higraph.m_out;									// Our output arcs
		out_size   = arcs.size(); 	
		for (i = out_size; i > 0; ) {								// For each of our output arcs
			arc       = (HiArc) arcs.elementAt(--i);				// Get arc
			other     = arc.to();									// Get node addressed on next row
			weight   += other.m_position;
		}

		arcs       = higraph.m_in;									// Our input arcs
		in_size    = arcs.size();
		for (i = in_size; i > 0; ) {								// For each of our input arcs
			arc       = (HiArc) arcs.elementAt(--i);				// Get arc
			other     = arc.from();									// Get node addressed on prior row
			weight   += other.m_position;
		}
		in_size += out_size;
		if (in_size == 0) {
			higraph.m_rowweight = -1.0;
		} else {
			weight += in_size;
			higraph.m_rowweight = ((double) weight)/((double) in_size);
		}
	}

	private static boolean shake(HiGraph row[])
	{
		HiGraph	higraph;
		int		i, position;
		double	max;
		boolean	ret;

		ret = false;
		for (i = row.length; i > 0; ) {
			higraph = row[--i];
			rowWeight(higraph);
		}

		for (position = row.length;;) {
			max = -2.0;
			for (i = row.length; i > 0; ) {
				higraph = row[--i];
				if (max < higraph.m_rowweight) {
					max = higraph.m_rowweight;
			}	}
			// Do the reverse way so flip equal weights at each cycle
			for (i = 0; i < row.length; ++i) {
				higraph = row[i];
				if (higraph.m_rowweight == max) {
					--position;
					if (higraph.m_position != position) {
						// Assign this row its new position
						higraph.m_position  = position;
						ret                 = true;
					}
					higraph.m_rowweight = -3.0;
					if (position == 0) {
						/*
						if (ret) {
							System.out.println("Baked row");
							dumpRow(0, row);
						}
						*/
						return(ret);
	}	}	}	}	}

	// Returns true if children have been considered for reordering

	private static boolean orderchildren(SimplexLayout options, HiGraph node) throws HiGraphException {

		HiArc		arc;
		HiGraph		child;
		Vector		children = node.m_children;
		int			size     = children.size();
		int			i, direction;
		boolean		ret      = false;
		
//		System.out.println("HiRow.orderChildren " + node);

		node.m_position = 0;
		if (size == 0) {
			return(ret);
		}

		// Order children from the bottom up
		for (i = 0; i < size; ++i) {
			arc = (HiArc) children.elementAt(i);
			child = arc.to();
			ret |= orderchildren(options, child);
		}
		
		// Only reorder at the lowest reorderable level
		if (ret) {
			return(ret);
		}
		

		if (node.dontReorder()) {
			System.out.println("Don't reorder " + node);
			return(true);
		}

		HiGraph board[][]      = buildBoard(node);		// [rank][offset within rank]
		int		cycles         = options.crossing();
		int		to;
		boolean	improved, changed;

		/*
		System.out.println("Initial board");
		dumpBoard(board);
		*/

		to        = board.length;
		if (to > 1) {
			direction = 1;
			improved  = false;
			changed   = false;
			for (i = 0;;) {
				/*
				System.out.println("Shake " + (direction > 0 ? "down" : "up"));
				dumpRow(i, board[i]);
				*/
				improved |= shake(board[i]);
				// Count each row as a cycle to balance load on graphs with many rows
				cycles -= to;
				if (cycles < 0) {
					break;
				}
				i        += direction;
				if (i < 0 || i >= to) {
					if (!improved) {
						break;
					}
					improved  = false;
					changed   = true;
					direction = -direction;
					i        += 2*direction;
			}	}

			if (changed) {
				SortVector.byPosition(children);
		}	}
		
//		System.out.println("After ordering lowest level of descendant");
//		dumpBoard(node, board);

		return(true);
	}
	
	static void order(SimplexLayout options, HiGraph root) throws HiGraphException 
	{
		orderchildren(options, root);

//		System.out.println("After ordering children");
//		root.dump();
	}
}
