package lsedit;

import java.awt.Color;
import java.awt.Graphics;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import javax.swing.JComponent;
import javax.swing.undo.UndoableEdit;

// Don't put any things in this class that can be avoided -- thousands of relations

public abstract class LandscapeObject {

	public  final static String COLOR_ID       =	 "color";
	public  final static String LABEL_COLOR_ID =	 "labelcolor";
	public  final static String OPEN_COLOR_ID  =	 "opencolor";
	public  final static String STYLE_ID    	=    "style";

	private final static int    MAX_RGB         = 255; 

	public static LandscapeObject	g_infoShown = null;	// Last identified thing mouse moved over

	/* All things that are not LandscapeClassObjects must have a parentClass
	 * Otherwise we can't getDiagram() for object in diagram
	 */

	// The class this entity belongs to (if entity or relation)
	// Probably null if a LandscapeClassObject

	private LandscapeClassObject	m_parentClass;				

	/* Logical color's need to be kept separate from the JComponent colors
	 * otherwise when we paint a component we set its actual color to its
	 * logical color loosing the fact that it may not have had a logical color
	 * to begin with (ie. null->red if the class was red)
	 */

	private Color					m_objectColor  = null;		// 
	private int						m_style        = -1;		// No style Not currently used by EntityInstance!
	private	JComponent				m_swingObject;				// The swing object that represents this thing in whatever

	// Attribute database

	private Hashtable				m_attributes;

	protected String qt(String str) 
	{
		return AttributeValueItem.qt(str);
	}

	private static String taColor(Color color1)
	{  
		int		red, green, blue, alpha; 
		String	ret;

		red    = color1.getRed();
		green  = color1.getGreen();
		blue   = color1.getBlue();
		alpha  = color1.getAlpha();

		ret  = "(" + red + " " + green + " " + blue;
		if (alpha != 255) {
			ret += " " + alpha;
		}
		ret += ")"; 
		return ret;
	}

	protected boolean logEdit(UndoableEdit undoableEdit)
	{
		return getDiagram().logEdit(undoableEdit);
	}

	protected boolean undoEnabled()
	{
		return getTa().undoEnabled();
	}
	
	// --------------
	// Public methods 
	// --------------

	public LandscapeObject() 
	{
	}

	public JComponent getSwingObject()
	{
		return m_swingObject;
	}

	public void setSwingObject(JComponent swingObject)
	{
		m_swingObject = swingObject;
	}

	public LandscapeClassObject getParentClass()
	{
		return m_parentClass;
	}
	
	public LandscapeClassObject derivedFrom(int i)
	{
		return(i == 0 ? m_parentClass : null);
	}

	public void setParentClass(LandscapeClassObject parentClass)
	{
		m_parentClass = parentClass;
	}

	class SetParentClass extends MyUndoableEdit implements UndoableEdit
	{
		LandscapeClassObject m_old;
		LandscapeClassObject m_new;

		SetParentClass(LandscapeClassObject old)
		{
			m_old = old;
			m_new = m_parentClass;
			logEdit(this);
		}

		public String getPresentationName() 
		{
			return LandscapeObject.this + " ParentClass " + m_new;
		}

		public void undo()
		{
			setParentClass(m_old);
		}

		public void redo()
		{
			setParentClass(m_new);
		}
	}	

	public void updateParentClass(LandscapeClassObject parentClass)
	{
		LandscapeClassObject old = m_parentClass;

		if (parentClass != old) {
			setParentClass(parentClass);
			if (undoEnabled()) {
				new SetParentClass(old);
	}	}	}

	// Entities and Relations obtain the diagram from their class..
	// This avoids a reference to the diagram on every edge/entity at small cost

	public Ta getTa()
	{
		return(m_parentClass.getTa());
	}

	public Diagram getDiagram()
	{
		return(m_parentClass.getDiagram());
	}

	public int getStyle()
	{
		return m_style;
	}

	public void setStyle(int value)
	{
		m_style = value;
	}

	class SetStyle extends MyUndoableEdit implements UndoableEdit
	{
		int                  m_old;
		int                  m_new;

		SetStyle(int old)
		{
			m_old = old;
			m_new = getStyle();
			logEdit(this);
		}

		public String getPresentationName() 
		{
			return LandscapeObject.this + " Style " + getStyleName(m_new);
		}

		public void undo()
		{
			setStyle(m_old);
		}

		public void redo()
		{
			setStyle(m_new);
	}	}

	public void updateStyle(int value)
	{
		int old = m_style;

		if (value != old) {
			setStyle(value);
			if (undoEnabled()) {
				new SetStyle(old);
	}	}	}

	/* This is broken re multiple inheritence rules IJD */

	public int getInheritedStyle() 
	{
		int	ret, i;

		ret = getStyle();
		if (ret < 0) {
			LandscapeClassObject	superclass;

			for (i = 0; (superclass = derivedFrom(i)) != null; ++i) {
				ret = superclass.getInheritedStyle();
				if (ret >= 0) {
					break;
		}	}	}
		return ret;
	}

	public String getStyleName(int style)
	{
		return "";
	}


	// Relations and RelationClasses can't be opened

	public Color getColorWhenOpen()
	{
		return null;
	}

	public Color getInheritedColorWhenOpen() 
	{
		return null;
	}

	public void setColorWhenOpen(Color color) 
	{
		System.out.println("Can't setColorWhenOpen(" + color + ") on " + this);
	}

	public Color getObjectColor()
	{
		return m_objectColor;
	}

	public void setObjectColor(Color color) 
	{
		m_objectColor = color;
	}

	public Color getInheritedObjectColor() 
	{
		Color	ret;

		ret = getObjectColor();
		if (ret == null) {
			LandscapeClassObject	superclass;
			int						i;

			for (i = 0; (superclass = derivedFrom(i)) != null; ++i) {
				ret = superclass.getInheritedObjectColor();
				if (ret != null) {
					break;
		}	}	}
		return ret;
	}

	class SetObjectColor extends MyPaintableUndoableEdit implements UndoableEdit
	{
		Color                m_old;
		Color                m_new;

		SetObjectColor(Color old)
		{
			m_old = old;
			m_new = getObjectColor();
			logEdit(this);
		}

		public String getPresentationName() 
		{
			return LandscapeObject.this + " ObjectColor ";
		}

		public void undo()
		{
			setObjectColor(m_old);
		}

		public void redo()
		{
			setObjectColor(m_new);
		}

		public void paintComponent(Graphics g, int x, int y)
		{
			paintComponentColor(g, x, y, m_new);
		}
	
		public int getPreferredWidth()
		{
			return(getPreferredWidthColor(LandscapeObject.this));
		}
	}

	public void updateObjectColor(Color color)
	{
		Color oldcolor = getObjectColor();

		if (color == null) {
			if (oldcolor == null) {
				return;
			}
		} else if (color.equals(oldcolor)) {
			return;
		}
		setObjectColor(color);
		if (undoEnabled()) {
			new SetObjectColor(oldcolor);
	}	}

	// Relations don't have labels

	public Color getLabelColor()
	{
		return null;
	}

	public Color getInheritedLabelColor() 
	{
		return null;
	}

	public void setLabelColor(Color color) 
	{
		System.out.println("Can't setLabelColor(" + color + ") on " + this);
	}

	public void updateLabelColor(Color color) 
	{
		System.out.println("Can't updateLabelColor(" + color + ") on " + this);
	}

	// Return the attribute with the associated id
	// Avoid name collision potential with Swing
	// Allow attributes to be null

	public int getLsAttributesSize()
	{
		if (m_attributes == null) {
			return 0;
		}
		return m_attributes.size();
	}

	public Attribute getLsAttribute(String id) 
	{
		if (m_attributes != null) {
			return (Attribute) m_attributes.get(id);
		}
		return(null);
	}

	public Attribute getLsAttributeAt(int index)
	{
		if (m_attributes != null) {
			Enumeration	en;
			Object		attr;

			for (en = m_attributes.elements(); en.hasMoreElements(); --index) {
				attr = en.nextElement();
				if (index == 0) {
					return((Attribute) attr);
		}	}	}
		return(null);
	}

	public Enumeration getLsAttributesEnum()
	{
		if (m_attributes == null) {
			return null;
		}
		return (m_attributes.elements());
	}

	public void putLsAttribute(Attribute attr)
	{
		if (m_attributes == null) {
			m_attributes = new Hashtable(5);
		}
		m_attributes.put(attr.id, attr);
	}

	public void addAttribute(Attribute attr) {

		if (processFirstOrder(attr)) { 
			return;
		}

		Attribute curAttr = getLsAttribute(attr.id);

		if (curAttr != null) {
			// Replace any old attributes with overrides
			m_attributes.remove(attr.id);
		}
		putLsAttribute(attr);
	}

	public void regRawAttribute(Vector v, Hashtable st, String id, Object type, Object val)
	{
		Object[] ent = new Object[3];

		ent[0] = st.get(id);
		ent[1] = type;
		ent[2] = val;

		v.addElement(ent);
	}

	public void regRawAttribute(Vector v, Hashtable st, String id, int val) 
	{
		regRawAttribute(v, st, id, Attribute.INT, new Integer(val));
	}

	public void regRawAttribute(Vector v, Hashtable st, String id, double val) 
	{
		regRawAttribute(v, st, id, Attribute.DOUBLE, new Double(val));
	}

	public void regRawAttribute(Vector v, Hashtable st, String id, String val) 
	{
		regRawAttribute(v, st, id, Attribute.STRING, val);
	}

	public void regRawAttribute(Vector v, Hashtable st, String id, int[] val)
	{
		regRawAttribute(v, st, id, Attribute.INT_LIST, val);
	}

	public void regRawAttribute(Vector v, Hashtable st, String id, double[] val)
	{
		regRawAttribute(v, st, id, Attribute.DOUBLE_LIST, val);
	}

	public void regRawAttribute(Vector v, Hashtable st, String id, String[] val)
	{
		regRawAttribute(v, st, id, Attribute.STRING_LIST, val);
	}

	public void regAttributeRaw(Vector v, Hashtable st, Attribute attr, boolean classType)
	{
		if (attr.avi == null) {
			if (classType) {
				regRawAttribute(v, st, attr.id, Attribute.NULL, null);
			}
		}
		else if (classType || !attr.m_cloneOnAssign) {
			if (attr.avi.next != null) {
				AttributeValueItem avi;
				int i;
				int num = 0;

				for (avi = attr.avi; avi != null; ++num) {
					avi = avi.next;
				}
				String[] sl = new String[num];
				avi = attr.avi;
				i = 0;
				for (avi = attr.avi; avi != null; ++i) {
					sl[i] = avi.value;
					avi   = avi.next;
				}
				regRawAttribute(v, st, attr.id, sl);
			} else {
				regRawAttribute(v, st, attr.id, attr.avi.value);
			}
		}
	}

	public void regRawAttribute(Vector v, Hashtable st, String id, Color c) 
	{
		int red = c.getRed();
		int green = c.getGreen();
		int blue = c.getBlue();

		double[] dl = new double[3];

		dl[0] = ((double) red)/((double) MAX_RGB);
		dl[1] = ((double) green)/((double) MAX_RGB);
		dl[2] = ((double) blue)/((double) MAX_RGB);

		regRawAttribute(v, st, id, dl);
	}

	// If a new attribute, add it to the attribute database,
	// otherwise merely update the value.

	public boolean processFirstOrder(Attribute attr) 
	{
		if (attr.id.equals(COLOR_ID)) {
			setObjectColor(attr.parseColor());
			return true;
		}

		if (attr.id.equals(LABEL_COLOR_ID)) {
			setLabelColor(attr.parseColor());
			return true;
		}

		if (attr.id.equals(OPEN_COLOR_ID)) {
			setColorWhenOpen(attr.parseColor());
			return true;
		}
		return false;
	}

	public void replaceAttribute(Attribute newAttr)
	{
		addAttribute(newAttr);
	}



	public void writeAttributes(PrintStream ps, LandscapeObject parentClass, boolean classType) throws IOException 
	{
		Color color, color1;
		Enumeration en;

		// Output the body of the attribute record 
		// The child method outputs the header and tail 

		color = getObjectColor();
		if (color != null) {
			if (parentClass != null) {
				color1 = parentClass.getInheritedObjectColor();
			} else {
				color1 = null;
			}
			if (!color.equals(color1)) {
				ps.print(Attribute.indent + COLOR_ID + " = " + taColor(color) + "\n"); 
		}	}

		color = getLabelColor();
		if (color != null) { 
			if (parentClass != null) {
				color1 = parentClass.getInheritedLabelColor();
			} else {
				color1 = null;
			}
			if (!color.equals(color1)) {
				ps.print(Attribute.indent + LABEL_COLOR_ID + " = " + taColor(color) + "\n");
		}	}

		color = getColorWhenOpen();
		if (color != null) { 
			color = getInheritedColorWhenOpen();
			if (parentClass != null) {
				color1 = parentClass.getInheritedColorWhenOpen();
			} else {
				color1 = null;
			}
			if (!color.equals(color1)) {
				ps.print(Attribute.indent + OPEN_COLOR_ID + " = " + taColor(color) + "\n");
		}	}

		if (m_attributes != null) {
			for (en = m_attributes.elements(); en.hasMoreElements(); ) {
				Attribute attr = (Attribute) en.nextElement();
				attr.writeAttribute(ps, parentClass, classType);
		}	}
	}

	// The routines that follow hide the complexity of getting/setting attribute values
	// from EditAttributes

	public int getPrimaryAttributeCount()
	{
		return(0);
	}

	public int getLsAttributeCount()
	{
		return(getPrimaryAttributeCount() + getLsAttributesSize() + 1);
	}

	public boolean canEditName(int index)
	{
		int	primary = getPrimaryAttributeCount();

		if (index < primary) {
			return(false);
		}
		return(true);
	}

	// Use overloading to set special rules

	public boolean canEditAttribute(int index)
	{
		int	primary;

		primary =  getPrimaryAttributeCount();
		if (index < primary) {
			return(true);
		}
		index -= primary;
		if (index < getLsAttributesSize()) {
			return(true);
		}
		return(false);
	}

	public String getLsAttributeNameAt(int index)
	{
		Attribute	attr;

		index -= getPrimaryAttributeCount();
		if (index == getLsAttributesSize()) {
			// Dummy row to allow insertion
			return("");
		}
		attr   = getLsAttributeAt(index);
		if (attr == null) {
			return(null);
		}
		return(attr.id);
	}

	public Object getLsAttributeValueAt(int index)
	{
		Attribute	attr;

		index -= getPrimaryAttributeCount();
		if (index == getLsAttributesSize()) {
			// Dummy row to allow insertion
			return("");
		}
		attr   = getLsAttributeAt(index);
		if (attr == null) {
			return(null);
		}
		return(attr.avi);
	}

	private boolean unknownAttributeName(String name)
	{
		int			i, primary;
		String		name1;

		for (i = 0; ; ++i) {
			name1 = getLsAttributeNameAt(i);
			if (name1 == null) {
				break;
			}
			if (name.equals(name1)) {
				System.out.println("Can't rename generic attribute to '" + name + "': attribute already exists");
				return(false);
		}	} 
		return(true);
	}

	public void setAttributeName(Attribute attribute, String oldName, String newName)
	{
		if (oldName != null) {
			m_attributes.remove(oldName);
		}
		if (newName != null) {
			attribute.id = newName;
			putLsAttribute(attribute);
		} else {
			if (getLsAttributesSize() == 0) {
				m_attributes = null;
	}	}	}

	class SetAttributeName extends MyUndoableEdit implements UndoableEdit
	{
		Attribute			 m_attribute;
		String				 m_oldName;
		String				 m_newName;

		SetAttributeName(Attribute attribute, String oldName, String newName)
		{
			m_attribute = attribute;
			m_oldName   = oldName;
			m_newName   = newName;
			logEdit(this);
		}

		public String getPresentationName() 
		{
			if (m_newName == null) {
				return LandscapeObject.this + " delete Attribute " + m_oldName;
			}
			if (m_oldName == null) {
				return LandscapeObject.this + " create Attribute " + m_newName;
			}
			return LandscapeObject.this + " rename Attribute " + m_oldName + " to " + m_newName;
		}

		public void undo()
		{
			setAttributeName(m_attribute, m_newName, m_oldName);
		}

		public void redo()
		{
			setAttributeName(m_attribute, m_oldName, m_newName);
	}	}

	public boolean updateAttributeNameAt(int index, Object value)
	{
		Attribute	attr;
		String		name;
		int			primary;

		primary = getPrimaryAttributeCount();

		if (index < primary) {
			// Can't change names of first order attributes
			return(false);
		}
		index -= primary;

		name = (String) value;
		if (index == getLsAttributesSize()) {
			if (name == null || name.equals("")) {
				// Dummy row hasn't changed
				return(false);
			}
			if (!unknownAttributeName(name)) {
				return(false);
			} 
			attr = new Attribute(name, null);
			setAttributeName(attr, null, name);
			if (undoEnabled()) {
				new SetAttributeName(attr, null, name);
			}
			return(true);
		}

		attr = getLsAttributeAt(index);
		if (attr == null) {
			return(false);
		}

		if (name.equals("")) {
			// Remove this attribute
			setAttributeName(attr, attr.id, null);
			if (undoEnabled()) {
				new SetAttributeName(attr, attr.id, null);
			}
			return(true);
		}
		if (name.equals(attr.id)) {
			return(false);
		}
		if (!unknownAttributeName(name)) {
			return(false);
		} 
		setAttributeName(attr, attr.id, name);
		if (undoEnabled()) {
			new SetAttributeName(attr, attr.id, name);
		}
		return(true);
	}


	class SetAttributeValue extends MyUndoableEdit implements UndoableEdit
	{
		AttributeValueItem	 m_oldValue;
		AttributeValueItem	 m_newValue;
		Attribute			 m_attribute;

		SetAttributeValue(Attribute attribute, AttributeValueItem oldValue)
		{
			m_attribute = attribute;
			m_oldValue  = oldValue;
			m_newValue  = attribute.avi;
			logEdit(this);
		}

		public String getPresentationName() 
		{
			LandscapeObject o = LandscapeObject.this;

			if (m_newValue == null) {
				return o + " remove Attribute " + m_attribute.id + " value";
			}
			if (m_oldValue == null) {
				return o + " add Attribute " + m_attribute.id + " value";
			}
			return o + " replace Attribute " +  m_attribute.id + " value";
		}

		public void undo()
		{
			m_attribute.setValue(m_oldValue);
		}

		public void redo()
		{
			m_attribute.setValue(m_newValue);
	}	}

	public void updateAttributeValueAt(int index, Object value)
	{
		Attribute			attr;
		AttributeValueItem	old;

		index -= getPrimaryAttributeCount();
		attr   = getLsAttributeAt(index);
		if (attr == null) {
			return;
		}
		old = attr.avi;
		attr.setValue((AttributeValueItem) value);
		if (undoEnabled()) {
			new SetAttributeValue(attr, old);
	}	}


	// Need to know the type in cases where value might be null
	// For example with some colors

	public int getLsAttributeTypeAt(int index)
	{
		if (index < getLsAttributeCount() - 1) {
			return(Attribute.AVI_TYPE);
		}
		// This attribute does not yet exist
		return(Attribute.NULL_TYPE);
	}

	public int getLsAttributeOffset(String id)
	{
		String	name;
		int		i;

		for (i = 0; ; ++i) {
			name = getLsAttributeNameAt(i);
			if (name == null) {
				return(-1);
			}
			if (name.equals(id)) {
				return(i);
	}	}	}

	public boolean defaultValue(String id, Object object)
	{
		if (object != null) {
			int		i;
			Object	object1;

			i = getLsAttributeOffset(id);
			if (i >= 0) {
				object1 = getLsAttributeValueAt(i);
				if (object.equals(object1)) {
					return(true);
		}	}	}
		return(false);
	}
}

