package lsedit;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Container;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import java.io.BufferedWriter;
import java.io.IOException;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

public class ArrowDimensions extends JDialog implements ActionListener { 

	private static final int	LINE_WIDTH   = 0;
	private static final int	ARROW_LENGTH = 1;
	private static final int	ARROW_ARC    = 2;
	private static final int	PIXELS_3D    = 3;
	private static final int	SHADOW_SIZE  = 4;
	private static final int    LABEL_ARC    = 5;


	protected final static String[] m_textfield_tags = 
							{
								"arrow:linewidth[",
								"arrow:linelength[",
								"arrow:arc[",
								"arrow:3dpixels[",
								"arrow:shadow[",
								"arrow:labelarc["
							};

	protected final static String[] m_textfield_titles = 
							{
								"Width of edge:",
								"Length of arrow edge:",
								"Arc between edge and arrow edge:",
								"3D highlighting weight:",
								"Incidental shadow size:",
								"Angle of label:"
							};

	protected final static String[] m_textfield_units = 
							{
								"Pixels",
								"Pixels",
								"Radians",
								"Pixels",
								"Pixels",
								"Degrees"
							};

	protected final static String[] m_textfield_resets = 
							{
								"1",
								"10.0",
								"0.4",
								"3",
								"5",
								"0.0"
							};

	protected static String[] m_textfield_defaults = 
							{
								"1",
								"10.0",
								"0.4",
								"3",
								"5",
								"0.0"
							};

	protected static String[] m_textfield_currents = 
							{
								"1",
								"10.0",
								"0.4",
								"3",
								"5",
								"0.0"
							};

	public	static int		m_line_width   = 1;
	public	static double	m_arrow_length = 10.0;
	public	static double	m_arrow_arc    = 0.4;
	public	static int		m_pixels_3d    = 3;
	public  static int		m_shadow_size  = 5;
	public  static double   m_label_angle  = 0.0;

	private static final int		FILL_ARROWHEAD     = 0;
	private static final int		CENTER_ARROWHEAD   = 1;
	private static final int		WEIGHTED_ARROWHEAD = 2;
	private static final int		BLACKWHITE_3D      = 3;
	private static final int		SHOW_EDGE_LABEL    = 4;
	private static final int        ROTATE_EDGE_LABEL  = 5;
	private static final int		SHOW_EDGE_TOOLTIP  = 6;
	private static final int		VARIABLE_ARROW_COLOR = 7;
	private static final int		INVERT_EDGE_LABEL_BACKGROUND = 8;
	private static final int		LEGEND_LABEL_BLACK = 9;
	private static final int		LABEL_INVERT_FORE  = 10;
	private static final int		LABEL_INVERT_BACK  = 11;
	private static final int		LABEL_BLACK_WHITE  = 12;


	protected final static String[] m_checkbox_tags = 
							{
								"arrow:fill[",
								"arrow:center[",
								"arrow:weighted[",
								"arrow:blackwhite[",
								"arrow:edgelabel[",
								"arrow:rotate[",
								"arrow:edgetip[",
								"arrow:color[",
								"arrow:invert[",
								"arrow:legend[",
								"arrow:labelfg[",
								"arrow:labelbg[",
								"arrow:labelbw[",
							};

	protected final static String[] m_checkbox_titles = 
							{
								 "Fill arrow head:",
								 "Center arrow head:",
								 "Permanently weight arrow head:",
								 "3D in Black&White:",
								 "Show fixed edge labels:",
								 "Edge labels angle w.r.t edge:",
								 "Show edge tooltip:",
								 "Variable arrow color:",
								 "Invert edge label background:",
								 "Legend labels black:",
								 "Label color inverts background",
								 "Invert entity label background",
 								 "Make entity label black/white"
							};

	protected final static boolean[] m_checkbox_resets = 
							{
								true,
								false,
								true,
								true,
								false,
								false,
								false,
								false,
								false,
								false,
								false,
								false,
								false
							};

	protected static boolean[] m_checkbox_defaults = 
							{
								true,
								false,
								true,
								true,
								false,
								false,
								false,
								false,
								false,
								false,
								false,
								false,
								false
							};

	protected static boolean[] m_checkbox_currents = 
							{
								true,
								false,
								true,
								true,
								false,
								false,
								false,
								false,
								false,
								false,
								false,
								false,
								false
							};



	static protected final int BUTTON_OK      = 0;
	static protected final int BUTTON_CANCEL  = 1;
	static protected final int BUTTON_DEFAULT = 2;
	static protected final int BUTTON_SET     = 3;
	static protected final int BUTTON_RESET   = 4;

	protected final static String[] m_button_titles =
							{
								"Ok",
								"Cancel",
								"Default",
								"Set",
								"Reset"
							};

	protected final static String[] m_button_tips =
							{
								null,
								null,
								"Use remembered default",
								"Set default to current",
								"Set default to initial"
							};

	protected LandscapeEditorCore	m_ls;
	protected JTextField[]			m_textfields;
	protected JCheckBox[]			m_checkboxes;
	protected JLabel				m_message;
	protected JButton[]				m_buttons;
	protected boolean				m_ret;

	protected static void setNumericParameter(int i, String string)
	{
		try {
			string = string.trim();
			switch (i) {
				case LINE_WIDTH:
				case PIXELS_3D:
				case SHADOW_SIZE:
				{
					int ival = Integer.parseInt(string);
					switch(i) {
					case LINE_WIDTH:
						m_line_width = ival;
						break;
					case PIXELS_3D:
						m_pixels_3d = ival;
						break;
					case SHADOW_SIZE:
						m_shadow_size = ival;
					}
					break;
				}
				default:
				{
					double dval = Double.parseDouble(string);
					switch (i) {
					case ARROW_LENGTH:
						m_arrow_length = dval;
						break;
					case ARROW_ARC:
						m_arrow_arc = dval;
						break;
					case LABEL_ARC:
						m_label_angle = Math.toRadians(dval);
						break;
			}	}	}
			m_textfield_currents[i] = string;
		} catch (Throwable exception) {
		}
	}

	protected static void setBooleanParameter(int i, boolean value)
	{
		m_checkbox_currents[i] = value;
	}

	public static void load(String string)
	{
		String	tag;
		int		i;
		int		j;
		boolean	bool;

		for (i = 0; i < m_textfield_tags.length; ++i) {
			tag = m_textfield_tags[i];
			if (string.startsWith(tag)) {
				string = string.substring(tag.length());
				if (string.charAt(1) != ']' || string.charAt(2) != '=') {
					return;
				}
				j      = string.charAt(0) - '0';
				string = string.substring(3);
				switch (j) {
				case 0:
					m_textfield_defaults[i] = string;
				case 1:
					setNumericParameter(i, string);
				}
				return;
		}	}
		for (i = 0; i < m_checkbox_tags.length; ++i) {
			tag = m_checkbox_tags[i];
			if (string.startsWith(tag)) {
				string = string.substring(tag.length());
				if (string.charAt(1) != ']' || string.charAt(2) != '=') {
					return;
				}
				j      = string.charAt(0) - '0';
				string = string.substring(3);
				bool   = ((string.charAt(0) == 't') ? true : false);
				switch (j) {
				case 0:
					m_checkbox_defaults[i] = bool;
					break;
				case 1:
					setBooleanParameter(i, bool);
					break;
				}
				return;
		}	}
	}

	public static void save(BufferedWriter bufferedWriter) throws IOException
	{
		String	tag, string, string1;
		int		i, j;
		String	prior_strings[];
		String	emit_strings[];
		boolean prior_booleans[];
		boolean emit_booleans[];
		boolean	bool;


		prior_strings  = m_textfield_resets;
		prior_booleans = m_checkbox_resets;
		emit_strings   = m_textfield_defaults;
		emit_booleans  = m_checkbox_defaults;

		for (j = 0; j < 2; ++j) {
			for (i = 0; i < m_textfield_tags.length; ++i) {
				string = emit_strings[i];
				if (string.equals(prior_strings[i])) {
					continue;
				}
				tag = m_textfield_tags[i];
				bufferedWriter.write(tag, 0, tag.length());
				string1 = "" + j + "]=";
				bufferedWriter.write(string1, 0, string1.length());
				bufferedWriter.write(string, 0, string.length());
				bufferedWriter.newLine();
			}
			for (i = 0; i < m_checkbox_tags.length; ++i) {
				bool = emit_booleans[i];
				if (bool == prior_booleans[i]) {
					continue;
				}
				tag = m_checkbox_tags[i];
				bufferedWriter.write(tag, 0, tag.length());
				string1 = "" + j + "]=";
				bufferedWriter.write(string1, 0, string1.length());
				string = (bool ? "true" : "false");
				bufferedWriter.write(string, 0, string.length());
				bufferedWriter.newLine();
			}

			prior_strings  = emit_strings;
			prior_booleans = emit_booleans;
			emit_strings   = m_textfield_currents;
			emit_booleans  = m_checkbox_currents;
		}
	}	

	public static boolean isFillArrowhead()
	{
		return m_checkbox_currents[FILL_ARROWHEAD];
	}

	public static boolean isCenterArrowhead()
	{
		return m_checkbox_currents[CENTER_ARROWHEAD];
	}

	public static boolean isWeightedArrowhead()
	{
		return m_checkbox_currents[WEIGHTED_ARROWHEAD];
	}

	public static boolean isBlackWhite3D()
	{
		return m_checkbox_currents[BLACKWHITE_3D];
	}

	public static boolean isShowEdgeLabel()
	{
		return m_checkbox_currents[SHOW_EDGE_LABEL];
	}

	public static boolean isRotateEdgeLabel()
	{
		return m_checkbox_currents[ROTATE_EDGE_LABEL];
	}

	public static boolean isShowEdgeTooltip()
	{
		return m_checkbox_currents[SHOW_EDGE_TOOLTIP];
	}

	public static boolean isVariableArrowColor()
	{
		return m_checkbox_currents[VARIABLE_ARROW_COLOR];
	}

	public static boolean isInvertEdgeLabelBackground()
	{
		return m_checkbox_currents[INVERT_EDGE_LABEL_BACKGROUND];
	}

	public static boolean isLegendLabelBlack()
	{
		return m_checkbox_currents[LEGEND_LABEL_BLACK];
	}

	public static boolean isLabelInvertForeground()
	{
		return m_checkbox_currents[LABEL_INVERT_FORE];
	}

	public static boolean isLabelBlackWhite()
	{
		return m_checkbox_currents[LABEL_BLACK_WHITE];
	}

	public static boolean isLabelInvertBackground()
	{
		return m_checkbox_currents[LABEL_INVERT_BACK];
	}

	protected ArrowDimensions(LandscapeEditorCore ls) //Constructor
	{
		super(ls.getFrame(), "Modify visualization settings",true); //false if non-modal

		Container	contentPane;
		JLabel		label;
		Font		font, bold;
		JTextField	textfield;
		JCheckBox	checkbox;
		JButton		button;
		int			i;
		String		tip;

		m_ls         = ls;
		m_ret        = false;
		font         = FontCache.getDialogFont();
		bold         = font.deriveFont(Font.BOLD);

		contentPane = getContentPane();
        contentPane.setLayout( new BorderLayout() );

		setForeground(ColorCache.get(0,0,0));
		setBackground(ColorCache.get(192,192,192));

		setFont(font);

		JPanel topPanel    = new JPanel();
		JPanel leftPanel   = new JPanel();
		JPanel centerPanel = new JPanel();
		JPanel rightPanel  = new JPanel();

		GridLayout gridLayout;

		topPanel.setLayout( new BorderLayout() );
		gridLayout = new GridLayout(19,1);
		gridLayout.setVgap(10);
		leftPanel.setLayout( gridLayout);

		gridLayout = new GridLayout(19,1);
		gridLayout.setVgap(10);
		centerPanel.setLayout( gridLayout);

		gridLayout = new GridLayout(19,1);
		gridLayout.setVgap(10);
		rightPanel.setLayout( gridLayout);

		topPanel.add( BorderLayout.WEST, leftPanel);
		topPanel.add( BorderLayout.EAST, rightPanel);
		topPanel.add( BorderLayout.CENTER, centerPanel);

		// ----------

		m_textfields = new JTextField[m_textfield_tags.length];
		for (i = 0; i < m_textfield_tags.length; ++i) {
			label = new JLabel(m_textfield_titles[i], JLabel.RIGHT);
			label.setFont(bold);
			leftPanel.add(label);
			m_textfields[i] = textfield = new JTextField(m_textfield_currents[i],  12);
			textfield.setFont(font);
			textfield.addActionListener(this);
			centerPanel.add(textfield);
			label = new JLabel(m_textfield_units[i], JLabel.LEFT);
			label.setFont(bold);
			rightPanel.add(label);
		}

		m_checkboxes = new JCheckBox[m_checkbox_tags.length];

		for (i = 0; i < m_checkbox_tags.length; ++i) {
			label = new JLabel(m_checkbox_titles[i], JLabel.RIGHT);
			label.setFont(bold);
			leftPanel.add(label);
			m_checkboxes[i] = checkbox = new JCheckBox("", m_checkbox_currents[i]);
			checkbox.setFont(font);
			centerPanel.add(checkbox);
		}

		// --------

        contentPane.add( BorderLayout.NORTH, topPanel );

		// ------------

		m_message = new JLabel(" ", JLabel.CENTER);
		m_message.setFont(font);
		m_message.setForeground(Color.RED);

		m_message.setSize(400,50);
		contentPane.add( BorderLayout.CENTER, m_message);

		// --------------
		// Use a FlowLayout to center the button and give it margins.

		JPanel bottomPanel = new JPanel();

		bottomPanel.setLayout(new FlowLayout(FlowLayout.CENTER, 15, 15));

		m_buttons = new JButton[m_button_titles.length];
		for (i = 0; i < m_button_titles.length; ++i) {
			m_buttons[i] = button = new JButton(m_button_titles[i]);
			button.setFont(bold);
			tip = m_button_tips[i];
			if (tip != null) {
				button.setToolTipText(tip);
			}
			button.addActionListener(this);
			bottomPanel.add(button);
		}

		contentPane.add( BorderLayout.SOUTH, bottomPanel);

		// Resize the window to the preferred size of its components
		pack();
		setVisible(true);
	}

	public boolean ok()
	{
		return m_ret;
	}

	// ActionListener interface

	public void actionPerformed(ActionEvent ev)
	{
		Object			source;
		JTextField		textfield;
		String			string;
		String			name;
		int				state, i;

		// Pop down the window when the button is clicked.
		// System.out.println("event: " + ev);

		source = ev.getSource();

		state = -1;
		for (i = 0; i < m_button_titles.length; ++i) {
			if (source == m_buttons[i]) {
				state = i;
				break;
		}	}

		switch (state) {
		case BUTTON_RESET:
			for (i = 0; i < m_textfield_tags.length; ++i) {
				m_textfield_defaults[i] = m_textfield_resets[i];
			}
			for (i = 0; i < m_checkbox_tags.length; ++i) {
				m_checkbox_defaults[i]  = m_checkbox_resets[i];
			}
		case BUTTON_DEFAULT:

			for (i = 0; i < m_textfield_tags.length; ++i) {
				m_textfields[i].setText(m_textfield_defaults[i]);
			}
			for (i = 0; i < m_checkbox_tags.length; ++i) {
				m_checkboxes[i].setSelected(m_checkbox_defaults[i]);
			}
			m_message.setText("Values set to initial default");
			return;
		case BUTTON_SET:
			for (i = 0; i < m_textfield_tags.length; ++i) {
				m_textfield_defaults[i] = m_textfields[i].getText();
			}
			for (i = 0; i < m_checkbox_tags.length; ++i) {
				m_checkbox_defaults[i] = m_checkboxes[i].isSelected();
			}
			return;
		default:
			for (i = 0; i < m_textfield_tags.length; ++i) {
				textfield = m_textfields[i];
				if (source == textfield || state == BUTTON_OK) {
					string = textfield.getText();
					string = string.trim();
					name   = m_textfield_titles[i];

					switch (i) {
					case LINE_WIDTH:
					case PIXELS_3D:
					case SHADOW_SIZE:
						try {
							int ival = Integer.parseInt(string);

							if (ival < 0) {
								m_message.setText(name + " may not be negative");
								return;
							}
							if (ival > 100) {
								m_message.setText(name + " may not exceed 99");
								return;
							}
						} catch (Throwable exception) {
							m_message.setText(name + " not an integer");
							return;
						}
						break;
					case ARROW_LENGTH:
					case ARROW_ARC:
						try {
							double dval;

							dval = Double.parseDouble(string);
							if (Double.isNaN(dval)) {
								m_message.setText("Length is not a double precision number");
								return;
							}
							if (dval < 0) {
								m_message.setText(name + " may not be negative");
								return;
							}
							switch (i) {
							case ARROW_ARC:
								if (dval > 1.5) {
									m_message.setText("The maximum allowed arc angle is 1.5 (very near pi/2)");
									return;
							}	}
						} catch (Throwable exception) {
							m_message.setText(name + " not a double precision value");
							return;
		}	}	}	}	}

		switch (state) {
		case BUTTON_OK:
			for (i = 0; i < m_textfield_tags.length; ++i) {
				setNumericParameter(i, m_textfields[i].getText());
			}
			for (i = 0; i < m_checkbox_tags.length; ++i) {
				setBooleanParameter(i, m_checkboxes[i].isSelected());
			}
			m_ret = true;
		case BUTTON_CANCEL:
			break;
		default:
			return;
		}

		setVisible(false);
		return;
	}

	public static void create(LandscapeEditorCore ls) 
	{
		int		old_line_width               = m_line_width;
		double	old_arrow_length             = m_arrow_length;
		double	old_arrow_arc                = m_arrow_arc;
		int		old_pixels_3d                = m_pixels_3d;
		int		old_shadow_size              = m_shadow_size;
		double	old_label_angle              = m_label_angle;

		boolean old_fill_arrowhead_state     = isFillArrowhead();
		boolean	old_center_arrowhead_state   = isCenterArrowhead();
		boolean	old_weighted_arrowhead_state = isWeightedArrowhead();
		boolean	old_blackwhite_3D_state      = isBlackWhite3D();
		boolean old_show_edge_label          = isShowEdgeLabel();
		boolean old_show_edge_tooltip        = isShowEdgeTooltip();

		ArrowDimensions arrowDimensions = new ArrowDimensions(ls);

		if (arrowDimensions.ok()) {

			if (old_show_edge_label              != isShowEdgeLabel()          ||
			    old_show_edge_tooltip            != isShowEdgeTooltip()        ||
				old_label_angle                  != m_label_angle) {
				// Must revalidate the diagram edges
				ls.refillDiagram();
			} else {
				if (old_line_width               != m_line_width               ||
					old_arrow_length             != m_arrow_length             ||
					old_arrow_arc                != m_arrow_arc                ||
					old_pixels_3d                != m_pixels_3d                ||
					old_shadow_size              != m_shadow_size              ||
					old_fill_arrowhead_state     != isFillArrowhead()          ||
					old_center_arrowhead_state   != isCenterArrowhead()        ||
					old_weighted_arrowhead_state != isWeightedArrowhead()      ||
					old_blackwhite_3D_state      != isBlackWhite3D() ) {

					ls.repaint(); 
		}	}	}
		arrowDimensions.dispose();

	}
} 



