package lsedit;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.FlowLayout;
import java.awt.Graphics;
import java.awt.GridLayout;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import java.util.Enumeration;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.DefaultCellEditor;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.border.Border;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;

import javax.swing.Icon;

class ElisionChooser extends JDialog implements ItemListener, ActionListener
{
	private JCheckBox	m_checkboxes[];

	private JButton		m_ok;
	private JButton		m_and  = null;
	private JButton		m_or   = null;
	private JButton		m_nand = null;
	private JButton		m_cancel;
	private int			m_result = -1;
	
	public ElisionChooser(JFrame frame, int elisions, boolean summary)
	{
		super(frame, "Select desired elisions", true);

		Container			contentPane;
		JCheckBox			checkBox;
		ToolBarButton		icon;
		String				desc, state;
		Font				font, bold;
		int					i, mask;
		boolean				flag, flag1;

		mask = elisions;

		font = FontCache.getDialogFont();
		bold = font.deriveFont(Font.BOLD);


		JPanel topPanel    = new JPanel();	

		GridBagLayout		gridBagLayout = new GridBagLayout();
		GridBagConstraints	c             = new GridBagConstraints();

		c.weightx = 1.0;

		topPanel.setLayout(gridBagLayout);

		setForeground(ColorCache.get(0,0,0));
		setBackground(ColorCache.get(192,192,192));
		setFont(font);

		m_checkboxes    = new JCheckBox[EntityInstance.CHANGED_ELISION + 1];

		for (i = 0; i <= EntityInstance.ELISIONS; ++i) {
			flag  = flag1 = ((mask & 1) != 0);
			switch (i) {
			case EntityInstance.CHANGED_ELISION:
				state = "";
				break;
			case EntityInstance.CLOSED_ELISION:
				flag1 = !flag;
			default:
				state = (flag1 ? "Hide" : "Show");
			}
			
			switch (i) {
			case EntityInstance.DST_ELISION:
				icon = new Elision_u_Button(null);
				desc = " destination edges";
				break;
			case EntityInstance.SRC_ELISION:
				icon = new Elision_s_Button(null);
				desc = " source edges";
				break;
			case EntityInstance.ENTERING_ELISION:
				icon = new Elision_CU_Button(null);
				desc = " entering edges";
				break;
			case EntityInstance.EXITING_ELISION:
				icon = new Elision_CS_Button(null);
				desc = " exiting edges";
			case EntityInstance.INTERNAL_ELISION:
				icon = new Elision_I_Button(null);
				desc = " internal edges";
				break;
			case EntityInstance.CLOSED_ELISION:
				icon = new Elision_c_Button(null);
				desc = " children";
				break;
			default:	// EntityInstance.CHANGED_ELISION
				icon = null;
				if (flag) {
					desc = "Set above elisions";
				} else {
					desc = "Preserve elisions unchanged";
				}
			}
			if (i != EntityInstance.CHANGED_ELISION) {
				desc = state + desc;
			}
			
			c.anchor    = GridBagConstraints.EAST;

			if (icon != null) {
				c.gridx = 0;
				gridBagLayout.setConstraints(icon, c);
				topPanel.add(icon);
				c.gridx     = 1;
			} else {
				c.gridx     = 0;
				c.gridwidth = 2;
			}

			m_checkboxes[i] = checkBox = new JCheckBox(desc, flag);
			c.anchor    = GridBagConstraints.WEST;
			c.gridwidth = GridBagConstraints.REMAINDER;

			gridBagLayout.setConstraints(checkBox, c);
			topPanel.add(checkBox);
			checkBox.addItemListener(this);

			c.weightx   = 0.0;
			c.gridwidth = GridBagConstraints.RELATIVE;
			mask >>= 1;
		}

		contentPane = getContentPane();
		contentPane.add( BorderLayout.NORTH, topPanel );

		// --------------
		// Use a FlowLayout to center the button and give it margins.

		JPanel bottomPanel = new JPanel();

		bottomPanel.setLayout(new FlowLayout(FlowLayout.CENTER, 15, 15));

		m_ok = new JButton("Set");
		m_ok.setFont(bold);
		bottomPanel.add(m_ok);
		m_ok.addActionListener(this);

		if (summary) {
			m_and = new JButton("And");
			m_and.setFont(bold);
			bottomPanel.add(m_and);
			m_and.addActionListener(this);

			m_or = new JButton("Or");
			m_or.setFont(bold);
			bottomPanel.add(m_or);
			m_or.addActionListener(this);

			m_nand = new JButton("Nand");
			m_nand.setFont(bold);
			bottomPanel.add(m_nand);
			m_nand.addActionListener(this);

		}

		m_cancel = new JButton("Cancel");
		m_cancel.setFont(bold);
		bottomPanel.add(m_cancel);
		m_cancel.addActionListener(this);

		contentPane.add( BorderLayout.SOUTH, bottomPanel);

		// Resize the window to the preferred size of its components
		pack();
		setVisible(true);
	}

	private int	getNewElisionValue()
	{
		JCheckBox[]	checkboxes = m_checkboxes;
		int			elisions   = checkboxes.length;
		int			ret	       = 0;
		int			mask       = 1;
		JCheckBox	checkbox;
		int			i;

		for (i = 0; i < elisions; ++i) {
			checkbox = checkboxes[i];
			if (checkbox != null && checkbox.isSelected()) {
				ret |= mask;
			}
			mask <<= 1;
		}
		return ret;
	}

	public int getElisions()
	{
		return m_result;
	}

	// ItemListener interface

	public void itemStateChanged(ItemEvent ev)
	{
		JCheckBox[]	checkboxes   = m_checkboxes;
		JCheckBox	checkBox     = (JCheckBox) ev.getItem();
		JCheckBox	unchangedBox = checkboxes[EntityInstance.CHANGED_ELISION];
		boolean		state        = (ev.getStateChange() == ItemEvent.SELECTED);
		String		desc;

		if (checkBox != unchangedBox) {
			String		text       = checkBox.getText();
			String		text1      = text.substring(4);
			boolean		state1     = state;

			if (checkBox == checkboxes[EntityInstance.CLOSED_ELISION]) {
				state1 = !state1;
			}
			checkBox.setText((state1 ? "Hide" : "Show") + text1);

			checkBox = unchangedBox;
			state    = true;
			checkBox.setSelected(state);
		} 
		if (state) {
			desc = "Set above elisions";
		} else {
			desc = "Preserve elisions unchanged";
		}
		checkBox.setText(desc);
 	}

	// ActionListener interface

	public void actionPerformed(ActionEvent ev)
	{
		Object	source = ev.getSource();
		int		action = -2;

		if (source == m_ok) {
			action = 0;
		} else if (source == m_cancel) {
			action = -1;
		} else if (source == m_and) {
			action = SelectedElisions.AND_ELISIONS;
		} else if (source == m_or) {
			action = SelectedElisions.OR_ELISIONS;
		} else if (source == m_nand) {
			action = SelectedElisions.NAND_ELISIONS;
		} else {
			return;
		}
		if (action >= 0) {
			m_result  = getNewElisionValue();
			m_result |= action;
		}
		setVisible(false);
		return;
	}
}

class SelectedElisions extends JLabel implements TableCellRenderer, Icon {

	public static final int DST			 = 0x01;	// Type DST_ELISION      = 0
	public static final int SRC			 = 0x02;	// Type SRC_ELISION      = 1
	public static final int ENTERS		 = 0x04;	// Type ENTERING_ELISION = 2
	public static final int EXITS		 = 0x08;	// Type EXITING_ELISION  = 3
	public static final int INTERNAL	 = 0x10;	// Type INTERNAL_ELISION = 4
	public static final int CONTAINS	 = 0x20;	// Type CLOSED_ELISION   = 5
	public static final int CHANGED		 = 0x40;	// Type CHANGED_ELISION  = 6

	public static final int ELIDED    = INTERNAL | DST | ENTERS | SRC |EXITS;

	/* Used for summary types */

	public static final int AND_ELISIONS  = 0x100;
	public static final int OR_ELISIONS   = 0x200;
	public static final int NAND_ELISIONS = 0x400;

	public static final int SUMMARY_ELISIONS = AND_ELISIONS | OR_ELISIONS | NAND_ELISIONS;


	private static final int GAP       = 4;

	private JFrame	m_frame;
	private	int		m_elisions = 0;
	private	boolean	m_summary  = false;

	public SelectedElisions(JFrame frame)
	{
		Font		font = FontCache.getDialogFont();
		Font		bold = font.deriveFont(Font.BOLD);

		m_frame    = frame;

		setFont(font);
		setIcon(this);
		setElisions(0);

		setSize(ToolBarButton.WIDTH*2, ToolBarButton.HEIGHT*2);
	}

	public boolean isSummary()
	{
		return m_summary;
	}

	public void setSummary(boolean summary)
	{
		m_summary = summary;
	}

	public int getElisions()
	{
		return m_elisions;
	}

	public void setElisions(int elisions)
	{
		String tip = "";

		m_elisions = elisions;

		if ((elisions & CHANGED) == 0) {
			tip = "Not yet specified";
		} else {
			if ((elisions & ELIDED) == 0) {
				tip = " all";
			} else if ((elisions & ELIDED) == ELIDED) {
				tip = " no";
			} else {
				if ((elisions & SRC) == 0) {
					tip = ", +source";
				} 
				if ((elisions & EXITS) == 0) {
					tip += ", +exits";
				}
				if ((elisions & ENTERS) == 0) {
					tip += ", +enters";
				}
				if ((elisions & DST) == 0) {
					tip += ", +destination";
				}
				if ((elisions & INTERNAL) == 0) {
					tip += ", +internal";
			}	}
			tip += " relations permitted";
		
			if ((elisions & CONTAINS) == 0) {
				tip = "Closed" + tip;
			} else {
				tip = "Open" + tip;
		}	}
		setToolTipText(tip);
	}

	public void updateElisions(int elisions, int mode) 
	{
		int elisions1 = elisions;

		if (mode == SelectedElisions.AND_ELISIONS) {
			elisions1 &= m_elisions;
		} else if (mode == SelectedElisions.OR_ELISIONS) {
			elisions1 |= m_elisions;
		} else if (mode == SelectedElisions.NAND_ELISIONS) {
			elisions1  = ~elisions1;
			elisions1 &= m_elisions;
		}
		setElisions(elisions1);
	}

	public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column)
	{
		return this;
	} 
 
	// Icon interface (used to paint image as icon)

	public int getIconWidth()
	{
		return(getWidth());
	}

	public int getIconHeight()
	{
		return(getHeight());
	}

	public void paintIcon(Component c, Graphics g, int x, int y)
	{
		int		w        = getWidth();
		int		h        = getHeight();
		int		x1       = x + GAP;      // x+gap -> x + width1
		int		y1       = y + h/3;
		int		width1   = w - 2 * GAP;
		int		height1  = (h * 2)/3 - GAP;
		int		elisions = m_elisions;
		Color	black    = Color.black;
		Color   red      = Color.red;
		Color	color;
		int		x2, y2, x3, y3;

		if ((elisions & CHANGED) == 0) {
			if (!isSummary()) {
				g.setColor(Color.RED);
				g.drawString("OK", x, y + h/2);
			}
		} else {
			if ((elisions & CONTAINS) == 0) {
				// Closed box
				g.setColor(Color.cyan);
				g.fillRect(x1, y1, width1, height1);
			}

			g.setColor(Color.black);
			g.drawRect(x1, y1, width1, height1);

			if ((elisions & INTERNAL) == 0) {
				x2 = x1 + (width1/3);
				y2 = y1 + (height1/2);
				x3 = x1 + (width1 * 2)/3;
				ToolBarButton.drawEdge(g, x2, y2, x3, y2); 
			}

			if ((elisions & SRC) == 0) {
				x2 = x1 + (width1/5);
				y3 = y + GAP;
				ToolBarButton.drawEdge(g, x2, y1, x2, y3);
			}

			if ((elisions & EXITS) == 0) {
				x2 = (x1 + (2*width1)/5);
				y2 = y1 + (height1/3);
				y3 = y + GAP;
				ToolBarButton.drawEdge(g, x2, y2, x2, y3);
			}

			if ((elisions & ENTERS) == 0) {
				x2 = x1 + (3*width1)/5;
				y2 = y1 + (height1/3);
				y3 = y + GAP;
				ToolBarButton.drawEdge(g, x2, y3, x2, y2);
			}

			if ((elisions & DST) == 0) {
				x2 = x1 + (4 * width1)/5;
				y3 = y + GAP;
				ToolBarButton.drawEdge(g, x2, y3, x2, y1);
		}	}
	}	
}

class ElisionTableModel extends AbstractTableModel {
	
	Diagram					m_diagram;
	Vector					m_entityClasses;
	Vector					m_relationClasses;
	SelectedElisions[][]	m_array;

	public ElisionTableModel(Diagram diagram)
	{
		LandscapeEditorCore		ls    = diagram.getLs();
		JFrame					frame = ls.getFrame();
		Enumeration				en;
		EntityClass				ec;
		Vector					entityClasses;
		int						rows, columns;
		SelectedElisions[][]	array;
		SelectedElisions[]		row;
		int						i, j;

		m_diagram         = diagram;

		m_entityClasses   = entityClasses = new Vector();

		i = 0;
		for (en = diagram.enumEntityClassesInOrder(); en.hasMoreElements(); ++i) {
			ec = (EntityClass) en.nextElement();
			ec.setOrderedId(i);
			entityClasses.addElement(ec);
		}

		m_relationClasses = diagram.getRelationClasses();
		rows              = getRowCount();
		columns           = getColumnCount() - 1;	// We have an additional first column

		m_array = array   = new SelectedElisions[rows][];

		row = null;
		for (i = 0; i < rows; ++i) {
			array[i] = row = new SelectedElisions[columns];
			for (j = 0; j < columns; ++j) {
				row[j] = new SelectedElisions(frame);
			}
			row[--j].setSummary(true);
		}
		for (j = row.length; --j >= 0; ) {
			row[j].setSummary(true);
		}
	}

	public int getRowCount()
	{
		return (m_entityClasses.size() + 1); 
	}

	public int getColumnCount()
	{
		return (m_relationClasses.size() + 2);
	}

	public Class getColumnClass(int column) 
	{
		if (column == 0) {
			return getValueAt(0, column).getClass();
		}
		return getValueAt(0, column).getClass();
	}

	public String getColumnName(int col)
	{
		if (col == 0) {
			return("");
		} else if (col > m_relationClasses.size()) {
			return "*ALL*";
		} else {
			RelationClass rc = (RelationClass) m_relationClasses.elementAt(col-1);
			return rc.getLabel();
		}
	}

	public boolean isCellEditable(int row, int col)
	{
		return (col != 0);
	}

	public Object getValueAt(int row, int col)
	{
		if (col == 0) {
			Vector		entityClasses = m_entityClasses;
			Enumeration en;
			EntityClass	ec;
			int			i;

			if (row < entityClasses.size()) {
				return entityClasses.elementAt(row);
			} 
			return "*ALL*";
		}	
		return m_array[row][col-1];
	}

	public void updateElisions(int row, int column, int elisions, int mode)
	{
		SelectedElisions selected = m_array[row][column];

		selected.updateElisions(elisions, mode);
//		System.out.println("Firing " + row + "x" + column + " " + elisions + " " + mode);
		fireTableCellUpdated(row, column);
	}

	public void setValueAt(Object value, int row, int col)
	{
		SelectedElisions selectedElisions = (SelectedElisions) value;

		if (selectedElisions.isSummary()) {
			int	elisions = selectedElisions.getElisions();
			int	mode     = elisions & SelectedElisions.SUMMARY_ELISIONS;
			int	scope    = 0;
			int	i, j;

			elisions &= ~SelectedElisions.SUMMARY_ELISIONS;

			if (row == getRowCount() - 1) {
				scope |= 1;		// Changing a column
			}
			if (col == getColumnCount() - 1) {
				scope |= 2;		// Changing a row
			}
			--col;
			switch (scope) {
			case 1:
				for (i = 0; i < row; ++i) {
					updateElisions(i, col, elisions, mode);
				}
				break;
			case 2:
				for (j = 0; j < col; ++j) {
					updateElisions(row, j, elisions, mode);
				}
				break;
			case 3:
				for (i = 0; i < row; ++i) {
					for (j = 0; j < col; ++j) {
						updateElisions(i, j, elisions, mode);
				}	}
				break;
			default:
				System.out.println("EditElisions failed");
				return;
	}	}	}

	public SelectedElisions[][] getArray()
	{
		return m_array;
	}
}

/*
 * This interface returns a button which is placed where the old value used to be and when fired brings up the
 * actual editor to change the old value.
 */

class ElisionEditor extends DefaultCellEditor implements ActionListener 
{
	JFrame				m_frame;
	AbstractTableModel	m_tableModel;
	SelectedElisions	m_selectedElisions;
	JButton				m_button;
	int					m_row;
	int					m_column;

	public ElisionEditor(JFrame frame, AbstractTableModel tableModel) 
	{
		super(new JCheckBox()); //Unfortunately, the constructor expects a check box, combo box, or text field.

		m_frame      = frame;
		m_tableModel = tableModel;

		//First, set up the button that brings up the dialog.

		m_button =  new JButton("EDITING");
		m_button.setBackground(Color.white);
		m_button.setBorderPainted(false);
		m_button.setMargin(new Insets(0,0,0,0));

		editorComponent = m_button;
		setClickCountToStart(1); //This is usually 1 or 2.

		//Here's the code that brings up the dialog.
		m_button.addActionListener(this);
	}

	public Object getCellEditorValue() 
	{
		return m_selectedElisions;
	}

	public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column) 
	{
		m_selectedElisions = (SelectedElisions) value;
		m_row              = row;
		m_column           = column;
		return editorComponent;
	}

	// Action listener interface

	public void actionPerformed(ActionEvent e) 
	{
		SelectedElisions selectedElisions = m_selectedElisions;
		boolean			 isSummary        = selectedElisions.isSummary();
		ElisionChooser	 elisionChooser   = new ElisionChooser(m_frame, selectedElisions.getElisions(), isSummary);
		int				 elisions         = elisionChooser.getElisions();

		if (elisions >= 0) {
			selectedElisions.setElisions(elisions);
			if (isSummary) {
				m_tableModel.setValueAt(selectedElisions, m_row, m_column); 
		}	}

		fireEditingStopped(); 
		elisionChooser.dispose();
	}
}


// This is the class which decides how the table is drawn and edited

class ElisionTable extends JTable {

	private ElisionEditor m_elisionEditor;

	public ElisionTable(JFrame frame, AbstractTableModel tableModel)
	{
		super(tableModel);
		m_elisionEditor = new ElisionEditor(frame, tableModel);
//		tableModel.addTableModelListener(this);
	}

	public TableCellRenderer getCellRenderer(int row, int column)
	{
		if (convertColumnIndexToModel(column) == 0) {
			return(super.getCellRenderer(row, column));
		}

		return (SelectedElisions) dataModel.getValueAt(row, column);
	}

	// Overload how cells are editted

	public TableCellEditor getCellEditor(int row, int column)
	{
		return m_elisionEditor;
	}
}

public class EditElisions extends JDialog implements ActionListener { 

	private	LandscapeEditorCore	m_ls;
	private ElisionTableModel	m_elisionTableModel;
	private ElisionTable		m_table;
	private JButton				m_ok      = null;
	private JButton				m_cancel  = null;
			
	protected EditElisions(JFrame frame, LandscapeEditorCore ls)
	{
		super(frame, "Edit elision rules",true); //false if non-modal

		Container	contentPane;
		JScrollPane	scrollPane;
		JPanel		panel, buttons;
		Font		font, bold;
		int			height, height1;

		m_ls         = ls;

		font         = FontCache.getDialogFont();
		bold         = font.deriveFont(Font.BOLD);

		setLocation(20, 20);

//		setSize(438,369);

		contentPane = getContentPane();
		contentPane.setLayout(new BorderLayout());

		setForeground(ColorCache.get(0,0,0));
		setBackground(ColorCache.get(192,192,192));
		setFont(font);

		ElisionTableModel	tableModel;
		ElisionTable		table;

		m_elisionTableModel = tableModel = new ElisionTableModel(ls.getDiagram());
		m_table             = table      = new ElisionTable(ls.getFrame(), tableModel);
		table.setFont(font);

		JTableHeader tableHeader = table.getTableHeader();
		tableHeader.setFont(bold);

		FontMetrics fm = getFontMetrics(font);
		height  = fm.getHeight() + 4;
		height1 = ToolBarButton.HEIGHT * 2;
		if (height < height1) {
			height = height1;
		}
		table.setRowHeight(height);

		table.setVisible(true);
		scrollPane = new JScrollPane(table);

		scrollPane.setVisible(true);
		contentPane.add(scrollPane, BorderLayout.CENTER);


		buttons = new JPanel();
		buttons.setLayout(new FlowLayout());

		m_ok     = new JButton("Ok");
		m_ok.setFont(bold);
		m_ok.addActionListener(this);
		buttons.add(m_ok);

		m_cancel = new JButton("Cancel");
		m_cancel.setFont(bold);
		m_cancel.addActionListener(this);
		buttons.add(m_cancel);

		contentPane.add(BorderLayout.SOUTH, buttons);

		pack();
		setVisible(true);

		table.removeEditor();
	}

	private static void descend(EntityInstance e, SelectedElisions[][] array)
	{
		int	i;

		{
			EntityClass			ec               = e.getEntityClass();
			int					row1             = ec.getOrderedId();
			SelectedElisions[]	selectedElisions = array[row1];	
			SelectedElisions	selectedElision;
			int					elisions;
			int					j, mask;

			for (i = selectedElisions.length - 1 /* Ignore the ALL column */; --i >= 0; ) {
				selectedElision = selectedElisions[i];
				elisions        = selectedElision.getElisions();
				if ((elisions & SelectedElisions.CHANGED) == 0) {
					continue;
				}
				mask = 1;
				for (j = 0; j < EntityInstance.ELISIONS; ++j) {
					if ((elisions & mask) != 0) {
						e.setElision(j, i);
					} else {
						e.clearElision(j, i);
					}
					mask <<= 1;
		}	}	}

		Vector				srcRelList       = e.getSrcRelList();

		if (srcRelList != null) {
			
			RelationInstance	ri;
			EntityInstance		child;

			for (i = srcRelList.size(); --i >= 0; ) {
				ri = (RelationInstance) srcRelList.elementAt(i);
				if (ri.isContainsClass()) {
					child = ri.getDst();
					descend(child, array);
	}	}	}	}

	private void doEditElisions()
	{
		SelectedElisions[][]	array   = m_elisionTableModel.getArray();
		Diagram					diagram = m_ls.getDiagram();
		SelectedElisions[]		selectedElisions;
		SelectedElisions		selectedElision;
		int						i, j, elisions;

		for (i = array.length - 1; --i >= 0; ) {
			selectedElisions = array[i];
			for (j = selectedElisions.length - 1; --j >= 0; ) {
				selectedElision = selectedElisions[j];
				elisions        = selectedElision.getElisions();
				if ((elisions & SelectedElisions.CHANGED) != 0) {
					descend(diagram.getRootInstance(), array);
					diagram.fill();
					return;
	}	}	}	}
		
	// ActionListener interface

	public void actionPerformed(ActionEvent ev)
	{
		Object	source = ev.getSource();

		if (source == m_ok || source == m_cancel) {
			if (source == m_ok) {
				doEditElisions();
			}	
			this.setVisible(false);
			return;
		}
	}
} 



