package lsedit;

import java.lang.Math;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Graphics;
import java.awt.Rectangle;

import java.util.Vector;

import javax.swing.Icon;
import javax.swing.JComponent;

// Entities in a landscape 

public class EntityComponent extends JComponent implements Icon {

	// Final values
	
	public final static int MARGIN             = 5; 
	
	/* Used in computing rules for shape of a folder */

	public  final static int MIN_FLAP_HT       = 3;
	public  final static int MAX_FLAP_HT       = 8;
	public  final static int FLAP_MARGIN       = 4;
	public  final static int TINY_FLAP_WD      = 10;
	public  final static int MIN_FLAP_WD       = 75;

	/* Position and size of the contents flag when entity closed */

	public final static int CONTENTS_FLAG_X   = 3;
	public final static int CONTENTS_FLAG_Y   = 9;
	public final static int CONTENTS_FLAG_DIM = 8;
	public final static int CONTENTS_FLAG_X_RESERVE = CONTENTS_FLAG_X + CONTENTS_FLAG_DIM;
	public final static int CONTENTS_FLAG_Y_RESERVE = CONTENTS_FLAG_Y + CONTENTS_FLAG_DIM;

	// These are the top left coordinates of the object on the screen

	private EntityInstance m_entityInstance; 

	// Sorting uses this value and it is really expensive to recompute for each key compare

	private double		   m_avgX;

 	private   Cardinal[]    m_dstCardinals;
	private   SrcCardinal[] m_srcCardinals;

	private	static int[]   m_xp = null;
	private static int[]   m_yp = null;

	// We need the position within the diagram to compute coordinates for edges quickly

	private int		m_diagramX         = 0;
	private int		m_diagramY         = 0;

	private Vector m_srcLiftedList = null;	// List of relations whose source is lifted to leave this entity
	private Vector m_dstLiftedList = null;	// List of relations whose destination is lifted to address this entity


	// --------------
	// Object methods
	// --------------

	public String toString() 
	{
		return "EntityComponent: " + m_entityInstance.toString();
	}

	// -----------------
	// Component methods
	// -----------------
	
	public void setBounds(int x, int y, int width, int height)
	{
/*
		if (m_entityInstance.toString().equals("clibrary")) {
			System.out.println(this + " setBounds from (" + getX() + "," + getY() + ":" + getWidth() + "x" + getHeight() + ") to (" + x + "," + y + ":" + width + "x" + height + ") old diagram = " + m_diagramX + "x" + m_diagramY);
		}
*/
		shiftDiagramLocations(x - getX(), y - getY());
		super.setBounds(x, y, width, height);
		refillEdges();
	}

	// ------------------
	// JComponent methods
	// ------------------

	public Component add(Component child)
	{
//		System.out.println("Adding " + child + " under " + this);

		EntityComponent ec    = (EntityComponent) child;
		EntityInstance	e     = ec.getEntityInstance();
		double			area  = e.widthRelLocal() * e.heightRelLocal();
		int				size  = getComponentCount();
		EntityComponent	ec1;
		EntityInstance	e1;
		double			area1;
		int		        i;

		for (i = 0; i < size; ++i) {
			ec1   = (EntityComponent) getComponent(i);
			e1    = ec1.getEntityInstance();
			area1 = e1.widthRelLocal() * e1.heightRelLocal();
			if (area1 > area) {
				// Put smaller areas before larger ones so they get painted later
				// We might shade larger things
				return add(child, i);
		}	}
		return super.add(child); 
	}

	public void paintComponent(Graphics g) 
	{
		EntityInstance	entityInstance = m_entityInstance;
		EntityClass		entityClass    = entityInstance.getEntityClass();
		boolean			isDrawRoot     = entityInstance.isDrawRoot();
		boolean			isOpen         = entityInstance.isOpen();
		int		x                      = 0;
		int		y                      = 0;
		int		width                  = getWidth();
		int		height                 = getHeight();
		int		margin                 = MARGIN;
		String	label1                 = entityInstance.getEntityLabel();
		int		labelWidth;
		boolean center;
		Color	backgroundColor;
		Color	foregroundColor;


//		System.out.println("EntityInstance.paintComponent: " + entityInstance + " redBox=" + entityInstance.isMarked(EntityInstance.REDBOX_MARK));		// IJD
/*		{
			int	cnt, i;

			cnt = getComponentCount();
			for (i = 0; i < cnt; ++i) {
				System.out.println("EntityInstance.paintComponent: child " + getComponent(i));
		}	}
*/

/*		java.lang.Thread.dumpStack();
		System.out.println("-----");
*/

		if (entityInstance.red_closed()) {
			backgroundColor = Color.red.darker();
//			System.out.print(backgroundColor);
		} else {
			backgroundColor = entityInstance.getCurrentObjectColor();
		}
		foregroundColor = Color.black;

		if (!isDrawRoot) {
			if (entityInstance.isMarked(EntityInstance.SHADES_MARK)) {
				int	shadow_size = ArrowDimensions.m_shadow_size;
				int	max         = ((width > height) ? height : width) >> 1;
				if (shadow_size > max) {
					shadow_size = max;
				}
				if (shadow_size > 0) {
					Color shadow;

					if (backgroundColor == null) {
						shadow = Color.GRAY;
					} else {
						shadow = backgroundColor;
					}
					shadow  = shadow.darker();
					width  -= shadow_size;
					height -= shadow_size;
					paintShape(g, entityInstance, entityClass, shadow_size, shadow_size, width, height, false, isOpen, foregroundColor, shadow);
		}	}	}

		if (entityInstance.red_open()) {
			foregroundColor = Color.red.darker();
		}
		paintShape(g, entityInstance, entityClass, 0, 0, width, height, isDrawRoot, isOpen, foregroundColor, backgroundColor); 

		Diagram				diagram = entityInstance.getDiagram();
		LandscapeEditorCore ls      = diagram.getLs();

		if (ls.isShowGrid()) {

			if (isDrawRoot) {
				int	grid = diagram.getGrid();

				if (grid > 1) {
					int i;
					g.setColor(diagram.getGridColor());
					for (i = grid; i < height; i += grid) {
						g.drawLine(0,i,width-1,i);
					}
					for (i = grid; i < width; i += grid) {
						g.drawLine(i,0,i, height-1);
		}	}	}	}

		if (entityInstance.closedWithChildren()) {

			/*  Draw a small mark as shown in top left of object

				x---
				|
				|   --------
				|  |        |
				   |    |   |
				   |    |   |
	  			   |  ----  |
				   |    |   |
				   |    |   |
				   |        |
				   |________|
			*/

			if (width > CONTENTS_FLAG_X_RESERVE && height > CONTENTS_FLAG_Y_RESERVE) {
				int x1, y1;
				Color	color1;

				color1 = entityInstance.getInheritedObjectColor();
				if (color1 != null) {
					color1 = ColorCache.getInverse(color1.getRGB());
				} else {
					color1 = Color.BLACK;
				}
				g.setColor(color1);

				g.drawRect(CONTENTS_FLAG_X, CONTENTS_FLAG_Y, CONTENTS_FLAG_DIM /* 8 */, CONTENTS_FLAG_DIM);

				g.drawLine(CONTENTS_FLAG_X+1,                     CONTENTS_FLAG_Y+(CONTENTS_FLAG_DIM/2), CONTENTS_FLAG_X_RESERVE-1,             CONTENTS_FLAG_Y+(CONTENTS_FLAG_DIM/2));
				g.drawLine(CONTENTS_FLAG_X+(CONTENTS_FLAG_DIM/2), CONTENTS_FLAG_Y+1,                     CONTENTS_FLAG_X+(CONTENTS_FLAG_DIM/2), CONTENTS_FLAG_Y_RESERVE-1);
				margin += CONTENTS_FLAG_X_RESERVE;
		}	}

		// Draw our own label

		g.setColor(entityInstance.getCurrentLabelColor());

		switch (entityInstance.getInheritedStyle()) {
		case EntityClass.ENTITY_STYLE_LABELLED_GROUP:
			Util.drawGroupBoxLabel(g, 0, 0, width, label1);
			break;
		default:
			labelWidth = getWidth() - MARGIN - margin;
			if (entityInstance.isOpen()) {
				center     = false;
				switch (entityInstance.getInheritedStyle()) {
				case EntityClass.ENTITY_STYLE_FOLDER:

					g.setFont(EntityInstance.getSmallFont());
					// Compute flap size
					int fw = (int) (((double) getWidth()) * .4);
					int fh = Math.max(MIN_FLAP_HT, Math.min(MAX_FLAP_HT, ((int) (getHeight() * .2))));

					if (fw < MIN_FLAP_WD) {
					   fw += fw/2;
					}
					if (fw < TINY_FLAP_WD) {
						fw = Math.min(fw + FLAP_MARGIN, getWidth() - getWidth()/3);
					}
					margin     = FLAP_MARGIN+fh/2+2;
//					Util.drawStringClipped(g, label1, FLAP_MARGIN+fh/2+2, 0, (double) fw, getHeight()-MARGIN*2, false, false);
					break;
				case EntityClass.ENTITY_STYLE_CLASS:
					g.setFont(EntityInstance.getOpenClassFont());
				
//					Util.drawStringClipped(g, label1, MARGIN*3, MARGIN, getWidth()-MARGIN*2, getHeight()-MARGIN*2, false, false);
					break;
				default:
					g.setFont(EntityInstance.getSmallFont());
//					g.setColor(Color.black); 
//					Util.drawStringClipped(g, label1, MARGIN, MARGIN, getWidth()-MARGIN*2, getHeight()-MARGIN*2, false, false);
				}
			} else {
				center = true;
				// centered, perhaps multi-line
				if (!entityInstance.isMarked(EntityInstance.CLIENT_SUPPLIER)) {
					g.setFont(entityInstance.getAdjustedClosedFont());
				} else {
					g.setFont(ClientSupplierSet.getClientSupplierFont());
					EntityInstance pe = entityInstance.getContainedBy();
					if (pe != null) {
						label1 = pe.getEntityLabel() + " .\n" + label1;
			}	}	}
			Util.drawStringWrapped(g, label1, margin, MARGIN, labelWidth, getHeight()-MARGIN*2, center, false, ArrowDimensions.isLabelInvertBackground());
			
		} 
	
		// Put flags around the edges of the box

		if (entityInstance.getGroupFlag()) {
			g.setColor(entityInstance.getInheritedLabelColor());

			int pdim = (height < 20) ? 4 : 6;

			// Draw resize points

			if (entityInstance.getGroupKeyFlag()) {
				g.fillRect(1,			   1,				pdim, pdim);
				g.fillRect(width/2-pdim/2, 1,				pdim, pdim);
				g.fillRect(width-pdim,	   1,				pdim, pdim);
				g.fillRect(1,			   height/2-pdim/2, pdim, pdim);
				g.fillRect(width-pdim,	   height/2-pdim/2, pdim, pdim);
				g.fillRect(1,			   height-pdim,		pdim, pdim);
				g.fillRect(width/2-pdim/2, height-pdim,		pdim, pdim);
				g.fillRect(width-pdim,	   height-pdim,		pdim, pdim);
			}  else {
				// Stupidity here.. An outline is one extra byte wide and high because the two
				// edges of a one pixel box occupy 2 pixels.

				g.drawRect(1,			   1,				pdim, pdim);
				g.drawRect(width/2-pdim/2, 1,				pdim, pdim);
				g.drawRect(width-pdim-1,   1,				pdim, pdim);
				g.drawRect(1,			   height/2-pdim/2, pdim, pdim);
				g.drawRect(width-pdim-1,   height/2-pdim/2, pdim, pdim);
				g.drawRect(1,			   height-pdim-1,	pdim, pdim);
				g.drawRect(width/2-pdim/2, height-pdim-1,	pdim, pdim);
				g.drawRect(width-pdim-1,   height-pdim-1,	pdim, pdim);
			}
		}

		/* For debugging 
		g.setColor(Color.black);
		g.drawLine(0,0,width,height);
		g.drawLine(0,height, width, 0);
		*/
	}

	// --------------
	// Public methods
	// --------------

	private static int[] getXp()
	{
		if (m_xp == null) {
			m_xp = new int[21];
		}
		return m_xp;
	}

	private static int[] getYp()
	{
		if (m_yp == null) {
			m_yp = new int[21];
		}
		return m_yp;
	}

 	public EntityComponent(EntityInstance entityInstance) 
	{
		setLayout(null);
		m_entityInstance = entityInstance;
		entityInstance.setEntityComponent(this);
		if (m_xp == null) {
			m_xp = new int[21];
			m_yp = new int[21];
		}
	}

	public EntityInstance getEntityInstance()
	{
		return(m_entityInstance);
	}

	public Vector getSrcLiftedList()
	{
		return m_srcLiftedList;
	}

	public Vector getDstLiftedList()
	{
		return m_dstLiftedList;
	}

	public Vector getNeededSrcLiftedList()
	{
		Vector	srcLiftedList = m_srcLiftedList;

		if (srcLiftedList == null) {
			m_srcLiftedList = srcLiftedList = new Vector();
		}
		return srcLiftedList;
	}

	public Vector getNeededDstLiftedList()
	{
		Vector	dstLiftedList = m_dstLiftedList;

		if (dstLiftedList == null) {
			m_dstLiftedList = dstLiftedList = new Vector();
		}
		return dstLiftedList;
	}

	public void clearLiftedEdges()
	{
		m_srcLiftedList = null;		// Have no lifted relation sources
		m_dstLiftedList = null;		// Have no lifted relation destinations
	}

	protected void shiftDiagramLocations(int shiftX, int shiftY)
	{
		Component	component;
		int			i;

		m_diagramX += shiftX;
		m_diagramY += shiftY;

/*
		if (m_entityInstance.toString().equals("clibrary")) {
			System.out.println(this + " shiftDiagramLocation(" + shiftX + "," + shiftY + ") diagram = " + m_diagramX + "," + m_diagramY);
		}
 */
		for (i = getComponentCount(); --i >= 0; ) {
			component = getComponent(i);
			if (component instanceof EntityComponent) {
				((EntityComponent) component).shiftDiagramLocations(shiftX, shiftY);
	}	}	}

	// Initialise the root of the entity component tree so that it is
	// sync'd with the same diagram location as its parent.  Then when
	// we relocate it everything works correctly.  N.B. The initial
	// location is set to (0,0) apriori when setting the entity size

	public void setDiagramLocation(int diagramX, int diagramY)
	{
		Component	component;
		int			i;

		m_diagramX = diagramX;;
		m_diagramY = diagramY;

/*
		if (m_entityInstance.toString().equals("clibrary")) {
			System.out.println(this + " setDiagramLocation " + this + " diagram = " + m_diagramX + "x" + m_diagramY);
		}
 */
		for (i = getComponentCount(); --i >= 0; ) {
			component = getComponent(i);
			if (component instanceof EntityComponent) {
				((EntityComponent) component).setDiagramLocation(diagramX+component.getX(), diagramY+component.getY());
	}	}	}

	public int getDiagramX()
	{
		return m_diagramX;
	}

	public int getDiagramY()
	{
		return m_diagramY;
	}

	public Rectangle getDiagramBounds() 
	{
		return new Rectangle(m_diagramX, m_diagramY, getWidth(), getHeight());
	}

	public boolean shades(EntityComponent base)
	{
		int	x     = getX();
		int	basex = base.getX();

		if (x >= basex) {
			int	y     = getY();
			int basey = base.getY();
			if (y >= basey) {
				x     += getWidth();
				basex += base.getWidth();
				if (x <= basex) {
					y     += getHeight();
					basey += base.getHeight();
					if (y <= basey) {
						return true;
		}	}	}	}
		return false; 
	}

	// Compute shading for each of our children if that shading not known

	public void computeShading()
	{
		EntityInstance	e    = m_entityInstance;
		EntityComponent ec1;
		int				size = getComponentCount();
		int				i;

		if (!e.isMarked(EntityInstance.SHADING_KNOWN_MARK)) {
			int				j;
			EntityComponent	ec2;
			EntityInstance	child;

			// From smallest to largest
			for (i = 0; i < size; ++i) {
				ec1   = (EntityComponent) getComponent(i);
				child = ec1.getEntityInstance();
				// From largest to smallest
				for (j = size; ;) {
					if (--j == i) {
						// The entityInstance e doesn't shade any larger areas
						child.nandMark(EntityInstance.SHADES_MARK);
						break;
					}
					ec2 = (EntityComponent) getComponent(j);
					if (ec1.shades(ec2)) {
						// The entity instance e shades ec2
//						System.out.println(ec1 + " shades " + ec2);
						child.orMark(EntityInstance.SHADES_MARK);
						break;
			}	}	}
			e.orMark(EntityInstance.SHADING_KNOWN_MARK);
		}

		for (i = 0; i < size; ++i) {
			ec1   = (EntityComponent) getComponent(i);
			ec1.computeShading();
	}	}

	// Reset all cardinals associated with this entity and things it contains

	public void resetDstCardinals(int numRelations) 
	{
		if (m_dstCardinals == null) {
			if (numRelations > 0) {
				m_dstCardinals = new Cardinal[numRelations];
			}
		} else {
			int		 i;
			Cardinal cardinal;
			
			for (i = 0; i < numRelations; ++i) {
				cardinal = m_dstCardinals[i];
				if (cardinal != null) {
					cardinal.reset();
	}	}	}	}

	public Cardinal[] getDstCardinals()
	{
		return m_dstCardinals;
	}

	public void resetSrcCardinals(int numRelations) 
	{
		if (m_srcCardinals == null) {
			if (numRelations > 0) {
				m_srcCardinals = new SrcCardinal[numRelations];
			}
		} else {
			int		 i;
			Cardinal cardinal;
			
			for (i = 0; i < numRelations; ++i) {
				cardinal = m_srcCardinals[i];
				if (cardinal != null) {
					cardinal.reset();
	}	}	}	}

	public Cardinal[] getSrcCardinals()
	{
		return m_srcCardinals;
	}

	// Used for sorting clients/suppliers

	public void setAvgX(double avgX)
	{
		m_avgX = avgX;
	}

	public double getAvgX()
	{
		return m_avgX;
	}

	// Invalidate our edges because of some change in our shape or position etc.

	public void refillEdges()
	{
		EntityComponent	component;
		int	i;

		{
			Vector				liftedList = m_srcLiftedList;
			RelationInstance	ri;
			int					dir;

			for (dir = 0; dir < 2; ++dir) {
				if (liftedList != null) {
					for (i = liftedList.size(); --i >= 0; ) {
						ri = (RelationInstance) liftedList.elementAt(i);
						ri.nandMark(RelationInstance.FILLED_MARK);
				}	}
				liftedList = m_dstLiftedList;
		}	}

		// Any edges to things inside us will also need to be refilled

		for (i = getComponentCount(); --i >= 0; ) {
			component = (EntityComponent) getComponent(i);
			component.refillEdges();
	}	}

	// Keep the Z ordering satisfying the fact that smaller areas get painted after later ones
	// Small areas occur before later ones in the component array

	private void entitySizeChanged()
	{
		Container parent = getParent();

		if (!(parent instanceof EntityComponent)) {
			return;
		}

		int				size  = parent.getComponentCount();
		int				i, j, k;
		EntityInstance	e, e1;
		EntityComponent	ec;
		double			area, area1;

		for (i = 0;; ++i) {
			if (i == size) {
				return;
			}
			if (this == parent.getComponent(i)) {
				break;
		}	}

		// This component used to occur at i

		e     = getEntityInstance();
		area  = e.widthRelLocal() * e.heightRelLocal();

		for (j = i-1; j >= 0; --j) {
			ec    = (EntityComponent) parent.getComponent(j);
			e1    = ec.getEntityInstance();
			area1 = e1.widthRelLocal() * e1.heightRelLocal();
			if (area1 <= area) {
				break;
		}	}
		if (++j == i) {
			// Everything before old position remains <= new area size (perhaps our size has grown)
			for (j = i+1; j < size; ++j) {
				ec    = (EntityComponent) parent.getComponent(j);
				e1    = ec.getEntityInstance();
				area1 = e1.widthRelLocal() * e1.heightRelLocal();
				if (area1 >= area) {
					break;
			}	}
			--j;
		}
		// New position for component currently at position i is at position j

		if (j != i) {
			parent.remove(i);
			parent.add(this, j);	// Add after the entity with smaller area and before entity with larger area
	}	}

	// This must be called whenever our relative bounds change when we are
	// visible to recompute our coordinates within the diagram

	public void setEntityBounds(EntityComponent parentComponent)
	{
		EntityInstance	e               = m_entityInstance;
		EntityInstance	parent          = parentComponent.getEntityInstance();
		double			parentWidth     = (double) parentComponent.getWidth();
		double			parentHeight    = (double) parentComponent.getHeight();
		int				oldX            = getX();
		int				oldY            = getY();
		int				oldWidth        = getWidth();
		int				oldHeight       = getHeight();
		int				newX, newY, newWidth, newHeight;
		boolean			sameSize;

		parent.nandMark(EntityInstance.SHADING_KNOWN_MARK);

		if (parentWidth <= 0 || parentHeight <= 0) {
			newX      = oldX;
			newY      = oldY;
			newWidth  = 0;
			newHeight = 0;
		} else {
			newX      = (int) (e.xRelLocal()      * parentWidth);
			newY      = (int) (e.yRelLocal()      * parentHeight);
			newWidth  = (int) (e.widthRelLocal()  * parentWidth);
			newHeight = (int) (e.heightRelLocal() * parentHeight);
		}
		sameSize = (oldWidth == newWidth) && (oldHeight == newHeight);
		if (sameSize && oldX == newX && oldY == newY) {
			return;
		}
		setBounds(newX, newY, newWidth,newHeight);

		if (!sameSize) {
			EntityComponent	child;
			int				i;

			entitySizeChanged();

			for (i = getComponentCount(); --i >= 0; ) {
				child = (EntityComponent) getComponent(i);
				child.setEntityBounds(this);
	}	}	}

	public void setEntityBounds()
	{
		Component parent = getParent();

		if (parent != null && parent instanceof EntityComponent) {
			setEntityBounds((EntityComponent) parent);
	}	}

	/* N.B. Must set graphics color prior to entry */

	public static void paintImage(Graphics g, int image, int x, int y, int width, int height)
	{
		int	i;

		for (i = EntityClass.ENTITY_IMAGE_LAST; i != 0; i >>= 1) {
			if ((i & image) == 0) {
				continue;
			}
			switch (i) {
				case EntityClass.ENTITY_IMAGE_ACTOR:
				{
					int	min, centerx, centery;

					min = width;
					if (height < min) {
						min = height;
					}
					centerx = x+width/2;
					centery = y+height/2;

					g.drawOval(centerx-(min/10), centery-((7*min)/20), min/5, min/5);						// Head
					g.drawLine(centerx, centery-((3*min)/20), centerx, centery+((3*min)/20));				// Body
					g.drawLine(centerx-(min/4), centery-(min/20), centerx+(min/4), centery-(min/20));		// Arms
					g.drawLine(centerx, centery+((3*min)/20), centerx-(min/5), centery+((2*min)/5));		// Left leg
					g.drawLine(centerx, centery+((3*min)/20), centerx+(min/5), centery+((2*min)/5));		// Right leg
					break;
				}
				case EntityClass.ENTITY_IMAGE_OVAL:
				{
					g.drawOval(x, y, width, height);
					break;
				}
				case EntityClass.ENTITY_IMAGE_FRAME:
				{
					g.drawRect(x, y, width-1, height-1);
					break;
				}
				case EntityClass.ENTITY_IMAGE_ROUNDED_FRAME:
				{
					int arc_w = width/5;
					int arc_h = height/5;
					int arc   = Math.min(arc_w, arc_h);

					g.drawRoundRect(x, y, width-1, height-1, arc, arc);
					break;

				}
				case EntityClass.ENTITY_IMAGE_X:
				{
					int	min, centerx, centery;

					min = width;
					if (height < min) {
						min = height;
					}
					centerx = x+width/2;
					centery = y+height/2;

					g.drawLine(centerx-(min/4), centery-(min/4), centerx+(min/4), centery+(min/4));	
					g.drawLine(centerx+(min/4), centery-(min/4), centerx-(min/4), centery+(min/4));	
					break;
	}	}	}	}

	private static void draw3DRectRaised(Graphics g, Color c, Color fill, int x, int y, int width, int height)
	{
		Color brighter = Color.white;
		Color darker   = Color.darkGray;

		if (!ArrowDimensions.isBlackWhite3D()) {
			brighter = c.brighter();
			darker   = c.darker();
		}

		if (fill != null) {
			g.setColor(fill);
			g.fillRect(x+ArrowDimensions.m_pixels_3d, y+ArrowDimensions.m_pixels_3d, width-2*ArrowDimensions.m_pixels_3d, height-2*ArrowDimensions.m_pixels_3d);
		}
		g.setColor(brighter);
		g.fillRect(x, y, ArrowDimensions.m_pixels_3d, height);																									// Band down left side
		g.fillRect(x + ArrowDimensions.m_pixels_3d, y, width - ArrowDimensions.m_pixels_3d, ArrowDimensions.m_pixels_3d);										// Band across top
		g.setColor(darker);
		g.fillRect(x + ArrowDimensions.m_pixels_3d, y + height-ArrowDimensions.m_pixels_3d, width-ArrowDimensions.m_pixels_3d, ArrowDimensions.m_pixels_3d);	// Band across bottom
		g.fillRect(x + width-ArrowDimensions.m_pixels_3d, y, ArrowDimensions.m_pixels_3d, height);																// Band down right side
		g.setColor(Color.black);
		g.drawLine(x,y, x+ArrowDimensions.m_pixels_3d-1, y+ArrowDimensions.m_pixels_3d-1);
    }    

	private static void draw3DRectSunken(Graphics g, Color c, Color fill, int x, int y, int width, int height)
	{
		Color brighter = Color.white;
		Color darker   = Color.darkGray;

		if (!ArrowDimensions.isBlackWhite3D()) {
			brighter = c.brighter();
			darker   = c.darker();
		}

		if (fill != null) {
			g.setColor(fill);
			g.fillRect(x+ArrowDimensions.m_pixels_3d, y+ArrowDimensions.m_pixels_3d, width-2*ArrowDimensions.m_pixels_3d, height-2*ArrowDimensions.m_pixels_3d);
		}
		g.setColor(darker);
		g.fillRect(x, y, ArrowDimensions.m_pixels_3d, height);																									// Band down left side
		g.fillRect(x + ArrowDimensions.m_pixels_3d, y, width - ArrowDimensions.m_pixels_3d, ArrowDimensions.m_pixels_3d);										// Band across top
		g.setColor(brighter);
		g.fillRect(x + ArrowDimensions.m_pixels_3d, y + height-ArrowDimensions.m_pixels_3d, width-ArrowDimensions.m_pixels_3d, ArrowDimensions.m_pixels_3d);	// Band across bottom
		g.fillRect(x + width-ArrowDimensions.m_pixels_3d, y, ArrowDimensions.m_pixels_3d, height);																// Band down right side
		g.setColor(Color.black);
		g.drawLine(x+width-ArrowDimensions.m_pixels_3d ,y+height-ArrowDimensions.m_pixels_3d, x+width-1, y+height-1);
    }    

	/* N.B. Any change to shapes requires potential correction to EdgePoint.adjustEdgePoint() */

	public static void paintShape(Graphics g, EntityInstance entityInstance, EntityClass entityClass, int x, int y, int width, int height, boolean isDrawRoot, boolean isOpen, Color foregroundColor, Color backgroundColor) 
	{
		if (width > 0 && height > 0) {
			int			   style, direction, image;
			int[]		   xp = m_xp;
			int[]		   yp = m_yp;
			int			   polygon_dimension = 0;
			int			   regular_dimension = 0;
			Color		   color;

			if (isDrawRoot) {
				style     = EntityClass.ENTITY_STYLE_3DBOX;
				direction = 0;
			} else {
				style     = entityClass.getInheritedStyle();
				direction = entityClass.getDirection();
			}

			switch (style) {
				case EntityClass.ENTITY_STYLE_3DBOX:
				{
					draw3DRectRaised(g, foregroundColor, backgroundColor, x, y, width-1, height-1);
					break;
				}
				case EntityClass.ENTITY_STYLE_SUNK_BOX:
				{
					draw3DRectSunken(g, foregroundColor, backgroundColor, x, y, width-1, height-1);
					break;
				}
				case EntityClass.ENTITY_STYLE_2DBOX:
				{
					if (backgroundColor != null) {
						g.setColor(backgroundColor);
						g.fillRect(x, y, width, height);
					}
					g.setColor(foregroundColor);
					g.drawRect(x, y, width-1, height-1);
					break;
				}

				case EntityClass.ENTITY_STYLE_FILE:
				{
					int fd = Math.min(Math.min(width, height)/2, 16 /* FLAP_DIM */);

					/*
					   0--(width-fd)----1 \        
					   |                     \    (fd)
					   |				        \ 
					   |                           2
					   | 						    |
					   |						 (height)
					   |							|
					   4 -------------------------- 3
					 */

					xp[0] = x;
					yp[0] = y;
					xp[1] = x + width - fd;
					yp[1] = y;
					xp[2] = x + width - 1;
					yp[2] = y + fd;
					xp[3] = xp[2];
					yp[3] = y + height - 1;
					xp[4] = x;
					yp[4] = yp[3];
					xp[5] = xp[0];
					yp[5] = yp[0];

					if (backgroundColor != null) {
						g.setColor(backgroundColor);
						g.fillPolygon(xp, yp, 6);
					}
					g.setColor(foregroundColor);
					g.drawPolygon(xp, yp, 6);

					xp[0] = xp[1];
					yp[0] = yp[2];
					xp[3] = xp[0];
					yp[3] = yp[0];

					g.drawPolygon(xp, yp, 4);
					break;
				}

				case EntityClass.ENTITY_STYLE_DISK:
				{
					int ad = Math.min(Math.min(width, height)/2, 8 /* ARC_DIM */);

					// Draw a cylinder

					if (backgroundColor != null) {
						g.setColor(backgroundColor);
						g.fillRect(x, y+ad/2, width, height-ad);
						g.fillOval(x, y, width, ad);
						g.fillOval(x, y+height-ad, width, ad);
					}
					g.setColor(foregroundColor);

					if (isOpen) {
						g.drawArc(x, y, width, ad, 180, -180);
					} else {
						g.drawOval(x, y, width, ad);
					}
					g.drawArc(x, y+height-ad-1, width, ad, 180, 180);
					g.drawLine(x, y+ad/2, x, y+height-ad/2);
					g.drawLine(x+width-1, y+ad/2, x+width-1, y+height-ad/2);
					break;
				}

				case EntityClass.ENTITY_STYLE_FOLDER:
				{
					// Compute flap size

					int fw = ((int) (((double) width) * .4));
					int fh = Math.max(MIN_FLAP_HT, Math.min(MAX_FLAP_HT, ((int) (height * .2))));
					int fm = FLAP_MARGIN;

					if (fw < MIN_FLAP_WD) {
					   fw += fw/2;
					}
					if (fw < TINY_FLAP_WD) {
						fw = Math.min(fw + FLAP_MARGIN, width - width/3);
						fm = 0;
					}

				/*   
                     <--------------fw-------------------------->

                                     2--------------------------3
                   (fh)             /                            \
				     0<--- fm---->1 fh/2                      fh/2  4---------------------------5
				     |                                                                         |
					 |																		   |
					 |																		   |
					 7------------------------------------------------------------------------ 6
			     */

					int x0, y0, x1, y1, x2, y2, x3, y3, x4, y4, x5, y5;


					if (backgroundColor != null) {
						g.setColor(backgroundColor);
						g.fillRect(x, y, width-1, height-1);
					}

					g.setColor(foregroundColor);
					g.drawRect(x,y,width-1, height-1);

					x0 = x;
					y0 = y+fh;
					x5 = x + width - 1;
					y5 = y0;

					if (isOpen) {
						g.drawLine(x0, y0, x5, y5);
					} else {
						x1 = x+fm;
						y1 = y+fh;

						g.drawLine(x0, y0, x1, y1);

						x2 = x1+fh/2;
						y2 = y;

						g.drawLine(x1, y1, x2, y2);

						x3 = x + fw;
						y3 = y2;

						g.drawLine(x2, y2, x3, y3);

						x4 = x3 + fh/2;
						y4 = y0;

						g.drawLine(x3, y3, x4, y4);
						g.drawLine(x4, y4, x5, y5);
					}
/*
					xp[0] = x;
					yp[0] = y+fh;
					xp[1] = x+fm;
					yp[1] = y+fh;
					xp[2] = xp[1]+fh/2;
					yp[2] = y;
					xp[3] = x + fw;
					yp[3] = yp[2];
					xp[4] = xp[3]+fh/2;
					yp[4] = yp[0];
					xp[5] = x + width-1;
					yp[5] = yp[0];
					xp[6] = xp[5];
					yp[6] = y + height-1;
					xp[7] = xp[0];
					yp[7] = yp[6];
					xp[8] = xp[0];
					yp[8] = yp[0];

					if (backgroundColor != null) {
						g.setColor(backgroundColor);
						g.fillPolygon(xp, yp, 9);
					}

					g.setColor(foregroundColor);
					g.drawPolygon(xp, yp, 9);

					if (!isOpen) {
						g.drawLine(xp[1], yp[1], xp[4], yp[4]);
					}
 */
					break;
				}
				case EntityClass.ENTITY_STYLE_SOURCEOBJ:
				{
					if (backgroundColor != null) {
						g.setColor(backgroundColor);
						g.fillOval(x, y, width, height);
					}
					g.setColor(foregroundColor);
					g.drawOval(x, y, width, height);
					break;
				}

				case EntityClass.ENTITY_STYLE_CLASS:
				{
					int arc_w = width/4;
					int arc_h = height/4;
					int arc   = Math.min(arc_w, arc_h);

					if (backgroundColor != null) {
						g.setColor(backgroundColor);
						g.fillRoundRect(x, y, width-1, height-1, arc, arc);
					}
					g.setColor(foregroundColor);
					g.drawRoundRect(x, y, width-1, height-1, arc, arc);
					break;
				}

				case EntityClass.ENTITY_STYLE_GROUP:
				{
					g.setColor(backgroundColor);
					Util.drawOutlineBox(g, x, y, width, height);
					break;
				}

				case EntityClass.ENTITY_STYLE_LABELLED_GROUP:
				{
					String label;

					if (entityInstance == null) {
						label = "";
					} else {
						label = entityInstance.getEntityLabel();
					}
					g.setColor(backgroundColor);
					Util.drawGroupBox(g, x, y, width, height, label /* Label is used only for sizing info */);
					break;
				}
				case EntityClass.ENTITY_STYLE_TRIANGLE:
				{
					switch (direction) {
					case 0:
						xp[0] = x;
						yp[0] = y+height-1;
						xp[1] = x+width/2;
						yp[1] = y;
						xp[2] = x+width-1;
						yp[2] = yp[0];
						break;
					case 1:
						xp[0] = x;
						yp[0] = y;
						xp[1] = x+width-1;
						yp[1] = y+height/2;
						xp[2] = xp[0];
						yp[2] = y+height-1;
						break;
					case 2:
						xp[0] = x;
						yp[0] = y;
						xp[1] = x + width - 1;
						yp[1] = yp[0];
						xp[2] = x + width/2;
						yp[2] = y + height - 1;
						break;
					case 3:
						xp[0] = x + width - 1;
						yp[0] = y;
						xp[1] = x;
						yp[1] = y + height/2;
						xp[2] = xp[0];
						yp[2] = y + height - 1;
						break;
					}
					polygon_dimension = 4;
					break;
				}
				case EntityClass.ENTITY_STYLE_ROMBUS:
				{
					switch (direction) {
					case 0:
						xp[0] = x;
						yp[0] = y+height-1;
						xp[1] = x+width/5;
						yp[1] = y;
						xp[2] = x+width-1;
						yp[2] = y;
						xp[3] = x+(width*4)/5;
						yp[3] = y+height-1;
						break;
					case 1:
						xp[0] = x;
						yp[0] = y;
						xp[1] = x+width-1;
						yp[1] = y+height/5;
						xp[2] = xp[1];
						yp[2] = y+height-1;
						xp[3] = xp[0];
						yp[3] = y+(4*height)/5;
						break;
					case 2:
						xp[0] = x+width/5;
						yp[0] = y+height-1;
						xp[1] = x;
						yp[1] = y;
						xp[2] = x+(4 *width)/5;
						yp[2] = y;
						xp[3] = x+width-1;
						yp[3] = y+height-1;
						break;
					case 3:
						xp[0] = x;
						yp[0] = y+height/5;
						xp[1] = x+width-1;
						yp[1] = y;
						xp[2] = xp[1];
						yp[2] = y+(4*height)/5;
						xp[3] = xp[0];
						yp[3] = y+height-1;
						break;
					}
					polygon_dimension = 5;
					break;
				}
				case EntityClass.ENTITY_STYLE_TRAPEZOID:
				{
					switch (direction) {
					case 0:
						xp[0] = x;
						yp[0] = y+height-1;
						xp[1] = x+width/5;
						yp[1] = y;
						xp[2] = x+(4*width)/5;
						yp[2] = yp[1];
						xp[3] = x+width-1;
						yp[3] = yp[0];
						break;
					case 1:
						xp[0] = x;
						yp[0] = y;
						xp[1] = x + width - 1;
						yp[1] = y + height/5;
						xp[2] = xp[1];
						yp[2] = y + (4*height)/5;
						xp[3] = xp[0];
						yp[3] = y + height - 1;
						break;
					case 2:
						xp[0] = x;
						yp[0] = y;
						xp[1] = x+width-1;
						yp[1] = yp[0];
						xp[2] = x+(4*width)/5;
						yp[2] = y+height-1;
						xp[3] = x+width/5;
						yp[3] = yp[2];
						break;
					case 3:
						xp[0] = x + width - 1;
						yp[0] = y;
						xp[1] = x;
						yp[1] = y + height/5;
						xp[2] = xp[1];
						yp[2] = y + (4*height)/5;
						xp[3] = xp[0];
						yp[3] = y + height - 1;
						break;
					}
					polygon_dimension = 5;
					break;
				}
				case EntityClass.ENTITY_STYLE_TRIANGLE2:
				{
					regular_dimension = 3;
					break;
				}
				case EntityClass.ENTITY_STYLE_RECTANGLE:
				{
					regular_dimension = 4;
					break;
				}

				case EntityClass.ENTITY_STYLE_PENTAGON:
				{
					regular_dimension = 5;
					break;
				}
				case EntityClass.ENTITY_STYLE_HEXAGON:
				{
					regular_dimension = 6;
					break;
				}
				case EntityClass.ENTITY_STYLE_OCTAGON:
				{
					regular_dimension = 8;
					break;
				}
				case EntityClass.ENTITY_STYLE_DECAHEDRON:
				{
					regular_dimension = 10;
					break;
				}
				case EntityClass.ENTITY_STYLE_12SIDED:
				{
					regular_dimension = 12;
					break;
				}
				case EntityClass.ENTITY_STYLE_14SIDED:
				{
					regular_dimension = 14;
					break;
				}
				case EntityClass.ENTITY_STYLE_16SIDED:
				{
					regular_dimension = 16;
					break;
				}
				case EntityClass.ENTITY_STYLE_18SIDED:
				{
					regular_dimension = 18;
					break;
				}
				case EntityClass.ENTITY_STYLE_20SIDED:
				{
					regular_dimension = 20;
					break;
				}
				case EntityClass.ENTITY_STYLE_PAPER:
				{
					int	baseline = y+(height*4)/5;

					xp[0] = x+(3*width)/8;			// x-
					yp[0] = baseline;				// | /
					xp[1] = xp[0];					// |/
					yp[1] = y+((193*height)/200);	// x
					xp[2] = x+(2*width)/3;			//    Diagonal up
					yp[2] = baseline;				//    Right top point			
					xp[3] = xp[0];
					yp[3] = yp[0];

					if (backgroundColor != null) {
						g.setColor(backgroundColor);
						g.fillRect(x,y, width, (height*4)/5);
						g.fillArc(x, y+height/2, width/2, height/2, -180, 180);
						g.fillPolygon(xp,yp, 4);								// This triangle connects the arc with the baseline

					}

					g.setColor(foregroundColor);
					g.drawLine(x, y, x, y+(height*3)/4);						// Left vertical edge
					g.drawLine(x, y, x+width-1, y);								// Top horizontal edge
					g.drawLine(x+width-1, y, x+width-1, baseline);				// Right vertical edge
					g.drawLine(x+(2*width)/3, baseline, x+width-1, baseline);

					g.drawArc(x, y+(height/2)-1, (width/2), (height/2)-1, -180, 120);
					g.drawLine( xp[1],yp[1], xp[2], yp[2]);

//					g.drawArc(x+(width/12),  baseline, x+((7*width)/6), y+((5*height)/2)-1, 120, -30);
					break;
				}
/*
				default:
				{
					// For debugging (This shows the root)
					g.setColor(Color.red);
					g.drawLine(x, y,        x+width-1, y+height-1);
					g.drawLine(x, y+height-1, x+width-1, y);
				}
*/
			}
			if (regular_dimension != 0) {
				polygon_dimension = regular_dimension + 1;

				double	angle = Math.toRadians(entityClass.getAngle() - 90.0);		// -90.0 to point up N.B. coordinate at (0,-.5)
				double	shift = 2.0 * Math.PI/((double) regular_dimension);
				double	dwidth, dheight;
				int		i;

				dwidth  = (double) width;
				dheight = (double) height;
				for (i = 0; i < regular_dimension; ++i) {
					xp[i] = x + (int) (dwidth  * (1.0 + Math.cos(angle)) * 0.5);
					yp[i] = y + (int) (dheight * (1.0 + Math.sin(angle)) * 0.5);
					angle -= shift;
				}
			}

			if (polygon_dimension != 0) {
				xp[polygon_dimension-1] = xp[0];
				yp[polygon_dimension-1] = yp[0];
				if (backgroundColor != null) {
					g.setColor(backgroundColor);
					g.fillPolygon(xp, yp, polygon_dimension);
				}

				g.setColor(foregroundColor);
				g.drawPolygon(xp, yp, polygon_dimension);
			}

			if (!isDrawRoot && !isOpen) {
				image = entityClass.getImage();
				if (image != EntityClass.ENTITY_IMAGE_NONE) {
					Color	inverse;

					g.setColor(backgroundColor);
					inverse = ColorCache.getInverse(backgroundColor.getRGB());
					g.setColor(inverse);

					paintImage(g, image, x, y, width, height);
			}	}
	}	}		

	public void paintMap(Graphics g, int x, int y, int width, int height, EntityInstance onPath, int depth)
	{
		if (width > 0 && height > 0) {
			EntityInstance	entityInstance = m_entityInstance;
			Color			backgroundColor;

			if (entityInstance.hasChildren() && depth != 1) {
				backgroundColor = null;
			} else {
				if (entityInstance == onPath) {
					backgroundColor = Color.red.darker();
				} else {
					backgroundColor = entityInstance.getInheritedObjectColor();
			}	}
			paintShape(g, entityInstance, entityInstance.getEntityClass(), x, y, width, height, false /* Not draw root */, entityInstance.isOpen(), Color.black, backgroundColor);

			if (depth < 1) {
				Vector				srcRelList = entityInstance.getSrcRelList();

				if (srcRelList != null) {
					RelationInstance	ri;
					EntityInstance		child;
					EntityClass			ec;
					int					x1, y1,width1, height1, i;


					for (i = srcRelList.size(); --i >= 0; ) {
						ri = (RelationInstance) srcRelList.elementAt(i);
						if (ri.isContainsClass()) {
							child   = ri.getDst();
							ec      = child.getEntityClass();
							if (ec.isShown()) {
								x1      = x + (int) (width  * child.xRelLocal());
								y1      = y + (int) (height * child.yRelLocal()); 
								width1  = (int) (width  * child.widthRelLocal());
								height1 = (int) (height * child.heightRelLocal());
								child.paintMap(g, x1, y1, width1, height1, onPath, depth+1);
	}	}	}	}	}	}	}

	// Icon interface (used to paint legends)

	public int getIconWidth()
	{
		return(getWidth());
	}

	public int getIconHeight()
	{
		return(getHeight());
	}

	public void paintIcon(Component c, Graphics g, int x, int y)
	{
//		System.out.println("EntityComponent.paintIcon");
		g.translate(x, y);
		paintComponent(g);
	}
}
