package lsedit;

import java.io.BufferedOutputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.OutputStream;
import java.io.PrintStream;
import java.net.URL;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Vector;

import java.awt.Color;

import javax.swing.JFrame;
import javax.swing.JPanel;

import java.util.zip.ZipInputStream;
import java.util.zip.GZIPInputStream;
import java.util.jar.JarInputStream;
import java.util.zip.ZipEntry;

/* This layer of software is merely concerned with representing the TA graph structures 
   Any access to update methods at this level will perform no signalling of changes
   These methods should only be used during the actual loading of an initial TA graph
 */

class NoFeedback implements TaFeedback 
{
	public void showProgress(String message) {}
	public void doFeedback(String message) {}
	public void showInfo(String message) {}
	public void error(String message) {}
	public void showCycle(RelationClass rc, EntityInstance e, int maxCycleSize) {}
	public void noContainRelation(String taPath) {}
	public void hasMultipleParents(RelationClass rc, EntityInstance e) {}
}

public class Ta extends JPanel implements TaFeedback
{
	public final static String INSTANCE_ID				= "$INSTANCE";	
	public final static String INHERIT_RELN				= "$INHERIT";
	public final static String CONTAIN_ID				= "contain"; 
	public final static String ROOT_ID                  = "$ROOT";

	public final static int	   UPDATE_FREQ              = 1000; 

	public static boolean		m_strict_TA			    = false;		// Set by -s option

	// Values

	protected TaFeedback			m_taFeedback;				// Listener to recieve progress feedback etc.
	protected static TaFeedback		m_noFeedback = null;

	private   Diagram				m_diagram;					// The actual diagram instance else null
	protected EntityCache			m_entityCache;				// Cache of entities in the graph
	protected RelationClass			m_defaultContainsClass;		// Default contains class

	protected EntityInstance		m_rootInstance;				// The root instance of the diagram
	protected EntityInstance		m_drawRoot;					// The current entity being drawn

	protected RelationClass			m_containsClass;			// The class that defines containment.
	protected int					m_cIndex;					// The index of the contains class that non-cached position information is for
	protected boolean				m_undoEnabled       = false;

	protected Hashtable m_entityClasses	  = new Hashtable(10);	// Entity classes

	// The set of relationClasses
	protected Hashtable m_relationClasses = new Hashtable(5);	// Relation classes
	private   Vector	m_numToRel		  = new Vector(10);

	public    RelationClass		m_relationBaseClass; 
	public    EntityClass		m_entityBaseClass;

	protected RelationClass		m_defaultRelationClass = null;
	protected int				m_numRelationClasses = 0;

	protected EntityClass		m_defaultEntityClass = null;
	private   int				m_numEntityClasses = 0; 

	private   boolean			m_universalScheme = false;

	private   Object			m_context;
	private	  String			m_zipEntry;

	/* Must detect use of legacy coordinates and convert after diagram loaded
	 * since attributes may not be in tree order so won't know parent dimensions
	 * necessarily when doing conversion from legacy coordinates to relative ones
	 */

	boolean						m_uses_local_coordinates;

	private boolean				m_schemeSetup = false;
	protected String			m_resString = null;

	protected int				m_progressCount;
	protected boolean			m_changedFlag = false;
	
	protected int				m_numberRelations;

	private	  String			m_comments = null;

	public Ta(TaFeedback taFeedback) 
	{
		if (taFeedback == null) {
			if (m_noFeedback == null) {
				m_noFeedback = new NoFeedback();
			}
			taFeedback = m_noFeedback;
		}
		m_taFeedback   = taFeedback;
		m_entityCache  = new EntityCache();
		m_diagram      = getDiagram();
		m_rootInstance = null;

		// Must create this before create $ROOT
		// Need a navlink attribute on rootInstance

		m_entityBaseClass = new EntityClass(EntityClass.ENTITY_BASE_CLASS_ID, m_numEntityClasses++, this);							// $ENTITY
		m_entityBaseClass.setStyle(EntityClass.ENTITY_STYLE_3DBOX);
		m_entityBaseClass.setObjectColor(Color.blue);		// Fill color
		m_entityBaseClass.setLabelColor(Color.cyan);		// Label color

		m_entityClasses.put(EntityClass.ENTITY_BASE_CLASS_ID, m_entityBaseClass);

		m_relationBaseClass =	 new RelationClass(RelationClass.RELATION_BASE_CLASS_ID, m_numRelationClasses++, this);	// $RELATION
		m_relationBaseClass.setIOfactor(0.5);
		m_relationBaseClass.setStyle(Util.LINE_STYLE_NORMAL);
		m_relationBaseClass.setObjectColor(Color.black);

		m_relationClasses.put(RelationClass.RELATION_BASE_CLASS_ID, m_relationBaseClass);													// Add to hash table
		m_numToRel.addElement(m_relationBaseClass);

		RelationClass rc = new RelationClass(CONTAIN_ID, m_numRelationClasses++, this);								// contain
		rc.setCIndex(0);
		m_defaultContainsClass = rc;
		m_relationClasses.put(CONTAIN_ID, rc);																							// Add to hash table
		m_numToRel.addElement(rc);
	}

	// Overloaded by Diagram to return the actual diagram pointer

	protected Diagram getDiagram()
	{
		return(null);
	}

	public int getCIndex()
	{
		return m_cIndex;
	}

	public RelationClass getContainsClass()
	{
		return m_containsClass;
	}

	public String getContainsId()
	{
		return m_containsClass.getLabel();
	}

	public int getContainsNid()
	{
		return m_containsClass.getNid();
	}

	public boolean getChangedFlag()
	{
		return m_changedFlag;
	}

	public boolean undoEnabled()
	{
		return m_undoEnabled;
	}

	public Vector getClassAndSubclasses(EntityClass ec)
	{
		return (ec.getClassAndSubclasses(m_entityClasses));
	}

	public EntityClass getEntityClass(String id)
	{
		return (EntityClass) m_entityClasses.get(id); 
	}

	public EntityInstance getRootInstance()
	{
		return m_rootInstance;
	}

	public EntityInstance getDrawRoot() 
	{
		return m_drawRoot; 
	}

	public Vector getClassAndSubclasses(RelationClass rc)
	{
		return rc.getClassAndSubclasses(m_relationClasses);
	}

	public RelationClass getRelationClass(String id)
	{
		return (RelationClass) m_relationClasses.get(id);
	}

	protected int getMaxCIndex()
	{
		Enumeration		en;
		RelationClass	rc;
		int				cindex, cindex1;

		cindex = -1;
		for (en = enumRelationClasses(); en.hasMoreElements(); ) {
			rc = (RelationClass) en.nextElement();
			cindex1 = rc.getCIndex();
			if (cindex1 > cindex) {
				cindex = cindex1;
		}	}
		return cindex;
	}

	protected int getCIndex(RelationClass containsClass)
	{
		int cindex = containsClass.getCIndex();
		
		if (cindex < 0) {
			cindex = getMaxCIndex();
			containsClass.setCIndex(++cindex);
		} 
		return cindex;
	}

	// Setup for universal scheme
	// This is employed if there are no SCHEME TUPLES

	public boolean isUniversalScheme()
	{
		return m_universalScheme;
	}

	public boolean entityExists(String name) 
	{
		return (getCache(name) != null);
	}

	public EntityInstance getCache(String id)
	{
		return m_entityCache.get(id);
	} 

	public void removeCache(EntityInstance e)
	{
		m_entityCache.remove(e);
	}

	public void putCache(EntityInstance e)
	{
		m_entityCache.put(e);
	}

	public Enumeration enumRelationClasses() 
	{
		return m_relationClasses.elements(); 
	}

	public Enumeration enumRelationClassesInOrder() 
	{
		return OrderedHashTableEnumeration.elements(m_relationClasses); 
	}

	public int numRelationClasses() 
	{
		return m_numRelationClasses;
	}

	public Enumeration enumEntityClasses() 
	{
		return m_entityClasses.elements(); 
	}

	public Enumeration enumEntityClassesInOrder() 
	{
		return OrderedHashTableEnumeration.elements(m_entityClasses); 
	}

	public int numEntityClasses() 
	{
		return m_entityClasses.size();
	}

	public Object getContext() 
	{
		return m_context;
	}

	public void setContext(Object context) 
	{
		m_context = context;
	}

	public String getContextName() 
	{
		String ret;

		if (m_context instanceof File) {
			ret = Util.nameFromPath(((File) m_context).getPath());
		} else if (m_rootInstance == null) { 
			ret = null;
		} else {
			ret = m_rootInstance.getEntityLabel();
		}
		if (m_zipEntry != null) {
			ret += "[" + m_zipEntry + "]";
		}
		return ret;
	}

	public String getDir(File file) 
	{
		if (file.isAbsolute()) {
			return file.getParent();
		}
		return (new File(file.getAbsolutePath())).getParent();
	}

	public String getDir() 
	{
		return getDir((File) m_context);
	}

	public String getAbsolutePath() 
	{
		if (m_context != null) {
			if (m_context instanceof File) {
				return ((File) m_context).getAbsolutePath();
			}
//			return ((URL) m_context).toExternalForm();
		}
		return null;
	}

	public Vector getRelationClasses()
	{
		return m_numToRel;
	}

	public RelationClass numToRelationClass(int n) 
	{
		return (RelationClass) m_numToRel.elementAt(n);
	}

	public int getNumberEntitiesLoaded()
	{
		return m_entityCache.size();
	}

	public int getNumberRelationsLoaded()
	{
		return m_numberRelations;
	}

	// --------------
	// Update methods
	// --------------

	private void createRootInstance()
	{
		m_rootInstance = new EntityInstance(null, ROOT_ID); 	// $ROOT
		m_rootInstance.setRelLocal(0.0, 0.0, 1.0, 1.0);
	}

	private void setupUniversalScheme() 
	{
		m_relationBaseClass.addRelationConstraint(m_entityBaseClass, m_entityBaseClass);
		m_universalScheme = true;
	}

	// After this method has been called however many times
	// need to fill again

	protected EntityClass addEntityClass(String id) 
	{
		EntityClass ec = getEntityClass(id); 

		if (ec == null) {
			ec = new EntityClass(id, m_numEntityClasses++, this);
			m_entityClasses.put(id, ec);
			if (m_defaultEntityClass == null) {
				setDefaultEntityClass(ec);
			}
		}
		return ec;
	}

	public void setDefaultRelationClass(RelationClass rc)
	{
		m_defaultRelationClass = rc;
	}

	protected RelationClass addRelationClass(String id) 
	{
		RelationClass rc = getRelationClass(id);		// Lookup in the hash table

		if (rc == null) {
			rc = new RelationClass(id, m_numRelationClasses, this);
			m_relationClasses.put(id, rc);
			m_numToRel.addElement(rc);
			m_numRelationClasses++;
			if (m_defaultRelationClass == null)
				setDefaultRelationClass(rc);
		}
		return rc;
	}

	public EntityInstance newCachedEntity(EntityClass ec, String id)
	{
		EntityInstance e = ec.newEntity(id); 

		putCache(e);
		return e;
	}

	protected RelationInstance newRelation(RelationClass rc, EntityInstance src, EntityInstance dst)
	{
		if (rc == null) {
			rc = m_defaultRelationClass;
			if (rc == null) {
				rc = m_relationBaseClass;
		}	}
		return new RelationInstance(rc, src, dst);
	}

	public RelationInstance addEdge(RelationClass rc, EntityInstance src, EntityInstance dst)
	{
		RelationInstance ri = newRelation(rc, src, dst);
		src.addSrcRelation(ri);
		dst.addDstRelation(ri);

		return ri;
	}

	public void changeIOfactor(RelationClass rc)
	{
		Enumeration en;

		for (en = m_entityClasses.elements(); en.hasMoreElements(); ) {
			EntityClass ec = (EntityClass) en.nextElement(); 
			ec.changeIOfactor(rc);
		}
	}

	private void computeRelCoordinates()
	{
		m_rootInstance.computeRelCoordinates(m_rootInstance.xRelLocal(), m_rootInstance.yRelLocal(), m_rootInstance.widthRelLocal(), m_rootInstance.heightRelLocal() );
	}

	public void setDefaultEntityClass(EntityClass ec)
	{
		m_defaultEntityClass   = ec;
	}

	public void noDiagram() 
	{
		m_uses_local_coordinates = false;
		m_schemeSetup = true;
		setupUniversalScheme();

		setDefaultEntityClass(m_entityBaseClass);
		setDefaultRelationClass(m_relationBaseClass);
		setContainsClass(m_defaultContainsClass);
		m_rootInstance = null;
	} 

	public void emptyDiagram() 
	{
		m_uses_local_coordinates = false;
		m_schemeSetup = true;
		setupUniversalScheme();

		setDefaultEntityClass(m_entityBaseClass);
		setDefaultRelationClass(m_relationBaseClass);
		m_rootInstance = newCachedEntity(m_entityBaseClass, ROOT_ID);
		switchContainsClass(m_defaultContainsClass, null);
	} 

	private void compactEntities()
	{
		EntityCache		entityCache = m_entityCache;
		EntityInstance	e;

		for (e = entityCache.getFirst(); e != null; e = entityCache.getNext()) {
			e.compact();
	}	}

	protected void switchContainsClass(RelationClass containsClass, Vector newForest)
	{
		EntityInstance	rootInstance  = m_rootInstance;

		setContainsClass(containsClass);
		rootInstance.removeAllEdges();

		if (newForest != null) {
			int				size = newForest.size();
			EntityInstance	child;
			int				i;

			for (i = 0; i < size; ++i) {
				child = (EntityInstance) newForest.elementAt(i);
				addEdge(containsClass, m_rootInstance, child);	// Not cached
		}	}	

		// This will assign it a contain index if not having one
		int cindex = getCIndex(containsClass);

		if (cindex != m_cIndex) {
			int	lth    = getMaxCIndex() + 1;
			
			m_rootInstance.exchangePositioning(m_cIndex, cindex, lth);
			m_cIndex = cindex;
		}
		containsClass.setActive(false);

		if (m_defaultRelationClass == containsClass) {
			setDefaultRelationClass(m_relationBaseClass);
		}
		prepostorder();
	}

	// ------------------
	// TA Reading methods
	// ------------------

	private void 
	processSchemeTuples(LandscapeTokenStream ts) throws IOException 
	{
		RelationClass	rc1, rc2;
		String			verb, object;
		EntityClass		ec1, ec2;
		String			msg;

		while (ts.nextSchemaTriple()) {

			verb    = ts.m_verb;
			object  = ts.m_object;

			if (verb.equals(INHERIT_RELN)) {							// $INHERIT
				switch (ts.m_relations) {
				case 0:	/* Neither term is a relation */
					if (object.equals(EntityClass.ENTITY_BASE_CLASS_ID)) {	// $ENTITY
						ts.errorNS("Improper use of $ENTITY with $INHERIT");
						break;
					}
					ec1 = addEntityClass(object); 
					ec2 = addEntityClass(ts.m_subject);
					msg = ec1.addParentClass(ec2);
					if (msg != null) {
						ts.errorNS(msg);
					}
					break;
				case 1:	/* First  is relation		*/
				case 2:	/* Second is relation		*/
					ts.errorNS("Mismatched entity/relation with $INHERIT -- presuming both relations");
				case 3:	/* Both term is a relation	*/
					if (object.equals(RelationClass.RELATION_BASE_CLASS_ID)) {	// $ENTITY
						ts.errorNS("Improper use of $RELATION with $INHERIT");
						break;
					}
					rc1 = addRelationClass(object);
					rc2 = addRelationClass(ts.m_subject);
					msg = rc1.addParentClass(rc2);
					if (msg != null) {
						ts.errorNS(msg);
					}
					break;
				}

			} else { 
				if (ts.m_relations != 0) {
					ts.errorNS("Cannot create relationships between relationships");
				} else {
					ec1 = addEntityClass(object); 
					ec2 = addEntityClass(ts.m_subject);
					rc1 = addRelationClass(verb);
					rc1.addRelationConstraint(ec1, ec2);	// If not already present
			}	}
		}
	}

	private void processFactTuples(LandscapeTokenStream ts) throws IOException 
	{
		String			verb, object, subject;
		EntityInstance	e, e1, e2;
		int				ne = 0;
		int				nr = 0;
		EntityClass		ec, ec1;
		RelationClass	rc;

		m_numberRelations = 0;
		MsgOut.vprint("\nFACT TUPLE : ");

		while (ts.nextFactTriple()) {

			verb    = ts.m_verb;
			object  = ts.m_object;
			subject = ts.m_subject;

			if (verb.equals(INSTANCE_ID)) {

				// 
				// $INSTANCE instanceId classId
				//

				ne++;
		
				if ((ne % UPDATE_FREQ) == 0) {
					MsgOut.vprint(".");
					m_taFeedback.showProgress("Entities: " + ne);
				}		

				if (object.equals(ROOT_ID)) {
					e = getRootInstance();
				} else {
					e = getCache(object);
				}

				if (!m_strict_TA) {
					ec = addEntityClass(subject);
				} else {
					ec = getEntityClass(subject);
				}
				if (ec == null) {
					ts.errorNS("Strict TA: Entity '" + object + "' instance of undeclared entity class '" + subject + "'");
				} else if (e == null) {

					// New $INSTANCE
					e = newCachedEntity(ec, object);
				} else {
					ec1 = e.getEntityClass();
					if (ec1 == null) {
						// Special case for $ROOT
						e.setParentClass(ec);
					} else {
						if (ec1.hasId(subject)) {
							ts.warning("Redeclaration of " + e.toString() + " as instanceof " + subject);
						} else {
							ts.errorNS("Attempt to declare " + e.getId() + " as instanceof " + subject + ". Currently declared as instanceof " + ec1.getId());
						}
				}	}
			} else {

				// Relation tuple definition
				//
				// relationClass entityInstance entityInstance
				//

				nr++;
				++m_numberRelations;

				if ((nr % UPDATE_FREQ) == 0) {

					MsgOut.vprint(".");
					m_taFeedback.showProgress("Relations: " + nr);
				}		

				e1 = getCache(object);
				e2 = getCache(subject);

				if (!m_strict_TA) {
					rc  = addRelationClass(verb);
					if (e1 == null) {
						e1 = newCachedEntity(m_entityBaseClass, object);
					}

					if (e2 == null) {
						e2 = newCachedEntity(m_entityBaseClass, subject);
					}
				} else {
					rc  = getRelationClass(verb);
				}

				if (rc == null || e1 == null || e2 == null) {
					String msg = "Strict TA: Relation " + "(" + verb + " " + object + " " + subject + ")";
					if (rc == null) {
						msg += " member of undeclared relation class '" + verb + "'";
					}
					if (e1 == null) {
						msg += " has undeclared source entity '" + object + "'";
					} 
					if (e2 == null) {
						msg += " has undeclared destination entity '" + subject + "'";
					}
					ts.errorNS(msg);
				} else {
					addEdge(rc, e1, e2);
				}
			}
		}
		m_numberRelations = nr;
	}

	private void parseStream(InputStream is, String src, URL context) 
	{
		Enumeration		en;
		RelationClass	rc1;
		EntityClass		ec1;
		EntityInstance	root;

		MsgOut.vprintln("Parse TA file: " + src);

//		System.out.println("" + Thread.currentThread());

		m_uses_local_coordinates = false;
		m_cIndex                 = 0;	// It is the contains that get loaded into the active positioning info

		try {
			InputStreamReader	reader       = new InputStreamReader(is);
			LineNumberReader	linenoReader = new LineNumberReader(reader);

			LandscapeTokenStream ts = new LandscapeTokenStream(linenoReader, src, m_entityCache);

			linenoReader.setLineNumber(1);

			ts.m_comments = "";

			for (;;) {
				int sec = ts.nextSection();

				if (sec == LandscapeTokenStream.EOF) {
					MsgOut.vprintln("");
					break;
				}
				try { 
					switch(sec) {

					case LandscapeTokenStream.SCHEME_TUPLE:

						m_schemeSetup = true;
						m_universalScheme = false;
				
						processSchemeTuples(ts);
						break;



					case LandscapeTokenStream.SCHEME_ATTRIBUTE:

						if (!m_schemeSetup) {
							m_schemeSetup = true;
							setupUniversalScheme();
						}
						ts.processSchemeAttributes(this);
						break;



					case LandscapeTokenStream.FACT_TUPLE:

						if (ts.m_comments.length() > 0) {
							m_comments = ts.m_comments;
						} else {
							m_comments = null;
						}
						ts.m_comments = null;

						if (!m_schemeSetup) {
							m_schemeSetup = true;
							setupUniversalScheme();
						}

						processFactTuples(ts);
						break;



					case LandscapeTokenStream.FACT_ATTRIBUTE:

						compactEntities();
						ts.processFactAttributes(this);
						break;


					case LandscapeTokenStream.INCLUDE_FILE:

						if (m_context == null) {
							if (!parseFile(ts.getIncludeFile(), null, false)) {
								return;
							}
						} else {
							if (!parseURL(ts.getIncludeFile(), context, false)) {
								return;
						}	}
						MsgOut.vprintln("Back to TA file: " + src);
						break; 
					}

				} catch (IOException e) {
					MsgOut.println("IO error reading landscape"); 
					m_resString = e.toString();
					break; 
				}
			}
			linenoReader.close(); 

		} catch (Exception e) {
			m_resString = e.toString();
			System.out.println("Parse error: " + m_resString);
		}

		root = getRootInstance();
		if (root.getParentClass() == null) {
			root.setParentClass(m_entityBaseClass);
		}
		for (en = m_entityClasses.elements(); en.hasMoreElements(); ) {
			ec1 = (EntityClass) en.nextElement(); 
			if (ec1 != m_entityBaseClass  && ec1.getParentCnt() == 0) {
				ec1.addParentClass(m_entityBaseClass);
		}	}

		for (en = m_relationClasses.elements(); en.hasMoreElements(); ) {
			rc1  = (RelationClass) en.nextElement();
			if (rc1 != m_relationBaseClass && rc1.getParentCnt() == 0) {
				rc1.addParentClass(m_relationBaseClass);
		}	}


	} // end parseStream() 

	private InputStream decompress(InputStream is, String source, String subfile)
	{
		InputStream ret      = is;
		int			lth      = source.length();

		if (lth > 4) {
			String ends = source.substring(lth - 4);
			if (ends.equalsIgnoreCase(".zip")) {
				ZipInputStream	zipInputStream;
				ZipEntry		zipEntry;

				try {
					zipInputStream = new ZipInputStream(is);
					for (;;) {
						zipEntry = zipInputStream.getNextEntry();
						if (subfile == null || subfile.equalsIgnoreCase(zipEntry.getName())) {
							break;
						}
						zipInputStream.closeEntry();
					}
				} catch (Exception e) {
					System.out.println("Attempt to open " + source + ((subfile == null) ? "" : "#" + subfile) + " as zip file failed");
					m_resString    = e.toString();
					zipInputStream = null;
				}
				return zipInputStream;
			}
			if (ends.equalsIgnoreCase(".jar")) {
				JarInputStream	jarInputStream;
				ZipEntry		zipEntry;

				try {
					jarInputStream = new JarInputStream(is);
					for (;;) {
						zipEntry = jarInputStream.getNextEntry();
						if (subfile == null || subfile.equalsIgnoreCase(zipEntry.getName())) {
							break;
						}
						jarInputStream.closeEntry();
					}
				} catch (Exception e) {
					System.out.println("Attempt to open " + source + ((subfile == null) ? "" : "#" + subfile) + " as jar file failed");
					m_resString    = e.toString();
					jarInputStream = null;
				}
				return jarInputStream;
			}

			if (lth > 5) {
				ends = source.substring(lth - 5);
				if (ends.equalsIgnoreCase(".gzip")) {
					GZIPInputStream	gzipInputStream;

					try {
						gzipInputStream = new GZIPInputStream(is);
					} catch (Exception e) {
						System.out.println("Attempt to open " + source + " as gzip file failed");
						m_resString     = e.toString();
						gzipInputStream = null;
					}
					return gzipInputStream;
		}	}	}
		return is;
	}

	private boolean parseURL(String taPath, URL context, boolean topLevel) 
	{
		int		lth   = taPath.length();
		char	lc	  = taPath.charAt(lth-1);
		String	entry = null;

		if (lth > 2 &&  lc == ']') {
			int	i = taPath.lastIndexOf('['); 

			if (i > 0 && i < lth - 2) {
				entry  = taPath.substring(i+1, lth-1);
				taPath = taPath.substring(0, i);
				if (m_zipEntry == null) {
					m_zipEntry = entry;
				}
				lth    = i;
				lc     = taPath.charAt(lth-1);
		}	}

		if (lc == File.separatorChar) {
			taPath = taPath.substring(0, lth-1);
		}

		URL		lsURL;

		try {
			if (context == null) {
				lsURL = new URL(taPath);
			} else {
				lsURL = new URL(context, taPath);
			}

			MsgOut.dprintln("Opening URL: " + taPath);
			InputStream is = lsURL.openStream();

			is = decompress(is, taPath, entry);
			if (is == null) {
				return false;
			}
			MsgOut.dprintln("opened");

			parseStream(is, taPath, lsURL);
		}
		catch(Exception e) {
			m_resString = e.toString();
			return false;
		}

		if (topLevel) {
			setContext(lsURL);
		}
		return true;
	}


	// Called from diagram.loadDiagram to parse a file or an include within a file
	// Also called from ClusterInterface to load an import

	public boolean parseFile(String taPath, Object context, boolean topLevel) 
	{
		String	entry = null;
		int		lth   = taPath.length();
		File	file  = null;

		if (lth > 0) {
			char	lc	  = taPath.charAt(lth-1);

			if (lth > 2 &&  lc == ']') {
				int	i = taPath.lastIndexOf('['); 

				if (i > 0 && i < lth - 2) {
					entry  = taPath.substring(i+1, lth-1);
					if (m_zipEntry == null) {
						m_zipEntry = entry;
					}
					taPath = taPath.substring(0, i);
					lth    = i;
					lc     = taPath.charAt(lth-1);
			}	}
			if (lc == File.separatorChar) {
				taPath = taPath.substring(0, lth-1);
		}	}	
		
		try {
			InputStream	is   = null;

			if (lth == 0) {
				if (context instanceof InputStream) {
					is = (InputStream) context;
				} 
			} else {
				if (context instanceof File) {
					String dir = getDir((File) context);
					file = new File(dir, taPath);
				} else {
					file = new File(taPath);
				}
				if (file != null) {
//					System.out.println(taPath);
					is = new FileInputStream(file);
					is = decompress(is, taPath, entry);
			}	}
			if (is == null) {
				m_resString = "No input stream specified";
				return false;
			}

			parseStream(is, taPath, null);
			is.close();

		} catch (Exception e) {
			m_resString = e.toString();
			return false;
		}
		if (topLevel && file != null) {
			setContext(file);
		}
		return true;
	}

	public void prepostorder()
	{
		m_rootInstance.prepostorder(1);
	}

	// This does just enough to allow navigation by contains edge
	// Call this if root == null

	public void setContainsClass(RelationClass containsClass)
	{
		Enumeration		en;
		RelationClass	rc;

		for (en = enumRelationClasses(); en.hasMoreElements(); ) {
			rc = (RelationClass) en.nextElement();
			rc.setContainsClass(rc == containsClass);
		}
		m_containsClass = containsClass;
	}

	public Vector getForest()
	{
		RelationClass		oldContainsClass = m_containsClass;
		Enumeration			en = m_rootInstance.srcRelationElements();
		Vector				forest;
		RelationInstance	ri;
		RelationClass		rc;

		forest = new Vector();
		if (en != null) {
			while (en.hasMoreElements()) {
				ri = (RelationInstance) en.nextElement();
				rc = ri.getRelationClass();
				if (rc == oldContainsClass) {
					forest.addElement(ri.getDst());
		}	}	}
		return(forest);
	}

	protected Vector establishForest(RelationClass containsClass)
	{
		RelationClass	oldContainsClass = m_containsClass;
		EntityInstance	root, parent, nextroot;
		int				i, size;
		String			containsLabel;
		Vector			oldForest, newForest;
		boolean			ok;


		containsLabel = containsClass.getLabel();

		removeCache(m_rootInstance);		// Dont want to see this while searching for forests
		nextroot   = null;
		newForest  = new Vector();
		size       = m_entityCache.size();
		root       = m_entityCache.someEntity();
		if (root == null) {
			ok = true;
		} else {
			ok = false;

			for (;; root = nextroot) {
				if (root == null) {
					break;
				}
				for (i = 0; ; ++i) {
					if (i > size) {
						// Must be a loop can't be a tree
						m_taFeedback.showCycle(containsClass, root, size);
						root = null;
						break;
					}
					// N.B. must pass in containsClass here or TA in stand-alone mode breaks
					// Due to attempt by getContainsBy to determine Diagram.  The entityClass
					// has diagram == null when loaded as TA.

					parent = root.getContainedBy(containsClass);
					if (parent == null || parent == m_rootInstance) {
						break;
					}
					root = parent;
				}
				if (root == null) {
					break;
				}
				// This sets the IN_TREE_MARK of every node visited
				// It visits all of the new children of root recursively
				// It explicitly uses containsClass in the navigation
				// If it hits an already marked node it knows that some node has multiple parents
				// In this case i is set -1

				i = root.cntNodesInSubtree(containsClass);
				if (i < 0) {
					break;
				}
				newForest.addElement(root);
				size -= i;

				if (size == 0) {
					ok = true;
					break;
				} 

				for (nextroot = m_entityCache.getFirst(); nextroot != null; nextroot = m_entityCache.getNext()) {
					if (!nextroot.isMarked(EntityInstance.IN_TREE_MARK)) {
						break;
				}	}
		}	}

		// Add the root instance back in
		putCache(m_rootInstance);

		m_rootInstance.clearTreeMark();

		if (!ok) {	
			newForest = null;
		}
		return newForest;
	}

	// If taPath is "" and context is Process then read from processes stdout

	public String loadTA(String taPath, Object context) 
	{
		Attribute	attr;
		boolean		option;
		boolean		ok;

		m_progressCount = 0;

		m_containsClass = null;
		m_resString     = null;
		m_zipEntry      = null;
		if (taPath == null) {
			noDiagram();
		} else {
			int		lth   = taPath.length();

			if (lth == 0 && context == null) {
				emptyDiagram();
			} else {
				createRootInstance();
				AttributeCache.activate();
				if (context instanceof URL || Util.isHTTP(taPath)) {
					MsgOut.dprintln("Parse a URL");
					ok = parseURL(taPath, (URL) context, true);
				} else {
					MsgOut.dprintln("Parse a file");
					ok = parseFile(taPath, context, true);
				}
				AttributeCache.deactivate();
				if (!ok) {
					if (m_resString == null) {
						m_resString = "Unknown error parsing " + context;
					}
					m_rootInstance = null;
					return m_resString;
				}
				putCache(m_rootInstance);
		}	}
	
		if (m_defaultEntityClass == null) {
			setDefaultEntityClass(m_entityBaseClass);
		}
		if (m_defaultRelationClass == null) {
			setDefaultRelationClass(m_relationBaseClass);
		}
		
		StringCache.clear();

		Enumeration		en;
		RelationClass	containsClass, rc;
		String			containsLabel;
		Vector			newForest;

		containsClass = null;
		for (en = m_relationClasses.elements(); en.hasMoreElements(); ) {
			rc = (RelationClass) en.nextElement();
			if (rc.isContainsClass()) {
				if (containsClass != null) {
					containsClass.setContainsClass(false);
				}
				containsClass = rc;
		}	}	
		if (containsClass == null) {
			containsClass = getRelationClass(CONTAIN_ID);
		}
		if (containsClass  == null) {
			m_taFeedback.noContainRelation(taPath);
			return "No contains relation class defined";
		}

		newForest = establishForest(containsClass);
		if (newForest == null) {
			return "Contains class does not form a forest";
		} 
		switchContainsClass(containsClass, newForest);
		
		if (m_uses_local_coordinates) {
			computeRelCoordinates();
		}
		
		if (m_rootInstance != null) {
			m_rootInstance.setDefaultOpenStatus();
		}			
		if (m_context == null) {
			return m_resString;
		}
		return null;
	}

	// -------------------
	// TA writing methoods
	// -------------------

	public void writeSchemeTuples(PrintStream ps) throws IOException 
	{
		Enumeration en, en1;
		EntityClass	ec, parent;
		RelationClass rc, rparent;

		ps.print("SCHEME TUPLE :\n\n"); 

		for (en = enumEntityClassesInOrder(); en.hasMoreElements(); ) {
			ec = (EntityClass) en.nextElement();
			if (ec != m_entityBaseClass) {
				for (en1 = ec.getParentElements(); en1.hasMoreElements(); ) {
					parent = (EntityClass) en1.nextElement();
					ps.print(INHERIT_RELN + " " + ec.getId() + " " + parent.getId() + "\n");
				}
			}
		}

		for (en = enumRelationClassesInOrder(); en.hasMoreElements(); ) {
			rc = (RelationClass) en.nextElement();
			if (rc != m_relationBaseClass) {
				for (en1 = rc.getParentElements(); en1.hasMoreElements(); ) {
					rparent = (RelationClass) en1.nextElement();
					ps.print(INHERIT_RELN + " (" + rc.getId() + ") (" + rparent.getId() + ")\n");
				}
			}
		}

		ps.print("\n");
		for (en = enumRelationClassesInOrder(); en.hasMoreElements(); ) {	// Output in same order as input
			rc = (RelationClass) en.nextElement();
			rc.writeRelations(ps); 
		}
	}

	public void writeSchemeAttributes(PrintStream ps) throws IOException 
	{
		Enumeration en;

		ps.print("\n\nSCHEME ATTRIBUTE :\n\n");
		// Write attributes for entity classes 

		for (en = m_entityClasses.elements(); en.hasMoreElements(); ) {
			EntityClass ec = (EntityClass) en.nextElement(); 
			ec.writeAttributes(ps);
		}

		// Write attributes for relation classes 

		for (en = m_relationClasses.elements(); en.hasMoreElements(); ) {
			RelationClass rc = (RelationClass) en.nextElement();
			rc.writeAttributes(ps);
		}
	}

	private void writeFactAttributes(PrintStream ps) throws IOException
	{
		Enumeration		en;
		EntityInstance	child;

		ps.print("\n\nFACT ATTRIBUTE :\n\n"); 
		m_rootInstance.writeOptionsAttributes(ps); 

		for (en = m_rootInstance.getChildren(); en.hasMoreElements(); ) {
			child = (EntityInstance) en.nextElement();
			child.writeAttributes(ps, m_cIndex);
	}	}

	private void writeFactTuples(PrintStream ps) throws IOException 
	{
		Enumeration		 en; 
		EntityInstance	 e;
		
		ps.print("\n\nFACT TUPLE :\n\n"); 

		for (en = m_rootInstance.getChildren(); en.hasMoreElements(); ) {
			EntityInstance child = (EntityInstance) en.nextElement();
			child.writeInstances(ps);
		}

		// Avoid writing m_rootInstance relations

		for (en = m_rootInstance.getChildren(); en.hasMoreElements(); ) {
			EntityInstance child = (EntityInstance) en.nextElement();
			child.writeRelations(ps);
		}
	}

	// Save the landscape in the output stream

	public void saveDiagram(OutputStream os, boolean markEnd) throws IOException
	{
		if (m_rootInstance != null) {
			BufferedOutputStream bos = new BufferedOutputStream(os);
			PrintStream ps = new PrintStream(bos);

			ps.print("// Landscape TA file written by LSEdit " + Version.Number() + "\n");
			if (m_comments != null) {
				ps.print(m_comments);
			}
			ps.print("\n");

			writeSchemeTuples(ps);
			writeSchemeAttributes(ps);
			writeFactTuples(ps);
			writeFactAttributes(ps);

			if (markEnd) {
				ps.print("END\n");
			}
			ps.flush();
			ps.close();
			m_changedFlag = false;
	}	}

	protected String saveByFile(String newPath) 
	{
		try {
			OutputStream os;

			if (newPath != null && newPath.length() == 0) {
				os = System.out;
			} else {
				File file  = (File) getContext();

				if (newPath == null) {
					String name = file.getPath();
					file.renameTo(new File(name + ".old"));
				} else {
					// New filename
					file = new File(newPath);
					setContext(file);
				}
				os = new FileOutputStream(file);
			}

			saveDiagram(os, false);
			return null;
		}
		catch(IOException e) {
			return "IOException on file open";
		}
	}

	// Wraps the TaFeedback interface

	public void showProgress(String message)
	{
		m_taFeedback.showProgress(message);
	}

	public void doFeedback(String message)
	{
		m_taFeedback.doFeedback(message);
	}

	public void showInfo(String message)
	{
		m_taFeedback.showInfo(message);
	}

	public void error(String message)
	{
		m_taFeedback.error(message);
	}

	public void showCycle(RelationClass rc, EntityInstance e, int maxCycleSize)
	{
		m_taFeedback.showCycle(rc, e, maxCycleSize);
	}

	public void noContainRelation(String taPath)
	{
		m_taFeedback.noContainRelation(taPath);
	}

	public void hasMultipleParents(RelationClass rc, EntityInstance e)
	{
		m_taFeedback.hasMultipleParents(rc, e);
	}
}

