#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <alloca.h>
#include <fcntl.h>
#include <time.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>

#include "xmalloc.h"
#include "util.h"
#include "object.h"
#include "classmember.h"
#include "directory.h"
#include "archive.h"
#include "source.h"
#include "collection.h"
#include "file.h"
#include "template.h"
#include "class.h"
#include "variable.h"
#include "function.h"
#include "edge.h"
#include "asxo/asxo.h"
#include "cx/cx.h"

#include "label.h"
#include "dwarf.h"
#include "signature_buffer.h"
#include "signature.h"

Csignature	g_signature;

//#define TRACE
// #define SANITY
#define SHOW_FILES

static char	*cwdP = 0;
static char *g_program_nameP = 0;
static int	g_ret            = 0;

static char	*asx_compileP = 0;
static char *asx_unlinkP  = 0;
static char *asx_ignoreP  = 0;
static char *asx_forceP   = 0;
static char *asx_silentP  = 0;
static char *asx_skipP    = 0;
static char *asx_liftP    = 0;
static int  compile_flag  = 1;
static int	unlink_flag   = 1;
static int	silent_flag   = 1;
static int  skip_flag     = 0;
static int  use_stdin     = 0;
static int	lineno        = 0;

variablesE	g_show_variables    = Variables_not_set;
functionsE  g_function_calls    = Functions_not_set;
int			g_lift_addresses = 0;
int			g_addresses      = 1;

int g_use_dwarf   = 1;
int g_debug_dwarf = 0;

#define SUFFIX_a    0
#define SUFFIX_so   1
#define SUFFIX_o 	2
#define SUFFIX_s 	3
#define SUFFIX_S    4
#define SUFFIX_c 	5
#define SUFFIX_C 	6
#define SUFFIX_cc 	7
#define SUFFIX_cxx	8
#define SUFFIX_cpp	9
#define SUFFIX_cpp1 10
#define SUFFIX_none 11

/* Must agree with SUFFIX_* */

static const char *suffixes[] =
{
	"a",
	"so",
	"o",
	"s",
	"S",
	"c",
	"C",
	"cc",
	"cxx",
	"cpp",
	"c++",
	0
};

static int ignore[sizeof(suffixes)/sizeof(*suffixes)] = {0};

static int force[sizeof(suffixes)/sizeof(*suffixes)] = {0};


static char	label_char[256]    = {0};
static char	function_char[256] = {0};

void
trap(void)
{
}

/* Convert the pathP to an absolute path.
 */

static char *
absolute_path(char *pathP)
{
	static char	full_path[2048];

	char	*P, *P1;

	P = full_path;
	if (*pathP == '/') {
		/* If the path is an absolute path we are done */
		strcpy(full_path, pathP);
	} else if (!cwdP || cwdP[0] != '/') {
		/* Don't know what the cwd is */
		strcpy(P, pathP);
	} else {
		strcpy(P, cwdP);
		P += strlen(cwdP);
		*P++ = '/';
		strcpy(P, pathP);
	}

	for (P = full_path; (P1 = strchr(P, '.')); ) {
		switch (P1[1]) {
		case '/':
			if (P1 == full_path || P1[-1] == '/') {
				// Erase ./
				P = P1;
				strcpy(P, P+2);
				continue;
			}
			break;
		case '.':
			if (P1[2] == '/' && P1 > full_path+1 && P1[-1] == '/') {
				// Remove <parent>/../
				for (P = P1-2; P != pathP && P[-1] != '/'; --P);
				strcpy(P, P1+3);
				continue;
			}
			break;
		}
		P = P1 + 1;
	}
	return(full_path);
}

static int
call_cc(char *argv[], char * const env[], int suffix)
{
	int		ret, pid;
	int		status;
	int 	i;

	if (!silent_flag) {
		fprintf(stderr, "%4d: ", lineno);
		for (i = 0; argv[i]; ++i) {
			fprintf(stderr, "%s ", argv[i]);
		}
		fprintf(stderr,"\n");
	}
	
	ret = fork();
	switch (ret) {
	case 0:
		execve(argv[0], argv, env);
		fprintf(stderr,"** Line %d: Return from execve %s\n", lineno, argv[0]);
		error_exit(1);
	case -1:
		fprintf(stderr,"** Line %d: Fork to cc failed\n", lineno);
		perror("");
		error_exit(2);
	}
	pid = wait(&status);
	if (pid != ret) {
		fprintf(stderr,"** Line %d: Wait for cc failed\n", lineno);
		perror("");
		error_exit(3);
	}
	if (!WIFEXITED(status)) {
		fprintf(stderr,"** Line %d: cc did not terminate normally\n", lineno);
		return(-1);
	}
	return(WEXITSTATUS(status));
}

static void
dump_instances(void)
{
	Cdirectory	*directoryP;
	Carchive	*archiveP;
	Csource		*sourceP;
	Cclass		*classP;
	Ctemplate	*templateP;
	Cfunction	*functionP;
	Cvariable	*variableP;
	Centity		*parentP;
	unsigned int	i, flags, cnt;
	int			mask;

	cnt = 0;
	for (directoryP = Cdirectory::g_headP; directoryP; directoryP = directoryP->m_nextP) {
		if (!(directoryP->m_flags & IgnoreX)) {
			directoryP->emit_instance();
			directoryP->add_emitted();
		}
		++cnt;
	}
	if (cnt != Cdirectory::g_cnt) {
		fprintf(stderr, "** !!Warning!! - Lost directories %d seen but %d created\n", cnt, Cdirectory::g_cnt);
	}

	cnt = 0;
	for (archiveP = Carchive::g_headP; archiveP; archiveP = archiveP->m_nextP) {
		archiveP->emit_instance();
		archiveP->add_emitted();
		++cnt;
	}
	if (cnt != Carchive::g_cnt) {
		fprintf(stderr, "** !!Warning!! - Lost archives %d seen but %d created\n", cnt, Carchive::g_cnt);
	}

	cnt = 0;
	for (sourceP = Csource::g_headP; sourceP; sourceP = sourceP->m_nextP) {
		sourceP->emit_instance();
		sourceP->add_emitted();
		++cnt;
	}
	if (cnt != Csource::g_cnt) {
		fprintf(stderr, "** !!Warning!! - Lost sources %d seen but %d created\n", cnt, Csource::g_cnt);
	}

	cnt = 0;
	if ((classP = Cclass::g_headP)) {
		fputs("$INSTANCE C B\n", stdout);
		for (; classP; classP = classP->m_nextP) {
			classP->emit_instance();
			classP->add_emitted();
			++cnt;
	}	}
	if (cnt != Cclass::g_cnt) {
		fprintf(stderr, "** !!Warning!! - Lost classes %d seen but %d created\n", cnt, Cclass::g_cnt);
	}

	cnt = 0;
	if ((templateP = Ctemplate::g_headP)) {
		fputs("$INSTANCE T B\n", stdout);
		for (; templateP; templateP = templateP->m_nextP) {
			templateP->emit_instance();
			templateP->add_emitted();
			++cnt;
	}	}
	if (cnt != Ctemplate::g_cnt) {
		fprintf(stderr, "** !!Warning!! - Lost templates %d seen but %d created\n", cnt, Ctemplate::g_cnt);
	}

	cnt = 0;
	for (i = 0; i < HASHSIZE; ++i) {
		for (functionP = Cfunction::g_hash[i]; functionP; functionP = functionP->m_hashP) {
			functionP->emit_instance();
			functionP->add_emitted();
			++cnt;
	}	}
	if (cnt != Cfunction::g_hash_cnt) {
		fprintf(stderr, "** !!Warning!! - Lost functions %d seen but %d created\n", cnt, Cfunction::g_hash_cnt);
	}

	switch (g_show_variables) {
	case All_variables:
		mask = 0;
		break;
	case Global_variables:
		mask = (LocalX | DynamicX | NestedX);
		break;
	case Static_variables:
		mask = (DynamicX | NestedX);
		break;
	case Static_function_variables:
		mask = DynamicX;
		break;
	case Function_parameters:
		mask = NotParamX;
		break;
	case No_variables:
		mask = -1;
	case Variables_not_set:
		break;
	}
	cnt = 0;
	for (i = 0; i < HASHSIZE; ++i) {
		for (variableP = Cvariable::g_hash[i]; variableP; variableP = variableP->m_hashP) {
			flags = variableP->m_flags;
			if (mask) {
				if (mask < 0 || (flags & mask)) {
					flags |= IgnoreX;
			}	}

			if (flags & IgnoreX) {
				variableP->m_flags = flags;
			} else {
				variableP->emit_instance();
				variableP->add_emitted();
			}
			++cnt;
	}	}
	if (cnt != Cvariable::g_hash_cnt) {
		fprintf(stderr, "** !!Warning!! - Lost variables %d seen but %d created\n", cnt, Cvariable::g_hash_cnt);
	}

	cnt = 0;
	for (i = 0; i < HASHSIZE; ++i) {
		for (variableP = Cvariable::g_addr_hash[i]; variableP; variableP = variableP->m_hashP) {
			parentP = variableP->m_parentP;
			assert(parentP);
			if (parentP->m_flags & IgnoreX) {
				variableP->m_flags |= IgnoreX;
			} else {
				variableP->emit_instance();
				variableP->add_emitted();
			}
			++cnt;
	}	}
	if (cnt != Cvariable::g_addr_hash_cnt) {
		fprintf(stderr, "** !!Warning!! - Lost addr variables %d seen but %d created\n", cnt, Cvariable::g_addr_hash_cnt);
	}

}

static void
dump_edges(void)
{
	Cobject		*objectP;

	for (objectP = Cobject::g_emitted_headP; objectP; objectP = objectP->m_emitted_nextP) {
		objectP->emit_edges();
}	}

static void
dump_attributes(void)
{
	Cobject		*objectP;
	Cedge		*edgeP;
	unsigned int i;
	unsigned int cnt;

	printf("\nFACT ATTRIBUTE :\n\n");

	if (Cclass::g_headP) {
		printf("C { label=\":classes:\" }\n");
	}
	if (Ctemplate::g_headP) {
		printf("T { label=\":templates:\" }\n");
	}

	cnt = 0;
	for (objectP = Cobject::g_emitted_headP; objectP; objectP = objectP->m_emitted_nextP) {
		objectP->emit_attributes();
		++cnt;
	}
	if (cnt != Cobject::Cobject::g_cnt) {
		fprintf(stderr, "** !!Warning!! - Lost attributes %d seen but %d created\n", cnt, Cobject::g_cnt);
	}

	cnt = 0;
	for (i = 0; i < HASHSIZE; ++i) {
		for (edgeP = Cedge::g_hash[i]; edgeP; edgeP = edgeP->m_hashP) {
			edgeP->emit_attributes();
			++cnt;
	}	}
	if (cnt != Cedge::g_hash_cnt) {
		fprintf(stderr, "** !!Warning!! - Lost edge attributes %d seen but %d created\n", cnt, Cedge::g_hash_cnt);
	}
}

static void
dump_schema(void)
{
	fputs(				"\nSCHEME TUPLE :\n\n"
		  				"$INHERIT D   $ENTITY\n"	// Directory
						"$INHERIT A   D\n"			// Archive
						"$INHERIT S   $ENTITY\n"	// Source
						"$INHERIT B   $ENTITY\n"	// Collection
			, stdout);

	if (Cclass::g_headP) {
		fputs(			"$INHERIT C   $ENTITY\n"	// Class
			,stdout);
	}

	if (Ctemplate::g_headP) {
		fputs(			"$INHERIT T   $ENTITY\n"	// Template
			,stdout);
	}

	fputs(				"$INHERIT F   $ENTITY\n"	// Function
						"$INHERIT GF  F\n"			// GlobalFunction
						"$INHERIT VF  F\n"			// VirtualFunction
						"$INHERIT EF  GF\n"			// ExternalFunction
						"$INHERIT LF  F\n"			// LocalFunction
			,stdout);

	if (g_show_variables != No_variables || g_function_calls != No_functions) {
		fputs(			"$INHERIT V   $ENTITY\n"	// Variable
			,stdout);
	}


	// Want to get the correct order in the legend

	if (Global_variables <= g_show_variables) {
		fputs(			"$INHERIT GV  V\n"			// GlobalVariable
						"$INHERIT WV  GV\n"			// WeakGlobalVariable
						"$INHERIT EV  GV\n"			// ExternalVariable
				, stdout);
	}
	if (Static_variables <= g_show_variables) {
		fputs(			"$INHERIT SV  V\n"			// StaticVariable
				, stdout);
	}
	if (Static_function_variables <= g_show_variables) {
		fputs(			"$INHERIT FV  SV\n"			// StaticFunctionVariable
				, stdout);
	}
	if (Function_parameters <= g_show_variables) {
		fputs(			"$INHERIT PV  V\n"			// ParameterVariable
				, stdout);
	}
	if (All_variables <= g_show_variables) {
		fputs(			"$INHERIT LV  V\n"			// LocalVariable
				, stdout);
	}

	if (g_addresses && (g_function_calls != No_functions || g_show_variables != No_variables)) {
		fputs(			"$INHERIT A   V\n"			// Address
			,stdout);

		if (g_function_calls != No_functions) {
			fputs(		"$INHERIT FA  A\n"			// FunctionAddress
				,stdout);
		}
		if (g_show_variables != No_variables) {
			fputs(		"$INHERIT VA  A\n"			// VariableAddress
			,stdout);
	}	}
	
	if (Cclass::g_headP) {
		if (g_show_variables != No_variables) {
			fputs(		"$INHERIT TV  WV\n"			// Vtable
						"$INHERIT TI  WV\n"			// Typeinfo
						"$INHERIT TN  WV\n"			// Typename
					    "$INHERIT TH  VF\n"			// Thunk
			, stdout);
		}
	}

	fputs(				"contain  D   D\n" 		// Directory contains Directory
		  				"contain  D   S\n"		// Directory contains Source
		  				"contain  S   F\n"		// Source    contains Functions
						"contain  C   F\n"		// Class     contains Functions
		, stdout);

	if (!g_function_calls != No_functions  || g_show_variables != No_variables) {
		if (g_show_variables != No_variables) {
			fputs(		"contain  S   V\n"		// Source    contains Variables
						"contain  C   V\n"		// Class     contains Variables
						"contain  F   V\n"		// Function  contains Variables
			, stdout);
			if (g_addresses) {
				fputs(	"contain  V   VA\n"		// Variables contain  Address
			, stdout);
			}
		} 
		if (g_addresses) {
			fputs(		"contain  F   FA\n"		// Function  contains Address
			, stdout);
	}	}

	if (Cclass::g_headP) {
		fputs(			"contain  C   TV\n"		// Class contains vtable
						"contain  C   TI\n"		// Class contains typeinfo
						"contain  C   TN\n"		// Class contains typename
						"contain  C   TH\n"		// Class contains thunks
			, stdout);
	}
	if (g_function_calls != No_functions) {
		fputs(			"CGF      F   GF\n"     // CallsGlobalFunction
						"CEF      F   EF\n"     // CallsExternalFunction
						"CLF      F   LF\n"		// CallsLocalFunction
						"CVF      F   VF\n"		// CallsVirtualFunction
						"CIF      F   FA\n"		// CallsIndirectFunction
			,stdout);
	}

	// Show in the right order in the legend

	if (Global_variables <= g_show_variables) {
		fputs(			"RGV      F   GV\n"      // ReadGlobalVariable
						"UGV      F   GV\n"      // UsesGlobalVariable
						"REV      F   EV\n"      // ReadExternalVariable
						"UEV      F   EV\n"      // UsesExternalVariable
			,stdout);
		if (g_addresses) {
			fputs(		"UVA      F   VA\n"		 // UsesVariableAddress
			,stdout);
	}	}
	if (Static_variables <= g_show_variables) {
		fputs(			"RSV      F   SV\n"      // ReadFunctionVariable
						"USV      F   SV\n"      // UsesFunctionVariable
				, stdout);
	}
	if (Static_function_variables <= g_show_variables) {
		fputs(			"RFV      F   FV\n"      // ReadFunctionVariable
						"UFV      F   FV\n"      // UsesFunctionVariable
				, stdout);
	}
	if (Function_parameters <= g_show_variables) {
		fputs(			"RPV      F   PV\n"      // ReadParameterVariable
						"UPV      F   PV\n"      // UsesParameterVariable
				, stdout);
	}
	if (All_variables <= g_show_variables) {
		fputs(			"RLV      F   LV\n"      // ReadLocalVariable
						"ULV      F   LV\n"      // UsesLocalVariable
				, stdout);
	}

	if (Cclass::g_headP) {
		fputs(			"M        C   F\n", stdout);	// Member of class is Function
		if (g_show_variables != No_variables) {
			fputs(		"M        C   V\n", stdout);	// Member of class is Variable
	}	}

	if (Ctemplate::g_headP) {
		fputs(			"MT       T   F\n"		// Member of template
			, stdout);
	}

	fputs(				"\nSCHEME ATTRIBUTE :\n\n"
		  				"$ENTITY {\n"
		  				"   color       = (255 255 255)\n"
          				"   labelcolor  = (  0   0   0)\n"
          				"   class_style = 1\n"
          				"   class_icon  = entity.png\n"
          				"   file\n"
		  				"}\n"
						"D {\n"
                        "   class_label = \"Directory\"\n"
		  				"   color       = (255 255 153)\n"
          				"   labelcolor  = (  0   0   0)\n"
          				"   class_icon  = directory.png\n"
          				"   class_style = 4\n"
		  				"}\n"
						"A {\n"
						"   class_label = \"Library\"\n"
		  				"   color       = (153 255 102)\n"
          				"   labelcolor  = (  0   0   0)\n"
          				"   class_icon  = library.png\n"
          				"   class_style = 4\n"
		  				"}\n"
						"B {\n"
						"   class_label = \"Collection\"\n"
		  				"   color       = (255 255 204)\n"
          				"   labelcolor  = (  0   0   0)\n"
          				"   class_icon  = collection.png\n"
          				"   class_style = 0\n"
		  				"}\n"
						"S {\n"
						"   class_label = \"Source\"\n"
		  				"   color       = (  0 204   0)\n"
          				"   labelcolor  = (  0   0   0)\n"
          				"   class_icon  = source.png\n"
          				"   class_style = 0\n"
		  				"}\n"
		  				"F {\n"
						"   class_label = \"Function\"\n"
          				"   class_style = 1\n"
          				"   class_icon  = function.png\n"
          				"   lineno\n"
		  				"}\n"
		  				"GF {\n"
						"   class_label = \"Global Function\"\n"
		  				"   color       = (255 102 102)\n"
          				"   class_icon  = globalfunction.png\n"
          				"   labelcolor  = (  0   0   0)\n"
          				"   class_style = 1\n"
		  				"}\n"
						"EF {\n"
						"   class_label = \"External Function\"\n"
		  				"   color       = (255   0 204)\n"
          				"   labelcolor  = (  0   0   0)\n"
          				"   class_style = 1\n"
          				"   class_icon  = externalfunction.png\n"
						"   class_description = \"Undeclared function\"\n"
		  				"}\n"
						"VF {\n"
						"   class_label = \"Virtual Function\"\n"
		  				"   color       = (170  98  98)\n"
          				"   labelcolor  = (  0   0   0)\n"
          				"   class_icon  = virtualfunction.png\n"
          				"   class_style = 1\n"
		  				"}\n"
						"LF {\n"
						"   class_label = \"Local Function\"\n"
		  				"   color       = (255 153 153)\n"
          				"   labelcolor  = (  0   0   0)\n"
          				"   class_icon  = localfunction.png\n"
          				"   class_style = 1\n"
		  				"}\n", stdout);

	if (g_show_variables != No_variables || g_function_calls != No_functions) {
		fputs(			"V {\n"
						"   class_label = \"Variable\"\n"
          				"   class_icon  = variable.png\n"
                  		"   class_style = 6\n"
			      		"}\n"
			,stdout);
	}
	switch (g_show_variables) {
	case All_variables:
		fputs(			"LV {\n"
						"  class_label = \"Local Variable\"\n"
                      	"  color       = (102 255 255)\n"
                      	"  labelcolor  = (  0   0   0)\n"
          				"  class_icon  = localvariable.png\n"
                      	"  class_style = 6\n"
						"  class_description = \"Function local variable\"\n"
			          	"}\n"
				, stdout);
	case Function_parameters:
		fputs(			"PV {\n"
						"  class_label = \"Parameter Variable\"\n"
                      	"  color       = ( 47 187 227)\n"
                      	"  labelcolor  = (  0   0   0)\n"
          				"  class_icon  = parametervariable.png\n"
                      	"  class_style = 6\n"
						"  class_description = \"Function parameter variable\"\n"
			          	"}\n"
				, stdout);
	case Static_function_variables:
		fputs(			"FV {\n"
						"  class_label = \"Static Function Variable\"\n"
                      	"  color       = (  0 204 204)\n"
                      	"  labelcolor  = (  0   0   0)\n"
          				"  class_icon  = functionvariable.png\n"
                      	"  class_style = 6\n"
						"  class_description = \"Function scope variable\"\n"
			          	"}\n"
				, stdout);
	case Static_variables:
		fputs(			"SV {\n"
						"  class_label = \"Static Variable\"\n"
                  		"  color       = (  0 204 204)\n"
                  		"  labelcolor  = (  0   0   0)\n"
          				"  class_icon  = staticvariable.png\n"
                  		"  class_style = 6\n"
						"  class_description = \"File scope variable\"\n"
			      		"}\n"
				, stdout);
	case Global_variables:
		fputs( 			"GV {\n"
						"  class_label = \"Global Variable\"\n"
                  		"  color       = (  0   0 255)\n"
                  		"  labelcolor  = (  0   0   0)\n"
          				"  class_icon  = globalvariable.png\n"
                  		"  class_style = 6\n"
			      		"}\n"
						"WV {\n"
						"  class_label = \"Weak Global Variable\"\n"
                  		"  color       = (204 204 255)\n"
                  		"  labelcolor  = (  0   0   0)\n"
          				"  class_icon  = weakvariable.png\n"
                  		"  class_style = 6\n"
			      		"}\n"
						"EV {\n"
						"  class_label = \"External Variable\"\n"
                  		"  color       = (153  51 255)\n"
                  		"  labelcolor  = (  0   0   0)\n"
          				"  class_icon  = externalvariable.png\n"
                  		"  class_style = 6\n"
						"  class_description = \"Undeclared variable\"\n"
			      		"}\n"
		,stdout);
	case No_variables:
	case Variables_not_set:
		break;
	}

	if (g_function_calls != No_functions || g_show_variables != No_variables) {
		if (g_addresses) {
			fputs(		"A {\n"
						"  class_label = \"Address\"\n"
          				"   class_icon  = address.png\n"
                      	"  class_style = 6\n"
						"  class_description = \"Address\"\n"
			          	"}\n", stdout);

			if (g_function_calls != No_functions) {
				fputs(	"FA {\n"
						"  class_label = \"Function Address\"\n"
                      	"  color       = (102 255 128)\n"
                      	"  labelcolor  = (  0   0   0)\n"
          				"   class_icon  = functionaddress.png\n"
                      	"  class_style = 6\n"
						"  class_description = \"Address of function\"\n"
			          	"}\n", stdout);
			}
			if (g_show_variables != No_variables) {
				fputs(	"VA {\n"
						"  class_label = \"Variable Address\"\n"
                      	"  color       = (102 255 128)\n"
                      	"  labelcolor  = (  0   0   0)\n"
          				"   class_icon  = variableaddress.png\n"
                      	"  class_style = 6\n"
						"  class_description = \"Address of variable\"\n"
			          	"}\n", stdout);
	}	}	}

	if (Cclass::g_headP) {
		fputs(			"C {\n"
						"   class_label = \"Class\"\n"
		      			"   color       = (255 204   0)\n"
              			"   labelcolor  = (  0   0   0)\n"
          				"   class_icon  = class.png\n"
              			"   class_style = 1\n"
			          	"}\n", stdout);
		if (g_show_variables != No_variables) {
			fputs(		"TV {\n"
						"  class_label = \"VTable\"\n"
                      	"  color       = (233 233 252)\n"
                      	"  labelcolor  = (  0   0   0)\n"
          				"   class_icon  = vtable.png\n"
                      	"  class_style = 6\n"
						"  class_description = \"Class VTable\"\n"
			          	"}\n"
						"TI {\n"
						"  class_label = \"Type Info\"\n"
                      	"  color       = (233 233 252)\n"
                      	"  labelcolor  = (  0   0   0)\n"
          				"   class_icon  = typeinfo.png\n"
                      	"  class_style = 6\n"
						"  class_description = \"Class Type Info\"\n"
			          	"}\n"
						"TN {\n"
						"  class_label = \"Type Name\"\n"
                      	"  color       = (233 233 252)\n"
                      	"  labelcolor  = (  0   0   0)\n"
          				"   class_icon  = typename.png\n"
                      	"  class_style = 6\n"
						"  class_description = \"Class Type Name\"\n"
			          	"}\n"
						"TH {\n"
						"  class_label = \"Thunk\"\n"
                      	"  color       = (233 233 252)\n"
                      	"  labelcolor  = (  0   0   0)\n"
          				"   class_icon  = thunk.png\n"
                      	"  class_style = 6\n"
						"  class_description = \"Class Thunk\"\n"
			          	"}\n"
				, stdout);
	}	}

	if (Ctemplate::g_headP) {
		fputs(			"T {\n"
						"   class_label = \"Template\"\n"
		      			"   color       = (255 204   0)\n"
              			"   labelcolor  = (  0   0   0)\n"
          				"   class_icon  = template.png\n"
              			"   class_style = 1\n"
			          	"}\n", stdout);
	}

	if (g_show_variables != No_variables || g_function_calls != No_functions) {
		fputs(			"($RELATION) {\n"
						"   file\n"
						"   lineno\n"
						"   freq\n"
						"}\n"
			, stdout);
	}

	if (g_function_calls != No_functions) {
		fputs(			"(CGF) {\n"
						"   class_label = \"Calls Global Function\"\n"
						"   color       = (153   0   0)\n"
						"   class_style = 0\n"
						"}\n"
						"(CEF) {\n"
						"   class_label = \"Calls External Function\"\n"
						"   color       = (153   0   0)\n"
						"   class_style = 0\n"
						"}\n"
						"(CLF)  {\n"
						"   class_label = \"Calls Local Function\"\n"
						"   color       = (153   0   0)\n"
						"   class_style = 0\n"
						"}\n"
						"(CVF)  {\n"
						"   class_label = \"Calls Virtual Function\"\n"
						"   color       = (255 167   0)\n"
						"   class_style = 0\n"
						"}\n"
						"(CIF) {\n"
						"   class_label = \"Calls Function by address\"\n"
						"   color       = (255 0  0)\n"
						"   class_style = 0\n"
						"}\n" 
			,stdout);
	}

	switch(g_show_variables) {
	case All_variables:
		fputs(			"(RLV) {\n"
						"   class_label = \"Read Local Variable\"\n"
						"   color       = (  0 153   0)\n"
						"   class_style = 0\n"
						"}\n"
						"(ULV) {\n"
						"   class_label = \"Updates Local Variable\"\n"
						"   color       = (204   0 204)\n"
						"   class_style = 0\n"
						"}\n"
				, stdout);
	case Function_parameters:
		fputs(			"(RPV) {\n"
						"   class_label = \"Read Parameter\"\n"
						"   color       = (  0 153   0)\n"
						"   class_style = 0\n"
						"}\n"
						"(UPV) {\n"
						"   class_label = \"Updates Parameter\"\n"
						"   color       = (204   0 204)\n"
						"   class_style = 0\n"
						"}\n"
				, stdout);
	case Static_function_variables:
		fputs(			"(RFV) {\n"
						"   class_label = \"Read Static Function Variable\"\n"
						"   color       = (  0 153   0)\n"
						"   class_style = 0\n"
						"}\n"
						"(UFV) {\n"
						"   class_label = \"Updates Static Function Variable\"\n"
						"   color       = (204   0 204)\n"
						"   class_style = 0\n"
						"}\n"
				, stdout);
	case Static_variables:
		fputs(			"(RSV) {\n"
						"   class_label = \"Read Static Variable\"\n"
						"   color       = (  0 153   0)\n"
						"   class_style = 0\n"
						"}\n"
						"(USV) {\n"
						"   class_label = \"Updates Static Variable\"\n"
						"   color       = (204   0 204)\n"
						"   class_style = 0\n"
						"}\n"
				, stdout);
	case Global_variables:
		fputs(			"(RGV) {\n"
						"   class_label = \"Read Global Variable\"\n"
						"   color       = (  0 153   0)\n"
						"   class_style = 0\n"
						"}\n"
						"(UGV) {\n"
						"   class_label = \"Updates Global Variable\"\n"
						"   color       = (204   0 204)\n"
						"   class_style = 0\n"
						"}\n"
						"(REV) {\n"
						"   class_label = \"Read External Variable\"\n"
						"   color       = (  0 153   0)\n"
						"   class_style = 0\n"
						"}\n"
						"(UEV) {\n"
						"   class_label = \"Updates External Variable\"\n"
						"   color       = (204   0 204)\n"
						"   class_style = 0\n"
						"}\n"
				, stdout);
		if (g_addresses) {
			fputs(		 "(UVA) {\n"
						"   class_label = \"Uses Variable Address\"\n"
						"   color       = (204   0 204)\n"
						"   class_style = 0\n"
						"}\n"
				, stdout);
		}
	case No_variables:
	case Variables_not_set:
		break;
	}

	if (Cclass::g_headP) {
		fputs(			"(M) {\n"
						"   class_label = \"Class Member\"\n"
						"   color       = (255 255 255)\n"
	    				"   class_style = 0\n"
						"}\n", stdout);
	}

	if (Ctemplate::g_headP) {
		fputs(			"(MT) {\n"
						"   class_label = \"Template Member\"\n"
						"   color       = (255 255 255)\n"
	    				"   class_style = 0\n"
						"}\n", stdout);
	}

	fputs(				"\nFACT TUPLE :\n\n", stdout);
}

static void
collapse_directories(void)
{
	Cdirectory	*directoryP, *parentP;
	char		*bufferP, *nameP, *parent_nameP;
	int			repeat, need;

	// Look at each directory and set m_child_dir to the number of children
	// That are directories

	for (directoryP = Cdirectory::g_headP; directoryP; directoryP = directoryP->m_nextP) {
		if ((parentP = (Cdirectory *) directoryP->m_parentP)) {
			parentP->m_child_dir++;
	}	}

	for (repeat = 1 ; repeat; ) {
		repeat = 0;
		for (directoryP = Cdirectory::g_headP; directoryP; directoryP = directoryP->m_nextP) {
			if (directoryP->m_flags & IgnoreX) {
				continue;
			}
			parentP = (Cdirectory *) directoryP->m_parentP;
			if (!parentP || parentP->m_child_dir != 1) {
				continue;
			}
			directoryP->m_parentP = parentP->m_parentP;
			parentP->m_flags     |= IgnoreX;
			parent_nameP          = parentP->m_nameP;
			repeat                = 1;

			if (!strcmp(parent_nameP, ".")) {
				continue;
			}
			nameP 		 = directoryP->m_nameP;
			if (!strcmp(nameP, ".")) {
				directoryP->name(parent_nameP);
			} else {
				need    = strlen(parent_nameP) + strlen(nameP) + 2;
				bufferP = (char *) alloca(need);
				if (!parent_nameP[0]) {
					sprintf(bufferP, "/%s", nameP);
				} else {
					sprintf(bufferP, "%s/%s", parent_nameP, nameP);
				}
				directoryP->name(bufferP);
}	}	}	}
		

static void
dump_ta(void)
{
#ifdef SANITY
	void sanity();

	fprintf(stderr, "-- Final sanity check\n");
	sanity();
#endif

	dump_schema();
	dump_instances();
	dump_edges();
	dump_attributes();
}

static int			read_object_suffix;
static int			read_object_seen_section;
static Csource		*read_object_sourceP;
static Csource		*read_object_executableP;
static Centity		*read_object_source_underP;
static Carchive		*read_object_archiveP;
static Cfunction	*read_object_functionP;
static Cfile		*read_object_fileP;
static int			read_object_lineno;
static int			read_object_seen_end;


static void
read_object_callback(int lth, char *inputP)
{
	Cfunction	*callingP;
	Cedge		*edgeP;
	char		*P, *P1, *P2, *absolute_pathP;
	int			c;

	if (!lth) {
		goto done;
	}
	switch (c = *inputP) {
	case 'A':
		if (!strncmp(inputP, "Archive ", 8)) {
			P              = inputP + 8;
			absolute_pathP = absolute_path(P);
			read_object_archiveP = new Carchive(absolute_pathP);
			if (!read_object_archiveP) {
				outofmemory();
			}
			read_object_source_underP = read_object_archiveP;
		}
		goto done;
	case 'F':
		if (!strncmp(inputP, "File ", 5)) {
			P    = inputP + 5;
			lth -= 5;
			read_object_sourceP = new Csource(P, read_object_archiveP);
			if (!read_object_sourceP) {
				outofmemory();
			}
			goto done;
		}
		if (!strncmp(inputP, "Function ", 9)) {
			P = inputP + 9;
			goto is_function;
		}
		goto done;
	case 'C':
		if (!strncmp(inputP, "Calls ", 6)) {
			P = inputP + 6;
			goto is_call;
		}
		goto done;
	case 'D':
		if (!read_object_seen_section) {
			if (!strncmp(inputP, "Disassembly of section ", 23)) {
				if (!strcmp(inputP+23, ".init")) {
					/* This is an object or .so file */
					read_object_executableP = read_object_sourceP;
			}	}
			read_object_seen_section = 1;
		}
		goto done;
	case 'S':
		if (!strncmp(inputP, "Source ", 7)) {
			P              = inputP + 7;
			absolute_pathP = 0;
			if (read_object_executableP) {
				absolute_pathP = absolute_path(P);
				read_object_sourceP = Csource::locate(absolute_pathP, read_object_executableP);
			}
			read_object_fileP = Cfile::locate(read_object_sourceP, 0);
			if (!read_object_fileP) {
				if (!absolute_pathP) {
					absolute_pathP = absolute_path(P);
				}
				read_object_fileP = new Cfile(read_object_sourceP, 0, absolute_pathP);
				if (!read_object_fileP) {
					outofmemory();
			}	}
			read_object_lineno = 0;
		}
		goto done;
	case 'L':
		if (!strncmp(inputP, "Line ", 5)) {
			P = inputP + 5;
			read_object_lineno = strtol(P, &P1, 10);
		}
		goto done;
	case 'E':
		if (!strncmp(inputP, "End of report", 13)) {
			read_object_seen_end = 1;
		}
		goto done;
	}
	if (lth < 10) {
		goto done;
	}
	if (lth > 28 && inputP[8] == ':') {
		P = inputP + 32;
		if (!strncmp(P, "call ", 5)) {
			P = strchr(P, '<');
			if (!P) {
				goto done;
			}
			++P;
			P1 = strchr(P, '>');
			if (!P1) {
				goto done;
			}
			*P1 = 0;
			if (P1-P > 4) {
				P2 = P1 - 4;
				if (!strcmp(P2, "@PLT")) {
					*P2 = 0;
			}	}
is_call:
			callingP = Cfunction::locate(P, FindX, read_object_sourceP);
			if (!read_object_functionP) {
				/* Don't know how this could happen */
				goto done;
			}
			edgeP = Cedge::locate(read_object_functionP, callingP, read_object_lineno, read_object_fileP);
			edgeP->m_freq++;
		}
		goto done;
	}

	if ((c >= '0' && c <= '9') || (c >= 'a' && c <= 'f')) {

		P = inputP + 9;
		if (*P++ != '<') {
			goto done;
		}
		P1 = strchr(P, '>');
		if (!P1) {
			goto done;
		}
		*P1 = 0;
is_function:
		read_object_functionP = Cfunction::locate(P, FindX, read_object_sourceP);
		if (!read_object_functionP->m_fileP) {
			read_object_functionP->m_fileP  = read_object_fileP;
			read_object_functionP->m_lineno = read_object_lineno;
		} 
		goto done;
	}
done:
	return;
}

static void
read_object_init(int suffix)
{
	read_object_suffix      = suffix;
	read_object_seen_section= 0;
	read_object_archiveP    = 0;
	read_object_executableP = 0;
	read_object_source_underP = 0;
	read_object_sourceP     = 0;
	read_object_fileP       = 0;
	read_object_lineno      = 0;
	read_object_functionP   = 0;
	read_object_seen_end    = 0;
}

static void
read_object_final(char *filenameP)
{
	if (!read_object_seen_end)  {
		fprintf(stderr, "** Line %d: Failed to fully process %s\n", lineno, filenameP);
		g_ret = 1;
}	}

static void
read_object(char *filenameP, int suffix)
{
	read_object_init(suffix);
	objdump_init(g_program_nameP);
	objdumpfile(filenameP, read_object_callback);
	objdump_finalize();
	read_object_final(filenameP);
}

static void
read_cx(char *filenameP, int suffix, char *argv[])
{
	read_object_init(suffix);
	cx(argv, read_object_callback);
	read_object_final(filenameP);
}

static char *
end_symbol_name(char *P)
{
	char	*P1;
	int		c;

	c = *P;
	if (c == '_' || isalpha(c)) {
		for (P1 = P; function_char[(int) *P1]; ++P1);
		return(P1);
	}
	return(0);
}

int		g_file_memory_size = 0;
char	*g_file_memoryP    = 0;

labelT	*g_labelsP         = 0;
int		g_labels_max       = -1;
int		g_labels_seen;

static void
remove_comments(void)
{
	char			*P;
	unsigned char	c, quote;
	
	g_labels_seen = 0;
	quote = 0;
	for (P = g_file_memoryP; (c = *((unsigned char *) P)); ++P) {
		if (quote) {
			if (c == quote) {
				quote = 0;
			} else if (c == '\\' && P[1]) {
				// Ignore character after escape
				++P;
			}
			continue;
		}
		switch (c) {
		case '"':
		case '\'':
			quote = c;
			break;
		case ':':
			// Might be a label
			++g_labels_seen;
			break;
		case '\t':
			// Convert tabs to space
			*P = ' ';
			break;
		case '#':
			// This is an assembler style comment
			// Erase characters from # to just before newline
			for (; (c = *((unsigned char *) P)) && c != '\n'; *P++ = ' ');
			--P;
			break;
		case '/':
			if (P[1] != '*') {
				break;
			}
			// Erase comment
			*P++ = ' ';
			*P++ = ' ';
			for (; ; ) {
				if (!(c = *((unsigned char *) P))) {
					--P;
					*P = '\n';
					--P;
					break;
				}
				*P++ = ' ';
				if (c == '*' && *P == '/') {
					*P    = ' ';
					break;
			}	}
			break;
	}	}

	if (g_use_dwarf && g_labels_max < g_labels_seen) {
		Xfree(g_labelsP);
		g_labelsP    = (labelT *) Xmalloc((g_labels_seen + 1) * sizeof(labelT));
		g_labels_max = g_labels_seen;
}	}

Cdwarf		g_dwarf;

/* In the first pass we discover any functions or variables defined in this
 * source.  That permits us to be certain that if we are looking for a
 * local function or variable (remember we don't know what it is when
 * looking for it we will bind such references to the local variable
 * rather than presuming it is global.  Without two passes we might
 * not know that a local copy of foo() existed and incorrectly bind to
 * a global copy of foo() instead.  Note that there can be at most one
 * local copy and at most one global copy, so the challenge is to simply
 * distinguish one copy from the other.
 */
 
char *g_startP;

static void
read_asm_pass1(Csource *sourceP)
{
	char		*debug_abbrevP, *debug_infoP, *debug_strP;
	char		*P;

	Cfunction	*functionP;
	Cfunction	*function1P;
	Cfile		*fileP;
	Cvariable	*usesP, *uses_nextP;
	char		*P1, *P2, *absolute_pathP;
	int			number;
	int			lno;
	int			lines_seen;
	int			seen_global, seen_local, seen_weak, flags, state;
	unsigned int	c, quote;
	labelT		*labelP, *string_labelP;
		
	fileP         = 0;
	lno           = 0;
	functionP     = 0;
	seen_global   = 0;
	seen_local    = 0;
	seen_weak     = 0;
	debug_infoP   = 0;
	debug_abbrevP = 0;
	debug_strP    = 0;

	quote         = 0;
	state         = 1;
	labelP        = g_labelsP;
	string_labelP = 0;
	lines_seen    = 1;
	for (P = g_startP = g_file_memoryP; (c = *((unsigned char *) P)); ++P) {
		if (quote) {
			if (c == quote) {
				quote = 0;
			} else if (c == '\\' && P[1]) {
				++P;
			}
			continue;
		} 
		switch (c) {
		case '"':
		case '\'':
			quote = c;
			state = 4;
			continue;
		}

		switch (state) {
		case 1:
			// Looking for start of label
			if (c == ' ') {
				continue;
			}
#if 0
			for (P1 = P; ; ++P1) {
				putchar(*P1);
				if (*P1 == '\n') {
					break;
			}	}
#endif
			if (g_use_dwarf) {
				labelP->labelP = P;
				labelP->lineno = lines_seen;
				labelP->stringP = 0;
			}
			state = 2;
		case 2:
			// Looking for end of label
			if (label_char[c]) {
				continue;
			}
			state = 3;
			if (c == ':') {
				if (g_use_dwarf) {
					string_labelP     = labelP;
					labelP->label_lth = P - labelP->labelP;
					if (labelP->label_lth) {
						++labelP;
				}	}
				// Move past the label
				continue;
			}	
			// move back to start of line
			P = g_startP;
			c = *P;
		case 3:
			// Skip initial space
			switch (c) {
			case ' ':
				continue;
			case ';':
			case '\n':
				goto end_line;
			}
			state = 0;
			break;
		case 4:
			// Find end of line
			switch (c) {
			case ';':
			case '\n':
end_line:		++lines_seen;
				state  = 1;
				g_startP = P + 1;
			}
			continue;
		}

		// State 0 if get here
			
		switch (c) {
		case '.':
			++P;
			switch (*P) {
			case 'c':
				if (!strncmp(P, "comm ", 5)) {		// .comm
					flags = 0;
					if (seen_local) {
						seen_local = 0;
						flags      = LocalX;
					}
					if (g_show_variables == No_variables) {
						break;
					}
					for (P += 5; *P == ' '; ++P);
					P1 = end_symbol_name(P);
					if (P1) {

						c     = *P1;
						*P1   = 0;

						function1P = functionP;
						// We have a local static variable inside a function
						// Determine the signature of the function it is inside
						if (!strncmp(P, "_ZZ", 3)) {
							// eg: _ZZN1IL4doitEPNS_1AEE8remember

 							if (g_signature.signature(P)) {
								P2 = g_signature.get_function();
								if (P2) {
									function1P = Cfunction::locate(P2, 0, sourceP);
									flags     |= NestedX;
						}	}	}

						// Flags [ NestedX ] [ LocalX ]
						usesP = Cvariable::locate(P, flags, sourceP, function1P);
						*P1   = c;
				}	}
				break;
			case 's':
				// Functions end at .size function name
				if (functionP && !strncmp(P, "size ", 5)) {
					P1 = functionP->m_nameP;
					P2 = P + 5;
					for (;;) {
						if (!*P1) {
							if (*P2 == ',' || *P2 == ' ') {
								functionP = 0;
							}
							break;
						}
						if (*P1++ != *P2++) {
							break;
					}	}
					break;
				}
				if (!strncmp(P, "section .", 9)) {
					P += 9;
					if (!strncmp(P, "debug_info\n", 11)) {
						assert(!debug_infoP);
						debug_infoP = P + 11;
						break;
					}
					if (!strncmp(P, "debug_abbrev\n", 13)) {
						assert(!debug_abbrevP);
						debug_abbrevP = P + 13;
						break;
					}
					if (!strncmp(P, "debug_str,", 10)) {
						assert(!debug_strP);
						for (P += 10; *P != '\n'; ++P);
						debug_strP = P+1;
						goto end_line;
				}	}
				if (string_labelP && !strncmp(P, "string ", 7)) {
					for (P += 7; *P == ' '; ++P);
					string_labelP->stringP = P;
					if (*P != '"') {
						for (;; ++P) {
							switch (*P) {
							case ' ':
							case ';':
							case '\n':
								break;
							default:
								continue;
							}
							break;
						}
					} else {
						while ((c = *++P) != '"') {
							if (c == '\\') {
								++P;
						}	}
						++P;
					}
					string_labelP->string_lth = P - string_labelP->stringP;
					string_labelP = 0;
					break;
				}
				break;
			case 'g':
				if (!strncmp(P, "globl ", 6)) {		// .globl
					seen_global = 1;
				}
				break;
			case 'w':
				if (!strncmp(P, "weak ", 5)) {		// .weak
					seen_weak = 1;
				}
				break;
			case 't':
				if (!strncmp(P, "type ", 5)) {		// .type
					flags = LocalX;
					if (seen_weak) {
						flags       = WeakX;
						seen_weak   = 0;
					}
					if (seen_global) {
						flags       = 0;
						seen_global = 0;
					}
					for (P += 5; *P == ' '; ++P);
					P1 = end_symbol_name(P);
					if (!P1) {
						break;
					}
					for (P2 = P1; *P2 == ' '; ++P2);
					if (*P2 != ',') {
						break;
					}
					for (++P2; *P2 == ' '; ++P2);
					if (!strncmp(P2, "@object", 7)) {
						if (g_show_variables != No_variables) {
							c   = *P1;
							*P1 = 0;
							flags |= InitX;
							// Flags InitX [WeakX] [LocalX]
							usesP = Cvariable::locate(P, flags, sourceP, functionP);
							*P1 = c;
						}
						break;
					}
					if (!strncmp(P2, "@function", 9)) {
						c   = *P1;
						*P1 = 0;
						functionP = Cfunction::locate(P, flags, sourceP);
						functionP->m_flags |= flags;
						*P1 = c;
						// Nested variables are declared before the
						// function that contains them
						for (usesP = Cvariable::g_local_headP; usesP; usesP = uses_nextP) {
							uses_nextP           = usesP->m_local_nextP;
							usesP->m_parentP     = functionP;
							usesP->m_flags      &= ~ExternalX;
							usesP->m_local_nextP = 0;
						}
						Cvariable::g_local_headP = 0;
					}
				}
				break;
			case 'f':
				if (!strncmp(P, "file ", 5)) {		// .file
					for (P += 5; *P == ' '; ++P);
					number = strtol(P, &P1, 10);
					if (P == P1) {
						break;
					}
					if (*P1 != ' ') {
						break;
					}
					for (++P1; *P1 == ' '; ++P1);
					if (*P1 != '"') {
						break;
					}
					P = P1 + 1;
					P1 = strchr(P, '"');
					if (!P1) {
						break;
					}
					c   = *P1;
					*P1 = 0;
					absolute_pathP = absolute_path(P);
					*P1 = c;
					P   = P1+1;
					fileP = new Cfile(sourceP, number, absolute_pathP);
					if (!fileP) {
						outofmemory();
					}
				}
				break;
			case 'l':
				if (!strncmp(P, "loc", 3)) {		// .loc
					P += 3;
					if (*P == ' ') {	
						for (; *P == ' '; ++P);
						number = strtol(P, &P1, 10);
						if (P == P1) {
							break;
						}
						fileP = Cfile::locate(sourceP, number);
						if (!fileP) {
							// This should never happen
							break;
						}
						for (P = P1; *P == ' '; ++P);
						lno = strtol(P, &P1, 10);
						if (P == P1) {
							break;
						}
						if (functionP && !functionP->m_fileP) {
							functionP->m_fileP  = fileP;
							functionP->m_lineno = lno;
						}
						break;
					}
					if (!strncmp(P, "al ", 3)) {	// .local
						seen_local = 1;
				}	}
			}
			break;
		default:
			string_labelP = 0;
		}
		state = 4;
		--P;
	}
	if (g_use_dwarf) {
		labelP->labelP = 0;
		assert((labelP - g_labelsP) <= g_labels_max);
		if (debug_abbrevP && debug_infoP) {
			g_dwarf.load(g_labelsP, debug_abbrevP, debug_infoP, debug_strP);
		} else {
			fprintf(stderr, "** Dwarf symbol abbrev=%p debug=%p information not found in %s\n", debug_abbrevP, debug_infoP, sourceP->m_nameP);
}	}	}


/*
 http://www.kernel-traffic.org/wine/wn20000228_32.html

 The linker/loader maintain a global offset table, which consists of an array
 of pointers to exported variables. All access to those variables is supposed
 to go through these pointers, if the variable could reside within another
 dynamic object. 

 If 'variable' is an exported symbol, then 'variable@GOT' is replaced by the
 linker with a constant: the offset of the GOT slot containing the pointer to
 variable. So, if the %ebx register contains the start address of the GOT, you
 can retrieve this pointer via 

 movl variable@GOT(%ebx), %eax 

 If you want the value of the variable, you need another indirection: 

 movl (%eax), %eax 

 If you know, however, that the variable in question must reside in the same
 dynamic object as the user, then you can avoid that double indirection, by
 using the GOTOFF facility. 'variable@GOTOFF' is replaced by the linker with
 another constant: the difference between the absolute address of the variable
 and the address of the global offset table (of the current dynamic object).
 Thus, you can retrieve the value of the variable in one step using: 

 movl variable@GOTOFF(%ebx), %eax 

*/

static void
read_asm_pass2(Csource *sourceP)
{
	char		*P;

	Centity		*entityP;
	Cfunction	*functionP;
	Cfile		*fileP;
	Cfunction	*callingP;
	Cclassmember *usesP;
	Cedge		*edgeP;
	char		*P1, *P2;
	int			number;
	int			lno;
	int			lines_seen, bracket, update1, update2, flags, state;
	int			in_vtable, skip_section;
	unsigned int c, c2, c3, quote;
	char		*nameP;
	dwarf_typeT	type, *typeP;

	// New versions of gcc call virtually through the *%rdx register
	// Older versions of gcc call virtually through the *%rax register
	// 32 bit machines call through *%eax

	dwarf_typeT	ventries[4];	// Reg A,B,C and D
	char		*instructionP, *firstP, *secondP;
	int			reg1, reg2;
	unsigned int offset;
	dwarf_typeT	*ventry1P, *ventry2P;

/*	
	Detecting virtual calls by watching register contents

		64-bit	32-bit 16-bit 8-bit
		RAX		EAX		AX	  AH|AL	ventries[0]
		RBX		EBX		BX	  BH|BL ventries[1]
		RCX		ECX		CX	  CH|CL ventries[2]
		RDX		EDX		DX	  DH|DL ventries[3]

		RSI		ESI		SI	  SIL
		RDI		EDI		DI	  DIL
		RBP		EBP		BP	  BPL
		RSP		ESP		SP	  SPL
		RIP		EIP		IP

	Our state machine trying to detect virtual calls is looking for

	movq    -24(%rbp), %rax 	// Class pointer -> %rax
	... 						// Optional noise not changing rax
	movq	(%rax),	%rax		// Vtable -> %rax
	... 						// Optional noise not changing rax
	addq    $8,	%rax     		// Optional -- absent if $0
	... 						// Optional noise not changing rax
   	movq    (%rax), %rax		// Function pointer
	... 						// Optional noise not changing rax
	call	*%rax

	with arbitrary switching of registers

	On a 32bit machine replace movq->movl addq->addl and rax->eax etc.

	Optional noise must not contain any labels that code might branch to.

 */
	fileP         = 0;
	lno           = 0;
	functionP     = 0;
	entityP       = 0;
	state         = 1;
	quote         = 0;
	lines_seen    = 1;
	in_vtable     = NOT_IN_VTABLE;
	skip_section  = 0;

	ventries[0].nodeP = 0;
	ventries[1].nodeP = 0;
	ventries[2].nodeP = 0;
	ventries[3].nodeP = 0;

	for (P = g_startP = g_file_memoryP; (c = *((unsigned char *) P)); ++P) {
		if (quote) {
			if (c == quote) {
				quote = 0;
			} else if (c == '\\' && P[1]) {
				++P;
			}
			continue;
		}
		switch (c) {
		case '"':
		case '\'':
			quote = c;
			goto next_line;
		}

		switch (state) {
		case 1:
			// Look for start of label
			if (c == ' ') {
				continue;
			}
#if 0
			fprintf(stderr, "%d: ", lines_seen);
			for (P1 = P; ; ++P1) {
				fputc(*P1, stderr);
				if (*P1 == '\n') {
					break;
			}	}
			if (lines_seen == 96) {
				trap();
			}
#endif
			state = 2;
		case 2:
			// Look for end of label
			if (label_char[c]) {
				continue;
			}
			state = 3;
			if (c == ':') {
				// Move past any label
				ventries[0].nodeP = 0;
				ventries[1].nodeP = 0;
				ventries[2].nodeP = 0;
				ventries[3].nodeP = 0;
				continue;
			}
			// move back to start of line
			P = g_startP;
			c = *((unsigned char *) P);
		case 3:
			// Skip initial whitespace
			switch (c) {
			case ' ':
				continue;
			case ';':
			case '\n':
				goto next_line;
			}
			state = 0;
			break;
		case 4:
			// Find end of line
			switch (c) {
			case ';':
			case '\n':
		 		++lines_seen;
				state  = 1;
				// state 0 if arrive here
				g_startP = P + 1;
			}
			continue;
		}
			
		if (skip_section) {
			if (strncmp(P, ".section ", 9)) {
				goto next_line;
		}	}
				
			
		if (c == '.') {
			// Assembler directive
			++P;		
			switch (*P) {
			case 'f':
				if (!strncmp(P, "file ", 5)) {		// .file
					for (P += 5; *P == ' '; ++P);
					number = strtol(P, &P1, 10);
					if (P == P1) {
						goto next_line;
					}
					if (*P1 != ' ') {
						goto next_line;
					}
					fileP = Cfile::locate(sourceP, number);
				}
				goto next_line;
			case 'l':
			case 'q':
				if (!strncmp(P, "quad ", 5) ||		// 8 byte pointer
					!strncmp(P, "long ", 5)) {		// 4 byte pointer
					if (!g_addresses) {
						goto next_line;
					}
					for (P += 5; *P == ' '; ++P);
					P1 = end_symbol_name(P);
					if (!P1) {
						goto next_line;
					}
					c   = *P1;
					*P1 = 0;
					flags = AddressesX | FindX;
#ifdef TODO
					if (g_function_calls > Direct_functions) {
						if (!strcmp(P, "__cxa_pure_virtual")) {
							P = 0;
							if (entityP && !strncmp(entityP->m_nameP, "_ZTV", 4) & 0 <= in_vtable) {
								P2 = g_dwarf.pure_virtual_function_name(entityP->m_nameP, in_vtable);
								if (P2) {
									flags |= PureX;
									*P1 = c;
									P   = P2;
									P1  = P + strlen(P);
									c   = *P1;
							}	}
							assert(P);
					}	}
#endif
					// Flags AddressesX FindX
					usesP = Cvariable::locate(P, flags, sourceP, functionP);
					if (!strncmp(P, "_ZTI", 4)) {
						/* For multiple inheritance the vtable is divided
						 * up into one section for each thing inherited
						 * from with a typeinfo pointer at the start of
						 * each
						 */
						in_vtable = 0;
					} else if (in_vtable != NOT_IN_VTABLE) {
						usesP->m_flags |= VirtualX;
                        if (usesP->m_vtable == NOT_IN_VTABLE) {
							usesP->m_vtable = in_vtable;
						}
						++in_vtable;
					}

					*P1 = c;
					if (entityP) {
						edgeP = Cedge::locate(entityP, usesP, lno, fileP);
						edgeP->m_freq++;
					}
					usesP->m_flags |= UsedX;
					P   = P1;
					goto next_line;
				}
				if (!strncmp(P, "loc", 3)) {		// .loc
					P += 3;
					if (*P == ' ') {	// .loc
						for (; *P == ' '; ++P);
						number = strtol(P, &P1, 10);
						if (P == P1) {
							goto next_line;
						}
						fileP = Cfile::locate(sourceP, number);
						if (!fileP) {
							// This should never happen
							goto next_line;
						}
						for (P = P1; *P == ' '; ++P);
						lno = strtol(P, &P1, 10);
					}
					goto next_line;
				}
				goto next_line;
			case 's':
				// terminate a function X if we see .size X [ ,]
				if (functionP && !strncmp(P, "size ", 5)) {
					P1 = functionP->m_nameP;
					P2 = P + 5;
					for (;;) {
						if (!*P1) {
							if (*P2 == ' ' || *P2 == ',') {
								functionP = 0;
								entityP   = 0;
								ventries[0].nodeP = 0;
								ventries[1].nodeP = 0;
								ventries[2].nodeP = 0;
								ventries[3].nodeP = 0;
							}
							goto next_line;
						}
						if (*P1++ != *P2++) {
							goto next_line;
					}	}
				}
				if (!strncmp(P, "section ", 8)) {
					skip_section = 0;
					entityP      = 0;
					in_vtable    = NOT_IN_VTABLE;
					P += 8;
					if (!strncmp(P, ".debug_", 7)) {
						P += 7;
						if (!strncmp(P, "info\n", 5) ||
							!strncmp(P, "abbrev\n", 7) ||
							!strncmp(P, "str,", 4)) {
							skip_section = 1;
					}	}
					goto next_line;
				}
				goto next_line;
			case 't':
				if (!strncmp(P, "type ", 5)) {		// .type
					entityP   = 0;
					in_vtable = NOT_IN_VTABLE;
					for (P += 5; *P == ' '; ++P);
					P1 = end_symbol_name(P);
					if (!P1) {
						goto next_line;
					}
					for (P2 = P1; *P2 == ' '; ++P2);
					if (*P2 != ',') {
						goto next_line;
					}
					for (++P2; *P2 == ' '; ++P2);
					if (!strncmp(P2, "@function", 9)) {
						ventries[0].nodeP = 0;
						ventries[1].nodeP = 0;
						ventries[2].nodeP = 0;
						ventries[3].nodeP = 0;
						c   = *P1;
						*P1 = 0;
						entityP = functionP = Cfunction::locate(P, 0, sourceP);
						*P1 = c;
						goto next_line;
					}
					if (!strncmp(P2, "@object", 7)) {
						if (g_show_variables != No_variables) {
							c   = *P1;
							*P1 = 0;
							if (!strncmp(P, "_ZTV", 4)) {
								/* This is a vtable
									Format:
										.quad offset	
										.quad _ZTI
										.quad vfunction
										.quad vfunction
						
									The offset is normally 0 but gives the
									value to add to a pointer to a supertype
									to recover a pointer to this type.
									Used when employing multiple inheritance.
								 */
								in_vtable = 0;
							} else if (!strncmp(P, "_ZTC", 4)) {
								/* This is a construction vtable
									Format:
										.quad ??
										.quad offset	
										.quad _ZTI
										.quad vfunction
										.quad vfunction
						
									The offset is normally 0 but gives the
									value to add to a pointer to a supertype
									to recover a pointer to this type.
									Used when employing multiple inheritance.
								 */
								in_vtable = 0;
							}
							flags |= InitX;
							entityP = Cvariable::locate(P, 0, sourceP, functionP);
							*P1 = c;
						}
						goto next_line;
					}
			}	}
			goto next_line;
		}

		// This is a normal machine instruction

		instructionP = P;
		c = *P;
		if (c < 'a') {
			c += 'a' - 'A';
		}
		c2 = P[1];
		if (c2 < 'a') {
			c2 += 'a' - 'A';
		}
		c3 = P[2];
		if (c3 < 'a') {
			c3 += 'a' - 'A';
		}
		update1  = 0;		// By default first argument is read
		update2  = UpdateX;	// By default second argument is updated
		switch (c) {
		case 'a':
			if (c2 == 'a') {						// Ascii adjust aa[a|d|m|s]
				ventries[0].nodeP = 0;	// Changes the A register
			}
			break;
		case 'b':
			switch (c2) {
			case 's':						// bswap
				update1 = 1;
				break;	
			case 't':						// bt
				if (c3 == 'c' ||			// btc	(compliments)
				    c3 == 'r' ||			// btr  (reset)
					c3 == 's') {			// bts  (set)
					break;
				}
				update2 = 0;
				break;
			case 'o':
				if (c3 == 'u') {			// bound
					update2 = 0;
				}
				break;
			}
			break;
		case 'c':
			switch (c2) {
			case 'a':
				if (c3 != 'l' || P[3] != 'l' || P[4] != ' ') {
					// not a call
					break;
				}
				if (g_function_calls == No_functions) {
					goto skip_call;
				}
				for (P += 5; *P == ' '; ++P);
				nameP = 0;
				if (g_function_calls > Direct_functions &&
					P[0] == '*' &&
					P[1] == '%' &&
					(P[2] == 'r' || P[2] == 'e') &&
					P[4] == 'x') {
					reg1 = P[3] - 'a';
					if (reg1 >= 0 && reg1 < 4) {
						// eg: 	movq    -24(%rbp), %rax 
						//		movq	(%rax),	%rax
						// 		addq    $8,	%rax     (Optional)
    					//		movq    (%rax), %rdx
						//		...
						//		call	*%rdx
						//     		    ^
						ventry1P = ventries + reg1;
						if (ventry1P->nodeP && ventry1P->is_ptr == -1) {
							nameP = g_dwarf.resolve_vfunction(ventry1P);
							if (nameP) {
								P1    = 0;
								goto have_function_name;
				}	}	}	}
				P1  = P;
				if (*P1 == '*') {
					++P1;
				}
				if (*P1 == '%') {
					++P1;
				}
				P1  = end_symbol_name(P1);
				if (!P1) {
					goto skip_call;
				}
				c   = *P1;
				*P1 = 0;
				nameP = P;
have_function_name:
				callingP = Cfunction::locate(nameP, FindX, sourceP);
				if (P1) {
					*P1 = c;
				}
				if (functionP) {
					edgeP = Cedge::locate(functionP, callingP, lno, fileP);
					edgeP->m_freq++;
					if (nameP != P) {
						edgeP->m_flags |= VirtualX;
					}
				}
skip_call:	 	ventries[0].nodeP = 0;
				ventries[1].nodeP = 0;
				ventries[2].nodeP = 0;
				ventries[3].nodeP = 0;
				goto next_line;
			case 'b':						// cbw
				ventries[0].nodeP = 0;
				break;
			case 'd':						// cdq
			case 'w':						// cwd
				ventries[0].nodeP = 0;
				ventries[3].nodeP = 0;
				break;
			case 'm':
				if (c3 == 'p') {	// cmp
					if (P[3] != 'x' && P[3] != 'X') {	// cmpx
						update2 = 0;
				}	}
				break;
			}
			break;
		case 'd':
			switch (c2) {
			case 'a':						// da[a|s]
				ventries[0].nodeP = 0;
				break;
			case 'e':
				if (c3 == 'c') {			// dec
					update1 = UpdateX;
				}
				break;
			case 'i':						// div
				ventries[0].nodeP = 0;
				ventries[3].nodeP = 0;
				break;
			}
			break;
		case 'e':
			if (c2 == 'n' && c3 == 't') {	// enter
				update2 = 0;
			}
			break;
		case 'f':
			switch (c2) {
			case 's':
				if (c3 == 't') {			// fst[p]
					update1 = 1;
				}
				break;
			case 'b':
			case 'i': 
				if (c3 == 's')  {			// fist[p] fbst
					update1 = 1;
				}
				break;
			}
			break;
		case 'i':
			switch (c2) {
			case 'd':						// idiv
			case 'm':						// imul
				ventries[0].nodeP = 0;
				ventries[3].nodeP = 0;
				break;
			case 'n':						// inc or in or ins[b|w|d]
				update1 = UpdateX;
				break;
			}
			break;
		case 'j':							// jmp etc.
			goto next_line;
		case 'l':
			switch (c2) {
			case 'a':						// lahf
				ventries[0].nodeP = 0;
				break;
			case 'o':
				if (c3 == 'o') {			// loop
					ventries[2].nodeP = 0;	// changes CX
					goto next_line;
				}
				ventries[0].nodeP = 0;		// lods[b|w|d]
				break;
			}
			break;
		case 'm':
			if (c2 == 'u') {				// mul
				ventries[0].nodeP = 0;
				ventries[3].nodeP = 0;
				break;
			}
			break;
		case 'n':
			switch (c2) {
			case 'e':
				if (c3 == 'g') {			// neg
					update1 = UpdateX;
				}
				break;
			case 'o':
				if (c3 == 't') {			// not
					update1 = UpdateX;
				}
				break;
			}
			break;
		case 'o':
			if (c2 == 'u' && c3 == 't') { 	// out
				goto next_line;
			}
			break;
		case 'p':
			if (c2 == 'o' && c3 == 'p') {	// pop
				update1 = UpdateX;
				if (P[3] == 'a') {
					ventries[0].nodeP = 0;
					ventries[1].nodeP = 0;
					ventries[2].nodeP = 0;
					ventries[3].nodeP = 0;
				}
			}
			break;
		case 'r':
			switch (c2) {
			case 'e':
				if (c3 == 'p') {			// rep[pe|pz|pne|nz]?
					ventries[2].nodeP = 0;
				}
				break;
			case 'c':						// ro[lr] | rc[lr]
			case 'o':
				update1 = UpdateX;
				update2 = 0;
				break;
			}
			break;
		case 's':
			switch (c2) {
			case 'e':
				if (c3 == 't') {			// set...
					update1 = UpdateX;
				}
				break;
			case 'a':						// sa[lr] | sh[lr][d]
			case 'h':
				update1 = UpdateX;
				update2 = 0;
				break;
			}
			break;
		case 't':
			if (c2 == 'e' && c3 == 's') {	// test
				update2 = 0;
			}
			break;
		case 'x':
			switch (c2) {
			case 'c':
				if (c3 == 'h') {			// xchg
					update1 = UpdateX;
				}
				break;
			case 'l':						// xlat
				ventries[0].nodeP = 0;
				break;
			}
			break;
		}

		// Move past instruction
		for (; (c = *P) && isalnum(c); ++P);
		if (c != ' ') {
			goto next_line;
		}
		for (++P; *P == ' '; ++P);

		firstP    = P;
		typeP     = 0;
		flags     = FindX;
		switch (*P) {
		case '$':
			if (!g_addresses) {
				goto skip_first_variable;
			}
			flags    |= AddressesX;
			update1   = UpdateX;
			++P;
			break;
		case '%':
			if (update1) {
				if ((P[1] == 'r' || P[1] == 'e') &&
					P[3] == 'x') {
					reg1 = P[2] - 'a';
					if (0 <= reg1 && reg1 < 4) {
						ventries[reg1].nodeP = 0;
			}	}	}
			break;
		default:
			if (functionP && g_use_dwarf) {
				typeP = g_dwarf.resolve(functionP, lines_seen, P, ventries, &type);
				if (typeP && typeP->type == LOCATION_member) {
					goto skip_first_variable;
				}
			}
			break;
		}


		if (!(flags & AddressesX) && g_show_variables == No_variables) {
			goto skip_first_variable;
		}
		P1 = end_symbol_name(P);
		if (typeP) {
			flags |= NestedX | DynamicX | NotParamX;
			flags &= ~FindX;
			if (typeP->type == LOCATION_parameter) {
				flags &= ~NotParamX;
			}
			nameP  = typeP->nameP;
			if (!nameP) {
				if (!P1) {
					goto skip_first_variable;
				}
				nameP = P;
			}
		} else {
			if (!P1) {
				goto skip_first_variable;
			}
			nameP = P;
		}
		if (P1) {
			c   = *P1;
			*P1 = 0;
		}
		// Variable anywhere
		// Flags [Addresses] ([FindX] | [NestedX|DynamicX] [Parameter])
		usesP = Cvariable::locate(nameP, flags, sourceP, functionP);
		if (functionP) {
			edgeP = Cedge::locate(functionP, usesP, lno, fileP);
			edgeP->m_freq++;
			edgeP->m_flags |= update1;
			usesP->m_flags |= UsedX;
		}
		if (P1) {
			*P1 = c;
			P   = P1;
		}
skip_first_variable:
		// Search for ','
		bracket = 0;
		for (;; ++P) {
			switch (*P) {
			case ',':
				if (bracket) {
					// Ignore comma in brackets
					continue;
				}
			case '\n':
			case 0:
				break;
			case '(':
				++bracket;
				continue;
			case ')':
				--bracket;
			default:
				continue;
			}
			break;
		}

		if (*P != ',') {
			goto next_line;
		}
		for (; (c = *++P) && c == ' '; );

		secondP = P;

		if (g_use_dwarf &&
            P[0] == '%' &&
			(P[1] == 'r' || P[1] == 'e') &&
			P[3] == 'x') {
			reg2 = P[2] - 'a';
			if (0 <= reg2 && reg2 < 4) {
				ventry2P = ventries + reg2;
				switch (instructionP[0]) {
				case 'a':
					if (instructionP[1] != 'd' ||
						instructionP[2] != 'd' ||
					   (instructionP[3] != 'q' && instructionP[3] != 'l')||
						instructionP[4] != ' ') {
						break;
					}
					if (firstP[0] != '$') {
						update2 = 1;
						break;
					}
   					// eg: 	movq    -24(%rbp), %rax 
					//		movq	(%rax),	%rax
					//		addq    $8,	%rax
					if (ventry2P->nodeP) {
						P1 = satou(firstP+1, &offset);
						if (!P1) {
							// Not sure what happened here
							update2 = 1;
							break;
						}
						ventry2P->offset += offset;
						update2           = 0;
					}
					break;
				case 'm':
					if (!g_use_dwarf) {
						break;
					}
					if (instructionP[1] != 'o' ||
						instructionP[2] != 'v' ||
					   (instructionP[3] != 'q' && instructionP[3] != 'l') ||
						instructionP[4] != ' ') {
						// not mov[q|l]
						break;
					}
					update2 = 1;
					if (typeP) {
						if (typeP->is_ptr) {
							// eg: 	movq    -24(%rbp), %rax 
							*ventry2P  = *typeP;
							update2    = 0;
						}
						break;
					}
					number = strtol(firstP, &P1, 10);
					if (P1 == firstP) {
						number = 0;
					} else {
						firstP = P1;
					}
					if (firstP[0] != '(' ||
						firstP[1] != '%' ||
					   (firstP[2] != 'r' && firstP[2] != 'e') ||
						firstP[4] != 'x' ||
						firstP[5] != ')' ) {
						ventry2P->nodeP = 0;
						break;
					}
					reg1 = firstP[3] - 'a';
					if (reg1 < 0 || 4 <= reg1) {
						ventry2P->nodeP = 0;
						break;
					}
					ventry1P = ventries + reg1;
					g_dwarf.dereference(number, ventry1P, ventry2P);
					update2 = 0;
					break;
				}
				if (update2) {
					ventry2P->nodeP = 0;
		}	}	}

		typeP = 0;
		if (c != '$') {
			flags = FindX;
			if (functionP && g_use_dwarf) {
				typeP = g_dwarf.resolve(functionP, lines_seen, P, ventries, &type);
				if (typeP && typeP->offset) {
					goto next_line;
			}	}
		} else {
			if (!g_addresses) {
				goto next_line;
			}
			flags = AddressesX | FindX;
			update2 = UpdateX;
			++P;
		}

		if (!(flags & AddressesX) && g_show_variables == No_variables) {
			goto next_line;
		}
		P1 = end_symbol_name(P);
		if (typeP) {
			flags |= NestedX|DynamicX|NotParamX;
			flags &= ~FindX;
			if (typeP->type == LOCATION_parameter) {
				flags &= ~NotParamX;
			}
			nameP  = typeP->nameP;
			if (!nameP) {
				if (!P1) {
					goto next_line;
				}
				nameP = P;
			}
		} else {
			if (!P1) {
				goto next_line;
			}
			nameP = P;
		}
		if (P1) {
			c   = *P1;
			*P1 = 0;
		}
		// Flags [AddressesX] (FindX | NestedX | DynamicX [NotParamX])
		usesP = Cvariable::locate(nameP, flags, sourceP, functionP);
		if (functionP) {
			edgeP = Cedge::locate(functionP, usesP, lno, fileP);
			edgeP->m_freq++;
			edgeP->m_flags |= update2;
			usesP->m_flags |= UsedX;
		}
		if (P1) {
			*P1 = c;
			P   = P1;
		}
next_line:
#ifdef TRACE
		if (state != 4) {
			fprintf(stderr, "%4d %d|%d|%d|%d ", 
				lines_seen,
				ventries[0].state,
				ventries[1].state,
				ventries[2].state,
				ventries[3].state);
			for (P1 = g_startP; ; ++P1) {
				fputc(*P1, stderr);
				if (*P1 == '\n') {
					break;
			}	}
		}
#endif
		state = 4;
		--P;
	}
	return;
}

#ifndef SSIZE_MAX
#define SSIZE_MAX 0x7FFFFFFF
#endif

static void
read_asm(int fileno, Csource *sourceP)
{
	struct stat buf;
	char		*P, *endP;
	int			lth, segment;
	int			ret;

	// We have to scan the input more than once so read the entire input
	// into memory

	ret = fstat(fileno, &buf);
	if (ret != 0) {
		fprintf(stderr,"** Line %d: fstat %s returned %d\n", lineno, sourceP->m_nameP, ret);
		perror("");
		return;
	}

	// Allocate a large enough memory buffer

	lth = buf.st_size;
	if (!lth) {
		fprintf(stderr, "** Line %d: %s is empty\n", lineno, sourceP->m_nameP);
		return;
	}

	if (lth >= g_file_memory_size) {
		Xfree(g_file_memoryP);
		g_file_memory_size = ((lth + 2) + 2047) & ~2047;
		g_file_memoryP = (char *) Xmalloc(g_file_memory_size);
	}

	P = g_file_memoryP;
	for (endP = P + lth; P < endP; P += ret) {
		segment = endP - P;
		if (segment > SSIZE_MAX) {
			segment = SSIZE_MAX;
		}
		ret = read(fileno, P, segment);
		if (ret <= 0) {
			fprintf(stderr, "** Line %d: read %s returned %d\n", lineno, sourceP->m_nameP, ret);
			perror("");
			return;
	}	}
	assert(P == endP);
	if (P[-1] != '\n') {
		*endP++ = '\n';
	}
	*endP = 0;
		
	remove_comments();
	read_asm_pass1(sourceP);
	read_asm_pass2(sourceP);
}


static int
get_environment(char **argv)
{
	int		i, c, c1, c2, suffix;
	char	*P, *P1, *P2;
	const char *suffixP;
	int		args;

	for (i = 'A'; i <= 'Z'; ++i) {
		label_char[i]    = 1;
		function_char[i] = 1;
	}
	for (i = 'a'; i <= 'z'; ++i) {
		label_char[i]    = 2;
		function_char[i] = 2;
	}
	for (i = '0'; i <= '9'; ++i) {
		label_char[i]    = 3;
		function_char[i] = 3;
	}
	label_char[(int) '_']    = 4;
	function_char[(int) '_'] = 4;
	label_char[(int) '.']    = 5;
	function_char[(int) '.'] = 5;
	function_char[(int) '$'] = 6;

	for (i = args = 1; (P = argv[i]); ++i) {
		if (P[0] == '-' && P[1] == '-') {
			if (!strncmp(P, "--ASX_COMPILE=", 14)) {
				asx_compileP = P + 14;
				continue;
			}
			if (!strncmp(P, "--ASX_UNLINK=", 13)) {
				asx_unlinkP = P + 13;
				continue;
			}
			if (!strncmp(P, "--ASX_IGNORE=", 13)) {
				asx_ignoreP = P + 13;
				continue;
			}
			if (!strncmp(P, "--ASX_FORCE=", 12)) {
				asx_forceP = P + 12;
				continue;
			}
			if (!strncmp(P, "--ASX_SILENT=", 13)) {
				asx_silentP = P + 13;
				continue;
			}
			if (!strncmp(P, "--ASX_SKIP=", 11)) {
				asx_skipP = P + 11;
				continue;
			}
			if (!strncmp(P, "--ASX_DWARF", 11)) {
				g_debug_dwarf = 1;
				continue;
			}
			if (!strncmp(P, "--ASX_LIFT=", 11)) {
				asx_liftP = P + 11;
				continue;
		}	}
		argv[args++] = P;
	}
	argv[args] = 0;

	if (!asx_compileP) {
		asx_compileP = getenv("ASX_COMPILE");
	}
	if (!asx_silentP) {
		asx_silentP = getenv("ASX_SILENT");
	}
	if (!asx_ignoreP) {
		asx_ignoreP = getenv("ASX_IGNORE");
		if (!asx_ignoreP) {
			asx_ignoreP = dupstring("a;o;so");
	}	}

	if (!asx_skipP) {
		asx_skipP = getenv("ASX_SKIP");
	}
    if (!asx_liftP) {
		asx_liftP = getenv("ASX_LIFT");
    }

	if (!g_debug_dwarf) {	
		if (getenv("ASX_DWARF")) {
			g_debug_dwarf = 1;
	}	}

	if (asx_liftP && strchr(asx_liftP, 'a')) {
		g_lift_addresses = 1;
	}

	if (asx_compileP) {
		printf("// ASX_COMPILE=%s// Will not compile source code", asx_compileP);
		if (!strcasecmp(asx_compileP, "no")) {
			printf(" (Extract by directly examining source code)\n");
			compile_flag = 0;
			unlink_flag  = 0;
		} else {
			printf(" (Extract by compiling source code to assembler)\n");
	}	}
		
	if (asx_silentP) {
		printf("// ASX_SILENT=%s// Will not echo commands", asx_silentP);
		if (!strcasecmp(asx_silentP, "no")) {
			printf(" (Echoing compiler commands)\n");
			silent_flag = 0;
		} else {
			printf(" (Compiler commands not shown)\n");
			silent_flag = 1;
	}	}
		
	if (compile_flag) {
		if (!asx_unlinkP) {
			asx_unlinkP = getenv("ASX_UNLINK");
		}
		if (asx_unlinkP) {
			printf("// ASX_UNLINK=%s", asx_unlinkP);
			if (!strcasecmp(asx_unlinkP, "no")) {
				unlink_flag = 0;
				printf(" (Generated *.s files preserved)\n");
			} else {
				printf(" (Generated *.s files removed)\n");
		}	}
		if (asx_skipP) {
			printf("// ASX_SKIP=%s", asx_skipP);
			if (!strcasecmp(asx_skipP, "yes")) {
				skip_flag = 1;
				printf(" (Pre-existing *.s files used)\n");
			} else {
				printf(" (Pre-existing *.s files ignored)\n");
	}	}	}

	if (asx_ignoreP) {
		printf("// ASX_IGNORE=%s\n", asx_ignoreP);

		for (P = asx_ignoreP; P; P = (c ? P1+1 : 0)) {
			P1 = P;
			c  = *P;
			// Remove leading space
			if (isspace(c)) {
				continue;
			}
			for (; (c = *P1) && c != ';' && c != ':' && c != ','; ++P1);
			*P1 = 0;
			// Remove trailing space
			for (P2 = P1; --P2 >= P; ) {
				if (!isspace(*P2)) {
					break;
				}
				*P2 = 0;
			}
			c1 = *P;
			if ('a' <= c1) {
				c1 += 'A' - 'a';
			}
			switch (c1) {
			case 'R':
				g_addresses = 0;
				break;
			case 'F':
				c2 = P[1];
				if ('a' <= c2) {
					c2 += 'A' - 'a';
				}
				switch (c2) {
				case 'V':
					g_function_calls = Direct_functions;
					break;
				default:
					g_function_calls = No_functions;
				}
				continue;
			case 'V':
				c2 = P[1];
				if ('a' <= c2) {
					c2 += 'A' - 'a';
				}
				switch (c2) {
				case 'S':
					g_show_variables = Global_variables;
					continue;
				case 'F':
					g_show_variables = Static_variables;
					continue;
				case 'P':
					g_show_variables = Static_function_variables;
					continue;
				case 'L':
					g_show_variables = Function_parameters;
					continue;
				case 'A':
				case 'G':
				case 0:
					g_show_variables = No_variables;
					continue;
				case 'N':
					g_show_variables = All_variables;
					continue;
				}
				continue;
			case 0:
				// Ignore things that only contained whitespace
				continue;
			}

			for (suffix = 0; ; ++suffix) {
				suffixP = suffixes[suffix];
				if (!suffixP) {
					fprintf(stderr, "** Can't ignore suffix \"%s\" - unknown\n", P);
					break;
				}
				if (!strcmp(P, suffixP)) {
					ignore[suffix] = 1;
					break;
			}	}
	}	}

	if (asx_forceP) {
		printf("// ASX_FORCE=%s\n", asx_forceP);
		for (P = asx_forceP; P; P = (c ? P1+1 : 0)) {
			P1 = P;
			c  = *P;
			// Remove leading space
			if (isspace(c)) {
				continue;
			}
			for (; (c = *P1) && c != ';' && c != ':' && c != ','; ++P1);
			*P1 = 0;
			// Remove trailing space
			for (P2 = P1; --P2 >= P; ) {
				if (!isspace(*P2)) {
					break;
				}
				*P2 = 0;
			}
			c1 = *P;
			if ('a' <= c1) {
				c1 += 'A' -'a';
			}
			switch (c1) {
			case 'R':
				g_addresses = 1;
				break;
			case 'F':
				c2 = P[1];
				if ('a' <= c2) {
					c2 += 'A' - 'a';
				}
				switch (c2) {
				case 'V':
					g_function_calls = All_functions;
					break;
				default:
					g_function_calls = Direct_functions;
				}
				continue;
			case 'V':
				c2 = P[1];
				if ('a' <= c2) {
					c2 += 'A' - 'a';
				}
				switch(c2) {
				case 'G':
					g_show_variables = Global_variables;
					continue;
				case 'S':
					g_show_variables = Static_variables;
					continue;
				case 'F':
					g_show_variables = Static_function_variables;
					continue;
				case 'P':
					g_show_variables = Function_parameters;
					continue;
				case 'L':
				case 'A':
				case 0:
					g_show_variables = All_variables;
					continue;
				case 'N':
					g_show_variables = No_variables;
					continue;
				}
				continue;
			case 0:
				// Ignore things that only contained whitespace
				continue;
			}

			for (suffix = 0; ; ++suffix) {
				suffixP = suffixes[suffix];
				if (!suffixP) {
					fprintf(stderr, "** Can't force suffix \"%s\" - unknown\n", P);
					break;
				}
				if (!strcmp(P, suffixP)) {
					force[suffix] = 1;
					break;
	}	}	}	}
	if (g_function_calls == Functions_not_set) {
		g_function_calls = All_functions;
	}
	if (g_show_variables == Variables_not_set) {
		g_show_variables = All_variables;
	}
	g_use_dwarf = g_function_calls > Direct_functions || g_show_variables > Static_function_variables;
	if (!g_use_dwarf) {
		printf("// Not using Dwarf\n");
	}
	return(args);
}

static int
parse_command(char *commandP, char **argv1P)
{
	char	*startP, *endP, *nextP;
	int		i, c;

	startP = commandP;
	if (*startP != '"') {
		fprintf(stderr, "** Line %d: Missing cwd start '\"' >%s\n", lineno, startP);
		return(-1);
	}

	for (i = -1; startP; ++i) {
		for (endP = nextP = ++startP; (c = *nextP) != '"'; ++nextP) {
			switch (c) {
			case 0:
				fprintf(stderr, "** Line %d: Missing end '\" >%s'\n", lineno, startP);
				return(-1);
			case '\\':
				c = *(++nextP);
			default:
				*endP++ = c;
		}	}
		*endP = 0;
		if (i < 0) {
			cwdP = startP;
			if (chdir(cwdP)) {
				fprintf(stderr,"** Line %d: Can't change directory to %s\n", lineno, cwdP);
				perror("");
				return(-1);
			}
		} else {
			argv1P[i] = startP;
		}
		startP = strchr(nextP+1, '"');
	}
	if (i < 1) {
		fprintf(stderr, "** Line %d: Missing cmd '\" >%s'\n", lineno, commandP);
		return(-1);
	}
	argv1P[i]   = 0;
	return(i);
}

static int
process_command(int argc2, char **argv1P, char * const env[])
{
	static char	filename1[1024];
	static char	*compilePP[2048];

	struct stat	source_stat;
	struct stat assembler_stat;

	int			compile;

	char	*programP, *P;
	char 	*compileP, *targetP, *assemblerP, *absolute_pathP;
	char	*P1;
	const char *suffixP;
	int		i, suffix, unlink_me, lth, ret, fno, argc1;
	int		seen_dash_S, seen_dash_E, running_as, skip_compile;
	int		c2;
	Csource	*sourceP;
	char	**argv2PP;

	/* Find the source and target files */

	seen_dash_S = 0;
	seen_dash_E = 0;
	targetP     = 0;
	assemblerP  = 0;
	compile     = 0;
	running_as  = 0;
	skip_compile= skip_flag;
	programP    = argv1P[0];
	lth         = strlen(programP);
	if (lth > 3) {
		if (!strcmp(programP+lth-3, "/as")) {
			running_as = 1;
		} else if (lth > 4 && !strcmp(programP+lth-4, "/gas")) {
			running_as = 1;
	}	}
	
	if (running_as) {
		for (argc1 = i = 1; (P = argv1P[argc1]); ++argc1) {
			if (argc1 >= argc2) {
				fprintf(stderr, "** lineno %d: Can't scan args for %s\n", lineno, programP);
				return(-1);
			}
			if (*P != '-') {	
				if (!strcmp(P, "/dev/null")) {
					return(0);
				}
				compilePP[compile++] = P;
				continue;		// Discard (to add back in later)
			}
			switch(P[1]) {
			case '-':
				if (!strcmp(P,"--defsym")) {
					++argc1;	// Discard two arguments
				}
				continue;
			case 'I':
			case 'o':
				++argc1;
			case 'a':
			case 'D':
			case 'f':
			case 'g':
			case 'J':
			case 'K':
			case 'L':
			case 'n':
			case 'R':
			case 'v':
			case 'W':
			case 'w':
			case 'x':
			case 'Z':
				continue;
			}
			fprintf(stderr, "** Line %d: Unknown %s option '%s'\n", lineno, programP, P);
			goto bad;
		}
	} else {
		for (argc1 = i = 1; (P = argv1P[argc1]); ++argc1) {
			if (argc1 >= argc2) {
				fprintf(stderr, "** lineno %d: Can't scan args for %s\n", lineno, programP);
				return(-1);
			}
			if (*P != '-') {	
				if (!strcmp(P, "/dev/null")) {
					return(0);
				}
				compilePP[compile++] = P;
				continue;		// Discard (to add back in later)
			}
			c2 = P[2];
			switch(P[1]) {
			case 'a':
				if (!strcmp(P, "-aux-info")) { 		// Discard 2 args
					++argc1;
					continue;
				}
				argv1P[i++] = P;
				continue;
			case 'f':
				if (!strcmp(P, "-fpic") || !strcmp(P, "-fPIC")) {
					continue;
				}
			case 'B':
			case 'n':
			case 't':
			case 'Y':
				argv1P[i++] = P;
				continue;
			case 'D':
			case 'I':
			case 'U':
				if (!c2) {
					argv1P[i++] = P;
					P           = argv1P[++argc1];
				}
				argv1P[i++] = P;
				continue;
			case 'M':
				switch (c2) {
				case 'F':
				case 'T':
					++argc1;	// Discard 2 arguments
				}
				continue;
			case 'b':
				if (!c2) {
					++argc1;						// Discard 2 args
					continue;
				}
				argv1P[i++] = P;
			case 'c':
			case 'C':
			case 'g':
			case 'h':
			case '-':
			case 'O':
			case 'P':
			case 'Q':
			case 'r':
			case 'v':
			case 'w':
			case 'l':
			case 'L':
				continue;							// Discard
			case 'd':
				if (c2 == 'D' || c2 == 'I' || c2 == 'M' || c2 == 'N') {
					argv1P[i++] = P;
				}
				continue;
			case 'E':
				if (!c2) {
					seen_dash_E = 1;
					continue;
				}
				argv1P[i++] = P;
				continue;
			case 'G':
			case 'm':
				if (!c2) {
					++argc1;
				} else if (!strcmp(P, "-m32") || !strcmp(P, "-m64")) {
					argv1P[i++] = P;
				}
				continue;
			case 'i':
				if (!strcmp(P, "-idirafter")     ||
					!strcmp(P, "-imacros")       ||
					!strcmp(P, "-include")       ||
					!strcmp(P, "-iprefix")       ||
					!strcmp(P, "-iwithprefix")   ||
					!strcmp(P, "-isystem")) {
					argv1P[i++] = P;
					P           = argv1P[++argc1];
				}
				argv1P[i++] = P;
				continue;
			case 'o':	// <file>
				if (!c2) {
					targetP = argv1P[++argc1];	// Discard parameter
					continue;
				}
				break;
			case 'p':
				if (strcmp(P, "-print")) {		// Not -print
					argv1P[i++] = P;
				} else {
					if (!strcmp(P, "-print-file-name")) {
						return(0);
				}	}
				continue;
			case 'S':
				if (!c2) {
					seen_dash_S = 1;
					continue;	
				}
				break;
			case 's':
				if (!strcmp(P, "-std")) {
					argv1P[i++] = P;
				}
				continue;
			case 'u':
			case 'V':
				if (!c2) {
					argv1P[i++] = P;
					P           = argv1P[++argc1];
				}
				argv1P[i++] = P;
				continue;
			case 'W':
				if (!strcmp(P, "-Wp,")) {
					if (!strcmp(P, "-Wp,-M")) {
						argv1P[i++] = P;
				}	}
				continue;
			case 'x':
				if (c2) {
					argv1P[i++] = P;
					continue;
				}
			case 'X':
				argv1P[i++] = P;
				argv1P[i++] = argv1P[++argc1];
				continue;
			case '#':
				if (!strcmp(P, "-###")) {
					continue;
				}
				break;
			}
			/* If the compiler was not compiling we shouldn't */
			fprintf(stderr, "** Line %d: Unknown compiler option '%s'\n", lineno, P);
			goto bad;
		}
	}

	argc1 = i;

	compilePP[compile] = 0;

	if (seen_dash_S) {
		assemblerP = targetP;
	}

	for (compile = 0; (compileP = compilePP[compile]); ++compile) {
		P1 = strrchr(compileP, '.');
		if (!P1) {
			suffix = SUFFIX_none;
		} else {
			++P1;
			for (suffix= 0;; ++suffix) {
			
				if (!(suffixP = suffixes[suffix])) {
					fprintf(stderr, "** Line %d: Unknown suffix %s\n", lineno, compileP);
					goto next_command;
				}
				if (!strcmp(P1, suffixP)) {
					break;
		}	}	}
	
		if (ignore[suffix]) {
			goto next_command;
		}

		unlink_me  = 0;
		if (suffix == SUFFIX_S && running_as) {
			suffix = SUFFIX_s;
		}
		switch (suffix) {
		case SUFFIX_so:
		case SUFFIX_o:
			if (use_stdin && !force[suffix]) {
				int		s;
				const char	*suffixP;
				
				lth = P1 - compileP;
				strncpy(filename1, compileP, lth);
	
				if (suffix == SUFFIX_so) {
					s = SUFFIX_o;
				} else {
					s = SUFFIX_s;
				}
				for (; s < SUFFIX_none; ++s) {
					suffixP = suffixes[s];
					if (!suffixP) {
						continue;
					}
					strcpy(filename1+lth, suffixP);
					absolute_pathP = absolute_path(filename1);
					if (Csource::lookup(absolute_pathP)) {
						fprintf(stderr, "-- Line %d: Assuming %s generated from %s\n", lineno, compileP, absolute_pathP);
						goto next_command;
			}	}	}
		case SUFFIX_none:
		case SUFFIX_a:
			read_object(compileP, suffix);
			break;
		case SUFFIX_cxx:
		case SUFFIX_cpp:
		case SUFFIX_cpp1:
			if (!use_stdin) {
				argv1P[0] = "/usr/bin/g++";
			}
		case SUFFIX_c:
		case SUFFIX_C:
		case SUFFIX_cc:
			if (seen_dash_E) {
				goto next_command;
			}
#ifdef SHOW_FILES
			fprintf(stderr,"%s/%s\n", cwdP, compileP);
#endif
		case SUFFIX_S:
			if (!assemblerP) {
				lth = P1-compileP;
				memcpy(filename1, compileP, lth);
				strcpy(filename1+lth, "s");
				assemblerP = filename1;
			}
			if (!compile_flag) {
				goto read_assembler;
			}
			
			argv1P[argc1++] = compileP;
			argv1P[argc1]   = 0;
			argv1P         -= 4;
			argv1P[0]       = argv1P[4];
			argv1P[1]       = "-o";
			argv1P[2]       = assemblerP;
			argv1P[3]       = "-g";
			if (suffix == SUFFIX_S) {
				argv1P[4]   = "-E";
			} else {
				argv1P[4]   = "-S";
			}
			if (unlink_flag || skip_compile) {
				if (access(assemblerP, F_OK)) {
					// Looks like file does not exist
					unlink_me    = unlink_flag;
					skip_compile = 0;
				} else if (skip_compile) {
					if (!access(compileP, F_OK)) {
						// Looks like the source exists
						if (stat(compileP, &source_stat) ||
							stat(assemblerP, &assembler_stat) ||
							source_stat.st_mtime >= assembler_stat.st_mtime) {
							skip_compile = 0;
			}	}	}	}
	
			if (!skip_compile) {
				ret = call_cc(argv1P, env, suffix);
				if (ret) {
					fprintf(stderr, "** Line %d: Failed to compile %s\n", lineno, assemblerP);
					argv1P[argc1-1] = absolute_path(assemblerP);
					read_cx(assemblerP, suffix, argv1P);
					unlink_me = 0;
					goto next_command;
			}	}
			goto read_assembler;
	
		case SUFFIX_s:
			assemblerP     = compileP;
read_assembler:
			absolute_pathP = absolute_path(compileP);
			sourceP        = new Csource(absolute_pathP, 0);
			if (!sourceP) {
				outofmemory();
				goto bad;
			}

			for (argc1 = 0; argv1P[argc1]; ++argc1);
			sourceP->m_argsPP = argv2PP = (char **) Xmalloc(sizeof(char *) * argc1 + 1);
			*argv2PP++ = dupstring(cwdP);
			for (i = 0; i < argc1; ++i) {
				argv2PP[i] = dupstring(argv1P[i]);
			}
			argv2PP[argc1] = 0;

			fno = open(assemblerP, O_RDONLY);
			if (fno < 0) {
				fprintf(stderr, "Line %d: Can't open %s\n", lineno, assemblerP);
				perror("");
				goto next_command;
			}
			read_asm(fno, sourceP);
			close(fno);
	
			if (!unlink_me) {
				break;
			}
	
			unlink_me = 0;
	
			/* Sanity checks follow */
			if (assemblerP != filename1) {
				fprintf(stderr, "Line %d: May not unlink file %s\n", lineno, assemblerP);
				exit(5);
				goto bad;
			} 
			lth = strlen(assemblerP);
			if (lth < 3 || assemblerP[lth-2] != '.' || assemblerP[lth-1] != 's') {
				fprintf(stderr, "Line %d: May not delete %s\n", lineno, assemblerP);
				exit(5);
				goto bad;
			}
	
			/* Discard this assembly file when done */
	
			if (unlink(assemblerP)) {
				fprintf(stderr, "Line %d: Can't delete %s\n", lineno, assemblerP);
				perror("");
				break;
			}
			break;
		}
next_command:
		continue;
	}
	return(0);
bad:
	return(-1);
}

// Leave room for -g -S -o filename
#define ExtraSlots 4

int
main(int argc, char *argv[], char * const env[])
{
	static char	*argv1[2048];
	
	char		**argv1P;
	int 		i;
	char		*stdin_bufferP;
	size_t		stdin_buffer_lth;
	clock_t		start_clock, end_clock;
	
	start_clock = time(0);

	printf("// ASX " VERSION " Compiled " __DATE__ ":"  __TIME__ " Start: %s\n", ctime(&start_clock));

	argc = get_environment(argv);

	printf("// Arguments to %s:\n//", argv[0]);

	for (i = 1; argv[i]; ++i) {
		fputc(' ', stdout);
		fputs(argv[i], stdout);
	}
	fputc('\n', stdout);
	fflush(stdout);

	g_program_nameP = argv[0];
	g_ret           = 0;

	if (argc < 2 || !strcmp(argv[1], "-")) {
    	use_stdin        = 1;
		stdin_buffer_lth = 120;
		stdin_bufferP    = (char *) malloc(stdin_buffer_lth);
		while (getline(&stdin_bufferP, &stdin_buffer_lth, stdin) >= 0) {
			argv1P = argv1 + ExtraSlots;
			++lineno;
			if (!strncmp(stdin_bufferP, "reset", 5)) {
				Cfunction::reset();
				Cvariable::reset();
				Cclass::reset();
				Ctemplate::reset();
				continue;
			}
			i = parse_command(stdin_bufferP, argv1P);
			if (i < 1) {
				continue;
			}
			process_command(i, argv1P, env);
		}
	} else {
		cwdP      = getcwd(0, 0);
		argv1P    = argv1 + ExtraSlots;
		argv1P[0] = "/usr/bin/gcc";
		for (i = 1; i < argc; ++i) {
			argv1P[i] = argv[i];	
		}
		argv1P[i]   = 0;
		process_command(i, argv1P, env);
	}

	Cvariable::resolve(g_show_variables, g_function_calls);
	Cfunction::resolve();
	collapse_directories();
	dump_ta();
	end_clock = time(0);
	printf("\n// End: %s \n", ctime(&end_clock));
	return(g_ret);
}

