#ifdef WIN32
#define _CRT_SECURE_NO_WARNINGS
#endif

#ifndef WIN32
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "xmalloc.h"
#include "util.h"
#include "object.h"
#include "label.h"
#include "classmember.h"
#include "function.h"
#include "dwarf.h"

#define INDENT "  "
//#define DUMP_STRINGTABLE
//#define DUMP_ABBREVIATIONS
//#define DUMP_HEADER

extern int g_debug_dwarf;

/* Process debug information in Assembler as per Dwalf 2 & 3 specification */

void
trap_dwarf(void)
{
}

static dw_formE	op_args_addr[]  = { DW_FORM_addr,DW_FORM_null};
static dw_formE	op_args_int8[]  = { DW_FORM_data1,DW_FORM_null};
static dw_formE	op_args_int16[] = { DW_FORM_data2,DW_FORM_null};
static dw_formE	op_args_int32[] = { DW_FORM_data4,DW_FORM_null};
static dw_formE	op_args_int64[] = { DW_FORM_data8,DW_FORM_null};
static dw_formE	op_args_udata[] = { DW_FORM_udata,DW_FORM_null};
static dw_formE	op_args_sdata[] = { DW_FORM_sdata,DW_FORM_null};
static dw_formE	op_args_udata_sdata[] = {
	DW_FORM_udata, DW_FORM_sdata, DW_FORM_null};
static dw_formE	op_args_udata_udata[] = {
	DW_FORM_udata,DW_FORM_udata, DW_FORM_null};
static dw_formE	op_args_indirect[] = { DW_FORM_indirect, DW_FORM_null};

typedef struct {
	const char	*nameP;
	dw_formE 	*argsP;
} op_tableT;

static op_tableT op_table[] = {
/* 0x00 */ 	 {0			,0}
/* 0x01 */	,{0			,0}
/* 0x02 */ 	,{0			,0}
/* 0x03 */	,{"addr"	,op_args_addr}	// constant address (size target specific)
/* 0x04 */ 	,{0			,0}
/* 0x05 */ 	,{0			,0}
/* 0x06 */	,{"deref"	,0}
/* 0x07 */ 	,{0			,0}
/* 0x08 */	,{"const1u"	,op_args_int8}	// 1-byte constant
/* 0x09 */	,{"const1s"	,op_args_int8} 	// 1-byte constant
/* 0x0a */	,{"const2u"	,op_args_int16} // 2-byte constant
/* 0x0b */	,{"const2s"	,op_args_int16}	// 2-byte constant
/* 0x0c */	,{"const4u"	,op_args_int32}	// 4-byte constant
/* 0x0d */	,{"const4s"	,op_args_int32}	// 4-byte constant
/* 0x0e */	,{"const8u"	,op_args_int64}	// 8-byte constant
/* 0x0f */	,{"const8s"	,op_args_int64}	// 8-byte constant
/* 0x10 */	,{"constu"	,op_args_udata}	// ULEB128 constant
/* 0x11 */	,{"consts"	,op_args_sdata}	// SLEB128 constant
/* 0x12 */	,{"dup"		,0} 
/* 0x13 */	,{"drop"	,0}
/* 0x14 */	,{"over"	,0}
/* 0x15 */	,{"pick"	,op_args_int8}	// 1-byte stack index
/* 0x16 */	,{"swap"	,0}
/* 0x17 */	,{"rot"		,0}
/* 0x18 */	,{"xderef"	,0}
/* 0x19 */	,{"abs"		,0}
/* 0x1a */	,{"and"		,0}
/* 0x1b */	,{"div"		,0}
/* 0x1c */	,{"minus"	,0}
/* 0x1d */	,{"mod"		,0}
/* 0x1e */	,{"mul"		,0}
/* 0x1f */	,{"neg"		,0}
/* 0x20 */	,{"not"		,0}
/* 0x21 */ ,{"or"		,0}
/* 0x22 */ ,{"plus"		,0}
/* 0x23 */ ,{"plus_uconst"	,op_args_udata}	// ULEB128 addend
/* 0x24 */ ,{"shl"		,0}
/* 0x25 */ ,{"shr"		,0}
/* 0x26 */ ,{"shra"		,0}
/* 0x27 */ ,{"xor"		,0}
/* 0x28 */ ,{"bra"		,op_args_int16}	// signed 2-byte constant
/* 0x29 */ ,{"eq"		,0}
/* 0x2a */ ,{"ge"		,0}
/* 0x2b */ ,{"gt"		,0}
/* 0x2c */ ,{"le"		,0}
/* 0x2d */ ,{"lt"		,0}
/* 0x2e */ ,{"ne"		,0}
/* 0x2f */ ,{"skip"		,op_args_int16}	// signed 2-byte constant
/* 0x30 */ ,{"lit0"		,0}
/* 0x31 */ ,{"lit1"		,0}
/* 0x32 */ ,{"lit2"		,0}
/* 0x33 */ ,{"lit3"		,0}
/* 0x34 */ ,{"lit4"		,0}
/* 0x35 */ ,{"lit5"		,0}
/* 0x36 */ ,{"lit6"		,0}
/* 0x37 */ ,{"lit7"		,0}
/* 0x38 */ ,{"lit8"		,0}
/* 0x39 */ ,{"lit9"		,0}
/* 0x3a */ ,{"lit10"	,0}
/* 0x3b */ ,{"lit11"	,0}
/* 0x3c */ ,{"lit12"	,0}
/* 0x3d */ ,{"lit13"	,0}
/* 0x3e */ ,{"lit14"	,0}
/* 0x3f */ ,{"lit15"	,0}
/* 0x40 */ ,{"lit16"	,0}
/* 0x41 */ ,{"lit17"	,0}
/* 0x42 */ ,{"lit18"	,0}
/* 0x43 */ ,{"lit19"	,0}
/* 0x44 */ ,{"lit20"	,0}
/* 0x45 */ ,{"lit21"	,0}
/* 0x46 */ ,{"lit22"	,0}
/* 0x47 */ ,{"lit23"	,0}
/* 0x48 */ ,{"lit24"	,0}
/* 0x49 */ ,{"lit25"	,0}
/* 0x4a */ ,{"lit26"	,0}
/* 0x4b */ ,{"lit27"	,0}
/* 0x4c */ ,{"lit28"	,0}
/* 0x4d */ ,{"lit29"	,0}
/* 0x4e */ ,{"lit30"	,0}
/* 0x4f */ ,{"lit31"	,0} // literals 0..31 = (DW_OP_lit0 + literal)
/* 0x50 */ ,{"reg0"		,0}
/* 0x51 */ ,{"reg1"		,0}
/* 0x52 */ ,{"reg2"		,0}
/* 0x53 */ ,{"reg3"		,0}
/* 0x54 */ ,{"reg4"		,0}
/* 0x55 */ ,{"reg5"		,0}
/* 0x56 */ ,{"reg6"		,0}
/* 0x57 */ ,{"reg7"		,0}
/* 0x58 */ ,{"reg8"		,0}
/* 0x59 */ ,{"reg9"		,0}
/* 0x5a */ ,{"reg10"	,0}
/* 0x5b */ ,{"reg11"	,0}
/* 0x5c */ ,{"reg12"	,0}
/* 0x5d */ ,{"reg13"	,0}
/* 0x5e */ ,{"reg14"	,0}
/* 0x5f */ ,{"reg15"	,0}
/* 0x60 */ ,{"reg16"	,0}
/* 0x61 */ ,{"reg17"	,0}
/* 0x62 */ ,{"reg18"	,0}
/* 0x63 */ ,{"reg19"	,0}
/* 0x64 */ ,{"reg20"	,0}
/* 0x65 */ ,{"reg21"	,0}
/* 0x66 */ ,{"reg22"	,0}
/* 0x67 */ ,{"reg23"	,0}
/* 0x68 */ ,{"reg24"	,0}
/* 0x69 */ ,{"reg25"	,0}
/* 0x6a */ ,{"reg26"	,0}
/* 0x6b */ ,{"reg27"	,0}
/* 0x6c */ ,{"reg28"	,0}
/* 0x6d */ ,{"reg29"	,0}
/* 0x6e */ ,{"reg30"	,0}
/* 0x6f */ ,{"reg31"	,0} // reg 0..31 = (DW_OP_reg0 + regnum)
/* 0x70 */ ,{"breg0"	,op_args_sdata}
/* 0x71 */ ,{"breg1"	,op_args_sdata}
/* 0x72 */ ,{"breg2"	,op_args_sdata}
/* 0x73 */ ,{"breg3"	,op_args_sdata}
/* 0x74 */ ,{"breg4"	,op_args_sdata}
/* 0x75 */ ,{"breg5"	,op_args_sdata}
/* 0x76 */ ,{"breg6"	,op_args_sdata}
/* 0x77 */ ,{"breg7"	,op_args_sdata}
/* 0x78 */ ,{"breg8"	,op_args_sdata}
/* 0x79 */ ,{"breg9"	,op_args_sdata}
/* 0x7a */ ,{"breg10" 	,op_args_sdata}
/* 0x7b */ ,{"breg11" 	,op_args_sdata}
/* 0x7c */ ,{"breg12" 	,op_args_sdata}
/* 0x7d */ ,{"breg13" 	,op_args_sdata}
/* 0x7e */ ,{"breg14" 	,op_args_sdata}
/* 0x7f */ ,{"breg15" 	,op_args_sdata}
/* 0x80 */ ,{"breg16" 	,op_args_sdata}
/* 0x81 */ ,{"breg17" 	,op_args_sdata}
/* 0x82 */ ,{"breg18" 	,op_args_sdata}
/* 0x83 */ ,{"breg19" 	,op_args_sdata}
/* 0x84 */ ,{"breg20" 	,op_args_sdata}
/* 0x85 */ ,{"breg21" 	,op_args_sdata}
/* 0x86 */ ,{"breg22" 	,op_args_sdata}
/* 0x87 */ ,{"breg23" 	,op_args_sdata}
/* 0x88 */ ,{"breg24" 	,op_args_sdata}
/* 0x89 */ ,{"breg25" 	,op_args_sdata}
/* 0x8a */ ,{"breg26" 	,op_args_sdata}
/* 0x8b */ ,{"breg27" 	,op_args_sdata}
/* 0x8c */ ,{"breg28" 	,op_args_sdata}
/* 0x8d */ ,{"breg29" 	,op_args_sdata}
/* 0x8e */ ,{"breg30" 	,op_args_sdata}
/* 0x8f */ ,{"breg31" 	,op_args_sdata} // SLEB128 offset base register 0..31 = (DW_OP_breg0 + regnum)
/* 0x90 */ ,{"regx"		,op_args_udata} // ULEB128 register
/* 0x91 */ ,{"fbreg"	,op_args_sdata} // SLEB128 offset
/* 0x92 */ ,{"bregx"	,op_args_udata_sdata} // ULEB128 register followed by SLEB128 offset
/* 0x93 */ ,{"piece"	,op_args_udata}	// ULEB128 size of piece addressed
/* 0x94 */ ,{"deref_size"	,op_args_int8}	// 1-byte size of data retrieved
/* 0x95 */ ,{"xderef_size"	,op_args_int8}	// 1-byte size of data retrieved
/* 0x96 */ ,{"nop"		,0}
/* 0x97 */ ,{"push_object_address"	,0}
/* 0x98 */ ,{"call2"	,op_args_int16}	// 2-byte offset of DIE
/* 0x99 */ ,{"call4"	,op_args_int32}	// 4-byte offset of DIE
/* 0x9a */ ,{"call_ref"	,op_args_indirect}	// 4- or 8-byte offset of DIE
/* 0x9b */ ,{"form_tls_address"	,0}
/* 0x9c */ ,{"call_frame_cfa"	,0}
/* 0x9d */ ,{"bit_piece",op_args_udata_udata}
};

static const char *
get_form_name(dw_formE form)
{
	static const char *name[] = {
/* 0x00 */	"null",
/* 0x01 */	"addr",
/* 0x02 */	0,
/* 0x03 */	"block2",
/* 0x04 */	"block4",
/* 0x05 */	"data2",
/* 0x06 */	"data4",
/* 0x07 */	"data8",
/* 0x08 */	"string",
/* 0x09 */	"block",
/* 0x0a */	"block1",
/* 0x0b */	"data1",
/* 0x0c */	"flag",
/* 0x0d */	"sdata",
/* 0x0e */	"strp",
/* 0x0f */	"udata",
/* 0x10 */	"ref_addr",
/* 0x11 */	"ref1",
/* 0x12 */	"ref2",
/* 0x13 */	"ref4",
/* 0x14 */	"ref8",
/* 0x15 */	"ref_udata",
/* 0x16 */	"indirect"
	};

	const char *nameP;
	char	error[20];

	assert((sizeof(name)/sizeof(*name)) == 0x17);
	if (form < 0 || form > 0x16) {
		nameP = 0;
	} else {
		nameP = name[form];
	}
	if (!nameP) {
		sprintf(error, "0x%08x", form);
		nameP = error;
	}
	return nameP;
}

static const char *
get_tag_name(dw_tagE tag)
{
	static const char *name[] = {
/* 0x00 */	"null",
/* 0x01 */	"array_type",
/* 0x02 */	"class_type",
/* 0x03 */	"entry_point",
/* 0x04 */	"enumeration_type",
/* 0x05 */	"formal_parameter",
/* 0x06 */	0,
/* 0x07 */	0,
/* 0x08 */	"imported_declaration",
/* 0x09 */	0,
/* 0x0a */	"label",
/* 0x0b */	"lexical_block",
/* 0x0c */	0,
/* 0x0d */	"member",
/* 0x0e */	0,
/* 0x0f */	"pointer_type",
/* 0x10 */	"reference_type",
/* 0x11 */	"compile_unit",
/* 0x12 */	"string_type",
/* 0x13 */	"structure_type",
/* 0x14 */	0,
/* 0x15 */	"subroutine_type",
/* 0x16 */	"typedef",
/* 0x17 */	"union_type",
/* 0x18 */	"unspecified_parameters",
/* 0x19 */	"variant",
/* 0x1a */	"common_block",
/* 0x1b */	"common_inclusion",
/* 0x1c */	"inheritance",
/* 0x1d */	"inlined_subroutine",
/* 0x1e */	"module",
/* 0x1f */	"ptr_to_member_type",
/* 0x20 */	"set_type",
/* 0x21 */	"subrange_type",
/* 0x22 */	"with_stmt",
/* 0x23 */	"access_declaration",
/* 0x24 */	"base_type",
/* 0x25 */	"catch_block",
/* 0x26 */	"const_type",
/* 0x27 */	"constant",
/* 0x28 */	"enumerator",
/* 0x29 */	"file_type",
/* 0x2a */	"friend",
/* 0x2b */	"namelist",
/* 0x2c */	"namelist_item",
/* 0x2d */	"packed_type",
/* 0x2e */	"subprogram",
/* 0x2f */	"template_type_parameter",
/* 0x30 */	"template_value_parameter",
/* 0x31 */	"thrown_type",
/* 0x32 */	"try_block",
/* 0x33 */	"variant_part",
/* 0x34 */	"variable",
/* 0x35 */	"volatile_type",
/* 0x36 */	"dwarf_procedure",
/* 0x37 */	"restrict_type",
/* 0x38 */	"interface_type",
/* 0x39 */	"namespace",
/* 0x3a */	"imported_module",
/* 0x3b */	"unspecified_type",
/* 0x3c */	"partial_unit",
/* 0x3d */	"imported_unit",
/* 0x3e */	0,
/* 0x3f */	"condition",
/* 0x40 */	"shared_type"
	};

	const char	*nameP;
	static char error[20];

	assert((sizeof(name)/sizeof(*name)) == 0x41);
	if (tag < 0 || tag > 0x40) {
		nameP = 0;
	} else {
		nameP = name[tag];
	}
	if (!nameP) {
		sprintf(error, "0x%08x", tag);
		nameP = error;
	}
	return nameP;
}

static const char *
get_attribute_name(dw_atE attribute)
{
	static const char *name[] = {
/* 0x00 */	0,
/* 0x01 */	"sibling",
/* 0x02 */	"location",
/* 0x03 */	"name",
/* 0x04 */	0,
/* 0x05 */	0,
/* 0x06 */	0,
/* 0x07 */	0,
/* 0x08 */	0,
/* 0x09 */	"ordering",
/* 0x0a */	0,
/* 0x0b */	"byte_size",
/* 0x0c */	"bit_offset",
/* 0x0d */	"bit_size",
/* 0x0e */	0,
/* 0x0f */	0,
/* 0x10 */	"stmt_list",
/* 0x11 */	"low_pc",
/* 0x12 */	"high_pc",
/* 0x13 */	"language",
/* 0x14 */	0,
/* 0x15 */	"discr",
/* 0x16 */	"discr_value",
/* 0x17 */	"visibility",
/* 0x18 */	"import",
/* 0x19 */	"string_length",
/* 0x1a */	"common_reference",
/* 0x1b */	"comp_dir",
/* 0x1c */	"const_value",
/* 0x1d */	"containing_type",
/* 0x1e */	"default_value",
/* 0x1f */	0,
/* 0x20 */	"inline",
/* 0x21 */	"is_optional",
/* 0x22 */	"lower_bound",
/* 0x23 */	0,
/* 0x24 */	0,
/* 0x25 */	"producer	",
/* 0x26 */	0,
/* 0x27 */	"prototyped",
/* 0x28 */	0,
/* 0x29 */	0,
/* 0x2a */	"return_addr",
/* 0x2b */	0,
/* 0x2c */	"start_scope",
/* 0x2d */	0,
/* 0x2e */	"bit_stride",
/* 0x2f */	"upper_bound",
/* 0x30 */	0,
/* 0x31 */	"abstract_origin",
/* 0x32 */	"accessibility",
/* 0x33 */	"address_class",
/* 0x34 */	"artificial",
/* 0x35 */	"base_types",
/* 0x36 */	"calling_convention",
/* 0x37 */	"count",
/* 0x38 */	"data_member_location",
/* 0x39 */	"decl_column",
/* 0x3a */	"decl_file",
/* 0x3b */	"decl_line",
/* 0x3c */	"declaration",
/* 0x3d */	"discr_list",
/* 0x3e */	"encoding",
/* 0x3f */	"external",
/* 0x40 */	"frame_base",
/* 0x41 */	"friend",
/* 0x42 */	"identifier_case",
/* 0x43 */	"macro_info",
/* 0x44 */	"namelist_item",
/* 0x45 */	"priority",
/* 0x46 */	"segment",
/* 0x47 */	"specification",
/* 0x48 */	"static_link",
/* 0x49 */	"type",
/* 0x4a */	"use_location",
/* 0x4b */	"variable_parameter",
/* 0x4c */	"virtuality",
/* 0x4d */	"vtable_elem_location",
/* 0x4e */	"allocated	",
/* 0x4f */	"associated",
/* 0x50 */	"data_location",
/* 0x51 */	"byte_stride",
/* 0x52 */	"entry_pc",
/* 0x53 */	"use_UTF8",
/* 0x54 */	"extension",
/* 0x55 */	"ranges",
/* 0x56 */	"trampoline",
/* 0x57 */	"call_column",
/* 0x58 */	"call_file",
/* 0x59 */	"call_line",
/* 0x5a */	"description",
/* 0x5b */	"binary_scale",
/* 0x5c */	"decimal_scale",
/* 0x5d */	"small",
/* 0x5e */	"decimal_sign",
/* 0x5f */	"digit_count",
/* 0x60 */	"picture_string",
/* 0x61 */	"mutable",
/* 0x62 */	"threads_scaled",
/* 0x63 */	"explicit",
/* 0x64 */	"object_pointer",
/* 0x65 */	"endianity",
/* 0x66 */	"elemental",
/* 0x67 */	"pure",
/* 0x68 */	"recursive"
};
	const char	*nameP;
	static char error[20];

	assert((sizeof(name)/sizeof(*name)) == 0x69);
	if (attribute == DW_AT_signature) {
		nameP = "signature";
	} else {
		if (attribute < 0 || attribute > 0x68) {
			nameP = 0;
		} else {
			nameP = name[attribute];
		}
		if (!nameP) {
			sprintf(error, "0x%08x", attribute);
			nameP = error;
	}	}
	return nameP;
}

#if 0

// Code to test that signed shift works correctly

#include <stdio.h>
int
main(int argc, char **argv)
{
	long long i;

	union {
		long long			k;
		unsigned long long  j;
	} val;

	i       = -2;
	val.k   = i;
	i     >>= 1;
	val.j >>= 1;

	printf("0x%16llx %lld 0x%16llx %lld\n", i, i, val.j, val.j);
	return(0);
}
#endif

static int g_offset;

static void
encode_uleb128(uint64T value, char **bufferPP, int load)
{
	char	*bufferP;
	uint8T	byte;

	bufferP = *bufferPP;
	do {
		byte    = value & 0x7F;	// Low order 7 bits of value
		value >>= 7;
		if (value != 0) { 		// more bytes to come
			byte |= 0x80;		// Set high bit of byte
		}
		if (load) {
			*bufferP = byte;
		}
		++bufferP;
	} while (value);
	*bufferPP = bufferP;
}

static uint64T
decode_uleb128(uint8T **bufferPP)
{
	uint8T			*bufferP;
	uint64T			ret, temp;
	int				shift;
	unsigned int	c;


	ret     = 0;
	shift   = 0;
	bufferP = *bufferPP;

	for (;;) {
		c = *bufferP++;
		assert(shift < 64);
		temp   = (c & 0x7F);
		temp <<= shift;
		ret   |= temp;
		if (!(c & 0x80)) {
			break;
		}
		shift += 7;
	}
	*bufferPP = bufferP;
	return(ret);
}

static void
encode_sleb128(sint64T value, char **bufferPP, int load)
{
	char	*bufferP;
	char	byte;
	int		more;

	bufferP = *bufferPP;
	more    = 1;

	do {
		byte    = value & 0x7F;	// low order 7 bits of value
		value >>= 7;
		if ((value ==  0 && !(byte & 0x40)) ||
		    (value == -1 &&  (byte & 0x40))) {
			more = 0;
		} else {
			byte |= 0x80;
		}
		if (load) {
			*bufferP = byte;
		}
		++bufferP;
	} while (more);
	*bufferPP = bufferP;
}

static sint64T
decode_sleb128(uint8T **bufferPP)
{
	uint8T	*bufferP;
	sint64T	ret;
	uint64T	temp;
	int		shift;
	uint32T	c;


	ret     = 0;
	shift   = 0;
	bufferP = *bufferPP;

	do {
		c = *bufferP++;
		assert(shift < 64);
		temp   = (c & 0x7F);
		temp <<= shift;
		ret   |= temp;
		shift += 7;
	} while (c & 0x80);

	if ((shift < 64) && (c & 0x40)) {
		temp   = 1;
		temp <<= shift;
		ret   |= -((sint64T) temp);
	}
	*bufferPP = bufferP;
	return(ret);
}

#if 0
// Code to test encode and decode

int
main(int argc, char **argv)
{
	unsigned long long uint64_1, uint64_2;
	signed long long   sint64_1, sint64_2;
	char		 buffer[24];
	char		*bufferP, *buffer1P;

#if 0
	uint64_1 = 0;
	do {
		bufferP = buffer;
		encode_uleb128(uint64_1, &bufferP ,1);
		buffer1P = buffer;
		uint64_2 = decode_uleb128(&buffer1P);
		assert(uint64_1 == uint64_2);
		assert(bufferP == buffer1P);
		++uint64_1;
	} while (uint64_1 <= 0xffffffff);
#endif


	sint64_1 = 0xffffffff80000000ll;
	do {
		bufferP = buffer;
		if (!(sint64_1 & 0xFFFFFFll)) {
			printf("sleb128: 0x%16llx\n", sint64_1);
		}
		encode_sleb128(sint64_1, &bufferP ,1);
		buffer1P = buffer;
		sint64_2 = decode_sleb128(&buffer1P);
		if (sint64_1 != sint64_2) {
			printf("sleb128: 0x%16llx 0x%16llx\n", sint64_1, sint64_2);
			assert(sint64_1 == sint64_2);
		}
		assert(bufferP == buffer1P);
		++sint64_1;
	} while (sint64_1 <= 0xffffffffll);


	return(0);
}
#endif

void
Cdwarf::print_ops(int lth, uint8T *blockP)
{
	uint8T		*P, *endP;
	const char 	*nameP;
	uint32T		code;
	dw_formE	*formP, form;
	uint8T		uint8;
	uint16T		uint16;
	uint32T		uint32;
	urefT		uref;
	uint64T		uint64;
	sint64T		sint64;

	assert((sizeof(op_table)/sizeof(*op_table)) == 0x9e);
	P = blockP + sizeof(uint32T);
	for (endP = P + lth; P < endP; ) {
		code = *P++;
		assert(code <= 0x9d);
		nameP = op_table[code].nameP;
		assert(nameP);
		fprintf(stderr, " [%s", nameP);
		formP = op_table[code].argsP;
		if (formP) {
			for (; (form = *formP) != DW_FORM_null; ++formP) {
				fprintf(stderr, " %s", get_form_name(form));
				switch (form) {
				case DW_FORM_data1:
					memcpy(&uint8, P, sizeof(uint8T));
					P += sizeof(uint8T);
					fprintf(stderr, " %d", (int) uint8);
					break;
				case DW_FORM_data2:
					memcpy(&uint16, P, sizeof(uint16T));
					P += sizeof(uint16T);
					fprintf(stderr, " %d", (int) uint16);
					break;
				case DW_FORM_data4:
				case DW_FORM_indirect:
					memcpy(&uint32, P, sizeof(uint32T));
					P += sizeof(uint32T);
					fprintf(stderr, " %d", uint32);
					break;
				case DW_FORM_addr:
					uref = get_uref(&P);
					fprintf(stderr, " %lld", uref);
					break;
				case DW_FORM_data8:
					memcpy(&uint64, P, sizeof(uint64T));
					P += sizeof(uint64T);
					fprintf(stderr, " %lld", uint64);
					break;
				case DW_FORM_udata:
					uint64 = decode_uleb128(&P);
					fprintf(stderr, " %llu", uint64);
					break;
				case DW_FORM_sdata:
					sint64 = decode_sleb128(&P);
					fprintf(stderr, " %lld", sint64);
					break;
				default:
					assert(0);
		}	}	}
		fprintf(stderr, "]");
	}
	assert(P == endP);
}

static int
load_unsigned(char **atPP, uint64T *valueP)
{
	char	*atP;
	uint64T	ret;
	char	c;
	
	atP = *atPP;

	ret = 0;
	if ((c = *atP) == '0') {
		if ((c = atP[1]) == 'x' || c == 'X') {
			for (++atP ;;) {
				c = *(++atP);
				if (c >= '0' && c <= '9') {
					c -= '0';
				} else if ( c >= 'a' && c <= 'f') {
					c += 10 - 'a';
				} else if (c >= 'A' && c <= 'F') {
					c += 10 - 'A';
				} else {
					goto done;
				}
				ret = ret * 16 + c;
		}	}
	} else {
		if (c < '0' || c > '9') {
			assert(0);
			return(0);
	}	}

	do {
		ret = ret * 10 + c - '0';
		c   = *(++atP);
	} while (c >= '0' && c <= '9');

done:
	*valueP = ret;
	*atPP   = atP;
	return(1);
}

static int
load_signed(char **atPP, sint64T *valueP)
{
	char	*atP;
	sint64T	ret;
	int		sign;
	char	c, c1;
	
	sign = 0;
	atP  = *atPP;

	ret = 0;
	if ((c = *atP) == '0') {
		if ((c1 = atP[1]) == 'x' || c1 == 'X') {
			for (++atP ;;) {
				c = *(++atP);
				if (c >= '0' && c <= '9') {
					c -= '0';
				} else if ( c >= 'a' && c <= 'f') {
					c += 10 - 'a';
				} else if (c >= 'A' && c <= 'F') {
					c += 10 - 'A';
				} else {
					goto done;
				}
				ret = ret * 16 + c;
		}	}
	} else {
		if (c == '-') {
			sign = 1;
			c = *(++atP);
		}
		if (c < '0' || c > '9') {
			assert(0);
			return(0);
	}	}

	do {
		ret = ret * 10 + c - '0';
		c   = *(++atP);
	} while (c >= '0' && c <= '9');

	if (sign) {
		ret = -ret;
	}

done:
	*valueP = ret;
	*atPP   = atP;
	return(1);
}

static int
load_uint8(char **atPP, uint8T *valueP)
{
	char 	*atP;
	uint64T	uint64;

	atP = *atPP;
	if (strncmp(atP, " .byte ", 7)) {
		assert(0);
		return(0);
	}
	atP += 7;
	if (!load_unsigned(&atP, &uint64)) {
		assert(0);
		return(0);
	}
	if (uint64 > 0xff) {
		assert(0);
		return(0);
	}
	if (*atP++ != '\n') {
		assert(0);
		return(0);

	}
	*valueP = (uint8T) uint64;
	*atPP   = atP;
	++g_offset;
	return(1);
}

static int
load_uint16(char **atPP, uint16T *valueP)
{
	char 	*atP;
	uint64T	uint64;

	atP = *atPP;
	if (strncmp(atP, " .value ", 8)) {
		assert(0);
		return(0);
	}
	atP += 8;
	if (!load_unsigned(&atP, &uint64)) {
		assert(0);
		return(0);
	}
	if (uint64 > 0xffff) {
		assert(0);
		return(0);
	}
	if (*atP++ != '\n') {
		assert(0);
		return(0);
	}
	*valueP   = (uint16T) uint64;
	*atPP     = atP;
	g_offset += 2;
	return(1);
}

int
Cdwarf::load_label(char **atPP, urefT *valueP)
{
	char	*atP, *P;
	char	*P1;
	char	c;
	labelT	*labelP;
	int		lth;

	atP = P = *atPP;
	for (; ; ++P) {
		switch (c = *P) {
		case 0:
		case ' ':
		case ',':
		case '\n':
			break;
		default:
			continue;
		}
		break;
	}
	*P  = 0;
	lth = P - atP;

	for (labelP = m_labelsP; (P1 = labelP->labelP); ++labelP) {
		if (lth == labelP->label_lth && !strncmp(atP, P1, lth)) {
			*valueP = (urefT) labelP->lineno;
			*P    = c;
			*atPP = P;
			return(1);
	}	}
	*P    = c;
	*atPP = P;
	// Unknown label (hopefully something we don't care about)
	*valueP = 0xFFFF1111;
	return(1);
}

int
Cdwarf::load_uint32(char **atPP, uint32T *valueP)
{
	char 	*atP;
	uint32T	uint32;
	uint64T	uint64;
	urefT	uref;
	char	c;

	atP = *atPP;
	if (strncmp(atP, " .long ", 7)) {
		assert(0);
		return(0);
	}
	atP += 7;
	if ((c = *atP) >= '0' && c <= '9') {
		if (!load_unsigned(&atP, &uint64)) {
			assert(0);
			return(0);
		}
		if (uint64 > 0xffffffff) {
			assert(0);
			return(0);
		}
		uint32 = (uint32T) uint64;
	} else {
		if (!load_label(&atP, &uref)) {
			assert(0);
			return(0);
		}
		if (uref > 0xffffffff) {
			assert(0);
			return(0);
		}
		uint32 = uref;
	}
	if (*atP++ != '\n') {
		assert(0);
		return(0);
	}
	*valueP   = uint32;
	*atPP     = atP;
	g_offset += sizeof(uint32T);
	return(1);
}

int
Cdwarf::load_uint64(char **atPP, uint64T *valueP)
{
	char 	*atP;
	uint64T	uint64;
	urefT	uref;
	char	c;

	atP = *atPP;
	if (strncmp(atP, " .quad ", 7)) {
		assert(0);
		return(0);
	}
	atP += 7;
	if ((c = *atP) >= '0' && c <= '9') {
		if (!load_unsigned(&atP, &uint64)) {
			assert(0);
			return(0);
		}
	} else {
		if (!load_label(&atP, &uref)) {
			assert(0);
			return(0);
		}
		uint64 = uref;
	}
	if (*atP++ != '\n') {
		assert(0);
		return(0);
	}
	*valueP   = uint64;
	*atPP     = atP;
	g_offset += sizeof(uint64T);
	return(1);
}

static const uint64T max_uleb128[] = {
	0x000000000000007fll,
	0x0000000000003fffll,
	0x00000000001fffffll,
	0x000000000fffffffll,
	0x00000007ffffffffll,
	0x000003ffffffffffll,
	0x0001ffffffffffffll,
	0x00ffffffffffffffll,
	0x7fffffffffffffffll,
	0xffffffffFFFFFFFFll
};

static int
load_uleb128(char **atPP, uint32T *valueP)
{
	char 			*atP;
	uint64T			uint64;
	const uint64T	*maxP;

	atP = *atPP;
	if (strncmp(atP, " .uleb128 ", 10)) {
		if (!strncmp(atP, " .byte ", 7)) {
			atP += 7;
			if (!load_unsigned(&atP, &uint64)) {
				assert(0);
				return(0);
			}
			assert(!uint64);
			goto done;
		}
		assert(0);
		return(0);
	}
	atP += 10;
	if (!load_unsigned(&atP, &uint64)) {
		assert(0);
		return(0);
	}
done:
	if (uint64 > 0xffffffff) {
		assert(0);
		return(0);
	}
	*valueP = (uint32T) uint64;
	if (*atP++ != '\n') {
		assert(0);
		return(0);
	}
	for (maxP = max_uleb128; ; ++maxP) {
		++g_offset;
		if (uint64 <= *maxP) {
			break;
	}	}
	*atPP = atP;
	return(1);
}

static const sint64T max_sleb128[] = {
	0x000000000000003fll,
	0x0000000000001fffll,
	0x00000000000fffffll,
	0x0000000007ffffffll,
	0x00000003ffffffffll,
	0x000001ffffffffffll,
	0x0000ffffffffffffll,
	0x007fffffffffffffll,
	0x3fffffffffffffffll,
	0x7fffffffFFFFFFFFll
};

static const sint64T min_sleb128[] = {
	0xffffffffffffffc0ll,
	0xffffffffffffe000ll,
	0xfffffffffff00000ll,
	0xfffffffff8000000ll,
	0xfffffffc00000000ll,
	0xfffffe0000000000ll,
	0xffff000000000000ll,
	0xff80000000000000ll,
	0xc000000000000000ll,
	0x8000000000000000ll
};

static int
load_sleb128(char **atPP, sint32T *valueP)
{
	char 			*atP;
	sint64T			sint64;
	const sint64T	*maxP;

	atP = *atPP;
	if (strncmp(atP, " .sleb128 ", 10)) {
/*
		if (!strncmp(atP, " .byte ", 7)) {
			atP += 7;
			if (!load_unsigned(&atP, valueP)) {
				assert(0);
				return(0);
			}
			assert(!*valueP);
			return(1);
		}
 */
		assert(0);
		return(0);
	}
	atP += 10;
	if (!load_signed(&atP, &sint64)) {
		assert(0);
		return(0);
	}
	*valueP = (sint32T) sint64;
	if (*atP++ != '\n') {
		assert(0);
		return(0);
	}
	if (0 <= sint64) { 
		for (maxP = max_sleb128; ; ++maxP) {
			++g_offset;
			if (sint64 <= *maxP) {
				break;
		}	}
	} else {
		for (maxP = min_sleb128; ; ++maxP) {
			++g_offset;
			if (*maxP <= sint64) {
				break;
	}	}	}
	*atPP = atP;
	return(1);
}

int
load_string1(char **atPP, char **bufferPP, int load, int escape)
{
	char 	*atP, *bufferP, *P, *P1;
	char	c;
	int		val;

	atP     = *atPP;
	bufferP = P1 = *bufferPP;

	for (P = atP; (c = *P) != '"'; ++P) {
		if (c == '\\') {
			c   = *(++P);
			if (escape) {
				if (load) {
					*P1 = '\\';
				}
				++P1;
				if (c >= '0' && c <= '3') {
					if (load) {
						*P1 = c;		// Put back first octal digit
					}
					++P1;
					c = *(++P);
					assert(c >= '0' && c <= '7');
					if (load) {
						*P1 = c;		// Put back second octal digit
					}
					++P1;
					c = *(++P);
					assert(c >= '0' && c <= '7');
				}
			} else {
				switch (c) {
				case 'n':
					c = '\n';
					break;
				case 't':
					c = '\t';
					break;
				case 'v':
					c = '\v';
					break;
				case 'b':
					c = '\b';
					break;
				case 'r':
					c = '\r';
					break;
				case 'f':
					c = '\f';
					break;
				case 'a':
					c = '\a';
					break;
				case '0':
				case '1':
				case '2':
				case '3':
					// Three digit octal number
					val = c - '0';
					c   = *(++P);
					assert(c >= '0' && c <= '7');
					val = val * 8 + c - '0';
					c   = *(++P);
					c   = val * 8 + c - '0';
					break;
		}	}	}
		if (load) {
			*P1 = c;
		}
		++g_offset;
		++P1;
	}
	if (load) {
		*P1 = 0;
	}
	++g_offset;
	++P1;
	if (*(++P) != '\n') {
		assert(0);
		return(0);
	}
	*atPP     = P+1;
	*bufferPP = P1;
	return(1);
}	

int
load_string(char **atPP, char **bufferPP, int load, int escape)
{
	char 	*atP, *bufferP;
	int		ret;

	atP     = *atPP;
	bufferP = *bufferPP;

	// Utter stupidity on part of old compilers
	while (!strncmp(atP, " .ascii \"", 9)) {
		atP += 9;
		if (!load_string1(&atP, &bufferP, load, escape)) {
			return(0);
		}
		// Later overwrite trailing \0
		--bufferP;
		// .ascii is not null terminated
		--g_offset;
	}
	if (strncmp(atP, " .string \"", 10)) {
		assert(0);
		return(0);
	}
	atP      += 10;
	ret       = load_string1(&atP, &bufferP, load, escape);
	*atPP     = atP;
	*bufferPP = bufferP;
	return(ret);
}	

int
Cdwarf::load_addr(char **atPP, urefT *valueP)
{
	uint32T	uint32;
	uint64T	uint64;
	urefT	uref;

	if (!strncmp(*atPP, " .l", 3)) {
		if (!load_uint32(atPP, &uint32)) {
			assert(0);
			return(0);
		}
		uref = (urefT) uint32;
	} else {
		if (!load_uint64(atPP, &uint64)) {
			assert(0);
			return(0);
		}
		uref = (urefT) uint64;
	}
	*valueP = uref;
	return(1);
}

int
Cdwarf::load_uint12(char **atPP, urefT *valueP)
{
	uint32T 	uint32;
	uint64T		uint64;

	if (!load_uint32(atPP, &uint32)) {
		assert(0);
		return(0);
	}
	if (uint32 != 0xffffffff) {
		uint64 = uint32;
	} else {
		if (!load_uint64(atPP, &uint64)) {
			assert(0);
			return(0);
		}
	}
	*valueP   = uint64;
	return(1);
}

int
Cdwarf::load_block(char **atPP, char **bufferPP, int load, int lth)
{
	char 	*atP, *bufferP, *endP;
	uint64T	uint64;
	uint32T	uint32;
	sint32T	sint32;
	uint16T	uint16;
	uint8T	uint8;

	atP     = *atPP;
	bufferP = *bufferPP;
	endP    = bufferP + lth;

	while (bufferP < endP) {
		if (atP[0] != ' ' || atP[1] != '.') {
			assert(0);
			return(0);
		}
		switch (atP[2]) {
		case 'q':
			if (!load_uint64(&atP, &uint64)) {
				assert(0);
				return(0);
			}
			if (load) {
				memcpy(bufferP, &uint64, sizeof(uint64T));
			}
			bufferP += sizeof(uint64T);
			break;
		case 'l':
			if (!load_uint32(&atP, &uint32)) {
				assert(0);
				return(0);
			}
			if (load) {
				memcpy(bufferP, &uint32, sizeof(uint32T));
			}
			bufferP += sizeof(uint32T);
			break;
		case 'v':
			if (!load_uint16(&atP, &uint16)) {
				assert(0);
				return(0);
			}
			if (load) {
				memcpy(bufferP, &uint16, sizeof(uint16T));
			}
			bufferP += sizeof(uint16T);
			break;
		case 'b':
			if (!load_uint8(&atP, &uint8)) {
				assert(0);
				return(0);
			}
			if (load) {
				*bufferP = uint8;
			}
			++bufferP;
			break;
		case 'u':
			if (!load_uleb128(&atP, &uint32)) {
				assert(0);
				return(0);
			}
			encode_uleb128((uint64T) uint32, &bufferP, load);
			break;
		case 's':
			switch (atP[3]) {
			case 't':
				if (!load_string(&atP, &bufferP, load, 0 /* Dont escape */)) {
					assert(0);
					return(0);
				}
				break;
			case 'l':
				if (!load_sleb128(&atP, &sint32)) {
					assert(0);
					return(0);
				}
				encode_sleb128((sint64T) sint32, &bufferP, load);
				break;
			default:
				assert(0);
				return(0);
			}
			break;
		default:
			assert(0);
			return(0);
	}	}
	assert(bufferP == endP);
	*bufferPP = bufferP;
	*atPP     = atP;
	return(1);
}

/* Make a crude attempt to evaluate a block */

static int
evaluate_block(dwarf_attrT	*attributeP)
{
	uint8T			*blockP, *endP;
	unsigned int	lth;
	uint64T			uint64;
	sint64T			sint64;

	blockP  = (uint8T *) attributeP->value.P;
	lth     = *((uint32T *) blockP);
	blockP += sizeof(uint32T);
	endP    = blockP + lth;
	switch (*blockP++) {
	case DW_OP_const1u:
		attributeP->value.uint32 = *((unsigned char *) blockP);
		++blockP;
		attributeP->form         = DW_FORM_ref4;
		break;
	case DW_OP_const1s:
		attributeP->value.sint32 = *((signed char *) blockP);
		++blockP;
		attributeP->form         = DW_FORM_data4;
		break;
	case DW_OP_const2u:
		attributeP->value.uint32 = *((uint16T *) blockP);
		blockP                  += sizeof(uint16T);
		attributeP->form         = DW_FORM_ref4;
		break;
	case DW_OP_const2s:
		attributeP->value.sint32 = *((sint16T *) blockP);
		blockP                  += sizeof(sint16T);
		attributeP->form         = DW_FORM_data4;
		break;
	case DW_OP_const4u:
		attributeP->value.uint32 = *((uint32T *) blockP);
		blockP                  += sizeof(uint32T);
		attributeP->form         = DW_FORM_ref4;
		break;
	case DW_OP_const4s:
		attributeP->value.sint32 = *((sint32T *) blockP);
		blockP                  += sizeof(sint32T);
		attributeP->form         = DW_FORM_data4;
		break;
	case DW_OP_const8u:
		uint64                   = *((uint64T *) blockP);
		blockP                  += sizeof(uint64T);
		goto check_uint64;
	case DW_OP_const8s:
		sint64                   = *((sint64T *) blockP);
		blockP                  += sizeof(sint64T);
		goto check_sint64;
	case DW_OP_constu:
	case  DW_OP_plus_uconst:
		uint64 = decode_uleb128(&blockP);
check_uint64:
		if (uint64 <= 0xFFFFFFFFu) {
			attributeP->value.uint32 = (uint32T) uint64;
			attributeP->form         = DW_FORM_ref4;
		} else {
			attributeP->value.uint64 = uint64;
			attributeP->form         = DW_FORM_ref8;
		}
		break;
	case DW_OP_consts:
		sint64 = decode_sleb128(&blockP);
check_sint64:
		if (sint64 <= 0x7FFFFFFF && sint64 >= 0x10000000) {
			attributeP->value.sint32 = (sint32T) sint64;
			attributeP->form         = DW_FORM_data4;
		} else {
			attributeP->value.sint64 = sint64;
			attributeP->form         = DW_FORM_data8;
		}
		break;
	case DW_OP_lit0:
	case DW_OP_lit1:
	case DW_OP_lit2:
	case DW_OP_lit3:
	case DW_OP_lit4:
	case DW_OP_lit5:
	case DW_OP_lit6:
	case DW_OP_lit7:
	case DW_OP_lit8:
	case DW_OP_lit9:
	case DW_OP_lit10:
	case DW_OP_lit11:
	case DW_OP_lit12:
	case DW_OP_lit13:
	case DW_OP_lit14:
	case DW_OP_lit15:
	case DW_OP_lit16:
	case DW_OP_lit17:
	case DW_OP_lit18:
	case DW_OP_lit19:
	case DW_OP_lit20:
	case DW_OP_lit21:
	case DW_OP_lit22:
	case DW_OP_lit23:
	case DW_OP_lit24:
	case DW_OP_lit25:
	case DW_OP_lit26:
	case DW_OP_lit27:
	case DW_OP_lit28:
	case DW_OP_lit29:
	case DW_OP_lit30:
	case DW_OP_lit31:
		attributeP->value.uint32 = blockP[-1] - DW_OP_lit0;
		attributeP->form         = DW_FORM_ref4;
		break;
	default:
		return(0);
	}
	assert(blockP == endP);
	return(1);
}

Cdwarf::Cdwarf(void)
{
	m_abbrevP     = 0;
	m_abbrev_max  = 0;
	m_abbrev_need = 0;

	m_nodes_lth   = 0;
	m_nodeP       = 0;

	m_strings_lth = 0;
	m_strings_max = 0;
	m_stringsP    = 0;

	m_functionP   = 0;
	m_locations   = 0;
	m_max_locations = 0;
	m_locationsP  = 0;

	// Try to optimise access to vtable information

	m_vfunctions     = -1;	// unknown
	m_vfunctions_max = 0;
	m_vfunctionsP    = 0;
}

Cdwarf::~Cdwarf(void)
{
	Xfree(m_vfunctionsP);
	Xfree(m_abbrevP);
	Xfree(m_nodeP);
	Xfree(m_stringsP);
	Xfree(m_locationsP);
}

#ifdef DUMP_HEADER
static void
dump_header(dwarf_headerT *headerP)
{
	fprintf(stderr, "Header Lth=%lld Version=%d Offset=%lld Size=%d\n",
		headerP->unit_length,
		(int) headerP->version,
		headerP->debug_abbrev_offset,
		(int) headerP->address_size);
}
#endif

int
Cdwarf::load_header(char **atPP)
{
	if (!load_uint12(atPP, &m_header.unit_length)) {
		assert(0);
		return(0);
	}
	if (!load_uint16(atPP, &m_header.version)) {
		assert(0);
		return(0);
	}
	assert(m_header.version >= 2 && m_header.version <= 3);
	if (!load_uint12(atPP, &m_header.debug_abbrev_offset)) {
		assert(0);
		return(0);
	}
	if (!load_uint8(atPP, &m_header.address_size)) {
		assert(0);
		return(0);
	}
	assert(m_header.address_size == 4 || m_header.address_size == 8);
#ifdef DUMP_HEADER
	dump_header(&m_header);
#endif
	return(1);
}

#ifdef DUMP_ABBREVIATIONS

static void
dump_abbr_attributes(int attributes, dwarf_abbrev_attrT *attributesP)
{
	dwarf_abbrev_attrT	*attributeP, *end_attributeP;

	attributeP = attributesP;
	for (end_attributeP = attributeP + attributes; attributeP < end_attributeP; ++attributeP) {
		fprintf(stderr,"%s%s %s\n",
			INDENT,
			get_attribute_name(attributeP->name),
			get_form_name(attributeP->form));
	}
}

static void
dump_abbreviations(int abbrev_lth, dwarf_abbrevT *startP)
{
	dwarf_abbrevT		*abbrevP, *end_abbrevP;

	abbrevP = startP;
	for (end_abbrevP = abbrevP + abbrev_lth; abbrevP < end_abbrevP; ++abbrevP) {
		fprintf(stderr, "%3d: %s", (int) (abbrevP - startP), get_tag_name(abbrevP->tag));
		
		if (abbrevP->children) {
			fputs(" children", stderr);
		}
		fputc('\n', stderr);
		dump_abbr_attributes(abbrevP->attributes, abbrevP->attributesP);
}	}
#endif

int
Cdwarf::load_debug_abbrev(char *debug_abbrevP)
{
	char				*atP;
	uint32T				abbrev_code, abbrev_max, tag;
	uint8T				children;
	uint32T				name, form;
	int					abbrev_attr;
	dwarf_abbrevT		*abbrevP, *end_abbrevP, *abbrev1P;
	unsigned int		need;
	dwarf_abbrev_attrT	*start_attributesP, *attributeP, *startP;

	atP         = debug_abbrevP;
	abbrev_max  = 0;
	abbrev_attr = 0;
	for (;;) {
		if (!load_uleb128(&atP, &abbrev_code)) {
			assert(0);
			return(0);
		}
		if (!abbrev_code) {
			break;
		}
		if (abbrev_code > abbrev_max) {
			abbrev_max = abbrev_code;
		}
		if (!load_uleb128(&atP, &tag)) {
			assert(0);
			return(0);
		}
		if (!load_uint8(&atP, &children)) {
			assert(0);
			return(0);
		}
		for (;;) {
			if (!load_uleb128(&atP, &name)) {
				assert(0);
				return(0);
			}
			if (!load_uleb128(&atP, &form)) {
				assert(0);
				return(0);
			}
			if (!name && !form) {
				break;
			}
			++abbrev_attr;
	}	}

	need = (abbrev_max + 1) * sizeof(dwarf_abbrevT) + abbrev_attr * sizeof(dwarf_abbrev_attrT);
	if (!m_abbrevP || m_abbrev_need < need) {
		Xfree(m_abbrevP);
		m_abbrevP     = (dwarf_abbrevT *) Xmalloc(need);
		m_abbrev_need = need;
	}
	m_abbrev_max = abbrev_max;

	abbrevP = m_abbrevP;
	for (end_abbrevP  = abbrevP + abbrev_max; abbrevP <= end_abbrevP; ++abbrevP) {
		Xcheck(m_abbrevP, need, abbrevP);
		abbrevP->tag         = DW_TAG_null;
		abbrevP->attributesP = 0;
	}
	attributeP = start_attributesP = (dwarf_abbrev_attrT *) abbrevP;
	abbrevP    = m_abbrevP;
	atP        = debug_abbrevP;
	for (;;) {
		if (!load_uleb128(&atP, &abbrev_code)) {
			assert(0);
			return(0);
		}
		if (!abbrev_code) {
			break;
		}
		assert(abbrev_code <= abbrev_max);
		abbrev1P = abbrevP + abbrev_code;
		Xcheck(abbrevP, need, abbrev1P);
		assert(abbrev1P->tag == DW_TAG_null);
		assert(!abbrev1P->attributesP);
		if (!load_uleb128(&atP, (unsigned int *) &abbrev1P->tag)) {
			assert(0);
			return(0);
		}
		if (!load_uint8(&atP, &children)) {
			assert(0);
			return(0);
		}
		assert(children < 2);
		abbrev1P->children    = (dw_childrenE) children;
		abbrev1P->attributesP = startP = attributeP;
		for (;;) {
			if (!load_uleb128(&atP, &name)) {
				assert(0);
				return(0);
			}
			if (!load_uleb128(&atP, &form)) {
				assert(0);
				return(0);
			}
			if (!name && !form) {
				abbrev1P->attributes = attributeP - startP;
				break;
			}
			Xcheck(abbrevP, need, attributeP);
			attributeP->name = (dw_atE)   name;
			attributeP->form = (dw_formE) form;
			++attributeP;
	}	}
	assert(((char *) attributeP) == ((char *) m_abbrevP) + need);
#ifdef DUMP_ABBREVIATIONS
	dump_abbreviations(m_abbrev_max, m_abbrevP);
#endif
	return(1);
}

urefT
Cdwarf::get_uref(uint8T	**PP)
{
	uint8T	*P;
	uint32T	uint32;
	uint64T	uint64;

	P = *PP;
	switch (m_header.address_size) {
	case 4:
		memcpy(&uint32, P, sizeof(uint32T));
		P   += sizeof(uint32T);
		*PP  = P;
		return (urefT) uint32;
	case 8:
		memcpy(&uint64, P, sizeof(uint64T));
		P    += sizeof(uint64T);
		*PP   = P;
		return (urefT) uint64;
	default:
		assert(0);
		return 0;
	}
}

urefT
Cdwarf::get_uref(dw_valueT	*valueP)
{
	switch (m_header.address_size) {
	case 4:
		return (urefT) valueP->uint32;
	case 8:
		return (urefT) valueP->uint64;
	default:
		assert(0);
		return(0);
	}
}

/* Returns
   0:	No node found
   1:	Node found and loaded
  -1:	Error
 */

int
Cdwarf::load_node(char **atPP, char **bufferPP, dwarf_nodeT **nodePP)
{
	char				*startP, *atP;
	char				*bufferP, *extraP;
	dwarf_nodeT			*nodeP, **tailPP, *childP;
	dwarf_attrT			*attributeP;
	dwarf_abbrevT		*abbrevP;
	dwarf_abbrev_attrT	*abbrev_attributeP, *end_abbrev_attributeP;
	unsigned int		code;
	dw_formE			form;
	dw_valueT			value;
	uint32T				lth;
	uint32T				uint32;
	uint16T		      	uint16;
	uint8T				uint8;
	int					i;
	int					seen_children;
	int					offset;

	offset        = g_offset;
	atP = startP  = *atPP;
	for (bufferP = *bufferPP; ((long) bufferP) & 7; ++bufferP);

	if (!load_uleb128(&atP, &code)) {
		assert(0);
		return -1;
	}
	if (!code) {
		*atPP = atP;
		return(0);
	}

	assert(code <= m_abbrev_max);
	abbrevP           = m_abbrevP + code;
	nodeP             = (dwarf_nodeT *) bufferP;
	attributeP        = (dwarf_attrT *) (nodeP+1);
	abbrev_attributeP = abbrevP->attributesP;
	extraP            = (char *) (attributeP + abbrevP->attributes);

	if (nodePP) {
		*nodePP            = nodeP;
		nodeP->tag         = abbrevP->tag;
		nodeP->offset      = offset;
		nodeP->attributes  = abbrevP->attributes;
		nodeP->attributesP = attributeP;

		if (g_debug_dwarf) {
			for (i = m_depth; --i >= 0; ) {
				fputc(' ', stderr);
				fputc(' ', stderr);
			}
			fprintf(stderr, "%u: %s", offset, get_tag_name(abbrevP->tag));
		}
	}
	for (end_abbrev_attributeP = abbrev_attributeP + abbrevP->attributes; abbrev_attributeP < end_abbrev_attributeP; ++abbrev_attributeP) {
		form = abbrev_attributeP->form;
retry:
		switch (form) {
		case DW_FORM_null:
			value.uint32 = 0;
			break;
		case DW_FORM_flag:		// flag (one byte 0 or 1)
			if (!load_uint8(&atP, &uint8)) {
				assert(0);
				return -1;
			}
			assert(uint8 == 0 || uint8 == 1);
			value.uint32 = uint8;
			break;
		case DW_FORM_ref1:		// 1 byte offset reference
			if (!load_uint8(&atP, &uint8)) {
				assert(0);
				return -1;
			}
			value.uref = uint8;
			break;

		case DW_FORM_data1:		// 1 byte constant
			if (!load_uint8(&atP, &uint8)) {
				assert(0);
				return -1;
			}
			value.uint32 = uint8;
			break;
		case DW_FORM_ref2:		// 2 byte reference
			if (!load_uint16(&atP, &uint16)) {
				assert(0);
				return(-1);
			}
			value.uref = uint16;
			break;

		case DW_FORM_data2:		// 2 byte constant
			if (!load_uint16(&atP, &uint16)) {
				assert(0);
				return(-1);
			}
			value.uint32 = uint16;
			break;
		case DW_FORM_ref4:		// 4 byte reference
			if (!load_uint32(&atP, &uint32)) {
				assert(0);
				return(-1);
			}
			value.uref = uint32;
			break;

		case DW_FORM_data4:		// int (constant, lineptr, loclistptr, macptr, rangelistptr)
			if (!load_uint32(&atP, &value.uint32)) {
				assert(0);
				return(-1);
			}
			break;
		case DW_FORM_ref_udata:	// unsigned LEB128 reference
			if (!load_uleb128(&atP, &uint32)) {
				assert(0);
				return(-1);
			}
			value.uref = uint32;
			break;
		case DW_FORM_addr:		// address
			if (!load_addr(&atP, &value.uref)) {
				assert(0);
				return(-1);
			}
			break;
		case DW_FORM_strp:		// offset into string table in .debug_str
			if (!load_uint12(&atP, &value.uref)) {
				assert(0);
				return(-1);
			}
			break;
		case DW_FORM_ref_addr:	// offset reference from start of debug info
			if (!load_uint12(&atP, &value.uref)) {
				assert(0);
				return(-1);
			}
			break;
		case DW_FORM_ref8:		// 8 byte reference
		case DW_FORM_data8:		// long (constant, lineptr, loclistptr, macptr, rangelistptr)
			if (!load_uint64(&atP, &value.uint64)) {
				assert(0);
				return(-1);
			}
			break;
		case DW_FORM_udata:		// unsigned LEB128 constant
			if (!load_uleb128(&atP, &value.uint32)) {
				assert(0);
				return(-1);
			}
			break;
		case DW_FORM_sdata:		// signed LEB128 constant
			if (!load_sleb128(&atP, &value.sint32)) {
				assert(0);
				return(-1);
			}
			break;
		case DW_FORM_block1:	// An unsigned char length followed by bytes
			if (!load_uint8(&atP, &uint8)) {
				assert(0);
				return(-1);
			}
			lth = uint8;
			goto handle_block;
		case DW_FORM_block2:	// An unsigned short length followed by bytes
			if (!load_uint16(&atP, &uint16)) {
				assert(0);
				return(-1);
			}
			lth = uint16;
			goto handle_block;
		case DW_FORM_block4:	// An unsigned int length followed by byte
			if (!load_uint32(&atP, &uint32)) {
				assert(0);
				return(-1);
			}
			lth = uint32;
			goto handle_block;
		case DW_FORM_block:		// An unsigned LEB128 length followed by bytes
			if (!load_uleb128(&atP, &uint32)) {
				assert(0);
				return(-1);
			}
			lth = uint32;
handle_block:
			for (; ((long) extraP) & 3; ++extraP);
			value.P = extraP;
			if (nodePP) {
				*((uint32T *) extraP) = lth;
			}
			extraP += sizeof(uint32T);

			if (!load_block(&atP, &extraP, nodePP != 0, lth)) {
				assert(0);
				return(-1);
			}
			break;
		case DW_FORM_string:	// inlined string in the data
			value.P = extraP;
			if (!load_string(&atP, &extraP, nodePP != 0, 1)) {
				assert(0);
				return(-1);
			}
			break;
		case DW_FORM_indirect:	// (see Section 7.5.3)
			if (!load_uleb128(&atP, &value.uint32)) {
				assert(0);
				return(-1);
			}
			form = (dw_formE) value.uint32;
			goto retry;
		default:
			assert(0);
			return(-1);
		}
		if (nodePP) {
			attributeP->name  = abbrev_attributeP->name;
			attributeP->form  = form;
			attributeP->value = value;

			if (g_debug_dwarf) {
				fprintf(stderr, " (%s %s ", get_attribute_name(abbrev_attributeP->name), get_form_name(form));
				switch(form) {
				case DW_FORM_flag:
					if (uint8) {
						fprintf(stderr, "yes");
					} else {
						fprintf(stderr, "no");
					}
					break;
				case DW_FORM_data1:
				case DW_FORM_data2:
				case DW_FORM_data4:
				case DW_FORM_udata:
				case DW_FORM_sdata:
					fprintf(stderr, "%d", value.uint32);
					break;
				case DW_FORM_addr:
				case DW_FORM_ref_addr:
				case DW_FORM_strp:
				case DW_FORM_ref1:
				case DW_FORM_ref2:
				case DW_FORM_ref4:
				case DW_FORM_ref_udata:
				case DW_FORM_ref8:
					fprintf(stderr, "%llu", value.uref);
					if (form == DW_FORM_strp) {
						char *P = get_attribute_string(attributeP);
		
						if (P) {
							fprintf(stderr, "\"%s\"", P);
						} else {
							fprintf(stderr, "NULL");
					}	}
					break;
				case DW_FORM_data8:
					fprintf(stderr, "%lld", value.uint64);
					break;
				case DW_FORM_block1:
				case DW_FORM_block2:
				case DW_FORM_block4:
				case DW_FORM_block:
					switch (attributeP->name) {
					case DW_AT_location:
						print_ops(lth, (uint8T *) value.P);
						break;
					default:
						fprintf(stderr, "%d...", lth);
					}
					break;
				case DW_FORM_string:	// inlined string in the data
					fprintf(stderr, "\"%s\"", value.P);
					break;
				case DW_FORM_null:
				case DW_FORM_indirect:
					break;
				}
				fprintf(stderr, ")");
		}	}
		++attributeP;
	}

	if (nodePP) {
		if (g_debug_dwarf) {
			fprintf(stderr, "\n");
		}
		tailPP = &nodeP->first_childP;
	} else {
		tailPP = 0;
	}

	if (abbrevP->children) {
		++m_depth;
		seen_children = 0;
		for (;;) {
			switch (load_node(&atP, &extraP, tailPP)) {
			case 1:
				if (tailPP) {
					childP = *tailPP;
					tailPP = &(childP->next_siblingP);
				}
				++seen_children;
				continue;
			case 0:
				if (!seen_children) {
					continue;
				}
				break;
			case -1:
				return(-1);
			}
			break;
		}
		--m_depth;
	}
	if (tailPP) {
		*tailPP = 0;
	}
	*atPP     = atP;
	*bufferPP = extraP;

	return(1);
} 

int
Cdwarf::load_debug_info(char *debug_infoP)
{
	char		*atP, *P;
	char		*startP, *bufferP;
	int			lth, lth1, offset, end_offset;

	m_vfunctions = -1;			// Number of virtual functions unknown
	g_offset = 0;
	atP = debug_infoP;
	if (!load_header(&atP)) {
		assert(0);
		return(0);
	}

	offset   = g_offset;
	P        = atP;
	bufferP  = 0;
	m_depth  = 0;
	if (load_node(&P, &bufferP, 0) != 1) {
		assert(0);
		return(0);
	}
	end_offset = g_offset;
	g_offset   = offset;
	lth      = bufferP - (char *) 0;

	if (m_nodes_lth < (unsigned int) lth) {
		Xfree(m_nodeP);
		m_nodeP     = (dwarf_nodeT *) Xmalloc(lth);
		m_nodes_lth = lth;
	}
	startP = bufferP = (char *) m_nodeP;

	P        = atP;
	m_depth  = 0;
	if (load_node(&P, &bufferP, &m_nodeP) != 1) {
		assert(0);
		return(0);
	}

	assert(end_offset == g_offset);
	lth1 = bufferP - startP;
	assert(lth == lth1);
	assert(m_nodeP->tag == DW_TAG_compile_unit || m_nodeP->tag == DW_TAG_partial_unit);
	return(1);
}

#ifdef DUMP_STRINGTABLE
static void
dump_stringtable(int lth, char *startP)
{
	char	*P, *endP;
	int		i;

	P = startP;
	i = 0;
	fprintf(stderr, "String table lth=%d\n", lth);
	for (endP = P + lth; P < endP; ) {
		fprintf(stderr, "%3d: \"%s\"\n", ++i, P);
		P += strlen(P) + 1;
}	}
#endif
		

int
Cdwarf::load_string_table(labelT *labelsP, char *debug_strP)
{
	char 	*stringsP, *P;
	labelT	*labelP;
	uint32T	lth;

	stringsP = 0;
	labelP   = labelsP;
	for (P = debug_strP; ; ) {
		if (!strncmp(P, " .ident", 7) || !strncmp(P, " .hidden", 8) || !strncmp(P, " .weak", 6)) {
			for (; *P++ != '\n'; );
			continue;
		}

		if (strncmp(P, " .s", 3) && strncmp(P, " .a", 3)) {
			// Assume this is a label
		 	for (;;++labelP) {
				if (!labelP->labelP) {
					assert(0);
					return(0);
				}
				if (labelP->labelP == P) {
					labelP->lineno = stringsP - ((char *) 0);
					break;
				}
				if (labelP->labelP > P) {
					assert(0);
					return(0);
			}	}
			for (; *P++ != '\n'; );
			continue;
		}
		if (!strncmp(P, " .section", 9)) {
			break;
		}
		if (!load_string(&P, &stringsP, 0, 0)) {
			assert(0);
			return(0);
	}	}
	m_strings_lth = lth = stringsP - ((char *) 0);
	if (m_strings_max < lth) {
		if (!m_strings_max) {
			m_strings_max = 1024;
		}
		do {
			m_strings_max <<= 1;
		} while (m_strings_max < lth);
		Xfree(m_stringsP);
		m_stringsP =  stringsP = (char *) Xmalloc(m_strings_max);
		if (!stringsP) {
			outofmemory();
	}	}
	stringsP = m_stringsP;
	for (P = debug_strP; ; ) {
		if (strncmp(P, " .s", 3)  && strncmp(P, " .a", 3)) {
			for (; *P++ != '\n'; );
			continue;
		}
		if (!strncmp(P, " .section", 9)) {
			break;
		}
		if (!load_string(&P, &stringsP, 1, 0)) {
			assert(0);
			return(0);
	}	}
	assert(lth == (uint32T) (stringsP - m_stringsP));
	assert(lth <= m_strings_max);
#ifdef DUMP_STRINGTABLE
	dump_stringtable(lth, m_stringsP);
#endif
	return(1);
}

void
Cdwarf::load(labelT *labelsP, char *debug_abbrevP, char *debug_infoP, char *debug_strP)
{
	m_labelsP     = labelsP;
	m_strings_lth = 0;
	m_functionP   = 0;

	if (debug_strP) {
		if (!load_string_table(labelsP, debug_strP)) {
			goto fail;
	}	}
	
	if (!load_debug_abbrev(debug_abbrevP)) {
		goto fail;
	}
	if (!load_debug_info(debug_infoP)) {
		goto fail;
	}
fail:
	return;
}

char *
Cdwarf::get_attribute_string(dwarf_attrT *attributeP)
{
	switch (attributeP->form) {
	case DW_FORM_string:
		return( attributeP->value.P);
	case DW_FORM_strp:
		assert(attributeP->value.uref < m_strings_lth);
		return(m_stringsP + attributeP->value.uref);
	default:
		assert(0);
	}
	return(0);
}

/* Find the node with offset under nodeP */

dwarf_nodeT *
Cdwarf::find_node(urefT	offset, char **under_namespacePP)
{
	dwarf_nodeT	*nodeP, *childP, *nextP;
	
	for (nodeP = m_nodeP; ; nodeP = childP) {
		if (under_namespacePP && nodeP->tag == DW_TAG_namespace) {
			*under_namespacePP = recover_name(nodeP, 0);
		}
		// Descend
		childP = nodeP->first_childP;
		if (!childP) {
			goto fail;
		}
		for (; ; childP = nextP) {
			if (childP->offset == offset) {
				return(childP);
			}
			if (childP->offset > offset) {
				goto fail;
			}
			nextP = childP->next_siblingP;
			if (!nextP || offset < nextP->offset) {
				// Descend under childP
				break;
	}	}	}
fail:
	assert(0);
	return(0);
}

dwarf_nodeT *
Cdwarf::find_specification(dwarf_attrT	*specificationP, int except, char **under_namespacePP)
{
	int offset;

	if (!specificationP) {
		return(0);
	}
	switch (specificationP->form) {
	case DW_FORM_ref1:
	case DW_FORM_ref2:
	case DW_FORM_ref4:
		offset = specificationP->value.uint32;
		if (offset != except) {
			return find_node(offset, under_namespacePP);
		}
	default:
		return 0;
}	}

/* Find the name of this node: We presume the name is not to be found
 * via yet another specification
 */

char *
Cdwarf::recover_name(dwarf_nodeT *nodeP, char **nsPP)
{
	dwarf_nodeT	*currentP;
	dwarf_attrT	*attributeP, *end_attributesP, *specificationP;
	int			attributes;

	for (currentP = nodeP; currentP; currentP = find_specification(specificationP, currentP->offset, nsPP)) {
		specificationP = 0;
		attributeP     = currentP->attributesP;
		attributes     = currentP->attributes;
		for (end_attributesP = attributeP + attributes; attributeP < end_attributesP; ++attributeP) {
			switch (attributeP->name) {
			case DW_AT_specification:
				specificationP = attributeP;
				break;
			case DW_AT_name:
				return get_attribute_string(attributeP);
			default:
				break;
	}	}	}
	return(0);
}

/* Find the signature of this node */

char *
Cdwarf::recover_signature(dwarf_nodeT *nodeP)
{
	dwarf_nodeT	*currentP;
	dwarf_attrT	*attributeP, *end_attributesP, *specificationP;
	int			attributes;


	for (currentP = nodeP; currentP; currentP = find_specification(specificationP, currentP->offset, 0)) {
		specificationP = 0;
		attributeP     = currentP->attributesP;
		attributes     = currentP->attributes;
		for (end_attributesP = attributeP + attributes; attributeP < end_attributesP; ++attributeP) {
			switch (attributeP->name) {
			case DW_AT_specification:
				specificationP = attributeP;
				break;
			case DW_AT_signature:
				return(get_attribute_string(attributeP));
			default:
				break;
	}	}	}
	return(0);
}

dwarf_attrT	*
Cdwarf::recover_frame_base(dwarf_nodeT *nodeP)
{
	dwarf_nodeT	*currentP;
	dwarf_attrT	*attributeP, *end_attributesP, *specificationP;
	int			attributes;


	for (currentP = nodeP; currentP; currentP = find_specification(specificationP, currentP->offset, 0)) {
		specificationP = 0;
		attributeP     = currentP->attributesP;
		attributes     = currentP->attributes;
		for (end_attributesP = attributeP + attributes; attributeP < end_attributesP; ++attributeP) {
			switch (attributeP->name) {
			case DW_AT_specification:
				specificationP = attributeP;
				break;
			case DW_AT_frame_base:
				return(attributeP);
			default:
				break;
	}	}	}
	return(0);
}

static void
load_frame_offset(dwarf_attrT *attributeP, locationT *locationP)
{
	uint8T			*startP, *operationP;
	uint32T			lth;
	uint8T			operation;
	int				offset;

	assert(locationP->frame_offset == -1);
	operationP  = startP = (uint8T *) attributeP->value.P;
	lth         = *((uint32T *) operationP);
	operationP += sizeof(uint32T);
	switch (operation = *operationP++) {
	case DW_OP_fbreg:
		offset = (int) decode_sleb128((uint8T **) &operationP);
		if ((operationP - startP) == (int) (lth + sizeof(uint32T))) {
			locationP->frame_offset = offset;
		}
		break;
	case DW_OP_addr: // This is a static variable
		locationP->type = LOCATION_static;
		break;
	default:		// No idea what this is
		locationP->type = LOCATION_bizarre;
		break;
}	}

static dw_opE
get_frame_base(dwarf_attrT *frame_baseP)
{
	uint8T	*blockP;
	uint32T	lth;

	switch (frame_baseP->form) {
	case DW_FORM_block1:
	case DW_FORM_block2:
	case DW_FORM_block4:
	case DW_FORM_block:
		blockP  = (uint8T *) frame_baseP->value.P;
		lth     = *((uint32T *) blockP);
		if (lth != 1) {
			break;
		}
		blockP += sizeof(uint32T);
		return ((dw_opE) *blockP);
	default:
		break;
	}
	return DW_OP_unknown;
}
			
static int
byte_size(dwarf_nodeT *nodeP)
{
	dwarf_attrT	*attributeP, *end_attributesP;

	attributeP = nodeP->attributesP;
	for (end_attributesP = attributeP + nodeP->attributes; attributeP < end_attributesP; ++attributeP) {
		if (attributeP->name == DW_AT_byte_size) {
			return attributeP->value.sint32;
	}	}
	return -1;
}

void
Cdwarf::resolve_type(dwarf_typeT *typeP, urefT typeinfo)
{
	dwarf_nodeT	*childP;
	dwarf_attrT	*attributeP, *end_attributesP;
	urefT		classinfo;
	int			is_ptr;

	assert(typeP);
	typeP->nodeP      = 0;
	typeP->dwarf_type = DW_TAG_null;
	if (!typeinfo) {
		// No idea why no typeinfo
		goto done;
	}

	for (is_ptr = 0;; typeinfo = classinfo) {
		childP = find_node(typeinfo, 0);
		if (!childP) {
			goto done;
		}
		switch (childP->tag) {
		case DW_TAG_pointer_type:
        case DW_TAG_reference_type:
			++is_ptr;
		case DW_TAG_const_type:
			attributeP = childP->attributesP;
			for (end_attributesP = attributeP + childP->attributes; ; ++attributeP) {
				if (end_attributesP <= attributeP) {
					goto done;
				}
				if (attributeP->name == DW_AT_type) {
					classinfo = attributeP->value.uref;
					break;
			}	}
			if (classinfo == typeinfo) {
				// Avoid circular references
				goto done;
			}
			continue;
		default:
			break;
		}
		break;
	}

	// Typeinfo is a pointer to a class or structure

	typeP->is_ptr      = is_ptr;
	typeP->nodeP       = childP;
	typeP->dwarf_type  = childP->tag;
	typeP->offset      = 0;
	typeP->struct_size = -2;
done:
	return;
}

static int
get_data_member_location(dwarf_attrT *attributeP)
{
	switch (attributeP->form) {
	case DW_FORM_block1:
	case DW_FORM_block2:
	case DW_FORM_block4:
	case DW_FORM_block:
		evaluate_block(attributeP);
	default:
		break;
	}
	switch (attributeP->form) {
	case DW_FORM_data4:
	case DW_FORM_ref4:
		return (attributeP->value.sint32);
	default:
		break;
	}
	return(-1);
}

void
Cdwarf::load_subprogram(dwarf_nodeT *nodeP, locationT **locationPP, int load)
{
	locationT		*locationP;
	location_typeE	type;
	dwarf_nodeT		*node1P, *childP;
	dwarf_attrT		*attributeP, *end_attributesP;
	urefT			typeinfo;

	locationP = *locationPP;
	for (childP = nodeP->first_childP; childP; childP = childP->next_siblingP) {
		switch (childP->tag) {
		case DW_TAG_lexical_block:
		{
			locationT	*location1P;
			int			low_pc, high_pc;

			location1P = locationP;
			load_subprogram(childP, &locationP, load);
			if (load) {
				attributeP = childP->attributesP;
				low_pc     = -1;
				high_pc    = -1;
				for (end_attributesP = attributeP + childP->attributes; attributeP < end_attributesP; ++attributeP) {
					switch (attributeP->name) {
					case DW_AT_low_pc:
						low_pc = (int) attributeP->value.uint64;
						break;
					case DW_AT_high_pc:
						high_pc = (int) attributeP->value.uint64;
					default:
						continue;
				}	}
				for (; location1P < locationP; ++location1P) {
					if (location1P->from_address == -1) {
						location1P->from_address = low_pc;
						location1P->to_address   = high_pc;
			}	}	}
			continue;
		}
		case DW_TAG_formal_parameter:
			type = LOCATION_parameter;
			goto common;
		case DW_TAG_variable:
			type = LOCATION_variable;
common:		if (load) {
				locationP->type			= type;
				locationP->name_ref     = 0;
				locationP->nameP		= 0;
				locationP->frame_offset = -1;
				locationP->struct_size  = 0;
				locationP->from_address = -1;
				locationP->to_address   = -1;
				locationP->is_ptr       = 0;
				locationP->classnameP   = 0;
				typeinfo   = 0;
				attributeP = childP->attributesP;
				for (end_attributesP = attributeP + childP->attributes; attributeP < end_attributesP; ++attributeP) {
					switch (attributeP->name) {
					case DW_AT_name:
						assert(!locationP->nameP);
						locationP->nameP    = get_attribute_string(attributeP);
						locationP->name_ref = childP->offset;
						continue;
					case DW_AT_location:
						load_frame_offset(attributeP, locationP);
						continue;
					case DW_AT_type:
						typeinfo = get_uref(&attributeP->value);
					default:
						continue;
				}	}
				resolve_type(locationP, typeinfo);
				if ((node1P = locationP->nodeP)) {
					locationP->classnameP = recover_name(node1P, 0);
					if (locationP->struct_size == -2) {
						locationP->struct_size = byte_size(node1P);
			}	}	}
			++locationP;
		default:
			continue;
	}	}
	*locationPP = locationP;
}

/* The specification attribute causes two distinct problems:
 *
 * (1) We may have to read attributes from the identified specification
 *     node as well as our own node in order to read all the attributes
 *     associated with a node
 * (2) We may have to read attributes from the identified specification
 *     node to determine if we are a node with the right name
 *     from which attributes should be read.  The name of a function
 *     within a namespace has to occur under the namespace node for
 *     us to know that the name is qualified with a namespace.
 *
 * Because we have no parent pointers in the dwarf structure we pass
 * down any namespace a node is under. This is 0 if not under a
 * namespace at present
 */

void
Cdwarf::load_locations(dwarf_nodeT *nodeP, char *function_signatureP, char *function_nameP, locationT **locationsPP, int load, char *under_namespaceP)
{
	dwarf_nodeT	*childP;
	dwarf_attrT	*frame_baseP;
	int			ret, lth;
	char		*nameP, *signatureP, *current_nsP;
	char		*P, *P1;

	for (childP = nodeP->first_childP; childP; childP = childP->next_siblingP) {
		switch (childP->tag) {
		case DW_TAG_namespace:
			nameP = recover_name(childP, 0);
			if (nameP) {
				lth   = strlen(nameP);
				if (function_nameP && 
					!strncmp(nameP, function_nameP, lth) &&
					function_nameP[lth]   == ':' &&
					function_nameP[lth+1] == ':') {
					load_locations(childP, function_signatureP, function_nameP, locationsPP, load, nameP);
				}
			}	
			break;
		case DW_TAG_class_type:
		case DW_TAG_structure_type:
			//load_locations(childP, signatureP, function_nameP, locationsPP, load, under_namespaceP);
			break;
		case DW_TAG_subprogram:
			signatureP = recover_signature(childP);
			if (signatureP && strcmp(function_signatureP, signatureP)) {
				current_nsP = under_namespaceP;
				signatureP  = 0;
			}
			if (!signatureP) {
				if (!function_nameP) {
					break;
				}
				current_nsP = under_namespaceP;
				nameP = recover_name(childP, &current_nsP);
				if (!nameP) {
					break;
				}
				P = function_nameP;
				if (current_nsP) {
					lth = strlen(current_nsP);
					if (strncmp(P, current_nsP, lth) || P[lth] != ':' || P[lth+1] != ':') {
						break;
					}
					P += lth + 2;
				}
				P1 = strchr(P, '(');
				if (P1) {
					*P1 = 0;
				}
				ret = strcmp(P, nameP);
				if (P1) {
					*P1 = '(';
				}
				if (ret) {
					break;
			}	}

			if (load) {
				frame_baseP = recover_frame_base(childP);
				if (frame_baseP) {
					m_frame_base = get_frame_base(frame_baseP);
			}	}
			// Load parameters/variables of this function
			load_subprogram(childP, locationsPP, load);
			// Can't assume this is the only place to recover data from
			break;
		default:
			break;
	}	}
	return;
}

void
Cdwarf::load_abstract_origins(void)
{
	dwarf_nodeT	*childP, *gchildP;
	dwarf_attrT	*attributeP, *end_attributesP;
	dwarf_attrT	*frame_baseP, *frame_offsetP;
	locationT	*locationsP, *locationP, *end_locationP;
	location_typeE type;
	int			attributes, abstract_origin;
	urefT		ref;

	locationsP    = m_locationsP;
	end_locationP = locationsP + m_locations;
	for (childP = m_nodeP->first_childP; childP; childP = childP->next_siblingP) {
		switch (childP->tag) {
		case DW_TAG_subprogram:
			abstract_origin = 0;
			frame_baseP     = 0;
			attributeP      = childP->attributesP;
			attributes      = childP->attributes;
			for (end_attributesP = attributeP + attributes; attributeP < end_attributesP; ++attributeP) {
				switch (attributeP->name) {
				case DW_AT_frame_base:
					frame_baseP = attributeP;
					break;
				case DW_AT_abstract_origin:
					abstract_origin = 1;
					break;
				default:
					break;
			}	}
			if (!abstract_origin) {
				break;
			}
			abstract_origin = 0;
			for (gchildP = childP->first_childP; gchildP; gchildP = gchildP->next_siblingP) {
				switch (gchildP->tag) {
				case DW_TAG_formal_parameter:
					type = LOCATION_parameter;
					goto common;
				case DW_TAG_variable:
					type = LOCATION_variable;
common:				ref        = 0;
					locationP  = 0;
					attributeP = gchildP->attributesP;
					attributes = gchildP->attributes;
					for (end_attributesP = attributeP + attributes; attributeP < end_attributesP; ++attributeP) {
						switch (attributeP->name) {
						case DW_AT_abstract_origin:
							ref = get_uref(&attributeP->value);
							break;
						case DW_AT_location:
							frame_offsetP = attributeP;
							break;
						default:
							break;
					}	}
					if (!ref || !frame_offsetP) {
						break;
					}
					for (locationP = locationsP; locationP < end_locationP; ++locationP) {
						if (locationP->frame_offset != -1) {
							continue;
						}
						if (locationP->name_ref != ref) {
							continue;
						}
						assert(locationP->type == type);
						abstract_origin = 1;
						load_frame_offset(frame_offsetP, locationP);
						if (frame_baseP && m_frame_base == DW_OP_unknown) {
							m_frame_base = get_frame_base(frame_baseP);
					}	}
					break;
				default:
					break;
			}	}
			if (abstract_origin) {
				goto done;
			}
			break;
		default:
			break;
	}	}
done:
#if 0
	for (locationP = locationsP; locationP < end_locationP; ++locationP) {
		if (locationP->frame_offset == -1) {
			trap_dwarf();
			break;
	}	}
#endif
	return;
}

dwarf_typeT *
Cdwarf::resolve_offset(Cfunction *functionP, int lines_seen, int frame_offset, dwarf_typeT *typeP)
{
	locationT	*locationsP, *locationP, *end_locationsP;
	int 		from_address, to_address, frame_offset1;

	locationsP = m_locationsP;
	for (end_locationsP = locationP = locationsP + m_locations; locationsP <= --locationP; ) {
		if (locationP->type >= LOCATION_static) {
			continue;
		}
		frame_offset1 = locationP->frame_offset;
		if (frame_offset >= frame_offset1 && frame_offset < frame_offset1 + locationP->struct_size) {
			from_address = locationP->from_address;
			if (from_address == -1 || from_address <= lines_seen) {
				to_address = locationP->to_address;
				if (to_address == -1 || lines_seen <= to_address) {
					*typeP = *((dwarf_typeT *) locationP);
					typeP->offset = frame_offset - frame_offset1;
					return(typeP);
	}	}	}	}
	// Can't determine location
	return (0);
}

/* Deduce the name and the type of the iten at the start of the expression
   Registers conveys the current dwarf_typeT of registers A, B, C, D
 */

dwarf_typeT *
Cdwarf::resolve(Cfunction *functionP, int lines_seen, const char *expressionP, dwarf_typeT registers[4], dwarf_typeT *typeP)
{
	int			state;
	const char	*P, *numberP;
	int			c, frame_offset, frame_adjust;

	if (!m_nodeP) {
		// Attempt to earlier load dwarf failed
		return(0);
	}

	if (functionP != m_functionP) {
		char		*signatureP  = functionP->m_nameP;
		char		*demangleP   = functionP->m_demangleP;
		locationT	*locationP   = 0;
		locationT	*end_locationsP;
		int			load, locations;

		if (*signatureP != '_' || signatureP[1] != 'Z') {
			demangleP = signatureP;
		}
		m_frame_base = DW_OP_unknown;
		for (load = 0; ; ++load) {
			load_locations(m_nodeP, signatureP, demangleP, &locationP, load, 0);
			if (load) {
				assert((locationP - m_locationsP) == locations);
				break;
			}
			m_locations = locations = locationP - (locationT *) 0;
			if (!locations) {
				break;
			}
			if (m_max_locations < locations) {
				Xfree(m_locationsP);
				m_locationsP = (locationT *) Xmalloc(locations * sizeof(locationT));
				m_max_locations = locations;
			}
			locationP   = m_locationsP;
		}
		end_locationsP = locationP;
		for (locationP = m_locationsP; locationP < end_locationsP; ++locationP) {
			if (locationP->frame_offset == -1 && locationP->name_ref > 0) {
				// Have to hate the dwarf standard -- really I ask you
				// Getting desparate
				load_abstract_origins();
				break;
		}	}

		m_functionP     = functionP;
	}

	if (!m_locations) {
		return(0);
	}

	state   = 1;
	numberP = 0;
	frame_offset = 0;
	for (P = expressionP; ; ++P) {
		switch (c = *P) {
		case '-':
			if (state == 1) {
				numberP = P;
				state   = 2;
				break;
			}
			return 0;
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
			switch (state) {
			case 1:
				numberP      = P;
			case 2:
				state        = 3;
			}
			frame_offset = frame_offset * 10 + c - '0';
			break;
		case '(':
			if (state > 3) {
				return 0;
			}

/*
64-BIT
Dwarf#  Register Name  Linux#  Solaris#  Instruction Opcode \ Field Encoding  
0  			RAX  		10  	14  		0 
1  			RDX  		12  	12 			2 
2  			RCX  		11  	13 			1 
3  			RBX  		5  		11 			3 
4  			RSI  		13  	9  			6 
5  			RDI  		14  	8  			7 
6  			RBP  		4  		10 			5 
7  			RSP  		19  	20 			4 
8  			R8  		9  		7  			8 
9  			R9  		8  		6  			9 
10  		R10  		7  		5  			10 
11  		R11  		6  		4  			11 
12  		R12  		3  		3  			12 
13  		R13  		2  		2  			13 
14  		R14  		1  		1  			14 
15  		R15  		0  		0  			15 

32 bit x86
Dwarf#	Register Name  Linux#	Solaris#	Instruction Opcode / Field Encoding
0  			EAX  		6  		11  		0  
1  			ECX  		1  		10  		1  
2  			EDX  		2  		9  			2  
3  			EBX  		0  		8  			3  
4  			UESP  		15  	17  		4  
5  			EBP  		5  		6  			5  
6  			ESI  		3  		5  			6  
7  			EDI  		4  		4  			7  

http://www.x86-64.org/documentation/abi.pdf
http://wikis.sun.com/display/SunStudio/Dwarf+Register+Numbering
 */
			if (!strncmp(P, "(%ebp)", 6)) {
				if (m_frame_base == DW_OP_reg5) {	// EBP
					frame_adjust = 0;
				} else {
					frame_adjust = -8;
				}
				if (numberP && *numberP == '-') {
					frame_offset = -frame_offset;
				}
				frame_offset += frame_adjust;
				return resolve_offset(functionP, lines_seen, frame_offset, typeP);
			} 
			if (!strncmp(P, "(%rbp)", 6)) {
				if (m_frame_base == DW_OP_reg6) {	// RBP
					frame_adjust = 0;
				} else {
					frame_adjust = -16;
				}
				if (numberP && *numberP == '-') {
					frame_offset = -frame_offset;
				}
				frame_offset += frame_adjust;
				return resolve_offset(functionP, lines_seen, frame_offset, typeP);
            } 
		default:
			return(0);
	}	}
}

/* Compute what the item we arrive at if we have some pointer in location
 * we add offset to that pointer, and then dereference the pointer at this
 * new address

 * the oldP indicates the type of what the register used to contain
 * is_ptr =  1 -> it contains the address of a pointer to the object
 * is_ptr =  0 -> it contains the address of the start of the class
 * is_ptr = -1 -> it contains the address of a vtable for the class

 */

void
Cdwarf::dereference(int shift,  dwarf_typeT *oldP, dwarf_typeT *newP)
{
	dwarf_typeT	*inputP;
	dwarf_nodeT	*old_nodeP;
	int			offset, offset1;
	dwarf_attrT	*attributeP, *end_attributesP;
	int			attributes, best_offset, struct_size, n;
	char		*nameP;
	urefT		typeinfo, has_vtable;
	dw_tagE		dwarf_type;

	dwarf_nodeT	*childP, *bestP;
	old_nodeP   = oldP->nodeP;
	if (!old_nodeP) {
		// If old has unknown type so does new
		goto fail;
	}
	offset = oldP->offset + shift;

	if (oldP->is_ptr != 1) {
		switch (oldP->is_ptr) {
		case -1:
			// oldP register already addresses start of function table
			goto fail;
		case 0:
			// Have the address of pointer to class vtable in oldP
			// This dereference turns it into a pointer to a function
			newP->nodeP  = oldP->nodeP;
			newP->is_ptr = -1;
			newP->offset = offset;
			break;
		default:
			// Presume any offset is simply indexing into same underlying type
			// We are referencing a pointer so offset should be multiple 4 or 8
			assert(!(offset & 3));
			newP->nodeP  = oldP->nodeP;
			newP->is_ptr = oldP->is_ptr - 1;
			newP->offset = 0;
			newP->struct_size = oldP->struct_size;
			break;
		}
		return;
	}

	// Decide what we now have in register

	for (inputP = oldP; (dwarf_type = inputP->dwarf_type) == DW_TAG_structure_type || dwarf_type == DW_TAG_class_type; inputP = newP) {
		struct_size = inputP->struct_size;
		if (struct_size <= 0) {
			if (struct_size == -2) {
				inputP->struct_size = struct_size = byte_size(old_nodeP);
		}	}
		if (struct_size > 0) {
			if (offset < 0) {
				// struct_offset + struct_size * n >= 0
				// -struct_offset <= struct_size * n
				// -struct_offset/struct_size <= n
		
				n = (struct_size - 1 - offset) / struct_size;	// round up
				offset += n * struct_size;
				assert(offset >= 0 && offset < struct_size);
			}
			if (offset >= struct_size) {
				// Presume an array of structs
				offset %= struct_size;
		}	}
	/* 
typedef struct {
    int a[5];
    struct s1S {
        int b;
        A   *P;
    } x;
} sT;

49: structure_type (name strp 105"<anonymous struct>") (byte_size data1 40) (decl_file data1 1) (decl_line data1 30) (sibling ref4 123)
	61: structure_type (name string "s1S") (byte_size data1 16) (decl_file data1 1) (decl_line data1 32) (sibling ref4 98)
		73: member (name string "b") (decl_file data1 1) (decl_line data1 33) (type ref4 123) (data_member_location block1 2...)
		85: member (name string "P") (decl_file data1 1) (decl_line data1 34) (type ref4 265) (data_member_location block1 2...)
	98: member (name string "a") (decl_file data1 1) (decl_line data1 31) (type ref4 271) (data_member_location block1 2...)
    110: member (name string "x") (decl_file data1 1) (decl_line data1 35) (type ref4 61) (data_member_location block1 2...)
	*/
		bestP      = 0;
		has_vtable = 0;
		for (childP = old_nodeP->first_childP; childP; childP = childP->next_siblingP) {
			switch (childP->tag) {
			case DW_TAG_subprogram:
				if (!has_vtable) {
					attributeP = childP->attributesP;
					attributes = childP->attributes;
					for (end_attributesP = attributeP + attributes; attributeP < end_attributesP; ++attributeP) {
						if (attributeP->name == DW_AT_vtable_elem_location) {
							// Class has vtable
							has_vtable = 1;
							break;
				}	}	}
				break;
			case DW_TAG_member:
				if (struct_size <= 0) {
					continue;
				}
				attributeP     = childP->attributesP;
				attributes     = childP->attributes;
				for (end_attributesP = attributeP + attributes; attributeP < end_attributesP; ++attributeP) {
					if (attributeP->name == DW_AT_data_member_location) {
						offset1 = get_data_member_location(attributeP);
						if (offset1 <= offset) {
							if (!bestP || offset1 > best_offset) {
								bestP       = childP;
								best_offset = offset1;
						}	}
						break;
				}	}
			default:
				break;
		}	}
		if (!bestP) {
			if (has_vtable) {
				// Didn't find any suitable member variables but did discover
				// at least one virtual function
				newP->nodeP  = old_nodeP;
				newP->is_ptr = 0;
				newP->offset = 0;
				newP->struct_size = -2;
				return;
			}
			goto fail;
		}
		attributeP     = bestP->attributesP;
		attributes     = bestP->attributes;
		typeinfo       = 0;
		nameP          = 0;
		for (end_attributesP = attributeP + attributes; attributeP < end_attributesP; ++attributeP) {
			switch (attributeP->name) {
			case DW_AT_name:
				nameP = get_attribute_string(attributeP);
				break;
			case DW_AT_type:
				typeinfo = get_uref(&attributeP->value);
			default:
				break;
		}	}
		newP->nameP = nameP;
		if (nameP && !strncmp(nameP, "_vptr.", 6)) {
			newP->nodeP  = old_nodeP;
			newP->is_ptr = 0;
			newP->offset = 0;
			newP->struct_size = -2;
			return;
		}
		assert(typeinfo);
		resolve_type(newP, typeinfo);
		old_nodeP = newP->nodeP;
		if (!old_nodeP) {
			break;
		}
		if (newP->is_ptr) {
			return;
		}
		newP->offset = offset = offset - best_offset;
	}
fail:
	newP->nodeP = 0;
}

// We must delay loading until after all debug information is available
// because things like signature might involve references to other nodes

int
Cdwarf::retrieve_vfunction(dwarf_nodeT *nodeP, char **signaturePP)
{
	char 		*signatureP;
	int			vtable_offset;
	dwarf_nodeT	*currentP;
	dwarf_attrT	*attributeP, *end_attributesP, *specificationP;

	signatureP    = 0;
	vtable_offset = -1;

	for (currentP = nodeP; currentP; currentP = find_specification(specificationP, currentP->offset, 0)) {
		specificationP = 0;
		attributeP     = currentP->attributesP;
		for (end_attributesP = attributeP + currentP->attributes; attributeP < end_attributesP; ++attributeP) {
			switch (attributeP->name) {
			case DW_AT_specification:
				specificationP = attributeP;
				break;
			case DW_AT_signature:
				signatureP = get_attribute_string(attributeP);
				if (signatureP[0] != '_' || signatureP[1] != 'Z') {
					return(-1);
				}
				if (0 <= vtable_offset) {
					goto found;
				}
				break;
			case DW_AT_vtable_elem_location:
				switch (attributeP->form) {
				case DW_FORM_block1:
				case DW_FORM_block2:
				case DW_FORM_block4:
				case DW_FORM_block:
					evaluate_block(attributeP);
				default:
					break;
				}
				switch (attributeP->form) {
				case DW_FORM_data4:
				case DW_FORM_ref4:
					vtable_offset = attributeP->value.uint32;
					if (0 <= vtable_offset) {
						if (signatureP) {
							goto found;
						}
						break;
					}
				default:
					return(-1);
				}
			default:
				break;
	}	}	}
	return(-1);
found:
	*signaturePP = signatureP;
	return(vtable_offset);
}

// A possible class identified by reg_location is having its vtable offset
// by reg_offset and is then having the result called.  Determine what
// function name is being called.

char *
Cdwarf::resolve_vfunction(dwarf_typeT *typeP)
{
	dwarf_nodeT	*nodeP, *childP;
	unsigned int entry;
	char		*signatureP;
	int			vtable_offset;

	if (!(nodeP = typeP->nodeP)) {
		return 0;
	}
	entry = typeP->offset / m_header.address_size;

	for (childP = nodeP->first_childP; ; childP = childP->next_siblingP) {
		if (!childP) {
			// Couldn't find declaration of correct virtual function
			trap_dwarf();
			return 0;
		}
		if (childP->tag == DW_TAG_subprogram) {
			vtable_offset = retrieve_vfunction(childP, &signatureP);
			if (vtable_offset < 0) {
				continue;
			}
			if (entry == (uint32T) vtable_offset) {
				return(signatureP);
		}	}
	}	

	// Look for childP->tag == DW_TAG_inheritance to find direct subclasses

	return 0;
}

// We must delay loading until after all debug information is available
// because things like signature might involve references to other nodes

void
Cdwarf::load_vfunctions(dwarf_nodeT *nodeP)
{
	char 		*signatureP;
	int			vtable_offset;
	vfunctionT	*vfunctionP;
	dwarf_nodeT *childP;

	if (nodeP->tag == DW_TAG_subprogram) {
		vtable_offset = retrieve_vfunction(nodeP, &signatureP);
		if (0 <= vtable_offset) {
			if (++m_vfunctions >= m_vfunctions_max) {
				if (!m_vfunctions_max) {
					m_vfunctions_max = 512;
					m_vfunctionsP    = (vfunctionT *) Xmalloc(sizeof(vfunctionT) * m_vfunctions_max);
				} else {
					m_vfunctions_max <<= 1;
					m_vfunctionsP    = (vfunctionT *) Xrealloc(m_vfunctionsP, sizeof(vfunctionT) * m_vfunctions_max);
			}	}
			vfunctionP                 = m_vfunctionsP + m_vfunctions;
			vfunctionP->signatureP    = signatureP;
			vfunctionP->vtable_offset = vtable_offset;
	}	}

	for (childP = nodeP->first_childP; childP; childP = childP->next_siblingP) {
		load_vfunctions(childP);
}	}

// Find the name of a pure virtual function
// Have to use a stupid approach because classes are not assigned a
// signature and templated classes are ugly to parse

char *
Cdwarf::pure_virtual_function_name(const char *vtable_nameP, int vtable_offset) 
{
	const char	*classP;
	char 		*signatureP, *signature1P;
	int			lth;
	vfunctionT	*vfunctionP, *endP;

	if (strncmp(vtable_nameP, "_ZTV", 4)) {
		// Not a C++ function
		return 0;
	}

	if (m_vfunctions == -1) {
		m_vfunctions = 0;
		load_vfunctions(m_nodeP);
	}

	// Skip _ZTV
	classP = vtable_nameP + 4;
	if (*classP == 'N') {
		++classP;
	}
	lth    = strlen(classP);

	vfunctionP = m_vfunctionsP;
	for (endP = vfunctionP + m_vfunctions; vfunctionP < endP; ++vfunctionP) {
		if (vfunctionP->vtable_offset == vtable_offset) {
			signatureP   = signature1P = vfunctionP->signatureP;
			signature1P += 2;
			if (*signature1P == 'N') {
				++signature1P;
			}
			if (!strncmp(signature1P, classP, lth)) {
				return signatureP;
	}	}	}
	return(0);
}
#endif
