#include <assert.h>
#include <malloc.h>
#include <stdio.h>
#include <string.h>

#include "xmalloc.h"
#include "util.h"
#include "signature_buffer.h"
#include "signature.h"

//#define TESTING
//#define TESTING1
//#define TESTING2

void
Csignature_node::dump(FILE *F, int entering, int indent, int levels)
{
	int 			i;
	Csignature_node	*childP;
	const char		*P, *endP;

	for (i = 0; i < indent; ++i) {
		putc(' ', F);
	}
	switch (entering) {
	case 1:
		fputs("> ", F);
		break;
	case 2:
		fputs("< ", F);
		break;
	case 3:
		fputs("+ ", F);
		break;
	}
	fprintf(F, "%p ", this);

	switch (m_type) {
	case SIGNATURE_substitution:
		fputs("substitution", F);
		break;
	case SIGNATURE_number:
		fputs("number", F);
		break;
	case SIGNATURE_compact_number:
		fputs("compact_number", F);
		break;
	case SIGNATURE_discriminator:
		fputs("discriminator", F);
		break;
	case SIGNATURE_call_offset:
		fputs("call_offset", F);
		break;
	case SIGNATURE_identifier:
		fputs("identifier", F);
		break;
	case SIGNATURE_data_member_prefix:
		fputs("data_member_prefix", F);
		break;
	case SIGNATURE_source_name:
		fputs("source_name", F);
		break;
	case SIGNATURE_digit:
		fputs("digit", F);
		break;
	case SIGNATURE_operator_name:
		fputs("operator_name", F);
		break;
	case SIGNATURE_ctor_dtor_name:
		fputs("ctor_dtor_name", F);
		break;
	case SIGNATURE_closure_type_name:
		fputs("closure_type_name", F);
		break;
	case SIGNATURE_unnamed_type_name:
		fputs("unnamed_type_name", F);
		break;
	case SIGNATURE_unqualified_name:
		fputs("unqualified_name", F);
		break;
	case SIGNATURE_template_template_param:
		fputs("template_template_param", F);
		break;
	case SIGNATURE_template_param:
		fputs("template_param", F);
		break;
	case SIGNATURE_expr_primary:
		fputs("expr_primary", F);
		break;
	case SIGNATURE_template_prefix:
		fputs("template_prefix", F);
		break;
	case SIGNATURE_template_arg:
		fputs("template_arg", F);
		break;
	case SIGNATURE_template_args:
		fputs("template_args", F);
		break;
	case SIGNATURE_simple_id:
		fputs("simple_id", F);
		break;
	case SIGNATURE_destructor_name:
		fputs("destructor_name",F );
		break;
	case SIGNATURE_base_unresolved_name:
		fputs("base_unresolved_name", F);
		break;
	case SIGNATURE_decltype:
		fputs("decltype", F);
		break;
	case SIGNATURE_unresolved_type:
		fputs("unresolved_type", F);
		break;
	case SIGNATURE_unresolved_qualifier_level:
		fputs("unresolved_qualifier_level", F);
		break;
	case SIGNATURE_unresolved_name:
		fputs("unresolved_name", F);
		break;
	case SIGNATURE_cv_qualifiers:
		fputs("cv_qualifiers", F);
		break;
	case SIGNATURE_function_param:
		fputs("function_param", F);
		break;
	case SIGNATURE_initializer:
		fputs("initializer", F);
		break;
	case SIGNATURE_expression:
		fputs("expression", F);
		break;
	case SIGNATURE_prefix:
		fputs("prefix", F);
		break;
	case SIGNATURE_nested_name:
		fputs("nested_name", F);
		break;
	case SIGNATURE_local_name:
		fputs("local_name", F);
		break;
	case SIGNATURE_St:
		fputs("St", F);
		break;
	case SIGNATURE_unscoped_name:
		fputs("unscoped_name", F);
		break;
	case SIGNATURE_unscoped_template_name:
		fputs("unscoped_template_name", F);
		break;
	case SIGNATURE_name:
		fputs("name", F);
		break;
	case SIGNATURE_class_enum_type:
		fputs("class_enum_type", F);
		break;
	case SIGNATURE_function_type:
		fputs("function_type", F);
		break;
	case SIGNATURE_array_type:
		fputs("array_type", F);
		break;
	case SIGNATURE_pointer_to_member_type:
		fputs("pointer_to_member_type", F);
		break;
	case SIGNATURE_fixed_point_type:
		fputs("fixed_point_type", F);
		break;
	case SIGNATURE_vector_type:
		fputs("vector_type", F);
		break;
	case SIGNATURE_type:
		fputs("type", F);
		break;
	case SIGNATURE_bare_function_type:
		fputs("bare_function_type", F);
		break;
	case SIGNATURE_java_resource:
		fputs("java_resource", F);
		break;
	case SIGNATURE_special_name:
		fputs("special_name", F);
		break;
	case SIGNATURE_encoding:
		fputs("encoding", F);
		break;
	case SIGNATURE_mangled_name:
		fputs("mangled_name", F);
		break;
	case SIGNATURE_root:
		fputs("root", F);
		break;
	default:
		fprintf(F, "???");
	}
	endP = m_endP;
	if (endP) {
		fputs(" \"", F);
		for (P = m_startP; P < endP; ++P) {
			fputc(*P, F);
		}
		fputs("\"", F);
	} else {
		fprintf(F, " '%s'", m_startP);
	}
		
	if (m_value) {
		fprintf(F, " (%ld)", m_value);
	}
	fputc('\n', F);
		
	if (!levels) {
		return;
	}
	for (childP = m_childP; childP; childP = childP->m_siblingP) {
		childP->dump(F, entering, indent+2, levels);
}	}

void
Csignature_node::reverse(void)
{
	Csignature_node	*childP, *lastP, *nextP;

	lastP = 0;
	if (m_type == SIGNATURE_substitution) {
		// Don't reverse things under a substitution
		// They get reversed in the tree outside substitution
		return;
	}
	for (childP = m_childP; childP; childP = nextP ) {
		childP->reverse();
		nextP              = childP->m_siblingP;
		childP->m_siblingP = lastP;
		lastP              = childP;
	}
	m_childP = lastP;
}

class Csignature_error {
};

class Csignature_expand {
};

class Cnew_node {
public:
	Csignature	*m_signatureP;

	Cnew_node(Csignature *signatureP, signature_typeE type)
	{
		Csignature_node	*nodeP, *childP;

		m_signatureP          = signatureP;
		nodeP                 = signatureP->m_nodeP;
		childP                = signatureP->get_node();
		childP->m_parentP     = nodeP;
		childP->m_type        = type;
		childP->m_startP      = signatureP->m_atP;
		childP->m_siblingP    = nodeP->m_childP;
		nodeP->m_childP       = childP;
		signatureP->m_nodeP   = childP;

#ifdef TESTING1
		int indent;

		indent = 0;
		for (nodeP = childP; (nodeP = nodeP->m_parentP); ) {
			++indent;
		}
		childP->dump(stderr, 1, indent, 0);
#endif
	}

	~Cnew_node(void) 
	{
		Csignature		*signatureP;
		Csignature_node	*nodeP;

		signatureP          = m_signatureP;
		nodeP               = signatureP->m_nodeP;
		nodeP->m_endP       = signatureP->m_atP;
		signatureP->m_nodeP = nodeP->m_parentP;
#ifdef TESTING1
		int indent = 0;
		Csignature_node	*parentP;

		for (parentP = nodeP; (parentP = parentP->m_parentP); ) {
			++indent;
		}
		nodeP->dump(stderr, 2, indent, 0);
#endif
	}
};

#define MAKE_NODE(X) Cnew_node new_node(this, SIGNATURE_ ## X)
#define LAST_SEEN(X) last_seen(SIGNATURE_ ## X)
#define ALL_SEEN(X)  all_seen(SIGNATURE_ ## X)

Csignature::Csignature(void)
{
	m_nodesP        = 0;
	m_nodes_endP    = 0;
	m_nodes_max     = 0;
}

Csignature::~Csignature(void)
{
	Xfree(m_nodesP);
}

Csignature_node *
Csignature::get_node(void)
{
	Csignature_node *nodeP;

	nodeP = --m_nodes_endP;
	if (nodeP < m_nodesP) {
		throw Csignature_expand();
	}
	nodeP->m_parentP  = 0;
	nodeP->m_childP   = 0;
	nodeP->m_siblingP = 0;
	nodeP->m_nextP    = 0;
	nodeP->m_value    = 0;
	return nodeP;
}

// Push the last thing seen under a node of the indicated type

void
Csignature::last_seen(signature_typeE type)
{
	Csignature_node	*newP;
	Csignature_node	*childP;

	childP             = m_nodeP->m_childP;	// Last seen
	childP->m_endP     = m_atP;
	newP               = get_node();
	newP->m_type       = type;
	newP->m_parentP    = childP->m_parentP;
	newP->m_startP     = childP->m_startP;
	newP->m_siblingP   = childP->m_siblingP;
	newP->m_endP       = m_atP;
	newP->m_childP     = childP;
	childP->m_parentP  = newP;
	childP->m_siblingP = 0;
	m_nodeP->m_childP  = newP;
}

// Push everything seen in the current production under a new of this type

void
Csignature::all_seen(signature_typeE type)
{
	Csignature_node	*newP;
	Csignature_node	*childP, *nextP;

	childP             = m_nodeP->m_childP;
	childP->m_endP     = m_atP;
	newP               = get_node();
	newP->m_type       = type;
	newP->m_parentP    = childP->m_parentP;
	newP->m_endP       = m_atP;
	newP->m_childP     = childP;
	for (;; childP = nextP) {
		childP->m_parentP = newP;
		if (!(nextP = childP->m_siblingP)) {
			break;
	}	}
	newP->m_startP = childP->m_startP;
	m_nodeP->m_childP = newP;
}
		
void
Csignature::fail(void)
{
	assert(0);
	throw Csignature_error();
}

void
Csignature::dump(FILE *F)
{
	fprintf(F, "%ld nodes in signature\n", m_nodes_max - (m_nodes_endP - m_nodesP));
	m_nodeP->dump(F, 0, 0, 1);
}

void
Csignature::add_dictionary(Csignature_node *nodeP)
{
	switch(nodeP->m_type) {
	case SIGNATURE_prefix:
	case SIGNATURE_template_prefix:
	case SIGNATURE_type:
	case SIGNATURE_template_template_param:
	case SIGNATURE_unresolved_type:
	case SIGNATURE_unscoped_template_name:
		break;
	default:
		assert(0);
		return;
	}

	if (!m_headP) {
		m_headP = nodeP;
	} else {
		m_tailP->m_nextP = nodeP;
	}
	m_tailP = nodeP;
#ifdef TESTING2
	{
		Csignature_node	*node1P;
		int				cnt = -1;

		nodeP->m_endP = m_atP;
		for (node1P = m_headP; node1P; node1P = node1P->m_nextP) {
			++cnt;
		}
		fprintf(stderr, "Added to dictionary entry %d\n", cnt);
		nodeP->dump(stderr, 3, 0, 1);
		fprintf(stderr, "--Added--\n");
	}
#endif
}

// <substitution> ::= S <seq-id> _
//                ::= S_

void
Csignature::substitution(void)
{
	MAKE_NODE(substitution);

	int				c, offset;
	Csignature_node	*nodeP;

	if (*m_atP++ != 'S') {
		fail();
	}
	switch (*m_atP++) {
	case 't':	// St # ::std::
   	case 'a':	// Sa # ::std::allocator
   	case 'b':	// Sb # ::std::basic_string
   	case 's':	// Ss # ::std::basic_string < char, ::std::char_traits<char>, ::std::allocator<char> >
				//    # ::std::string
	case 'i':	// Si # ::std::basic_istream<char,  std::char_traits<char> >
				//    # ::std::istream
   	case 'o':	// So # ::std::basic_ostream<char,  std::char_traits<char> >
				//    # ::std::ostream
   	case 'd':	// Sd # ::std::basic_iostream<char, std::char_traits<char> >
				//	  # ::std::iostream
		m_nodeP->m_value = -1;
		return;
	}
	offset = 0;
	if (*--m_atP != '_') {
		for (; ((c = *m_atP) >= '0' && c <= '9') || (c >= 'A' && c <= 'Z'); ++m_atP) {
			offset = offset * 36 + c;
			if (c <= '9') {
				offset -= '0';
			} else {
				offset -= 'A' - 10;
		}	}
		if (c != '_') {
			fail();
		}
		++offset;
	}
	++m_atP;

	m_nodeP->m_value = offset;
	for (nodeP = m_headP; --offset >= 0; nodeP = nodeP->m_nextP) {
		if (!nodeP) {
			fail();
	}	}
	m_nodeP->m_childP = nodeP;
}

long
Csignature::number(void)
{
	MAKE_NODE(number);
	long	val, sign;
	int		c;

	if (*m_atP == 'n') {	// Negative
		sign = -1;
		++m_atP;
	} else {
		sign = 1;
	}
	// Recover the value
	for (val = 0; (c = *m_atP) >= '0' && c <= '9'; ++m_atP) {
		val = val * 10 + c - '0';
	}
	if (sign < 0) {
		val = -val;
	}
	m_nodeP->m_value = val;
	return(val);
}

long
Csignature::compact_number(void)
{
	MAKE_NODE(compact_number);
	long	val;
	int		c;

	if (*m_atP == '_') {
		val = 0;
	} else {
		// Recover the value
		for (val = 0; (c = *m_atP) >= '0' && c <= '9'; ++m_atP) {
			val = val * 10 + c - '0';
		}
		if (*m_atP != '_') {
			fail();
		}
		++val;
	}
	++m_atP;
	m_nodeP->m_value = val;
	return(val);
}

// <discriminator> := _ <non-negative number> 

void
Csignature::discriminator(void)
{
	MAKE_NODE(discriminator);
	long val;


	if (*m_atP++ != '_') {
		fail();
	}
	val = number();
	if (val < 0) {
		fail();
}	}

// <call-offset> ::= h <nv-offset> _
//        		 ::= v <v-offset> _
//   <nv-offset> ::= <offset number>
//      		     # non-virtual base override
//   <v-offset>  ::= <offset number> _ <virtual offset number>
//      	         # virtual base override, with vcall offset

void
Csignature::call_offset(void)
{
	MAKE_NODE(call_offset);

	int 	i, max;

	switch(*m_atP++) {
	case 'h':
		max = 1;
		break;
	case 'v':
		max = 2;
		break;
	default:
		fail();
	}
	for (i = 0; i < max; ++i) {
		// Recover the call offset
		number();
		if (*m_atP++ != '_') {
			fail();
	}	}
}

void
Csignature::identifier(long lth)
{
	MAKE_NODE(identifier);

	m_atP += lth;
}

// <source-name> ::= <positive length number> <identifier>

void
Csignature::source_name(void)
{
	MAKE_NODE(source_name);
	long	lth;

	lth = number();
	if (lth <= 0) {
		fail();
	}
	identifier(lth);
}

int
Csignature::digit(void)
{
	MAKE_NODE(digit);
	int	c;

	if ((c = *m_atP++) < '0' || c > '9') {
		fail();
	}
	c -= '0';
	m_nodeP->m_value = c;
	return(c);
}

// Returns the number of arguments

#define NL(s) s, (sizeof s) - 1

const struct operatorT operators[] =
{
  { "aN", NL ("&="),        2 },
  { "aS", NL ("="),         2 },
  { "aa", NL ("&&"),        2 },
  { "ad", NL ("&"),         1 },
  { "an", NL ("&"),         2 },
  { "cl", NL ("()"),        2 },
  { "cm", NL (","),         2 },
  { "co", NL ("~"),         1 },
  { "dV", NL ("/="),        2 },
  { "da", NL ("delete[]"),  1 },
  { "de", NL ("*"),         1 },
  { "dl", NL ("delete"),    1 },
  { "dt", NL ("."),         2 },
  { "dv", NL ("/"),         2 },
  { "eO", NL ("^="),        2 },
  { "eo", NL ("^"),         2 },
  { "eq", NL ("=="),        2 },
  { "ge", NL (">="),        2 },
  { "gt", NL (">"),         2 },
  { "ix", NL ("[]"),        2 },
  { "lS", NL ("<<="),       2 },
  { "le", NL ("<="),        2 },
  { "ls", NL ("<<"),        2 },
  { "lt", NL ("<"),         2 },
  { "mI", NL ("-="),        2 },
  { "mL", NL ("*="),        2 },
  { "mi", NL ("-"),         2 },
  { "ml", NL ("*"),         2 },
  { "mm", NL ("--"),        1 },
  { "na", NL ("new[]"),     1 },
  { "ne", NL ("!="),        2 },
  { "ng", NL ("-"),         1 },
  { "nt", NL ("!"),         1 },
  { "nw", NL ("new"),       1 },
  { "oR", NL ("|="),        2 },
  { "oo", NL ("||"),        2 },
  { "or", NL ("|"),         2 },
  { "pL", NL ("+="),        2 },
  { "pl", NL ("+"),         2 },
  { "pm", NL ("->*"),       2 },
  { "pp", NL ("++"),        1 },
  { "ps", NL ("+"),         1 },
  { "pt", NL ("->"),        2 },
  { "qu", NL ("?"),         3 },
  { "rM", NL ("%="),        2 },
  { "rS", NL (">>="),       2 },
  { "rm", NL ("%"),         2 },
  { "rs", NL (">>"),        2 },
  { "st", NL ("sizeof "),   1 },
  { "sz", NL ("sizeof "),   1 },
  { "at", NL ("alignof "),   1 },
  { "az", NL ("alignof "),   1 },
  { NULL, NULL, 0,          0 }
};


/* static */ const operatorT *
Csignature::lookup_operator(char c1, char c2)
{
	int	 			low, high, i;
	const operatorT	*operatorP;

	/* LOW is the inclusive lower bound.  */
	low = 0;
	/* HIGH is the exclusive upper bound.  We subtract one to ignore
	 the sentinel at the end of the array.  */
	high = ((sizeof(operators) / sizeof (operators[0])) - 1);
	for (;;) {
		i = low + (high - low) / 2;
	  	operatorP = operators + i;

	  	if (c1 == operatorP->codeP[0] && c2 == operatorP->codeP[1]) {
			return operatorP;
		}
	  	if (c1 < operatorP->codeP[0] || (c1 == operatorP->codeP[0] && c2 < operatorP->codeP[1])) {
	    	high = i;
		} else {
	    	low = i + 1;
		}
		if (low == high) {
	    	return 0;
}	}	}

int
Csignature::operator_name(void)
{
	MAKE_NODE(operator_name);
  	char 			c1, c2, ret;
	const operatorT	*operatorP;

  	c1 = *m_atP++;
	c2 = *m_atP++;

	switch (c1) {
	case 'v':
		// ::= v <digit> <source-name>	# vendor extended operator
		if (c2 >= '0' && c2 <= '9') {
			--m_atP;
			digit();
			ret = m_nodeP->m_childP->m_value;
			source_name();
			m_nodeP->m_value = -1;
			goto done;
		}
		break;
	case 'c':
		// ::= cv <type>	# (cast)        
		if (c2 == 'v') {
			ret = -1;
			type();
			m_nodeP->m_value = -2;
			goto done;
		}
		break;
	}
	operatorP = lookup_operator(c1, c2);
	if (!operatorP) {
		fail();
	}
	m_nodeP->m_value = operatorP - operators;
	ret = operatorP->args;
done:
	// Not substitutable
	return(ret);
}

// <ctor-dtor-name> ::= C1	# complete object constructor
//             	    ::= C2	# base object constructor
//					::= C3	# complete object allocating constructor
//					::= D0	# deleting destructor
//					::= D1	# complete object destructor
//					::= D2	# base object destructor

void
Csignature::ctor_dtor_name(void)
{
	MAKE_NODE(ctor_dtor_name);
	int	c;

	switch (*m_atP++) {
	case 'C':
		if ((c = *m_atP++) < '1' || c > '3') {
			fail();
		}
		break;
	case 'D':
		if ((c = *m_atP++) < '0' && c > '2') {
			fail();
		}
		break;
	default:
		fail();
}	}

// <closure-type-name> ::= Ul <lambda-sig> E [ <nonnegative number> ] _ 

void
Csignature::closure_type_name(void)
{
	MAKE_NODE(closure_type_name);

	if (*m_atP++ != 'U' || *m_atP++ != 'l') {
		fail();
	}
	bare_function_type();
	if (*m_atP != 'E') {
		fail();
	}
	++m_atP;
	compact_number();
}

// <unnamed-type-name> ::= Ut [ <nonnegative number> ] _ 
//                     ::= <closure-type-name>

void
Csignature::unnamed_type_name(void)
{
	MAKE_NODE(unnamed_type_name);

	if (*m_atP++ != 'U') {
		fail();
	}
	switch (*m_atP++) {
	case 't':
		compact_number();
		break;
	case 'l':
		// <closure-type-name> ::= Ul <lambda-sig> E [ <nonnegative number> ] _ 
		closure_type_name();
		break;
	default:
		fail();
}	}

// <unqualified-name> ::= <operator-name>
//                    ::= <ctor-dtor-name>  
//                    ::= <source-name>   
//					  ::= <unnamed-type-name>

void
Csignature::unqualified_name(int maybe_data_member_prefix)
{
	MAKE_NODE(unqualified_name);
	int		c;

	switch ((c = *m_atP)) {
	case 'C':
	case 'D':
		ctor_dtor_name();
		break;
	case 'L':		// gnu extension
		++m_atP;
		source_name();
		if (*m_atP == '_') {
			discriminator();
		}
		break;
	case 'U':
		switch (m_atP[1]) {
		case 'l':
		case 't':
			unnamed_type_name();
			break;
		default:
			fail();
		}
		break;
	default:
		if (c >= '0' && c <= '9') {
			source_name();
			if (maybe_data_member_prefix && *m_atP == 'M') {
				LAST_SEEN(data_member_prefix);
				++m_atP;
			}
			break;
		}
		if (c >= 'a' && c <= 'z') {
			// <operator-name>
			operator_name();
			break;
		}
		fail();
	}
}

// <template-param> ::= T_	# first template parameter
//    		        ::= T <parameter-2 non-negative number> _

void
Csignature::template_param(void)
{
	MAKE_NODE(template_param);
	
	if (*m_atP++ != 'T') {
		fail();
	}
	m_nodeP->m_value = compact_number();
}

//  <expr-primary> ::= L <type> <value number> E   # integer literal
//                 ::= L <type <value float> E     # floating literal
//                 ::= L <mangled-name> E          # external name

void
Csignature::expr_primary(void)
{
	MAKE_NODE(expr_primary);
	int	c;

	if (*m_atP++ != 'L') {
		fail();
	}
	if (*m_atP == '_' && m_atP[1] == 'Z') {
		mangled_name();
	} else {
		type();
		if (*m_atP == 'n') {
			++m_atP;
		}
		for (; ((c = *m_atP) >= '0' && c <= '9') || (c >= 'a' && c <= 'f'); ++m_atP);
	}
	if (*m_atP != 'E') {
		fail();
	}
	++m_atP;
}

//  <template-arg> ::= <type>			# type or template
//			  	   ::= X <expression> E	# expression
//                 ::= <expr-primary>   # simple expressions
//				   ::= J <template-arg>* E

void
Csignature::template_arg(void)
{
	MAKE_NODE(template_arg);

	switch (*m_atP) {
	case 'X':
		++m_atP;
		expression();
		if (*m_atP != 'E') {
			fail();
		}
		++m_atP;
		break;
	case 'L':
		expr_primary();
		break;
	case 'J':
		++m_atP;
		for (; *m_atP != 'E';) {
			template_arg();
		}
		++m_atP;
		break;
	default:
		type();
		break;
	}
}

// <template-args> ::= I <template-arg>+ E

void
Csignature::template_args(void)
{
	MAKE_NODE(template_args);
	int			args;

	if (*m_atP++ != 'I') {
		fail();
	}
	for (args = 0; *m_atP != 'E'; ++args) {
		template_arg();
	}
	++m_atP;
	if (args < 1) {
		fail();
	}
}

// <simple-id> ::= <source-name> [ <template-args> ]

void
Csignature::simple_id(void)
{
	MAKE_NODE(simple_id);

	source_name();
	if (*m_atP == 'I') {
		template_args();
	}
	return;
}

// <destructor-name> ::= <unresolved-type> # e.g., ~T or ~decltype(f())
//                   ::= <simple-id>       # e.g., ~A<2*N>

void
Csignature::destructor_name(void)
{
	MAKE_NODE(destructor_name);

	switch (*m_atP) {
	case 'T':
	case 'D':
	case 'S':
		unresolved_type();
		break;
	default:
		simple_id();
}	}

// <base-unresolved-name> ::= <simple-id> 
//                            # unresolved name
//                        ::= on <operator-name> 
//                            # unresolved operator-function-id
//                        ::= on <operator-name> <template-args>
//                            # unresolved operator template-id
//                        ::= dn <destructor-name>
//                            # destructor or pseudo-destructor;
//                            # e.g. ~X or ~X<N-1>


void
Csignature::base_unresolved_name(void)
{
	MAKE_NODE(base_unresolved_name);
	
	switch (*m_atP) {
	case 'o':	// on <operator-name>  # unresolved operator-function-id
				// on <operator-name> <template-args>  # unresolved operator template-id
		if (*++m_atP != 'n') {
			fail();
		}
		++m_atP;
		operator_name();
		if (*m_atP == 'I') {
			template_args();
		}
		break;
	case 'd':
		if (*++m_atP != 'n') {
			fail();
		}
		destructor_name();
		break;
	default:
		simple_id();
	}
	return;
}

// <decltype>  ::= Dt <expression> E  
//                 # decltype of an id-expression or class member access (C++0x)
//             ::= DT <expression> E  
//                 # decltype of an expression (C++0x)

void
Csignature::decltype1()
{
	MAKE_NODE(decltype);

	if (*m_atP != 'D') {
		fail();
	}
	switch (*++m_atP) {
	case 't':
	case 'T':
		++m_atP;
		expression();
		if (*m_atP != 'E') {
			fail();
		}
		++m_atP;
		break;
	default:
		fail();
	}
}

// <unresolved-type> ::= <template-param>
//                    ::= <decltype>
//                    ::= <substitution>

void
Csignature::unresolved_type(void)
{
	MAKE_NODE(unresolved_type);

	switch (*m_atP) {
	case 'S':
		substitution();
		return;
	case 'T':
		template_param();
		break;
	case 'D':
		decltype1();
		break;
	default:
		fail();
	}
	add_dictionary(m_nodeP);
}

// <unresolved-qualifier-level> ::= <simple-id>

void
Csignature::unresolved_qualifier_level(void)
{
	MAKE_NODE(unresolved_qualifier_level);

	simple_id();
}

// <unresolved-name> ::= [gs] <base-unresolved-name> 
//                       # x or (with "gs") ::x
//                       sr <unresolved-type> <base-unresolved-name>
//                       # T::x / decltype(p)::x
//                       srN <unresolved-type> <unresolved-qualifier-level>+ E <base-unresolved-name>
//						 # T::N::x /decltype(p)::N::x
//                       [gs] sr <unresolved-qualifier-level>+ E <base-unresolved-name>  
//						 # A::x, N::y, A<T>::z; "gs" means leading "::"


void
Csignature::unresolved_name(void)
{
	MAKE_NODE(unresolved_name);
	int		saw_gs;

	saw_gs = 0;
	if (*m_atP == 'g') {
		if (*++m_atP != 's') {
			fail();
		}
		saw_gs = 1;
		++m_atP;
	}

	if (*m_atP == 's') {
		++m_atP;
		if (*m_atP != 'r') {
			fail();
		}
		++m_atP;
		// sr
		switch (*m_atP) {
		case 'N':
			// srN
			if (saw_gs) {
				fail();
			}
			++m_atP;
			unresolved_type();
			break;
		case 'S':
			if (m_atP[1] == 't') {
				m_atP += 2;
				break;
			}
		case 'T':
		case 'D':
			unresolved_type();
			goto skip;
		}
		for (; *m_atP != 'E'; ) {
			unresolved_qualifier_level();
		}
		++m_atP;
	}
skip:
	base_unresolved_name();
}

// <CV-qualifiers> ::= [r] [V] [K] 	# restrict (C99), volatile, const

void
Csignature::cv_qualifiers(void)
{
	MAKE_NODE(cv_qualifiers);

	for (;; ++m_atP) {
		switch(*m_atP) {
		case 'r':
		case 'V':
		case 'K':
			continue;
		}
		return;
}	}

// <function-param> ::= fp <top-level CV-qualifiers> _
//                      # L == 0, first parameter
//        		    ::= fp <top-level CV-qualifiers> <parameter-2 non-negative number> _
//                      # L == 0, second and later parameters
//         		    ::= fL <L-1 non-negative number> p <top-level CV-qualifiers> _
//						# L > 0, first parameter
//				    ::= fL <L-1 non-negative number> p <top-level CV-qualifiers>
//                         <parameter-2 non-negative number> _   
//                      # L > 0, second and later parameters

void
Csignature::function_param(void)
{
	MAKE_NODE(function_param);

	if (*m_atP != 'f') {
		fail();
	}
	switch (*++m_atP) {
	case 'L':
		++m_atP;
		number();
		if (*m_atP != 'p') {
			fail();
		}
	case 'p':
		++m_atP;
		switch (*m_atP) {
		case 'r':
		case 'V':
		case 'K':
			cv_qualifiers();
		}
		compact_number();
		break;
	default:
		fail();
	}
}

// <initializer> ::= pi <expression>* E       

void
Csignature::initializer(void)
{
	MAKE_NODE(initializer);

	if (*m_atP != 'p' || m_atP[1] != 'i') {
		fail();
	}
	m_atP += 2;
	while (*m_atP != 'E') {
		expression();
	}
	++m_atP;
	return;
}

//<expression> ::= <unary operator-name> <expression>
//	           ::= <binary operator-name> <expression>  * 2
//	           ::= <ternary operator-name> <expression> * 3
//			   ::= c1 <expression>+ E

//             ::= st <type>
//             ::= <template-param>
//             ::= sr <type> <unqualified-name>   # dependent name
//             ::= sr <type> <unqualified-name> <template-args>
//                 # dependent template-id
//	           ::= <expr-primary>

void
Csignature::expression(void)
{
	MAKE_NODE(expression);
	int		saw_gs;

	saw_gs = 0;
	switch (*m_atP) {
	case 'L': // <expr-primary> ::= L <type> <value number> E 
			  // # integer literal
			  //                 ::= L <type <value float> E
			  // # floating literal
			  //                 ::= L <mangled-name> E
			  // # external name
		expr_primary();
		return;
	case 'T': // <template-param>
		template_param();
		return;
	case 'a':
		switch (m_atP[1]) {
		case 't':	// at <type> # alignof (a type)
			m_atP += 2;
			type();
			return;
		}
		break;
	case 'c':
		switch (m_atP[1]) {
		case 'v':	// cv <type> <expression> # conversion with one argument
					//::= cv <type> _ <expression>* E
			m_atP += 2;
			type();
			if (*m_atP != '_') {
				expression();
				return;
			}
			for (; *m_atP != 'E'; ) {
				expression();
			}
			++m_atP;
			return;
		case 'l':	// cl <expression>+ E  # call
			m_atP += 2;
			for (; *m_atP != 'E'; )  {
				expression();
			}
			++m_atP;
			return;
		case 'c': // cc <type> <expression> # const_cast<type> (expression)
			m_atP += 2;
			type();
			expression();
			return;
		}
		break;
	case 'd':
		switch (m_atP[1]) {
		case 'l': // dl <expression> # delete expression
		case 'a': // da <expression> # delete[] expression
			m_atP += 2;
			expression();
			return;
		case 'c': // dc <type> <expression> # dynamic_cast<type> (expression)
			m_atP += 2;
			type();
			expression();
			return;
		case 's': // ds <expression> <expression> # expr.*expr
			m_atP += 2;
			expression();
			expression();
			return;
		case 't': // dt <expression> <unresolved-name> # expr.name
			m_atP += 2;
			expression();
			unresolved_name();
			return;
		}	
		break;
	case 'f':	// <function-param>
		switch (m_atP[1]) {
		case 'L':
		case 'p':
			function_param();
			return;
		}
		break;
	case 'g':
		if (m_atP[1] == 's') {
			m_atP += 2;
			saw_gs = 1;
			break;
		}
		break;
	case 'm':	// mm_ <expression> # prefix --
		switch (m_atP[1]) {
		case 'm':
			if (m_atP[2] == '_') {
				m_atP += 3;
				expression();
				return;
			}
			break;
		}
		break;
	case 'p':	// pp_ <expression> # prefix ++
		switch (m_atP[1]) {
		case 'p':
			if (m_atP[2] == '_') {
				m_atP += 3;
				expression();
				return;
			}
			break;
		case 't': // pt <expression> <unresolved-name> # expr->name
			m_atP += 2;
			expression();
			unresolved_name();
			return;
		}
		break;
	case 'r':	// rc <type> <expression> 
				// # reinterpret_cast<type> (expression)
		switch (m_atP[1]) {
		case 'c':
			m_atP += 2;
			type();
			expression();
			return;
		}
		break;
	case 's':
		switch (m_atP[1]) {
		case 'c': // sc <type> <expression>
				  // # static_cast<type> (expression)
			m_atP += 2;
			type();
			expression();
			return;
		case 'p': // sp <expression> # pack expansion
			m_atP += 2;
			expression();
			return;
        case 'r': // sr
			// ::= sr <type> <unqualified-name>   # dependent name
			// ::= sr <type> <unqualified-name> <template-args>
			// This is very different from the standards <unresolved-name>
			m_atP += 2;
            type();
			unqualified_name(0);
			if (*m_atP == 'I') {
				template_args();
			}
			return;
		case 't': // st <type> # sizeof (a type)
			++m_atP;
			type();
			return;
		case 'Z': // sZ <template-param> # size of a parameter pack
           	  	  // sZ <function-param> # size of a function parameter pack
			m_atP += 2;
			if (*m_atP == 'T') {
				template_param();
			} else {
				function_param();
			}
			return;
		}
		break;
	case 't':
		switch (m_atP[1]) {
		case 'i': // ti <type> # typeid (type)
			++m_atP;
			type();
			return;
		case 'e': // te <expression> # typeid (expression)
		case 'w': // tw <expression> # throw expression
			m_atP += 2;
			expression();
			return;
		case 'r': // tr # throw with no operand (rethrow)
			m_atP += 2;
			return;
		}
		break;
	}

	// May have seen gs if arrive here
	// Decide if we have an unresolved name or an operator_name

	switch (*m_atP) {
	case 'd':
		if (m_atP[1] == 'n') {
			saw_gs |= 2;
			break;
		}
		break;
	case 'n':
		switch (m_atP[1]) {
		case 'w': // nw <expression>* _ <type> E 
				  // # new (expr-list) type
       	          // nw <expression>* _ <type> <initializer>
				  // # new (expr-list) type (init)
		case 'a': // na <expression>* _ <type> E
				  // # new[] (expr-list) type
           	  	  // na <expression>* _ <type> <initializer>
				  // # new[] (expr-list) type (init)
			for (m_atP += 2; *m_atP != '_'; ) {
				expression();
			}
			++m_atP;
			type();
			if (*m_atP != 'E') {
				initializer();
			} else {
				++m_atP;
			}
			return;
		}
		break;
	case 'o':
		if (m_atP[1] == 'n') {
			saw_gs |= 2;
			break;
		}
		break;
	case 's':
		if (m_atP[1] == 'r') {
			saw_gs |= 2;
			break;
	}	}

	//<expression> ::= <unary operator-name> <expression>
	//	           ::= <binary operator-name> <expression> * 2
	//	           ::= <trinary operator-name> <expression> * 3

	if (!saw_gs && *m_atP >= 'a' && *m_atP <= 'z') {
		int expressions;

		for (expressions = operator_name(); expressions > 0; --expressions) {
			expression();
		}
		return;
	}
	if (saw_gs & 1) {
		m_atP -= 2;
	}
	unresolved_name();
}

//          <prefix> ::= <prefix> <unqualified-name>
//   	             ::= <template-prefix> <template-args>
//                   ::= <template-param>		# T...
//					 ::= <decltype>				# D[tT]
// 	                 ::= # empty
//					 ::= <prefix> <data-member-prefix> 
//							# <prefix> <source-name> M
// 	                 ::= <substitution>
// 
// <template-prefix> ::= <prefix> <template unqualified-name>
//                   ::= <template-param> 	# T...
//                   ::= <substitution>		# S...

void
Csignature::prefix(void)
{
	int		c;

	for (;;) {
		switch (c = *m_atP) {
		case 'S':
			substitution();
			break;
		case 'T':
			template_param();
			break;
		case 'D':
			if (m_atP[1] == 't' || m_atP[1] == 'T') {
				decltype1();
				return;
			}
		default:
			unqualified_name(1);
		}
	
		switch (*m_atP) {
		case 'E':
			return;
		case 'I':
			ALL_SEEN(template_prefix);
			if (c != 'S') {
				add_dictionary(m_nodeP->m_childP);
			}
			template_args();
			if (*m_atP == 'E') {
				return;
			}
		default:
			ALL_SEEN(prefix);
			if (c != 'S') {
				add_dictionary(m_nodeP->m_childP);
}	}	}	}

// <nested-name> ::= N [<CV-qualifiers>] <prefix> <unqualified-name> E
//               ::= N [<CV-qualifiers>] <template-prefix> <template-args> E

void
Csignature::nested_name(void)
{
	MAKE_NODE(nested_name);

	if (*m_atP != 'N') {
		fail();
	}
	++m_atP;
	switch (*m_atP) {
	case 'r':
	case 'V':
	case 'K':
		cv_qualifiers();
	}
	prefix();
	if (*m_atP != 'E') {
		fail();
	}
	++m_atP;
}

// <local-name> := Z <function encoding> E <entity name> [<discriminator>]
//              := Z <function encoding> E s [<discriminator>]
// <local-name> := Z <function encoding> Ed [ <parameter number> ] _ <entity name>

void
Csignature::local_name(void)
{
	MAKE_NODE(local_name);

	if (*m_atP != 'Z') {
		fail();
	}
	++m_atP;
	encoding();
	if (*m_atP++ != 'E') {
		fail();
	}
	switch (*m_atP) {
	case 's':
		++m_atP;
		break;
	case 'd':
		++m_atP;
		compact_number();
	default:
		name();
	}
	if (*m_atP == '_') {
		discriminator();
	}
}

void
Csignature::St(void)
{
	MAKE_NODE(St);

	m_atP += 2;
}

// <unscoped-name> ::= <unqualified-name>
//      		   ::= St <unqualified-name>   # ::std::

void
Csignature::unscoped_name(void)
{
	MAKE_NODE(unscoped_name);

	if (*m_atP == 'S' && m_atP[1] == 't') {
		St();
	}
	unqualified_name(0);
}
		
// <name> ::= <nested-name>
//        ::= <unscoped-name>
//        ::= <unscoped-template-name> <template-args>
//        ::= <local-name>	# See Scope Encoding below
//

// <unscoped-template-name> ::= <unscoped-name>
//          			    ::= <substitution>

// Returns false if the entire name was obtained through a substition

int
Csignature::name()
{
	MAKE_NODE(name);

	switch (*m_atP) {
	case 'N':
		nested_name();
		return(1);
	case 'Z':
		local_name();
		return(1);
	case 'S':
		// <unscoped-template-name> ::= <substituion>
		if (m_atP[1] != 't') {
			substitution();
			if (*m_atP != 'I') {
				return(0);
			}
			break;
		} 
		// Drop St through to be part of unscoped name
	default:
		unscoped_name();
		if (*m_atP != 'I') {
			return(1);
		}
		/* This is <template_args> which means that we just saw
	   	   <unscoped-template_name> which is a substitution candidate
	       if we didn't just get it from a substition
	 	*/
		LAST_SEEN(unscoped_template_name);
		add_dictionary(m_nodeP->m_childP);
		break;
	}
	template_args();
	return(1);
}

//   <class-enum-type> ::= <name>

int
Csignature::class_enum_type(void)
{
	MAKE_NODE(class_enum_type);

	return name();
}

// <function-type> ::= F [Y] <bare-function-type> E
// Y prefix encodes extern "C"
// <bare-function-type> ::= <signature type>+
// # types are possible return type, then parameter types

void
Csignature::function_type(void)
{
	MAKE_NODE(function_type);


	if (*m_atP != 'F') {
		fail();
	}
	++m_atP;
	if (*m_atP == 'Y') {	// extern "C". 
		++m_atP;
	}
	bare_function_type();
	if (*m_atP != 'E') {
		fail();
	}
	++m_atP;
}

// <array-type> ::= A <positive dimension number> _ <element type>
//              ::= A [<dimension expression>] _ <element type>

void
Csignature::array_type(void)
{
	MAKE_NODE(array_type);
	int	c;

	if (*m_atP != 'A') {
		fail();
	}
	++m_atP;
	if ((c = *m_atP) >= '0' && c <= '9') {
		number();
	} else if (c != '_') {
		expression();
	}
	if (*m_atP != '_') {
		fail();
	}
	++m_atP;
	type();
}

/* <pointer-to-member-type> ::= M <class type> <member type>

    The ABI specifies that any type can be a substitution source, and
    that M is followed by two types, and that when a CV-qualified
    type is seen both the base type and the CV-qualified types are
    substitution sources.  The ABI also specifies that for a pointer
    to a CV-qualified member function, the qualifiers are attached to
    the second type.  Given the grammar, a plain reading of the ABI
    suggests that both the CV-qualified member function and the
    non-qualified member function are substitution sources.  However,
    g++ does not work that way.  g++ treats only the CV-qualified
    member function as a substitution source.  FIXME.  So to work
    with g++, we need to pull off the CV-qualifiers here, in order to
    avoid calling add_substitution() in cplus_demangle_type().  But
    for a CV-qualified member which is not a function, g++ does
    follow the ABI, so we need to handle that case here by calling
    d_add_substitution ourselves.  */
	
void
Csignature::pointer_to_member_type(void)
{
	MAKE_NODE(pointer_to_member_type);
	const char *P;

	if (*m_atP != 'M') {
		fail();
	}
	++m_atP;
	type();		// Class type
	for (P = m_atP; ; ++P) {
		switch (*P) {
		case 'r':
		case 'V':
		case 'K':
			continue;
		}
		break;
	}
	if (P != m_atP && *P != 'F') {
		type();
	} else {
		if (P != m_atP) {
			cv_qualifiers();
		}
		type();
}	}

// Fixed point types. DF<int bits><length><fract bits><sat>
// GNU extension

void
Csignature::fixed_point_type(void)
{
	MAKE_NODE(fixed_point_type);

	if (*m_atP != 'D') {
		fail();
	}
	if (*++m_atP != 'F') {
		fail();
	}
	++m_atP;
	number();
	type();
	number();
	++m_atP;
}

// <vector-type> ::= Dv <number> _ <type> GNU extension
//               ::= Dv _ <expression> _ <type> */

void
Csignature::vector_type(void)
{
	MAKE_NODE(vector_type);
	int	c;

	if (*m_atP != 'D') {
		fail();
	}
	if (*++m_atP != 'v') {
		fail();
	}
	++m_atP;
	if ((c = *m_atP) == '_') {
		++m_atP;
		expression();
	} else {
		number();
	}
	if (*m_atP != '_') {
		fail();
	}
	++m_atP;
	type();
	return;
}

// <type> ::= <builtin-type>
// 	      ::= <function-type>
// 	      ::= <class-enum-type>
// 	      ::= <array-type>
// 	      ::= <pointer-to-member-type>
// 	      ::= <template-param>
// 	      ::= <template-template-param> <template-args>
// 	      ::= <decltype>
// 	      ::= <substitution>
//        ::= <CV-qualifiers> <type>
//        ::= P <type>	# pointer-to
// 	      ::= R <type>	# reference-to
// 	      ::= C <type>	# complex pair (C 2000)
// 	      ::= G <type>	# imaginary (C 2000)
//        ::= U <source-name> <type> # vendor extended type qualifier
//        ::= Dp <type>          # pack expansion (C++0x)
//
// <template-template-param> ::= <template-param>
//						     ::= <substitution>

void
Csignature::type(void)
{
	MAKE_NODE(type);

	int			c, can_subst;

	can_subst = 1;

	switch (*m_atP) {
	case 'r':
	case 'V':
	case 'K':
		cv_qualifiers();
		type();
		break;
	case 'a':	// signed char
	case 'b':	// bool
	case 'c':	// char
	case 'd':	// double
	case 'e':	// long double, __float80
	case 'f':	// float
	case 'g':	// __float128
	case 'h':	// unsigned char
	case 'i':	// int
	case 'j':	// unsigned int
	case 'l':	// long
	case 'm':	// unsigned long
	case 'n': 	// __int128
	case 'o':	// unsigned __int128
	case 's':	// short
	case 't':	// unsigned short
	case 'v':	// void
	case 'w':	// wchar_t
	case 'x':	// long long, __int64
	case 'y':	// unsigned long long, __int64
	case 'z':	// ellipsis ...
		// builtin
		// Not substitutable
		++m_atP;
		return;
	case 'u': 	// u <source-name>	# vendor extended builtin type
		++m_atP;
		source_name();
		break;
	case 'F':
		function_type();
		break;
	case 'S':
		c = m_atP[1];
        if (c != 't') {
		    if (c >= 'a' && c <= 'z') {
			    /* This is a special substitution - start of class-enum-type>
                   If the substitution was a complete type, then it is not
                   a new substitution candidate.  However, if the
                   substitution was followed by template arguments, then
                   the whole thing is a substitution candidate.  */
			    can_subst = class_enum_type();
			    break;
		    }
		    substitution();
		    /* The substituted name may have been a template name and
		       may be followed by tepmlate args.  */
		    if (*m_atP != 'I') {
		    	can_subst = 0;
		    } else {
		    	template_args();
		    }
		    break;
		}
	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	case 'N':
	case 'Z':
		class_enum_type();
		break;
	case 'A':
		array_type();
		break;
	case 'M':
		pointer_to_member_type();
		break;
	case 'T':
		template_param();
		if (*m_atP != 'I') {
			break;
		}
		// ::= <template-template-param> <template-args>
		// <template-template-param> ::= <template-param>
		//                           ::= <substitution>
		LAST_SEEN(template_template_param);
		add_dictionary(m_nodeP->m_childP);
		template_args();
		break;
	case 'P': // P <type>	# pointer-to
	case 'R': // R <type>	# reference-to
	case 'O': // rvalue reference-to (C++0x)
	case 'C': // complex pair (C 2000)
	case 'G': // imaginary (C 2000)
		++m_atP;
		type();
		break;
	case 'U': 
		// ::= U <source-name> <type> # vendor extended type qualifier
		++m_atP;
		source_name();
		type();
		break;
	case 'D':
		can_subst = 0;
		switch(*++m_atP) {
		case 't':	// <decltype>
		case 'T':	// <decltype>
			--m_atP;
			decltype1();
			break;
		case 'p':	// Dp <type> # pack expansion (C++0x)
			++m_atP;
			type();
			break;
		// IEEE 754r floating point
		case 'd':	// 64 bit
		case 'e':	// 128 bit
		case 'f':	// 32 bit
		case 'h':	// 16 bit
		case 'i':	// char32_t
		case 's':	// char16_t
		case 'a':	// auto
		case 'n':	// std::nullptr_t (ie. decltype(nullptr))
			// Builtin type
			++m_atP;
			return;
		case 'F':	// Fixed point types. DF<int bits><length><fract bits><sat>
					// GNU extension
			--m_atP;
			fixed_point_type();
			break;
		case 'v':	// <vector-type> ::= Dv <number> _ <type> GNU extension
                 	//               ::= Dv _ <expression> _ <type> */
			--m_atP;
			vector_type();
			break;
		default:
			fail();
		}
		break;
	default:
		fail();
	}
	if (can_subst) {
		add_dictionary(m_nodeP);
	}
}

// <bare-function-type> ::= <signature type>+
//	# types are possible return type, then parameter types

void
Csignature::bare_function_type(void)
{
	MAKE_NODE(bare_function_type);
	int	seen = 0;
	
	if (*m_atP == 'J') {	// gnu extension indicating first parm return type
		++m_atP;
	}

	for (;;) {
		switch (*m_atP) {
		case 0:
		case 'E':
			goto done;
		default:
			seen = 1;
			type();
	}	}
		
done:
	if (!seen) {
		fail();
}	}

void
Csignature::java_resource(void)
{
	MAKE_NODE(java_resource);
	long	lth;

	lth = compact_number();
	if (lth <= 1) {
		fail();
	}
	identifier(lth-1);
}

// <special-name> ::= TV <type>	# virtual table
//       		  ::= TT <type>	# VTT structure (construction vtable index)
//		          ::= TI <type>	# typeinfo structure
//		          ::= TS <type>	# typeinfo name (null-terminated byte string)
//                ::= T <call-offset> <base encoding>
//		              # base is the nominal target function of thunk
//                ::= Tc <call-offset> <call-offset> <base encoding>
//		              # base is the nominal target function of thunk
//		              # first call-offset is 'this' adjustment
//		              # second call-offset is result adjustment
//                ::= GV <object name>	# Guard variable for one-time initialization
//			          # No <type>

void
Csignature::special_name(void)
{
	MAKE_NODE(special_name);

	switch (*m_atP) {
	case 'T':
		switch (*++m_atP) {
		case 'V':	// ::= TV <type>	# virtual table
		case 'T':	// ::= TT <type>	# VTT structure (construction vtable index)
		case 'I':	// ::= TI <type>	# typeinfo structure
		case 'S':	// ::= TS <type>	# typeinfo name (null-terminated byte str)
		case 'F':	// gnu extension typeinfo_fn
		case 'J':	// gnu extension java_class
			++m_atP;
			// The type is not a name but the classname
			type();
			return;
		case 'h':	// Virtual overide thunk with fixed this adjustment
					// <special-name> ::= Th <offset number> _ <base encoding>
		case 'v':	// Virtual override thunk with virtual adjustment
					// <special-name> ::= Tv <offset number> _ <virtual offset number> _ <base encoding>
					// # base is the nominal target function of thunk
			call_offset();
			encoding();	// <base encoding>
			return;
		case 'c': 	// ::= Tc <call-offset> <call-offset> <base encoding>
		    		//  # base is the nominal target function of thunk
		    		//  # first call-offset is 'this' adjustment
		    		//  # second call-offset is result adjustment
			++m_atP;
			call_offset();
			call_offset();
			encoding();	// <base encoding>
			return;
		case 'C':	// gnu extension construction vtable
			++m_atP;
			type();	// Derived type
			compact_number();
			type();	// Base type
			return;
		}
		break;
	case 'G':
		switch (*++m_atP) {
		case 'V': 	// <special-name> ::= GV <object name>
					// # Guard variable for one-time initialization # No <type>
		case 'R':	// # Reftemp GNU extension ?
			++m_atP;
			name();
			return;
		case 'A':	// # Hidden_alias GNU extension ?
			++m_atP;
			encoding();
			return;
		case 'r':	// # Java resource GNU extension ?
			++m_atP;
			java_resource();
			return;
		}
		break;
	}
	fail();
}

// <encoding> ::= <function name> <bare-function-type>
//            ::= <data name>
//			  ::= <special-name>

void
Csignature::encoding(void)
{
	MAKE_NODE(encoding);

	switch (*m_atP) {
	case 'G':
	case 'T':
		special_name();
		return;
	default:
		// <encoding> ::= <function name> <bare-function-type>
		//            ::= <data name>
		name();
		if (*m_atP && *m_atP != 'E') {
			bare_function_type();
}	}	}

// <mangled-name> ::= _Z <encoding>

void
Csignature::mangled_name(void)
{
	MAKE_NODE(mangled_name);

	if (m_atP[0] == '_' && m_atP[1] == 'Z') {
		m_atP += 2;
		encoding();
	} else {
		// Not mangled
		m_atP += strlen(m_atP);
}	}

/* This logic must fully scan a mangled_name and return the logical end
   of that mangled name or 0 on error. 
 */

const char *
Csignature::signature(const char *startP)
{
	int				need;

	m_classname.invalidate();
	m_template.invalidate();
	m_function.invalidate();
	m_function_name.invalidate();

	if (!m_nodes_max) {
		need        = 256;
		m_nodesP    = (Csignature_node *) Xmalloc(need * sizeof(Csignature_node));
		m_nodes_max = need;
	}
	for (;;) {
		try {
			m_nodes_endP = m_nodesP + m_nodes_max;
			m_headP      = 0;
			m_tailP      = 0;
			m_nodeP      = get_node();
			m_nodeP->m_startP = startP;
			m_nodeP->m_type   = SIGNATURE_root;
			m_atP             = startP;
			mangled_name();
			assert(m_nodeP->m_type == SIGNATURE_root);
			m_nodeP->m_endP   = m_atP;
			m_nodeP->reverse();
			if (*m_atP) {
				fail();
			}

#ifdef TESTING
			{
				Csignature_node *nodeP;
				int				cnt = -1;
				fprintf(stderr, "Dictionary\n");
				for (nodeP = m_headP; nodeP; nodeP = nodeP->m_nextP) {
					fprintf(stderr, "%d\n", ++cnt);
					nodeP->dump(stderr, 0, 0, 1);
				}
				fprintf(stderr, "Full tree\n");
				dump(stderr);
				fprintf(stderr, "End of tree\n");
			}
#endif
			return(m_atP);
		} catch (Csignature_error e) {
			return(0);
		} catch (Csignature_expand e) {
			need = m_nodes_max << 1;
			Xfree(m_nodesP);
			m_nodesP    = (Csignature_node *) Xmalloc(need * sizeof(Csignature_node));
			m_nodes_max = need;
}	}	}

int
Csignature::dictionary_count(void)
{
	Csignature_node *nodeP;
	int 			cnt;

	cnt = 0;
	for (nodeP = m_headP; nodeP; nodeP = nodeP->m_nextP) {
		++cnt;
	}
	return cnt;
}

char *
Csignature::get_classname(void)
{
	if (!m_nodeP) {
		return 0;
	}
	return m_classname.get_classname(m_nodeP);
}

char *
Csignature::get_template(void)
{
	if (!m_nodeP) {
		return 0;
	}
	return m_template.get_template(m_nodeP);
}

// Find the function name from a static variable declared inside that function
// or the function name from a standard function call etc.

char *
Csignature::get_function(void)
{
	if (!m_nodeP) {
		return 0;
	}
	return m_function.get_function(m_nodeP);
}

// Just get the actual function name -- not the class it is in
char *
Csignature::get_function_name(void)
{
	if (!m_nodeP) {
		return 0;
	}
	return m_function_name.get_function_name(m_nodeP);
}

int
Csignature::same_class(Csignature *otherP)
{
	char	*P1, *P2;

	P1 = get_classname();
	if (!P1) {
		return(0);
	}
	P2 = otherP->get_classname();
	if (!P2) {
		return(0);
	}
	return !strcmp(P1, P2);
}

int
Csignature::same_function(Csignature *otherP)
{
	char	*P1, *P2;

	P1 = get_function();
	if (!P1) {
		return(0);
	}
	P2 = otherP->get_function();
	if (!P2) {
		return(0);
	}
	return !strcmp(P1, P2);
}

int
Csignature::same_function_name(Csignature *otherP)
{
	char	*P1, *P2;

	P1 = get_function_name();
	if (!P1) {
		return(0);
	}
	P2 = otherP->get_function_name();
	if (!P2) {
		return(0);
	}
	return !strcmp(P1, P2);
}

int
Csignature::is_function(const char *startP)
{
	Csignature_node *nodeP, *nextP;

	if (startP[0] != '_' || startP[1] != 'Z') {
		return(0);
	}
	if (!signature(startP)) {
		return(0);
	}
	for (nodeP = m_nodeP; nodeP;) {
		switch (nodeP->m_type) {
		case SIGNATURE_bare_function_type:
			return 1;
		case SIGNATURE_name:
			return 0;
		default:
			// Move to last child if one
			nodeP = nodeP->m_childP;
			if (nodeP) {
				for (; (nextP = nodeP->m_siblingP); nodeP = nextP);
	}	}	}
	return 0;
}
